# -*- Mode: Python -*-
# GObject-Introspection - a framework for introspecting GObject libraries
# Copyright (C) 2008  Johan Dahlin
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.
#

# Poor mans GObject python bindings
# Incidentally portable beyond CPython, due to usage of ctypes
# Why not PyGObject?
# - 1) Avoid PyGObject dependency
# - 2) Low-level binding
# - 3) Implementation independent
# - 4) Usage lower in the stack
#
# based on defsgen.py (C) 2006 John Finlay
#

import os
import ctypes
from ctypes.util import find_library


# Constants

# from gobject/gtype.h


def _make_fundamental(x):
    G_TYPE_FUNDAMENTAL_SHIFT = 2
    return x << G_TYPE_FUNDAMENTAL_SHIFT


TYPE_INVALID = _make_fundamental(0)
TYPE_NONE = _make_fundamental(1)
TYPE_INTERFACE = _make_fundamental(2)
TYPE_CHAR = _make_fundamental(3)
TYPE_UCHAR = _make_fundamental(4)
TYPE_BOOLEAN = _make_fundamental(5)
TYPE_INT = _make_fundamental(6)
TYPE_UINT = _make_fundamental(7)
TYPE_LONG = _make_fundamental(8)
TYPE_ULONG = _make_fundamental(9)
TYPE_INT64 = _make_fundamental(10)
TYPE_UINT64 = _make_fundamental(11)
TYPE_ENUM = _make_fundamental(12)
TYPE_FLAGS = _make_fundamental(13)
TYPE_FLOAT = _make_fundamental(14)
TYPE_DOUBLE = _make_fundamental(15)
TYPE_STRING = _make_fundamental(16)
TYPE_POINTER = _make_fundamental(17)
TYPE_BOXED = _make_fundamental(18)
TYPE_PARAM = _make_fundamental(19)
TYPE_OBJECT = _make_fundamental(20)

# Typedefs

# FIXME - this is wrong on win64, where long == 32 but size_t == 64
GType = ctypes.c_ulong

TYPE_FLAG_ABSTRACT = 1 << 4

# Structs


class GTypeClass(ctypes.Structure):
    _fields_ = [('g_type', GType)]


class GTypeInstance(ctypes.Structure):
    _fields_ = [('g_class', ctypes.POINTER(GTypeClass))]


class GEnumValue(ctypes.Structure):
    _fields_ = [('value', ctypes.c_int),
                ('value_name', ctypes.c_char_p),
                ('value_nick', ctypes.c_char_p)]


class GFlagsValue(ctypes.Structure):
    _fields_ = [('value', ctypes.c_uint),
                ('value_name', ctypes.c_char_p),
                ('value_nick', ctypes.c_char_p)]


class GEnumClass(ctypes.Structure):
    _fields_ = [('g_type_class', GTypeClass),
                ('minimum', ctypes.c_int),
                ('maximum', ctypes.c_int),
                ('n_values', ctypes.c_uint),
                ('values', ctypes.POINTER(GEnumValue))]

    def get_values(self):
        for i in range(self.n_values):
            yield self.values[i]


class GFlagsClass(ctypes.Structure):
    _fields_ = [('g_type_class', GTypeClass),
               ('mask', ctypes.c_uint),
               ('n_values', ctypes.c_uint),
               ('values', ctypes.POINTER(GFlagsValue))]

    def get_values(self):
        for i in range(self.n_values):
            yield self.values[i]


class GTypeInterface(ctypes.Structure):
    _fields_ = [('g_type', GType),
                ('g_instance_type', GType)]


class GParamSpec(ctypes.Structure):
    _fields_ = [('g_type_instance', GTypeInstance),
                ('name', ctypes.c_char_p),
                ('flags', ctypes.c_uint),
                ('value_type', GType),
                ('owner_type', GType)]


class GSignalInfo(ctypes.Structure):
    _fields_ = [('signal_id', ctypes.c_uint),
                ('signal_name', ctypes.c_char_p),
                ('itype', GType),
                ('signal_flags', ctypes.c_uint),
                ('return_type', GType),
                ('n_params', ctypes.c_uint),
                ('param_types', ctypes.POINTER(GType))]

    def get_params(self):
        for i in range(self.n_params):
            yield self.param_types[i]


if os.name == 'nt':
    _library_path = find_library('libgobject-2.0-0')
else:
    _library_path = find_library('gobject-2.0')
if not _library_path:
    raise ImportError("Could not find gobject-2.0 library")
_gobj = ctypes.CDLL(_library_path, ctypes.RTLD_GLOBAL)
_gobj.g_type_init()

# We need to initialize threads just in case part of the library
# wants to use gthreads; libsoup seems to use g_once for its
# type init functions.
if os.name == 'nt':
    _threads_library_path = find_library('libgthread-2.0-0')
else:
    _threads_library_path = find_library('gthread-2.0')
if not _threads_library_path:
    raise ImportError("Could not find gthread-2.0 library")
_gthreads = ctypes.CDLL(_threads_library_path, ctypes.RTLD_GLOBAL)
_gthreads.g_thread_init.argtypes = [ctypes.c_void_p]
_gthreads.g_thread_init(None)

# Functions


def _gwrap(fname, ret, *argtypes):

    def _deco(f):
        f._cfunc = getattr(_gobj, fname)
        f._cfunc.restype = ret
        f._cfunc.argtypes = argtypes
        return f
    return _deco


@_gwrap('g_object_new', ctypes.c_void_p, GType)
def object_new(type_id):
    return _gobj.g_object_new(type_id, None)


# Workaround this error:
#   GLib-GObject-CRITICAL **: g_param_spec_pool_list:
#   assertion `pool != NULL' failed
# which happens when trying to introspect an interface before instantiating
# a GObject.

object_new(TYPE_OBJECT)
_gobj.g_initially_unowned_get_type()


@_gwrap('g_type_name', ctypes.c_char_p, GType)
def type_name(type_id):
    return _gobj.g_type_name(type_id)


@_gwrap('g_type_from_name', GType, ctypes.c_char_p)
def type_from_name(name):
    return _gobj.g_type_from_name(name)


@_gwrap('g_type_fundamental', GType, GType)
def type_fundamental(type_id):
    return _gobj.g_type_fundamental(type_id)


@_gwrap('g_type_test_flags', ctypes.c_int, GType, ctypes.c_int)
def type_test_flags(type_id, flag):
    return _gobj.g_type_test_flags(type_id, flag)


def type_is_abstract(type_id):
    return type_test_flags(type_id, TYPE_FLAG_ABSTRACT) != 0


@_gwrap('g_type_parent', GType, GType)
def type_parent(type_id):
    return _gobj.g_type_parent(type_id)


@_gwrap('g_type_class_ref', ctypes.POINTER(GTypeClass), GType)
def type_class_ref(type_id):
    fundamental_type = type_fundamental(type_id)
    if fundamental_type == TYPE_INVALID:
        return None
    if fundamental_type == TYPE_ENUM:
        typeclass = GEnumClass
    elif fundamental_type == TYPE_FLAGS:
        typeclass = GFlagsClass
    else:
        raise NotImplementedError(fundamental_type)
    ptr = _gobj.g_type_class_ref(type_id)
    return ctypes.cast(ptr, ctypes.POINTER(typeclass)).contents


@_gwrap('g_object_class_list_properties',
       ctypes.POINTER(ctypes.POINTER(GParamSpec)),
       ctypes.POINTER(GTypeClass), ctypes.POINTER(ctypes.c_uint))
def object_class_list_properties(type_id):
    klass = _gobj.g_type_class_ref(type_id)
    n = ctypes.c_uint()
    pspecs = _gobj.g_object_class_list_properties(klass, ctypes.byref(n))
    for i in range(n.value):
        yield ctypes.cast(pspecs[i], ctypes.POINTER(GParamSpec)).contents


@_gwrap('g_type_default_interface_ref', ctypes.c_void_p, GType)
def type_default_interface_ref(type_id):
    return _gobj.g_type_default_interface_ref(type_id)


@_gwrap('g_object_interface_list_properties',
       ctypes.POINTER(ctypes.POINTER(GParamSpec)),
       ctypes.c_void_p, ctypes.POINTER(ctypes.c_uint))
def object_interface_list_properties(type_id):
    iface = type_default_interface_ref(type_id)
    n = ctypes.c_uint()
    pspecs = _gobj.g_object_interface_list_properties(iface, ctypes.byref(n))
    for i in range(n.value):
        yield ctypes.cast(pspecs[i], ctypes.POINTER(GParamSpec)).contents


@_gwrap('g_type_interfaces', ctypes.POINTER(GType), GType,
       ctypes.POINTER(ctypes.c_uint))
def type_interfaces(type_id):
    n = ctypes.c_uint()
    type_ids = _gobj.g_type_interfaces(type_id, ctypes.byref(n))
    for i in range(n.value):
        yield type_ids[i]


@_gwrap('g_type_interface_prerequisites', ctypes.POINTER(GType), GType,
       ctypes.POINTER(ctypes.c_uint))
def type_interface_prerequisites(type_id):
    n = ctypes.c_uint()
    type_ids = _gobj.g_type_interface_prerequisites(type_id, ctypes.byref(n))
    for i in range(n.value):
        yield type_ids[i]


@_gwrap('g_signal_query', None, ctypes.c_uint, ctypes.POINTER(GSignalInfo))
def signal_query(sigid, qptr):
    _gobj.g_signal_query(sigid, qptr)


@_gwrap('g_signal_list_ids', ctypes.POINTER(ctypes.c_uint),
       GType, ctypes.POINTER(ctypes.c_uint))
def signal_list(type_id):
    n = ctypes.c_uint()
    signal_ids = _gobj.g_signal_list_ids(type_id, ctypes.byref(n))
    for i in range(n.value):
        info = GSignalInfo()
        signal_query(signal_ids[i], ctypes.byref(info))
        yield info

TYPE_GTYPE = type_from_name('GType')
