/* vi:set ts=8 sts=0 sw=8:
 * $Id: file.c,v 1.20 1999/01/18 19:45:53 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn <kahn@zk3.dec.com>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * file locking/unlocking code borrowed from mutt (www.mutt.org).
 */
#include <stdio.h>
#include <dirent.h>
#include <errno.h>
#include <stdlib.h>
#include <ctype.h>
#include <utime.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/file.h>
#include "win.h"
#include "doc.h"
#include "dialog.h"
#include "msgbar.h"
#include "msgbox.h"
#include "file.h"
#include "prefs.h"

#define MAXLOCKATTEMPT 5


/*** external functions ***/
extern void win_set_title(doc_t *d);


/*** global function definitions ***/
/*
 * PUBLIC: file_exist
 *
 * returns true if file exists
 */
bool_t
file_exist(char *fname)
{
	struct stat sb;

	if (stat(fname, &sb) == -1)
		return FALSE;

	return TRUE;
} /* file_exist */


/*
 * PUBLIC: file_is_readonly
 *
 * returns true/false if file is readonly.  requires filename.
 */
bool_t
file_is_readonly(char *fname)
{
	if (strcmp(fname, UNTITLED) == 0 || access(fname, F_OK))
		return FALSE;

	return (access(fname, W_OK) != 0);
} /* file_is_readonly */


/*
 * PUBLIC: file_full_pathname_make
 *
 * given a filename, tries to construct the full pathname to the file.  if
 * successfuly, returns a newly allocated buffer with the full pathname.
 * if the filename is already a full pathname, the pointer to the filename
 * is returned as is (so no new buffer is allocated).
 */
char *
file_full_pathname_make(char *fname)
{
	char *full, *cwd;
	int len;

	if (fname[0] == '/')
		return g_strdup(fname);

	if ((cwd = getcwd(NULL, MAXPATH)) == NULL)
		return g_strdup(fname);

	if (cwd[strlen(cwd) - 1] == '/')
		cwd[strlen(cwd) - 1] = '\0';

	len = strlen(cwd) + strlen(fname) + 2;
	full = (char *)g_malloc(len);
	if ((strlen(fname) > 2) && fname[0] == '.' && fname[1] == '/')
		g_snprintf(full, len, "%s/%s", cwd, fname + 2);
	else
		g_snprintf(full, len, "%s/%s", cwd, fname);
	g_free(cwd);

	return full;
} /* file_full_pathname_make */


#ifdef APP_GNP
/*
 * PUBLIC: file_open_execute
 *
 * actually do the low-level file opening and stuff it into the document's
 * text widget.
 */
int
file_open_execute(win_t *w, doc_t *d)
{
	size_t size;
	off_t bytesread;
	long num;
	char *buf;
	FILE *fp;

	/* get file stats (e.g., file size is used by files list window) */
	if (strcmp(d->fname, UNTITLED)) {
		d->sb = (struct stat *)g_malloc(sizeof(struct stat));
		if (stat(d->fname, d->sb) == -1) {
			g_free(d->sb);
			d->sb = NULL;
		}
	}

	if (d->sb && d->sb->st_size > 0) {
		/* open file */
		if ((fp = fopen(d->fname, "r")) == NULL) {
			char msg[255];

			g_snprintf(msg, 255,
				"file_open_execute: could not open '%s'",
				d->fname);
			perror(msg);
			return 1;
		}
		num = file_lock(d->fname, fp, FALSE, FALSE, TRUE);
		gtk_text_freeze(GTK_TEXT(d->data));

#ifdef USE_SIMPLE_READ_ALGORITHM
#define READ_CHUNK	32767
		/*
		 * the simple read algorithm basically just reads in READ_CHUNK
		 * bytes at a time.  we can still do better, but this a >>lot<<
		 * better than before, where we were reading 10 bytes at a
		 * time!  ideally, we'd like to alloc a buffer to read the
		 * entire file in at once, or a buffer as large as possible if
		 * the file is really really large.
		 */
		size = READ_CHUNK;
		buf = (char *)g_malloc(size+1);
#else
		/*
		 * the more aggressive/intelligent method is to try to allocate
		 * a buffer to fit all the contents of the file; that way, we
		 * only read the file once.  basically, we try to allocate a
		 * buffer that is twice the number of bytes that is needed.
		 * this is because gtk_text_insert() does a memcpy() of the
		 * read buffer into the text widget's buffer, so we must ensure
		 * that there is enough memory for both the read and the
		 * memcpy().  since we'd like to read the entire file at once,
		 * so we start with a buffer that is twice the filesize.  if
		 * this fails (e.g., we're reading a really large file), we
		 * keep reducing the size by half until we are able to get a
		 * buffer.
		 */
		size = 2 * (d->sb->st_size + 1);
		while ((buf = (char *)malloc(size)) == NULL)
			size /= 2;

		GNPDBG_FILE(("size %lu is ok (using half)\n", (gulong)size));
		g_free(buf);
		size /= 2;
		if ((buf = (char *)malloc(size + 1)) == NULL) {
			perror("file_open_execute: unable to malloc read buf");
			fclose(fp);
			return 1;
		}
#endif	/* ifdef USE_CONSERVATIVE_READ_ALGORITHM */

		GNPDBG_FILE(("malloc'd %lu bytes for read buf\n",
			     (gulong)(size+1)));

		/* buffer allocated, now actually read the file */
		bytesread = 0;
		while (bytesread < d->sb->st_size) {
			num = fread(buf, 1, size, fp);
			if (num > 0) {
				gtk_text_insert(
					GTK_TEXT(d->data),
					NULL, NULL, NULL,
#if 0
					text_font,
					&text_fg_color, &text_bg_color,
#endif
					buf, num);
				bytesread += num;
			} else if (ferror(fp)) {
				/* need to print warning/error message */
				perror("read error");
				break;
			} else /* if (num == 0) */ {
				GNPDBG_FILE(("hmm... read zero bytes"));
			}
		}
		g_free(buf);
		num = file_unlock(d->fname, fp);
		fclose(fp);
		gtk_text_thaw(GTK_TEXT(d->data));
	} /* filesize > 0 */

	/* misc settings */
	gtk_label_set(GTK_LABEL(d->tablabel), doc_basefname(d));
	gtk_text_set_point(GTK_TEXT(d->data), 0);

	return 0;
} /* file_open_execute */
#endif


/*
 * PUBLIC: file_perm_string
 *
 * returns the rwx permissions of a file in a string.  non-reentrant.
 * does not look at setgid, setuid, or sticky bit settings.
 */
char *
file_perm_string(struct stat *sb)
{
	static char perm[10];

	g_snprintf(perm, 10, "%c%c%c%c%c%c%c%c%c",
		(sb->st_mode & S_IRUSR) ? 'r' : '-',
		(sb->st_mode & S_IWUSR) ? 'w' : '-',
		(sb->st_mode & S_IXUSR) ? 'x' : '-',
		(sb->st_mode & S_IRGRP) ? 'r' : '-',
		(sb->st_mode & S_IWGRP) ? 'w' : '-',
		(sb->st_mode & S_IXGRP) ? 'x' : '-',
		(sb->st_mode & S_IROTH) ? 'r' : '-',
		(sb->st_mode & S_IWOTH) ? 'w' : '-',
		(sb->st_mode & S_IXOTH) ? 'x' : '-'
		);
	return perm;
} /* file_perm_string */


#ifdef APP_GNP
/*
 * PUBLIC: file_save_execute
 *
 * actually dump the contents of the document's text widget and do the
 * low-level file save.
 */
bool_t
file_save_execute(doc_t *d, bool_t saveas)
{
	FILE *fp;
	int ret;
	char *buf;

	if (!saveas && file_is_readonly(d->fname)) {
		(void)do_dialog_ok("Read only file!",
			" Cannot save file.  File is read only! ");
		return FALSE;
	}

	if ((fp = fopen(d->fname, "w")) == NULL) {
		g_warning("file_save_execute: can't open '%s' for saving",
			d->fname);
		return FALSE;
	}
	ret = file_lock(d->fname, fp, TRUE, FALSE, TRUE);
	gtk_text_thaw(GTK_TEXT(d->data));

	/*
	 * this isn't very memory efficient; gtk_editable_get_chars()
	 * (actually, it's gtk_text_get_chars()), returns the text in a newly
	 * malloc'd buffer.  we're basically making another copy of the
	 * document just to save it.  if the document is large, we might run
	 * into problems.
	 */
	buf = gtk_editable_get_chars(GTK_EDITABLE(d->data), 0,
					gtk_text_get_length(GTK_TEXT(d->data)));
	if (fputs(buf, fp) == EOF) {
		perror("Error saving file");
		g_free(buf);
		fclose(fp);
		return FALSE;
	}
	g_free(buf);
	fflush(fp);
	ret = file_unlock(d->fname, fp);
	fclose(fp);

	if (d->sb == NULL)
		d->sb = (struct stat *)g_malloc(sizeof(struct stat));
	if (stat(d->fname, d->sb) == -1) {
		g_free(d->sb);
		d->sb = NULL;
	}

	gtk_label_set(GTK_LABEL(d->tablabel), doc_basefname(d));
	d->changed = FALSE;
	if (!d->changed_id)
		d->changed_id = gtk_signal_connect(
					GTK_OBJECT(d->data), "changed",
					GTK_SIGNAL_FUNC(doc_changed_cb), d);

	win_set_title(d);

	msgbar_printf(d->w, "Saved %s", doc_basefname(d));
	msgbox_printf("saved %s", doc_basefname(d));
	return TRUE;
} /* file_save_execute */
#endif



/*
 * PUBLIC: file_lock
 *
 * this routine was taken from mutt (www.mutt.org).  uses either fcntl() or
 * flock() to perform file locking of a file.  dot-locking code is currently
 * not supported.
 *
 * Args:
 *	excl	if excl != 0, request an exclusive lock
 *	dot	if dot != 0, try to dotlock the file
 *	timeout	should retry locking?
 */
int 
file_lock(const char *path, FILE *fp, bool_t excl, bool_t dot, bool_t timeout)
{
	int r = 0;
	int fd = fileno(fp);
#if defined (USE_FCNTL) || defined (USE_FLOCK)
	int count;
	int attempt;
	struct stat prev_sb;
#endif
#ifdef USE_FCNTL
	struct flock lck;

	memset(&lck, 0, sizeof(struct flock));
	lck.l_type = excl ? F_WRLCK : F_RDLCK;
	lck.l_whence = SEEK_SET;

	count = 0;
	attempt = 0;
	while (fcntl(fd, F_SETLK, &lck) == -1) {
		struct stat sb;

		if (errno != EAGAIN && errno != EACCES) {
			perror("file_lock: fcntl error");
			return -1;
		}

		if (fstat(fd, &sb) != 0)
			sb.st_size = 0;

		if (count == 0)
			prev_sb = sb;

		/* only unlock file if it is unchanged */
		if (prev_sb.st_size == sb.st_size && ++count >=
			(timeout ? MAXLOCKATTEMPT : 0)) {

			if (timeout)
				g_warning("Timeout exceeded while attempting "
					  "fcntl lock!");
			return -1;
		}
		prev_sb = sb;

		printf("Waiting for fcntl lock... %d", ++attempt);
		sleep(1);
	}
#endif /* USE_FCNTL */

#ifdef USE_FLOCK
	count = 0;
	attempt = 0;
	while (flock(fd, (excl ? LOCK_EX : LOCK_SH) | LOCK_NB) == -1) {
		struct stat sb;
		if (errno != EWOULDBLOCK) {
			perror("flock");
			r = -1;
			break;
		}
		if (fstat(fd, &sb) != 0)
			sb.st_size = 0;

		if (count == 0)
			prev_sb = sb;

		/* only unlock file if it is unchanged */
		if (prev_sb.st_size == sb.st_size && ++count >=
			(timeout ? MAXLOCKATTEMPT : 0)) {
			if (timeout)
				g_warning("Timeout exceeded while attempting "
					  "flock lock!");
			r = -1;
			break;
		}
		prev_sb = sb;

		printf("Waiting for flock attempt... %d", ++attempt);
		sleep(1);
	}
#endif /* USE_FLOCK */

#ifdef USE_DOTLOCK
	if (r == 0 && dot)
		r = dotlock_file(path, timeout);
#endif /* USE_DOTLOCK */

	if (r == -1) {
		/* release any other locks obtained in this routine */

#ifdef USE_FCNTL
		lck.l_type = F_UNLCK;
		fcntl(fd, F_SETLK, &lck);
#endif /* USE_FCNTL */

#ifdef USE_FLOCK
		flock(fd, LOCK_UN);
#endif /* USE_FLOCK */

		return (-1);
	}
	return 0;
} /* file_lock */


/*
 * PUBLIC: file_unlock
 *
 * this routine was taken from mutt (www.mutt.org).  uses either fcntl() or
 * flock() to unlock a file previously locked by lock_file().
 * dot-locking code is currently not supported.
 */
int 
file_unlock(const char *path, FILE *fp)
{
	int fd = fileno(fp);
#ifdef USE_FCNTL
	struct flock unlockit = {F_UNLCK, 0, 0, 0};

	memset(&unlockit, 0, sizeof(struct flock));
	unlockit.l_type = F_UNLCK;
	unlockit.l_whence = SEEK_SET;
	fcntl(fd, F_SETLK, &unlockit);
#endif

#ifdef USE_FLOCK
	flock(fd, LOCK_UN);
#endif

#ifdef USE_DOTLOCK
	undotlock_file(path);
#endif

	return 0;
} /* file_unlock */


/****** dotlocking is currently not supported.  eventually, it probably will
	be, which is why the code is still here *****/
#ifdef USE_DOTLOCK
/* HP-UX and ConvexOS don't have this macro */
#ifndef S_ISLNK
#define S_ISLNK(x) (((x) & S_IFMT) == S_IFLNK ? 1 : 0)
#endif

/* parameters: 
 * path - file to lock
 * retry - should retry if unable to lock?
 */
static int 
dotlock_file(const char *path, int retry)
{
	const char *pathptr = path;
	char lockfile[MAXPATH];
	char nfslockfile[MAXPATH];
	char realpath[MAXPATH];
	struct stat sb;
	size_t prev_size = 0;
	int count = 0;
	int attempt = 0;
	int fd;

#if 0
	/* if the file is a symlink, find the real file to which it refers */
	for (;;) {
		if (lstat(pathptr, &sb) != 0) {
			perror(pathptr);
			return -1;
		}

		if (S_ISLNK(sb.st_mode)) {
			char linkfile[MAXPATH];
			char linkpath[MAXPATH];

			if ((count = readlink(pathptr, linkfile,
				sizeof(linkfile))) == -1) {
				perror(path);
				return -1;
			}

			linkfile[count] = 0;	/* readlink() does not NUL terminate the string! */
			mutt_expand_link(linkpath, pathptr, linkfile);
			strfcpy(realpath, linkpath, sizeof(realpath));
			pathptr = realpath;
		} else
			break;
	}
#endif

	snprintf(nfslockfile, sizeof(nfslockfile),
		"%s.%s.%d", pathptr, Hostname, (int)getpid());
	snprintf(lockfile, sizeof(lockfile), "%s.lock", pathptr);
	unlink(nfslockfile);

	while ((fd = open(nfslockfile, O_WRONLY | O_EXCL | O_CREAT, 0)) < 0) {
		if (errno != EAGAIN) {
			mutt_perror("cannot open NFS lock file!");
			return (-1);
		}
	}
	close(fd);

	count = 0;
	for (;;) {
		link(nfslockfile, lockfile);
		if (stat(nfslockfile, &sb) != 0) {
			perror("stat");
			return -1;
		}

		if (sb.st_nlink == 2)
			break;

		if (stat(path, &sb) != 0)
			sb.st_size = 0;

		if (count == 0)
			prev_size = sb.st_size;

		/* only try to remove the lock if the file is not changing */
		if (prev_size == sb.st_size && ++count >=
			(retry ? MAXLOCKATTEMPT : 0)) {
			if (retry && mutt_yesorno("Lock count exceeded, remove lock?", 1) == 1) {
				unlink(lockfile);
				count = 0;
				attempt = 0;
				continue;
			} else
				return (-1);
		}
		prev_size = sb.st_size;

		printf("Waiting for lock attempt #%d...", ++attempt);
		sleep(1);
	}

	unlink(nfslockfile);

	return 0;
} /* dotlock_file */


static int 
undotlock_file(const char *path)
{
	const char *pathptr = path;
	char lockfile[MAXPATH];
	char realpath[MAXPATH];
	struct stat sb;
	int n;

	FOREVER
	{
		dprint(2, (debugfile, "undotlock: unlocking %s\n", path));

		if (lstat(pathptr, &sb) != 0) {
			mutt_perror(pathptr);
			return (-1);
		}
		if (S_ISLNK(sb.st_mode)) {
			char linkfile[MAXPATH];
			char linkpath[MAXPATH];

			if ((n = readlink(pathptr, linkfile, sizeof(linkfile))) == -1) {
				mutt_perror(pathptr);
				return (-1);
			}
			linkfile[n] = 0;	/* readlink() does not NUL terminate the string! */
			mutt_expand_link(linkpath, pathptr, linkfile);
			strfcpy(realpath, linkpath, sizeof(realpath));
			pathptr = realpath;
			continue;
		} else
			break;
	}

	snprintf(lockfile, sizeof(lockfile), "%s.lock", pathptr);
	unlink(lockfile);
	return 0;
}
#endif /* USE_DOTLOCK */


/* the end */
