/* libfind - a helper library for use with rpmfind operations.
 * Copyright (C) 1999  James Henstridge
 * Concepts for rpmfind Copyright (C) 199x  Daniel Veillard
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "search.h"
#include <stdlib.h>
#include <unistd.h>
#include <libgnome/libgnome.h>
#include <time.h>
#include "rdf_api.h"
#include "rdf.h"
#include "trans.h"
#include "distrib.h"
#include "guess.h"

#ifdef HAVE_REGEX_H
#include <regex.h>
#endif

/* undef this if you don't want to cache rpmData structures */
#define RPM_DATA_CACHE

static GList *allPackages = NULL;
extern GVoidFunc libfind_idle_func;

GList *getFullPackageList(void) {
  rdfSchema rdf;
  rdfNamespace rdfNs, rpmNs;
  rdfDescription desc;
  char *file = update_file("resources/fullIndex.rdf.gz");
  GList *tail = NULL;
  GHashTable *check;

  if (allPackages != NULL)
    return allPackages;

  rdf = rdfRead(file);
  if (rdf == NULL) {
    g_warning("can't open catalog '%s'", file);
    g_free(file);
    return NULL;
  }
  rdfNs = rdfGetNamespace(rdf, "http://www.w3.org/TR/WD-rdf-syntax#");
  if (rdfNs == NULL) {
    g_warning("%s is not an RDF schema", file);
    rdfDestroySchema(rdf);
    g_free(file);
    return NULL;
  }
  rpmNs = rdfGetNamespace(rdf, "http://www.rpm.org/");
  if (rpmNs == NULL) {
    g_warning("%s is not an RPM specific RDF schema", file);
    rdfDestroySchema(rdf);
    g_free(file);
    return NULL;
  }
  desc = rdfFirstDescription(rdf);
  if (desc == NULL) {
    g_warning("%s RDF schema seems empty", file);
    rdfDestroySchema(rdf);
    g_free(file);
    return NULL;
  }

  check = g_hash_table_new(g_str_hash, g_str_equal);
  while (desc != NULL) {
    char *name;

    rdfGetValue(desc, "Name", rpmNs, &name, NULL);
    if (!g_hash_table_lookup(check, name)) {
      char *key = g_strdup(name);

      if (tail) {
	g_list_append(tail, key);
	tail = g_list_last(tail);
      } else
	allPackages = tail = g_list_append(NULL, key);
      g_hash_table_insert(check, key, GINT_TO_POINTER(TRUE));
    }
    if (name) free(name);
    desc = rdfNextDescription(desc);
  }
  g_list_sort(allPackages, (GCompareFunc)g_strcasecmp);
  g_hash_table_destroy(check);
  return allPackages;
}

GList *aproposSearch(const char *search) {
  GList *ret = NULL, *tail1 = NULL, *tail2 = NULL;
  rdfSchema rdf;
  rdfNamespace rdfNs, rpmNs;
  rdfDescription desc;
  gboolean buildPackages = FALSE;
  GHashTable *check = NULL;
#ifdef HAVE_REGEX_H
  regex_t reg;
  regmatch_t pmatch;
#endif
  char *file = update_file("resources/fullIndex.rdf.gz");

  if (!allPackages)
    buildPackages = TRUE;

  g_message("Searching for '%s'", search);
  rdf = rdfRead(file);
  if (rdf == NULL) {
    g_warning("can't open catalog '%s'", file);
    g_free(file);
    return NULL;
  }
  rdfNs = rdfGetNamespace(rdf, "http://www.w3.org/TR/WD-rdf-syntax#");
  if (rdfNs == NULL) {
    g_warning("%s is not an RDF schema", file);
    rdfDestroySchema(rdf);
    g_free(file);
    return NULL;
  }
  rpmNs = rdfGetNamespace(rdf, "http://www.rpm.org/");
  if (rpmNs == NULL) {
    g_warning("%s is not an RPM specific RDF schema", file);
    rdfDestroySchema(rdf);
    g_free(file);
    return NULL;
  }
  desc = rdfFirstDescription(rdf);
  if (desc == NULL) {
    g_warning("%s RDF schema seems empty", file);
    rdfDestroySchema(rdf);
    g_free(file);
    return NULL;
  }

#ifdef HAVE_REGEX_H
#ifdef REG_ICASE
# ifdef REG_NOSUB
  regcomp(&reg, search, REG_ICASE|REG_NOSUB);
# else
  regcomp(&reg, search, REG_ICASE);
# endif
#else
# ifdef REG_NOSUB
  regcomp(&reg, search, REG_NOSUB);
# else
  regcomp(&reg, search, 0);
# endif
#endif
#endif

  if (buildPackages)
    check = g_hash_table_new(g_str_hash, g_str_equal);

  while (desc != NULL) {
    char *URL, *name, *description;

    URL = rdfGetDescriptionAbout(rdf, desc);
    if (URL == NULL) {
      g_warning("%s RDF schema invalid: description without href", file);
      rdfDestroySchema(rdf);
      g_free(file);
      return NULL;
    }
    rdfGetValue(desc, "Name", rpmNs, &name, NULL);
    rdfGetValue(desc, "Summary", rpmNs, &description, NULL);

    if (buildPackages)
      if (!g_hash_table_lookup(check, name)) {
	char *key = g_strdup(name);

	if (tail1) {
	  g_list_append(tail1, key);
	  tail1 = g_list_last(tail1);
	} else
	  allPackages = tail1 = g_list_append(NULL, key);
	g_hash_table_insert(check, key, GINT_TO_POINTER(TRUE));
      }

#ifdef HAVE_REGEX_H
    if ((name && !regexec(&reg, name, 1, &pmatch, 0)) ||
	(description && !regexec(&reg, description, 1, &pmatch, 0)))
#elif defined(HAVE_STRSTR)
    if ((name && strstr(name, search)) ||
	(description && strstr(description, search)))
#else
#error "You need regex code or strstr for this to work"
#endif
      if (!g_list_find_custom(ret, name, (GCompareFunc)g_strcasecmp)) {
	if (tail2) {
	  g_list_append(tail2, g_strdup(name));
	  tail2 = g_list_last(tail2);
	} else
	  ret = tail2 = g_list_append(NULL, g_strdup(name));
      }
    free(URL);
    if (name) free(name);
    if (description) free(description);

    if (libfind_idle_func)
      (* libfind_idle_func)();

    desc = rdfNextDescription(desc);
  }
  rdfDestroySchema(rdf);

  if (buildPackages) {
    g_hash_table_destroy(check);
    g_list_sort(allPackages, (GCompareFunc)g_strcasecmp);
  }

  g_free(file);
#ifdef HAVE_REGEX_H
  regfree(&reg);
#endif

  ret = g_list_sort(ret, (GCompareFunc)g_strcasecmp);

  return ret;
}

void freeApropos(GList *apropos) {
  g_list_foreach(apropos, (GFunc)g_free, NULL);
  g_list_free(apropos);
}

void freeRpmPackageAlternate(rpmPackageAlternate *alt) {
  if (alt->name) g_free(alt->name);
  if (alt->version) g_free(alt->version);
  if (alt->release) g_free(alt->release);
  if (alt->arch) g_free(alt->arch);
  if (alt->os) g_free(alt->os);
  if (alt->distribution) g_free(alt->distribution);
  if (alt->vendor) g_free(alt->vendor);
  if (alt->subdir) g_free(alt->subdir);
  if (alt->key) g_free(alt->key);
  g_free(alt);
}

static gboolean alternateCompatible(rpmPackageAlternate *alt) {
  gboolean sources=gnome_config_get_bool("/gnorpm/rpmfind/wantSources=false");

  if (sources) {
    if (!g_strcasecmp(alt->arch, "src"))
      return TRUE;
    else
      return FALSE;
  }
  /* nparch packages are always valid */
  if (!g_strcasecmp(alt->arch, "noarch"))
    return TRUE;
  if (g_strcasecmp(alt->os, guessOs()))
    return FALSE;
  if (!g_strcasecmp(alt->arch, guessArch()))
    return TRUE;
  return FALSE;
}

GList *alternateSearch(const char *resource) {
  char *tmp = g_strdup_printf("resources/%s.rdf", resource);
  char *file = url_get_to_temp(tmp);
  rdfSchema rdf;
  rdfNamespace rdfNs, rpmNs;
  rdfDescription desc;
  GList *ret = NULL;

  g_free(tmp);
  if (file == NULL) {
    g_warning("couldn't get info for resource %s", resource);
    return NULL;
  }
  rdf = rdfRead(file);
  unlink(file);
  g_free(file);
  if (rdf == NULL)
    return NULL;

  rdfNs = rdfGetNamespace(rdf, "http://www.w3.org/TR/WD-rdf-syntax#");
  if (rdfNs == NULL) {
    g_warning("%s not an RDF schema", resource);
    rdfDestroySchema(rdf);
    return NULL;
  }
  rpmNs = rdfGetNamespace(rdf, "http://www.rpm.org/");
  if (rpmNs == NULL) {
    g_warning("%s not an RPM specific RDF schema", resource);
    rdfDestroySchema(rdf);
    return NULL;
  }
  desc = rdfFirstDescription(rdf);
  if (desc == NULL) {
    g_warning("%s RDF schema seems empty", resource);
    rdfDestroySchema(rdf);
    return NULL;
  }
  while (desc != NULL) {
    char *URL, *value = rdfGetDescriptionHref(rdf, desc);
    rpmPackageAlternate *alt;

    if (value != NULL) URL = value;
    else {
      g_warning("%s RDF schema invalid: description without href", resource);
      rdfDestroySchema(rdf);
      freeAlternates(ret);
      return NULL;
    }
    
    alt = g_new0(rpmPackageAlternate, 1);
    rdfGetValue(desc, "Name", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->name = g_strdup(value);
      free(value);
    } else {
      g_warning("%s RDF schema invalid: no name", resource);
      rdfDestroySchema(rdf);
      freeRpmPackageAlternate(alt);
      freeAlternates(ret);
      return NULL;
    }
    rdfGetValue(desc, "Version", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->version = g_strdup(value);
      free(value);
    } else {
      g_warning("%s RDF schema invalid: no version", resource);
      rdfDestroySchema(rdf);
      freeRpmPackageAlternate(alt);
      freeAlternates(ret);
      return NULL;
    }
    rdfGetValue(desc, "Release", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->release = g_strdup(value);
      free(value);
    } else {
      g_warning("%s RDF schema invalid: no release", resource);
      rdfDestroySchema(rdf);
      freeRpmPackageAlternate(alt);
      freeAlternates(ret);
      return NULL;
    }
    rdfGetValue(desc, "Arch", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->arch = g_strdup(value);
      free(value);
    }
    rdfGetValue(desc, "Os", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->os = g_strdup(value);
      free(value);
    }
    rdfGetValue(desc, "Distribution", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->distribution = g_strdup(value);
      free(value);
    }
    rdfGetValue(desc, "Vendor", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->vendor = g_strdup(value);
      free(value);
    }
    rdfGetValue(desc, "Date", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->date = strtol(value, NULL, 0);
      free(value);
    }
    rdfGetValue(desc, "Size", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->size = strtol(value, NULL, 0);
      free(value);
    }
    rdfGetValue(desc, "Subdir", rpmNs, &value, NULL);
    if (value != NULL) {
      alt->subdir = g_strdup(value);
      free(value);
    }

    /* a unique key for this alternate record */
    alt->key = g_strdup_printf("%s/%s-%s-%s", alt->subdir, alt->name,
				alt->version, alt->release);

    /* only list package if we could actualy install it on this system */
    if (alternateCompatible(alt))
      ret = g_list_append(ret, alt);
    else
      freeRpmPackageAlternate(alt);

    desc = rdfNextDescription(desc);
  }
  return ret;
}

/* generate a number based on a version number string that can be compared
 * against other such numbers to give a reasonable idea  of which package
 * is newer */
static guint normaliseVersion(const char *version) {
  gchar **components = g_strsplit(version, ".", 3);
  guint ret = 0;

  if (components[0] != NULL) {
    ret += strtol(components[0], NULL, 0) * 1000000;
    if (components[1] != NULL) {
      ret += strtol(components[1], NULL, 0) * 1000;
      if (components[2] != NULL)
	ret += strtol(components[2], NULL, 0);
    }
  }
  g_strfreev(components);
  return ret;
}

#define HIGH_RATING 1000
#define MED_RATING   500
#define LOW_RATING   100

guint alternateScore(rpmPackageAlternate *alt,
		     const char *preferredDist,
		     const char *preferredVendor) {
  gboolean version_only;
  guint score = 0;
  gint distrib_rating;

  distrib_rating = distribInfoGet(alt->subdir)->rating;
  if (distrib_rating < 0) return 0;
  score += distrib_rating;
  version_only =
    gnome_config_get_bool("/gnorpm/rpmfind/wantLatestVersion=false");
  if (version_only)
    return normaliseVersion(alt->version);
  else
    score += normaliseVersion(alt->version) / 1000;
  if (preferredDist && preferredVendor) {
    if (!strcmp(alt->vendor, preferredVendor)) {
      score += MED_RATING;
      if (!strcmp(alt->distribution, preferredDist))
	score += HIGH_RATING;
    }
    if (!strcmp(alt->vendor, guessVendor())) {
      score += LOW_RATING;
      if (!strcmp(alt->distribution, guessDistribution()))
	score += MED_RATING;
    }
  } else
    if (!strcmp(alt->vendor, guessVendor())) {
      score += MED_RATING;
      if (!strcmp(alt->distribution, guessDistribution()))
	score += HIGH_RATING;
    }

  /* maybe put the time check in here.  Is it really necessary? */
  /* file size weighting towards small rpms goes in here */

  return score;
}

/* XXX not thread safe, but how do I get the data in there? */
static struct _preferredInfo {
  const char *dist, *vendor;
} info;

static int alternate_compare(rpmPackageAlternate *a, rpmPackageAlternate *b) {
  gint score_a = alternateScore(a, info.dist, info.vendor);
  gint score_b = alternateScore(b, info.dist, info.vendor);

  if (score_a == score_b) return 0;
  if (score_a > score_b) return -1;
  return 1;
}

GList *sortAlternates(GList *alternates,
		      const char *preferredDist,
		      const char *preferredVendor) {
  info.dist = preferredDist;
  info.vendor = preferredVendor;
  return g_list_sort(alternates, (GCompareFunc)alternate_compare);
}

void freeAlternates(GList *alternates) {
  g_list_foreach(alternates, (GFunc)freeRpmPackageAlternate, NULL);
  g_list_free(alternates);
}

void listAlternates(GList *alternates) {
  GList *tmp;

  g_print("Guess info:\nvendor=%s dist=%s os=%s arch=%s\n",
	  guessVendor(), guessDistribution(), guessOs(), guessArch());

  g_print("Number of alternates: %d\n", g_list_length(alternates));
  for (tmp = alternates; tmp; tmp = tmp->next) {
    rpmPackageAlternate *alt = tmp->data;
    g_print("%s-%s-%s.%s.%s.rpm  Score: %d\n", alt->name, alt->version,
	    alt->release, alt->arch, alt->os,
	    alternateScore(alt, NULL, NULL));
    g_print("  %s/%s  Subdir: %s\n", alt->vendor, alt->distribution,
	    alt->subdir);
    g_print("  size=%d date=%s\n", alt->size, ctime(&(alt->date)));
  }
}

rpmData *alternateGetInfo(rpmPackageAlternate *alt) {
#ifdef RPM_DATA_CACHE
  static GHashTable *cache = NULL;
  rpmData *ret;

  if (!cache)
    cache = g_hash_table_new(g_str_hash, g_str_equal);
  ret = g_hash_table_lookup(cache, alt->key);
  if (!ret) {
    ret = rpmOpenPackage(alt->subdir, alt->name, alt->version,
			 alt->release, alt->arch);
    ret->nofree = TRUE;
    g_hash_table_insert(cache, g_strdup(alt->key), ret);
  }
  return ret;
#else
  return rpmOpenPackage(alt->subdir, alt->name, alt->version,
			alt->release, alt->arch);
#endif
}
