/* GnoRPM - A GNOME front end for the Redhat Package Manager (RPM)
 * Copyright (C) 1998-1999  James Henstridge
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <gnome.h>

#include <rpmlib.h>

#include "rpmprogress.h"

#include "install.h"
#include "misc.h"

/* I used the rpm source code as a template for these functions, so I just
 * thought I would say that portions of this are probably Copyright Redhat
 */

/* what these routines will assume the size of old source packages is */
#define UNKNOWN_SIZE 1024

static RpmProgress *prog;
static rpmInstallCb inst_cb;
static void *inst_cb_user_data;
  
static void updateDisp(gulong amount, gulong total) {
  rpm_progress_update(prog, (1.0 * amount / total));
}

static gint installPackages(gchar *rootdir, gchar **packages, gchar *location,
                       gint numPackages, gint totSize, gint installFlags,
                       gint interfaceFlags, rpmdb db) {
  gint i, fd, rc, curSize;
  gint numFailed = 0;
  gchar buf[512], *netsharedPath = NULL, *nameptr, *groupName;
  Header head;
  GtkWidget *win, *wid;
  unsigned long *sizep;

  netsharedPath = rpmGetVar(RPMVAR_NETSHAREDPATH);

  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  set_icon(win);
  gtk_widget_set_usize(win, 350, -1);
  gtk_container_set_border_width(GTK_CONTAINER(win), 10);
  if (installFlags & RPMINSTALL_UPGRADE)
    gtk_window_set_title(GTK_WINDOW(win), _("Upgrading"));
  else
    gtk_window_set_title(GTK_WINDOW(win), _("Installing"));

  wid = rpm_progress_new(numPackages, totSize);
  gtk_container_add(GTK_CONTAINER(win), wid);
  gtk_widget_show(wid);

  prog = RPM_PROGRESS(wid);
  gtk_widget_show(win);

  for (i = 0; i < numPackages; i++) {
    if (!packages[i]) continue;
    nameptr = strrchr(packages[i], '/');
    if (!nameptr) nameptr = packages[i];
    else nameptr++;
    groupName = NULL;

    /* install */
    fd = open(packages[i], O_RDONLY);
    if (fd < 0) {
      g_snprintf(buf, 511, _("error: cannot open file %s"), packages[i]);
      message_box(buf);
      numFailed++;
      packages[i] = NULL;
      continue;
    }
    if (!rpmReadPackageHeader(fd, &head, NULL, NULL, NULL)) {
      if (head) {
	headerGetEntry(head, RPMTAG_SIZE, NULL, (void **)&sizep, NULL);
	curSize = sizep[0];
	headerGetEntry(head, RPMTAG_GROUP, NULL, (void **)&groupName, NULL);
	groupName = g_strdup(groupName);
	headerFree(head);
      } else
	curSize = UNKNOWN_SIZE;
    } else
      curSize = UNKNOWN_SIZE;
    lseek(fd, 0, SEEK_SET);
    rpm_progress_next(prog, nameptr, curSize);

    if (db) {
#ifdef HAVE_RPM_2_5
      rc = rpmInstallPackage(rootdir, db, fd, NULL, installFlags,
			     updateDisp, NULL);
#else
      rc = rpmInstallPackage(rootdir, db, fd, location, installFlags,
                             updateDisp, NULL, netsharedPath);
#endif
      if (!rc && inst_cb)
	(* inst_cb)(nameptr, groupName, inst_cb_user_data);
    } else if (installFlags & RPMINSTALL_TEST)
        rc = 0;
    else
        rc = rpmInstallSourcePackage(rootdir,fd,NULL,updateDisp,NULL,NULL);
    if (rc) {
      numFailed++;
    }
    close(fd);
    if (groupName) g_free(groupName);
  }
  gtk_widget_destroy(win);
  return numFailed;
}

/* returns TRUE if yes was pressed */
static gboolean printDepProblems(struct rpmDependencyConflict *conflicts,
                            int numConflicts) {
  GtkWidget *win, *w, *list;
  gchar buf[1024];
  gint i, flags, ret = FALSE;
  GList *items = NULL;

  win = gnome_dialog_new(_("Dependency Problems"), GNOME_STOCK_BUTTON_YES,
			 GNOME_STOCK_BUTTON_NO, NULL);
  gnome_dialog_close_hides(GNOME_DIALOG(win), FALSE);
  gtk_window_set_policy(GTK_WINDOW(win), FALSE, TRUE, TRUE);
  set_icon(win);
  gtk_widget_set_usize(win, 300, -1);
  gnome_dialog_set_default(GNOME_DIALOG(win), 1);
  gnome_dialog_set_close(GNOME_DIALOG(win), TRUE);

  w = gtk_label_new(_("The following dependency problems occured:"));
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(win)->vbox), w, FALSE, TRUE, 0);
  gtk_widget_show(w);
  for (i = 0; i < numConflicts; i++) {
    buf[0] = '\0';
    strcat(buf, conflicts[i].needsName);
    flags = conflicts[i].needsFlags;
    if (flags) {
      strcat(buf, " ");
      if (flags & RPMSENSE_LESS)
        strcat(buf, "<");
      if (flags & RPMSENSE_GREATER)
        strcat(buf, ">");
      if (flags & RPMSENSE_EQUAL)
        strcat(buf, "=");
      if (flags & RPMSENSE_SERIAL)
        strcat(buf, "S");
      strcat(buf, " ");
      strcat(buf, conflicts[i].needsVersion);
    }
    if (conflicts[i].sense == RPMDEP_SENSE_REQUIRES)
      strcat(buf, _(" is needed by "));
    else
      strcat(buf, _(" conflicts with "));
    strcat(buf, conflicts[i].byName);    strcat(buf, "-");
    strcat(buf, conflicts[i].byVersion); strcat(buf, "-");
    strcat(buf, conflicts[i].byRelease);
    w = gtk_list_item_new_with_label(buf);
    gtk_widget_show(w);
    items = g_list_append(items, w);
  }
  w = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(w, 320, 120);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(w), GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(win)->vbox), w, TRUE, TRUE, 0);
  gtk_widget_show(w);
  list = gtk_list_new();
  gtk_list_set_selection_mode(GTK_LIST(list), GTK_SELECTION_BROWSE);
  gtk_list_append_items(GTK_LIST(list), items);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(w), list);
  gtk_widget_show(list);

  w = gtk_label_new(_("Do you want to ignore these problems?\n"
		      "(saying yes may make your system unstable)"));
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(win)->vbox), w, FALSE, TRUE, 0);
  gtk_widget_show(w);

  gtk_widget_show(win);
  ret = gnome_dialog_run_and_close(GNOME_DIALOG(win));
  return !ret;
}

int do_install(gchar *rootdir, GList *pkgs, gchar *location,
	       gint installFlags, gint interfaceFlags,
	       rpmInstallCb cb, void *user_data) {
  gint mode, rc, i, fd, isSource, stopInstall = 0, pkgc;
  gint numPackages, numBinaryPackages = 0;
  gint numSourcePackages = 0, numFailed = 0;
  gint numConflicts, totSize = 0;
  gchar **packages, *pkg_file;
  Header *binaryHeaders;
  rpmdb db;
  rpmDependencies rpmdep;
  struct rpmDependencyConflict *conflicts;
  gulong *sizep;

  if (installFlags & RPMINSTALL_TEST)
    mode = O_RDONLY;
  else
    mode = O_RDWR | O_CREAT;

  inst_cb = cb;
  inst_cb_user_data = user_data;

  pkgc = g_list_length(pkgs);
  packages = g_new(gchar *, pkgc + 1);
  binaryHeaders = g_new(Header, pkgc + 1);
  for (numPackages = 0; pkgs != NULL; pkgs = pkgs->next) {
    pkg_file = pkgs->data;
    fd = open(pkg_file, O_RDONLY);
    if (fd < 0) {
      numFailed++;
      continue;
    }
    packages[numPackages++] = pkg_file;
    rc = rpmReadPackageHeader(fd, &binaryHeaders[numBinaryPackages],
                              &isSource, NULL, NULL);
    close(fd);
    if (binaryHeaders[numBinaryPackages]) {
      if (headerGetEntry(binaryHeaders[numBinaryPackages], RPMTAG_SIZE,
			 NULL, (void **) &sizep, NULL))
	totSize += *sizep;
    } else
      totSize += UNKNOWN_SIZE;
    if (rc) {
      numPackages--;
      numFailed++;
    } else if (isSource) {
      if (binaryHeaders[numBinaryPackages])
        headerFree(binaryHeaders[numBinaryPackages]);
      numSourcePackages++;
    } else
      numBinaryPackages++;
  }
  if (numBinaryPackages) {
    if (rpmdbOpen(rootdir, &db, mode, 0644)) {
      for (i=0; i < numBinaryPackages; i++)
        headerFree(binaryHeaders[i]);
      g_free(binaryHeaders);
      g_free(packages);
      return numPackages;
    }
    rpmdep = rpmdepDependencies(db);
    for (i =0; i < numBinaryPackages; i++)
      if (installFlags & RPMINSTALL_UPGRADE)
        rpmdepUpgradePackage(rpmdep, binaryHeaders[i], packages[i]);
      else
        rpmdepAddPackage(rpmdep, binaryHeaders[i], packages[i]);
    if (!(interfaceFlags & INTER_NODEPS)) {
      if (rpmdepCheck(rpmdep, &conflicts, &numConflicts)) {
        numFailed = numPackages;
        message_box(_("Dependency check failed."));
        stopInstall = 1;
      }
      if (!stopInstall && conflicts) {
        if (!printDepProblems(conflicts, numConflicts)) {
	  numFailed = numPackages;
	  stopInstall = 1;
	}
	rpmdepFreeConflicts(conflicts, numConflicts);
      }
    }
    if (!(interfaceFlags & INTER_NOORDER))
      if (rpmdepOrder(rpmdep, (void ***) &packages)) {
        numFailed = numPackages;
        stopInstall = 1;
      }
    rpmdepDone(rpmdep);
  } else
    db = NULL;
  if (!stopInstall) {
    /* do the actual install */
    numFailed += installPackages(rootdir, packages, location, numPackages,
                                 totSize, installFlags, interfaceFlags, db);
  }
  for (i = 0; i < numBinaryPackages; i++)
    headerFree(binaryHeaders[i]);
  g_free(binaryHeaders);
  if (db) rpmdbClose(db);
  g_free(packages);

  return numFailed;
}

static gboolean continue_remove(rpmdb db, GList *indices) {
  GList *tmp, *items = NULL;
  gchar *s1, *s2, *s3, buf[512];
  GtkWidget *win, *box, *w, *list;
  gboolean ret = FALSE;

  win = gnome_dialog_new(_("Continue Removal"), GNOME_STOCK_BUTTON_YES,
			 GNOME_STOCK_BUTTON_NO, NULL);
  gnome_dialog_close_hides(GNOME_DIALOG(win), FALSE);
  gtk_window_set_policy(GTK_WINDOW(win), FALSE, TRUE, TRUE);
  set_icon(win);
  gnome_dialog_set_default(GNOME_DIALOG(win), 0);
  gnome_dialog_set_close(GNOME_DIALOG(win), TRUE);

  box = GNOME_DIALOG(win)->vbox;
  w = gtk_label_new(_("Remove the following packages?"));
  gtk_box_pack_start(GTK_BOX(box), w, FALSE, TRUE, 0);
  gtk_widget_show(w);
  for (tmp = indices; tmp; tmp = tmp->next) {
    guint32 index = GPOINTER_TO_UINT(tmp->data);
    Header h = rpmdbGetRecord(db, index);
    GtkWidget *item;

    headerGetEntry(h, RPMTAG_NAME, NULL, (void**)&s1, NULL);
    headerGetEntry(h, RPMTAG_VERSION, NULL, (void**)&s2, NULL);
    headerGetEntry(h, RPMTAG_RELEASE, NULL, (void**)&s3, NULL);
    g_snprintf(buf, 511, "%s-%s-%s", s1, s2, s3);
    item = gtk_list_item_new_with_label(buf);
    gtk_widget_show(item);
    items = g_list_append(items, item);

    headerFree(h);
  }
  w = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(w, 320, 120);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(w), GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(box), w, TRUE, TRUE, 0);
  gtk_widget_show(w);
  list = gtk_list_new();
  gtk_list_set_selection_mode(GTK_LIST(list), GTK_SELECTION_BROWSE);
  gtk_list_append_items(GTK_LIST(list), items);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(w), list);
  gtk_widget_show(list);

  gtk_widget_show(win);
  ret = gnome_dialog_run_and_close(GNOME_DIALOG(win));
  return !ret;
}

gint do_uninstall(gchar *root, GList *indices, gint uninstallFlags,
		 gint interfaceFlags) {
  rpmdb db;
  gint mode, numFailed = 0, numConflicts, stopUninstall = 0;
  rpmDependencies rpmdep;
  struct rpmDependencyConflict *conflicts;
  GList *tmp;

  if (uninstallFlags &RPMUNINSTALL_TEST)
    mode = O_RDONLY;
  else
    mode = O_RDWR | O_EXCL;
  if (rpmdbOpen(root, &db, mode, 0644)) {
    return g_list_length(indices);
  }
  if (!continue_remove(db, indices)) return 0;
  if (!(interfaceFlags & INTER_NODEPS)) {
    rpmdep = rpmdepDependencies(db);
    for (tmp = indices; tmp != NULL; tmp = tmp->next)
      rpmdepRemovePackage(rpmdep, GPOINTER_TO_UINT(tmp->data));
    if (rpmdepCheck(rpmdep, &conflicts, &numConflicts)) {
      numFailed = g_list_length(indices);
      stopUninstall = 1;
    }
    rpmdepDone(rpmdep);
    if (!stopUninstall && conflicts) {
      if (!printDepProblems(conflicts, numConflicts)) {
	numFailed += g_list_length(indices);
	stopUninstall = 1;
      }
      rpmdepFreeConflicts(conflicts, numConflicts);
    }
  }
  if (!stopUninstall) {
    for (tmp = indices; tmp != NULL; tmp = tmp->next)
      if (rpmRemovePackage(root, db, GPOINTER_TO_UINT(tmp->data),
			   uninstallFlags))
	numFailed++;
  }
  rpmdbClose(db);
  return numFailed;
}

gint install_one(gchar *root, gchar *file, gchar *location,
		 gint installFlags, gint interfaceFlags,
		 rpmInstallCb cb, void *user_data) {
  GList *list;
  gint result;

  list = g_list_append(NULL, file);
  result = do_install(root, list, location, installFlags, interfaceFlags,
		      cb, user_data);
  g_list_free(list);
  return result;
}

gint uninstall_one(gchar *root, guint index, gint uninstallFlags,
		  gint interfaceFlags) {
  GList *list;
  gint result;
  list = g_list_append(NULL, GUINT_TO_POINTER(index));
  result = do_uninstall(root, list, uninstallFlags, interfaceFlags);
  g_list_free(list);
  return result;
}

