/* GnoRPM - A GNOME front end for the Redhat Package Manager (RPM)
 * Copyright (C) 1998-1999  James Henstridge
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include "rpminstalldlg.h"
#include "rpmquerydlg.h"
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "misc.h"
#include "pixmaps.h"

enum {
  QUERY,
  INSTALL,
  CHECKSIG,
  LAST_SIGNAL
};
static guint installdlg_signals[LAST_SIGNAL];

static GtkObjectClass *parent_class = NULL;

enum { TARGET_URI_LIST };
static GtkTargetEntry drop_types[] = {
  { "text/uri-list", 0, TARGET_URI_LIST}
};
static gint n_drop_types = sizeof(drop_types) / sizeof(drop_types[0]);

static void rpm_install_dialog_class_init(RpmInstallDialogClass *klass);
static void rpm_install_dialog_init(RpmInstallDialog *dlg);

static void rpm_install_dialog_show_filesel(RpmInstallDialog *dlg);
static void rpm_install_dialog_rem_item(RpmInstallDialog *dlg);
static void rpm_install_dialog_destroy(GtkObject *obj);

static void rpm_install_dialog_query(RpmInstallDialog *dlg);
static void rpm_install_dialog_install(RpmInstallDialog *dlg);
static void rpm_install_dialog_checksig(RpmInstallDialog *dlg);
static void rpm_install_dialog_marshal_signal(GtkObject *object,
					      GtkSignalFunc func,
					      gpointer data, GtkArg *args);
static void rpm_install_dialog_drop_cb(RpmInstallDialog *dlg,
				       GdkDragContext *context,
				       gint x, gint y,
				       GtkSelectionData *selection_data,
				       guint info, guint time);
					      
guint rpm_install_dialog_get_type(void) {
  static guint instdlg_type = 0;
  if (!instdlg_type) {
    GtkTypeInfo instdlg_info = {
      "RpmInstallDialog",
      sizeof(RpmInstallDialog),
      sizeof(RpmInstallDialogClass),
      (GtkClassInitFunc) rpm_install_dialog_class_init,
      (GtkObjectInitFunc) rpm_install_dialog_init,
      (GtkArgSetFunc) NULL,
      (GtkArgGetFunc) NULL,
    };
    instdlg_type = gtk_type_unique(gtk_dialog_get_type(), &instdlg_info);
  }
  return instdlg_type;
}

static void rpm_install_dialog_class_init(RpmInstallDialogClass *klass) {
  GtkObjectClass *object_class;

  object_class = GTK_OBJECT_CLASS(klass);
  parent_class = gtk_type_class(gtk_dialog_get_type());

  installdlg_signals[QUERY] =
    gtk_signal_new("query",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(RpmInstallDialogClass, query),
		   rpm_install_dialog_marshal_signal,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);
  installdlg_signals[INSTALL] =
    gtk_signal_new("install",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(RpmInstallDialogClass, install),
		   rpm_install_dialog_marshal_signal,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);
  installdlg_signals[CHECKSIG] =
    gtk_signal_new("checksig",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(RpmInstallDialogClass, checksig),
		   rpm_install_dialog_marshal_signal,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_POINTER);
  gtk_object_class_add_signals(object_class, installdlg_signals, LAST_SIGNAL);

  object_class->destroy = rpm_install_dialog_destroy;
}

static void rpm_install_dialog_init(RpmInstallDialog *self) {
  GtkWidget *hbox, *bbox, *button, *wid;

  /* set up drag and drop thingy */
  gtk_drag_dest_set(GTK_WIDGET(self),
		    GTK_DEST_DEFAULT_MOTION |
		    GTK_DEST_DEFAULT_HIGHLIGHT |
		    GTK_DEST_DEFAULT_DROP,
		    drop_types, n_drop_types,
		    GDK_ACTION_COPY);
  gtk_signal_connect(GTK_OBJECT(self), "drag_data_received",
		     GTK_SIGNAL_FUNC(rpm_install_dialog_drop_cb), NULL);

  set_icon(GTK_WIDGET(self));
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(self)->vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show(hbox);

  wid = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(wid, 200, 150);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(wid),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(hbox), wid, TRUE, TRUE, 0);
  gtk_widget_show(wid);

  self->list = gtk_list_new();
  gtk_list_set_selection_mode(GTK_LIST(self->list), GTK_SELECTION_MULTIPLE);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(wid), self->list);
  gtk_widget_show(self->list);

  bbox = gtk_vbutton_box_new();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_START);
  gtk_box_pack_start(GTK_BOX(hbox), bbox, FALSE, TRUE, 0);
  gtk_widget_show(bbox);

  button = gtk_button_new_with_label(_("Add"));
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(rpm_install_dialog_show_filesel),
			    GTK_OBJECT(self));
  gtk_container_add(GTK_CONTAINER(bbox), button);
  gtk_widget_show(button);

  button = gtk_button_new_with_label(_("Remove\nfrom List"));
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(rpm_install_dialog_rem_item),
			    GTK_OBJECT(self));
  gtk_container_add(GTK_CONTAINER(bbox), button);
  gtk_widget_show(button);

  hbox = GTK_DIALOG(self)->action_area;
  button = gnome_pixmap_button(gnome_stock_pixmap_widget(NULL,
				RPM_STOCK_PIXMAP_QUERY), _("Query"));
  gtk_widget_set_usize(button, 100 /* GNOME_BUTTON_WIDTH */, -1);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(rpm_install_dialog_query),
			    GTK_OBJECT(self));
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  gtk_widget_show(button);

  button = gnome_pixmap_button(gnome_stock_pixmap_widget(NULL,
				RPM_STOCK_PIXMAP_INSTALL), _("Install"));
  gtk_widget_set_usize(button, 100 /* GNOME_BUTTON_WIDTH */, -1);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(rpm_install_dialog_install),
			    GTK_OBJECT(self));
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  gtk_widget_show(button);

  button = gnome_pixmap_button(gnome_stock_pixmap_widget(NULL,
				RPM_STOCK_PIXMAP_VERIFY), _("Check Sig"));
  gtk_widget_set_usize(button, 100 /* GNOME_BUTTON_WIDTH */, -1);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(rpm_install_dialog_checksig),
			    GTK_OBJECT(self));
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  gtk_widget_show(button);

  button = gnome_stock_button(GNOME_STOCK_BUTTON_CLOSE);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(self));
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
  gtk_widget_show(button);

  self->filesel = NULL;
  self->hdl = NULL;
}

GtkWidget *rpm_install_dialog_new(DBHandle *hdl) {
  RpmInstallDialog *self = gtk_type_new(rpm_install_dialog_get_type());

  self->hdl = hdl;
  return GTK_WIDGET(self);
}

void rpm_install_dialog_add_file(RpmInstallDialog *self, gchar *fname) {
  GtkWidget *item;
  GList *tmp;
  gchar buf[512], *s1, *s2, *s3;
  int fd, isSource;
  Header h;

  fd = open(fname, O_RDONLY);
  if (fd < 0) {
    message_box(_("Can't open file"));
    return;
  }
  if (rpmReadPackageHeader(fd, &h, &isSource, NULL, NULL)) {
    s1 = strrchr(fname, '/');
    s1++;
    g_snprintf(buf, 511, _("%s doesn't appear to be a RPM package"), s1);
    message_box(buf);
    return;
  }
  if (!h && isSource) {
    s1 = strrchr(fname, '/');
    s1++;
    g_snprintf(buf, 511, "%s (S)", s1);
  } else {
    headerGetEntry(h, RPMTAG_NAME,    NULL, (void**)&s1, NULL);
    headerGetEntry(h, RPMTAG_VERSION, NULL, (void**)&s2, NULL);
    headerGetEntry(h, RPMTAG_RELEASE, NULL, (void**)&s3, NULL);
    g_snprintf(buf, 511, "%s-%s-%s%s", s1, s2, s3, isSource?" (S)":"");
    headerFree(h);
  }
  item = gtk_list_item_new_with_label(buf);
  gtk_object_set_data_full(GTK_OBJECT(item), "filename", g_strdup(fname),
			   (GtkDestroyNotify) g_free);
  gtk_widget_show(item);
  tmp = g_list_prepend(NULL, item);
  gtk_list_append_items(GTK_LIST(self->list), tmp);
  gtk_list_select_child(GTK_LIST(self->list), item);
}

static void rpm_install_dialog_drop_cb(RpmInstallDialog *self,
				       GdkDragContext *context,
				       gint x, gint y,
				       GtkSelectionData *selection_data,
				       guint info, guint time) {
  GList *names, *list;

  switch (info) {
  case TARGET_URI_LIST:
    list = gnome_uri_list_extract_filenames(selection_data->data);
    for (names = list; names; names = names->next)
      rpm_install_dialog_add_file(self, names->data);
    gnome_uri_list_free_strings(list);
    break;
  default:
  }
}

static void rpm_install_dialog_add(RpmInstallDialog *self) {
  gchar *fname;

  fname = gtk_file_selection_get_filename(GTK_FILE_SELECTION(self->filesel));
  rpm_install_dialog_add_file(self, fname);
}

static void filesel_destroy(RpmInstallDialog *self) {
  self->filesel = NULL;
}
static void rpm_install_dialog_show_filesel(RpmInstallDialog *self) {
  GtkWidget *button, *label;
  gchar *string, *dir;

  if (self->filesel != NULL) {
    gdk_window_raise(self->filesel->window);
    return;
  }
  self->filesel = gtk_file_selection_new(_("Add Packages"));
  gtk_signal_connect_object(GTK_OBJECT(self->filesel), "destroy",
			    GTK_SIGNAL_FUNC(filesel_destroy),GTK_OBJECT(self));
  /* security, if for some reason you want to run this SUID */
  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(self->filesel));

  button = GTK_FILE_SELECTION(self->filesel)->ok_button;
  label = GTK_BUTTON(button)->child;
  gtk_label_set(GTK_LABEL(label), _("Add"));
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(rpm_install_dialog_add),
			    GTK_OBJECT(self));

  button = GTK_FILE_SELECTION(self->filesel)->cancel_button;
  label = GTK_BUTTON(button)->child;
  gtk_label_set(GTK_LABEL(label), _("Close"));
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(self->filesel));

  string = gnome_config_get_string(
			"/gnorpm/Paths/rpmDir=/mnt/cdrom/RedHat/RPMS");
  if (string[strlen(string) - 1] != PATH_SEP) {
    dir = g_copy_strings(string, PATH_SEP_STR, NULL);
    g_free(string);
  } else
    dir = string;
  if (g_file_exists(dir))
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(self->filesel),
				    dir);
  g_free(dir);

  gtk_widget_show(self->filesel);
}

static void rpm_install_dialog_rem_item(RpmInstallDialog *self) {
  GList *sel, *tmp_list = NULL;

  for (sel = GTK_LIST(self->list)->selection; sel != NULL; sel = sel->next)
    tmp_list = g_list_prepend(tmp_list, sel->data);
  gtk_list_remove_items(GTK_LIST(self->list), tmp_list);
}

static void rpm_install_dialog_query(RpmInstallDialog *self) {
  GList *selection, *files = NULL;

  selection = GTK_LIST(self->list)->selection;
  for (; selection != NULL; selection = selection->next)
    files = g_list_prepend(files, gtk_object_get_data(
				GTK_OBJECT(selection->data), "filename"));

  gtk_signal_emit(GTK_OBJECT(self), installdlg_signals[QUERY], files);
  g_list_free(files);
}

static void rpm_install_dialog_install(RpmInstallDialog *self) {
  GList *selection, *files = NULL;

  selection = GTK_LIST(self->list)->selection;
  for (; selection != NULL; selection = selection->next)
    files = g_list_prepend(files, gtk_object_get_data(
				GTK_OBJECT(selection->data), "filename"));

  gtk_signal_emit(GTK_OBJECT(self), installdlg_signals[INSTALL], files);
  g_list_free(files);
}

static void rpm_install_dialog_checksig(RpmInstallDialog *self) {
  GList *selection, *files = NULL;

  selection = GTK_LIST(self->list)->selection;
  for (; selection != NULL; selection = selection->next)
    files = g_list_prepend(files, gtk_object_get_data(
				GTK_OBJECT(selection->data), "filename"));

  gtk_signal_emit(GTK_OBJECT(self), installdlg_signals[CHECKSIG], files);
  g_list_free(files);
}

static void rpm_install_dialog_destroy(GtkObject *object) {
  RpmInstallDialog *self = RPM_INSTALL_DIALOG(object);

  if (self->filesel) gtk_widget_destroy(self->filesel);
  if (parent_class->destroy)
    (* parent_class->destroy)(object);
}

static void rpm_install_dialog_marshal_signal(GtkObject *object,
					      GtkSignalFunc func,
					      gpointer data, GtkArg *args) {
  typedef void (*sig_func)(GtkObject *o, GList *files, gpointer data);
  sig_func rfunc = (sig_func)func;

  (*rfunc)(object, GTK_VALUE_POINTER(args[0]), data);
}
