/* brlmonui.c
 *
 * Copyright 2001, 2002 Sun Microsystems, Inc.,
 * Copyright 2001, 2002 BAUM Retec, A.G.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "brlmonui.h"
#include "brlmon.h"
#include <gdk/gdk.h>
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <gconf/gconf-value.h>
#include "SRMessages.h"
#include "srintl.h"
#include <glib.h>

#define STATUS_COUNT 	4

#define FIRST_POSITION 	0
#define INVALID_POSITION -1

#define FONT_COURIER   	"courier 14"
#define FONT_BRAILLE   	"braille 14"

#define CELL_WIDTH	20
#define CELL_HEIGHT	25
#define STATUS_WIDTH 	18

#define MIN_HEIGHT	40

#define CELL_TEXT_LENGTH 8

#define BRLMON_DOTNONE_GCONF_KEY "dotnone"
#define BRLMON_DOT7_GCONF_KEY 	 "dot7"
#define BRLMON_DOT8_GCONF_KEY 	 "dot8"
#define BRLMON_DOT78_GCONF_KEY 	 "dot78"

#define DEFAULT_DOTNONE_COLOR   "#000000000000"
#define DEFAULT_DOT78_COLOR	"#000000007D00"
#define DEFAULT_DOT7_COLOR	"#7D0000000000"
#define DEFAULT_DOT8_COLOR	"#00007D000000"


static gboolean brlmon_load_help_interface (void);
static gboolean	brlmon_load_brlmon_options (void);
static gboolean brlmon_save_brlmon_options (void);
static GladeXML* brlmon_load_glade_interface (const gchar *glade_file, 
		    			      const gchar *window);

/**
 *
 * BrlMon main window widget.
 *
**/
static GtkWidget *w_brlmon;

/**
 *
 * Table of horizontal boxes
 *
**/
static GtkWidget *hbox [PANEL_LENGHT];
static GtkWidget *hbox1;

/**
 *
 * Table of PANEL_LENGTH number frames
 *
**/
static GtkWidget *frame [PANEL_LENGHT];
static GtkWidget *frame_status [STATUS_COUNT];

static GtkWidget *label_status;
/**
 *
 * Tables of characters.
 *
**/
static GtkWidget *cell [PANEL_LENGHT];
static GtkWidget *cell_status [STATUS_COUNT];

static GtkWidget *ck_theme_color;
static GtkWidget *cp_dotnone;
static GtkWidget *cp_dot7;  
static GtkWidget *cp_dot78;
static GtkWidget *cp_dot8;

/**
 *
 * panel_size - number of characters
 *
**/
static gint panel_size;
static gint panel_pos;

/**
 *
 * cursor_pos - cursor position on display
 *
**/
static gint cursor_pos;

/**
 *
 * old_pos - last cursor displayed position
 *
**/
static gint old_pos;

/**
 *
 * brlmon_typedot - display dot type
 *
**/
static gint brlmon_typedot;

static gboolean brlmon_main_quit = FALSE;

/**
 *
 * Braille setting gconf client listener				
 *
**/
static GConfClient *brlmon_client = NULL;

gint brlmon_modetype = MODE_NORMAL;
static gboolean user_event = TRUE;
static gboolean brlmon_use_default_color = TRUE;
static GdkColor *colors;


/**
 *
 * Set dot type.
 *
**/
void 
brlmon_set_typedot (DotType val)
{
    brlmon_typedot = val;
}


gboolean 
brlmon_gconf_client_init (void)
{
    GError *error = NULL;
        
    brlmon_client = gconf_client_get_default ();
             
    if (brlmon_client == NULL) 
    {
	sru_warning ("brlmon:Failed to init GConf client:\n");
	sru_warning ("Recomanded to delete ~/.gconf/gnopernicus directories");
	return FALSE;
    }
	             
    gconf_client_add_dir(brlmon_client,
	BRLMON_PATH,GCONF_CLIENT_PRELOAD_NONE,&error);

    if (error != NULL)
    {
	sru_warning ("brlmon:Failed to add directory:\n");
	sru_warning ("Recomanded to delete ~/.gconf/apps/gnopernicus directories");
	g_error_free (error);
	error = NULL;
	brlmon_client = NULL;
	return FALSE;
    }
		
    return TRUE;
}


static gboolean
brlmon_check_type (const gchar* key, GConfValue* val, GConfValueType t, GError** err)
{
    if (val->type != t)
    {
        g_set_error (err, GCONF_ERROR, GCONF_ERROR_TYPE_MISMATCH,
	  	   _("Expected key: %s"),
                   key);
	      
        return FALSE;
    }
    else
	return TRUE;
}


gboolean
brlmon_set_int (gint val, const gchar *key)
{
    GError *error = NULL;
    gboolean  ret = TRUE;
    gchar *path;
    
    sru_return_val_if_fail (key != NULL, FALSE );
    sru_return_val_if_fail (brlmon_client != NULL, FALSE );
    
    path = gconf_concat_dir_and_key (BRLMON_PATH,key);
    
    sru_return_val_if_fail (gconf_client_key_is_writable (brlmon_client,path,NULL),FALSE);
    
    ret  = gconf_client_set_int (brlmon_client,path,val,&error);

    if (error != NULL)
    {
	sru_warning ("brlmon:Failed to set value:%d\n",val);
	sru_warning ("Recomanded to delete ~/.gconf/apps/gnopernicus directories");
	g_error_free (error);
	error = NULL;
    }
    
    g_free(path);
    return ret;
}

/**
 * Get Methods
**/
gint 
brlmon_get_int_with_default (const gchar *key, gint def)
{
    GError *error = NULL;
    GConfValue *value = NULL;
    gint ret_val;
    gchar *path = NULL;
    
    sru_return_val_if_fail (key != NULL, def );
    sru_return_val_if_fail (brlmon_client != NULL, def );

    path  = gconf_concat_dir_and_key (BRLMON_PATH,key);
    value = gconf_client_get (brlmon_client,path,&error);
    
    ret_val = def;
    
    if (value != NULL && error == NULL)
    {
	if (brlmon_check_type (key, value, GCONF_VALUE_INT, &error))
    	    ret_val = gconf_value_get_int (value);
        else
	{
    	    sru_warning ("brlmon:Invalid type of key %s\n",key);
	    sru_warning ("Recomanded to delete ~/.gconf/apps/gnopernicus directories");
		
	    if (!brlmon_set_int (ret_val,key))
	        sru_warning ("brlmon:Failed to set sring value:%d",ret_val);
	}
	    
	
	gconf_value_free (value);
	g_free (path);
	
        return ret_val;
    }
    else
    {
	if (error != NULL)
	{
	    sru_warning ("brlmon:Failed to get value %s\n",key);
	    sru_warning ("Recomanded to delete ~/.gconf/apps/gnopernicus directories");
	}
	    
	if (!brlmon_set_int (def,key))
	    sru_warning ("brlmon:Failed to set sring value:%d",def);
		
	g_free (path);    
	
        return def;
    }    
}


gboolean
brlmon_set_string (const gchar *val, const gchar *key)
{
    GError *error = NULL;
    gboolean ret  = TRUE;
    gchar *path   = NULL;
    
    g_return_val_if_fail (key != NULL, FALSE );
    g_return_val_if_fail (brlmon_client != NULL, FALSE );

    path = gconf_concat_dir_and_key (BRLMON_PATH,key);
    
    g_return_val_if_fail (gconf_client_key_is_writable (brlmon_client,path,NULL), FALSE);
    
    ret = gconf_client_set_string (brlmon_client, path, val, &error);

    if (error != NULL)
    {
	    sru_warning ("brlmon:Failed to set value:%s\n",val);
	    sru_warning ("Recomanded to delete ~/.gconf/apps/gnopernicus directories");
	    g_error_free (error);
	    error = NULL;
    }
    
    g_free (path);
    
    return ret;
}

gchar* 
brlmon_get_string_with_default (const gchar *key, gchar *def)
{
    GError *error = NULL;
    gchar *path   = NULL;
    gchar *retval = NULL;
    
    g_return_val_if_fail (key != NULL, def );
    g_return_val_if_fail (brlmon_client != NULL, def );

    path   = gconf_concat_dir_and_key (BRLMON_PATH,key);
    retval = gconf_client_get_string (brlmon_client,path,&error);

    if (error)
    {
	sru_warning("brlmon:Failed return string value %s\n",key);
	sru_warning("Recomanded to delete ~/.gconf/apps/gnopernicus directories");
	g_error_free(error);
	error = NULL;
	if (!brlmon_set_string (def,key))
	    sru_warning ("brlmon:Failed to set sring value:%s",def);
	
	g_free (path);
	
	return g_strdup (def);
    }
    
    if (!retval)
	return g_strdup (def);
    g_free (path);

    return g_strdup (retval);
}

static void 
brlmon_change_font (GtkWidget 	*view,
		    const gchar *font_name)
{
    PangoFontDescription *font_desc = NULL;
  
    font_desc = pango_font_description_from_string (font_name);
    
    if (font_desc)
    {
	gtk_widget_modify_font (view, font_desc);
	pango_font_description_free (font_desc);
    }
    else
	sru_warning ("%s font not installed", font_name);

    
}

#define NO_OF_DOT_TYPES  3
gboolean
brlmon_load_colors ()
{
    gboolean *succes;    
    gchar    *string_color; 
    
    colors = (GdkColor*) g_new0 (GdkColor , NO_OF_DOT_TYPES);
    succes = (gboolean*) g_new0 (gboolean , NO_OF_DOT_TYPES);

    string_color = brlmon_get_string_with_default (BRLMON_DOT78_GCONF_KEY, DEFAULT_DOT78_COLOR);
    if (!gdk_color_parse (string_color, &colors[0]))
    	g_warning ("Can not parse color string");
    g_free (string_color);
    string_color = brlmon_get_string_with_default (BRLMON_DOT7_GCONF_KEY, DEFAULT_DOT7_COLOR);
    if (!gdk_color_parse (string_color, &colors[1]))
    	g_warning ("Can not parse color string");
    g_free (string_color);
    string_color = brlmon_get_string_with_default (BRLMON_DOT8_GCONF_KEY, DEFAULT_DOT8_COLOR);
    if (!gdk_color_parse (string_color, &colors[2]))
    	g_warning ("Can not parse color string");
    g_free (string_color);

    if (gdk_colormap_alloc_colors (gdk_colormap_get_system (), 
			          colors, NO_OF_DOT_TYPES, 
			          FALSE, TRUE, succes))
	return FALSE;

    return TRUE;	
}
/**
 *
 * Display text with different colors, depended from dottype.
 *
**/
static void
 
brlmon_set_color (GtkWidget *widget)
{

    if (brlmon_use_default_color)
      {
	GtkStateType state;
	switch (brlmon_typedot)
	  {
	  case DOT78:
	    state = GTK_STATE_SELECTED;
	  case DOT7:
	    state = GTK_STATE_PRELIGHT;
	  case DOT8:
	    state = GTK_STATE_ACTIVE;
	  default:
	    state = GTK_STATE_NORMAL;
	    break;
	  }
	gtk_widget_modify_text (widget, state, NULL);
	gtk_widget_modify_text (label_status, state, NULL);
	return;
    }

    switch (brlmon_typedot)
    {
	case DOTNONE:
	    gtk_widget_modify_text (widget,GTK_STATE_NORMAL,NULL);
	    break;	
	
	case DOT78:
	    gtk_widget_modify_text (widget,GTK_STATE_NORMAL,&colors[0]);
	    break;
	
	case DOT7:
	    gtk_widget_modify_text (widget,GTK_STATE_NORMAL,&colors[1]);
	    break;
	    
	default:	
	    gtk_widget_modify_text (widget,GTK_STATE_NORMAL,&colors[2]);
	    break;
    }  

}

static void
brlmon_update_text_color ()
{
    gint i;
    
    for(i = 0 ; i < panel_size ; i++)
	brlmon_set_color (cell[i]);

    for (i = 0 ; i < STATUS_COUNT; i++)
	brlmon_set_color (cell_status[i]);
}

/**
 *
 * Move cursor in 1st cell
 *
**/
void 
brlmon_cursor_pos_clean (void)
{
    cursor_pos = FIRST_POSITION;
}

/**
 * Clean old_pos
**/
void 
brlmon_old_pos_clean (void)
{
    old_pos = INVALID_POSITION;
}

/**
 *
 * Set cursor pos on display
 * pos - position on display
 *
**/
void 
brlmon_cursor_pos (gint pos)
{
    if (brlmon_main_quit) 
	return;
    
    sru_return_if_fail (pos >= FIRST_POSITION && pos < panel_size);
    
    if (old_pos > INVALID_POSITION)
	gtk_frame_set_shadow_type (GTK_FRAME (frame[old_pos]), GTK_SHADOW_OUT);
	
    gtk_frame_set_shadow_type (GTK_FRAME ( frame[pos] ),GTK_SHADOW_IN);
	
    if (brlmon_modetype == MODE_DUAL)
    {
	if (old_pos > INVALID_POSITION)
	    gtk_frame_set_shadow_type (GTK_FRAME (frame[panel_size + old_pos] ), 
				       GTK_SHADOW_OUT);
	
	gtk_frame_set_shadow_type (GTK_FRAME (frame[panel_size + pos] ), GTK_SHADOW_IN);
    }
    
    old_pos = pos;
}

static void 
brlmon_clean_panel_with_size (gint size)
{
    GtkTextBuffer *buffer;
    gint i;
    
    for(i = 0 ; i < size ; i++)
    {
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell[i]));
	gtk_text_buffer_set_text (buffer," ",1);
	brlmon_set_typedot (DOTNONE);
	brlmon_set_color  (cell[i]);
	brlmon_change_font (cell[i], FONT_COURIER);
	gtk_frame_set_shadow_type (GTK_FRAME(frame[i]),GTK_SHADOW_OUT);
    }
    
    for(i = 0; i < STATUS_COUNT ; i++)
    {
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell_status[i]));
	gtk_text_buffer_set_text (buffer," ",1);
	brlmon_change_font (cell_status[i], FONT_COURIER);
	gtk_frame_set_shadow_type (GTK_FRAME(frame_status[i]),GTK_SHADOW_OUT);
    }
    
}

/**
 *
 * Clean display
 *
**/
void 
brlmon_clean_panel (void)
{
    if (brlmon_main_quit) 
	return;
    
    switch(brlmon_modetype)
    {
	case MODE_NORMAL:
	case MODE_BRAILLE:
	     brlmon_clean_panel_with_size (panel_size);
	    break;
	case MODE_DUAL:
	     brlmon_clean_panel_with_size (2 * panel_size);
	    break;
    }
}


/**
 *
 * Show status on status display
 * text - text to show
 *
**/
static void 
brlmon_print_text_on_status (gchar *text)
{
    GtkTextBuffer *buffer;
    gchar str[CELL_TEXT_LENGTH];
    gint  i;
    
    sru_return_if_fail (text);
    sru_return_if_fail (g_utf8_strlen (text,-1) == STATUS_COUNT);
                
    for (i = 0 ; i < STATUS_COUNT; i++)
    {
	g_utf8_strncpy (str,text,1);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell_status[i]));
	gtk_text_buffer_set_text (buffer,str,-1);
	brlmon_set_color (cell_status[i]);
	text = g_utf8_find_next_char (text, NULL);
	    
	switch (brlmon_modetype)
	{
	    case MODE_DUAL:
	    case MODE_NORMAL:
		brlmon_change_font (cell_status[i], FONT_COURIER);
		break;
	    case MODE_BRAILLE:
		brlmon_change_font (cell_status[i], FONT_BRAILLE);
		break;
        }
    }
}


/**
 *
 * Show text on display from current position
 * text - text to show
 *
**/
static void 
brlmon_print_text_from_cur_pos_on_display (gchar *text)
{
    gint len;
    gchar str[32];
    gint i;
    gint pos;
    GtkTextBuffer *buffer;
    
    sru_return_if_fail (text);
    sru_return_if_fail (g_utf8_validate (text, -1, NULL));
    len = g_utf8_strlen (text,-1);
    len = (cursor_pos + len < panel_size) ? len : (panel_size - cursor_pos);
    
    pos = cursor_pos;
    
    for (i = 0 ; i < len; i ++)
    {
	    g_utf8_strncpy (str,text,1);
	    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell[i + pos]));
	    gtk_text_buffer_set_text (buffer,str,-1);
	    brlmon_set_color (cell[i+pos]);
	    
	    if (brlmon_modetype == MODE_NORMAL)
		brlmon_change_font (cell[i+pos], FONT_COURIER);
	    else
		brlmon_change_font (cell[i+pos], FONT_BRAILLE);
	    cursor_pos++;
	    text = g_utf8_find_next_char (text, NULL);
    }
	
    brlmon_typedot = DOTNONE;
}


/**
 *
 * Show text on display in dual mode
 * text - text to show
 *
**/
static void 
brlmon_print_text_from_cur_pos_dual_mode (gchar *text)
{
    gint len;
    gchar str[CELL_TEXT_LENGTH];
    gint i;
    gint pos;
    GtkTextBuffer *buffer;

    sru_return_if_fail (text);    
    sru_return_if_fail (g_utf8_validate (text, -1, NULL));
    len = g_utf8_strlen (text,-1);
    len = (cursor_pos + len < panel_size) ? len : (panel_size - cursor_pos);
    
    pos = cursor_pos;
    
    for (i = 0 ; i < len; i ++)
    {
	g_utf8_strncpy (str, text, 1);
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell[i + pos]));
	gtk_text_buffer_set_text (buffer, str, -1);
	brlmon_set_color (cell[i+pos]);
	brlmon_change_font (cell[i+pos], FONT_COURIER);
	cursor_pos++;
	    
	buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell[i + pos + panel_size]));
	gtk_text_buffer_set_text (buffer, str, -1);
	brlmon_set_color (cell[i + pos + panel_size]);
	brlmon_change_font (cell[i + pos + panel_size], FONT_BRAILLE);
	text = g_utf8_find_next_char (text, NULL);
    }
	
    brlmon_typedot = DOTNONE;
}

/**
 *
 * Show text on Brlmon
 * text - text to show
 * role - type of text
 *
**/
void 
brlmon_print_text_from_cur_pos (gchar *text, DisplayRole role)
{
    if (brlmon_main_quit) 
	return;
    
    switch(role)
    {
	case ROLE_MAIN:
	case ROLE_OTHER:
		if (brlmon_modetype != MODE_DUAL)	
		    brlmon_print_text_from_cur_pos_on_display (text);
		else			
		    brlmon_print_text_from_cur_pos_dual_mode (text);
	    break;
	case ROLE_STATUS:
		brlmon_print_text_on_status (text);
	    break;
    }
}


#ifdef _NO_DEPRECATED_
/**
 *
 * Show text on display from first position.
 * text - text to show
 *
**/
void 
brlmon_print_text (gchar *text)
{
    GtkTextBuffer *buffer;
    gchar str[CELL_TEXT_LENGTH];
    gint len;
    gint i;

    sru_return_if_fail (text);    
    sru_return_if_fail (g_utf8_validate (text, -1, NULL));
    
    if (brlmon_main_quit) 
	return;    
    
    len = len < panel_size ? len : panel_size;
    len = g_utf8_strlen (text, -1);
    cursor_pos = 0;
    
    for (i = 0 ; i < panel_size; i ++)
    {
	if (i < len)
	{
		    g_utf8_strncpy (str,text,1);
		    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell[i]));
    		    gtk_text_buffer_set_text (buffer,str,-1);
		    brlmon_set_color (cell[i]);
		    brlmon_change_font (cell[i], brlmon_modetype);
		    cursor_pos++;
	}
	else
	{
		    buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW(cell[i]));
		    gtk_text_buffer_set_text (buffer,"",-1);
		    brlmon_set_color (cell[i]);
		    brlmon_change_font (cell[i], brlmon_modetype);
	}
    }
}
#endif

/**
 *
 * Key event callback function
 *
**/
static gboolean
brlmon_debugger_key_press_event         (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
		
		
{
    GtkWidget *temp_window = NULL;
    
    if (event->state & GDK_CONTROL_MASK)	
    { 
	    switch(event->keyval)
	    {
		case GDK_F:
		case GDK_f:
		    brlmon_modetype = (brlmon_modetype + 1 ) % MODE_TYPE_NUMBER;
		    
		    brlmon_set_int (brlmon_modetype, "font");
		    
		    switch (brlmon_modetype)
		    {
		     case MODE_BRAILLE:
		        break;
		     case MODE_NORMAL:
		     case MODE_DUAL:
		    	    user_event = FALSE;
			    gtk_widget_hide_all (hbox1);
			    temp_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
			    gtk_widget_reparent (hbox1, temp_window);			    
			    gtk_widget_destroy ( temp_window );
			    gtk_widget_destroyed ( temp_window, &temp_window );
			    hbox1 = NULL;
			break;
		    }
		    brlmon_refresh ();
		break;
		case GDK_B:
		case GDK_b:
		    break;
		case GDK_Q:
		case GDK_q:
		    brlmon_main_quit = TRUE;
		    gtk_main_quit ();
		    break;
		case GDK_D:
		case GDK_d:
		    brlmon_use_default_color = !brlmon_use_default_color;
		    brlmon_update_text_color ();
		    break;
		case GDK_O:
		case GDK_o:
		    brlmon_load_brlmon_options ();
		    break;
		default:break;
	    }
	    return TRUE;
	    
    }	
    else
    {
	switch(event->keyval)
	{
	    case GDK_F1:
		brlmon_load_help_interface ();
	        break;
	    default:
		break;
	}
    }

  return FALSE;
}


/**
 *
 * Create "text status" cells
 *
**/
static void 
brlmon_create_status (void)
{
    GtkWidget *vbox_status;    
    GtkWidget *hbox_status;
    gint i;

    vbox_status = gtk_vbox_new (FALSE, 0);
    gtk_box_set_homogeneous (GTK_BOX (vbox_status),TRUE);
    gtk_widget_ref (vbox_status);
    gtk_object_set_data_full (GTK_OBJECT (w_brlmon), "vbox", vbox_status,
	                        (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (vbox_status);
    gtk_box_pack_start (GTK_BOX (hbox1), vbox_status, FALSE, FALSE, 5);
    
    label_status = gtk_label_new (_("Status"));
    gtk_widget_show (label_status);
    gtk_widget_set_usize (label_status, 60, 10);
    gtk_box_pack_start (GTK_BOX (vbox_status), label_status, TRUE, TRUE, 0);
    gtk_label_set_justify (GTK_LABEL (label_status), GTK_JUSTIFY_LEFT);
        
    hbox_status = gtk_hbox_new (TRUE, 0);
    gtk_widget_ref  (hbox_status);
    gtk_widget_show (hbox_status);
    gtk_box_pack_start (GTK_BOX (vbox_status), hbox_status, TRUE, TRUE, 0);
    
    for(i = 0 ;i < STATUS_COUNT; i++)
    {
	frame_status[i] = gtk_frame_new (NULL);
	gtk_widget_ref (frame_status[i]);
	gtk_widget_set_usize (frame_status[i], 10, 20);
	gtk_object_set_data_full (GTK_OBJECT (hbox1), "frame1", frame_status[i],
                        		(GtkDestroyNotify) gtk_widget_unref);
	gtk_frame_set_shadow_type (GTK_FRAME(frame_status[i]),GTK_SHADOW_OUT);
	gtk_widget_show (frame_status[i]);
	gtk_box_pack_start (GTK_BOX (hbox_status), frame_status[i], TRUE, TRUE, 0);

	cell_status[i] = gtk_text_view_new();
	gtk_widget_ref (cell_status[i]);
	gtk_text_view_set_editable (GTK_TEXT_VIEW(cell_status[i]),FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(cell_status[i]),FALSE);
	gtk_text_view_set_pixels_above_lines (GTK_TEXT_VIEW (cell_status[i]), 4);
	gtk_text_view_set_pixels_below_lines (GTK_TEXT_VIEW (cell_status[i]), 2);
	gtk_object_set_data_full (GTK_OBJECT (hbox1), "c_status", cell_status[i],
                        		(GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (cell_status[i]);
	gtk_container_add (GTK_CONTAINER (frame_status[i]), cell_status[i]);
    }
}




/**
 *
 * Create "col" and "lines" number columns on display
 *
**/
static void 
brlmon_create_line_col (gint line)
{
    GtkWidget *vbox;    
    gint i;
        
    gtk_signal_connect (GTK_OBJECT (w_brlmon), "key_press_event",
                        GTK_SIGNAL_FUNC (brlmon_debugger_key_press_event),
                        NULL);

    hbox1 = gtk_hbox_new (FALSE, 0);
    gtk_widget_show (hbox1);
    
    gtk_container_add (GTK_CONTAINER (w_brlmon), hbox1);
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_set_homogeneous (GTK_BOX (vbox),TRUE);
    gtk_widget_ref (vbox);
    gtk_object_set_data_full (GTK_OBJECT (hbox1), "vbox", vbox,
	                        (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (vbox);
    gtk_box_pack_start (GTK_BOX (hbox1), vbox, TRUE, TRUE, 0);
    for(i = line - 1 ; i > -1 ; i--)
    {
	hbox[i] = gtk_hbox_new (TRUE, 0);
	gtk_box_set_homogeneous (GTK_BOX (hbox[i]),TRUE);
	gtk_widget_ref (hbox[i]);
	gtk_object_set_data_full (GTK_OBJECT (hbox1), "hbox1", hbox[i],
                        	(GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (hbox[i]);
	gtk_box_pack_start (GTK_BOX (vbox), hbox[i], TRUE, TRUE, 0);
    }    
}

/**
 *
 * Create cell table with "line" - row and "col" -column
 *
**/
static void 
brlmon_create_item (gint line, gint col)
{
    gint i;
    for(i = 0 ; i < col ; i++,panel_pos++)
    {
	frame[panel_pos] = gtk_frame_new (NULL);
	gtk_widget_ref (frame[panel_pos]);
	gtk_object_set_data_full (GTK_OBJECT (hbox1), "frame", frame[panel_pos],
                        		(GtkDestroyNotify) gtk_widget_unref);
	gtk_frame_set_shadow_type (GTK_FRAME(frame[panel_pos]),GTK_SHADOW_OUT);
	gtk_widget_set_usize (frame[panel_pos], 10, 10);
	gtk_widget_show (frame[panel_pos]);
	gtk_box_pack_start (GTK_BOX (hbox[line]), frame[panel_pos], TRUE, TRUE, 0);

	cell[panel_pos] = gtk_text_view_new();
	gtk_widget_ref (cell[panel_pos]);
	gtk_text_view_set_editable (GTK_TEXT_VIEW(cell[panel_pos]),FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(cell[panel_pos]),FALSE);
	gtk_text_view_set_pixels_above_lines (GTK_TEXT_VIEW (cell[panel_pos]), 4);
	gtk_text_view_set_pixels_below_lines (GTK_TEXT_VIEW (cell[panel_pos]), 2);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (cell[panel_pos]), 2);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (cell[panel_pos]), 2);
	gtk_object_set_data_full (GTK_OBJECT (hbox1), "c", cell[panel_pos],
                        		(GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (cell[panel_pos]);
	gtk_container_add (GTK_CONTAINER (frame[panel_pos]), cell[panel_pos]);
    }
}

/**
 *
 * Create display table.
 *
**/
void 
brlmon_create_text_area (gint line, gint col)
{
    gint i;
    gint height,width;
            
    panel_size = col * line;
    
    if (brlmon_modetype == MODE_DUAL) 
	line = line * 2;
    
    width = (col * CELL_WIDTH) + (STATUS_COUNT * STATUS_WIDTH);
    
    height = (line * CELL_HEIGHT < MIN_HEIGHT) ? MIN_HEIGHT : line * CELL_HEIGHT;
                
    brlmon_create_line_col (line);

    gtk_window_set_default_size ( GTK_WINDOW (w_brlmon), width, height);
            
    gtk_window_resize ( GTK_WINDOW (w_brlmon), width, height);    
        
    panel_pos = FIRST_POSITION;
        
    for(i = line - 1 ; i > INVALID_POSITION ; i--) 
	brlmon_create_item (i, col);
    
    brlmon_create_status ();
    
    gtk_window_set_focus (GTK_WINDOW (w_brlmon), NULL);    
}

/**
 *
 * Remove display window event callback. Exit from BrlMon
 *
**/ 
static void
brlmon_window_remove   (GtkWidget       *button,
                	gpointer         user_data)
{
    if (user_event)
    {
	brlmon_main_quit = TRUE;
	gtk_main_quit ();
    }
    else
	user_event = TRUE;
}
/**
 *
 * Load brlmon display glade interface
 *
**/

GladeXML* 
brlmon_load_glade_interface (const gchar *glade_file, 
		    	     const gchar *window)
{
    GladeXML *xml;
    gchar *path;
    
    path = g_strdup_printf("../brlmon/%s", glade_file);
	
    if (g_file_test ( path , G_FILE_TEST_EXISTS|G_FILE_TEST_IS_REGULAR))
    {
	xml = glade_xml_new_with_domain (path , window, GETTEXT_PACKAGE);
	g_free (path);
	if (!xml) 
	{
	    sru_warning (_("We could not load the interface!"));
	    return NULL;
	}
    }
    else	
    {
	g_free (path);
	path = g_strdup_printf("%sbrlmon_files/%s", BRLMON_GLADEDIR, glade_file);
	if (g_file_test (path, G_FILE_TEST_EXISTS|G_FILE_TEST_IS_REGULAR))
	{
	    xml = glade_xml_new_with_domain (path , window, GETTEXT_PACKAGE);
	    g_free (path);
	    if (!xml) 
	    {
	        sru_warning (_("We could not load the interface!"));
	        return NULL;
	    }
	}
	else
	{
	    g_free (path);
	    if (g_file_test (glade_file ,G_FILE_TEST_EXISTS|G_FILE_TEST_IS_REGULAR))
	    {
		xml = glade_xml_new_with_domain (glade_file, window, GETTEXT_PACKAGE);
		if (!xml) 
		{
		    sru_warning (_("We could not load the interface!"));
		    return NULL;
		}
	    }
	    else
	    {
		sru_warning (_("We could not load the interface!"));
		return NULL;
	    }
	}
    }
		
    return xml;
}

static void
brlmon_response (GtkDialog *dialog,
		gint       response_id,
		gpointer   user_data)
{
    if (response_id == GTK_RESPONSE_CLOSE)
	brlmon_save_brlmon_options ();

    gtk_widget_hide ((GtkWidget*)dialog);
}


static gint
brlmon_delete_emit_response_cancel (GtkDialog *dialog,
				   GdkEventAny *event,
				   gpointer data)
{
    gtk_dialog_response (GTK_DIALOG (dialog),
			 GTK_RESPONSE_CANCEL);
    return TRUE; /* Do not destroy */
}

gboolean 
brlmon_save_brlmon_options (void)
{	
    gchar *tmp;    

    gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (cp_dot7),    &colors[1].red, &colors[1].green, &colors[1].blue, 0);
    gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (cp_dot78),   &colors[0].red, &colors[0].green, &colors[0].blue, 0);
    gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (cp_dot8),    &colors[2].red, &colors[2].green, &colors[2].blue, 0);

    tmp = g_strdup_printf ("#%04x%04x%04x", colors[0].red,colors[0].green,colors[0].blue);
    brlmon_set_string (tmp , BRLMON_DOT78_GCONF_KEY);    
    g_free (tmp);
    
    tmp = g_strdup_printf ("#%04x%04x%04x", colors[1].red,colors[1].green,colors[1].blue);
    brlmon_set_string (tmp , BRLMON_DOT7_GCONF_KEY);    
    g_free (tmp);
    
    tmp = g_strdup_printf ("#%04x%04x%04x", colors[2].red,colors[2].green,colors[2].blue);
    brlmon_set_string (tmp , BRLMON_DOT8_GCONF_KEY);    
    g_free (tmp);
    
    brlmon_use_default_color = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ck_theme_color));

    brlmon_update_text_color ();
        
    return TRUE;
}

gboolean
brlmon_load_brlmon_options (void)
{
    GladeXML  *xml;
    static GtkWidget *w_brlmon_options;
    
    if (!w_brlmon_options)
    {
	xml = brlmon_load_glade_interface ("monitor.glade2", "w_brlmon_options");
    
	w_brlmon_options = glade_xml_get_widget (xml,"w_brlmon_options");
	
	ck_theme_color = glade_xml_get_widget (xml,"ck_theme_color");	
	cp_dotnone = glade_xml_get_widget (xml,"cp_dotnone");
	cp_dot7    = glade_xml_get_widget (xml,"cp_dot7");
	cp_dot78   = glade_xml_get_widget (xml,"cp_dot78");
	cp_dot8    = glade_xml_get_widget (xml,"cp_dot8");
    
	gtk_window_set_transient_for (GTK_WINDOW (w_brlmon_options),
	    			  GTK_WINDOW (w_brlmon));
	gtk_window_set_destroy_with_parent (GTK_WINDOW (w_brlmon_options), TRUE);

	g_signal_connect (w_brlmon_options, "response",
		          G_CALLBACK (brlmon_response), NULL);
	g_signal_connect (w_brlmon_options, "delete_event",
                    	  G_CALLBACK (brlmon_delete_emit_response_cancel), NULL);        

    }
    else
	gtk_widget_show (w_brlmon_options);
	 
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ck_theme_color), brlmon_use_default_color);

    gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (cp_dot7),    colors[1].red, colors[1].green, colors[1].blue, 0);
    gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (cp_dot78),   colors[0].red, colors[0].green, colors[0].blue, 0);
    gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (cp_dot8),    colors[2].red, colors[2].green, colors[2].blue, 0);

    return TRUE;
}

gboolean 
brlmon_load_interface (void)
{
    GladeXML *xml;
    
    xml = brlmon_load_glade_interface ("monitor.glade2", "w_brlmon");
	    
    w_brlmon = glade_xml_get_widget (xml,"w_brlmon");
                    
    glade_xml_signal_connect (xml,"on_w_brlmon_remove",	
			    GTK_SIGNAL_FUNC (brlmon_window_remove));	
    
    g_object_unref (G_OBJECT (xml));
    
    user_event = TRUE;
    
    brlmon_main_quit = FALSE;
        
    return TRUE;
}

/**
 *
 * Load brlmon help interface
 *
**/
gboolean 
brlmon_load_help_interface (void)
{
    GError *error = NULL;
    
    if (!g_file_test ("../help/braille_monitor/C/brlmonitor.xml", G_FILE_TEST_EXISTS))
	gnome_help_display ("brlmonitor", NULL, &error);
	
    if (error != NULL)
    {
	sru_warning (error->message);
	g_error_free (error);
    }
    
    return TRUE;
}
