/*******************************
 Handling of ICQ (sound) events
 (c) 1999 Jeremy Wise
 GnomeICU
********************************/

#include "common.h"
#include "events.h"
#include "gnomeicu.h"
#include "util.h"

#include <signal.h>
#include <sys/wait.h>

void sig_chld( int ); /* prototype of our signalhandler function */
	
/*** Global functions ***/
void gnomeicu_event(ICQEvType event, UIN_T uin)
{
	GSList *contact;
	gboolean enable_sound = FALSE;

#ifdef TRACE_FUNCTION
	g_print( "gnomeicu_event(%d)\n", event );
#endif

	if( uin )
	{
		contact = Find_User( uin );
		if( contact )
			gnomeicu_fire_external_program(event, kontakt);
	}

	if( toggles->sound_onlyonline == FALSE ||
	    (Current_Status & 0xffff) == STATUS_ONLINE ||
	    (Current_Status & 0xffff) == STATUS_FREE_CHAT )
	{
		enable_sound = TRUE;
	}

	if( !( enable_online_events && enable_sound) )
		return;

	if( toggles->beep && event != EV_USEROFF )
		gdk_beep();

#ifdef HAVE_ESD
	if( !toggles->sound )
		return;

	switch(event)
	{
	case EV_MSGRECV:
		gnome_triggers_do("", "program", "GnomeICU", "message", NULL);
		break;
	case EV_USERON:
		gnome_triggers_do("", "program", "GnomeICU", "online", NULL);
		break;
	case EV_USEROFF:
		gnome_triggers_do("", "program", "GnomeICU", "offline", NULL);
		break;
	case EV_CHATREQ:
		gnome_triggers_do("", "program", "GnomeICU", "chat", NULL);
		break;
	case EV_CHATBEEP:
		gnome_triggers_do("", "program", "GnomeICU", "chatbeep", NULL);
		break;
	case EV_URLRECV:
		gnome_triggers_do("", "program", "GnomeICU", "url", NULL);
		break;
	case EV_AUTHREQ:
		gnome_triggers_do("", "program", "GnomeICU", "authreq", NULL);
		break;
	case EV_AUTH:
		gnome_triggers_do("", "program", "GnomeICU", "auth", NULL);
		break;
	case EV_FILEREQ:
		gnome_triggers_do("", "program", "GnomeICU", "file", NULL);
		break;
	case EV_LISTADD:
		gnome_triggers_do("", "program", "GnomeICU", "notify", NULL);
		break;
	case EV_CONTLIST:
		gnome_triggers_do("", "program", "GnomeICU", "contlist", NULL);
		break;
	}
#endif
}

void gnomeicu_fire_external_program( ICQEvType event, CONTACT_PTR contact )
{
	char *command=NULL;
	gchar *arg0, *arg1, *arg2, *arg3, *arg4;
	static struct sigaction chld;

#ifdef TRACE_FUNCTION
	g_print( "gnomeicu_fire_external_program(%d,%x)\n", event, (int)contact );
#endif

	/* set a signalhandler for sigchld to prevente a zombie proccess */
	if( chld.sa_handler == NULL ){
		chld.sa_handler = sig_chld;
		sigemptyset(&chld.sa_mask);
		chld.sa_flags = SA_NODEFER;
		sigaction(SIGCHLD, &chld, NULL);
	}
	
	g_return_if_fail ( contact );

	if( !enable_online_events )
		return;

	switch(event)
	{
	case EV_MSGRECV:
		if( toggles->program_on_receive_message  )
			command = programs->receive_message;
		break;
	case EV_CHATREQ:
		if( toggles->program_on_chat_request )
			command = programs->chat_request;
		break;
	case EV_URLRECV:
		if( toggles->program_on_receive_url )
			command = programs->receive_url;
		break;
	case EV_FILEREQ:
		if( toggles->program_on_file_request )
			command = programs->file_request;
		break;
	case EV_USERON:
		if( toggles->program_on_user_online )
			command = programs->user_online;
		break;
	case EV_USEROFF:
		if( toggles->program_on_user_offline )
			command = programs->user_offline;
		break;
	case EV_AUTH:
		if( toggles->program_on_auth )
			command = programs->auth;
		break;
	case EV_AUTHREQ:
		if( toggles->program_on_auth_request )
			command = programs->auth_request;
		break;
	case EV_LISTADD:
		if( toggles->program_on_list_add )
			command = programs->list_add;
		break;
	case EV_CONTLIST:
		if( toggles->program_on_contact_list )
			command = programs->contact_list;
		break;
	default:
		return;
	}

	if( command )
	{
		if (fork() == 0)
		{
			arg1 = _("UNK");
			if (Current_Status == STATUS_OFFLINE)
				arg1 = _("OFF");

			switch (Current_Status & 0xffff) {
				case STATUS_ONLINE:
					arg1 = _("ON");
					break;
				case STATUS_DND:
					arg1 = _("DND");
					break;
				case STATUS_AWAY:
					arg1 = _("AWAY");
					break;
				case STATUS_OCCUPIED:
					arg1 = _("OCC");
					break;
				case STATUS_NA:
					arg1 = _("NA");
					break;
				case STATUS_INVISIBLE:
					arg1 = _("INV");
					break;
				case STATUS_FREE_CHAT:
					arg1 = _("FFC");
					break;
			}
			if (strchr (command, '/') == NULL)
				arg0 = command;
			else
				arg0 = strrchr (command, '/');

			arg2 = g_strdup_printf ("%u", contact->uin);
			arg3 = g_strdup (contact->nick);

			if( !contact->has_direct_connect)
			{
				arg4 = g_strdup_printf( "%u.%u.%u.%u",
				       (BYTE) (contact->current_ip >> 24),
				       (BYTE) (contact->current_ip >> 16),
				       (BYTE) (contact->current_ip >> 8), 
				       (BYTE) (contact->current_ip) );    
			}
			else
				arg4 = g_strdup( _("N/A") );

			execlp (command, arg0, arg1, arg2, arg3, arg4, NULL);
			_exit(1);
		}
	}
}

/* our signalhandler function for sigchld
 * nothing to do, just wait for child
 */
void sig_chld( int signo ) 
{
	while(waitpid(-1,NULL,WNOHANG) > 0);
}
