// Take a look at the license at the top of the repository in the LICENSE file.

// rustdoc-stripper-ignore-next
//! Traits intended for implementing the [`SymbolicPaintable`](crate::SymbolicPaintable) interface.

use crate::subclass::prelude::*;
use crate::SymbolicPaintable;
use gdk::subclass::prelude::PaintableImpl;
use glib::translate::*;
use glib::Cast;

pub trait SymbolicPaintableImpl: PaintableImpl {
    fn snapshot_symbolic(
        &self,
        paintable: &Self::Type,
        snapshot: &gdk::Snapshot,
        width: f64,
        height: f64,
        colors: &[gdk::RGBA],
    ) {
        self.parent_snapshot_symbolic(paintable, snapshot, width, height, colors)
    }
}

pub trait SymbolicPaintableImplExt: ObjectSubclass {
    fn parent_snapshot_symbolic(
        &self,
        _paintable: &Self::Type,
        _snapshot: &gdk::Snapshot,
        _width: f64,
        _height: f64,
        _colors: &[gdk::RGBA],
    );
}

impl<T: SymbolicPaintableImpl> SymbolicPaintableImplExt for T {
    fn parent_snapshot_symbolic(
        &self,
        paintable: &Self::Type,
        snapshot: &gdk::Snapshot,
        width: f64,
        height: f64,
        colors: &[gdk::RGBA],
    ) {
        unsafe {
            let type_data = Self::type_data();
            let parent_iface = type_data.as_ref().parent_interface::<SymbolicPaintable>()
                as *const ffi::GtkSymbolicPaintableInterface;

            let func = (*parent_iface).snapshot_symbolic.unwrap();
            func(
                paintable
                    .unsafe_cast_ref::<SymbolicPaintable>()
                    .to_glib_none()
                    .0,
                snapshot.to_glib_none().0,
                width,
                height,
                colors.to_glib_none().0,
                colors.len() as usize,
            )
        }
    }
}

unsafe impl<T: SymbolicPaintableImpl> IsImplementable<T> for SymbolicPaintable {
    fn interface_init(iface: &mut glib::Interface<Self>) {
        let iface = iface.as_mut();

        assert!(
            crate::rt::is_initialized(),
            "GTK has to be initialized first"
        );

        iface.snapshot_symbolic = Some(symbolic_paintable_snapshot_symbolic::<T>);
    }
}

unsafe extern "C" fn symbolic_paintable_snapshot_symbolic<T: SymbolicPaintableImpl>(
    paintable: *mut ffi::GtkSymbolicPaintable,
    snapshotptr: *mut gdk::ffi::GdkSnapshot,
    width: f64,
    height: f64,
    colors: *const gdk::ffi::GdkRGBA,
    n_colors: usize,
) {
    let instance = &*(paintable as *mut T::Instance);
    let imp = instance.imp();

    let snapshot: Borrowed<gdk::Snapshot> = from_glib_borrow(snapshotptr);

    imp.snapshot_symbolic(
        from_glib_borrow::<_, SymbolicPaintable>(paintable).unsafe_cast_ref(),
        &snapshot,
        width,
        height,
        std::slice::from_raw_parts(colors as *const gdk::RGBA, n_colors),
    )
}
