// Take a look at the license at the top of the repository in the LICENSE file.

use glib::translate::*;
use glib::GString;
use libc::c_uint;
use std::mem;

#[derive(Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub struct Key(u32);

impl ::std::ops::Deref for Key {
    type Target = u32;

    fn deref(&self) -> &u32 {
        &self.0
    }
}

impl ::std::ops::DerefMut for Key {
    fn deref_mut(&mut self) -> &mut u32 {
        &mut self.0
    }
}

impl ::std::convert::From<u32> for Key {
    fn from(value: u32) -> Self {
        skip_assert_initialized!();
        Self(value)
    }
}

impl FromGlib<u32> for Key {
    unsafe fn from_glib(value: u32) -> Self {
        skip_assert_initialized!();
        Self(value)
    }
}

impl IntoGlib for Key {
    type GlibType = u32;

    fn into_glib(self) -> u32 {
        *self
    }
}

impl Key {
    #[doc(alias = "gdk_keyval_from_name")]
    pub fn from_name(keyval_name: &str) -> Self {
        skip_assert_initialized!();
        Self(unsafe { ffi::gdk_keyval_from_name(keyval_name.to_glib_none().0) })
    }

    #[doc(alias = "gdk_unicode_to_keyval")]
    pub fn from_unicode(c: char) -> Self {
        skip_assert_initialized!();
        Self(unsafe { ffi::gdk_unicode_to_keyval(u32::from(c)) })
    }

    #[doc(alias = "gdk_keyval_to_unicode")]
    pub fn to_unicode(&self) -> Option<char> {
        skip_assert_initialized!();
        unsafe { ::std::char::from_u32(ffi::gdk_keyval_to_unicode(**self)).filter(|x| *x != '\0') }
    }

    #[doc(alias = "gdk_keyval_name")]
    pub fn name(&self) -> Option<GString> {
        skip_assert_initialized!();
        unsafe { from_glib_none(ffi::gdk_keyval_name(**self as c_uint)) }
    }

    #[doc(alias = "gdk_keyval_is_upper")]
    pub fn is_upper(&self) -> bool {
        skip_assert_initialized!();
        unsafe { from_glib(ffi::gdk_keyval_is_upper(**self)) }
    }

    #[doc(alias = "gdk_keyval_is_lower")]
    pub fn is_lower(&self) -> bool {
        skip_assert_initialized!();
        unsafe { from_glib(ffi::gdk_keyval_is_lower(**self)) }
    }

    #[doc(alias = "gdk_keyval_to_upper")]
    #[must_use]
    pub fn to_upper(&self) -> Self {
        skip_assert_initialized!();
        unsafe { ffi::gdk_keyval_to_upper(**self) }.into()
    }

    #[doc(alias = "gdk_keyval_to_lower")]
    #[must_use]
    pub fn to_lower(&self) -> Self {
        skip_assert_initialized!();
        unsafe { ffi::gdk_keyval_to_lower(**self) }.into()
    }

    #[doc(alias = "gdk_keyval_convert_case")]
    pub fn convert_case(&self) -> (Self, Self) {
        skip_assert_initialized!();
        unsafe {
            let mut lower = mem::MaybeUninit::uninit();
            let mut upper = mem::MaybeUninit::uninit();
            ffi::gdk_keyval_convert_case(self.0, lower.as_mut_ptr(), upper.as_mut_ptr());
            let lower = lower.assume_init();
            let upper = upper.assume_init();
            (Self(lower), Self(upper))
        }
    }
}

impl ::std::fmt::Display for Key {
    fn fmt(&self, f: &mut ::std::fmt::Formatter<'_>) -> ::std::fmt::Result {
        write!(f, "Key({})", self.0)
    }
}

#[allow(non_upper_case_globals)]
pub mod constants {
    use super::Key;

    #[doc(alias = "GDK_KEY_VoidSymbol")]
    pub const VoidSymbol: Key = Key(ffi::GDK_KEY_VoidSymbol as u32);
    #[doc(alias = "GDK_KEY_BackSpace")]
    pub const BackSpace: Key = Key(ffi::GDK_KEY_BackSpace as u32);
    #[doc(alias = "GDK_KEY_Tab")]
    pub const Tab: Key = Key(ffi::GDK_KEY_Tab as u32);
    #[doc(alias = "GDK_KEY_Linefeed")]
    pub const Linefeed: Key = Key(ffi::GDK_KEY_Linefeed as u32);
    #[doc(alias = "GDK_KEY_Clear")]
    pub const Clear: Key = Key(ffi::GDK_KEY_Clear as u32);
    #[doc(alias = "GDK_KEY_Return")]
    pub const Return: Key = Key(ffi::GDK_KEY_Return as u32);
    #[doc(alias = "GDK_KEY_Pause")]
    pub const Pause: Key = Key(ffi::GDK_KEY_Pause as u32);
    #[doc(alias = "GDK_KEY_Scroll_Lock")]
    pub const Scroll_Lock: Key = Key(ffi::GDK_KEY_Scroll_Lock as u32);
    #[doc(alias = "GDK_KEY_Sys_Req")]
    pub const Sys_Req: Key = Key(ffi::GDK_KEY_Sys_Req as u32);
    #[doc(alias = "GDK_KEY_Escape")]
    pub const Escape: Key = Key(ffi::GDK_KEY_Escape as u32);
    #[doc(alias = "GDK_KEY_Delete")]
    pub const Delete: Key = Key(ffi::GDK_KEY_Delete as u32);
    #[doc(alias = "GDK_KEY_Multi_key")]
    pub const Multi_key: Key = Key(ffi::GDK_KEY_Multi_key as u32);
    #[doc(alias = "GDK_KEY_Codeinput")]
    pub const Codeinput: Key = Key(ffi::GDK_KEY_Codeinput as u32);
    #[doc(alias = "GDK_KEY_SingleCandidate")]
    pub const SingleCandidate: Key = Key(ffi::GDK_KEY_SingleCandidate as u32);
    #[doc(alias = "GDK_KEY_MultipleCandidate")]
    pub const MultipleCandidate: Key = Key(ffi::GDK_KEY_MultipleCandidate as u32);
    #[doc(alias = "GDK_KEY_PreviousCandidate")]
    pub const PreviousCandidate: Key = Key(ffi::GDK_KEY_PreviousCandidate as u32);
    #[doc(alias = "GDK_KEY_Kanji")]
    pub const Kanji: Key = Key(ffi::GDK_KEY_Kanji as u32);
    #[doc(alias = "GDK_KEY_Muhenkan")]
    pub const Muhenkan: Key = Key(ffi::GDK_KEY_Muhenkan as u32);
    #[doc(alias = "GDK_KEY_Henkan_Mode")]
    pub const Henkan_Mode: Key = Key(ffi::GDK_KEY_Henkan_Mode as u32);
    #[doc(alias = "GDK_KEY_Henkan")]
    pub const Henkan: Key = Key(ffi::GDK_KEY_Henkan as u32);
    #[doc(alias = "GDK_KEY_Romaji")]
    pub const Romaji: Key = Key(ffi::GDK_KEY_Romaji as u32);
    #[doc(alias = "GDK_KEY_Hiragana")]
    pub const Hiragana: Key = Key(ffi::GDK_KEY_Hiragana as u32);
    #[doc(alias = "GDK_KEY_Katakana")]
    pub const Katakana: Key = Key(ffi::GDK_KEY_Katakana as u32);
    #[doc(alias = "GDK_KEY_Hiragana_Katakana")]
    pub const Hiragana_Katakana: Key = Key(ffi::GDK_KEY_Hiragana_Katakana as u32);
    #[doc(alias = "GDK_KEY_Zenkaku")]
    pub const Zenkaku: Key = Key(ffi::GDK_KEY_Zenkaku as u32);
    #[doc(alias = "GDK_KEY_Hankaku")]
    pub const Hankaku: Key = Key(ffi::GDK_KEY_Hankaku as u32);
    #[doc(alias = "GDK_KEY_Zenkaku_Hankaku")]
    pub const Zenkaku_Hankaku: Key = Key(ffi::GDK_KEY_Zenkaku_Hankaku as u32);
    #[doc(alias = "GDK_KEY_Touroku")]
    pub const Touroku: Key = Key(ffi::GDK_KEY_Touroku as u32);
    #[doc(alias = "GDK_KEY_Massyo")]
    pub const Massyo: Key = Key(ffi::GDK_KEY_Massyo as u32);
    #[doc(alias = "GDK_KEY_Kana_Lock")]
    pub const Kana_Lock: Key = Key(ffi::GDK_KEY_Kana_Lock as u32);
    #[doc(alias = "GDK_KEY_Kana_Shift")]
    pub const Kana_Shift: Key = Key(ffi::GDK_KEY_Kana_Shift as u32);
    #[doc(alias = "GDK_KEY_Eisu_Shift")]
    pub const Eisu_Shift: Key = Key(ffi::GDK_KEY_Eisu_Shift as u32);
    #[doc(alias = "GDK_KEY_Eisu_toggle")]
    pub const Eisu_toggle: Key = Key(ffi::GDK_KEY_Eisu_toggle as u32);
    #[doc(alias = "GDK_KEY_Kanji_Bangou")]
    pub const Kanji_Bangou: Key = Key(ffi::GDK_KEY_Kanji_Bangou as u32);
    #[doc(alias = "GDK_KEY_Zen_Koho")]
    pub const Zen_Koho: Key = Key(ffi::GDK_KEY_Zen_Koho as u32);
    #[doc(alias = "GDK_KEY_Mae_Koho")]
    pub const Mae_Koho: Key = Key(ffi::GDK_KEY_Mae_Koho as u32);
    #[doc(alias = "GDK_KEY_Home")]
    pub const Home: Key = Key(ffi::GDK_KEY_Home as u32);
    #[doc(alias = "GDK_KEY_Left")]
    pub const Left: Key = Key(ffi::GDK_KEY_Left as u32);
    #[doc(alias = "GDK_KEY_Up")]
    pub const Up: Key = Key(ffi::GDK_KEY_Up as u32);
    #[doc(alias = "GDK_KEY_Right")]
    pub const Right: Key = Key(ffi::GDK_KEY_Right as u32);
    #[doc(alias = "GDK_KEY_Down")]
    pub const Down: Key = Key(ffi::GDK_KEY_Down as u32);
    #[doc(alias = "GDK_KEY_Prior")]
    pub const Prior: Key = Key(ffi::GDK_KEY_Prior as u32);
    #[doc(alias = "GDK_KEY_Page_Up")]
    pub const Page_Up: Key = Key(ffi::GDK_KEY_Page_Up as u32);
    #[doc(alias = "GDK_KEY_Next")]
    pub const Next: Key = Key(ffi::GDK_KEY_Next as u32);
    #[doc(alias = "GDK_KEY_Page_Down")]
    pub const Page_Down: Key = Key(ffi::GDK_KEY_Page_Down as u32);
    #[doc(alias = "GDK_KEY_End")]
    pub const End: Key = Key(ffi::GDK_KEY_End as u32);
    #[doc(alias = "GDK_KEY_Begin")]
    pub const Begin: Key = Key(ffi::GDK_KEY_Begin as u32);
    #[doc(alias = "GDK_KEY_Select")]
    pub const Select: Key = Key(ffi::GDK_KEY_Select as u32);
    #[doc(alias = "GDK_KEY_Print")]
    pub const Print: Key = Key(ffi::GDK_KEY_Print as u32);
    #[doc(alias = "GDK_KEY_Execute")]
    pub const Execute: Key = Key(ffi::GDK_KEY_Execute as u32);
    #[doc(alias = "GDK_KEY_Insert")]
    pub const Insert: Key = Key(ffi::GDK_KEY_Insert as u32);
    #[doc(alias = "GDK_KEY_Undo")]
    pub const Undo: Key = Key(ffi::GDK_KEY_Undo as u32);
    #[doc(alias = "GDK_KEY_Redo")]
    pub const Redo: Key = Key(ffi::GDK_KEY_Redo as u32);
    #[doc(alias = "GDK_KEY_Menu")]
    pub const Menu: Key = Key(ffi::GDK_KEY_Menu as u32);
    #[doc(alias = "GDK_KEY_Find")]
    pub const Find: Key = Key(ffi::GDK_KEY_Find as u32);
    #[doc(alias = "GDK_KEY_Cancel")]
    pub const Cancel: Key = Key(ffi::GDK_KEY_Cancel as u32);
    #[doc(alias = "GDK_KEY_Help")]
    pub const Help: Key = Key(ffi::GDK_KEY_Help as u32);
    #[doc(alias = "GDK_KEY_Break")]
    pub const Break: Key = Key(ffi::GDK_KEY_Break as u32);
    #[doc(alias = "GDK_KEY_Mode_switch")]
    pub const Mode_switch: Key = Key(ffi::GDK_KEY_Mode_switch as u32);
    #[doc(alias = "GDK_KEY_script_switch")]
    pub const script_switch: Key = Key(ffi::GDK_KEY_script_switch as u32);
    #[doc(alias = "GDK_KEY_Num_Lock")]
    pub const Num_Lock: Key = Key(ffi::GDK_KEY_Num_Lock as u32);
    #[doc(alias = "GDK_KEY_KP_Space")]
    pub const KP_Space: Key = Key(ffi::GDK_KEY_KP_Space as u32);
    #[doc(alias = "GDK_KEY_KP_Tab")]
    pub const KP_Tab: Key = Key(ffi::GDK_KEY_KP_Tab as u32);
    #[doc(alias = "GDK_KEY_KP_Enter")]
    pub const KP_Enter: Key = Key(ffi::GDK_KEY_KP_Enter as u32);
    #[doc(alias = "GDK_KEY_KP_F1")]
    pub const KP_F1: Key = Key(ffi::GDK_KEY_KP_F1 as u32);
    #[doc(alias = "GDK_KEY_KP_F2")]
    pub const KP_F2: Key = Key(ffi::GDK_KEY_KP_F2 as u32);
    #[doc(alias = "GDK_KEY_KP_F3")]
    pub const KP_F3: Key = Key(ffi::GDK_KEY_KP_F3 as u32);
    #[doc(alias = "GDK_KEY_KP_F4")]
    pub const KP_F4: Key = Key(ffi::GDK_KEY_KP_F4 as u32);
    #[doc(alias = "GDK_KEY_KP_Home")]
    pub const KP_Home: Key = Key(ffi::GDK_KEY_KP_Home as u32);
    #[doc(alias = "GDK_KEY_KP_Left")]
    pub const KP_Left: Key = Key(ffi::GDK_KEY_KP_Left as u32);
    #[doc(alias = "GDK_KEY_KP_Up")]
    pub const KP_Up: Key = Key(ffi::GDK_KEY_KP_Up as u32);
    #[doc(alias = "GDK_KEY_KP_Right")]
    pub const KP_Right: Key = Key(ffi::GDK_KEY_KP_Right as u32);
    #[doc(alias = "GDK_KEY_KP_Down")]
    pub const KP_Down: Key = Key(ffi::GDK_KEY_KP_Down as u32);
    #[doc(alias = "GDK_KEY_KP_Prior")]
    pub const KP_Prior: Key = Key(ffi::GDK_KEY_KP_Prior as u32);
    #[doc(alias = "GDK_KEY_KP_Page_Up")]
    pub const KP_Page_Up: Key = Key(ffi::GDK_KEY_KP_Page_Up as u32);
    #[doc(alias = "GDK_KEY_KP_Next")]
    pub const KP_Next: Key = Key(ffi::GDK_KEY_KP_Next as u32);
    #[doc(alias = "GDK_KEY_KP_Page_Down")]
    pub const KP_Page_Down: Key = Key(ffi::GDK_KEY_KP_Page_Down as u32);
    #[doc(alias = "GDK_KEY_KP_End")]
    pub const KP_End: Key = Key(ffi::GDK_KEY_KP_End as u32);
    #[doc(alias = "GDK_KEY_KP_Begin")]
    pub const KP_Begin: Key = Key(ffi::GDK_KEY_KP_Begin as u32);
    #[doc(alias = "GDK_KEY_KP_Insert")]
    pub const KP_Insert: Key = Key(ffi::GDK_KEY_KP_Insert as u32);
    #[doc(alias = "GDK_KEY_KP_Delete")]
    pub const KP_Delete: Key = Key(ffi::GDK_KEY_KP_Delete as u32);
    #[doc(alias = "GDK_KEY_KP_Equal")]
    pub const KP_Equal: Key = Key(ffi::GDK_KEY_KP_Equal as u32);
    #[doc(alias = "GDK_KEY_KP_Multiply")]
    pub const KP_Multiply: Key = Key(ffi::GDK_KEY_KP_Multiply as u32);
    #[doc(alias = "GDK_KEY_KP_Add")]
    pub const KP_Add: Key = Key(ffi::GDK_KEY_KP_Add as u32);
    #[doc(alias = "GDK_KEY_KP_Separator")]
    pub const KP_Separator: Key = Key(ffi::GDK_KEY_KP_Separator as u32);
    #[doc(alias = "GDK_KEY_KP_Subtract")]
    pub const KP_Subtract: Key = Key(ffi::GDK_KEY_KP_Subtract as u32);
    #[doc(alias = "GDK_KEY_KP_Decimal")]
    pub const KP_Decimal: Key = Key(ffi::GDK_KEY_KP_Decimal as u32);
    #[doc(alias = "GDK_KEY_KP_Divide")]
    pub const KP_Divide: Key = Key(ffi::GDK_KEY_KP_Divide as u32);
    #[doc(alias = "GDK_KEY_KP_0")]
    pub const KP_0: Key = Key(ffi::GDK_KEY_KP_0 as u32);
    #[doc(alias = "GDK_KEY_KP_1")]
    pub const KP_1: Key = Key(ffi::GDK_KEY_KP_1 as u32);
    #[doc(alias = "GDK_KEY_KP_2")]
    pub const KP_2: Key = Key(ffi::GDK_KEY_KP_2 as u32);
    #[doc(alias = "GDK_KEY_KP_3")]
    pub const KP_3: Key = Key(ffi::GDK_KEY_KP_3 as u32);
    #[doc(alias = "GDK_KEY_KP_4")]
    pub const KP_4: Key = Key(ffi::GDK_KEY_KP_4 as u32);
    #[doc(alias = "GDK_KEY_KP_5")]
    pub const KP_5: Key = Key(ffi::GDK_KEY_KP_5 as u32);
    #[doc(alias = "GDK_KEY_KP_6")]
    pub const KP_6: Key = Key(ffi::GDK_KEY_KP_6 as u32);
    #[doc(alias = "GDK_KEY_KP_7")]
    pub const KP_7: Key = Key(ffi::GDK_KEY_KP_7 as u32);
    #[doc(alias = "GDK_KEY_KP_8")]
    pub const KP_8: Key = Key(ffi::GDK_KEY_KP_8 as u32);
    #[doc(alias = "GDK_KEY_KP_9")]
    pub const KP_9: Key = Key(ffi::GDK_KEY_KP_9 as u32);
    #[doc(alias = "GDK_KEY_F1")]
    pub const F1: Key = Key(ffi::GDK_KEY_F1 as u32);
    #[doc(alias = "GDK_KEY_F2")]
    pub const F2: Key = Key(ffi::GDK_KEY_F2 as u32);
    #[doc(alias = "GDK_KEY_F3")]
    pub const F3: Key = Key(ffi::GDK_KEY_F3 as u32);
    #[doc(alias = "GDK_KEY_F4")]
    pub const F4: Key = Key(ffi::GDK_KEY_F4 as u32);
    #[doc(alias = "GDK_KEY_F5")]
    pub const F5: Key = Key(ffi::GDK_KEY_F5 as u32);
    #[doc(alias = "GDK_KEY_F6")]
    pub const F6: Key = Key(ffi::GDK_KEY_F6 as u32);
    #[doc(alias = "GDK_KEY_F7")]
    pub const F7: Key = Key(ffi::GDK_KEY_F7 as u32);
    #[doc(alias = "GDK_KEY_F8")]
    pub const F8: Key = Key(ffi::GDK_KEY_F8 as u32);
    #[doc(alias = "GDK_KEY_F9")]
    pub const F9: Key = Key(ffi::GDK_KEY_F9 as u32);
    #[doc(alias = "GDK_KEY_F10")]
    pub const F10: Key = Key(ffi::GDK_KEY_F10 as u32);
    #[doc(alias = "GDK_KEY_F11")]
    pub const F11: Key = Key(ffi::GDK_KEY_F11 as u32);
    #[doc(alias = "GDK_KEY_L1")]
    pub const L1: Key = Key(ffi::GDK_KEY_L1 as u32);
    #[doc(alias = "GDK_KEY_F12")]
    pub const F12: Key = Key(ffi::GDK_KEY_F12 as u32);
    #[doc(alias = "GDK_KEY_L2")]
    pub const L2: Key = Key(ffi::GDK_KEY_L2 as u32);
    #[doc(alias = "GDK_KEY_F13")]
    pub const F13: Key = Key(ffi::GDK_KEY_F13 as u32);
    #[doc(alias = "GDK_KEY_L3")]
    pub const L3: Key = Key(ffi::GDK_KEY_L3 as u32);
    #[doc(alias = "GDK_KEY_F14")]
    pub const F14: Key = Key(ffi::GDK_KEY_F14 as u32);
    #[doc(alias = "GDK_KEY_L4")]
    pub const L4: Key = Key(ffi::GDK_KEY_L4 as u32);
    #[doc(alias = "GDK_KEY_F15")]
    pub const F15: Key = Key(ffi::GDK_KEY_F15 as u32);
    #[doc(alias = "GDK_KEY_L5")]
    pub const L5: Key = Key(ffi::GDK_KEY_L5 as u32);
    #[doc(alias = "GDK_KEY_F16")]
    pub const F16: Key = Key(ffi::GDK_KEY_F16 as u32);
    #[doc(alias = "GDK_KEY_L6")]
    pub const L6: Key = Key(ffi::GDK_KEY_L6 as u32);
    #[doc(alias = "GDK_KEY_F17")]
    pub const F17: Key = Key(ffi::GDK_KEY_F17 as u32);
    #[doc(alias = "GDK_KEY_L7")]
    pub const L7: Key = Key(ffi::GDK_KEY_L7 as u32);
    #[doc(alias = "GDK_KEY_F18")]
    pub const F18: Key = Key(ffi::GDK_KEY_F18 as u32);
    #[doc(alias = "GDK_KEY_L8")]
    pub const L8: Key = Key(ffi::GDK_KEY_L8 as u32);
    #[doc(alias = "GDK_KEY_F19")]
    pub const F19: Key = Key(ffi::GDK_KEY_F19 as u32);
    #[doc(alias = "GDK_KEY_L9")]
    pub const L9: Key = Key(ffi::GDK_KEY_L9 as u32);
    #[doc(alias = "GDK_KEY_F20")]
    pub const F20: Key = Key(ffi::GDK_KEY_F20 as u32);
    #[doc(alias = "GDK_KEY_L10")]
    pub const L10: Key = Key(ffi::GDK_KEY_L10 as u32);
    #[doc(alias = "GDK_KEY_F21")]
    pub const F21: Key = Key(ffi::GDK_KEY_F21 as u32);
    #[doc(alias = "GDK_KEY_R1")]
    pub const R1: Key = Key(ffi::GDK_KEY_R1 as u32);
    #[doc(alias = "GDK_KEY_F22")]
    pub const F22: Key = Key(ffi::GDK_KEY_F22 as u32);
    #[doc(alias = "GDK_KEY_R2")]
    pub const R2: Key = Key(ffi::GDK_KEY_R2 as u32);
    #[doc(alias = "GDK_KEY_F23")]
    pub const F23: Key = Key(ffi::GDK_KEY_F23 as u32);
    #[doc(alias = "GDK_KEY_R3")]
    pub const R3: Key = Key(ffi::GDK_KEY_R3 as u32);
    #[doc(alias = "GDK_KEY_F24")]
    pub const F24: Key = Key(ffi::GDK_KEY_F24 as u32);
    #[doc(alias = "GDK_KEY_R4")]
    pub const R4: Key = Key(ffi::GDK_KEY_R4 as u32);
    #[doc(alias = "GDK_KEY_F25")]
    pub const F25: Key = Key(ffi::GDK_KEY_F25 as u32);
    #[doc(alias = "GDK_KEY_R5")]
    pub const R5: Key = Key(ffi::GDK_KEY_R5 as u32);
    #[doc(alias = "GDK_KEY_F26")]
    pub const F26: Key = Key(ffi::GDK_KEY_F26 as u32);
    #[doc(alias = "GDK_KEY_R6")]
    pub const R6: Key = Key(ffi::GDK_KEY_R6 as u32);
    #[doc(alias = "GDK_KEY_F27")]
    pub const F27: Key = Key(ffi::GDK_KEY_F27 as u32);
    #[doc(alias = "GDK_KEY_R7")]
    pub const R7: Key = Key(ffi::GDK_KEY_R7 as u32);
    #[doc(alias = "GDK_KEY_F28")]
    pub const F28: Key = Key(ffi::GDK_KEY_F28 as u32);
    #[doc(alias = "GDK_KEY_R8")]
    pub const R8: Key = Key(ffi::GDK_KEY_R8 as u32);
    #[doc(alias = "GDK_KEY_F29")]
    pub const F29: Key = Key(ffi::GDK_KEY_F29 as u32);
    #[doc(alias = "GDK_KEY_R9")]
    pub const R9: Key = Key(ffi::GDK_KEY_R9 as u32);
    #[doc(alias = "GDK_KEY_F30")]
    pub const F30: Key = Key(ffi::GDK_KEY_F30 as u32);
    #[doc(alias = "GDK_KEY_R10")]
    pub const R10: Key = Key(ffi::GDK_KEY_R10 as u32);
    #[doc(alias = "GDK_KEY_F31")]
    pub const F31: Key = Key(ffi::GDK_KEY_F31 as u32);
    #[doc(alias = "GDK_KEY_R11")]
    pub const R11: Key = Key(ffi::GDK_KEY_R11 as u32);
    #[doc(alias = "GDK_KEY_F32")]
    pub const F32: Key = Key(ffi::GDK_KEY_F32 as u32);
    #[doc(alias = "GDK_KEY_R12")]
    pub const R12: Key = Key(ffi::GDK_KEY_R12 as u32);
    #[doc(alias = "GDK_KEY_F33")]
    pub const F33: Key = Key(ffi::GDK_KEY_F33 as u32);
    #[doc(alias = "GDK_KEY_R13")]
    pub const R13: Key = Key(ffi::GDK_KEY_R13 as u32);
    #[doc(alias = "GDK_KEY_F34")]
    pub const F34: Key = Key(ffi::GDK_KEY_F34 as u32);
    #[doc(alias = "GDK_KEY_R14")]
    pub const R14: Key = Key(ffi::GDK_KEY_R14 as u32);
    #[doc(alias = "GDK_KEY_F35")]
    pub const F35: Key = Key(ffi::GDK_KEY_F35 as u32);
    #[doc(alias = "GDK_KEY_R15")]
    pub const R15: Key = Key(ffi::GDK_KEY_R15 as u32);
    #[doc(alias = "GDK_KEY_Shift_L")]
    pub const Shift_L: Key = Key(ffi::GDK_KEY_Shift_L as u32);
    #[doc(alias = "GDK_KEY_Shift_R")]
    pub const Shift_R: Key = Key(ffi::GDK_KEY_Shift_R as u32);
    #[doc(alias = "GDK_KEY_Control_L")]
    pub const Control_L: Key = Key(ffi::GDK_KEY_Control_L as u32);
    #[doc(alias = "GDK_KEY_Control_R")]
    pub const Control_R: Key = Key(ffi::GDK_KEY_Control_R as u32);
    #[doc(alias = "GDK_KEY_Caps_Lock")]
    pub const Caps_Lock: Key = Key(ffi::GDK_KEY_Caps_Lock as u32);
    #[doc(alias = "GDK_KEY_Shift_Lock")]
    pub const Shift_Lock: Key = Key(ffi::GDK_KEY_Shift_Lock as u32);
    #[doc(alias = "GDK_KEY_Meta_L")]
    pub const Meta_L: Key = Key(ffi::GDK_KEY_Meta_L as u32);
    #[doc(alias = "GDK_KEY_Meta_R")]
    pub const Meta_R: Key = Key(ffi::GDK_KEY_Meta_R as u32);
    #[doc(alias = "GDK_KEY_Alt_L")]
    pub const Alt_L: Key = Key(ffi::GDK_KEY_Alt_L as u32);
    #[doc(alias = "GDK_KEY_Alt_R")]
    pub const Alt_R: Key = Key(ffi::GDK_KEY_Alt_R as u32);
    #[doc(alias = "GDK_KEY_Super_L")]
    pub const Super_L: Key = Key(ffi::GDK_KEY_Super_L as u32);
    #[doc(alias = "GDK_KEY_Super_R")]
    pub const Super_R: Key = Key(ffi::GDK_KEY_Super_R as u32);
    #[doc(alias = "GDK_KEY_Hyper_L")]
    pub const Hyper_L: Key = Key(ffi::GDK_KEY_Hyper_L as u32);
    #[doc(alias = "GDK_KEY_Hyper_R")]
    pub const Hyper_R: Key = Key(ffi::GDK_KEY_Hyper_R as u32);
    #[doc(alias = "GDK_KEY_ISO_Lock")]
    pub const ISO_Lock: Key = Key(ffi::GDK_KEY_ISO_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Level2_Latch")]
    pub const ISO_Level2_Latch: Key = Key(ffi::GDK_KEY_ISO_Level2_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Level3_Shift")]
    pub const ISO_Level3_Shift: Key = Key(ffi::GDK_KEY_ISO_Level3_Shift as u32);
    #[doc(alias = "GDK_KEY_ISO_Level3_Latch")]
    pub const ISO_Level3_Latch: Key = Key(ffi::GDK_KEY_ISO_Level3_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Level3_Lock")]
    pub const ISO_Level3_Lock: Key = Key(ffi::GDK_KEY_ISO_Level3_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Level5_Shift")]
    pub const ISO_Level5_Shift: Key = Key(ffi::GDK_KEY_ISO_Level5_Shift as u32);
    #[doc(alias = "GDK_KEY_ISO_Level5_Latch")]
    pub const ISO_Level5_Latch: Key = Key(ffi::GDK_KEY_ISO_Level5_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Level5_Lock")]
    pub const ISO_Level5_Lock: Key = Key(ffi::GDK_KEY_ISO_Level5_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Group_Shift")]
    pub const ISO_Group_Shift: Key = Key(ffi::GDK_KEY_ISO_Group_Shift as u32);
    #[doc(alias = "GDK_KEY_ISO_Group_Latch")]
    pub const ISO_Group_Latch: Key = Key(ffi::GDK_KEY_ISO_Group_Latch as u32);
    #[doc(alias = "GDK_KEY_ISO_Group_Lock")]
    pub const ISO_Group_Lock: Key = Key(ffi::GDK_KEY_ISO_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Next_Group")]
    pub const ISO_Next_Group: Key = Key(ffi::GDK_KEY_ISO_Next_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_Next_Group_Lock")]
    pub const ISO_Next_Group_Lock: Key = Key(ffi::GDK_KEY_ISO_Next_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Prev_Group")]
    pub const ISO_Prev_Group: Key = Key(ffi::GDK_KEY_ISO_Prev_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_Prev_Group_Lock")]
    pub const ISO_Prev_Group_Lock: Key = Key(ffi::GDK_KEY_ISO_Prev_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_First_Group")]
    pub const ISO_First_Group: Key = Key(ffi::GDK_KEY_ISO_First_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_First_Group_Lock")]
    pub const ISO_First_Group_Lock: Key = Key(ffi::GDK_KEY_ISO_First_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Last_Group")]
    pub const ISO_Last_Group: Key = Key(ffi::GDK_KEY_ISO_Last_Group as u32);
    #[doc(alias = "GDK_KEY_ISO_Last_Group_Lock")]
    pub const ISO_Last_Group_Lock: Key = Key(ffi::GDK_KEY_ISO_Last_Group_Lock as u32);
    #[doc(alias = "GDK_KEY_ISO_Left_Tab")]
    pub const ISO_Left_Tab: Key = Key(ffi::GDK_KEY_ISO_Left_Tab as u32);
    #[doc(alias = "GDK_KEY_ISO_Move_Line_Up")]
    pub const ISO_Move_Line_Up: Key = Key(ffi::GDK_KEY_ISO_Move_Line_Up as u32);
    #[doc(alias = "GDK_KEY_ISO_Move_Line_Down")]
    pub const ISO_Move_Line_Down: Key = Key(ffi::GDK_KEY_ISO_Move_Line_Down as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Line_Up")]
    pub const ISO_Partial_Line_Up: Key = Key(ffi::GDK_KEY_ISO_Partial_Line_Up as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Line_Down")]
    pub const ISO_Partial_Line_Down: Key = Key(ffi::GDK_KEY_ISO_Partial_Line_Down as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Space_Left")]
    pub const ISO_Partial_Space_Left: Key = Key(ffi::GDK_KEY_ISO_Partial_Space_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Partial_Space_Right")]
    pub const ISO_Partial_Space_Right: Key = Key(ffi::GDK_KEY_ISO_Partial_Space_Right as u32);
    #[doc(alias = "GDK_KEY_ISO_Set_Margin_Left")]
    pub const ISO_Set_Margin_Left: Key = Key(ffi::GDK_KEY_ISO_Set_Margin_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Set_Margin_Right")]
    pub const ISO_Set_Margin_Right: Key = Key(ffi::GDK_KEY_ISO_Set_Margin_Right as u32);
    #[doc(alias = "GDK_KEY_ISO_Release_Margin_Left")]
    pub const ISO_Release_Margin_Left: Key = Key(ffi::GDK_KEY_ISO_Release_Margin_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Release_Margin_Right")]
    pub const ISO_Release_Margin_Right: Key = Key(ffi::GDK_KEY_ISO_Release_Margin_Right as u32);
    #[doc(alias = "GDK_KEY_ISO_Release_Both_Margins")]
    pub const ISO_Release_Both_Margins: Key = Key(ffi::GDK_KEY_ISO_Release_Both_Margins as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Left")]
    pub const ISO_Fast_Cursor_Left: Key = Key(ffi::GDK_KEY_ISO_Fast_Cursor_Left as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Right")]
    pub const ISO_Fast_Cursor_Right: Key = Key(ffi::GDK_KEY_ISO_Fast_Cursor_Right as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Up")]
    pub const ISO_Fast_Cursor_Up: Key = Key(ffi::GDK_KEY_ISO_Fast_Cursor_Up as u32);
    #[doc(alias = "GDK_KEY_ISO_Fast_Cursor_Down")]
    pub const ISO_Fast_Cursor_Down: Key = Key(ffi::GDK_KEY_ISO_Fast_Cursor_Down as u32);
    #[doc(alias = "GDK_KEY_ISO_Continuous_Underline")]
    pub const ISO_Continuous_Underline: Key = Key(ffi::GDK_KEY_ISO_Continuous_Underline as u32);
    #[doc(alias = "GDK_KEY_ISO_Discontinuous_Underline")]
    pub const ISO_Discontinuous_Underline: Key =
        Key(ffi::GDK_KEY_ISO_Discontinuous_Underline as u32);
    #[doc(alias = "GDK_KEY_ISO_Emphasize")]
    pub const ISO_Emphasize: Key = Key(ffi::GDK_KEY_ISO_Emphasize as u32);
    #[doc(alias = "GDK_KEY_ISO_Center_Object")]
    pub const ISO_Center_Object: Key = Key(ffi::GDK_KEY_ISO_Center_Object as u32);
    #[doc(alias = "GDK_KEY_ISO_Enter")]
    pub const ISO_Enter: Key = Key(ffi::GDK_KEY_ISO_Enter as u32);
    #[doc(alias = "GDK_KEY_dead_grave")]
    pub const dead_grave: Key = Key(ffi::GDK_KEY_dead_grave as u32);
    #[doc(alias = "GDK_KEY_dead_acute")]
    pub const dead_acute: Key = Key(ffi::GDK_KEY_dead_acute as u32);
    #[doc(alias = "GDK_KEY_dead_circumflex")]
    pub const dead_circumflex: Key = Key(ffi::GDK_KEY_dead_circumflex as u32);
    #[doc(alias = "GDK_KEY_dead_tilde")]
    pub const dead_tilde: Key = Key(ffi::GDK_KEY_dead_tilde as u32);
    #[doc(alias = "GDK_KEY_dead_perispomeni")]
    pub const dead_perispomeni: Key = Key(ffi::GDK_KEY_dead_perispomeni as u32);
    #[doc(alias = "GDK_KEY_dead_macron")]
    pub const dead_macron: Key = Key(ffi::GDK_KEY_dead_macron as u32);
    #[doc(alias = "GDK_KEY_dead_breve")]
    pub const dead_breve: Key = Key(ffi::GDK_KEY_dead_breve as u32);
    #[doc(alias = "GDK_KEY_dead_abovedot")]
    pub const dead_abovedot: Key = Key(ffi::GDK_KEY_dead_abovedot as u32);
    #[doc(alias = "GDK_KEY_dead_diaeresis")]
    pub const dead_diaeresis: Key = Key(ffi::GDK_KEY_dead_diaeresis as u32);
    #[doc(alias = "GDK_KEY_dead_abovering")]
    pub const dead_abovering: Key = Key(ffi::GDK_KEY_dead_abovering as u32);
    #[doc(alias = "GDK_KEY_dead_doubleacute")]
    pub const dead_doubleacute: Key = Key(ffi::GDK_KEY_dead_doubleacute as u32);
    #[doc(alias = "GDK_KEY_dead_caron")]
    pub const dead_caron: Key = Key(ffi::GDK_KEY_dead_caron as u32);
    #[doc(alias = "GDK_KEY_dead_cedilla")]
    pub const dead_cedilla: Key = Key(ffi::GDK_KEY_dead_cedilla as u32);
    #[doc(alias = "GDK_KEY_dead_ogonek")]
    pub const dead_ogonek: Key = Key(ffi::GDK_KEY_dead_ogonek as u32);
    #[doc(alias = "GDK_KEY_dead_iota")]
    pub const dead_iota: Key = Key(ffi::GDK_KEY_dead_iota as u32);
    #[doc(alias = "GDK_KEY_dead_voiced_sound")]
    pub const dead_voiced_sound: Key = Key(ffi::GDK_KEY_dead_voiced_sound as u32);
    #[doc(alias = "GDK_KEY_dead_semivoiced_sound")]
    pub const dead_semivoiced_sound: Key = Key(ffi::GDK_KEY_dead_semivoiced_sound as u32);
    #[doc(alias = "GDK_KEY_dead_belowdot")]
    pub const dead_belowdot: Key = Key(ffi::GDK_KEY_dead_belowdot as u32);
    #[doc(alias = "GDK_KEY_dead_hook")]
    pub const dead_hook: Key = Key(ffi::GDK_KEY_dead_hook as u32);
    #[doc(alias = "GDK_KEY_dead_horn")]
    pub const dead_horn: Key = Key(ffi::GDK_KEY_dead_horn as u32);
    #[doc(alias = "GDK_KEY_dead_stroke")]
    pub const dead_stroke: Key = Key(ffi::GDK_KEY_dead_stroke as u32);
    #[doc(alias = "GDK_KEY_dead_abovecomma")]
    pub const dead_abovecomma: Key = Key(ffi::GDK_KEY_dead_abovecomma as u32);
    #[doc(alias = "GDK_KEY_dead_psili")]
    pub const dead_psili: Key = Key(ffi::GDK_KEY_dead_psili as u32);
    #[doc(alias = "GDK_KEY_dead_abovereversedcomma")]
    pub const dead_abovereversedcomma: Key = Key(ffi::GDK_KEY_dead_abovereversedcomma as u32);
    #[doc(alias = "GDK_KEY_dead_dasia")]
    pub const dead_dasia: Key = Key(ffi::GDK_KEY_dead_dasia as u32);
    #[doc(alias = "GDK_KEY_dead_doublegrave")]
    pub const dead_doublegrave: Key = Key(ffi::GDK_KEY_dead_doublegrave as u32);
    #[doc(alias = "GDK_KEY_dead_belowring")]
    pub const dead_belowring: Key = Key(ffi::GDK_KEY_dead_belowring as u32);
    #[doc(alias = "GDK_KEY_dead_belowmacron")]
    pub const dead_belowmacron: Key = Key(ffi::GDK_KEY_dead_belowmacron as u32);
    #[doc(alias = "GDK_KEY_dead_belowcircumflex")]
    pub const dead_belowcircumflex: Key = Key(ffi::GDK_KEY_dead_belowcircumflex as u32);
    #[doc(alias = "GDK_KEY_dead_belowtilde")]
    pub const dead_belowtilde: Key = Key(ffi::GDK_KEY_dead_belowtilde as u32);
    #[doc(alias = "GDK_KEY_dead_belowbreve")]
    pub const dead_belowbreve: Key = Key(ffi::GDK_KEY_dead_belowbreve as u32);
    #[doc(alias = "GDK_KEY_dead_belowdiaeresis")]
    pub const dead_belowdiaeresis: Key = Key(ffi::GDK_KEY_dead_belowdiaeresis as u32);
    #[doc(alias = "GDK_KEY_dead_invertedbreve")]
    pub const dead_invertedbreve: Key = Key(ffi::GDK_KEY_dead_invertedbreve as u32);
    #[doc(alias = "GDK_KEY_dead_belowcomma")]
    pub const dead_belowcomma: Key = Key(ffi::GDK_KEY_dead_belowcomma as u32);
    #[doc(alias = "GDK_KEY_dead_currency")]
    pub const dead_currency: Key = Key(ffi::GDK_KEY_dead_currency as u32);
    #[doc(alias = "GDK_KEY_dead_a")]
    pub const dead_a: Key = Key(ffi::GDK_KEY_dead_a as u32);
    #[doc(alias = "GDK_KEY_dead_A")]
    pub const dead_A: Key = Key(ffi::GDK_KEY_dead_A as u32);
    #[doc(alias = "GDK_KEY_dead_e")]
    pub const dead_e: Key = Key(ffi::GDK_KEY_dead_e as u32);
    #[doc(alias = "GDK_KEY_dead_E")]
    pub const dead_E: Key = Key(ffi::GDK_KEY_dead_E as u32);
    #[doc(alias = "GDK_KEY_dead_i")]
    pub const dead_i: Key = Key(ffi::GDK_KEY_dead_i as u32);
    #[doc(alias = "GDK_KEY_dead_I")]
    pub const dead_I: Key = Key(ffi::GDK_KEY_dead_I as u32);
    #[doc(alias = "GDK_KEY_dead_o")]
    pub const dead_o: Key = Key(ffi::GDK_KEY_dead_o as u32);
    #[doc(alias = "GDK_KEY_dead_O")]
    pub const dead_O: Key = Key(ffi::GDK_KEY_dead_O as u32);
    #[doc(alias = "GDK_KEY_dead_u")]
    pub const dead_u: Key = Key(ffi::GDK_KEY_dead_u as u32);
    #[doc(alias = "GDK_KEY_dead_U")]
    pub const dead_U: Key = Key(ffi::GDK_KEY_dead_U as u32);
    #[doc(alias = "GDK_KEY_dead_small_schwa")]
    pub const dead_small_schwa: Key = Key(ffi::GDK_KEY_dead_small_schwa as u32);
    #[doc(alias = "GDK_KEY_dead_capital_schwa")]
    pub const dead_capital_schwa: Key = Key(ffi::GDK_KEY_dead_capital_schwa as u32);
    #[doc(alias = "GDK_KEY_dead_greek")]
    pub const dead_greek: Key = Key(ffi::GDK_KEY_dead_greek as u32);
    #[doc(alias = "GDK_KEY_First_Virtual_Screen")]
    pub const First_Virtual_Screen: Key = Key(ffi::GDK_KEY_First_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_Prev_Virtual_Screen")]
    pub const Prev_Virtual_Screen: Key = Key(ffi::GDK_KEY_Prev_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_Next_Virtual_Screen")]
    pub const Next_Virtual_Screen: Key = Key(ffi::GDK_KEY_Next_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_Last_Virtual_Screen")]
    pub const Last_Virtual_Screen: Key = Key(ffi::GDK_KEY_Last_Virtual_Screen as u32);
    #[doc(alias = "GDK_KEY_Terminate_Server")]
    pub const Terminate_Server: Key = Key(ffi::GDK_KEY_Terminate_Server as u32);
    #[doc(alias = "GDK_KEY_AccessX_Enable")]
    pub const AccessX_Enable: Key = Key(ffi::GDK_KEY_AccessX_Enable as u32);
    #[doc(alias = "GDK_KEY_AccessX_Feedback_Enable")]
    pub const AccessX_Feedback_Enable: Key = Key(ffi::GDK_KEY_AccessX_Feedback_Enable as u32);
    #[doc(alias = "GDK_KEY_RepeatKeys_Enable")]
    pub const RepeatKeys_Enable: Key = Key(ffi::GDK_KEY_RepeatKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_SlowKeys_Enable")]
    pub const SlowKeys_Enable: Key = Key(ffi::GDK_KEY_SlowKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_BounceKeys_Enable")]
    pub const BounceKeys_Enable: Key = Key(ffi::GDK_KEY_BounceKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_StickyKeys_Enable")]
    pub const StickyKeys_Enable: Key = Key(ffi::GDK_KEY_StickyKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_MouseKeys_Enable")]
    pub const MouseKeys_Enable: Key = Key(ffi::GDK_KEY_MouseKeys_Enable as u32);
    #[doc(alias = "GDK_KEY_MouseKeys_Accel_Enable")]
    pub const MouseKeys_Accel_Enable: Key = Key(ffi::GDK_KEY_MouseKeys_Accel_Enable as u32);
    #[doc(alias = "GDK_KEY_Overlay1_Enable")]
    pub const Overlay1_Enable: Key = Key(ffi::GDK_KEY_Overlay1_Enable as u32);
    #[doc(alias = "GDK_KEY_Overlay2_Enable")]
    pub const Overlay2_Enable: Key = Key(ffi::GDK_KEY_Overlay2_Enable as u32);
    #[doc(alias = "GDK_KEY_AudibleBell_Enable")]
    pub const AudibleBell_Enable: Key = Key(ffi::GDK_KEY_AudibleBell_Enable as u32);
    #[doc(alias = "GDK_KEY_Pointer_Left")]
    pub const Pointer_Left: Key = Key(ffi::GDK_KEY_Pointer_Left as u32);
    #[doc(alias = "GDK_KEY_Pointer_Right")]
    pub const Pointer_Right: Key = Key(ffi::GDK_KEY_Pointer_Right as u32);
    #[doc(alias = "GDK_KEY_Pointer_Up")]
    pub const Pointer_Up: Key = Key(ffi::GDK_KEY_Pointer_Up as u32);
    #[doc(alias = "GDK_KEY_Pointer_Down")]
    pub const Pointer_Down: Key = Key(ffi::GDK_KEY_Pointer_Down as u32);
    #[doc(alias = "GDK_KEY_Pointer_UpLeft")]
    pub const Pointer_UpLeft: Key = Key(ffi::GDK_KEY_Pointer_UpLeft as u32);
    #[doc(alias = "GDK_KEY_Pointer_UpRight")]
    pub const Pointer_UpRight: Key = Key(ffi::GDK_KEY_Pointer_UpRight as u32);
    #[doc(alias = "GDK_KEY_Pointer_DownLeft")]
    pub const Pointer_DownLeft: Key = Key(ffi::GDK_KEY_Pointer_DownLeft as u32);
    #[doc(alias = "GDK_KEY_Pointer_DownRight")]
    pub const Pointer_DownRight: Key = Key(ffi::GDK_KEY_Pointer_DownRight as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button_Dflt")]
    pub const Pointer_Button_Dflt: Key = Key(ffi::GDK_KEY_Pointer_Button_Dflt as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button1")]
    pub const Pointer_Button1: Key = Key(ffi::GDK_KEY_Pointer_Button1 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button2")]
    pub const Pointer_Button2: Key = Key(ffi::GDK_KEY_Pointer_Button2 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button3")]
    pub const Pointer_Button3: Key = Key(ffi::GDK_KEY_Pointer_Button3 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button4")]
    pub const Pointer_Button4: Key = Key(ffi::GDK_KEY_Pointer_Button4 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Button5")]
    pub const Pointer_Button5: Key = Key(ffi::GDK_KEY_Pointer_Button5 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick_Dflt")]
    pub const Pointer_DblClick_Dflt: Key = Key(ffi::GDK_KEY_Pointer_DblClick_Dflt as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick1")]
    pub const Pointer_DblClick1: Key = Key(ffi::GDK_KEY_Pointer_DblClick1 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick2")]
    pub const Pointer_DblClick2: Key = Key(ffi::GDK_KEY_Pointer_DblClick2 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick3")]
    pub const Pointer_DblClick3: Key = Key(ffi::GDK_KEY_Pointer_DblClick3 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick4")]
    pub const Pointer_DblClick4: Key = Key(ffi::GDK_KEY_Pointer_DblClick4 as u32);
    #[doc(alias = "GDK_KEY_Pointer_DblClick5")]
    pub const Pointer_DblClick5: Key = Key(ffi::GDK_KEY_Pointer_DblClick5 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag_Dflt")]
    pub const Pointer_Drag_Dflt: Key = Key(ffi::GDK_KEY_Pointer_Drag_Dflt as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag1")]
    pub const Pointer_Drag1: Key = Key(ffi::GDK_KEY_Pointer_Drag1 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag2")]
    pub const Pointer_Drag2: Key = Key(ffi::GDK_KEY_Pointer_Drag2 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag3")]
    pub const Pointer_Drag3: Key = Key(ffi::GDK_KEY_Pointer_Drag3 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag4")]
    pub const Pointer_Drag4: Key = Key(ffi::GDK_KEY_Pointer_Drag4 as u32);
    #[doc(alias = "GDK_KEY_Pointer_Drag5")]
    pub const Pointer_Drag5: Key = Key(ffi::GDK_KEY_Pointer_Drag5 as u32);
    #[doc(alias = "GDK_KEY_Pointer_EnableKeys")]
    pub const Pointer_EnableKeys: Key = Key(ffi::GDK_KEY_Pointer_EnableKeys as u32);
    #[doc(alias = "GDK_KEY_Pointer_Accelerate")]
    pub const Pointer_Accelerate: Key = Key(ffi::GDK_KEY_Pointer_Accelerate as u32);
    #[doc(alias = "GDK_KEY_Pointer_DfltBtnNext")]
    pub const Pointer_DfltBtnNext: Key = Key(ffi::GDK_KEY_Pointer_DfltBtnNext as u32);
    #[doc(alias = "GDK_KEY_Pointer_DfltBtnPrev")]
    pub const Pointer_DfltBtnPrev: Key = Key(ffi::GDK_KEY_Pointer_DfltBtnPrev as u32);
    #[doc(alias = "GDK_KEY_ch")]
    pub const ch: Key = Key(ffi::GDK_KEY_ch as u32);
    #[doc(alias = "GDK_KEY_Ch")]
    pub const Ch: Key = Key(ffi::GDK_KEY_Ch as u32);
    #[doc(alias = "GDK_KEY_CH")]
    pub const CH: Key = Key(ffi::GDK_KEY_CH as u32);
    #[doc(alias = "GDK_KEY_c_h")]
    pub const c_h: Key = Key(ffi::GDK_KEY_c_h as u32);
    #[doc(alias = "GDK_KEY_C_h")]
    pub const C_h: Key = Key(ffi::GDK_KEY_C_h as u32);
    #[doc(alias = "GDK_KEY_C_H")]
    pub const C_H: Key = Key(ffi::GDK_KEY_C_H as u32);
    #[doc(alias = "GDK_KEY_3270_Duplicate")]
    pub const _3270_Duplicate: Key = Key(ffi::GDK_KEY_3270_Duplicate as u32);
    #[doc(alias = "GDK_KEY_3270_FieldMark")]
    pub const _3270_FieldMark: Key = Key(ffi::GDK_KEY_3270_FieldMark as u32);
    #[doc(alias = "GDK_KEY_3270_Right2")]
    pub const _3270_Right2: Key = Key(ffi::GDK_KEY_3270_Right2 as u32);
    #[doc(alias = "GDK_KEY_3270_Left2")]
    pub const _3270_Left2: Key = Key(ffi::GDK_KEY_3270_Left2 as u32);
    #[doc(alias = "GDK_KEY_3270_BackTab")]
    pub const _3270_BackTab: Key = Key(ffi::GDK_KEY_3270_BackTab as u32);
    #[doc(alias = "GDK_KEY_3270_EraseEOF")]
    pub const _3270_EraseEOF: Key = Key(ffi::GDK_KEY_3270_EraseEOF as u32);
    #[doc(alias = "GDK_KEY_3270_EraseInput")]
    pub const _3270_EraseInput: Key = Key(ffi::GDK_KEY_3270_EraseInput as u32);
    #[doc(alias = "GDK_KEY_3270_Reset")]
    pub const _3270_Reset: Key = Key(ffi::GDK_KEY_3270_Reset as u32);
    #[doc(alias = "GDK_KEY_3270_Quit")]
    pub const _3270_Quit: Key = Key(ffi::GDK_KEY_3270_Quit as u32);
    #[doc(alias = "GDK_KEY_3270_PA1")]
    pub const _3270_PA1: Key = Key(ffi::GDK_KEY_3270_PA1 as u32);
    #[doc(alias = "GDK_KEY_3270_PA2")]
    pub const _3270_PA2: Key = Key(ffi::GDK_KEY_3270_PA2 as u32);
    #[doc(alias = "GDK_KEY_3270_PA3")]
    pub const _3270_PA3: Key = Key(ffi::GDK_KEY_3270_PA3 as u32);
    #[doc(alias = "GDK_KEY_3270_Test")]
    pub const _3270_Test: Key = Key(ffi::GDK_KEY_3270_Test as u32);
    #[doc(alias = "GDK_KEY_3270_Attn")]
    pub const _3270_Attn: Key = Key(ffi::GDK_KEY_3270_Attn as u32);
    #[doc(alias = "GDK_KEY_3270_CursorBlink")]
    pub const _3270_CursorBlink: Key = Key(ffi::GDK_KEY_3270_CursorBlink as u32);
    #[doc(alias = "GDK_KEY_3270_AltCursor")]
    pub const _3270_AltCursor: Key = Key(ffi::GDK_KEY_3270_AltCursor as u32);
    #[doc(alias = "GDK_KEY_3270_KeyClick")]
    pub const _3270_KeyClick: Key = Key(ffi::GDK_KEY_3270_KeyClick as u32);
    #[doc(alias = "GDK_KEY_3270_Jump")]
    pub const _3270_Jump: Key = Key(ffi::GDK_KEY_3270_Jump as u32);
    #[doc(alias = "GDK_KEY_3270_Ident")]
    pub const _3270_Ident: Key = Key(ffi::GDK_KEY_3270_Ident as u32);
    #[doc(alias = "GDK_KEY_3270_Rule")]
    pub const _3270_Rule: Key = Key(ffi::GDK_KEY_3270_Rule as u32);
    #[doc(alias = "GDK_KEY_3270_Copy")]
    pub const _3270_Copy: Key = Key(ffi::GDK_KEY_3270_Copy as u32);
    #[doc(alias = "GDK_KEY_3270_Play")]
    pub const _3270_Play: Key = Key(ffi::GDK_KEY_3270_Play as u32);
    #[doc(alias = "GDK_KEY_3270_Setup")]
    pub const _3270_Setup: Key = Key(ffi::GDK_KEY_3270_Setup as u32);
    #[doc(alias = "GDK_KEY_3270_Record")]
    pub const _3270_Record: Key = Key(ffi::GDK_KEY_3270_Record as u32);
    #[doc(alias = "GDK_KEY_3270_ChangeScreen")]
    pub const _3270_ChangeScreen: Key = Key(ffi::GDK_KEY_3270_ChangeScreen as u32);
    #[doc(alias = "GDK_KEY_3270_DeleteWord")]
    pub const _3270_DeleteWord: Key = Key(ffi::GDK_KEY_3270_DeleteWord as u32);
    #[doc(alias = "GDK_KEY_3270_ExSelect")]
    pub const _3270_ExSelect: Key = Key(ffi::GDK_KEY_3270_ExSelect as u32);
    #[doc(alias = "GDK_KEY_3270_CursorSelect")]
    pub const _3270_CursorSelect: Key = Key(ffi::GDK_KEY_3270_CursorSelect as u32);
    #[doc(alias = "GDK_KEY_3270_PrintScreen")]
    pub const _3270_PrintScreen: Key = Key(ffi::GDK_KEY_3270_PrintScreen as u32);
    #[doc(alias = "GDK_KEY_3270_Enter")]
    pub const _3270_Enter: Key = Key(ffi::GDK_KEY_3270_Enter as u32);
    #[doc(alias = "GDK_KEY_space")]
    pub const space: Key = Key(ffi::GDK_KEY_space as u32);
    #[doc(alias = "GDK_KEY_exclam")]
    pub const exclam: Key = Key(ffi::GDK_KEY_exclam as u32);
    #[doc(alias = "GDK_KEY_quotedbl")]
    pub const quotedbl: Key = Key(ffi::GDK_KEY_quotedbl as u32);
    #[doc(alias = "GDK_KEY_numbersign")]
    pub const numbersign: Key = Key(ffi::GDK_KEY_numbersign as u32);
    #[doc(alias = "GDK_KEY_dollar")]
    pub const dollar: Key = Key(ffi::GDK_KEY_dollar as u32);
    #[doc(alias = "GDK_KEY_percent")]
    pub const percent: Key = Key(ffi::GDK_KEY_percent as u32);
    #[doc(alias = "GDK_KEY_ampersand")]
    pub const ampersand: Key = Key(ffi::GDK_KEY_ampersand as u32);
    #[doc(alias = "GDK_KEY_apostrophe")]
    pub const apostrophe: Key = Key(ffi::GDK_KEY_apostrophe as u32);
    #[doc(alias = "GDK_KEY_quoteright")]
    pub const quoteright: Key = Key(ffi::GDK_KEY_quoteright as u32);
    #[doc(alias = "GDK_KEY_parenleft")]
    pub const parenleft: Key = Key(ffi::GDK_KEY_parenleft as u32);
    #[doc(alias = "GDK_KEY_parenright")]
    pub const parenright: Key = Key(ffi::GDK_KEY_parenright as u32);
    #[doc(alias = "GDK_KEY_asterisk")]
    pub const asterisk: Key = Key(ffi::GDK_KEY_asterisk as u32);
    #[doc(alias = "GDK_KEY_plus")]
    pub const plus: Key = Key(ffi::GDK_KEY_plus as u32);
    #[doc(alias = "GDK_KEY_comma")]
    pub const comma: Key = Key(ffi::GDK_KEY_comma as u32);
    #[doc(alias = "GDK_KEY_minus")]
    pub const minus: Key = Key(ffi::GDK_KEY_minus as u32);
    #[doc(alias = "GDK_KEY_period")]
    pub const period: Key = Key(ffi::GDK_KEY_period as u32);
    #[doc(alias = "GDK_KEY_slash")]
    pub const slash: Key = Key(ffi::GDK_KEY_slash as u32);
    #[doc(alias = "GDK_KEY_0")]
    pub const _0: Key = Key(ffi::GDK_KEY_0 as u32); // equivalent to keypad '0'
    pub const _1: Key = Key(ffi::GDK_KEY_1 as u32); // equivalent to keypad '1'
    pub const _2: Key = Key(ffi::GDK_KEY_2 as u32); // equivalent to keypad '2'
    pub const _3: Key = Key(ffi::GDK_KEY_3 as u32); // equivalent to keypad '3'
    pub const _4: Key = Key(ffi::GDK_KEY_4 as u32); // equivalent to keypad '4'
    pub const _5: Key = Key(ffi::GDK_KEY_5 as u32); // equivalent to keypad '5'
    pub const _6: Key = Key(ffi::GDK_KEY_6 as u32); // equivalent to keypad '6'
    pub const _7: Key = Key(ffi::GDK_KEY_7 as u32); // equivalent to keypad '7'
    pub const _8: Key = Key(ffi::GDK_KEY_8 as u32); // equivalent to keypad '8'
    pub const _9: Key = Key(ffi::GDK_KEY_9 as u32); // equivalent to keypad '9'
    pub const colon: Key = Key(ffi::GDK_KEY_colon as u32);
    #[doc(alias = "GDK_KEY_semicolon")]
    pub const semicolon: Key = Key(ffi::GDK_KEY_semicolon as u32);
    #[doc(alias = "GDK_KEY_less")]
    pub const less: Key = Key(ffi::GDK_KEY_less as u32);
    #[doc(alias = "GDK_KEY_equal")]
    pub const equal: Key = Key(ffi::GDK_KEY_equal as u32);
    #[doc(alias = "GDK_KEY_greater")]
    pub const greater: Key = Key(ffi::GDK_KEY_greater as u32);
    #[doc(alias = "GDK_KEY_question")]
    pub const question: Key = Key(ffi::GDK_KEY_question as u32);
    #[doc(alias = "GDK_KEY_at")]
    pub const at: Key = Key(ffi::GDK_KEY_at as u32);
    #[doc(alias = "GDK_KEY_A")]
    pub const A: Key = Key(ffi::GDK_KEY_A as u32);
    #[doc(alias = "GDK_KEY_B")]
    pub const B: Key = Key(ffi::GDK_KEY_B as u32);
    #[doc(alias = "GDK_KEY_C")]
    pub const C: Key = Key(ffi::GDK_KEY_C as u32);
    #[doc(alias = "GDK_KEY_D")]
    pub const D: Key = Key(ffi::GDK_KEY_D as u32);
    #[doc(alias = "GDK_KEY_E")]
    pub const E: Key = Key(ffi::GDK_KEY_E as u32);
    #[doc(alias = "GDK_KEY_F")]
    pub const F: Key = Key(ffi::GDK_KEY_F as u32);
    #[doc(alias = "GDK_KEY_G")]
    pub const G: Key = Key(ffi::GDK_KEY_G as u32);
    #[doc(alias = "GDK_KEY_H")]
    pub const H: Key = Key(ffi::GDK_KEY_H as u32);
    #[doc(alias = "GDK_KEY_I")]
    pub const I: Key = Key(ffi::GDK_KEY_I as u32);
    #[doc(alias = "GDK_KEY_J")]
    pub const J: Key = Key(ffi::GDK_KEY_J as u32);
    #[doc(alias = "GDK_KEY_K")]
    pub const K: Key = Key(ffi::GDK_KEY_K as u32);
    #[doc(alias = "GDK_KEY_L")]
    pub const L: Key = Key(ffi::GDK_KEY_L as u32);
    #[doc(alias = "GDK_KEY_M")]
    pub const M: Key = Key(ffi::GDK_KEY_M as u32);
    #[doc(alias = "GDK_KEY_N")]
    pub const N: Key = Key(ffi::GDK_KEY_N as u32);
    #[doc(alias = "GDK_KEY_O")]
    pub const O: Key = Key(ffi::GDK_KEY_O as u32);
    #[doc(alias = "GDK_KEY_P")]
    pub const P: Key = Key(ffi::GDK_KEY_P as u32);
    #[doc(alias = "GDK_KEY_Q")]
    pub const Q: Key = Key(ffi::GDK_KEY_Q as u32);
    #[doc(alias = "GDK_KEY_R")]
    pub const R: Key = Key(ffi::GDK_KEY_R as u32);
    #[doc(alias = "GDK_KEY_S")]
    pub const S: Key = Key(ffi::GDK_KEY_S as u32);
    #[doc(alias = "GDK_KEY_T")]
    pub const T: Key = Key(ffi::GDK_KEY_T as u32);
    #[doc(alias = "GDK_KEY_U")]
    pub const U: Key = Key(ffi::GDK_KEY_U as u32);
    #[doc(alias = "GDK_KEY_V")]
    pub const V: Key = Key(ffi::GDK_KEY_V as u32);
    #[doc(alias = "GDK_KEY_W")]
    pub const W: Key = Key(ffi::GDK_KEY_W as u32);
    #[doc(alias = "GDK_KEY_X")]
    pub const X: Key = Key(ffi::GDK_KEY_X as u32);
    #[doc(alias = "GDK_KEY_Y")]
    pub const Y: Key = Key(ffi::GDK_KEY_Y as u32);
    #[doc(alias = "GDK_KEY_Z")]
    pub const Z: Key = Key(ffi::GDK_KEY_Z as u32);
    #[doc(alias = "GDK_KEY_bracketleft")]
    pub const bracketleft: Key = Key(ffi::GDK_KEY_bracketleft as u32);
    #[doc(alias = "GDK_KEY_backslash")]
    pub const backslash: Key = Key(ffi::GDK_KEY_backslash as u32);
    #[doc(alias = "GDK_KEY_bracketright")]
    pub const bracketright: Key = Key(ffi::GDK_KEY_bracketright as u32);
    #[doc(alias = "GDK_KEY_asciicircum")]
    pub const asciicircum: Key = Key(ffi::GDK_KEY_asciicircum as u32);
    #[doc(alias = "GDK_KEY_underscore")]
    pub const underscore: Key = Key(ffi::GDK_KEY_underscore as u32);
    #[doc(alias = "GDK_KEY_grave")]
    pub const grave: Key = Key(ffi::GDK_KEY_grave as u32);
    #[doc(alias = "GDK_KEY_quoteleft")]
    pub const quoteleft: Key = Key(ffi::GDK_KEY_quoteleft as u32);
    #[doc(alias = "GDK_KEY_a")]
    pub const a: Key = Key(ffi::GDK_KEY_a as u32);
    #[doc(alias = "GDK_KEY_b")]
    pub const b: Key = Key(ffi::GDK_KEY_b as u32);
    #[doc(alias = "GDK_KEY_c")]
    pub const c: Key = Key(ffi::GDK_KEY_c as u32);
    #[doc(alias = "GDK_KEY_d")]
    pub const d: Key = Key(ffi::GDK_KEY_d as u32);
    #[doc(alias = "GDK_KEY_e")]
    pub const e: Key = Key(ffi::GDK_KEY_e as u32);
    #[doc(alias = "GDK_KEY_f")]
    pub const f: Key = Key(ffi::GDK_KEY_f as u32);
    #[doc(alias = "GDK_KEY_g")]
    pub const g: Key = Key(ffi::GDK_KEY_g as u32);
    #[doc(alias = "GDK_KEY_h")]
    pub const h: Key = Key(ffi::GDK_KEY_h as u32);
    #[doc(alias = "GDK_KEY_i")]
    pub const i: Key = Key(ffi::GDK_KEY_i as u32);
    #[doc(alias = "GDK_KEY_j")]
    pub const j: Key = Key(ffi::GDK_KEY_j as u32);
    #[doc(alias = "GDK_KEY_k")]
    pub const k: Key = Key(ffi::GDK_KEY_k as u32);
    #[doc(alias = "GDK_KEY_l")]
    pub const l: Key = Key(ffi::GDK_KEY_l as u32);
    #[doc(alias = "GDK_KEY_m")]
    pub const m: Key = Key(ffi::GDK_KEY_m as u32);
    #[doc(alias = "GDK_KEY_n")]
    pub const n: Key = Key(ffi::GDK_KEY_n as u32);
    #[doc(alias = "GDK_KEY_o")]
    pub const o: Key = Key(ffi::GDK_KEY_o as u32);
    #[doc(alias = "GDK_KEY_p")]
    pub const p: Key = Key(ffi::GDK_KEY_p as u32);
    #[doc(alias = "GDK_KEY_q")]
    pub const q: Key = Key(ffi::GDK_KEY_q as u32);
    #[doc(alias = "GDK_KEY_r")]
    pub const r: Key = Key(ffi::GDK_KEY_r as u32);
    #[doc(alias = "GDK_KEY_s")]
    pub const s: Key = Key(ffi::GDK_KEY_s as u32);
    #[doc(alias = "GDK_KEY_t")]
    pub const t: Key = Key(ffi::GDK_KEY_t as u32);
    #[doc(alias = "GDK_KEY_u")]
    pub const u: Key = Key(ffi::GDK_KEY_u as u32);
    #[doc(alias = "GDK_KEY_v")]
    pub const v: Key = Key(ffi::GDK_KEY_v as u32);
    #[doc(alias = "GDK_KEY_w")]
    pub const w: Key = Key(ffi::GDK_KEY_w as u32);
    #[doc(alias = "GDK_KEY_x")]
    pub const x: Key = Key(ffi::GDK_KEY_x as u32);
    #[doc(alias = "GDK_KEY_y")]
    pub const y: Key = Key(ffi::GDK_KEY_y as u32);
    #[doc(alias = "GDK_KEY_z")]
    pub const z: Key = Key(ffi::GDK_KEY_z as u32);
    #[doc(alias = "GDK_KEY_braceleft")]
    pub const braceleft: Key = Key(ffi::GDK_KEY_braceleft as u32);
    #[doc(alias = "GDK_KEY_bar")]
    pub const bar: Key = Key(ffi::GDK_KEY_bar as u32);
    #[doc(alias = "GDK_KEY_braceright")]
    pub const braceright: Key = Key(ffi::GDK_KEY_braceright as u32);
    #[doc(alias = "GDK_KEY_asciitilde")]
    pub const asciitilde: Key = Key(ffi::GDK_KEY_asciitilde as u32);
    #[doc(alias = "GDK_KEY_nobreakspace")]
    pub const nobreakspace: Key = Key(ffi::GDK_KEY_nobreakspace as u32);
    #[doc(alias = "GDK_KEY_exclamdown")]
    pub const exclamdown: Key = Key(ffi::GDK_KEY_exclamdown as u32);
    #[doc(alias = "GDK_KEY_cent")]
    pub const cent: Key = Key(ffi::GDK_KEY_cent as u32);
    #[doc(alias = "GDK_KEY_sterling")]
    pub const sterling: Key = Key(ffi::GDK_KEY_sterling as u32);
    #[doc(alias = "GDK_KEY_currency")]
    pub const currency: Key = Key(ffi::GDK_KEY_currency as u32);
    #[doc(alias = "GDK_KEY_yen")]
    pub const yen: Key = Key(ffi::GDK_KEY_yen as u32);
    #[doc(alias = "GDK_KEY_brokenbar")]
    pub const brokenbar: Key = Key(ffi::GDK_KEY_brokenbar as u32);
    #[doc(alias = "GDK_KEY_section")]
    pub const section: Key = Key(ffi::GDK_KEY_section as u32);
    #[doc(alias = "GDK_KEY_diaeresis")]
    pub const diaeresis: Key = Key(ffi::GDK_KEY_diaeresis as u32);
    #[doc(alias = "GDK_KEY_copyright")]
    pub const copyright: Key = Key(ffi::GDK_KEY_copyright as u32);
    #[doc(alias = "GDK_KEY_ordfeminine")]
    pub const ordfeminine: Key = Key(ffi::GDK_KEY_ordfeminine as u32);
    #[doc(alias = "GDK_KEY_guillemotleft")]
    pub const guillemotleft: Key = Key(ffi::GDK_KEY_guillemotleft as u32);
    #[doc(alias = "GDK_KEY_notsign")]
    pub const notsign: Key = Key(ffi::GDK_KEY_notsign as u32);
    #[doc(alias = "GDK_KEY_hyphen")]
    pub const hyphen: Key = Key(ffi::GDK_KEY_hyphen as u32);
    #[doc(alias = "GDK_KEY_registered")]
    pub const registered: Key = Key(ffi::GDK_KEY_registered as u32);
    #[doc(alias = "GDK_KEY_macron")]
    pub const macron: Key = Key(ffi::GDK_KEY_macron as u32);
    #[doc(alias = "GDK_KEY_degree")]
    pub const degree: Key = Key(ffi::GDK_KEY_degree as u32);
    #[doc(alias = "GDK_KEY_plusminus")]
    pub const plusminus: Key = Key(ffi::GDK_KEY_plusminus as u32);
    #[doc(alias = "GDK_KEY_twosuperior")]
    pub const twosuperior: Key = Key(ffi::GDK_KEY_twosuperior as u32);
    #[doc(alias = "GDK_KEY_threesuperior")]
    pub const threesuperior: Key = Key(ffi::GDK_KEY_threesuperior as u32);
    #[doc(alias = "GDK_KEY_acute")]
    pub const acute: Key = Key(ffi::GDK_KEY_acute as u32);
    #[doc(alias = "GDK_KEY_mu")]
    pub const mu: Key = Key(ffi::GDK_KEY_mu as u32);
    #[doc(alias = "GDK_KEY_paragraph")]
    pub const paragraph: Key = Key(ffi::GDK_KEY_paragraph as u32);
    #[doc(alias = "GDK_KEY_periodcentered")]
    pub const periodcentered: Key = Key(ffi::GDK_KEY_periodcentered as u32);
    #[doc(alias = "GDK_KEY_cedilla")]
    pub const cedilla: Key = Key(ffi::GDK_KEY_cedilla as u32);
    #[doc(alias = "GDK_KEY_onesuperior")]
    pub const onesuperior: Key = Key(ffi::GDK_KEY_onesuperior as u32);
    #[doc(alias = "GDK_KEY_masculine")]
    pub const masculine: Key = Key(ffi::GDK_KEY_masculine as u32);
    #[doc(alias = "GDK_KEY_guillemotright")]
    pub const guillemotright: Key = Key(ffi::GDK_KEY_guillemotright as u32);
    #[doc(alias = "GDK_KEY_onequarter")]
    pub const onequarter: Key = Key(ffi::GDK_KEY_onequarter as u32);
    #[doc(alias = "GDK_KEY_onehalf")]
    pub const onehalf: Key = Key(ffi::GDK_KEY_onehalf as u32);
    #[doc(alias = "GDK_KEY_threequarters")]
    pub const threequarters: Key = Key(ffi::GDK_KEY_threequarters as u32);
    #[doc(alias = "GDK_KEY_questiondown")]
    pub const questiondown: Key = Key(ffi::GDK_KEY_questiondown as u32);
    #[doc(alias = "GDK_KEY_Agrave")]
    pub const Agrave: Key = Key(ffi::GDK_KEY_Agrave as u32);
    #[doc(alias = "GDK_KEY_Aacute")]
    pub const Aacute: Key = Key(ffi::GDK_KEY_Aacute as u32);
    #[doc(alias = "GDK_KEY_Acircumflex")]
    pub const Acircumflex: Key = Key(ffi::GDK_KEY_Acircumflex as u32);
    #[doc(alias = "GDK_KEY_Atilde")]
    pub const Atilde: Key = Key(ffi::GDK_KEY_Atilde as u32);
    #[doc(alias = "GDK_KEY_Adiaeresis")]
    pub const Adiaeresis: Key = Key(ffi::GDK_KEY_Adiaeresis as u32);
    #[doc(alias = "GDK_KEY_Aring")]
    pub const Aring: Key = Key(ffi::GDK_KEY_Aring as u32);
    #[doc(alias = "GDK_KEY_AE")]
    pub const AE: Key = Key(ffi::GDK_KEY_AE as u32);
    #[doc(alias = "GDK_KEY_Ccedilla")]
    pub const Ccedilla: Key = Key(ffi::GDK_KEY_Ccedilla as u32);
    #[doc(alias = "GDK_KEY_Egrave")]
    pub const Egrave: Key = Key(ffi::GDK_KEY_Egrave as u32);
    #[doc(alias = "GDK_KEY_Eacute")]
    pub const Eacute: Key = Key(ffi::GDK_KEY_Eacute as u32);
    #[doc(alias = "GDK_KEY_Ecircumflex")]
    pub const Ecircumflex: Key = Key(ffi::GDK_KEY_Ecircumflex as u32);
    #[doc(alias = "GDK_KEY_Ediaeresis")]
    pub const Ediaeresis: Key = Key(ffi::GDK_KEY_Ediaeresis as u32);
    #[doc(alias = "GDK_KEY_Igrave")]
    pub const Igrave: Key = Key(ffi::GDK_KEY_Igrave as u32);
    #[doc(alias = "GDK_KEY_Iacute")]
    pub const Iacute: Key = Key(ffi::GDK_KEY_Iacute as u32);
    #[doc(alias = "GDK_KEY_Icircumflex")]
    pub const Icircumflex: Key = Key(ffi::GDK_KEY_Icircumflex as u32);
    #[doc(alias = "GDK_KEY_Idiaeresis")]
    pub const Idiaeresis: Key = Key(ffi::GDK_KEY_Idiaeresis as u32);
    #[doc(alias = "GDK_KEY_ETH")]
    pub const ETH: Key = Key(ffi::GDK_KEY_ETH as u32);
    #[doc(alias = "GDK_KEY_Eth")]
    pub const Eth: Key = Key(ffi::GDK_KEY_Eth as u32);
    #[doc(alias = "GDK_KEY_Ntilde")]
    pub const Ntilde: Key = Key(ffi::GDK_KEY_Ntilde as u32);
    #[doc(alias = "GDK_KEY_Ograve")]
    pub const Ograve: Key = Key(ffi::GDK_KEY_Ograve as u32);
    #[doc(alias = "GDK_KEY_Oacute")]
    pub const Oacute: Key = Key(ffi::GDK_KEY_Oacute as u32);
    #[doc(alias = "GDK_KEY_Ocircumflex")]
    pub const Ocircumflex: Key = Key(ffi::GDK_KEY_Ocircumflex as u32);
    #[doc(alias = "GDK_KEY_Otilde")]
    pub const Otilde: Key = Key(ffi::GDK_KEY_Otilde as u32);
    #[doc(alias = "GDK_KEY_Odiaeresis")]
    pub const Odiaeresis: Key = Key(ffi::GDK_KEY_Odiaeresis as u32);
    #[doc(alias = "GDK_KEY_multiply")]
    pub const multiply: Key = Key(ffi::GDK_KEY_multiply as u32);
    #[doc(alias = "GDK_KEY_Oslash")]
    pub const Oslash: Key = Key(ffi::GDK_KEY_Oslash as u32);
    #[doc(alias = "GDK_KEY_Ooblique")]
    pub const Ooblique: Key = Key(ffi::GDK_KEY_Ooblique as u32);
    #[doc(alias = "GDK_KEY_Ugrave")]
    pub const Ugrave: Key = Key(ffi::GDK_KEY_Ugrave as u32);
    #[doc(alias = "GDK_KEY_Uacute")]
    pub const Uacute: Key = Key(ffi::GDK_KEY_Uacute as u32);
    #[doc(alias = "GDK_KEY_Ucircumflex")]
    pub const Ucircumflex: Key = Key(ffi::GDK_KEY_Ucircumflex as u32);
    #[doc(alias = "GDK_KEY_Udiaeresis")]
    pub const Udiaeresis: Key = Key(ffi::GDK_KEY_Udiaeresis as u32);
    #[doc(alias = "GDK_KEY_Yacute")]
    pub const Yacute: Key = Key(ffi::GDK_KEY_Yacute as u32);
    #[doc(alias = "GDK_KEY_THORN")]
    pub const THORN: Key = Key(ffi::GDK_KEY_THORN as u32);
    #[doc(alias = "GDK_KEY_Thorn")]
    pub const Thorn: Key = Key(ffi::GDK_KEY_Thorn as u32);
    #[doc(alias = "GDK_KEY_ssharp")]
    pub const ssharp: Key = Key(ffi::GDK_KEY_ssharp as u32);
    #[doc(alias = "GDK_KEY_agrave")]
    pub const agrave: Key = Key(ffi::GDK_KEY_agrave as u32);
    #[doc(alias = "GDK_KEY_aacute")]
    pub const aacute: Key = Key(ffi::GDK_KEY_aacute as u32);
    #[doc(alias = "GDK_KEY_acircumflex")]
    pub const acircumflex: Key = Key(ffi::GDK_KEY_acircumflex as u32);
    #[doc(alias = "GDK_KEY_atilde")]
    pub const atilde: Key = Key(ffi::GDK_KEY_atilde as u32);
    #[doc(alias = "GDK_KEY_adiaeresis")]
    pub const adiaeresis: Key = Key(ffi::GDK_KEY_adiaeresis as u32);
    #[doc(alias = "GDK_KEY_aring")]
    pub const aring: Key = Key(ffi::GDK_KEY_aring as u32);
    #[doc(alias = "GDK_KEY_ae")]
    pub const ae: Key = Key(ffi::GDK_KEY_ae as u32);
    #[doc(alias = "GDK_KEY_ccedilla")]
    pub const ccedilla: Key = Key(ffi::GDK_KEY_ccedilla as u32);
    #[doc(alias = "GDK_KEY_egrave")]
    pub const egrave: Key = Key(ffi::GDK_KEY_egrave as u32);
    #[doc(alias = "GDK_KEY_eacute")]
    pub const eacute: Key = Key(ffi::GDK_KEY_eacute as u32);
    #[doc(alias = "GDK_KEY_ecircumflex")]
    pub const ecircumflex: Key = Key(ffi::GDK_KEY_ecircumflex as u32);
    #[doc(alias = "GDK_KEY_ediaeresis")]
    pub const ediaeresis: Key = Key(ffi::GDK_KEY_ediaeresis as u32);
    #[doc(alias = "GDK_KEY_igrave")]
    pub const igrave: Key = Key(ffi::GDK_KEY_igrave as u32);
    #[doc(alias = "GDK_KEY_iacute")]
    pub const iacute: Key = Key(ffi::GDK_KEY_iacute as u32);
    #[doc(alias = "GDK_KEY_icircumflex")]
    pub const icircumflex: Key = Key(ffi::GDK_KEY_icircumflex as u32);
    #[doc(alias = "GDK_KEY_idiaeresis")]
    pub const idiaeresis: Key = Key(ffi::GDK_KEY_idiaeresis as u32);
    #[doc(alias = "GDK_KEY_eth")]
    pub const eth: Key = Key(ffi::GDK_KEY_eth as u32);
    #[doc(alias = "GDK_KEY_ntilde")]
    pub const ntilde: Key = Key(ffi::GDK_KEY_ntilde as u32);
    #[doc(alias = "GDK_KEY_ograve")]
    pub const ograve: Key = Key(ffi::GDK_KEY_ograve as u32);
    #[doc(alias = "GDK_KEY_oacute")]
    pub const oacute: Key = Key(ffi::GDK_KEY_oacute as u32);
    #[doc(alias = "GDK_KEY_ocircumflex")]
    pub const ocircumflex: Key = Key(ffi::GDK_KEY_ocircumflex as u32);
    #[doc(alias = "GDK_KEY_otilde")]
    pub const otilde: Key = Key(ffi::GDK_KEY_otilde as u32);
    #[doc(alias = "GDK_KEY_odiaeresis")]
    pub const odiaeresis: Key = Key(ffi::GDK_KEY_odiaeresis as u32);
    #[doc(alias = "GDK_KEY_division")]
    pub const division: Key = Key(ffi::GDK_KEY_division as u32);
    #[doc(alias = "GDK_KEY_oslash")]
    pub const oslash: Key = Key(ffi::GDK_KEY_oslash as u32);
    #[doc(alias = "GDK_KEY_ooblique")]
    pub const ooblique: Key = Key(ffi::GDK_KEY_ooblique as u32);
    #[doc(alias = "GDK_KEY_ugrave")]
    pub const ugrave: Key = Key(ffi::GDK_KEY_ugrave as u32);
    #[doc(alias = "GDK_KEY_uacute")]
    pub const uacute: Key = Key(ffi::GDK_KEY_uacute as u32);
    #[doc(alias = "GDK_KEY_ucircumflex")]
    pub const ucircumflex: Key = Key(ffi::GDK_KEY_ucircumflex as u32);
    #[doc(alias = "GDK_KEY_udiaeresis")]
    pub const udiaeresis: Key = Key(ffi::GDK_KEY_udiaeresis as u32);
    #[doc(alias = "GDK_KEY_yacute")]
    pub const yacute: Key = Key(ffi::GDK_KEY_yacute as u32);
    #[doc(alias = "GDK_KEY_thorn")]
    pub const thorn: Key = Key(ffi::GDK_KEY_thorn as u32);
    #[doc(alias = "GDK_KEY_ydiaeresis")]
    pub const ydiaeresis: Key = Key(ffi::GDK_KEY_ydiaeresis as u32);
    #[doc(alias = "GDK_KEY_Aogonek")]
    pub const Aogonek: Key = Key(ffi::GDK_KEY_Aogonek as u32);
    #[doc(alias = "GDK_KEY_breve")]
    pub const breve: Key = Key(ffi::GDK_KEY_breve as u32);
    #[doc(alias = "GDK_KEY_Lstroke")]
    pub const Lstroke: Key = Key(ffi::GDK_KEY_Lstroke as u32);
    #[doc(alias = "GDK_KEY_Lcaron")]
    pub const Lcaron: Key = Key(ffi::GDK_KEY_Lcaron as u32);
    #[doc(alias = "GDK_KEY_Sacute")]
    pub const Sacute: Key = Key(ffi::GDK_KEY_Sacute as u32);
    #[doc(alias = "GDK_KEY_Scaron")]
    pub const Scaron: Key = Key(ffi::GDK_KEY_Scaron as u32);
    #[doc(alias = "GDK_KEY_Scedilla")]
    pub const Scedilla: Key = Key(ffi::GDK_KEY_Scedilla as u32);
    #[doc(alias = "GDK_KEY_Tcaron")]
    pub const Tcaron: Key = Key(ffi::GDK_KEY_Tcaron as u32);
    #[doc(alias = "GDK_KEY_Zacute")]
    pub const Zacute: Key = Key(ffi::GDK_KEY_Zacute as u32);
    #[doc(alias = "GDK_KEY_Zcaron")]
    pub const Zcaron: Key = Key(ffi::GDK_KEY_Zcaron as u32);
    #[doc(alias = "GDK_KEY_Zabovedot")]
    pub const Zabovedot: Key = Key(ffi::GDK_KEY_Zabovedot as u32);
    #[doc(alias = "GDK_KEY_aogonek")]
    pub const aogonek: Key = Key(ffi::GDK_KEY_aogonek as u32);
    #[doc(alias = "GDK_KEY_ogonek")]
    pub const ogonek: Key = Key(ffi::GDK_KEY_ogonek as u32);
    #[doc(alias = "GDK_KEY_lstroke")]
    pub const lstroke: Key = Key(ffi::GDK_KEY_lstroke as u32);
    #[doc(alias = "GDK_KEY_lcaron")]
    pub const lcaron: Key = Key(ffi::GDK_KEY_lcaron as u32);
    #[doc(alias = "GDK_KEY_sacute")]
    pub const sacute: Key = Key(ffi::GDK_KEY_sacute as u32);
    #[doc(alias = "GDK_KEY_caron")]
    pub const caron: Key = Key(ffi::GDK_KEY_caron as u32);
    #[doc(alias = "GDK_KEY_scaron")]
    pub const scaron: Key = Key(ffi::GDK_KEY_scaron as u32);
    #[doc(alias = "GDK_KEY_scedilla")]
    pub const scedilla: Key = Key(ffi::GDK_KEY_scedilla as u32);
    #[doc(alias = "GDK_KEY_tcaron")]
    pub const tcaron: Key = Key(ffi::GDK_KEY_tcaron as u32);
    #[doc(alias = "GDK_KEY_zacute")]
    pub const zacute: Key = Key(ffi::GDK_KEY_zacute as u32);
    #[doc(alias = "GDK_KEY_doubleacute")]
    pub const doubleacute: Key = Key(ffi::GDK_KEY_doubleacute as u32);
    #[doc(alias = "GDK_KEY_zcaron")]
    pub const zcaron: Key = Key(ffi::GDK_KEY_zcaron as u32);
    #[doc(alias = "GDK_KEY_zabovedot")]
    pub const zabovedot: Key = Key(ffi::GDK_KEY_zabovedot as u32);
    #[doc(alias = "GDK_KEY_Racute")]
    pub const Racute: Key = Key(ffi::GDK_KEY_Racute as u32);
    #[doc(alias = "GDK_KEY_Abreve")]
    pub const Abreve: Key = Key(ffi::GDK_KEY_Abreve as u32);
    #[doc(alias = "GDK_KEY_Lacute")]
    pub const Lacute: Key = Key(ffi::GDK_KEY_Lacute as u32);
    #[doc(alias = "GDK_KEY_Cacute")]
    pub const Cacute: Key = Key(ffi::GDK_KEY_Cacute as u32);
    #[doc(alias = "GDK_KEY_Ccaron")]
    pub const Ccaron: Key = Key(ffi::GDK_KEY_Ccaron as u32);
    #[doc(alias = "GDK_KEY_Eogonek")]
    pub const Eogonek: Key = Key(ffi::GDK_KEY_Eogonek as u32);
    #[doc(alias = "GDK_KEY_Ecaron")]
    pub const Ecaron: Key = Key(ffi::GDK_KEY_Ecaron as u32);
    #[doc(alias = "GDK_KEY_Dcaron")]
    pub const Dcaron: Key = Key(ffi::GDK_KEY_Dcaron as u32);
    #[doc(alias = "GDK_KEY_Dstroke")]
    pub const Dstroke: Key = Key(ffi::GDK_KEY_Dstroke as u32);
    #[doc(alias = "GDK_KEY_Nacute")]
    pub const Nacute: Key = Key(ffi::GDK_KEY_Nacute as u32);
    #[doc(alias = "GDK_KEY_Ncaron")]
    pub const Ncaron: Key = Key(ffi::GDK_KEY_Ncaron as u32);
    #[doc(alias = "GDK_KEY_Odoubleacute")]
    pub const Odoubleacute: Key = Key(ffi::GDK_KEY_Odoubleacute as u32);
    #[doc(alias = "GDK_KEY_Rcaron")]
    pub const Rcaron: Key = Key(ffi::GDK_KEY_Rcaron as u32);
    #[doc(alias = "GDK_KEY_Uring")]
    pub const Uring: Key = Key(ffi::GDK_KEY_Uring as u32);
    #[doc(alias = "GDK_KEY_Udoubleacute")]
    pub const Udoubleacute: Key = Key(ffi::GDK_KEY_Udoubleacute as u32);
    #[doc(alias = "GDK_KEY_Tcedilla")]
    pub const Tcedilla: Key = Key(ffi::GDK_KEY_Tcedilla as u32);
    #[doc(alias = "GDK_KEY_racute")]
    pub const racute: Key = Key(ffi::GDK_KEY_racute as u32);
    #[doc(alias = "GDK_KEY_abreve")]
    pub const abreve: Key = Key(ffi::GDK_KEY_abreve as u32);
    #[doc(alias = "GDK_KEY_lacute")]
    pub const lacute: Key = Key(ffi::GDK_KEY_lacute as u32);
    #[doc(alias = "GDK_KEY_cacute")]
    pub const cacute: Key = Key(ffi::GDK_KEY_cacute as u32);
    #[doc(alias = "GDK_KEY_ccaron")]
    pub const ccaron: Key = Key(ffi::GDK_KEY_ccaron as u32);
    #[doc(alias = "GDK_KEY_eogonek")]
    pub const eogonek: Key = Key(ffi::GDK_KEY_eogonek as u32);
    #[doc(alias = "GDK_KEY_ecaron")]
    pub const ecaron: Key = Key(ffi::GDK_KEY_ecaron as u32);
    #[doc(alias = "GDK_KEY_dcaron")]
    pub const dcaron: Key = Key(ffi::GDK_KEY_dcaron as u32);
    #[doc(alias = "GDK_KEY_dstroke")]
    pub const dstroke: Key = Key(ffi::GDK_KEY_dstroke as u32);
    #[doc(alias = "GDK_KEY_nacute")]
    pub const nacute: Key = Key(ffi::GDK_KEY_nacute as u32);
    #[doc(alias = "GDK_KEY_ncaron")]
    pub const ncaron: Key = Key(ffi::GDK_KEY_ncaron as u32);
    #[doc(alias = "GDK_KEY_odoubleacute")]
    pub const odoubleacute: Key = Key(ffi::GDK_KEY_odoubleacute as u32);
    #[doc(alias = "GDK_KEY_rcaron")]
    pub const rcaron: Key = Key(ffi::GDK_KEY_rcaron as u32);
    #[doc(alias = "GDK_KEY_uring")]
    pub const uring: Key = Key(ffi::GDK_KEY_uring as u32);
    #[doc(alias = "GDK_KEY_udoubleacute")]
    pub const udoubleacute: Key = Key(ffi::GDK_KEY_udoubleacute as u32);
    #[doc(alias = "GDK_KEY_tcedilla")]
    pub const tcedilla: Key = Key(ffi::GDK_KEY_tcedilla as u32);
    #[doc(alias = "GDK_KEY_abovedot")]
    pub const abovedot: Key = Key(ffi::GDK_KEY_abovedot as u32);
    #[doc(alias = "GDK_KEY_Hstroke")]
    pub const Hstroke: Key = Key(ffi::GDK_KEY_Hstroke as u32);
    #[doc(alias = "GDK_KEY_Hcircumflex")]
    pub const Hcircumflex: Key = Key(ffi::GDK_KEY_Hcircumflex as u32);
    #[doc(alias = "GDK_KEY_Iabovedot")]
    pub const Iabovedot: Key = Key(ffi::GDK_KEY_Iabovedot as u32);
    #[doc(alias = "GDK_KEY_Gbreve")]
    pub const Gbreve: Key = Key(ffi::GDK_KEY_Gbreve as u32);
    #[doc(alias = "GDK_KEY_Jcircumflex")]
    pub const Jcircumflex: Key = Key(ffi::GDK_KEY_Jcircumflex as u32);
    #[doc(alias = "GDK_KEY_hstroke")]
    pub const hstroke: Key = Key(ffi::GDK_KEY_hstroke as u32);
    #[doc(alias = "GDK_KEY_hcircumflex")]
    pub const hcircumflex: Key = Key(ffi::GDK_KEY_hcircumflex as u32);
    #[doc(alias = "GDK_KEY_idotless")]
    pub const idotless: Key = Key(ffi::GDK_KEY_idotless as u32);
    #[doc(alias = "GDK_KEY_gbreve")]
    pub const gbreve: Key = Key(ffi::GDK_KEY_gbreve as u32);
    #[doc(alias = "GDK_KEY_jcircumflex")]
    pub const jcircumflex: Key = Key(ffi::GDK_KEY_jcircumflex as u32);
    #[doc(alias = "GDK_KEY_Cabovedot")]
    pub const Cabovedot: Key = Key(ffi::GDK_KEY_Cabovedot as u32);
    #[doc(alias = "GDK_KEY_Ccircumflex")]
    pub const Ccircumflex: Key = Key(ffi::GDK_KEY_Ccircumflex as u32);
    #[doc(alias = "GDK_KEY_Gabovedot")]
    pub const Gabovedot: Key = Key(ffi::GDK_KEY_Gabovedot as u32);
    #[doc(alias = "GDK_KEY_Gcircumflex")]
    pub const Gcircumflex: Key = Key(ffi::GDK_KEY_Gcircumflex as u32);
    #[doc(alias = "GDK_KEY_Ubreve")]
    pub const Ubreve: Key = Key(ffi::GDK_KEY_Ubreve as u32);
    #[doc(alias = "GDK_KEY_Scircumflex")]
    pub const Scircumflex: Key = Key(ffi::GDK_KEY_Scircumflex as u32);
    #[doc(alias = "GDK_KEY_cabovedot")]
    pub const cabovedot: Key = Key(ffi::GDK_KEY_cabovedot as u32);
    #[doc(alias = "GDK_KEY_ccircumflex")]
    pub const ccircumflex: Key = Key(ffi::GDK_KEY_ccircumflex as u32);
    #[doc(alias = "GDK_KEY_gabovedot")]
    pub const gabovedot: Key = Key(ffi::GDK_KEY_gabovedot as u32);
    #[doc(alias = "GDK_KEY_gcircumflex")]
    pub const gcircumflex: Key = Key(ffi::GDK_KEY_gcircumflex as u32);
    #[doc(alias = "GDK_KEY_ubreve")]
    pub const ubreve: Key = Key(ffi::GDK_KEY_ubreve as u32);
    #[doc(alias = "GDK_KEY_scircumflex")]
    pub const scircumflex: Key = Key(ffi::GDK_KEY_scircumflex as u32);
    #[doc(alias = "GDK_KEY_kra")]
    pub const kra: Key = Key(ffi::GDK_KEY_kra as u32);
    #[doc(alias = "GDK_KEY_kappa")]
    pub const kappa: Key = Key(ffi::GDK_KEY_kappa as u32);
    #[doc(alias = "GDK_KEY_Rcedilla")]
    pub const Rcedilla: Key = Key(ffi::GDK_KEY_Rcedilla as u32);
    #[doc(alias = "GDK_KEY_Itilde")]
    pub const Itilde: Key = Key(ffi::GDK_KEY_Itilde as u32);
    #[doc(alias = "GDK_KEY_Lcedilla")]
    pub const Lcedilla: Key = Key(ffi::GDK_KEY_Lcedilla as u32);
    #[doc(alias = "GDK_KEY_Emacron")]
    pub const Emacron: Key = Key(ffi::GDK_KEY_Emacron as u32);
    #[doc(alias = "GDK_KEY_Gcedilla")]
    pub const Gcedilla: Key = Key(ffi::GDK_KEY_Gcedilla as u32);
    #[doc(alias = "GDK_KEY_Tslash")]
    pub const Tslash: Key = Key(ffi::GDK_KEY_Tslash as u32);
    #[doc(alias = "GDK_KEY_rcedilla")]
    pub const rcedilla: Key = Key(ffi::GDK_KEY_rcedilla as u32);
    #[doc(alias = "GDK_KEY_itilde")]
    pub const itilde: Key = Key(ffi::GDK_KEY_itilde as u32);
    #[doc(alias = "GDK_KEY_lcedilla")]
    pub const lcedilla: Key = Key(ffi::GDK_KEY_lcedilla as u32);
    #[doc(alias = "GDK_KEY_emacron")]
    pub const emacron: Key = Key(ffi::GDK_KEY_emacron as u32);
    #[doc(alias = "GDK_KEY_gcedilla")]
    pub const gcedilla: Key = Key(ffi::GDK_KEY_gcedilla as u32);
    #[doc(alias = "GDK_KEY_tslash")]
    pub const tslash: Key = Key(ffi::GDK_KEY_tslash as u32);
    #[doc(alias = "GDK_KEY_ENG")]
    pub const ENG: Key = Key(ffi::GDK_KEY_ENG as u32);
    #[doc(alias = "GDK_KEY_eng")]
    pub const eng: Key = Key(ffi::GDK_KEY_eng as u32);
    #[doc(alias = "GDK_KEY_Amacron")]
    pub const Amacron: Key = Key(ffi::GDK_KEY_Amacron as u32);
    #[doc(alias = "GDK_KEY_Iogonek")]
    pub const Iogonek: Key = Key(ffi::GDK_KEY_Iogonek as u32);
    #[doc(alias = "GDK_KEY_Eabovedot")]
    pub const Eabovedot: Key = Key(ffi::GDK_KEY_Eabovedot as u32);
    #[doc(alias = "GDK_KEY_Imacron")]
    pub const Imacron: Key = Key(ffi::GDK_KEY_Imacron as u32);
    #[doc(alias = "GDK_KEY_Ncedilla")]
    pub const Ncedilla: Key = Key(ffi::GDK_KEY_Ncedilla as u32);
    #[doc(alias = "GDK_KEY_Omacron")]
    pub const Omacron: Key = Key(ffi::GDK_KEY_Omacron as u32);
    #[doc(alias = "GDK_KEY_Kcedilla")]
    pub const Kcedilla: Key = Key(ffi::GDK_KEY_Kcedilla as u32);
    #[doc(alias = "GDK_KEY_Uogonek")]
    pub const Uogonek: Key = Key(ffi::GDK_KEY_Uogonek as u32);
    #[doc(alias = "GDK_KEY_Utilde")]
    pub const Utilde: Key = Key(ffi::GDK_KEY_Utilde as u32);
    #[doc(alias = "GDK_KEY_Umacron")]
    pub const Umacron: Key = Key(ffi::GDK_KEY_Umacron as u32);
    #[doc(alias = "GDK_KEY_amacron")]
    pub const amacron: Key = Key(ffi::GDK_KEY_amacron as u32);
    #[doc(alias = "GDK_KEY_iogonek")]
    pub const iogonek: Key = Key(ffi::GDK_KEY_iogonek as u32);
    #[doc(alias = "GDK_KEY_eabovedot")]
    pub const eabovedot: Key = Key(ffi::GDK_KEY_eabovedot as u32);
    #[doc(alias = "GDK_KEY_imacron")]
    pub const imacron: Key = Key(ffi::GDK_KEY_imacron as u32);
    #[doc(alias = "GDK_KEY_ncedilla")]
    pub const ncedilla: Key = Key(ffi::GDK_KEY_ncedilla as u32);
    #[doc(alias = "GDK_KEY_omacron")]
    pub const omacron: Key = Key(ffi::GDK_KEY_omacron as u32);
    #[doc(alias = "GDK_KEY_kcedilla")]
    pub const kcedilla: Key = Key(ffi::GDK_KEY_kcedilla as u32);
    #[doc(alias = "GDK_KEY_uogonek")]
    pub const uogonek: Key = Key(ffi::GDK_KEY_uogonek as u32);
    #[doc(alias = "GDK_KEY_utilde")]
    pub const utilde: Key = Key(ffi::GDK_KEY_utilde as u32);
    #[doc(alias = "GDK_KEY_umacron")]
    pub const umacron: Key = Key(ffi::GDK_KEY_umacron as u32);
    #[doc(alias = "GDK_KEY_Wcircumflex")]
    pub const Wcircumflex: Key = Key(ffi::GDK_KEY_Wcircumflex as u32);
    #[doc(alias = "GDK_KEY_wcircumflex")]
    pub const wcircumflex: Key = Key(ffi::GDK_KEY_wcircumflex as u32);
    #[doc(alias = "GDK_KEY_Ycircumflex")]
    pub const Ycircumflex: Key = Key(ffi::GDK_KEY_Ycircumflex as u32);
    #[doc(alias = "GDK_KEY_ycircumflex")]
    pub const ycircumflex: Key = Key(ffi::GDK_KEY_ycircumflex as u32);
    #[doc(alias = "GDK_KEY_Babovedot")]
    pub const Babovedot: Key = Key(ffi::GDK_KEY_Babovedot as u32);
    #[doc(alias = "GDK_KEY_babovedot")]
    pub const babovedot: Key = Key(ffi::GDK_KEY_babovedot as u32);
    #[doc(alias = "GDK_KEY_Dabovedot")]
    pub const Dabovedot: Key = Key(ffi::GDK_KEY_Dabovedot as u32);
    #[doc(alias = "GDK_KEY_dabovedot")]
    pub const dabovedot: Key = Key(ffi::GDK_KEY_dabovedot as u32);
    #[doc(alias = "GDK_KEY_Fabovedot")]
    pub const Fabovedot: Key = Key(ffi::GDK_KEY_Fabovedot as u32);
    #[doc(alias = "GDK_KEY_fabovedot")]
    pub const fabovedot: Key = Key(ffi::GDK_KEY_fabovedot as u32);
    #[doc(alias = "GDK_KEY_Mabovedot")]
    pub const Mabovedot: Key = Key(ffi::GDK_KEY_Mabovedot as u32);
    #[doc(alias = "GDK_KEY_mabovedot")]
    pub const mabovedot: Key = Key(ffi::GDK_KEY_mabovedot as u32);
    #[doc(alias = "GDK_KEY_Pabovedot")]
    pub const Pabovedot: Key = Key(ffi::GDK_KEY_Pabovedot as u32);
    #[doc(alias = "GDK_KEY_pabovedot")]
    pub const pabovedot: Key = Key(ffi::GDK_KEY_pabovedot as u32);
    #[doc(alias = "GDK_KEY_Sabovedot")]
    pub const Sabovedot: Key = Key(ffi::GDK_KEY_Sabovedot as u32);
    #[doc(alias = "GDK_KEY_sabovedot")]
    pub const sabovedot: Key = Key(ffi::GDK_KEY_sabovedot as u32);
    #[doc(alias = "GDK_KEY_Tabovedot")]
    pub const Tabovedot: Key = Key(ffi::GDK_KEY_Tabovedot as u32);
    #[doc(alias = "GDK_KEY_tabovedot")]
    pub const tabovedot: Key = Key(ffi::GDK_KEY_tabovedot as u32);
    #[doc(alias = "GDK_KEY_Wgrave")]
    pub const Wgrave: Key = Key(ffi::GDK_KEY_Wgrave as u32);
    #[doc(alias = "GDK_KEY_wgrave")]
    pub const wgrave: Key = Key(ffi::GDK_KEY_wgrave as u32);
    #[doc(alias = "GDK_KEY_Wacute")]
    pub const Wacute: Key = Key(ffi::GDK_KEY_Wacute as u32);
    #[doc(alias = "GDK_KEY_wacute")]
    pub const wacute: Key = Key(ffi::GDK_KEY_wacute as u32);
    #[doc(alias = "GDK_KEY_Wdiaeresis")]
    pub const Wdiaeresis: Key = Key(ffi::GDK_KEY_Wdiaeresis as u32);
    #[doc(alias = "GDK_KEY_wdiaeresis")]
    pub const wdiaeresis: Key = Key(ffi::GDK_KEY_wdiaeresis as u32);
    #[doc(alias = "GDK_KEY_Ygrave")]
    pub const Ygrave: Key = Key(ffi::GDK_KEY_Ygrave as u32);
    #[doc(alias = "GDK_KEY_ygrave")]
    pub const ygrave: Key = Key(ffi::GDK_KEY_ygrave as u32);
    #[doc(alias = "GDK_KEY_OE")]
    pub const OE: Key = Key(ffi::GDK_KEY_OE as u32);
    #[doc(alias = "GDK_KEY_oe")]
    pub const oe: Key = Key(ffi::GDK_KEY_oe as u32);
    #[doc(alias = "GDK_KEY_Ydiaeresis")]
    pub const Ydiaeresis: Key = Key(ffi::GDK_KEY_Ydiaeresis as u32);
    #[doc(alias = "GDK_KEY_overline")]
    pub const overline: Key = Key(ffi::GDK_KEY_overline as u32);
    #[doc(alias = "GDK_KEY_kana_fullstop")]
    pub const kana_fullstop: Key = Key(ffi::GDK_KEY_kana_fullstop as u32);
    #[doc(alias = "GDK_KEY_kana_openingbracket")]
    pub const kana_openingbracket: Key = Key(ffi::GDK_KEY_kana_openingbracket as u32);
    #[doc(alias = "GDK_KEY_kana_closingbracket")]
    pub const kana_closingbracket: Key = Key(ffi::GDK_KEY_kana_closingbracket as u32);
    #[doc(alias = "GDK_KEY_kana_comma")]
    pub const kana_comma: Key = Key(ffi::GDK_KEY_kana_comma as u32);
    #[doc(alias = "GDK_KEY_kana_conjunctive")]
    pub const kana_conjunctive: Key = Key(ffi::GDK_KEY_kana_conjunctive as u32);
    #[doc(alias = "GDK_KEY_kana_middledot")]
    pub const kana_middledot: Key = Key(ffi::GDK_KEY_kana_middledot as u32);
    #[doc(alias = "GDK_KEY_kana_WO")]
    pub const kana_WO: Key = Key(ffi::GDK_KEY_kana_WO as u32);
    #[doc(alias = "GDK_KEY_kana_a")]
    pub const kana_a: Key = Key(ffi::GDK_KEY_kana_a as u32);
    #[doc(alias = "GDK_KEY_kana_i")]
    pub const kana_i: Key = Key(ffi::GDK_KEY_kana_i as u32);
    #[doc(alias = "GDK_KEY_kana_u")]
    pub const kana_u: Key = Key(ffi::GDK_KEY_kana_u as u32);
    #[doc(alias = "GDK_KEY_kana_e")]
    pub const kana_e: Key = Key(ffi::GDK_KEY_kana_e as u32);
    #[doc(alias = "GDK_KEY_kana_o")]
    pub const kana_o: Key = Key(ffi::GDK_KEY_kana_o as u32);
    #[doc(alias = "GDK_KEY_kana_ya")]
    pub const kana_ya: Key = Key(ffi::GDK_KEY_kana_ya as u32);
    #[doc(alias = "GDK_KEY_kana_yu")]
    pub const kana_yu: Key = Key(ffi::GDK_KEY_kana_yu as u32);
    #[doc(alias = "GDK_KEY_kana_yo")]
    pub const kana_yo: Key = Key(ffi::GDK_KEY_kana_yo as u32);
    #[doc(alias = "GDK_KEY_kana_tsu")]
    pub const kana_tsu: Key = Key(ffi::GDK_KEY_kana_tsu as u32);
    #[doc(alias = "GDK_KEY_kana_tu")]
    pub const kana_tu: Key = Key(ffi::GDK_KEY_kana_tu as u32);
    #[doc(alias = "GDK_KEY_prolongedsound")]
    pub const prolongedsound: Key = Key(ffi::GDK_KEY_prolongedsound as u32);
    #[doc(alias = "GDK_KEY_kana_A")]
    pub const kana_A: Key = Key(ffi::GDK_KEY_kana_A as u32);
    #[doc(alias = "GDK_KEY_kana_I")]
    pub const kana_I: Key = Key(ffi::GDK_KEY_kana_I as u32);
    #[doc(alias = "GDK_KEY_kana_U")]
    pub const kana_U: Key = Key(ffi::GDK_KEY_kana_U as u32);
    #[doc(alias = "GDK_KEY_kana_E")]
    pub const kana_E: Key = Key(ffi::GDK_KEY_kana_E as u32);
    #[doc(alias = "GDK_KEY_kana_O")]
    pub const kana_O: Key = Key(ffi::GDK_KEY_kana_O as u32);
    #[doc(alias = "GDK_KEY_kana_KA")]
    pub const kana_KA: Key = Key(ffi::GDK_KEY_kana_KA as u32);
    #[doc(alias = "GDK_KEY_kana_KI")]
    pub const kana_KI: Key = Key(ffi::GDK_KEY_kana_KI as u32);
    #[doc(alias = "GDK_KEY_kana_KU")]
    pub const kana_KU: Key = Key(ffi::GDK_KEY_kana_KU as u32);
    #[doc(alias = "GDK_KEY_kana_KE")]
    pub const kana_KE: Key = Key(ffi::GDK_KEY_kana_KE as u32);
    #[doc(alias = "GDK_KEY_kana_KO")]
    pub const kana_KO: Key = Key(ffi::GDK_KEY_kana_KO as u32);
    #[doc(alias = "GDK_KEY_kana_SA")]
    pub const kana_SA: Key = Key(ffi::GDK_KEY_kana_SA as u32);
    #[doc(alias = "GDK_KEY_kana_SHI")]
    pub const kana_SHI: Key = Key(ffi::GDK_KEY_kana_SHI as u32);
    #[doc(alias = "GDK_KEY_kana_SU")]
    pub const kana_SU: Key = Key(ffi::GDK_KEY_kana_SU as u32);
    #[doc(alias = "GDK_KEY_kana_SE")]
    pub const kana_SE: Key = Key(ffi::GDK_KEY_kana_SE as u32);
    #[doc(alias = "GDK_KEY_kana_SO")]
    pub const kana_SO: Key = Key(ffi::GDK_KEY_kana_SO as u32);
    #[doc(alias = "GDK_KEY_kana_TA")]
    pub const kana_TA: Key = Key(ffi::GDK_KEY_kana_TA as u32);
    #[doc(alias = "GDK_KEY_kana_CHI")]
    pub const kana_CHI: Key = Key(ffi::GDK_KEY_kana_CHI as u32);
    #[doc(alias = "GDK_KEY_kana_TI")]
    pub const kana_TI: Key = Key(ffi::GDK_KEY_kana_TI as u32);
    #[doc(alias = "GDK_KEY_kana_TSU")]
    pub const kana_TSU: Key = Key(ffi::GDK_KEY_kana_TSU as u32);
    #[doc(alias = "GDK_KEY_kana_TU")]
    pub const kana_TU: Key = Key(ffi::GDK_KEY_kana_TU as u32);
    #[doc(alias = "GDK_KEY_kana_TE")]
    pub const kana_TE: Key = Key(ffi::GDK_KEY_kana_TE as u32);
    #[doc(alias = "GDK_KEY_kana_TO")]
    pub const kana_TO: Key = Key(ffi::GDK_KEY_kana_TO as u32);
    #[doc(alias = "GDK_KEY_kana_NA")]
    pub const kana_NA: Key = Key(ffi::GDK_KEY_kana_NA as u32);
    #[doc(alias = "GDK_KEY_kana_NI")]
    pub const kana_NI: Key = Key(ffi::GDK_KEY_kana_NI as u32);
    #[doc(alias = "GDK_KEY_kana_NU")]
    pub const kana_NU: Key = Key(ffi::GDK_KEY_kana_NU as u32);
    #[doc(alias = "GDK_KEY_kana_NE")]
    pub const kana_NE: Key = Key(ffi::GDK_KEY_kana_NE as u32);
    #[doc(alias = "GDK_KEY_kana_NO")]
    pub const kana_NO: Key = Key(ffi::GDK_KEY_kana_NO as u32);
    #[doc(alias = "GDK_KEY_kana_HA")]
    pub const kana_HA: Key = Key(ffi::GDK_KEY_kana_HA as u32);
    #[doc(alias = "GDK_KEY_kana_HI")]
    pub const kana_HI: Key = Key(ffi::GDK_KEY_kana_HI as u32);
    #[doc(alias = "GDK_KEY_kana_FU")]
    pub const kana_FU: Key = Key(ffi::GDK_KEY_kana_FU as u32);
    #[doc(alias = "GDK_KEY_kana_HU")]
    pub const kana_HU: Key = Key(ffi::GDK_KEY_kana_HU as u32);
    #[doc(alias = "GDK_KEY_kana_HE")]
    pub const kana_HE: Key = Key(ffi::GDK_KEY_kana_HE as u32);
    #[doc(alias = "GDK_KEY_kana_HO")]
    pub const kana_HO: Key = Key(ffi::GDK_KEY_kana_HO as u32);
    #[doc(alias = "GDK_KEY_kana_MA")]
    pub const kana_MA: Key = Key(ffi::GDK_KEY_kana_MA as u32);
    #[doc(alias = "GDK_KEY_kana_MI")]
    pub const kana_MI: Key = Key(ffi::GDK_KEY_kana_MI as u32);
    #[doc(alias = "GDK_KEY_kana_MU")]
    pub const kana_MU: Key = Key(ffi::GDK_KEY_kana_MU as u32);
    #[doc(alias = "GDK_KEY_kana_ME")]
    pub const kana_ME: Key = Key(ffi::GDK_KEY_kana_ME as u32);
    #[doc(alias = "GDK_KEY_kana_MO")]
    pub const kana_MO: Key = Key(ffi::GDK_KEY_kana_MO as u32);
    #[doc(alias = "GDK_KEY_kana_YA")]
    pub const kana_YA: Key = Key(ffi::GDK_KEY_kana_YA as u32);
    #[doc(alias = "GDK_KEY_kana_YU")]
    pub const kana_YU: Key = Key(ffi::GDK_KEY_kana_YU as u32);
    #[doc(alias = "GDK_KEY_kana_YO")]
    pub const kana_YO: Key = Key(ffi::GDK_KEY_kana_YO as u32);
    #[doc(alias = "GDK_KEY_kana_RA")]
    pub const kana_RA: Key = Key(ffi::GDK_KEY_kana_RA as u32);
    #[doc(alias = "GDK_KEY_kana_RI")]
    pub const kana_RI: Key = Key(ffi::GDK_KEY_kana_RI as u32);
    #[doc(alias = "GDK_KEY_kana_RU")]
    pub const kana_RU: Key = Key(ffi::GDK_KEY_kana_RU as u32);
    #[doc(alias = "GDK_KEY_kana_RE")]
    pub const kana_RE: Key = Key(ffi::GDK_KEY_kana_RE as u32);
    #[doc(alias = "GDK_KEY_kana_RO")]
    pub const kana_RO: Key = Key(ffi::GDK_KEY_kana_RO as u32);
    #[doc(alias = "GDK_KEY_kana_WA")]
    pub const kana_WA: Key = Key(ffi::GDK_KEY_kana_WA as u32);
    #[doc(alias = "GDK_KEY_kana_N")]
    pub const kana_N: Key = Key(ffi::GDK_KEY_kana_N as u32);
    #[doc(alias = "GDK_KEY_voicedsound")]
    pub const voicedsound: Key = Key(ffi::GDK_KEY_voicedsound as u32);
    #[doc(alias = "GDK_KEY_semivoicedsound")]
    pub const semivoicedsound: Key = Key(ffi::GDK_KEY_semivoicedsound as u32);
    #[doc(alias = "GDK_KEY_kana_switch")]
    pub const kana_switch: Key = Key(ffi::GDK_KEY_kana_switch as u32);
    #[doc(alias = "GDK_KEY_Farsi_0")]
    pub const Farsi_0: Key = Key(ffi::GDK_KEY_Farsi_0 as u32);
    #[doc(alias = "GDK_KEY_Farsi_1")]
    pub const Farsi_1: Key = Key(ffi::GDK_KEY_Farsi_1 as u32);
    #[doc(alias = "GDK_KEY_Farsi_2")]
    pub const Farsi_2: Key = Key(ffi::GDK_KEY_Farsi_2 as u32);
    #[doc(alias = "GDK_KEY_Farsi_3")]
    pub const Farsi_3: Key = Key(ffi::GDK_KEY_Farsi_3 as u32);
    #[doc(alias = "GDK_KEY_Farsi_4")]
    pub const Farsi_4: Key = Key(ffi::GDK_KEY_Farsi_4 as u32);
    #[doc(alias = "GDK_KEY_Farsi_5")]
    pub const Farsi_5: Key = Key(ffi::GDK_KEY_Farsi_5 as u32);
    #[doc(alias = "GDK_KEY_Farsi_6")]
    pub const Farsi_6: Key = Key(ffi::GDK_KEY_Farsi_6 as u32);
    #[doc(alias = "GDK_KEY_Farsi_7")]
    pub const Farsi_7: Key = Key(ffi::GDK_KEY_Farsi_7 as u32);
    #[doc(alias = "GDK_KEY_Farsi_8")]
    pub const Farsi_8: Key = Key(ffi::GDK_KEY_Farsi_8 as u32);
    #[doc(alias = "GDK_KEY_Farsi_9")]
    pub const Farsi_9: Key = Key(ffi::GDK_KEY_Farsi_9 as u32);
    #[doc(alias = "GDK_KEY_Arabic_percent")]
    pub const Arabic_percent: Key = Key(ffi::GDK_KEY_Arabic_percent as u32);
    #[doc(alias = "GDK_KEY_Arabic_superscript_alef")]
    pub const Arabic_superscript_alef: Key = Key(ffi::GDK_KEY_Arabic_superscript_alef as u32);
    #[doc(alias = "GDK_KEY_Arabic_tteh")]
    pub const Arabic_tteh: Key = Key(ffi::GDK_KEY_Arabic_tteh as u32);
    #[doc(alias = "GDK_KEY_Arabic_peh")]
    pub const Arabic_peh: Key = Key(ffi::GDK_KEY_Arabic_peh as u32);
    #[doc(alias = "GDK_KEY_Arabic_tcheh")]
    pub const Arabic_tcheh: Key = Key(ffi::GDK_KEY_Arabic_tcheh as u32);
    #[doc(alias = "GDK_KEY_Arabic_ddal")]
    pub const Arabic_ddal: Key = Key(ffi::GDK_KEY_Arabic_ddal as u32);
    #[doc(alias = "GDK_KEY_Arabic_rreh")]
    pub const Arabic_rreh: Key = Key(ffi::GDK_KEY_Arabic_rreh as u32);
    #[doc(alias = "GDK_KEY_Arabic_comma")]
    pub const Arabic_comma: Key = Key(ffi::GDK_KEY_Arabic_comma as u32);
    #[doc(alias = "GDK_KEY_Arabic_fullstop")]
    pub const Arabic_fullstop: Key = Key(ffi::GDK_KEY_Arabic_fullstop as u32);
    #[doc(alias = "GDK_KEY_Arabic_0")]
    pub const Arabic_0: Key = Key(ffi::GDK_KEY_Arabic_0 as u32);
    #[doc(alias = "GDK_KEY_Arabic_1")]
    pub const Arabic_1: Key = Key(ffi::GDK_KEY_Arabic_1 as u32);
    #[doc(alias = "GDK_KEY_Arabic_2")]
    pub const Arabic_2: Key = Key(ffi::GDK_KEY_Arabic_2 as u32);
    #[doc(alias = "GDK_KEY_Arabic_3")]
    pub const Arabic_3: Key = Key(ffi::GDK_KEY_Arabic_3 as u32);
    #[doc(alias = "GDK_KEY_Arabic_4")]
    pub const Arabic_4: Key = Key(ffi::GDK_KEY_Arabic_4 as u32);
    #[doc(alias = "GDK_KEY_Arabic_5")]
    pub const Arabic_5: Key = Key(ffi::GDK_KEY_Arabic_5 as u32);
    #[doc(alias = "GDK_KEY_Arabic_6")]
    pub const Arabic_6: Key = Key(ffi::GDK_KEY_Arabic_6 as u32);
    #[doc(alias = "GDK_KEY_Arabic_7")]
    pub const Arabic_7: Key = Key(ffi::GDK_KEY_Arabic_7 as u32);
    #[doc(alias = "GDK_KEY_Arabic_8")]
    pub const Arabic_8: Key = Key(ffi::GDK_KEY_Arabic_8 as u32);
    #[doc(alias = "GDK_KEY_Arabic_9")]
    pub const Arabic_9: Key = Key(ffi::GDK_KEY_Arabic_9 as u32);
    #[doc(alias = "GDK_KEY_Arabic_semicolon")]
    pub const Arabic_semicolon: Key = Key(ffi::GDK_KEY_Arabic_semicolon as u32);
    #[doc(alias = "GDK_KEY_Arabic_question_mark")]
    pub const Arabic_question_mark: Key = Key(ffi::GDK_KEY_Arabic_question_mark as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamza")]
    pub const Arabic_hamza: Key = Key(ffi::GDK_KEY_Arabic_hamza as u32);
    #[doc(alias = "GDK_KEY_Arabic_maddaonalef")]
    pub const Arabic_maddaonalef: Key = Key(ffi::GDK_KEY_Arabic_maddaonalef as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaonalef")]
    pub const Arabic_hamzaonalef: Key = Key(ffi::GDK_KEY_Arabic_hamzaonalef as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaonwaw")]
    pub const Arabic_hamzaonwaw: Key = Key(ffi::GDK_KEY_Arabic_hamzaonwaw as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaunderalef")]
    pub const Arabic_hamzaunderalef: Key = Key(ffi::GDK_KEY_Arabic_hamzaunderalef as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamzaonyeh")]
    pub const Arabic_hamzaonyeh: Key = Key(ffi::GDK_KEY_Arabic_hamzaonyeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_alef")]
    pub const Arabic_alef: Key = Key(ffi::GDK_KEY_Arabic_alef as u32);
    #[doc(alias = "GDK_KEY_Arabic_beh")]
    pub const Arabic_beh: Key = Key(ffi::GDK_KEY_Arabic_beh as u32);
    #[doc(alias = "GDK_KEY_Arabic_tehmarbuta")]
    pub const Arabic_tehmarbuta: Key = Key(ffi::GDK_KEY_Arabic_tehmarbuta as u32);
    #[doc(alias = "GDK_KEY_Arabic_teh")]
    pub const Arabic_teh: Key = Key(ffi::GDK_KEY_Arabic_teh as u32);
    #[doc(alias = "GDK_KEY_Arabic_theh")]
    pub const Arabic_theh: Key = Key(ffi::GDK_KEY_Arabic_theh as u32);
    #[doc(alias = "GDK_KEY_Arabic_jeem")]
    pub const Arabic_jeem: Key = Key(ffi::GDK_KEY_Arabic_jeem as u32);
    #[doc(alias = "GDK_KEY_Arabic_hah")]
    pub const Arabic_hah: Key = Key(ffi::GDK_KEY_Arabic_hah as u32);
    #[doc(alias = "GDK_KEY_Arabic_khah")]
    pub const Arabic_khah: Key = Key(ffi::GDK_KEY_Arabic_khah as u32);
    #[doc(alias = "GDK_KEY_Arabic_dal")]
    pub const Arabic_dal: Key = Key(ffi::GDK_KEY_Arabic_dal as u32);
    #[doc(alias = "GDK_KEY_Arabic_thal")]
    pub const Arabic_thal: Key = Key(ffi::GDK_KEY_Arabic_thal as u32);
    #[doc(alias = "GDK_KEY_Arabic_ra")]
    pub const Arabic_ra: Key = Key(ffi::GDK_KEY_Arabic_ra as u32);
    #[doc(alias = "GDK_KEY_Arabic_zain")]
    pub const Arabic_zain: Key = Key(ffi::GDK_KEY_Arabic_zain as u32);
    #[doc(alias = "GDK_KEY_Arabic_seen")]
    pub const Arabic_seen: Key = Key(ffi::GDK_KEY_Arabic_seen as u32);
    #[doc(alias = "GDK_KEY_Arabic_sheen")]
    pub const Arabic_sheen: Key = Key(ffi::GDK_KEY_Arabic_sheen as u32);
    #[doc(alias = "GDK_KEY_Arabic_sad")]
    pub const Arabic_sad: Key = Key(ffi::GDK_KEY_Arabic_sad as u32);
    #[doc(alias = "GDK_KEY_Arabic_dad")]
    pub const Arabic_dad: Key = Key(ffi::GDK_KEY_Arabic_dad as u32);
    #[doc(alias = "GDK_KEY_Arabic_tah")]
    pub const Arabic_tah: Key = Key(ffi::GDK_KEY_Arabic_tah as u32);
    #[doc(alias = "GDK_KEY_Arabic_zah")]
    pub const Arabic_zah: Key = Key(ffi::GDK_KEY_Arabic_zah as u32);
    #[doc(alias = "GDK_KEY_Arabic_ain")]
    pub const Arabic_ain: Key = Key(ffi::GDK_KEY_Arabic_ain as u32);
    #[doc(alias = "GDK_KEY_Arabic_ghain")]
    pub const Arabic_ghain: Key = Key(ffi::GDK_KEY_Arabic_ghain as u32);
    #[doc(alias = "GDK_KEY_Arabic_tatweel")]
    pub const Arabic_tatweel: Key = Key(ffi::GDK_KEY_Arabic_tatweel as u32);
    #[doc(alias = "GDK_KEY_Arabic_feh")]
    pub const Arabic_feh: Key = Key(ffi::GDK_KEY_Arabic_feh as u32);
    #[doc(alias = "GDK_KEY_Arabic_qaf")]
    pub const Arabic_qaf: Key = Key(ffi::GDK_KEY_Arabic_qaf as u32);
    #[doc(alias = "GDK_KEY_Arabic_kaf")]
    pub const Arabic_kaf: Key = Key(ffi::GDK_KEY_Arabic_kaf as u32);
    #[doc(alias = "GDK_KEY_Arabic_lam")]
    pub const Arabic_lam: Key = Key(ffi::GDK_KEY_Arabic_lam as u32);
    #[doc(alias = "GDK_KEY_Arabic_meem")]
    pub const Arabic_meem: Key = Key(ffi::GDK_KEY_Arabic_meem as u32);
    #[doc(alias = "GDK_KEY_Arabic_noon")]
    pub const Arabic_noon: Key = Key(ffi::GDK_KEY_Arabic_noon as u32);
    #[doc(alias = "GDK_KEY_Arabic_ha")]
    pub const Arabic_ha: Key = Key(ffi::GDK_KEY_Arabic_ha as u32);
    #[doc(alias = "GDK_KEY_Arabic_heh")]
    pub const Arabic_heh: Key = Key(ffi::GDK_KEY_Arabic_heh as u32);
    #[doc(alias = "GDK_KEY_Arabic_waw")]
    pub const Arabic_waw: Key = Key(ffi::GDK_KEY_Arabic_waw as u32);
    #[doc(alias = "GDK_KEY_Arabic_alefmaksura")]
    pub const Arabic_alefmaksura: Key = Key(ffi::GDK_KEY_Arabic_alefmaksura as u32);
    #[doc(alias = "GDK_KEY_Arabic_yeh")]
    pub const Arabic_yeh: Key = Key(ffi::GDK_KEY_Arabic_yeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_fathatan")]
    pub const Arabic_fathatan: Key = Key(ffi::GDK_KEY_Arabic_fathatan as u32);
    #[doc(alias = "GDK_KEY_Arabic_dammatan")]
    pub const Arabic_dammatan: Key = Key(ffi::GDK_KEY_Arabic_dammatan as u32);
    #[doc(alias = "GDK_KEY_Arabic_kasratan")]
    pub const Arabic_kasratan: Key = Key(ffi::GDK_KEY_Arabic_kasratan as u32);
    #[doc(alias = "GDK_KEY_Arabic_fatha")]
    pub const Arabic_fatha: Key = Key(ffi::GDK_KEY_Arabic_fatha as u32);
    #[doc(alias = "GDK_KEY_Arabic_damma")]
    pub const Arabic_damma: Key = Key(ffi::GDK_KEY_Arabic_damma as u32);
    #[doc(alias = "GDK_KEY_Arabic_kasra")]
    pub const Arabic_kasra: Key = Key(ffi::GDK_KEY_Arabic_kasra as u32);
    #[doc(alias = "GDK_KEY_Arabic_shadda")]
    pub const Arabic_shadda: Key = Key(ffi::GDK_KEY_Arabic_shadda as u32);
    #[doc(alias = "GDK_KEY_Arabic_sukun")]
    pub const Arabic_sukun: Key = Key(ffi::GDK_KEY_Arabic_sukun as u32);
    #[doc(alias = "GDK_KEY_Arabic_madda_above")]
    pub const Arabic_madda_above: Key = Key(ffi::GDK_KEY_Arabic_madda_above as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamza_above")]
    pub const Arabic_hamza_above: Key = Key(ffi::GDK_KEY_Arabic_hamza_above as u32);
    #[doc(alias = "GDK_KEY_Arabic_hamza_below")]
    pub const Arabic_hamza_below: Key = Key(ffi::GDK_KEY_Arabic_hamza_below as u32);
    #[doc(alias = "GDK_KEY_Arabic_jeh")]
    pub const Arabic_jeh: Key = Key(ffi::GDK_KEY_Arabic_jeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_veh")]
    pub const Arabic_veh: Key = Key(ffi::GDK_KEY_Arabic_veh as u32);
    #[doc(alias = "GDK_KEY_Arabic_keheh")]
    pub const Arabic_keheh: Key = Key(ffi::GDK_KEY_Arabic_keheh as u32);
    #[doc(alias = "GDK_KEY_Arabic_gaf")]
    pub const Arabic_gaf: Key = Key(ffi::GDK_KEY_Arabic_gaf as u32);
    #[doc(alias = "GDK_KEY_Arabic_noon_ghunna")]
    pub const Arabic_noon_ghunna: Key = Key(ffi::GDK_KEY_Arabic_noon_ghunna as u32);
    #[doc(alias = "GDK_KEY_Arabic_heh_doachashmee")]
    pub const Arabic_heh_doachashmee: Key = Key(ffi::GDK_KEY_Arabic_heh_doachashmee as u32);
    #[doc(alias = "GDK_KEY_Farsi_yeh")]
    pub const Farsi_yeh: Key = Key(ffi::GDK_KEY_Farsi_yeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_farsi_yeh")]
    pub const Arabic_farsi_yeh: Key = Key(ffi::GDK_KEY_Arabic_farsi_yeh as u32);
    #[doc(alias = "GDK_KEY_Arabic_yeh_baree")]
    pub const Arabic_yeh_baree: Key = Key(ffi::GDK_KEY_Arabic_yeh_baree as u32);
    #[doc(alias = "GDK_KEY_Arabic_heh_goal")]
    pub const Arabic_heh_goal: Key = Key(ffi::GDK_KEY_Arabic_heh_goal as u32);
    #[doc(alias = "GDK_KEY_Arabic_switch")]
    pub const Arabic_switch: Key = Key(ffi::GDK_KEY_Arabic_switch as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_GHE_bar")]
    pub const Cyrillic_GHE_bar: Key = Key(ffi::GDK_KEY_Cyrillic_GHE_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ghe_bar")]
    pub const Cyrillic_ghe_bar: Key = Key(ffi::GDK_KEY_Cyrillic_ghe_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ZHE_descender")]
    pub const Cyrillic_ZHE_descender: Key = Key(ffi::GDK_KEY_Cyrillic_ZHE_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_zhe_descender")]
    pub const Cyrillic_zhe_descender: Key = Key(ffi::GDK_KEY_Cyrillic_zhe_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_KA_descender")]
    pub const Cyrillic_KA_descender: Key = Key(ffi::GDK_KEY_Cyrillic_KA_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ka_descender")]
    pub const Cyrillic_ka_descender: Key = Key(ffi::GDK_KEY_Cyrillic_ka_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_KA_vertstroke")]
    pub const Cyrillic_KA_vertstroke: Key = Key(ffi::GDK_KEY_Cyrillic_KA_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ka_vertstroke")]
    pub const Cyrillic_ka_vertstroke: Key = Key(ffi::GDK_KEY_Cyrillic_ka_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EN_descender")]
    pub const Cyrillic_EN_descender: Key = Key(ffi::GDK_KEY_Cyrillic_EN_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_en_descender")]
    pub const Cyrillic_en_descender: Key = Key(ffi::GDK_KEY_Cyrillic_en_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U_straight")]
    pub const Cyrillic_U_straight: Key = Key(ffi::GDK_KEY_Cyrillic_U_straight as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u_straight")]
    pub const Cyrillic_u_straight: Key = Key(ffi::GDK_KEY_Cyrillic_u_straight as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U_straight_bar")]
    pub const Cyrillic_U_straight_bar: Key = Key(ffi::GDK_KEY_Cyrillic_U_straight_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u_straight_bar")]
    pub const Cyrillic_u_straight_bar: Key = Key(ffi::GDK_KEY_Cyrillic_u_straight_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_HA_descender")]
    pub const Cyrillic_HA_descender: Key = Key(ffi::GDK_KEY_Cyrillic_HA_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ha_descender")]
    pub const Cyrillic_ha_descender: Key = Key(ffi::GDK_KEY_Cyrillic_ha_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_CHE_descender")]
    pub const Cyrillic_CHE_descender: Key = Key(ffi::GDK_KEY_Cyrillic_CHE_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_che_descender")]
    pub const Cyrillic_che_descender: Key = Key(ffi::GDK_KEY_Cyrillic_che_descender as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_CHE_vertstroke")]
    pub const Cyrillic_CHE_vertstroke: Key = Key(ffi::GDK_KEY_Cyrillic_CHE_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_che_vertstroke")]
    pub const Cyrillic_che_vertstroke: Key = Key(ffi::GDK_KEY_Cyrillic_che_vertstroke as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHHA")]
    pub const Cyrillic_SHHA: Key = Key(ffi::GDK_KEY_Cyrillic_SHHA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_shha")]
    pub const Cyrillic_shha: Key = Key(ffi::GDK_KEY_Cyrillic_shha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SCHWA")]
    pub const Cyrillic_SCHWA: Key = Key(ffi::GDK_KEY_Cyrillic_SCHWA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_schwa")]
    pub const Cyrillic_schwa: Key = Key(ffi::GDK_KEY_Cyrillic_schwa as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_I_macron")]
    pub const Cyrillic_I_macron: Key = Key(ffi::GDK_KEY_Cyrillic_I_macron as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_i_macron")]
    pub const Cyrillic_i_macron: Key = Key(ffi::GDK_KEY_Cyrillic_i_macron as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_O_bar")]
    pub const Cyrillic_O_bar: Key = Key(ffi::GDK_KEY_Cyrillic_O_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_o_bar")]
    pub const Cyrillic_o_bar: Key = Key(ffi::GDK_KEY_Cyrillic_o_bar as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U_macron")]
    pub const Cyrillic_U_macron: Key = Key(ffi::GDK_KEY_Cyrillic_U_macron as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u_macron")]
    pub const Cyrillic_u_macron: Key = Key(ffi::GDK_KEY_Cyrillic_u_macron as u32);
    #[doc(alias = "GDK_KEY_Serbian_dje")]
    pub const Serbian_dje: Key = Key(ffi::GDK_KEY_Serbian_dje as u32);
    #[doc(alias = "GDK_KEY_Macedonia_gje")]
    pub const Macedonia_gje: Key = Key(ffi::GDK_KEY_Macedonia_gje as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_io")]
    pub const Cyrillic_io: Key = Key(ffi::GDK_KEY_Cyrillic_io as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_ie")]
    pub const Ukrainian_ie: Key = Key(ffi::GDK_KEY_Ukrainian_ie as u32);
    #[doc(alias = "GDK_KEY_Ukranian_je")]
    pub const Ukranian_je: Key = Key(ffi::GDK_KEY_Ukranian_je as u32);
    #[doc(alias = "GDK_KEY_Macedonia_dse")]
    pub const Macedonia_dse: Key = Key(ffi::GDK_KEY_Macedonia_dse as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_i")]
    pub const Ukrainian_i: Key = Key(ffi::GDK_KEY_Ukrainian_i as u32);
    #[doc(alias = "GDK_KEY_Ukranian_i")]
    pub const Ukranian_i: Key = Key(ffi::GDK_KEY_Ukranian_i as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_yi")]
    pub const Ukrainian_yi: Key = Key(ffi::GDK_KEY_Ukrainian_yi as u32);
    #[doc(alias = "GDK_KEY_Ukranian_yi")]
    pub const Ukranian_yi: Key = Key(ffi::GDK_KEY_Ukranian_yi as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_je")]
    pub const Cyrillic_je: Key = Key(ffi::GDK_KEY_Cyrillic_je as u32);
    #[doc(alias = "GDK_KEY_Serbian_je")]
    pub const Serbian_je: Key = Key(ffi::GDK_KEY_Serbian_je as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_lje")]
    pub const Cyrillic_lje: Key = Key(ffi::GDK_KEY_Cyrillic_lje as u32);
    #[doc(alias = "GDK_KEY_Serbian_lje")]
    pub const Serbian_lje: Key = Key(ffi::GDK_KEY_Serbian_lje as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_nje")]
    pub const Cyrillic_nje: Key = Key(ffi::GDK_KEY_Cyrillic_nje as u32);
    #[doc(alias = "GDK_KEY_Serbian_nje")]
    pub const Serbian_nje: Key = Key(ffi::GDK_KEY_Serbian_nje as u32);
    #[doc(alias = "GDK_KEY_Serbian_tshe")]
    pub const Serbian_tshe: Key = Key(ffi::GDK_KEY_Serbian_tshe as u32);
    #[doc(alias = "GDK_KEY_Macedonia_kje")]
    pub const Macedonia_kje: Key = Key(ffi::GDK_KEY_Macedonia_kje as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_ghe_with_upturn")]
    pub const Ukrainian_ghe_with_upturn: Key = Key(ffi::GDK_KEY_Ukrainian_ghe_with_upturn as u32);
    #[doc(alias = "GDK_KEY_Byelorussian_shortu")]
    pub const Byelorussian_shortu: Key = Key(ffi::GDK_KEY_Byelorussian_shortu as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_dzhe")]
    pub const Cyrillic_dzhe: Key = Key(ffi::GDK_KEY_Cyrillic_dzhe as u32);
    #[doc(alias = "GDK_KEY_Serbian_dze")]
    pub const Serbian_dze: Key = Key(ffi::GDK_KEY_Serbian_dze as u32);
    #[doc(alias = "GDK_KEY_numerosign")]
    pub const numerosign: Key = Key(ffi::GDK_KEY_numerosign as u32);
    #[doc(alias = "GDK_KEY_Serbian_DJE")]
    pub const Serbian_DJE: Key = Key(ffi::GDK_KEY_Serbian_DJE as u32);
    #[doc(alias = "GDK_KEY_Macedonia_GJE")]
    pub const Macedonia_GJE: Key = Key(ffi::GDK_KEY_Macedonia_GJE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_IO")]
    pub const Cyrillic_IO: Key = Key(ffi::GDK_KEY_Cyrillic_IO as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_IE")]
    pub const Ukrainian_IE: Key = Key(ffi::GDK_KEY_Ukrainian_IE as u32);
    #[doc(alias = "GDK_KEY_Ukranian_JE")]
    pub const Ukranian_JE: Key = Key(ffi::GDK_KEY_Ukranian_JE as u32);
    #[doc(alias = "GDK_KEY_Macedonia_DSE")]
    pub const Macedonia_DSE: Key = Key(ffi::GDK_KEY_Macedonia_DSE as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_I")]
    pub const Ukrainian_I: Key = Key(ffi::GDK_KEY_Ukrainian_I as u32);
    #[doc(alias = "GDK_KEY_Ukranian_I")]
    pub const Ukranian_I: Key = Key(ffi::GDK_KEY_Ukranian_I as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_YI")]
    pub const Ukrainian_YI: Key = Key(ffi::GDK_KEY_Ukrainian_YI as u32);
    #[doc(alias = "GDK_KEY_Ukranian_YI")]
    pub const Ukranian_YI: Key = Key(ffi::GDK_KEY_Ukranian_YI as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_JE")]
    pub const Cyrillic_JE: Key = Key(ffi::GDK_KEY_Cyrillic_JE as u32);
    #[doc(alias = "GDK_KEY_Serbian_JE")]
    pub const Serbian_JE: Key = Key(ffi::GDK_KEY_Serbian_JE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_LJE")]
    pub const Cyrillic_LJE: Key = Key(ffi::GDK_KEY_Cyrillic_LJE as u32);
    #[doc(alias = "GDK_KEY_Serbian_LJE")]
    pub const Serbian_LJE: Key = Key(ffi::GDK_KEY_Serbian_LJE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_NJE")]
    pub const Cyrillic_NJE: Key = Key(ffi::GDK_KEY_Cyrillic_NJE as u32);
    #[doc(alias = "GDK_KEY_Serbian_NJE")]
    pub const Serbian_NJE: Key = Key(ffi::GDK_KEY_Serbian_NJE as u32);
    #[doc(alias = "GDK_KEY_Serbian_TSHE")]
    pub const Serbian_TSHE: Key = Key(ffi::GDK_KEY_Serbian_TSHE as u32);
    #[doc(alias = "GDK_KEY_Macedonia_KJE")]
    pub const Macedonia_KJE: Key = Key(ffi::GDK_KEY_Macedonia_KJE as u32);
    #[doc(alias = "GDK_KEY_Ukrainian_GHE_WITH_UPTURN")]
    pub const Ukrainian_GHE_WITH_UPTURN: Key = Key(ffi::GDK_KEY_Ukrainian_GHE_WITH_UPTURN as u32);
    #[doc(alias = "GDK_KEY_Byelorussian_SHORTU")]
    pub const Byelorussian_SHORTU: Key = Key(ffi::GDK_KEY_Byelorussian_SHORTU as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_DZHE")]
    pub const Cyrillic_DZHE: Key = Key(ffi::GDK_KEY_Cyrillic_DZHE as u32);
    #[doc(alias = "GDK_KEY_Serbian_DZE")]
    pub const Serbian_DZE: Key = Key(ffi::GDK_KEY_Serbian_DZE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_yu")]
    pub const Cyrillic_yu: Key = Key(ffi::GDK_KEY_Cyrillic_yu as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_a")]
    pub const Cyrillic_a: Key = Key(ffi::GDK_KEY_Cyrillic_a as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_be")]
    pub const Cyrillic_be: Key = Key(ffi::GDK_KEY_Cyrillic_be as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_tse")]
    pub const Cyrillic_tse: Key = Key(ffi::GDK_KEY_Cyrillic_tse as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_de")]
    pub const Cyrillic_de: Key = Key(ffi::GDK_KEY_Cyrillic_de as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ie")]
    pub const Cyrillic_ie: Key = Key(ffi::GDK_KEY_Cyrillic_ie as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ef")]
    pub const Cyrillic_ef: Key = Key(ffi::GDK_KEY_Cyrillic_ef as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ghe")]
    pub const Cyrillic_ghe: Key = Key(ffi::GDK_KEY_Cyrillic_ghe as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ha")]
    pub const Cyrillic_ha: Key = Key(ffi::GDK_KEY_Cyrillic_ha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_i")]
    pub const Cyrillic_i: Key = Key(ffi::GDK_KEY_Cyrillic_i as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_shorti")]
    pub const Cyrillic_shorti: Key = Key(ffi::GDK_KEY_Cyrillic_shorti as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ka")]
    pub const Cyrillic_ka: Key = Key(ffi::GDK_KEY_Cyrillic_ka as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_el")]
    pub const Cyrillic_el: Key = Key(ffi::GDK_KEY_Cyrillic_el as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_em")]
    pub const Cyrillic_em: Key = Key(ffi::GDK_KEY_Cyrillic_em as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_en")]
    pub const Cyrillic_en: Key = Key(ffi::GDK_KEY_Cyrillic_en as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_o")]
    pub const Cyrillic_o: Key = Key(ffi::GDK_KEY_Cyrillic_o as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_pe")]
    pub const Cyrillic_pe: Key = Key(ffi::GDK_KEY_Cyrillic_pe as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ya")]
    pub const Cyrillic_ya: Key = Key(ffi::GDK_KEY_Cyrillic_ya as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_er")]
    pub const Cyrillic_er: Key = Key(ffi::GDK_KEY_Cyrillic_er as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_es")]
    pub const Cyrillic_es: Key = Key(ffi::GDK_KEY_Cyrillic_es as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_te")]
    pub const Cyrillic_te: Key = Key(ffi::GDK_KEY_Cyrillic_te as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_u")]
    pub const Cyrillic_u: Key = Key(ffi::GDK_KEY_Cyrillic_u as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_zhe")]
    pub const Cyrillic_zhe: Key = Key(ffi::GDK_KEY_Cyrillic_zhe as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ve")]
    pub const Cyrillic_ve: Key = Key(ffi::GDK_KEY_Cyrillic_ve as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_softsign")]
    pub const Cyrillic_softsign: Key = Key(ffi::GDK_KEY_Cyrillic_softsign as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_yeru")]
    pub const Cyrillic_yeru: Key = Key(ffi::GDK_KEY_Cyrillic_yeru as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ze")]
    pub const Cyrillic_ze: Key = Key(ffi::GDK_KEY_Cyrillic_ze as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_sha")]
    pub const Cyrillic_sha: Key = Key(ffi::GDK_KEY_Cyrillic_sha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_e")]
    pub const Cyrillic_e: Key = Key(ffi::GDK_KEY_Cyrillic_e as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_shcha")]
    pub const Cyrillic_shcha: Key = Key(ffi::GDK_KEY_Cyrillic_shcha as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_che")]
    pub const Cyrillic_che: Key = Key(ffi::GDK_KEY_Cyrillic_che as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_hardsign")]
    pub const Cyrillic_hardsign: Key = Key(ffi::GDK_KEY_Cyrillic_hardsign as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_YU")]
    pub const Cyrillic_YU: Key = Key(ffi::GDK_KEY_Cyrillic_YU as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_A")]
    pub const Cyrillic_A: Key = Key(ffi::GDK_KEY_Cyrillic_A as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_BE")]
    pub const Cyrillic_BE: Key = Key(ffi::GDK_KEY_Cyrillic_BE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_TSE")]
    pub const Cyrillic_TSE: Key = Key(ffi::GDK_KEY_Cyrillic_TSE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_DE")]
    pub const Cyrillic_DE: Key = Key(ffi::GDK_KEY_Cyrillic_DE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_IE")]
    pub const Cyrillic_IE: Key = Key(ffi::GDK_KEY_Cyrillic_IE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EF")]
    pub const Cyrillic_EF: Key = Key(ffi::GDK_KEY_Cyrillic_EF as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_GHE")]
    pub const Cyrillic_GHE: Key = Key(ffi::GDK_KEY_Cyrillic_GHE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_HA")]
    pub const Cyrillic_HA: Key = Key(ffi::GDK_KEY_Cyrillic_HA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_I")]
    pub const Cyrillic_I: Key = Key(ffi::GDK_KEY_Cyrillic_I as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHORTI")]
    pub const Cyrillic_SHORTI: Key = Key(ffi::GDK_KEY_Cyrillic_SHORTI as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_KA")]
    pub const Cyrillic_KA: Key = Key(ffi::GDK_KEY_Cyrillic_KA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EL")]
    pub const Cyrillic_EL: Key = Key(ffi::GDK_KEY_Cyrillic_EL as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EM")]
    pub const Cyrillic_EM: Key = Key(ffi::GDK_KEY_Cyrillic_EM as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_EN")]
    pub const Cyrillic_EN: Key = Key(ffi::GDK_KEY_Cyrillic_EN as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_O")]
    pub const Cyrillic_O: Key = Key(ffi::GDK_KEY_Cyrillic_O as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_PE")]
    pub const Cyrillic_PE: Key = Key(ffi::GDK_KEY_Cyrillic_PE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_YA")]
    pub const Cyrillic_YA: Key = Key(ffi::GDK_KEY_Cyrillic_YA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ER")]
    pub const Cyrillic_ER: Key = Key(ffi::GDK_KEY_Cyrillic_ER as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ES")]
    pub const Cyrillic_ES: Key = Key(ffi::GDK_KEY_Cyrillic_ES as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_TE")]
    pub const Cyrillic_TE: Key = Key(ffi::GDK_KEY_Cyrillic_TE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_U")]
    pub const Cyrillic_U: Key = Key(ffi::GDK_KEY_Cyrillic_U as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ZHE")]
    pub const Cyrillic_ZHE: Key = Key(ffi::GDK_KEY_Cyrillic_ZHE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_VE")]
    pub const Cyrillic_VE: Key = Key(ffi::GDK_KEY_Cyrillic_VE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SOFTSIGN")]
    pub const Cyrillic_SOFTSIGN: Key = Key(ffi::GDK_KEY_Cyrillic_SOFTSIGN as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_YERU")]
    pub const Cyrillic_YERU: Key = Key(ffi::GDK_KEY_Cyrillic_YERU as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_ZE")]
    pub const Cyrillic_ZE: Key = Key(ffi::GDK_KEY_Cyrillic_ZE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHA")]
    pub const Cyrillic_SHA: Key = Key(ffi::GDK_KEY_Cyrillic_SHA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_E")]
    pub const Cyrillic_E: Key = Key(ffi::GDK_KEY_Cyrillic_E as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_SHCHA")]
    pub const Cyrillic_SHCHA: Key = Key(ffi::GDK_KEY_Cyrillic_SHCHA as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_CHE")]
    pub const Cyrillic_CHE: Key = Key(ffi::GDK_KEY_Cyrillic_CHE as u32);
    #[doc(alias = "GDK_KEY_Cyrillic_HARDSIGN")]
    pub const Cyrillic_HARDSIGN: Key = Key(ffi::GDK_KEY_Cyrillic_HARDSIGN as u32);
    #[doc(alias = "GDK_KEY_Greek_ALPHAaccent")]
    pub const Greek_ALPHAaccent: Key = Key(ffi::GDK_KEY_Greek_ALPHAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_EPSILONaccent")]
    pub const Greek_EPSILONaccent: Key = Key(ffi::GDK_KEY_Greek_EPSILONaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_ETAaccent")]
    pub const Greek_ETAaccent: Key = Key(ffi::GDK_KEY_Greek_ETAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTAaccent")]
    pub const Greek_IOTAaccent: Key = Key(ffi::GDK_KEY_Greek_IOTAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTAdieresis")]
    pub const Greek_IOTAdieresis: Key = Key(ffi::GDK_KEY_Greek_IOTAdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTAdiaeresis")]
    pub const Greek_IOTAdiaeresis: Key = Key(ffi::GDK_KEY_Greek_IOTAdiaeresis as u32);
    #[doc(alias = "GDK_KEY_Greek_OMICRONaccent")]
    pub const Greek_OMICRONaccent: Key = Key(ffi::GDK_KEY_Greek_OMICRONaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_UPSILONaccent")]
    pub const Greek_UPSILONaccent: Key = Key(ffi::GDK_KEY_Greek_UPSILONaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_UPSILONdieresis")]
    pub const Greek_UPSILONdieresis: Key = Key(ffi::GDK_KEY_Greek_UPSILONdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_OMEGAaccent")]
    pub const Greek_OMEGAaccent: Key = Key(ffi::GDK_KEY_Greek_OMEGAaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_accentdieresis")]
    pub const Greek_accentdieresis: Key = Key(ffi::GDK_KEY_Greek_accentdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_horizbar")]
    pub const Greek_horizbar: Key = Key(ffi::GDK_KEY_Greek_horizbar as u32);
    #[doc(alias = "GDK_KEY_Greek_alphaaccent")]
    pub const Greek_alphaaccent: Key = Key(ffi::GDK_KEY_Greek_alphaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_epsilonaccent")]
    pub const Greek_epsilonaccent: Key = Key(ffi::GDK_KEY_Greek_epsilonaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_etaaccent")]
    pub const Greek_etaaccent: Key = Key(ffi::GDK_KEY_Greek_etaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_iotaaccent")]
    pub const Greek_iotaaccent: Key = Key(ffi::GDK_KEY_Greek_iotaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_iotadieresis")]
    pub const Greek_iotadieresis: Key = Key(ffi::GDK_KEY_Greek_iotadieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_iotaaccentdieresis")]
    pub const Greek_iotaaccentdieresis: Key = Key(ffi::GDK_KEY_Greek_iotaaccentdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_omicronaccent")]
    pub const Greek_omicronaccent: Key = Key(ffi::GDK_KEY_Greek_omicronaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilonaccent")]
    pub const Greek_upsilonaccent: Key = Key(ffi::GDK_KEY_Greek_upsilonaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilondieresis")]
    pub const Greek_upsilondieresis: Key = Key(ffi::GDK_KEY_Greek_upsilondieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilonaccentdieresis")]
    pub const Greek_upsilonaccentdieresis: Key =
        Key(ffi::GDK_KEY_Greek_upsilonaccentdieresis as u32);
    #[doc(alias = "GDK_KEY_Greek_omegaaccent")]
    pub const Greek_omegaaccent: Key = Key(ffi::GDK_KEY_Greek_omegaaccent as u32);
    #[doc(alias = "GDK_KEY_Greek_ALPHA")]
    pub const Greek_ALPHA: Key = Key(ffi::GDK_KEY_Greek_ALPHA as u32);
    #[doc(alias = "GDK_KEY_Greek_BETA")]
    pub const Greek_BETA: Key = Key(ffi::GDK_KEY_Greek_BETA as u32);
    #[doc(alias = "GDK_KEY_Greek_GAMMA")]
    pub const Greek_GAMMA: Key = Key(ffi::GDK_KEY_Greek_GAMMA as u32);
    #[doc(alias = "GDK_KEY_Greek_DELTA")]
    pub const Greek_DELTA: Key = Key(ffi::GDK_KEY_Greek_DELTA as u32);
    #[doc(alias = "GDK_KEY_Greek_EPSILON")]
    pub const Greek_EPSILON: Key = Key(ffi::GDK_KEY_Greek_EPSILON as u32);
    #[doc(alias = "GDK_KEY_Greek_ZETA")]
    pub const Greek_ZETA: Key = Key(ffi::GDK_KEY_Greek_ZETA as u32);
    #[doc(alias = "GDK_KEY_Greek_ETA")]
    pub const Greek_ETA: Key = Key(ffi::GDK_KEY_Greek_ETA as u32);
    #[doc(alias = "GDK_KEY_Greek_THETA")]
    pub const Greek_THETA: Key = Key(ffi::GDK_KEY_Greek_THETA as u32);
    #[doc(alias = "GDK_KEY_Greek_IOTA")]
    pub const Greek_IOTA: Key = Key(ffi::GDK_KEY_Greek_IOTA as u32);
    #[doc(alias = "GDK_KEY_Greek_KAPPA")]
    pub const Greek_KAPPA: Key = Key(ffi::GDK_KEY_Greek_KAPPA as u32);
    #[doc(alias = "GDK_KEY_Greek_LAMDA")]
    pub const Greek_LAMDA: Key = Key(ffi::GDK_KEY_Greek_LAMDA as u32);
    #[doc(alias = "GDK_KEY_Greek_LAMBDA")]
    pub const Greek_LAMBDA: Key = Key(ffi::GDK_KEY_Greek_LAMBDA as u32);
    #[doc(alias = "GDK_KEY_Greek_MU")]
    pub const Greek_MU: Key = Key(ffi::GDK_KEY_Greek_MU as u32);
    #[doc(alias = "GDK_KEY_Greek_NU")]
    pub const Greek_NU: Key = Key(ffi::GDK_KEY_Greek_NU as u32);
    #[doc(alias = "GDK_KEY_Greek_XI")]
    pub const Greek_XI: Key = Key(ffi::GDK_KEY_Greek_XI as u32);
    #[doc(alias = "GDK_KEY_Greek_OMICRON")]
    pub const Greek_OMICRON: Key = Key(ffi::GDK_KEY_Greek_OMICRON as u32);
    #[doc(alias = "GDK_KEY_Greek_PI")]
    pub const Greek_PI: Key = Key(ffi::GDK_KEY_Greek_PI as u32);
    #[doc(alias = "GDK_KEY_Greek_RHO")]
    pub const Greek_RHO: Key = Key(ffi::GDK_KEY_Greek_RHO as u32);
    #[doc(alias = "GDK_KEY_Greek_SIGMA")]
    pub const Greek_SIGMA: Key = Key(ffi::GDK_KEY_Greek_SIGMA as u32);
    #[doc(alias = "GDK_KEY_Greek_TAU")]
    pub const Greek_TAU: Key = Key(ffi::GDK_KEY_Greek_TAU as u32);
    #[doc(alias = "GDK_KEY_Greek_UPSILON")]
    pub const Greek_UPSILON: Key = Key(ffi::GDK_KEY_Greek_UPSILON as u32);
    #[doc(alias = "GDK_KEY_Greek_PHI")]
    pub const Greek_PHI: Key = Key(ffi::GDK_KEY_Greek_PHI as u32);
    #[doc(alias = "GDK_KEY_Greek_CHI")]
    pub const Greek_CHI: Key = Key(ffi::GDK_KEY_Greek_CHI as u32);
    #[doc(alias = "GDK_KEY_Greek_PSI")]
    pub const Greek_PSI: Key = Key(ffi::GDK_KEY_Greek_PSI as u32);
    #[doc(alias = "GDK_KEY_Greek_OMEGA")]
    pub const Greek_OMEGA: Key = Key(ffi::GDK_KEY_Greek_OMEGA as u32);
    #[doc(alias = "GDK_KEY_Greek_alpha")]
    pub const Greek_alpha: Key = Key(ffi::GDK_KEY_Greek_alpha as u32);
    #[doc(alias = "GDK_KEY_Greek_beta")]
    pub const Greek_beta: Key = Key(ffi::GDK_KEY_Greek_beta as u32);
    #[doc(alias = "GDK_KEY_Greek_gamma")]
    pub const Greek_gamma: Key = Key(ffi::GDK_KEY_Greek_gamma as u32);
    #[doc(alias = "GDK_KEY_Greek_delta")]
    pub const Greek_delta: Key = Key(ffi::GDK_KEY_Greek_delta as u32);
    #[doc(alias = "GDK_KEY_Greek_epsilon")]
    pub const Greek_epsilon: Key = Key(ffi::GDK_KEY_Greek_epsilon as u32);
    #[doc(alias = "GDK_KEY_Greek_zeta")]
    pub const Greek_zeta: Key = Key(ffi::GDK_KEY_Greek_zeta as u32);
    #[doc(alias = "GDK_KEY_Greek_eta")]
    pub const Greek_eta: Key = Key(ffi::GDK_KEY_Greek_eta as u32);
    #[doc(alias = "GDK_KEY_Greek_theta")]
    pub const Greek_theta: Key = Key(ffi::GDK_KEY_Greek_theta as u32);
    #[doc(alias = "GDK_KEY_Greek_iota")]
    pub const Greek_iota: Key = Key(ffi::GDK_KEY_Greek_iota as u32);
    #[doc(alias = "GDK_KEY_Greek_kappa")]
    pub const Greek_kappa: Key = Key(ffi::GDK_KEY_Greek_kappa as u32);
    #[doc(alias = "GDK_KEY_Greek_lamda")]
    pub const Greek_lamda: Key = Key(ffi::GDK_KEY_Greek_lamda as u32);
    #[doc(alias = "GDK_KEY_Greek_lambda")]
    pub const Greek_lambda: Key = Key(ffi::GDK_KEY_Greek_lambda as u32);
    #[doc(alias = "GDK_KEY_Greek_mu")]
    pub const Greek_mu: Key = Key(ffi::GDK_KEY_Greek_mu as u32);
    #[doc(alias = "GDK_KEY_Greek_nu")]
    pub const Greek_nu: Key = Key(ffi::GDK_KEY_Greek_nu as u32);
    #[doc(alias = "GDK_KEY_Greek_xi")]
    pub const Greek_xi: Key = Key(ffi::GDK_KEY_Greek_xi as u32);
    #[doc(alias = "GDK_KEY_Greek_omicron")]
    pub const Greek_omicron: Key = Key(ffi::GDK_KEY_Greek_omicron as u32);
    #[doc(alias = "GDK_KEY_Greek_pi")]
    pub const Greek_pi: Key = Key(ffi::GDK_KEY_Greek_pi as u32);
    #[doc(alias = "GDK_KEY_Greek_rho")]
    pub const Greek_rho: Key = Key(ffi::GDK_KEY_Greek_rho as u32);
    #[doc(alias = "GDK_KEY_Greek_sigma")]
    pub const Greek_sigma: Key = Key(ffi::GDK_KEY_Greek_sigma as u32);
    #[doc(alias = "GDK_KEY_Greek_finalsmallsigma")]
    pub const Greek_finalsmallsigma: Key = Key(ffi::GDK_KEY_Greek_finalsmallsigma as u32);
    #[doc(alias = "GDK_KEY_Greek_tau")]
    pub const Greek_tau: Key = Key(ffi::GDK_KEY_Greek_tau as u32);
    #[doc(alias = "GDK_KEY_Greek_upsilon")]
    pub const Greek_upsilon: Key = Key(ffi::GDK_KEY_Greek_upsilon as u32);
    #[doc(alias = "GDK_KEY_Greek_phi")]
    pub const Greek_phi: Key = Key(ffi::GDK_KEY_Greek_phi as u32);
    #[doc(alias = "GDK_KEY_Greek_chi")]
    pub const Greek_chi: Key = Key(ffi::GDK_KEY_Greek_chi as u32);
    #[doc(alias = "GDK_KEY_Greek_psi")]
    pub const Greek_psi: Key = Key(ffi::GDK_KEY_Greek_psi as u32);
    #[doc(alias = "GDK_KEY_Greek_omega")]
    pub const Greek_omega: Key = Key(ffi::GDK_KEY_Greek_omega as u32);
    #[doc(alias = "GDK_KEY_Greek_switch")]
    pub const Greek_switch: Key = Key(ffi::GDK_KEY_Greek_switch as u32);
    #[doc(alias = "GDK_KEY_leftradical")]
    pub const leftradical: Key = Key(ffi::GDK_KEY_leftradical as u32);
    #[doc(alias = "GDK_KEY_topleftradical")]
    pub const topleftradical: Key = Key(ffi::GDK_KEY_topleftradical as u32);
    #[doc(alias = "GDK_KEY_horizconnector")]
    pub const horizconnector: Key = Key(ffi::GDK_KEY_horizconnector as u32);
    #[doc(alias = "GDK_KEY_topintegral")]
    pub const topintegral: Key = Key(ffi::GDK_KEY_topintegral as u32);
    #[doc(alias = "GDK_KEY_botintegral")]
    pub const botintegral: Key = Key(ffi::GDK_KEY_botintegral as u32);
    #[doc(alias = "GDK_KEY_vertconnector")]
    pub const vertconnector: Key = Key(ffi::GDK_KEY_vertconnector as u32);
    #[doc(alias = "GDK_KEY_topleftsqbracket")]
    pub const topleftsqbracket: Key = Key(ffi::GDK_KEY_topleftsqbracket as u32);
    #[doc(alias = "GDK_KEY_botleftsqbracket")]
    pub const botleftsqbracket: Key = Key(ffi::GDK_KEY_botleftsqbracket as u32);
    #[doc(alias = "GDK_KEY_toprightsqbracket")]
    pub const toprightsqbracket: Key = Key(ffi::GDK_KEY_toprightsqbracket as u32);
    #[doc(alias = "GDK_KEY_botrightsqbracket")]
    pub const botrightsqbracket: Key = Key(ffi::GDK_KEY_botrightsqbracket as u32);
    #[doc(alias = "GDK_KEY_topleftparens")]
    pub const topleftparens: Key = Key(ffi::GDK_KEY_topleftparens as u32);
    #[doc(alias = "GDK_KEY_botleftparens")]
    pub const botleftparens: Key = Key(ffi::GDK_KEY_botleftparens as u32);
    #[doc(alias = "GDK_KEY_toprightparens")]
    pub const toprightparens: Key = Key(ffi::GDK_KEY_toprightparens as u32);
    #[doc(alias = "GDK_KEY_botrightparens")]
    pub const botrightparens: Key = Key(ffi::GDK_KEY_botrightparens as u32);
    #[doc(alias = "GDK_KEY_leftmiddlecurlybrace")]
    pub const leftmiddlecurlybrace: Key = Key(ffi::GDK_KEY_leftmiddlecurlybrace as u32);
    #[doc(alias = "GDK_KEY_rightmiddlecurlybrace")]
    pub const rightmiddlecurlybrace: Key = Key(ffi::GDK_KEY_rightmiddlecurlybrace as u32);
    #[doc(alias = "GDK_KEY_topleftsummation")]
    pub const topleftsummation: Key = Key(ffi::GDK_KEY_topleftsummation as u32);
    #[doc(alias = "GDK_KEY_botleftsummation")]
    pub const botleftsummation: Key = Key(ffi::GDK_KEY_botleftsummation as u32);
    #[doc(alias = "GDK_KEY_topvertsummationconnector")]
    pub const topvertsummationconnector: Key = Key(ffi::GDK_KEY_topvertsummationconnector as u32);
    #[doc(alias = "GDK_KEY_botvertsummationconnector")]
    pub const botvertsummationconnector: Key = Key(ffi::GDK_KEY_botvertsummationconnector as u32);
    #[doc(alias = "GDK_KEY_toprightsummation")]
    pub const toprightsummation: Key = Key(ffi::GDK_KEY_toprightsummation as u32);
    #[doc(alias = "GDK_KEY_botrightsummation")]
    pub const botrightsummation: Key = Key(ffi::GDK_KEY_botrightsummation as u32);
    #[doc(alias = "GDK_KEY_rightmiddlesummation")]
    pub const rightmiddlesummation: Key = Key(ffi::GDK_KEY_rightmiddlesummation as u32);
    #[doc(alias = "GDK_KEY_lessthanequal")]
    pub const lessthanequal: Key = Key(ffi::GDK_KEY_lessthanequal as u32);
    #[doc(alias = "GDK_KEY_notequal")]
    pub const notequal: Key = Key(ffi::GDK_KEY_notequal as u32);
    #[doc(alias = "GDK_KEY_greaterthanequal")]
    pub const greaterthanequal: Key = Key(ffi::GDK_KEY_greaterthanequal as u32);
    #[doc(alias = "GDK_KEY_integral")]
    pub const integral: Key = Key(ffi::GDK_KEY_integral as u32);
    #[doc(alias = "GDK_KEY_therefore")]
    pub const therefore: Key = Key(ffi::GDK_KEY_therefore as u32);
    #[doc(alias = "GDK_KEY_variation")]
    pub const variation: Key = Key(ffi::GDK_KEY_variation as u32);
    #[doc(alias = "GDK_KEY_infinity")]
    pub const infinity: Key = Key(ffi::GDK_KEY_infinity as u32);
    #[doc(alias = "GDK_KEY_nabla")]
    pub const nabla: Key = Key(ffi::GDK_KEY_nabla as u32);
    #[doc(alias = "GDK_KEY_approximate")]
    pub const approximate: Key = Key(ffi::GDK_KEY_approximate as u32);
    #[doc(alias = "GDK_KEY_similarequal")]
    pub const similarequal: Key = Key(ffi::GDK_KEY_similarequal as u32);
    #[doc(alias = "GDK_KEY_ifonlyif")]
    pub const ifonlyif: Key = Key(ffi::GDK_KEY_ifonlyif as u32);
    #[doc(alias = "GDK_KEY_implies")]
    pub const implies: Key = Key(ffi::GDK_KEY_implies as u32);
    #[doc(alias = "GDK_KEY_identical")]
    pub const identical: Key = Key(ffi::GDK_KEY_identical as u32);
    #[doc(alias = "GDK_KEY_radical")]
    pub const radical: Key = Key(ffi::GDK_KEY_radical as u32);
    #[doc(alias = "GDK_KEY_includedin")]
    pub const includedin: Key = Key(ffi::GDK_KEY_includedin as u32);
    #[doc(alias = "GDK_KEY_includes")]
    pub const includes: Key = Key(ffi::GDK_KEY_includes as u32);
    #[doc(alias = "GDK_KEY_intersection")]
    pub const intersection: Key = Key(ffi::GDK_KEY_intersection as u32);
    #[doc(alias = "GDK_KEY_union")]
    pub const union: Key = Key(ffi::GDK_KEY_union as u32);
    #[doc(alias = "GDK_KEY_logicaland")]
    pub const logicaland: Key = Key(ffi::GDK_KEY_logicaland as u32);
    #[doc(alias = "GDK_KEY_logicalor")]
    pub const logicalor: Key = Key(ffi::GDK_KEY_logicalor as u32);
    #[doc(alias = "GDK_KEY_partialderivative")]
    pub const partialderivative: Key = Key(ffi::GDK_KEY_partialderivative as u32);
    #[doc(alias = "GDK_KEY_function")]
    pub const function: Key = Key(ffi::GDK_KEY_function as u32);
    #[doc(alias = "GDK_KEY_leftarrow")]
    pub const leftarrow: Key = Key(ffi::GDK_KEY_leftarrow as u32);
    #[doc(alias = "GDK_KEY_uparrow")]
    pub const uparrow: Key = Key(ffi::GDK_KEY_uparrow as u32);
    #[doc(alias = "GDK_KEY_rightarrow")]
    pub const rightarrow: Key = Key(ffi::GDK_KEY_rightarrow as u32);
    #[doc(alias = "GDK_KEY_downarrow")]
    pub const downarrow: Key = Key(ffi::GDK_KEY_downarrow as u32);
    #[doc(alias = "GDK_KEY_blank")]
    pub const blank: Key = Key(ffi::GDK_KEY_blank as u32);
    #[doc(alias = "GDK_KEY_soliddiamond")]
    pub const soliddiamond: Key = Key(ffi::GDK_KEY_soliddiamond as u32);
    #[doc(alias = "GDK_KEY_checkerboard")]
    pub const checkerboard: Key = Key(ffi::GDK_KEY_checkerboard as u32);
    #[doc(alias = "GDK_KEY_ht")]
    pub const ht: Key = Key(ffi::GDK_KEY_ht as u32);
    #[doc(alias = "GDK_KEY_ff")]
    pub const ff: Key = Key(ffi::GDK_KEY_ff as u32);
    #[doc(alias = "GDK_KEY_cr")]
    pub const cr: Key = Key(ffi::GDK_KEY_cr as u32);
    #[doc(alias = "GDK_KEY_lf")]
    pub const lf: Key = Key(ffi::GDK_KEY_lf as u32);
    #[doc(alias = "GDK_KEY_nl")]
    pub const nl: Key = Key(ffi::GDK_KEY_nl as u32);
    #[doc(alias = "GDK_KEY_vt")]
    pub const vt: Key = Key(ffi::GDK_KEY_vt as u32);
    #[doc(alias = "GDK_KEY_lowrightcorner")]
    pub const lowrightcorner: Key = Key(ffi::GDK_KEY_lowrightcorner as u32);
    #[doc(alias = "GDK_KEY_uprightcorner")]
    pub const uprightcorner: Key = Key(ffi::GDK_KEY_uprightcorner as u32);
    #[doc(alias = "GDK_KEY_upleftcorner")]
    pub const upleftcorner: Key = Key(ffi::GDK_KEY_upleftcorner as u32);
    #[doc(alias = "GDK_KEY_lowleftcorner")]
    pub const lowleftcorner: Key = Key(ffi::GDK_KEY_lowleftcorner as u32);
    #[doc(alias = "GDK_KEY_crossinglines")]
    pub const crossinglines: Key = Key(ffi::GDK_KEY_crossinglines as u32);
    #[doc(alias = "GDK_KEY_horizlinescan1")]
    pub const horizlinescan1: Key = Key(ffi::GDK_KEY_horizlinescan1 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan3")]
    pub const horizlinescan3: Key = Key(ffi::GDK_KEY_horizlinescan3 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan5")]
    pub const horizlinescan5: Key = Key(ffi::GDK_KEY_horizlinescan5 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan7")]
    pub const horizlinescan7: Key = Key(ffi::GDK_KEY_horizlinescan7 as u32);
    #[doc(alias = "GDK_KEY_horizlinescan9")]
    pub const horizlinescan9: Key = Key(ffi::GDK_KEY_horizlinescan9 as u32);
    #[doc(alias = "GDK_KEY_leftt")]
    pub const leftt: Key = Key(ffi::GDK_KEY_leftt as u32);
    #[doc(alias = "GDK_KEY_rightt")]
    pub const rightt: Key = Key(ffi::GDK_KEY_rightt as u32);
    #[doc(alias = "GDK_KEY_bott")]
    pub const bott: Key = Key(ffi::GDK_KEY_bott as u32);
    #[doc(alias = "GDK_KEY_topt")]
    pub const topt: Key = Key(ffi::GDK_KEY_topt as u32);
    #[doc(alias = "GDK_KEY_vertbar")]
    pub const vertbar: Key = Key(ffi::GDK_KEY_vertbar as u32);
    #[doc(alias = "GDK_KEY_emspace")]
    pub const emspace: Key = Key(ffi::GDK_KEY_emspace as u32);
    #[doc(alias = "GDK_KEY_enspace")]
    pub const enspace: Key = Key(ffi::GDK_KEY_enspace as u32);
    #[doc(alias = "GDK_KEY_em3space")]
    pub const em3space: Key = Key(ffi::GDK_KEY_em3space as u32);
    #[doc(alias = "GDK_KEY_em4space")]
    pub const em4space: Key = Key(ffi::GDK_KEY_em4space as u32);
    #[doc(alias = "GDK_KEY_digitspace")]
    pub const digitspace: Key = Key(ffi::GDK_KEY_digitspace as u32);
    #[doc(alias = "GDK_KEY_punctspace")]
    pub const punctspace: Key = Key(ffi::GDK_KEY_punctspace as u32);
    #[doc(alias = "GDK_KEY_thinspace")]
    pub const thinspace: Key = Key(ffi::GDK_KEY_thinspace as u32);
    #[doc(alias = "GDK_KEY_hairspace")]
    pub const hairspace: Key = Key(ffi::GDK_KEY_hairspace as u32);
    #[doc(alias = "GDK_KEY_emdash")]
    pub const emdash: Key = Key(ffi::GDK_KEY_emdash as u32);
    #[doc(alias = "GDK_KEY_endash")]
    pub const endash: Key = Key(ffi::GDK_KEY_endash as u32);
    #[doc(alias = "GDK_KEY_signifblank")]
    pub const signifblank: Key = Key(ffi::GDK_KEY_signifblank as u32);
    #[doc(alias = "GDK_KEY_ellipsis")]
    pub const ellipsis: Key = Key(ffi::GDK_KEY_ellipsis as u32);
    #[doc(alias = "GDK_KEY_doubbaselinedot")]
    pub const doubbaselinedot: Key = Key(ffi::GDK_KEY_doubbaselinedot as u32);
    #[doc(alias = "GDK_KEY_onethird")]
    pub const onethird: Key = Key(ffi::GDK_KEY_onethird as u32);
    #[doc(alias = "GDK_KEY_twothirds")]
    pub const twothirds: Key = Key(ffi::GDK_KEY_twothirds as u32);
    #[doc(alias = "GDK_KEY_onefifth")]
    pub const onefifth: Key = Key(ffi::GDK_KEY_onefifth as u32);
    #[doc(alias = "GDK_KEY_twofifths")]
    pub const twofifths: Key = Key(ffi::GDK_KEY_twofifths as u32);
    #[doc(alias = "GDK_KEY_threefifths")]
    pub const threefifths: Key = Key(ffi::GDK_KEY_threefifths as u32);
    #[doc(alias = "GDK_KEY_fourfifths")]
    pub const fourfifths: Key = Key(ffi::GDK_KEY_fourfifths as u32);
    #[doc(alias = "GDK_KEY_onesixth")]
    pub const onesixth: Key = Key(ffi::GDK_KEY_onesixth as u32);
    #[doc(alias = "GDK_KEY_fivesixths")]
    pub const fivesixths: Key = Key(ffi::GDK_KEY_fivesixths as u32);
    #[doc(alias = "GDK_KEY_careof")]
    pub const careof: Key = Key(ffi::GDK_KEY_careof as u32);
    #[doc(alias = "GDK_KEY_figdash")]
    pub const figdash: Key = Key(ffi::GDK_KEY_figdash as u32);
    #[doc(alias = "GDK_KEY_leftanglebracket")]
    pub const leftanglebracket: Key = Key(ffi::GDK_KEY_leftanglebracket as u32);
    #[doc(alias = "GDK_KEY_decimalpoint")]
    pub const decimalpoint: Key = Key(ffi::GDK_KEY_decimalpoint as u32);
    #[doc(alias = "GDK_KEY_rightanglebracket")]
    pub const rightanglebracket: Key = Key(ffi::GDK_KEY_rightanglebracket as u32);
    #[doc(alias = "GDK_KEY_marker")]
    pub const marker: Key = Key(ffi::GDK_KEY_marker as u32);
    #[doc(alias = "GDK_KEY_oneeighth")]
    pub const oneeighth: Key = Key(ffi::GDK_KEY_oneeighth as u32);
    #[doc(alias = "GDK_KEY_threeeighths")]
    pub const threeeighths: Key = Key(ffi::GDK_KEY_threeeighths as u32);
    #[doc(alias = "GDK_KEY_fiveeighths")]
    pub const fiveeighths: Key = Key(ffi::GDK_KEY_fiveeighths as u32);
    #[doc(alias = "GDK_KEY_seveneighths")]
    pub const seveneighths: Key = Key(ffi::GDK_KEY_seveneighths as u32);
    #[doc(alias = "GDK_KEY_trademark")]
    pub const trademark: Key = Key(ffi::GDK_KEY_trademark as u32);
    #[doc(alias = "GDK_KEY_signaturemark")]
    pub const signaturemark: Key = Key(ffi::GDK_KEY_signaturemark as u32);
    #[doc(alias = "GDK_KEY_trademarkincircle")]
    pub const trademarkincircle: Key = Key(ffi::GDK_KEY_trademarkincircle as u32);
    #[doc(alias = "GDK_KEY_leftopentriangle")]
    pub const leftopentriangle: Key = Key(ffi::GDK_KEY_leftopentriangle as u32);
    #[doc(alias = "GDK_KEY_rightopentriangle")]
    pub const rightopentriangle: Key = Key(ffi::GDK_KEY_rightopentriangle as u32);
    #[doc(alias = "GDK_KEY_emopencircle")]
    pub const emopencircle: Key = Key(ffi::GDK_KEY_emopencircle as u32);
    #[doc(alias = "GDK_KEY_emopenrectangle")]
    pub const emopenrectangle: Key = Key(ffi::GDK_KEY_emopenrectangle as u32);
    #[doc(alias = "GDK_KEY_leftsinglequotemark")]
    pub const leftsinglequotemark: Key = Key(ffi::GDK_KEY_leftsinglequotemark as u32);
    #[doc(alias = "GDK_KEY_rightsinglequotemark")]
    pub const rightsinglequotemark: Key = Key(ffi::GDK_KEY_rightsinglequotemark as u32);
    #[doc(alias = "GDK_KEY_leftdoublequotemark")]
    pub const leftdoublequotemark: Key = Key(ffi::GDK_KEY_leftdoublequotemark as u32);
    #[doc(alias = "GDK_KEY_rightdoublequotemark")]
    pub const rightdoublequotemark: Key = Key(ffi::GDK_KEY_rightdoublequotemark as u32);
    #[doc(alias = "GDK_KEY_prescription")]
    pub const prescription: Key = Key(ffi::GDK_KEY_prescription as u32);
    #[doc(alias = "GDK_KEY_permille")]
    pub const permille: Key = Key(ffi::GDK_KEY_permille as u32);
    #[doc(alias = "GDK_KEY_minutes")]
    pub const minutes: Key = Key(ffi::GDK_KEY_minutes as u32);
    #[doc(alias = "GDK_KEY_seconds")]
    pub const seconds: Key = Key(ffi::GDK_KEY_seconds as u32);
    #[doc(alias = "GDK_KEY_latincross")]
    pub const latincross: Key = Key(ffi::GDK_KEY_latincross as u32);
    #[doc(alias = "GDK_KEY_hexagram")]
    pub const hexagram: Key = Key(ffi::GDK_KEY_hexagram as u32);
    #[doc(alias = "GDK_KEY_filledrectbullet")]
    pub const filledrectbullet: Key = Key(ffi::GDK_KEY_filledrectbullet as u32);
    #[doc(alias = "GDK_KEY_filledlefttribullet")]
    pub const filledlefttribullet: Key = Key(ffi::GDK_KEY_filledlefttribullet as u32);
    #[doc(alias = "GDK_KEY_filledrighttribullet")]
    pub const filledrighttribullet: Key = Key(ffi::GDK_KEY_filledrighttribullet as u32);
    #[doc(alias = "GDK_KEY_emfilledcircle")]
    pub const emfilledcircle: Key = Key(ffi::GDK_KEY_emfilledcircle as u32);
    #[doc(alias = "GDK_KEY_emfilledrect")]
    pub const emfilledrect: Key = Key(ffi::GDK_KEY_emfilledrect as u32);
    #[doc(alias = "GDK_KEY_enopencircbullet")]
    pub const enopencircbullet: Key = Key(ffi::GDK_KEY_enopencircbullet as u32);
    #[doc(alias = "GDK_KEY_enopensquarebullet")]
    pub const enopensquarebullet: Key = Key(ffi::GDK_KEY_enopensquarebullet as u32);
    #[doc(alias = "GDK_KEY_openrectbullet")]
    pub const openrectbullet: Key = Key(ffi::GDK_KEY_openrectbullet as u32);
    #[doc(alias = "GDK_KEY_opentribulletup")]
    pub const opentribulletup: Key = Key(ffi::GDK_KEY_opentribulletup as u32);
    #[doc(alias = "GDK_KEY_opentribulletdown")]
    pub const opentribulletdown: Key = Key(ffi::GDK_KEY_opentribulletdown as u32);
    #[doc(alias = "GDK_KEY_openstar")]
    pub const openstar: Key = Key(ffi::GDK_KEY_openstar as u32);
    #[doc(alias = "GDK_KEY_enfilledcircbullet")]
    pub const enfilledcircbullet: Key = Key(ffi::GDK_KEY_enfilledcircbullet as u32);
    #[doc(alias = "GDK_KEY_enfilledsqbullet")]
    pub const enfilledsqbullet: Key = Key(ffi::GDK_KEY_enfilledsqbullet as u32);
    #[doc(alias = "GDK_KEY_filledtribulletup")]
    pub const filledtribulletup: Key = Key(ffi::GDK_KEY_filledtribulletup as u32);
    #[doc(alias = "GDK_KEY_filledtribulletdown")]
    pub const filledtribulletdown: Key = Key(ffi::GDK_KEY_filledtribulletdown as u32);
    #[doc(alias = "GDK_KEY_leftpointer")]
    pub const leftpointer: Key = Key(ffi::GDK_KEY_leftpointer as u32);
    #[doc(alias = "GDK_KEY_rightpointer")]
    pub const rightpointer: Key = Key(ffi::GDK_KEY_rightpointer as u32);
    #[doc(alias = "GDK_KEY_club")]
    pub const club: Key = Key(ffi::GDK_KEY_club as u32);
    #[doc(alias = "GDK_KEY_diamond")]
    pub const diamond: Key = Key(ffi::GDK_KEY_diamond as u32);
    #[doc(alias = "GDK_KEY_heart")]
    pub const heart: Key = Key(ffi::GDK_KEY_heart as u32);
    #[doc(alias = "GDK_KEY_maltesecross")]
    pub const maltesecross: Key = Key(ffi::GDK_KEY_maltesecross as u32);
    #[doc(alias = "GDK_KEY_dagger")]
    pub const dagger: Key = Key(ffi::GDK_KEY_dagger as u32);
    #[doc(alias = "GDK_KEY_doubledagger")]
    pub const doubledagger: Key = Key(ffi::GDK_KEY_doubledagger as u32);
    #[doc(alias = "GDK_KEY_checkmark")]
    pub const checkmark: Key = Key(ffi::GDK_KEY_checkmark as u32);
    #[doc(alias = "GDK_KEY_ballotcross")]
    pub const ballotcross: Key = Key(ffi::GDK_KEY_ballotcross as u32);
    #[doc(alias = "GDK_KEY_musicalsharp")]
    pub const musicalsharp: Key = Key(ffi::GDK_KEY_musicalsharp as u32);
    #[doc(alias = "GDK_KEY_musicalflat")]
    pub const musicalflat: Key = Key(ffi::GDK_KEY_musicalflat as u32);
    #[doc(alias = "GDK_KEY_malesymbol")]
    pub const malesymbol: Key = Key(ffi::GDK_KEY_malesymbol as u32);
    #[doc(alias = "GDK_KEY_femalesymbol")]
    pub const femalesymbol: Key = Key(ffi::GDK_KEY_femalesymbol as u32);
    #[doc(alias = "GDK_KEY_telephone")]
    pub const telephone: Key = Key(ffi::GDK_KEY_telephone as u32);
    #[doc(alias = "GDK_KEY_telephonerecorder")]
    pub const telephonerecorder: Key = Key(ffi::GDK_KEY_telephonerecorder as u32);
    #[doc(alias = "GDK_KEY_phonographcopyright")]
    pub const phonographcopyright: Key = Key(ffi::GDK_KEY_phonographcopyright as u32);
    #[doc(alias = "GDK_KEY_caret")]
    pub const caret: Key = Key(ffi::GDK_KEY_caret as u32);
    #[doc(alias = "GDK_KEY_singlelowquotemark")]
    pub const singlelowquotemark: Key = Key(ffi::GDK_KEY_singlelowquotemark as u32);
    #[doc(alias = "GDK_KEY_doublelowquotemark")]
    pub const doublelowquotemark: Key = Key(ffi::GDK_KEY_doublelowquotemark as u32);
    #[doc(alias = "GDK_KEY_cursor")]
    pub const cursor: Key = Key(ffi::GDK_KEY_cursor as u32);
    #[doc(alias = "GDK_KEY_leftcaret")]
    pub const leftcaret: Key = Key(ffi::GDK_KEY_leftcaret as u32);
    #[doc(alias = "GDK_KEY_rightcaret")]
    pub const rightcaret: Key = Key(ffi::GDK_KEY_rightcaret as u32);
    #[doc(alias = "GDK_KEY_downcaret")]
    pub const downcaret: Key = Key(ffi::GDK_KEY_downcaret as u32);
    #[doc(alias = "GDK_KEY_upcaret")]
    pub const upcaret: Key = Key(ffi::GDK_KEY_upcaret as u32);
    #[doc(alias = "GDK_KEY_overbar")]
    pub const overbar: Key = Key(ffi::GDK_KEY_overbar as u32);
    #[doc(alias = "GDK_KEY_downtack")]
    pub const downtack: Key = Key(ffi::GDK_KEY_downtack as u32);
    #[doc(alias = "GDK_KEY_upshoe")]
    pub const upshoe: Key = Key(ffi::GDK_KEY_upshoe as u32);
    #[doc(alias = "GDK_KEY_downstile")]
    pub const downstile: Key = Key(ffi::GDK_KEY_downstile as u32);
    #[doc(alias = "GDK_KEY_underbar")]
    pub const underbar: Key = Key(ffi::GDK_KEY_underbar as u32);
    #[doc(alias = "GDK_KEY_jot")]
    pub const jot: Key = Key(ffi::GDK_KEY_jot as u32);
    #[doc(alias = "GDK_KEY_quad")]
    pub const quad: Key = Key(ffi::GDK_KEY_quad as u32);
    #[doc(alias = "GDK_KEY_uptack")]
    pub const uptack: Key = Key(ffi::GDK_KEY_uptack as u32);
    #[doc(alias = "GDK_KEY_circle")]
    pub const circle: Key = Key(ffi::GDK_KEY_circle as u32);
    #[doc(alias = "GDK_KEY_upstile")]
    pub const upstile: Key = Key(ffi::GDK_KEY_upstile as u32);
    #[doc(alias = "GDK_KEY_downshoe")]
    pub const downshoe: Key = Key(ffi::GDK_KEY_downshoe as u32);
    #[doc(alias = "GDK_KEY_rightshoe")]
    pub const rightshoe: Key = Key(ffi::GDK_KEY_rightshoe as u32);
    #[doc(alias = "GDK_KEY_leftshoe")]
    pub const leftshoe: Key = Key(ffi::GDK_KEY_leftshoe as u32);
    #[doc(alias = "GDK_KEY_lefttack")]
    pub const lefttack: Key = Key(ffi::GDK_KEY_lefttack as u32);
    #[doc(alias = "GDK_KEY_righttack")]
    pub const righttack: Key = Key(ffi::GDK_KEY_righttack as u32);
    #[doc(alias = "GDK_KEY_hebrew_doublelowline")]
    pub const hebrew_doublelowline: Key = Key(ffi::GDK_KEY_hebrew_doublelowline as u32);
    #[doc(alias = "GDK_KEY_hebrew_aleph")]
    pub const hebrew_aleph: Key = Key(ffi::GDK_KEY_hebrew_aleph as u32);
    #[doc(alias = "GDK_KEY_hebrew_bet")]
    pub const hebrew_bet: Key = Key(ffi::GDK_KEY_hebrew_bet as u32);
    #[doc(alias = "GDK_KEY_hebrew_beth")]
    pub const hebrew_beth: Key = Key(ffi::GDK_KEY_hebrew_beth as u32);
    #[doc(alias = "GDK_KEY_hebrew_gimel")]
    pub const hebrew_gimel: Key = Key(ffi::GDK_KEY_hebrew_gimel as u32);
    #[doc(alias = "GDK_KEY_hebrew_gimmel")]
    pub const hebrew_gimmel: Key = Key(ffi::GDK_KEY_hebrew_gimmel as u32);
    #[doc(alias = "GDK_KEY_hebrew_dalet")]
    pub const hebrew_dalet: Key = Key(ffi::GDK_KEY_hebrew_dalet as u32);
    #[doc(alias = "GDK_KEY_hebrew_daleth")]
    pub const hebrew_daleth: Key = Key(ffi::GDK_KEY_hebrew_daleth as u32);
    #[doc(alias = "GDK_KEY_hebrew_he")]
    pub const hebrew_he: Key = Key(ffi::GDK_KEY_hebrew_he as u32);
    #[doc(alias = "GDK_KEY_hebrew_waw")]
    pub const hebrew_waw: Key = Key(ffi::GDK_KEY_hebrew_waw as u32);
    #[doc(alias = "GDK_KEY_hebrew_zain")]
    pub const hebrew_zain: Key = Key(ffi::GDK_KEY_hebrew_zain as u32);
    #[doc(alias = "GDK_KEY_hebrew_zayin")]
    pub const hebrew_zayin: Key = Key(ffi::GDK_KEY_hebrew_zayin as u32);
    #[doc(alias = "GDK_KEY_hebrew_chet")]
    pub const hebrew_chet: Key = Key(ffi::GDK_KEY_hebrew_chet as u32);
    #[doc(alias = "GDK_KEY_hebrew_het")]
    pub const hebrew_het: Key = Key(ffi::GDK_KEY_hebrew_het as u32);
    #[doc(alias = "GDK_KEY_hebrew_tet")]
    pub const hebrew_tet: Key = Key(ffi::GDK_KEY_hebrew_tet as u32);
    #[doc(alias = "GDK_KEY_hebrew_teth")]
    pub const hebrew_teth: Key = Key(ffi::GDK_KEY_hebrew_teth as u32);
    #[doc(alias = "GDK_KEY_hebrew_yod")]
    pub const hebrew_yod: Key = Key(ffi::GDK_KEY_hebrew_yod as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalkaph")]
    pub const hebrew_finalkaph: Key = Key(ffi::GDK_KEY_hebrew_finalkaph as u32);
    #[doc(alias = "GDK_KEY_hebrew_kaph")]
    pub const hebrew_kaph: Key = Key(ffi::GDK_KEY_hebrew_kaph as u32);
    #[doc(alias = "GDK_KEY_hebrew_lamed")]
    pub const hebrew_lamed: Key = Key(ffi::GDK_KEY_hebrew_lamed as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalmem")]
    pub const hebrew_finalmem: Key = Key(ffi::GDK_KEY_hebrew_finalmem as u32);
    #[doc(alias = "GDK_KEY_hebrew_mem")]
    pub const hebrew_mem: Key = Key(ffi::GDK_KEY_hebrew_mem as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalnun")]
    pub const hebrew_finalnun: Key = Key(ffi::GDK_KEY_hebrew_finalnun as u32);
    #[doc(alias = "GDK_KEY_hebrew_nun")]
    pub const hebrew_nun: Key = Key(ffi::GDK_KEY_hebrew_nun as u32);
    #[doc(alias = "GDK_KEY_hebrew_samech")]
    pub const hebrew_samech: Key = Key(ffi::GDK_KEY_hebrew_samech as u32);
    #[doc(alias = "GDK_KEY_hebrew_samekh")]
    pub const hebrew_samekh: Key = Key(ffi::GDK_KEY_hebrew_samekh as u32);
    #[doc(alias = "GDK_KEY_hebrew_ayin")]
    pub const hebrew_ayin: Key = Key(ffi::GDK_KEY_hebrew_ayin as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalpe")]
    pub const hebrew_finalpe: Key = Key(ffi::GDK_KEY_hebrew_finalpe as u32);
    #[doc(alias = "GDK_KEY_hebrew_pe")]
    pub const hebrew_pe: Key = Key(ffi::GDK_KEY_hebrew_pe as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalzade")]
    pub const hebrew_finalzade: Key = Key(ffi::GDK_KEY_hebrew_finalzade as u32);
    #[doc(alias = "GDK_KEY_hebrew_finalzadi")]
    pub const hebrew_finalzadi: Key = Key(ffi::GDK_KEY_hebrew_finalzadi as u32);
    #[doc(alias = "GDK_KEY_hebrew_zade")]
    pub const hebrew_zade: Key = Key(ffi::GDK_KEY_hebrew_zade as u32);
    #[doc(alias = "GDK_KEY_hebrew_zadi")]
    pub const hebrew_zadi: Key = Key(ffi::GDK_KEY_hebrew_zadi as u32);
    #[doc(alias = "GDK_KEY_hebrew_qoph")]
    pub const hebrew_qoph: Key = Key(ffi::GDK_KEY_hebrew_qoph as u32);
    #[doc(alias = "GDK_KEY_hebrew_kuf")]
    pub const hebrew_kuf: Key = Key(ffi::GDK_KEY_hebrew_kuf as u32);
    #[doc(alias = "GDK_KEY_hebrew_resh")]
    pub const hebrew_resh: Key = Key(ffi::GDK_KEY_hebrew_resh as u32);
    #[doc(alias = "GDK_KEY_hebrew_shin")]
    pub const hebrew_shin: Key = Key(ffi::GDK_KEY_hebrew_shin as u32);
    #[doc(alias = "GDK_KEY_hebrew_taw")]
    pub const hebrew_taw: Key = Key(ffi::GDK_KEY_hebrew_taw as u32);
    #[doc(alias = "GDK_KEY_hebrew_taf")]
    pub const hebrew_taf: Key = Key(ffi::GDK_KEY_hebrew_taf as u32);
    #[doc(alias = "GDK_KEY_Hebrew_switch")]
    pub const Hebrew_switch: Key = Key(ffi::GDK_KEY_Hebrew_switch as u32);
    #[doc(alias = "GDK_KEY_Thai_kokai")]
    pub const Thai_kokai: Key = Key(ffi::GDK_KEY_Thai_kokai as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhai")]
    pub const Thai_khokhai: Key = Key(ffi::GDK_KEY_Thai_khokhai as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhuat")]
    pub const Thai_khokhuat: Key = Key(ffi::GDK_KEY_Thai_khokhuat as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhwai")]
    pub const Thai_khokhwai: Key = Key(ffi::GDK_KEY_Thai_khokhwai as u32);
    #[doc(alias = "GDK_KEY_Thai_khokhon")]
    pub const Thai_khokhon: Key = Key(ffi::GDK_KEY_Thai_khokhon as u32);
    #[doc(alias = "GDK_KEY_Thai_khorakhang")]
    pub const Thai_khorakhang: Key = Key(ffi::GDK_KEY_Thai_khorakhang as u32);
    #[doc(alias = "GDK_KEY_Thai_ngongu")]
    pub const Thai_ngongu: Key = Key(ffi::GDK_KEY_Thai_ngongu as u32);
    #[doc(alias = "GDK_KEY_Thai_chochan")]
    pub const Thai_chochan: Key = Key(ffi::GDK_KEY_Thai_chochan as u32);
    #[doc(alias = "GDK_KEY_Thai_choching")]
    pub const Thai_choching: Key = Key(ffi::GDK_KEY_Thai_choching as u32);
    #[doc(alias = "GDK_KEY_Thai_chochang")]
    pub const Thai_chochang: Key = Key(ffi::GDK_KEY_Thai_chochang as u32);
    #[doc(alias = "GDK_KEY_Thai_soso")]
    pub const Thai_soso: Key = Key(ffi::GDK_KEY_Thai_soso as u32);
    #[doc(alias = "GDK_KEY_Thai_chochoe")]
    pub const Thai_chochoe: Key = Key(ffi::GDK_KEY_Thai_chochoe as u32);
    #[doc(alias = "GDK_KEY_Thai_yoying")]
    pub const Thai_yoying: Key = Key(ffi::GDK_KEY_Thai_yoying as u32);
    #[doc(alias = "GDK_KEY_Thai_dochada")]
    pub const Thai_dochada: Key = Key(ffi::GDK_KEY_Thai_dochada as u32);
    #[doc(alias = "GDK_KEY_Thai_topatak")]
    pub const Thai_topatak: Key = Key(ffi::GDK_KEY_Thai_topatak as u32);
    #[doc(alias = "GDK_KEY_Thai_thothan")]
    pub const Thai_thothan: Key = Key(ffi::GDK_KEY_Thai_thothan as u32);
    #[doc(alias = "GDK_KEY_Thai_thonangmontho")]
    pub const Thai_thonangmontho: Key = Key(ffi::GDK_KEY_Thai_thonangmontho as u32);
    #[doc(alias = "GDK_KEY_Thai_thophuthao")]
    pub const Thai_thophuthao: Key = Key(ffi::GDK_KEY_Thai_thophuthao as u32);
    #[doc(alias = "GDK_KEY_Thai_nonen")]
    pub const Thai_nonen: Key = Key(ffi::GDK_KEY_Thai_nonen as u32);
    #[doc(alias = "GDK_KEY_Thai_dodek")]
    pub const Thai_dodek: Key = Key(ffi::GDK_KEY_Thai_dodek as u32);
    #[doc(alias = "GDK_KEY_Thai_totao")]
    pub const Thai_totao: Key = Key(ffi::GDK_KEY_Thai_totao as u32);
    #[doc(alias = "GDK_KEY_Thai_thothung")]
    pub const Thai_thothung: Key = Key(ffi::GDK_KEY_Thai_thothung as u32);
    #[doc(alias = "GDK_KEY_Thai_thothahan")]
    pub const Thai_thothahan: Key = Key(ffi::GDK_KEY_Thai_thothahan as u32);
    #[doc(alias = "GDK_KEY_Thai_thothong")]
    pub const Thai_thothong: Key = Key(ffi::GDK_KEY_Thai_thothong as u32);
    #[doc(alias = "GDK_KEY_Thai_nonu")]
    pub const Thai_nonu: Key = Key(ffi::GDK_KEY_Thai_nonu as u32);
    #[doc(alias = "GDK_KEY_Thai_bobaimai")]
    pub const Thai_bobaimai: Key = Key(ffi::GDK_KEY_Thai_bobaimai as u32);
    #[doc(alias = "GDK_KEY_Thai_popla")]
    pub const Thai_popla: Key = Key(ffi::GDK_KEY_Thai_popla as u32);
    #[doc(alias = "GDK_KEY_Thai_phophung")]
    pub const Thai_phophung: Key = Key(ffi::GDK_KEY_Thai_phophung as u32);
    #[doc(alias = "GDK_KEY_Thai_fofa")]
    pub const Thai_fofa: Key = Key(ffi::GDK_KEY_Thai_fofa as u32);
    #[doc(alias = "GDK_KEY_Thai_phophan")]
    pub const Thai_phophan: Key = Key(ffi::GDK_KEY_Thai_phophan as u32);
    #[doc(alias = "GDK_KEY_Thai_fofan")]
    pub const Thai_fofan: Key = Key(ffi::GDK_KEY_Thai_fofan as u32);
    #[doc(alias = "GDK_KEY_Thai_phosamphao")]
    pub const Thai_phosamphao: Key = Key(ffi::GDK_KEY_Thai_phosamphao as u32);
    #[doc(alias = "GDK_KEY_Thai_moma")]
    pub const Thai_moma: Key = Key(ffi::GDK_KEY_Thai_moma as u32);
    #[doc(alias = "GDK_KEY_Thai_yoyak")]
    pub const Thai_yoyak: Key = Key(ffi::GDK_KEY_Thai_yoyak as u32);
    #[doc(alias = "GDK_KEY_Thai_rorua")]
    pub const Thai_rorua: Key = Key(ffi::GDK_KEY_Thai_rorua as u32);
    #[doc(alias = "GDK_KEY_Thai_ru")]
    pub const Thai_ru: Key = Key(ffi::GDK_KEY_Thai_ru as u32);
    #[doc(alias = "GDK_KEY_Thai_loling")]
    pub const Thai_loling: Key = Key(ffi::GDK_KEY_Thai_loling as u32);
    #[doc(alias = "GDK_KEY_Thai_lu")]
    pub const Thai_lu: Key = Key(ffi::GDK_KEY_Thai_lu as u32);
    #[doc(alias = "GDK_KEY_Thai_wowaen")]
    pub const Thai_wowaen: Key = Key(ffi::GDK_KEY_Thai_wowaen as u32);
    #[doc(alias = "GDK_KEY_Thai_sosala")]
    pub const Thai_sosala: Key = Key(ffi::GDK_KEY_Thai_sosala as u32);
    #[doc(alias = "GDK_KEY_Thai_sorusi")]
    pub const Thai_sorusi: Key = Key(ffi::GDK_KEY_Thai_sorusi as u32);
    #[doc(alias = "GDK_KEY_Thai_sosua")]
    pub const Thai_sosua: Key = Key(ffi::GDK_KEY_Thai_sosua as u32);
    #[doc(alias = "GDK_KEY_Thai_hohip")]
    pub const Thai_hohip: Key = Key(ffi::GDK_KEY_Thai_hohip as u32);
    #[doc(alias = "GDK_KEY_Thai_lochula")]
    pub const Thai_lochula: Key = Key(ffi::GDK_KEY_Thai_lochula as u32);
    #[doc(alias = "GDK_KEY_Thai_oang")]
    pub const Thai_oang: Key = Key(ffi::GDK_KEY_Thai_oang as u32);
    #[doc(alias = "GDK_KEY_Thai_honokhuk")]
    pub const Thai_honokhuk: Key = Key(ffi::GDK_KEY_Thai_honokhuk as u32);
    #[doc(alias = "GDK_KEY_Thai_paiyannoi")]
    pub const Thai_paiyannoi: Key = Key(ffi::GDK_KEY_Thai_paiyannoi as u32);
    #[doc(alias = "GDK_KEY_Thai_saraa")]
    pub const Thai_saraa: Key = Key(ffi::GDK_KEY_Thai_saraa as u32);
    #[doc(alias = "GDK_KEY_Thai_maihanakat")]
    pub const Thai_maihanakat: Key = Key(ffi::GDK_KEY_Thai_maihanakat as u32);
    #[doc(alias = "GDK_KEY_Thai_saraaa")]
    pub const Thai_saraaa: Key = Key(ffi::GDK_KEY_Thai_saraaa as u32);
    #[doc(alias = "GDK_KEY_Thai_saraam")]
    pub const Thai_saraam: Key = Key(ffi::GDK_KEY_Thai_saraam as u32);
    #[doc(alias = "GDK_KEY_Thai_sarai")]
    pub const Thai_sarai: Key = Key(ffi::GDK_KEY_Thai_sarai as u32);
    #[doc(alias = "GDK_KEY_Thai_saraii")]
    pub const Thai_saraii: Key = Key(ffi::GDK_KEY_Thai_saraii as u32);
    #[doc(alias = "GDK_KEY_Thai_saraue")]
    pub const Thai_saraue: Key = Key(ffi::GDK_KEY_Thai_saraue as u32);
    #[doc(alias = "GDK_KEY_Thai_sarauee")]
    pub const Thai_sarauee: Key = Key(ffi::GDK_KEY_Thai_sarauee as u32);
    #[doc(alias = "GDK_KEY_Thai_sarau")]
    pub const Thai_sarau: Key = Key(ffi::GDK_KEY_Thai_sarau as u32);
    #[doc(alias = "GDK_KEY_Thai_sarauu")]
    pub const Thai_sarauu: Key = Key(ffi::GDK_KEY_Thai_sarauu as u32);
    #[doc(alias = "GDK_KEY_Thai_phinthu")]
    pub const Thai_phinthu: Key = Key(ffi::GDK_KEY_Thai_phinthu as u32);
    #[doc(alias = "GDK_KEY_Thai_maihanakat_maitho")]
    pub const Thai_maihanakat_maitho: Key = Key(ffi::GDK_KEY_Thai_maihanakat_maitho as u32);
    #[doc(alias = "GDK_KEY_Thai_baht")]
    pub const Thai_baht: Key = Key(ffi::GDK_KEY_Thai_baht as u32);
    #[doc(alias = "GDK_KEY_Thai_sarae")]
    pub const Thai_sarae: Key = Key(ffi::GDK_KEY_Thai_sarae as u32);
    #[doc(alias = "GDK_KEY_Thai_saraae")]
    pub const Thai_saraae: Key = Key(ffi::GDK_KEY_Thai_saraae as u32);
    #[doc(alias = "GDK_KEY_Thai_sarao")]
    pub const Thai_sarao: Key = Key(ffi::GDK_KEY_Thai_sarao as u32);
    #[doc(alias = "GDK_KEY_Thai_saraaimaimuan")]
    pub const Thai_saraaimaimuan: Key = Key(ffi::GDK_KEY_Thai_saraaimaimuan as u32);
    #[doc(alias = "GDK_KEY_Thai_saraaimaimalai")]
    pub const Thai_saraaimaimalai: Key = Key(ffi::GDK_KEY_Thai_saraaimaimalai as u32);
    #[doc(alias = "GDK_KEY_Thai_lakkhangyao")]
    pub const Thai_lakkhangyao: Key = Key(ffi::GDK_KEY_Thai_lakkhangyao as u32);
    #[doc(alias = "GDK_KEY_Thai_maiyamok")]
    pub const Thai_maiyamok: Key = Key(ffi::GDK_KEY_Thai_maiyamok as u32);
    #[doc(alias = "GDK_KEY_Thai_maitaikhu")]
    pub const Thai_maitaikhu: Key = Key(ffi::GDK_KEY_Thai_maitaikhu as u32);
    #[doc(alias = "GDK_KEY_Thai_maiek")]
    pub const Thai_maiek: Key = Key(ffi::GDK_KEY_Thai_maiek as u32);
    #[doc(alias = "GDK_KEY_Thai_maitho")]
    pub const Thai_maitho: Key = Key(ffi::GDK_KEY_Thai_maitho as u32);
    #[doc(alias = "GDK_KEY_Thai_maitri")]
    pub const Thai_maitri: Key = Key(ffi::GDK_KEY_Thai_maitri as u32);
    #[doc(alias = "GDK_KEY_Thai_maichattawa")]
    pub const Thai_maichattawa: Key = Key(ffi::GDK_KEY_Thai_maichattawa as u32);
    #[doc(alias = "GDK_KEY_Thai_thanthakhat")]
    pub const Thai_thanthakhat: Key = Key(ffi::GDK_KEY_Thai_thanthakhat as u32);
    #[doc(alias = "GDK_KEY_Thai_nikhahit")]
    pub const Thai_nikhahit: Key = Key(ffi::GDK_KEY_Thai_nikhahit as u32);
    #[doc(alias = "GDK_KEY_Thai_leksun")]
    pub const Thai_leksun: Key = Key(ffi::GDK_KEY_Thai_leksun as u32);
    #[doc(alias = "GDK_KEY_Thai_leknung")]
    pub const Thai_leknung: Key = Key(ffi::GDK_KEY_Thai_leknung as u32);
    #[doc(alias = "GDK_KEY_Thai_leksong")]
    pub const Thai_leksong: Key = Key(ffi::GDK_KEY_Thai_leksong as u32);
    #[doc(alias = "GDK_KEY_Thai_leksam")]
    pub const Thai_leksam: Key = Key(ffi::GDK_KEY_Thai_leksam as u32);
    #[doc(alias = "GDK_KEY_Thai_leksi")]
    pub const Thai_leksi: Key = Key(ffi::GDK_KEY_Thai_leksi as u32);
    #[doc(alias = "GDK_KEY_Thai_lekha")]
    pub const Thai_lekha: Key = Key(ffi::GDK_KEY_Thai_lekha as u32);
    #[doc(alias = "GDK_KEY_Thai_lekhok")]
    pub const Thai_lekhok: Key = Key(ffi::GDK_KEY_Thai_lekhok as u32);
    #[doc(alias = "GDK_KEY_Thai_lekchet")]
    pub const Thai_lekchet: Key = Key(ffi::GDK_KEY_Thai_lekchet as u32);
    #[doc(alias = "GDK_KEY_Thai_lekpaet")]
    pub const Thai_lekpaet: Key = Key(ffi::GDK_KEY_Thai_lekpaet as u32);
    #[doc(alias = "GDK_KEY_Thai_lekkao")]
    pub const Thai_lekkao: Key = Key(ffi::GDK_KEY_Thai_lekkao as u32);
    #[doc(alias = "GDK_KEY_Hangul")]
    pub const Hangul: Key = Key(ffi::GDK_KEY_Hangul as u32);
    #[doc(alias = "GDK_KEY_Hangul_Start")]
    pub const Hangul_Start: Key = Key(ffi::GDK_KEY_Hangul_Start as u32);
    #[doc(alias = "GDK_KEY_Hangul_End")]
    pub const Hangul_End: Key = Key(ffi::GDK_KEY_Hangul_End as u32);
    #[doc(alias = "GDK_KEY_Hangul_Hanja")]
    pub const Hangul_Hanja: Key = Key(ffi::GDK_KEY_Hangul_Hanja as u32);
    #[doc(alias = "GDK_KEY_Hangul_Jamo")]
    pub const Hangul_Jamo: Key = Key(ffi::GDK_KEY_Hangul_Jamo as u32);
    #[doc(alias = "GDK_KEY_Hangul_Romaja")]
    pub const Hangul_Romaja: Key = Key(ffi::GDK_KEY_Hangul_Romaja as u32);
    #[doc(alias = "GDK_KEY_Hangul_Codeinput")]
    pub const Hangul_Codeinput: Key = Key(ffi::GDK_KEY_Hangul_Codeinput as u32);
    #[doc(alias = "GDK_KEY_Hangul_Jeonja")]
    pub const Hangul_Jeonja: Key = Key(ffi::GDK_KEY_Hangul_Jeonja as u32);
    #[doc(alias = "GDK_KEY_Hangul_Banja")]
    pub const Hangul_Banja: Key = Key(ffi::GDK_KEY_Hangul_Banja as u32);
    #[doc(alias = "GDK_KEY_Hangul_PreHanja")]
    pub const Hangul_PreHanja: Key = Key(ffi::GDK_KEY_Hangul_PreHanja as u32);
    #[doc(alias = "GDK_KEY_Hangul_PostHanja")]
    pub const Hangul_PostHanja: Key = Key(ffi::GDK_KEY_Hangul_PostHanja as u32);
    #[doc(alias = "GDK_KEY_Hangul_SingleCandidate")]
    pub const Hangul_SingleCandidate: Key = Key(ffi::GDK_KEY_Hangul_SingleCandidate as u32);
    #[doc(alias = "GDK_KEY_Hangul_MultipleCandidate")]
    pub const Hangul_MultipleCandidate: Key = Key(ffi::GDK_KEY_Hangul_MultipleCandidate as u32);
    #[doc(alias = "GDK_KEY_Hangul_PreviousCandidate")]
    pub const Hangul_PreviousCandidate: Key = Key(ffi::GDK_KEY_Hangul_PreviousCandidate as u32);
    #[doc(alias = "GDK_KEY_Hangul_Special")]
    pub const Hangul_Special: Key = Key(ffi::GDK_KEY_Hangul_Special as u32);
    #[doc(alias = "GDK_KEY_Hangul_switch")]
    pub const Hangul_switch: Key = Key(ffi::GDK_KEY_Hangul_switch as u32);
    #[doc(alias = "GDK_KEY_Hangul_Kiyeog")]
    pub const Hangul_Kiyeog: Key = Key(ffi::GDK_KEY_Hangul_Kiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangKiyeog")]
    pub const Hangul_SsangKiyeog: Key = Key(ffi::GDK_KEY_Hangul_SsangKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_KiyeogSios")]
    pub const Hangul_KiyeogSios: Key = Key(ffi::GDK_KEY_Hangul_KiyeogSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_Nieun")]
    pub const Hangul_Nieun: Key = Key(ffi::GDK_KEY_Hangul_Nieun as u32);
    #[doc(alias = "GDK_KEY_Hangul_NieunJieuj")]
    pub const Hangul_NieunJieuj: Key = Key(ffi::GDK_KEY_Hangul_NieunJieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_NieunHieuh")]
    pub const Hangul_NieunHieuh: Key = Key(ffi::GDK_KEY_Hangul_NieunHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_Dikeud")]
    pub const Hangul_Dikeud: Key = Key(ffi::GDK_KEY_Hangul_Dikeud as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangDikeud")]
    pub const Hangul_SsangDikeud: Key = Key(ffi::GDK_KEY_Hangul_SsangDikeud as u32);
    #[doc(alias = "GDK_KEY_Hangul_Rieul")]
    pub const Hangul_Rieul: Key = Key(ffi::GDK_KEY_Hangul_Rieul as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulKiyeog")]
    pub const Hangul_RieulKiyeog: Key = Key(ffi::GDK_KEY_Hangul_RieulKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulMieum")]
    pub const Hangul_RieulMieum: Key = Key(ffi::GDK_KEY_Hangul_RieulMieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulPieub")]
    pub const Hangul_RieulPieub: Key = Key(ffi::GDK_KEY_Hangul_RieulPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulSios")]
    pub const Hangul_RieulSios: Key = Key(ffi::GDK_KEY_Hangul_RieulSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulTieut")]
    pub const Hangul_RieulTieut: Key = Key(ffi::GDK_KEY_Hangul_RieulTieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulPhieuf")]
    pub const Hangul_RieulPhieuf: Key = Key(ffi::GDK_KEY_Hangul_RieulPhieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulHieuh")]
    pub const Hangul_RieulHieuh: Key = Key(ffi::GDK_KEY_Hangul_RieulHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_Mieum")]
    pub const Hangul_Mieum: Key = Key(ffi::GDK_KEY_Hangul_Mieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_Pieub")]
    pub const Hangul_Pieub: Key = Key(ffi::GDK_KEY_Hangul_Pieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangPieub")]
    pub const Hangul_SsangPieub: Key = Key(ffi::GDK_KEY_Hangul_SsangPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_PieubSios")]
    pub const Hangul_PieubSios: Key = Key(ffi::GDK_KEY_Hangul_PieubSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_Sios")]
    pub const Hangul_Sios: Key = Key(ffi::GDK_KEY_Hangul_Sios as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangSios")]
    pub const Hangul_SsangSios: Key = Key(ffi::GDK_KEY_Hangul_SsangSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_Ieung")]
    pub const Hangul_Ieung: Key = Key(ffi::GDK_KEY_Hangul_Ieung as u32);
    #[doc(alias = "GDK_KEY_Hangul_Jieuj")]
    pub const Hangul_Jieuj: Key = Key(ffi::GDK_KEY_Hangul_Jieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_SsangJieuj")]
    pub const Hangul_SsangJieuj: Key = Key(ffi::GDK_KEY_Hangul_SsangJieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_Cieuc")]
    pub const Hangul_Cieuc: Key = Key(ffi::GDK_KEY_Hangul_Cieuc as u32);
    #[doc(alias = "GDK_KEY_Hangul_Khieuq")]
    pub const Hangul_Khieuq: Key = Key(ffi::GDK_KEY_Hangul_Khieuq as u32);
    #[doc(alias = "GDK_KEY_Hangul_Tieut")]
    pub const Hangul_Tieut: Key = Key(ffi::GDK_KEY_Hangul_Tieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_Phieuf")]
    pub const Hangul_Phieuf: Key = Key(ffi::GDK_KEY_Hangul_Phieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_Hieuh")]
    pub const Hangul_Hieuh: Key = Key(ffi::GDK_KEY_Hangul_Hieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_A")]
    pub const Hangul_A: Key = Key(ffi::GDK_KEY_Hangul_A as u32);
    #[doc(alias = "GDK_KEY_Hangul_AE")]
    pub const Hangul_AE: Key = Key(ffi::GDK_KEY_Hangul_AE as u32);
    #[doc(alias = "GDK_KEY_Hangul_YA")]
    pub const Hangul_YA: Key = Key(ffi::GDK_KEY_Hangul_YA as u32);
    #[doc(alias = "GDK_KEY_Hangul_YAE")]
    pub const Hangul_YAE: Key = Key(ffi::GDK_KEY_Hangul_YAE as u32);
    #[doc(alias = "GDK_KEY_Hangul_EO")]
    pub const Hangul_EO: Key = Key(ffi::GDK_KEY_Hangul_EO as u32);
    #[doc(alias = "GDK_KEY_Hangul_E")]
    pub const Hangul_E: Key = Key(ffi::GDK_KEY_Hangul_E as u32);
    #[doc(alias = "GDK_KEY_Hangul_YEO")]
    pub const Hangul_YEO: Key = Key(ffi::GDK_KEY_Hangul_YEO as u32);
    #[doc(alias = "GDK_KEY_Hangul_YE")]
    pub const Hangul_YE: Key = Key(ffi::GDK_KEY_Hangul_YE as u32);
    #[doc(alias = "GDK_KEY_Hangul_O")]
    pub const Hangul_O: Key = Key(ffi::GDK_KEY_Hangul_O as u32);
    #[doc(alias = "GDK_KEY_Hangul_WA")]
    pub const Hangul_WA: Key = Key(ffi::GDK_KEY_Hangul_WA as u32);
    #[doc(alias = "GDK_KEY_Hangul_WAE")]
    pub const Hangul_WAE: Key = Key(ffi::GDK_KEY_Hangul_WAE as u32);
    #[doc(alias = "GDK_KEY_Hangul_OE")]
    pub const Hangul_OE: Key = Key(ffi::GDK_KEY_Hangul_OE as u32);
    #[doc(alias = "GDK_KEY_Hangul_YO")]
    pub const Hangul_YO: Key = Key(ffi::GDK_KEY_Hangul_YO as u32);
    #[doc(alias = "GDK_KEY_Hangul_U")]
    pub const Hangul_U: Key = Key(ffi::GDK_KEY_Hangul_U as u32);
    #[doc(alias = "GDK_KEY_Hangul_WEO")]
    pub const Hangul_WEO: Key = Key(ffi::GDK_KEY_Hangul_WEO as u32);
    #[doc(alias = "GDK_KEY_Hangul_WE")]
    pub const Hangul_WE: Key = Key(ffi::GDK_KEY_Hangul_WE as u32);
    #[doc(alias = "GDK_KEY_Hangul_WI")]
    pub const Hangul_WI: Key = Key(ffi::GDK_KEY_Hangul_WI as u32);
    #[doc(alias = "GDK_KEY_Hangul_YU")]
    pub const Hangul_YU: Key = Key(ffi::GDK_KEY_Hangul_YU as u32);
    #[doc(alias = "GDK_KEY_Hangul_EU")]
    pub const Hangul_EU: Key = Key(ffi::GDK_KEY_Hangul_EU as u32);
    #[doc(alias = "GDK_KEY_Hangul_YI")]
    pub const Hangul_YI: Key = Key(ffi::GDK_KEY_Hangul_YI as u32);
    #[doc(alias = "GDK_KEY_Hangul_I")]
    pub const Hangul_I: Key = Key(ffi::GDK_KEY_Hangul_I as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Kiyeog")]
    pub const Hangul_J_Kiyeog: Key = Key(ffi::GDK_KEY_Hangul_J_Kiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_SsangKiyeog")]
    pub const Hangul_J_SsangKiyeog: Key = Key(ffi::GDK_KEY_Hangul_J_SsangKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_KiyeogSios")]
    pub const Hangul_J_KiyeogSios: Key = Key(ffi::GDK_KEY_Hangul_J_KiyeogSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Nieun")]
    pub const Hangul_J_Nieun: Key = Key(ffi::GDK_KEY_Hangul_J_Nieun as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_NieunJieuj")]
    pub const Hangul_J_NieunJieuj: Key = Key(ffi::GDK_KEY_Hangul_J_NieunJieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_NieunHieuh")]
    pub const Hangul_J_NieunHieuh: Key = Key(ffi::GDK_KEY_Hangul_J_NieunHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Dikeud")]
    pub const Hangul_J_Dikeud: Key = Key(ffi::GDK_KEY_Hangul_J_Dikeud as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Rieul")]
    pub const Hangul_J_Rieul: Key = Key(ffi::GDK_KEY_Hangul_J_Rieul as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulKiyeog")]
    pub const Hangul_J_RieulKiyeog: Key = Key(ffi::GDK_KEY_Hangul_J_RieulKiyeog as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulMieum")]
    pub const Hangul_J_RieulMieum: Key = Key(ffi::GDK_KEY_Hangul_J_RieulMieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulPieub")]
    pub const Hangul_J_RieulPieub: Key = Key(ffi::GDK_KEY_Hangul_J_RieulPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulSios")]
    pub const Hangul_J_RieulSios: Key = Key(ffi::GDK_KEY_Hangul_J_RieulSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulTieut")]
    pub const Hangul_J_RieulTieut: Key = Key(ffi::GDK_KEY_Hangul_J_RieulTieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulPhieuf")]
    pub const Hangul_J_RieulPhieuf: Key = Key(ffi::GDK_KEY_Hangul_J_RieulPhieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_RieulHieuh")]
    pub const Hangul_J_RieulHieuh: Key = Key(ffi::GDK_KEY_Hangul_J_RieulHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Mieum")]
    pub const Hangul_J_Mieum: Key = Key(ffi::GDK_KEY_Hangul_J_Mieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Pieub")]
    pub const Hangul_J_Pieub: Key = Key(ffi::GDK_KEY_Hangul_J_Pieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_PieubSios")]
    pub const Hangul_J_PieubSios: Key = Key(ffi::GDK_KEY_Hangul_J_PieubSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Sios")]
    pub const Hangul_J_Sios: Key = Key(ffi::GDK_KEY_Hangul_J_Sios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_SsangSios")]
    pub const Hangul_J_SsangSios: Key = Key(ffi::GDK_KEY_Hangul_J_SsangSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Ieung")]
    pub const Hangul_J_Ieung: Key = Key(ffi::GDK_KEY_Hangul_J_Ieung as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Jieuj")]
    pub const Hangul_J_Jieuj: Key = Key(ffi::GDK_KEY_Hangul_J_Jieuj as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Cieuc")]
    pub const Hangul_J_Cieuc: Key = Key(ffi::GDK_KEY_Hangul_J_Cieuc as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Khieuq")]
    pub const Hangul_J_Khieuq: Key = Key(ffi::GDK_KEY_Hangul_J_Khieuq as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Tieut")]
    pub const Hangul_J_Tieut: Key = Key(ffi::GDK_KEY_Hangul_J_Tieut as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Phieuf")]
    pub const Hangul_J_Phieuf: Key = Key(ffi::GDK_KEY_Hangul_J_Phieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_Hieuh")]
    pub const Hangul_J_Hieuh: Key = Key(ffi::GDK_KEY_Hangul_J_Hieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_RieulYeorinHieuh")]
    pub const Hangul_RieulYeorinHieuh: Key = Key(ffi::GDK_KEY_Hangul_RieulYeorinHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_SunkyeongeumMieum")]
    pub const Hangul_SunkyeongeumMieum: Key = Key(ffi::GDK_KEY_Hangul_SunkyeongeumMieum as u32);
    #[doc(alias = "GDK_KEY_Hangul_SunkyeongeumPieub")]
    pub const Hangul_SunkyeongeumPieub: Key = Key(ffi::GDK_KEY_Hangul_SunkyeongeumPieub as u32);
    #[doc(alias = "GDK_KEY_Hangul_PanSios")]
    pub const Hangul_PanSios: Key = Key(ffi::GDK_KEY_Hangul_PanSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_KkogjiDalrinIeung")]
    pub const Hangul_KkogjiDalrinIeung: Key = Key(ffi::GDK_KEY_Hangul_KkogjiDalrinIeung as u32);
    #[doc(alias = "GDK_KEY_Hangul_SunkyeongeumPhieuf")]
    pub const Hangul_SunkyeongeumPhieuf: Key = Key(ffi::GDK_KEY_Hangul_SunkyeongeumPhieuf as u32);
    #[doc(alias = "GDK_KEY_Hangul_YeorinHieuh")]
    pub const Hangul_YeorinHieuh: Key = Key(ffi::GDK_KEY_Hangul_YeorinHieuh as u32);
    #[doc(alias = "GDK_KEY_Hangul_AraeA")]
    pub const Hangul_AraeA: Key = Key(ffi::GDK_KEY_Hangul_AraeA as u32);
    #[doc(alias = "GDK_KEY_Hangul_AraeAE")]
    pub const Hangul_AraeAE: Key = Key(ffi::GDK_KEY_Hangul_AraeAE as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_PanSios")]
    pub const Hangul_J_PanSios: Key = Key(ffi::GDK_KEY_Hangul_J_PanSios as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_KkogjiDalrinIeung")]
    pub const Hangul_J_KkogjiDalrinIeung: Key = Key(ffi::GDK_KEY_Hangul_J_KkogjiDalrinIeung as u32);
    #[doc(alias = "GDK_KEY_Hangul_J_YeorinHieuh")]
    pub const Hangul_J_YeorinHieuh: Key = Key(ffi::GDK_KEY_Hangul_J_YeorinHieuh as u32);
    #[doc(alias = "GDK_KEY_Korean_Won")]
    pub const Korean_Won: Key = Key(ffi::GDK_KEY_Korean_Won as u32);
    #[doc(alias = "GDK_KEY_Armenian_ligature_ew")]
    pub const Armenian_ligature_ew: Key = Key(ffi::GDK_KEY_Armenian_ligature_ew as u32);
    #[doc(alias = "GDK_KEY_Armenian_full_stop")]
    pub const Armenian_full_stop: Key = Key(ffi::GDK_KEY_Armenian_full_stop as u32);
    #[doc(alias = "GDK_KEY_Armenian_verjaket")]
    pub const Armenian_verjaket: Key = Key(ffi::GDK_KEY_Armenian_verjaket as u32);
    #[doc(alias = "GDK_KEY_Armenian_separation_mark")]
    pub const Armenian_separation_mark: Key = Key(ffi::GDK_KEY_Armenian_separation_mark as u32);
    #[doc(alias = "GDK_KEY_Armenian_but")]
    pub const Armenian_but: Key = Key(ffi::GDK_KEY_Armenian_but as u32);
    #[doc(alias = "GDK_KEY_Armenian_hyphen")]
    pub const Armenian_hyphen: Key = Key(ffi::GDK_KEY_Armenian_hyphen as u32);
    #[doc(alias = "GDK_KEY_Armenian_yentamna")]
    pub const Armenian_yentamna: Key = Key(ffi::GDK_KEY_Armenian_yentamna as u32);
    #[doc(alias = "GDK_KEY_Armenian_exclam")]
    pub const Armenian_exclam: Key = Key(ffi::GDK_KEY_Armenian_exclam as u32);
    #[doc(alias = "GDK_KEY_Armenian_amanak")]
    pub const Armenian_amanak: Key = Key(ffi::GDK_KEY_Armenian_amanak as u32);
    #[doc(alias = "GDK_KEY_Armenian_accent")]
    pub const Armenian_accent: Key = Key(ffi::GDK_KEY_Armenian_accent as u32);
    #[doc(alias = "GDK_KEY_Armenian_shesht")]
    pub const Armenian_shesht: Key = Key(ffi::GDK_KEY_Armenian_shesht as u32);
    #[doc(alias = "GDK_KEY_Armenian_question")]
    pub const Armenian_question: Key = Key(ffi::GDK_KEY_Armenian_question as u32);
    #[doc(alias = "GDK_KEY_Armenian_paruyk")]
    pub const Armenian_paruyk: Key = Key(ffi::GDK_KEY_Armenian_paruyk as u32);
    #[doc(alias = "GDK_KEY_Armenian_AYB")]
    pub const Armenian_AYB: Key = Key(ffi::GDK_KEY_Armenian_AYB as u32);
    #[doc(alias = "GDK_KEY_Armenian_ayb")]
    pub const Armenian_ayb: Key = Key(ffi::GDK_KEY_Armenian_ayb as u32);
    #[doc(alias = "GDK_KEY_Armenian_BEN")]
    pub const Armenian_BEN: Key = Key(ffi::GDK_KEY_Armenian_BEN as u32);
    #[doc(alias = "GDK_KEY_Armenian_ben")]
    pub const Armenian_ben: Key = Key(ffi::GDK_KEY_Armenian_ben as u32);
    #[doc(alias = "GDK_KEY_Armenian_GIM")]
    pub const Armenian_GIM: Key = Key(ffi::GDK_KEY_Armenian_GIM as u32);
    #[doc(alias = "GDK_KEY_Armenian_gim")]
    pub const Armenian_gim: Key = Key(ffi::GDK_KEY_Armenian_gim as u32);
    #[doc(alias = "GDK_KEY_Armenian_DA")]
    pub const Armenian_DA: Key = Key(ffi::GDK_KEY_Armenian_DA as u32);
    #[doc(alias = "GDK_KEY_Armenian_da")]
    pub const Armenian_da: Key = Key(ffi::GDK_KEY_Armenian_da as u32);
    #[doc(alias = "GDK_KEY_Armenian_YECH")]
    pub const Armenian_YECH: Key = Key(ffi::GDK_KEY_Armenian_YECH as u32);
    #[doc(alias = "GDK_KEY_Armenian_yech")]
    pub const Armenian_yech: Key = Key(ffi::GDK_KEY_Armenian_yech as u32);
    #[doc(alias = "GDK_KEY_Armenian_ZA")]
    pub const Armenian_ZA: Key = Key(ffi::GDK_KEY_Armenian_ZA as u32);
    #[doc(alias = "GDK_KEY_Armenian_za")]
    pub const Armenian_za: Key = Key(ffi::GDK_KEY_Armenian_za as u32);
    #[doc(alias = "GDK_KEY_Armenian_E")]
    pub const Armenian_E: Key = Key(ffi::GDK_KEY_Armenian_E as u32);
    #[doc(alias = "GDK_KEY_Armenian_e")]
    pub const Armenian_e: Key = Key(ffi::GDK_KEY_Armenian_e as u32);
    #[doc(alias = "GDK_KEY_Armenian_AT")]
    pub const Armenian_AT: Key = Key(ffi::GDK_KEY_Armenian_AT as u32);
    #[doc(alias = "GDK_KEY_Armenian_at")]
    pub const Armenian_at: Key = Key(ffi::GDK_KEY_Armenian_at as u32);
    #[doc(alias = "GDK_KEY_Armenian_TO")]
    pub const Armenian_TO: Key = Key(ffi::GDK_KEY_Armenian_TO as u32);
    #[doc(alias = "GDK_KEY_Armenian_to")]
    pub const Armenian_to: Key = Key(ffi::GDK_KEY_Armenian_to as u32);
    #[doc(alias = "GDK_KEY_Armenian_ZHE")]
    pub const Armenian_ZHE: Key = Key(ffi::GDK_KEY_Armenian_ZHE as u32);
    #[doc(alias = "GDK_KEY_Armenian_zhe")]
    pub const Armenian_zhe: Key = Key(ffi::GDK_KEY_Armenian_zhe as u32);
    #[doc(alias = "GDK_KEY_Armenian_INI")]
    pub const Armenian_INI: Key = Key(ffi::GDK_KEY_Armenian_INI as u32);
    #[doc(alias = "GDK_KEY_Armenian_ini")]
    pub const Armenian_ini: Key = Key(ffi::GDK_KEY_Armenian_ini as u32);
    #[doc(alias = "GDK_KEY_Armenian_LYUN")]
    pub const Armenian_LYUN: Key = Key(ffi::GDK_KEY_Armenian_LYUN as u32);
    #[doc(alias = "GDK_KEY_Armenian_lyun")]
    pub const Armenian_lyun: Key = Key(ffi::GDK_KEY_Armenian_lyun as u32);
    #[doc(alias = "GDK_KEY_Armenian_KHE")]
    pub const Armenian_KHE: Key = Key(ffi::GDK_KEY_Armenian_KHE as u32);
    #[doc(alias = "GDK_KEY_Armenian_khe")]
    pub const Armenian_khe: Key = Key(ffi::GDK_KEY_Armenian_khe as u32);
    #[doc(alias = "GDK_KEY_Armenian_TSA")]
    pub const Armenian_TSA: Key = Key(ffi::GDK_KEY_Armenian_TSA as u32);
    #[doc(alias = "GDK_KEY_Armenian_tsa")]
    pub const Armenian_tsa: Key = Key(ffi::GDK_KEY_Armenian_tsa as u32);
    #[doc(alias = "GDK_KEY_Armenian_KEN")]
    pub const Armenian_KEN: Key = Key(ffi::GDK_KEY_Armenian_KEN as u32);
    #[doc(alias = "GDK_KEY_Armenian_ken")]
    pub const Armenian_ken: Key = Key(ffi::GDK_KEY_Armenian_ken as u32);
    #[doc(alias = "GDK_KEY_Armenian_HO")]
    pub const Armenian_HO: Key = Key(ffi::GDK_KEY_Armenian_HO as u32);
    #[doc(alias = "GDK_KEY_Armenian_ho")]
    pub const Armenian_ho: Key = Key(ffi::GDK_KEY_Armenian_ho as u32);
    #[doc(alias = "GDK_KEY_Armenian_DZA")]
    pub const Armenian_DZA: Key = Key(ffi::GDK_KEY_Armenian_DZA as u32);
    #[doc(alias = "GDK_KEY_Armenian_dza")]
    pub const Armenian_dza: Key = Key(ffi::GDK_KEY_Armenian_dza as u32);
    #[doc(alias = "GDK_KEY_Armenian_GHAT")]
    pub const Armenian_GHAT: Key = Key(ffi::GDK_KEY_Armenian_GHAT as u32);
    #[doc(alias = "GDK_KEY_Armenian_ghat")]
    pub const Armenian_ghat: Key = Key(ffi::GDK_KEY_Armenian_ghat as u32);
    #[doc(alias = "GDK_KEY_Armenian_TCHE")]
    pub const Armenian_TCHE: Key = Key(ffi::GDK_KEY_Armenian_TCHE as u32);
    #[doc(alias = "GDK_KEY_Armenian_tche")]
    pub const Armenian_tche: Key = Key(ffi::GDK_KEY_Armenian_tche as u32);
    #[doc(alias = "GDK_KEY_Armenian_MEN")]
    pub const Armenian_MEN: Key = Key(ffi::GDK_KEY_Armenian_MEN as u32);
    #[doc(alias = "GDK_KEY_Armenian_men")]
    pub const Armenian_men: Key = Key(ffi::GDK_KEY_Armenian_men as u32);
    #[doc(alias = "GDK_KEY_Armenian_HI")]
    pub const Armenian_HI: Key = Key(ffi::GDK_KEY_Armenian_HI as u32);
    #[doc(alias = "GDK_KEY_Armenian_hi")]
    pub const Armenian_hi: Key = Key(ffi::GDK_KEY_Armenian_hi as u32);
    #[doc(alias = "GDK_KEY_Armenian_NU")]
    pub const Armenian_NU: Key = Key(ffi::GDK_KEY_Armenian_NU as u32);
    #[doc(alias = "GDK_KEY_Armenian_nu")]
    pub const Armenian_nu: Key = Key(ffi::GDK_KEY_Armenian_nu as u32);
    #[doc(alias = "GDK_KEY_Armenian_SHA")]
    pub const Armenian_SHA: Key = Key(ffi::GDK_KEY_Armenian_SHA as u32);
    #[doc(alias = "GDK_KEY_Armenian_sha")]
    pub const Armenian_sha: Key = Key(ffi::GDK_KEY_Armenian_sha as u32);
    #[doc(alias = "GDK_KEY_Armenian_VO")]
    pub const Armenian_VO: Key = Key(ffi::GDK_KEY_Armenian_VO as u32);
    #[doc(alias = "GDK_KEY_Armenian_vo")]
    pub const Armenian_vo: Key = Key(ffi::GDK_KEY_Armenian_vo as u32);
    #[doc(alias = "GDK_KEY_Armenian_CHA")]
    pub const Armenian_CHA: Key = Key(ffi::GDK_KEY_Armenian_CHA as u32);
    #[doc(alias = "GDK_KEY_Armenian_cha")]
    pub const Armenian_cha: Key = Key(ffi::GDK_KEY_Armenian_cha as u32);
    #[doc(alias = "GDK_KEY_Armenian_PE")]
    pub const Armenian_PE: Key = Key(ffi::GDK_KEY_Armenian_PE as u32);
    #[doc(alias = "GDK_KEY_Armenian_pe")]
    pub const Armenian_pe: Key = Key(ffi::GDK_KEY_Armenian_pe as u32);
    #[doc(alias = "GDK_KEY_Armenian_JE")]
    pub const Armenian_JE: Key = Key(ffi::GDK_KEY_Armenian_JE as u32);
    #[doc(alias = "GDK_KEY_Armenian_je")]
    pub const Armenian_je: Key = Key(ffi::GDK_KEY_Armenian_je as u32);
    #[doc(alias = "GDK_KEY_Armenian_RA")]
    pub const Armenian_RA: Key = Key(ffi::GDK_KEY_Armenian_RA as u32);
    #[doc(alias = "GDK_KEY_Armenian_ra")]
    pub const Armenian_ra: Key = Key(ffi::GDK_KEY_Armenian_ra as u32);
    #[doc(alias = "GDK_KEY_Armenian_SE")]
    pub const Armenian_SE: Key = Key(ffi::GDK_KEY_Armenian_SE as u32);
    #[doc(alias = "GDK_KEY_Armenian_se")]
    pub const Armenian_se: Key = Key(ffi::GDK_KEY_Armenian_se as u32);
    #[doc(alias = "GDK_KEY_Armenian_VEV")]
    pub const Armenian_VEV: Key = Key(ffi::GDK_KEY_Armenian_VEV as u32);
    #[doc(alias = "GDK_KEY_Armenian_vev")]
    pub const Armenian_vev: Key = Key(ffi::GDK_KEY_Armenian_vev as u32);
    #[doc(alias = "GDK_KEY_Armenian_TYUN")]
    pub const Armenian_TYUN: Key = Key(ffi::GDK_KEY_Armenian_TYUN as u32);
    #[doc(alias = "GDK_KEY_Armenian_tyun")]
    pub const Armenian_tyun: Key = Key(ffi::GDK_KEY_Armenian_tyun as u32);
    #[doc(alias = "GDK_KEY_Armenian_RE")]
    pub const Armenian_RE: Key = Key(ffi::GDK_KEY_Armenian_RE as u32);
    #[doc(alias = "GDK_KEY_Armenian_re")]
    pub const Armenian_re: Key = Key(ffi::GDK_KEY_Armenian_re as u32);
    #[doc(alias = "GDK_KEY_Armenian_TSO")]
    pub const Armenian_TSO: Key = Key(ffi::GDK_KEY_Armenian_TSO as u32);
    #[doc(alias = "GDK_KEY_Armenian_tso")]
    pub const Armenian_tso: Key = Key(ffi::GDK_KEY_Armenian_tso as u32);
    #[doc(alias = "GDK_KEY_Armenian_VYUN")]
    pub const Armenian_VYUN: Key = Key(ffi::GDK_KEY_Armenian_VYUN as u32);
    #[doc(alias = "GDK_KEY_Armenian_vyun")]
    pub const Armenian_vyun: Key = Key(ffi::GDK_KEY_Armenian_vyun as u32);
    #[doc(alias = "GDK_KEY_Armenian_PYUR")]
    pub const Armenian_PYUR: Key = Key(ffi::GDK_KEY_Armenian_PYUR as u32);
    #[doc(alias = "GDK_KEY_Armenian_pyur")]
    pub const Armenian_pyur: Key = Key(ffi::GDK_KEY_Armenian_pyur as u32);
    #[doc(alias = "GDK_KEY_Armenian_KE")]
    pub const Armenian_KE: Key = Key(ffi::GDK_KEY_Armenian_KE as u32);
    #[doc(alias = "GDK_KEY_Armenian_ke")]
    pub const Armenian_ke: Key = Key(ffi::GDK_KEY_Armenian_ke as u32);
    #[doc(alias = "GDK_KEY_Armenian_O")]
    pub const Armenian_O: Key = Key(ffi::GDK_KEY_Armenian_O as u32);
    #[doc(alias = "GDK_KEY_Armenian_o")]
    pub const Armenian_o: Key = Key(ffi::GDK_KEY_Armenian_o as u32);
    #[doc(alias = "GDK_KEY_Armenian_FE")]
    pub const Armenian_FE: Key = Key(ffi::GDK_KEY_Armenian_FE as u32);
    #[doc(alias = "GDK_KEY_Armenian_fe")]
    pub const Armenian_fe: Key = Key(ffi::GDK_KEY_Armenian_fe as u32);
    #[doc(alias = "GDK_KEY_Armenian_apostrophe")]
    pub const Armenian_apostrophe: Key = Key(ffi::GDK_KEY_Armenian_apostrophe as u32);
    #[doc(alias = "GDK_KEY_Georgian_an")]
    pub const Georgian_an: Key = Key(ffi::GDK_KEY_Georgian_an as u32);
    #[doc(alias = "GDK_KEY_Georgian_ban")]
    pub const Georgian_ban: Key = Key(ffi::GDK_KEY_Georgian_ban as u32);
    #[doc(alias = "GDK_KEY_Georgian_gan")]
    pub const Georgian_gan: Key = Key(ffi::GDK_KEY_Georgian_gan as u32);
    #[doc(alias = "GDK_KEY_Georgian_don")]
    pub const Georgian_don: Key = Key(ffi::GDK_KEY_Georgian_don as u32);
    #[doc(alias = "GDK_KEY_Georgian_en")]
    pub const Georgian_en: Key = Key(ffi::GDK_KEY_Georgian_en as u32);
    #[doc(alias = "GDK_KEY_Georgian_vin")]
    pub const Georgian_vin: Key = Key(ffi::GDK_KEY_Georgian_vin as u32);
    #[doc(alias = "GDK_KEY_Georgian_zen")]
    pub const Georgian_zen: Key = Key(ffi::GDK_KEY_Georgian_zen as u32);
    #[doc(alias = "GDK_KEY_Georgian_tan")]
    pub const Georgian_tan: Key = Key(ffi::GDK_KEY_Georgian_tan as u32);
    #[doc(alias = "GDK_KEY_Georgian_in")]
    pub const Georgian_in: Key = Key(ffi::GDK_KEY_Georgian_in as u32);
    #[doc(alias = "GDK_KEY_Georgian_kan")]
    pub const Georgian_kan: Key = Key(ffi::GDK_KEY_Georgian_kan as u32);
    #[doc(alias = "GDK_KEY_Georgian_las")]
    pub const Georgian_las: Key = Key(ffi::GDK_KEY_Georgian_las as u32);
    #[doc(alias = "GDK_KEY_Georgian_man")]
    pub const Georgian_man: Key = Key(ffi::GDK_KEY_Georgian_man as u32);
    #[doc(alias = "GDK_KEY_Georgian_nar")]
    pub const Georgian_nar: Key = Key(ffi::GDK_KEY_Georgian_nar as u32);
    #[doc(alias = "GDK_KEY_Georgian_on")]
    pub const Georgian_on: Key = Key(ffi::GDK_KEY_Georgian_on as u32);
    #[doc(alias = "GDK_KEY_Georgian_par")]
    pub const Georgian_par: Key = Key(ffi::GDK_KEY_Georgian_par as u32);
    #[doc(alias = "GDK_KEY_Georgian_zhar")]
    pub const Georgian_zhar: Key = Key(ffi::GDK_KEY_Georgian_zhar as u32);
    #[doc(alias = "GDK_KEY_Georgian_rae")]
    pub const Georgian_rae: Key = Key(ffi::GDK_KEY_Georgian_rae as u32);
    #[doc(alias = "GDK_KEY_Georgian_san")]
    pub const Georgian_san: Key = Key(ffi::GDK_KEY_Georgian_san as u32);
    #[doc(alias = "GDK_KEY_Georgian_tar")]
    pub const Georgian_tar: Key = Key(ffi::GDK_KEY_Georgian_tar as u32);
    #[doc(alias = "GDK_KEY_Georgian_un")]
    pub const Georgian_un: Key = Key(ffi::GDK_KEY_Georgian_un as u32);
    #[doc(alias = "GDK_KEY_Georgian_phar")]
    pub const Georgian_phar: Key = Key(ffi::GDK_KEY_Georgian_phar as u32);
    #[doc(alias = "GDK_KEY_Georgian_khar")]
    pub const Georgian_khar: Key = Key(ffi::GDK_KEY_Georgian_khar as u32);
    #[doc(alias = "GDK_KEY_Georgian_ghan")]
    pub const Georgian_ghan: Key = Key(ffi::GDK_KEY_Georgian_ghan as u32);
    #[doc(alias = "GDK_KEY_Georgian_qar")]
    pub const Georgian_qar: Key = Key(ffi::GDK_KEY_Georgian_qar as u32);
    #[doc(alias = "GDK_KEY_Georgian_shin")]
    pub const Georgian_shin: Key = Key(ffi::GDK_KEY_Georgian_shin as u32);
    #[doc(alias = "GDK_KEY_Georgian_chin")]
    pub const Georgian_chin: Key = Key(ffi::GDK_KEY_Georgian_chin as u32);
    #[doc(alias = "GDK_KEY_Georgian_can")]
    pub const Georgian_can: Key = Key(ffi::GDK_KEY_Georgian_can as u32);
    #[doc(alias = "GDK_KEY_Georgian_jil")]
    pub const Georgian_jil: Key = Key(ffi::GDK_KEY_Georgian_jil as u32);
    #[doc(alias = "GDK_KEY_Georgian_cil")]
    pub const Georgian_cil: Key = Key(ffi::GDK_KEY_Georgian_cil as u32);
    #[doc(alias = "GDK_KEY_Georgian_char")]
    pub const Georgian_char: Key = Key(ffi::GDK_KEY_Georgian_char as u32);
    #[doc(alias = "GDK_KEY_Georgian_xan")]
    pub const Georgian_xan: Key = Key(ffi::GDK_KEY_Georgian_xan as u32);
    #[doc(alias = "GDK_KEY_Georgian_jhan")]
    pub const Georgian_jhan: Key = Key(ffi::GDK_KEY_Georgian_jhan as u32);
    #[doc(alias = "GDK_KEY_Georgian_hae")]
    pub const Georgian_hae: Key = Key(ffi::GDK_KEY_Georgian_hae as u32);
    #[doc(alias = "GDK_KEY_Georgian_he")]
    pub const Georgian_he: Key = Key(ffi::GDK_KEY_Georgian_he as u32);
    #[doc(alias = "GDK_KEY_Georgian_hie")]
    pub const Georgian_hie: Key = Key(ffi::GDK_KEY_Georgian_hie as u32);
    #[doc(alias = "GDK_KEY_Georgian_we")]
    pub const Georgian_we: Key = Key(ffi::GDK_KEY_Georgian_we as u32);
    #[doc(alias = "GDK_KEY_Georgian_har")]
    pub const Georgian_har: Key = Key(ffi::GDK_KEY_Georgian_har as u32);
    #[doc(alias = "GDK_KEY_Georgian_hoe")]
    pub const Georgian_hoe: Key = Key(ffi::GDK_KEY_Georgian_hoe as u32);
    #[doc(alias = "GDK_KEY_Georgian_fi")]
    pub const Georgian_fi: Key = Key(ffi::GDK_KEY_Georgian_fi as u32);
    #[doc(alias = "GDK_KEY_Xabovedot")]
    pub const Xabovedot: Key = Key(ffi::GDK_KEY_Xabovedot as u32);
    #[doc(alias = "GDK_KEY_Ibreve")]
    pub const Ibreve: Key = Key(ffi::GDK_KEY_Ibreve as u32);
    #[doc(alias = "GDK_KEY_Zstroke")]
    pub const Zstroke: Key = Key(ffi::GDK_KEY_Zstroke as u32);
    #[doc(alias = "GDK_KEY_Gcaron")]
    pub const Gcaron: Key = Key(ffi::GDK_KEY_Gcaron as u32);
    #[doc(alias = "GDK_KEY_Ocaron")]
    pub const Ocaron: Key = Key(ffi::GDK_KEY_Ocaron as u32);
    #[doc(alias = "GDK_KEY_Obarred")]
    pub const Obarred: Key = Key(ffi::GDK_KEY_Obarred as u32);
    #[doc(alias = "GDK_KEY_xabovedot")]
    pub const xabovedot: Key = Key(ffi::GDK_KEY_xabovedot as u32);
    #[doc(alias = "GDK_KEY_ibreve")]
    pub const ibreve: Key = Key(ffi::GDK_KEY_ibreve as u32);
    #[doc(alias = "GDK_KEY_zstroke")]
    pub const zstroke: Key = Key(ffi::GDK_KEY_zstroke as u32);
    #[doc(alias = "GDK_KEY_gcaron")]
    pub const gcaron: Key = Key(ffi::GDK_KEY_gcaron as u32);
    #[doc(alias = "GDK_KEY_ocaron")]
    pub const ocaron: Key = Key(ffi::GDK_KEY_ocaron as u32);
    #[doc(alias = "GDK_KEY_obarred")]
    pub const obarred: Key = Key(ffi::GDK_KEY_obarred as u32);
    #[doc(alias = "GDK_KEY_SCHWA")]
    pub const SCHWA: Key = Key(ffi::GDK_KEY_SCHWA as u32);
    #[doc(alias = "GDK_KEY_schwa")]
    pub const schwa: Key = Key(ffi::GDK_KEY_schwa as u32);
    #[doc(alias = "GDK_KEY_EZH")]
    pub const EZH: Key = Key(ffi::GDK_KEY_EZH as u32);
    #[doc(alias = "GDK_KEY_ezh")]
    pub const ezh: Key = Key(ffi::GDK_KEY_ezh as u32);
    #[doc(alias = "GDK_KEY_Lbelowdot")]
    pub const Lbelowdot: Key = Key(ffi::GDK_KEY_Lbelowdot as u32);
    #[doc(alias = "GDK_KEY_lbelowdot")]
    pub const lbelowdot: Key = Key(ffi::GDK_KEY_lbelowdot as u32);
    #[doc(alias = "GDK_KEY_Abelowdot")]
    pub const Abelowdot: Key = Key(ffi::GDK_KEY_Abelowdot as u32);
    #[doc(alias = "GDK_KEY_abelowdot")]
    pub const abelowdot: Key = Key(ffi::GDK_KEY_abelowdot as u32);
    #[doc(alias = "GDK_KEY_Ahook")]
    pub const Ahook: Key = Key(ffi::GDK_KEY_Ahook as u32);
    #[doc(alias = "GDK_KEY_ahook")]
    pub const ahook: Key = Key(ffi::GDK_KEY_ahook as u32);
    #[doc(alias = "GDK_KEY_Acircumflexacute")]
    pub const Acircumflexacute: Key = Key(ffi::GDK_KEY_Acircumflexacute as u32);
    #[doc(alias = "GDK_KEY_acircumflexacute")]
    pub const acircumflexacute: Key = Key(ffi::GDK_KEY_acircumflexacute as u32);
    #[doc(alias = "GDK_KEY_Acircumflexgrave")]
    pub const Acircumflexgrave: Key = Key(ffi::GDK_KEY_Acircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_acircumflexgrave")]
    pub const acircumflexgrave: Key = Key(ffi::GDK_KEY_acircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_Acircumflexhook")]
    pub const Acircumflexhook: Key = Key(ffi::GDK_KEY_Acircumflexhook as u32);
    #[doc(alias = "GDK_KEY_acircumflexhook")]
    pub const acircumflexhook: Key = Key(ffi::GDK_KEY_acircumflexhook as u32);
    #[doc(alias = "GDK_KEY_Acircumflextilde")]
    pub const Acircumflextilde: Key = Key(ffi::GDK_KEY_Acircumflextilde as u32);
    #[doc(alias = "GDK_KEY_acircumflextilde")]
    pub const acircumflextilde: Key = Key(ffi::GDK_KEY_acircumflextilde as u32);
    #[doc(alias = "GDK_KEY_Acircumflexbelowdot")]
    pub const Acircumflexbelowdot: Key = Key(ffi::GDK_KEY_Acircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_acircumflexbelowdot")]
    pub const acircumflexbelowdot: Key = Key(ffi::GDK_KEY_acircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_Abreveacute")]
    pub const Abreveacute: Key = Key(ffi::GDK_KEY_Abreveacute as u32);
    #[doc(alias = "GDK_KEY_abreveacute")]
    pub const abreveacute: Key = Key(ffi::GDK_KEY_abreveacute as u32);
    #[doc(alias = "GDK_KEY_Abrevegrave")]
    pub const Abrevegrave: Key = Key(ffi::GDK_KEY_Abrevegrave as u32);
    #[doc(alias = "GDK_KEY_abrevegrave")]
    pub const abrevegrave: Key = Key(ffi::GDK_KEY_abrevegrave as u32);
    #[doc(alias = "GDK_KEY_Abrevehook")]
    pub const Abrevehook: Key = Key(ffi::GDK_KEY_Abrevehook as u32);
    #[doc(alias = "GDK_KEY_abrevehook")]
    pub const abrevehook: Key = Key(ffi::GDK_KEY_abrevehook as u32);
    #[doc(alias = "GDK_KEY_Abrevetilde")]
    pub const Abrevetilde: Key = Key(ffi::GDK_KEY_Abrevetilde as u32);
    #[doc(alias = "GDK_KEY_abrevetilde")]
    pub const abrevetilde: Key = Key(ffi::GDK_KEY_abrevetilde as u32);
    #[doc(alias = "GDK_KEY_Abrevebelowdot")]
    pub const Abrevebelowdot: Key = Key(ffi::GDK_KEY_Abrevebelowdot as u32);
    #[doc(alias = "GDK_KEY_abrevebelowdot")]
    pub const abrevebelowdot: Key = Key(ffi::GDK_KEY_abrevebelowdot as u32);
    #[doc(alias = "GDK_KEY_Ebelowdot")]
    pub const Ebelowdot: Key = Key(ffi::GDK_KEY_Ebelowdot as u32);
    #[doc(alias = "GDK_KEY_ebelowdot")]
    pub const ebelowdot: Key = Key(ffi::GDK_KEY_ebelowdot as u32);
    #[doc(alias = "GDK_KEY_Ehook")]
    pub const Ehook: Key = Key(ffi::GDK_KEY_Ehook as u32);
    #[doc(alias = "GDK_KEY_ehook")]
    pub const ehook: Key = Key(ffi::GDK_KEY_ehook as u32);
    #[doc(alias = "GDK_KEY_Etilde")]
    pub const Etilde: Key = Key(ffi::GDK_KEY_Etilde as u32);
    #[doc(alias = "GDK_KEY_etilde")]
    pub const etilde: Key = Key(ffi::GDK_KEY_etilde as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexacute")]
    pub const Ecircumflexacute: Key = Key(ffi::GDK_KEY_Ecircumflexacute as u32);
    #[doc(alias = "GDK_KEY_ecircumflexacute")]
    pub const ecircumflexacute: Key = Key(ffi::GDK_KEY_ecircumflexacute as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexgrave")]
    pub const Ecircumflexgrave: Key = Key(ffi::GDK_KEY_Ecircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_ecircumflexgrave")]
    pub const ecircumflexgrave: Key = Key(ffi::GDK_KEY_ecircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexhook")]
    pub const Ecircumflexhook: Key = Key(ffi::GDK_KEY_Ecircumflexhook as u32);
    #[doc(alias = "GDK_KEY_ecircumflexhook")]
    pub const ecircumflexhook: Key = Key(ffi::GDK_KEY_ecircumflexhook as u32);
    #[doc(alias = "GDK_KEY_Ecircumflextilde")]
    pub const Ecircumflextilde: Key = Key(ffi::GDK_KEY_Ecircumflextilde as u32);
    #[doc(alias = "GDK_KEY_ecircumflextilde")]
    pub const ecircumflextilde: Key = Key(ffi::GDK_KEY_ecircumflextilde as u32);
    #[doc(alias = "GDK_KEY_Ecircumflexbelowdot")]
    pub const Ecircumflexbelowdot: Key = Key(ffi::GDK_KEY_Ecircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_ecircumflexbelowdot")]
    pub const ecircumflexbelowdot: Key = Key(ffi::GDK_KEY_ecircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_Ihook")]
    pub const Ihook: Key = Key(ffi::GDK_KEY_Ihook as u32);
    #[doc(alias = "GDK_KEY_ihook")]
    pub const ihook: Key = Key(ffi::GDK_KEY_ihook as u32);
    #[doc(alias = "GDK_KEY_Ibelowdot")]
    pub const Ibelowdot: Key = Key(ffi::GDK_KEY_Ibelowdot as u32);
    #[doc(alias = "GDK_KEY_ibelowdot")]
    pub const ibelowdot: Key = Key(ffi::GDK_KEY_ibelowdot as u32);
    #[doc(alias = "GDK_KEY_Obelowdot")]
    pub const Obelowdot: Key = Key(ffi::GDK_KEY_Obelowdot as u32);
    #[doc(alias = "GDK_KEY_obelowdot")]
    pub const obelowdot: Key = Key(ffi::GDK_KEY_obelowdot as u32);
    #[doc(alias = "GDK_KEY_Ohook")]
    pub const Ohook: Key = Key(ffi::GDK_KEY_Ohook as u32);
    #[doc(alias = "GDK_KEY_ohook")]
    pub const ohook: Key = Key(ffi::GDK_KEY_ohook as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexacute")]
    pub const Ocircumflexacute: Key = Key(ffi::GDK_KEY_Ocircumflexacute as u32);
    #[doc(alias = "GDK_KEY_ocircumflexacute")]
    pub const ocircumflexacute: Key = Key(ffi::GDK_KEY_ocircumflexacute as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexgrave")]
    pub const Ocircumflexgrave: Key = Key(ffi::GDK_KEY_Ocircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_ocircumflexgrave")]
    pub const ocircumflexgrave: Key = Key(ffi::GDK_KEY_ocircumflexgrave as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexhook")]
    pub const Ocircumflexhook: Key = Key(ffi::GDK_KEY_Ocircumflexhook as u32);
    #[doc(alias = "GDK_KEY_ocircumflexhook")]
    pub const ocircumflexhook: Key = Key(ffi::GDK_KEY_ocircumflexhook as u32);
    #[doc(alias = "GDK_KEY_Ocircumflextilde")]
    pub const Ocircumflextilde: Key = Key(ffi::GDK_KEY_Ocircumflextilde as u32);
    #[doc(alias = "GDK_KEY_ocircumflextilde")]
    pub const ocircumflextilde: Key = Key(ffi::GDK_KEY_ocircumflextilde as u32);
    #[doc(alias = "GDK_KEY_Ocircumflexbelowdot")]
    pub const Ocircumflexbelowdot: Key = Key(ffi::GDK_KEY_Ocircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_ocircumflexbelowdot")]
    pub const ocircumflexbelowdot: Key = Key(ffi::GDK_KEY_ocircumflexbelowdot as u32);
    #[doc(alias = "GDK_KEY_Ohornacute")]
    pub const Ohornacute: Key = Key(ffi::GDK_KEY_Ohornacute as u32);
    #[doc(alias = "GDK_KEY_ohornacute")]
    pub const ohornacute: Key = Key(ffi::GDK_KEY_ohornacute as u32);
    #[doc(alias = "GDK_KEY_Ohorngrave")]
    pub const Ohorngrave: Key = Key(ffi::GDK_KEY_Ohorngrave as u32);
    #[doc(alias = "GDK_KEY_ohorngrave")]
    pub const ohorngrave: Key = Key(ffi::GDK_KEY_ohorngrave as u32);
    #[doc(alias = "GDK_KEY_Ohornhook")]
    pub const Ohornhook: Key = Key(ffi::GDK_KEY_Ohornhook as u32);
    #[doc(alias = "GDK_KEY_ohornhook")]
    pub const ohornhook: Key = Key(ffi::GDK_KEY_ohornhook as u32);
    #[doc(alias = "GDK_KEY_Ohorntilde")]
    pub const Ohorntilde: Key = Key(ffi::GDK_KEY_Ohorntilde as u32);
    #[doc(alias = "GDK_KEY_ohorntilde")]
    pub const ohorntilde: Key = Key(ffi::GDK_KEY_ohorntilde as u32);
    #[doc(alias = "GDK_KEY_Ohornbelowdot")]
    pub const Ohornbelowdot: Key = Key(ffi::GDK_KEY_Ohornbelowdot as u32);
    #[doc(alias = "GDK_KEY_ohornbelowdot")]
    pub const ohornbelowdot: Key = Key(ffi::GDK_KEY_ohornbelowdot as u32);
    #[doc(alias = "GDK_KEY_Ubelowdot")]
    pub const Ubelowdot: Key = Key(ffi::GDK_KEY_Ubelowdot as u32);
    #[doc(alias = "GDK_KEY_ubelowdot")]
    pub const ubelowdot: Key = Key(ffi::GDK_KEY_ubelowdot as u32);
    #[doc(alias = "GDK_KEY_Uhook")]
    pub const Uhook: Key = Key(ffi::GDK_KEY_Uhook as u32);
    #[doc(alias = "GDK_KEY_uhook")]
    pub const uhook: Key = Key(ffi::GDK_KEY_uhook as u32);
    #[doc(alias = "GDK_KEY_Uhornacute")]
    pub const Uhornacute: Key = Key(ffi::GDK_KEY_Uhornacute as u32);
    #[doc(alias = "GDK_KEY_uhornacute")]
    pub const uhornacute: Key = Key(ffi::GDK_KEY_uhornacute as u32);
    #[doc(alias = "GDK_KEY_Uhorngrave")]
    pub const Uhorngrave: Key = Key(ffi::GDK_KEY_Uhorngrave as u32);
    #[doc(alias = "GDK_KEY_uhorngrave")]
    pub const uhorngrave: Key = Key(ffi::GDK_KEY_uhorngrave as u32);
    #[doc(alias = "GDK_KEY_Uhornhook")]
    pub const Uhornhook: Key = Key(ffi::GDK_KEY_Uhornhook as u32);
    #[doc(alias = "GDK_KEY_uhornhook")]
    pub const uhornhook: Key = Key(ffi::GDK_KEY_uhornhook as u32);
    #[doc(alias = "GDK_KEY_Uhorntilde")]
    pub const Uhorntilde: Key = Key(ffi::GDK_KEY_Uhorntilde as u32);
    #[doc(alias = "GDK_KEY_uhorntilde")]
    pub const uhorntilde: Key = Key(ffi::GDK_KEY_uhorntilde as u32);
    #[doc(alias = "GDK_KEY_Uhornbelowdot")]
    pub const Uhornbelowdot: Key = Key(ffi::GDK_KEY_Uhornbelowdot as u32);
    #[doc(alias = "GDK_KEY_uhornbelowdot")]
    pub const uhornbelowdot: Key = Key(ffi::GDK_KEY_uhornbelowdot as u32);
    #[doc(alias = "GDK_KEY_Ybelowdot")]
    pub const Ybelowdot: Key = Key(ffi::GDK_KEY_Ybelowdot as u32);
    #[doc(alias = "GDK_KEY_ybelowdot")]
    pub const ybelowdot: Key = Key(ffi::GDK_KEY_ybelowdot as u32);
    #[doc(alias = "GDK_KEY_Yhook")]
    pub const Yhook: Key = Key(ffi::GDK_KEY_Yhook as u32);
    #[doc(alias = "GDK_KEY_yhook")]
    pub const yhook: Key = Key(ffi::GDK_KEY_yhook as u32);
    #[doc(alias = "GDK_KEY_Ytilde")]
    pub const Ytilde: Key = Key(ffi::GDK_KEY_Ytilde as u32);
    #[doc(alias = "GDK_KEY_ytilde")]
    pub const ytilde: Key = Key(ffi::GDK_KEY_ytilde as u32);
    #[doc(alias = "GDK_KEY_Ohorn")]
    pub const Ohorn: Key = Key(ffi::GDK_KEY_Ohorn as u32);
    #[doc(alias = "GDK_KEY_ohorn")]
    pub const ohorn: Key = Key(ffi::GDK_KEY_ohorn as u32);
    #[doc(alias = "GDK_KEY_Uhorn")]
    pub const Uhorn: Key = Key(ffi::GDK_KEY_Uhorn as u32);
    #[doc(alias = "GDK_KEY_uhorn")]
    pub const uhorn: Key = Key(ffi::GDK_KEY_uhorn as u32);
    #[doc(alias = "GDK_KEY_EcuSign")]
    pub const EcuSign: Key = Key(ffi::GDK_KEY_EcuSign as u32);
    #[doc(alias = "GDK_KEY_ColonSign")]
    pub const ColonSign: Key = Key(ffi::GDK_KEY_ColonSign as u32);
    #[doc(alias = "GDK_KEY_CruzeiroSign")]
    pub const CruzeiroSign: Key = Key(ffi::GDK_KEY_CruzeiroSign as u32);
    #[doc(alias = "GDK_KEY_FFrancSign")]
    pub const FFrancSign: Key = Key(ffi::GDK_KEY_FFrancSign as u32);
    #[doc(alias = "GDK_KEY_LiraSign")]
    pub const LiraSign: Key = Key(ffi::GDK_KEY_LiraSign as u32);
    #[doc(alias = "GDK_KEY_MillSign")]
    pub const MillSign: Key = Key(ffi::GDK_KEY_MillSign as u32);
    #[doc(alias = "GDK_KEY_NairaSign")]
    pub const NairaSign: Key = Key(ffi::GDK_KEY_NairaSign as u32);
    #[doc(alias = "GDK_KEY_PesetaSign")]
    pub const PesetaSign: Key = Key(ffi::GDK_KEY_PesetaSign as u32);
    #[doc(alias = "GDK_KEY_RupeeSign")]
    pub const RupeeSign: Key = Key(ffi::GDK_KEY_RupeeSign as u32);
    #[doc(alias = "GDK_KEY_WonSign")]
    pub const WonSign: Key = Key(ffi::GDK_KEY_WonSign as u32);
    #[doc(alias = "GDK_KEY_NewSheqelSign")]
    pub const NewSheqelSign: Key = Key(ffi::GDK_KEY_NewSheqelSign as u32);
    #[doc(alias = "GDK_KEY_DongSign")]
    pub const DongSign: Key = Key(ffi::GDK_KEY_DongSign as u32);
    #[doc(alias = "GDK_KEY_EuroSign")]
    pub const EuroSign: Key = Key(ffi::GDK_KEY_EuroSign as u32);
    #[doc(alias = "GDK_KEY_zerosuperior")]
    pub const zerosuperior: Key = Key(ffi::GDK_KEY_zerosuperior as u32);
    #[doc(alias = "GDK_KEY_foursuperior")]
    pub const foursuperior: Key = Key(ffi::GDK_KEY_foursuperior as u32);
    #[doc(alias = "GDK_KEY_fivesuperior")]
    pub const fivesuperior: Key = Key(ffi::GDK_KEY_fivesuperior as u32);
    #[doc(alias = "GDK_KEY_sixsuperior")]
    pub const sixsuperior: Key = Key(ffi::GDK_KEY_sixsuperior as u32);
    #[doc(alias = "GDK_KEY_sevensuperior")]
    pub const sevensuperior: Key = Key(ffi::GDK_KEY_sevensuperior as u32);
    #[doc(alias = "GDK_KEY_eightsuperior")]
    pub const eightsuperior: Key = Key(ffi::GDK_KEY_eightsuperior as u32);
    #[doc(alias = "GDK_KEY_ninesuperior")]
    pub const ninesuperior: Key = Key(ffi::GDK_KEY_ninesuperior as u32);
    #[doc(alias = "GDK_KEY_zerosubscript")]
    pub const zerosubscript: Key = Key(ffi::GDK_KEY_zerosubscript as u32);
    #[doc(alias = "GDK_KEY_onesubscript")]
    pub const onesubscript: Key = Key(ffi::GDK_KEY_onesubscript as u32);
    #[doc(alias = "GDK_KEY_twosubscript")]
    pub const twosubscript: Key = Key(ffi::GDK_KEY_twosubscript as u32);
    #[doc(alias = "GDK_KEY_threesubscript")]
    pub const threesubscript: Key = Key(ffi::GDK_KEY_threesubscript as u32);
    #[doc(alias = "GDK_KEY_foursubscript")]
    pub const foursubscript: Key = Key(ffi::GDK_KEY_foursubscript as u32);
    #[doc(alias = "GDK_KEY_fivesubscript")]
    pub const fivesubscript: Key = Key(ffi::GDK_KEY_fivesubscript as u32);
    #[doc(alias = "GDK_KEY_sixsubscript")]
    pub const sixsubscript: Key = Key(ffi::GDK_KEY_sixsubscript as u32);
    #[doc(alias = "GDK_KEY_sevensubscript")]
    pub const sevensubscript: Key = Key(ffi::GDK_KEY_sevensubscript as u32);
    #[doc(alias = "GDK_KEY_eightsubscript")]
    pub const eightsubscript: Key = Key(ffi::GDK_KEY_eightsubscript as u32);
    #[doc(alias = "GDK_KEY_ninesubscript")]
    pub const ninesubscript: Key = Key(ffi::GDK_KEY_ninesubscript as u32);
    #[doc(alias = "GDK_KEY_partdifferential")]
    pub const partdifferential: Key = Key(ffi::GDK_KEY_partdifferential as u32);
    #[doc(alias = "GDK_KEY_emptyset")]
    pub const emptyset: Key = Key(ffi::GDK_KEY_emptyset as u32);
    #[doc(alias = "GDK_KEY_elementof")]
    pub const elementof: Key = Key(ffi::GDK_KEY_elementof as u32);
    #[doc(alias = "GDK_KEY_notelementof")]
    pub const notelementof: Key = Key(ffi::GDK_KEY_notelementof as u32);
    #[doc(alias = "GDK_KEY_containsas")]
    pub const containsas: Key = Key(ffi::GDK_KEY_containsas as u32);
    #[doc(alias = "GDK_KEY_squareroot")]
    pub const squareroot: Key = Key(ffi::GDK_KEY_squareroot as u32);
    #[doc(alias = "GDK_KEY_cuberoot")]
    pub const cuberoot: Key = Key(ffi::GDK_KEY_cuberoot as u32);
    #[doc(alias = "GDK_KEY_fourthroot")]
    pub const fourthroot: Key = Key(ffi::GDK_KEY_fourthroot as u32);
    #[doc(alias = "GDK_KEY_dintegral")]
    pub const dintegral: Key = Key(ffi::GDK_KEY_dintegral as u32);
    #[doc(alias = "GDK_KEY_tintegral")]
    pub const tintegral: Key = Key(ffi::GDK_KEY_tintegral as u32);
    #[doc(alias = "GDK_KEY_because")]
    pub const because: Key = Key(ffi::GDK_KEY_because as u32);
    #[doc(alias = "GDK_KEY_approxeq")]
    pub const approxeq: Key = Key(ffi::GDK_KEY_approxeq as u32);
    #[doc(alias = "GDK_KEY_notapproxeq")]
    pub const notapproxeq: Key = Key(ffi::GDK_KEY_notapproxeq as u32);
    #[doc(alias = "GDK_KEY_notidentical")]
    pub const notidentical: Key = Key(ffi::GDK_KEY_notidentical as u32);
    #[doc(alias = "GDK_KEY_stricteq")]
    pub const stricteq: Key = Key(ffi::GDK_KEY_stricteq as u32);
    #[doc(alias = "GDK_KEY_braille_dot_1")]
    pub const braille_dot_1: Key = Key(ffi::GDK_KEY_braille_dot_1 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_2")]
    pub const braille_dot_2: Key = Key(ffi::GDK_KEY_braille_dot_2 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_3")]
    pub const braille_dot_3: Key = Key(ffi::GDK_KEY_braille_dot_3 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_4")]
    pub const braille_dot_4: Key = Key(ffi::GDK_KEY_braille_dot_4 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_5")]
    pub const braille_dot_5: Key = Key(ffi::GDK_KEY_braille_dot_5 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_6")]
    pub const braille_dot_6: Key = Key(ffi::GDK_KEY_braille_dot_6 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_7")]
    pub const braille_dot_7: Key = Key(ffi::GDK_KEY_braille_dot_7 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_8")]
    pub const braille_dot_8: Key = Key(ffi::GDK_KEY_braille_dot_8 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_9")]
    pub const braille_dot_9: Key = Key(ffi::GDK_KEY_braille_dot_9 as u32);
    #[doc(alias = "GDK_KEY_braille_dot_10")]
    pub const braille_dot_10: Key = Key(ffi::GDK_KEY_braille_dot_10 as u32);
    #[doc(alias = "GDK_KEY_braille_blank")]
    pub const braille_blank: Key = Key(ffi::GDK_KEY_braille_blank as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1")]
    pub const braille_dots_1: Key = Key(ffi::GDK_KEY_braille_dots_1 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2")]
    pub const braille_dots_2: Key = Key(ffi::GDK_KEY_braille_dots_2 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12")]
    pub const braille_dots_12: Key = Key(ffi::GDK_KEY_braille_dots_12 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3")]
    pub const braille_dots_3: Key = Key(ffi::GDK_KEY_braille_dots_3 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13")]
    pub const braille_dots_13: Key = Key(ffi::GDK_KEY_braille_dots_13 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23")]
    pub const braille_dots_23: Key = Key(ffi::GDK_KEY_braille_dots_23 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123")]
    pub const braille_dots_123: Key = Key(ffi::GDK_KEY_braille_dots_123 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4")]
    pub const braille_dots_4: Key = Key(ffi::GDK_KEY_braille_dots_4 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14")]
    pub const braille_dots_14: Key = Key(ffi::GDK_KEY_braille_dots_14 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24")]
    pub const braille_dots_24: Key = Key(ffi::GDK_KEY_braille_dots_24 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124")]
    pub const braille_dots_124: Key = Key(ffi::GDK_KEY_braille_dots_124 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34")]
    pub const braille_dots_34: Key = Key(ffi::GDK_KEY_braille_dots_34 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134")]
    pub const braille_dots_134: Key = Key(ffi::GDK_KEY_braille_dots_134 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234")]
    pub const braille_dots_234: Key = Key(ffi::GDK_KEY_braille_dots_234 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234")]
    pub const braille_dots_1234: Key = Key(ffi::GDK_KEY_braille_dots_1234 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_5")]
    pub const braille_dots_5: Key = Key(ffi::GDK_KEY_braille_dots_5 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_15")]
    pub const braille_dots_15: Key = Key(ffi::GDK_KEY_braille_dots_15 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_25")]
    pub const braille_dots_25: Key = Key(ffi::GDK_KEY_braille_dots_25 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_125")]
    pub const braille_dots_125: Key = Key(ffi::GDK_KEY_braille_dots_125 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_35")]
    pub const braille_dots_35: Key = Key(ffi::GDK_KEY_braille_dots_35 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_135")]
    pub const braille_dots_135: Key = Key(ffi::GDK_KEY_braille_dots_135 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_235")]
    pub const braille_dots_235: Key = Key(ffi::GDK_KEY_braille_dots_235 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1235")]
    pub const braille_dots_1235: Key = Key(ffi::GDK_KEY_braille_dots_1235 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_45")]
    pub const braille_dots_45: Key = Key(ffi::GDK_KEY_braille_dots_45 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_145")]
    pub const braille_dots_145: Key = Key(ffi::GDK_KEY_braille_dots_145 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_245")]
    pub const braille_dots_245: Key = Key(ffi::GDK_KEY_braille_dots_245 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1245")]
    pub const braille_dots_1245: Key = Key(ffi::GDK_KEY_braille_dots_1245 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_345")]
    pub const braille_dots_345: Key = Key(ffi::GDK_KEY_braille_dots_345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1345")]
    pub const braille_dots_1345: Key = Key(ffi::GDK_KEY_braille_dots_1345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2345")]
    pub const braille_dots_2345: Key = Key(ffi::GDK_KEY_braille_dots_2345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12345")]
    pub const braille_dots_12345: Key = Key(ffi::GDK_KEY_braille_dots_12345 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_6")]
    pub const braille_dots_6: Key = Key(ffi::GDK_KEY_braille_dots_6 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_16")]
    pub const braille_dots_16: Key = Key(ffi::GDK_KEY_braille_dots_16 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_26")]
    pub const braille_dots_26: Key = Key(ffi::GDK_KEY_braille_dots_26 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_126")]
    pub const braille_dots_126: Key = Key(ffi::GDK_KEY_braille_dots_126 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_36")]
    pub const braille_dots_36: Key = Key(ffi::GDK_KEY_braille_dots_36 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_136")]
    pub const braille_dots_136: Key = Key(ffi::GDK_KEY_braille_dots_136 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_236")]
    pub const braille_dots_236: Key = Key(ffi::GDK_KEY_braille_dots_236 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1236")]
    pub const braille_dots_1236: Key = Key(ffi::GDK_KEY_braille_dots_1236 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_46")]
    pub const braille_dots_46: Key = Key(ffi::GDK_KEY_braille_dots_46 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_146")]
    pub const braille_dots_146: Key = Key(ffi::GDK_KEY_braille_dots_146 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_246")]
    pub const braille_dots_246: Key = Key(ffi::GDK_KEY_braille_dots_246 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1246")]
    pub const braille_dots_1246: Key = Key(ffi::GDK_KEY_braille_dots_1246 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_346")]
    pub const braille_dots_346: Key = Key(ffi::GDK_KEY_braille_dots_346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1346")]
    pub const braille_dots_1346: Key = Key(ffi::GDK_KEY_braille_dots_1346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2346")]
    pub const braille_dots_2346: Key = Key(ffi::GDK_KEY_braille_dots_2346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12346")]
    pub const braille_dots_12346: Key = Key(ffi::GDK_KEY_braille_dots_12346 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_56")]
    pub const braille_dots_56: Key = Key(ffi::GDK_KEY_braille_dots_56 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_156")]
    pub const braille_dots_156: Key = Key(ffi::GDK_KEY_braille_dots_156 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_256")]
    pub const braille_dots_256: Key = Key(ffi::GDK_KEY_braille_dots_256 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1256")]
    pub const braille_dots_1256: Key = Key(ffi::GDK_KEY_braille_dots_1256 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_356")]
    pub const braille_dots_356: Key = Key(ffi::GDK_KEY_braille_dots_356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1356")]
    pub const braille_dots_1356: Key = Key(ffi::GDK_KEY_braille_dots_1356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2356")]
    pub const braille_dots_2356: Key = Key(ffi::GDK_KEY_braille_dots_2356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12356")]
    pub const braille_dots_12356: Key = Key(ffi::GDK_KEY_braille_dots_12356 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_456")]
    pub const braille_dots_456: Key = Key(ffi::GDK_KEY_braille_dots_456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1456")]
    pub const braille_dots_1456: Key = Key(ffi::GDK_KEY_braille_dots_1456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2456")]
    pub const braille_dots_2456: Key = Key(ffi::GDK_KEY_braille_dots_2456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12456")]
    pub const braille_dots_12456: Key = Key(ffi::GDK_KEY_braille_dots_12456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3456")]
    pub const braille_dots_3456: Key = Key(ffi::GDK_KEY_braille_dots_3456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13456")]
    pub const braille_dots_13456: Key = Key(ffi::GDK_KEY_braille_dots_13456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23456")]
    pub const braille_dots_23456: Key = Key(ffi::GDK_KEY_braille_dots_23456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123456")]
    pub const braille_dots_123456: Key = Key(ffi::GDK_KEY_braille_dots_123456 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_7")]
    pub const braille_dots_7: Key = Key(ffi::GDK_KEY_braille_dots_7 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_17")]
    pub const braille_dots_17: Key = Key(ffi::GDK_KEY_braille_dots_17 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_27")]
    pub const braille_dots_27: Key = Key(ffi::GDK_KEY_braille_dots_27 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_127")]
    pub const braille_dots_127: Key = Key(ffi::GDK_KEY_braille_dots_127 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_37")]
    pub const braille_dots_37: Key = Key(ffi::GDK_KEY_braille_dots_37 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_137")]
    pub const braille_dots_137: Key = Key(ffi::GDK_KEY_braille_dots_137 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_237")]
    pub const braille_dots_237: Key = Key(ffi::GDK_KEY_braille_dots_237 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1237")]
    pub const braille_dots_1237: Key = Key(ffi::GDK_KEY_braille_dots_1237 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_47")]
    pub const braille_dots_47: Key = Key(ffi::GDK_KEY_braille_dots_47 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_147")]
    pub const braille_dots_147: Key = Key(ffi::GDK_KEY_braille_dots_147 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_247")]
    pub const braille_dots_247: Key = Key(ffi::GDK_KEY_braille_dots_247 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1247")]
    pub const braille_dots_1247: Key = Key(ffi::GDK_KEY_braille_dots_1247 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_347")]
    pub const braille_dots_347: Key = Key(ffi::GDK_KEY_braille_dots_347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1347")]
    pub const braille_dots_1347: Key = Key(ffi::GDK_KEY_braille_dots_1347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2347")]
    pub const braille_dots_2347: Key = Key(ffi::GDK_KEY_braille_dots_2347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12347")]
    pub const braille_dots_12347: Key = Key(ffi::GDK_KEY_braille_dots_12347 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_57")]
    pub const braille_dots_57: Key = Key(ffi::GDK_KEY_braille_dots_57 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_157")]
    pub const braille_dots_157: Key = Key(ffi::GDK_KEY_braille_dots_157 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_257")]
    pub const braille_dots_257: Key = Key(ffi::GDK_KEY_braille_dots_257 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1257")]
    pub const braille_dots_1257: Key = Key(ffi::GDK_KEY_braille_dots_1257 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_357")]
    pub const braille_dots_357: Key = Key(ffi::GDK_KEY_braille_dots_357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1357")]
    pub const braille_dots_1357: Key = Key(ffi::GDK_KEY_braille_dots_1357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2357")]
    pub const braille_dots_2357: Key = Key(ffi::GDK_KEY_braille_dots_2357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12357")]
    pub const braille_dots_12357: Key = Key(ffi::GDK_KEY_braille_dots_12357 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_457")]
    pub const braille_dots_457: Key = Key(ffi::GDK_KEY_braille_dots_457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1457")]
    pub const braille_dots_1457: Key = Key(ffi::GDK_KEY_braille_dots_1457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2457")]
    pub const braille_dots_2457: Key = Key(ffi::GDK_KEY_braille_dots_2457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12457")]
    pub const braille_dots_12457: Key = Key(ffi::GDK_KEY_braille_dots_12457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3457")]
    pub const braille_dots_3457: Key = Key(ffi::GDK_KEY_braille_dots_3457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13457")]
    pub const braille_dots_13457: Key = Key(ffi::GDK_KEY_braille_dots_13457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23457")]
    pub const braille_dots_23457: Key = Key(ffi::GDK_KEY_braille_dots_23457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123457")]
    pub const braille_dots_123457: Key = Key(ffi::GDK_KEY_braille_dots_123457 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_67")]
    pub const braille_dots_67: Key = Key(ffi::GDK_KEY_braille_dots_67 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_167")]
    pub const braille_dots_167: Key = Key(ffi::GDK_KEY_braille_dots_167 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_267")]
    pub const braille_dots_267: Key = Key(ffi::GDK_KEY_braille_dots_267 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1267")]
    pub const braille_dots_1267: Key = Key(ffi::GDK_KEY_braille_dots_1267 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_367")]
    pub const braille_dots_367: Key = Key(ffi::GDK_KEY_braille_dots_367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1367")]
    pub const braille_dots_1367: Key = Key(ffi::GDK_KEY_braille_dots_1367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2367")]
    pub const braille_dots_2367: Key = Key(ffi::GDK_KEY_braille_dots_2367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12367")]
    pub const braille_dots_12367: Key = Key(ffi::GDK_KEY_braille_dots_12367 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_467")]
    pub const braille_dots_467: Key = Key(ffi::GDK_KEY_braille_dots_467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1467")]
    pub const braille_dots_1467: Key = Key(ffi::GDK_KEY_braille_dots_1467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2467")]
    pub const braille_dots_2467: Key = Key(ffi::GDK_KEY_braille_dots_2467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12467")]
    pub const braille_dots_12467: Key = Key(ffi::GDK_KEY_braille_dots_12467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3467")]
    pub const braille_dots_3467: Key = Key(ffi::GDK_KEY_braille_dots_3467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13467")]
    pub const braille_dots_13467: Key = Key(ffi::GDK_KEY_braille_dots_13467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23467")]
    pub const braille_dots_23467: Key = Key(ffi::GDK_KEY_braille_dots_23467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123467")]
    pub const braille_dots_123467: Key = Key(ffi::GDK_KEY_braille_dots_123467 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_567")]
    pub const braille_dots_567: Key = Key(ffi::GDK_KEY_braille_dots_567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1567")]
    pub const braille_dots_1567: Key = Key(ffi::GDK_KEY_braille_dots_1567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2567")]
    pub const braille_dots_2567: Key = Key(ffi::GDK_KEY_braille_dots_2567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12567")]
    pub const braille_dots_12567: Key = Key(ffi::GDK_KEY_braille_dots_12567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3567")]
    pub const braille_dots_3567: Key = Key(ffi::GDK_KEY_braille_dots_3567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13567")]
    pub const braille_dots_13567: Key = Key(ffi::GDK_KEY_braille_dots_13567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23567")]
    pub const braille_dots_23567: Key = Key(ffi::GDK_KEY_braille_dots_23567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123567")]
    pub const braille_dots_123567: Key = Key(ffi::GDK_KEY_braille_dots_123567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4567")]
    pub const braille_dots_4567: Key = Key(ffi::GDK_KEY_braille_dots_4567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14567")]
    pub const braille_dots_14567: Key = Key(ffi::GDK_KEY_braille_dots_14567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24567")]
    pub const braille_dots_24567: Key = Key(ffi::GDK_KEY_braille_dots_24567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124567")]
    pub const braille_dots_124567: Key = Key(ffi::GDK_KEY_braille_dots_124567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34567")]
    pub const braille_dots_34567: Key = Key(ffi::GDK_KEY_braille_dots_34567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134567")]
    pub const braille_dots_134567: Key = Key(ffi::GDK_KEY_braille_dots_134567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234567")]
    pub const braille_dots_234567: Key = Key(ffi::GDK_KEY_braille_dots_234567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234567")]
    pub const braille_dots_1234567: Key = Key(ffi::GDK_KEY_braille_dots_1234567 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_8")]
    pub const braille_dots_8: Key = Key(ffi::GDK_KEY_braille_dots_8 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_18")]
    pub const braille_dots_18: Key = Key(ffi::GDK_KEY_braille_dots_18 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_28")]
    pub const braille_dots_28: Key = Key(ffi::GDK_KEY_braille_dots_28 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_128")]
    pub const braille_dots_128: Key = Key(ffi::GDK_KEY_braille_dots_128 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_38")]
    pub const braille_dots_38: Key = Key(ffi::GDK_KEY_braille_dots_38 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_138")]
    pub const braille_dots_138: Key = Key(ffi::GDK_KEY_braille_dots_138 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_238")]
    pub const braille_dots_238: Key = Key(ffi::GDK_KEY_braille_dots_238 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1238")]
    pub const braille_dots_1238: Key = Key(ffi::GDK_KEY_braille_dots_1238 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_48")]
    pub const braille_dots_48: Key = Key(ffi::GDK_KEY_braille_dots_48 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_148")]
    pub const braille_dots_148: Key = Key(ffi::GDK_KEY_braille_dots_148 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_248")]
    pub const braille_dots_248: Key = Key(ffi::GDK_KEY_braille_dots_248 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1248")]
    pub const braille_dots_1248: Key = Key(ffi::GDK_KEY_braille_dots_1248 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_348")]
    pub const braille_dots_348: Key = Key(ffi::GDK_KEY_braille_dots_348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1348")]
    pub const braille_dots_1348: Key = Key(ffi::GDK_KEY_braille_dots_1348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2348")]
    pub const braille_dots_2348: Key = Key(ffi::GDK_KEY_braille_dots_2348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12348")]
    pub const braille_dots_12348: Key = Key(ffi::GDK_KEY_braille_dots_12348 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_58")]
    pub const braille_dots_58: Key = Key(ffi::GDK_KEY_braille_dots_58 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_158")]
    pub const braille_dots_158: Key = Key(ffi::GDK_KEY_braille_dots_158 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_258")]
    pub const braille_dots_258: Key = Key(ffi::GDK_KEY_braille_dots_258 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1258")]
    pub const braille_dots_1258: Key = Key(ffi::GDK_KEY_braille_dots_1258 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_358")]
    pub const braille_dots_358: Key = Key(ffi::GDK_KEY_braille_dots_358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1358")]
    pub const braille_dots_1358: Key = Key(ffi::GDK_KEY_braille_dots_1358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2358")]
    pub const braille_dots_2358: Key = Key(ffi::GDK_KEY_braille_dots_2358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12358")]
    pub const braille_dots_12358: Key = Key(ffi::GDK_KEY_braille_dots_12358 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_458")]
    pub const braille_dots_458: Key = Key(ffi::GDK_KEY_braille_dots_458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1458")]
    pub const braille_dots_1458: Key = Key(ffi::GDK_KEY_braille_dots_1458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2458")]
    pub const braille_dots_2458: Key = Key(ffi::GDK_KEY_braille_dots_2458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12458")]
    pub const braille_dots_12458: Key = Key(ffi::GDK_KEY_braille_dots_12458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3458")]
    pub const braille_dots_3458: Key = Key(ffi::GDK_KEY_braille_dots_3458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13458")]
    pub const braille_dots_13458: Key = Key(ffi::GDK_KEY_braille_dots_13458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23458")]
    pub const braille_dots_23458: Key = Key(ffi::GDK_KEY_braille_dots_23458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123458")]
    pub const braille_dots_123458: Key = Key(ffi::GDK_KEY_braille_dots_123458 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_68")]
    pub const braille_dots_68: Key = Key(ffi::GDK_KEY_braille_dots_68 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_168")]
    pub const braille_dots_168: Key = Key(ffi::GDK_KEY_braille_dots_168 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_268")]
    pub const braille_dots_268: Key = Key(ffi::GDK_KEY_braille_dots_268 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1268")]
    pub const braille_dots_1268: Key = Key(ffi::GDK_KEY_braille_dots_1268 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_368")]
    pub const braille_dots_368: Key = Key(ffi::GDK_KEY_braille_dots_368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1368")]
    pub const braille_dots_1368: Key = Key(ffi::GDK_KEY_braille_dots_1368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2368")]
    pub const braille_dots_2368: Key = Key(ffi::GDK_KEY_braille_dots_2368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12368")]
    pub const braille_dots_12368: Key = Key(ffi::GDK_KEY_braille_dots_12368 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_468")]
    pub const braille_dots_468: Key = Key(ffi::GDK_KEY_braille_dots_468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1468")]
    pub const braille_dots_1468: Key = Key(ffi::GDK_KEY_braille_dots_1468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2468")]
    pub const braille_dots_2468: Key = Key(ffi::GDK_KEY_braille_dots_2468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12468")]
    pub const braille_dots_12468: Key = Key(ffi::GDK_KEY_braille_dots_12468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3468")]
    pub const braille_dots_3468: Key = Key(ffi::GDK_KEY_braille_dots_3468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13468")]
    pub const braille_dots_13468: Key = Key(ffi::GDK_KEY_braille_dots_13468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23468")]
    pub const braille_dots_23468: Key = Key(ffi::GDK_KEY_braille_dots_23468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123468")]
    pub const braille_dots_123468: Key = Key(ffi::GDK_KEY_braille_dots_123468 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_568")]
    pub const braille_dots_568: Key = Key(ffi::GDK_KEY_braille_dots_568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1568")]
    pub const braille_dots_1568: Key = Key(ffi::GDK_KEY_braille_dots_1568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2568")]
    pub const braille_dots_2568: Key = Key(ffi::GDK_KEY_braille_dots_2568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12568")]
    pub const braille_dots_12568: Key = Key(ffi::GDK_KEY_braille_dots_12568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3568")]
    pub const braille_dots_3568: Key = Key(ffi::GDK_KEY_braille_dots_3568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13568")]
    pub const braille_dots_13568: Key = Key(ffi::GDK_KEY_braille_dots_13568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23568")]
    pub const braille_dots_23568: Key = Key(ffi::GDK_KEY_braille_dots_23568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123568")]
    pub const braille_dots_123568: Key = Key(ffi::GDK_KEY_braille_dots_123568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4568")]
    pub const braille_dots_4568: Key = Key(ffi::GDK_KEY_braille_dots_4568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14568")]
    pub const braille_dots_14568: Key = Key(ffi::GDK_KEY_braille_dots_14568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24568")]
    pub const braille_dots_24568: Key = Key(ffi::GDK_KEY_braille_dots_24568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124568")]
    pub const braille_dots_124568: Key = Key(ffi::GDK_KEY_braille_dots_124568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34568")]
    pub const braille_dots_34568: Key = Key(ffi::GDK_KEY_braille_dots_34568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134568")]
    pub const braille_dots_134568: Key = Key(ffi::GDK_KEY_braille_dots_134568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234568")]
    pub const braille_dots_234568: Key = Key(ffi::GDK_KEY_braille_dots_234568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234568")]
    pub const braille_dots_1234568: Key = Key(ffi::GDK_KEY_braille_dots_1234568 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_78")]
    pub const braille_dots_78: Key = Key(ffi::GDK_KEY_braille_dots_78 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_178")]
    pub const braille_dots_178: Key = Key(ffi::GDK_KEY_braille_dots_178 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_278")]
    pub const braille_dots_278: Key = Key(ffi::GDK_KEY_braille_dots_278 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1278")]
    pub const braille_dots_1278: Key = Key(ffi::GDK_KEY_braille_dots_1278 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_378")]
    pub const braille_dots_378: Key = Key(ffi::GDK_KEY_braille_dots_378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1378")]
    pub const braille_dots_1378: Key = Key(ffi::GDK_KEY_braille_dots_1378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2378")]
    pub const braille_dots_2378: Key = Key(ffi::GDK_KEY_braille_dots_2378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12378")]
    pub const braille_dots_12378: Key = Key(ffi::GDK_KEY_braille_dots_12378 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_478")]
    pub const braille_dots_478: Key = Key(ffi::GDK_KEY_braille_dots_478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1478")]
    pub const braille_dots_1478: Key = Key(ffi::GDK_KEY_braille_dots_1478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2478")]
    pub const braille_dots_2478: Key = Key(ffi::GDK_KEY_braille_dots_2478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12478")]
    pub const braille_dots_12478: Key = Key(ffi::GDK_KEY_braille_dots_12478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3478")]
    pub const braille_dots_3478: Key = Key(ffi::GDK_KEY_braille_dots_3478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13478")]
    pub const braille_dots_13478: Key = Key(ffi::GDK_KEY_braille_dots_13478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23478")]
    pub const braille_dots_23478: Key = Key(ffi::GDK_KEY_braille_dots_23478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123478")]
    pub const braille_dots_123478: Key = Key(ffi::GDK_KEY_braille_dots_123478 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_578")]
    pub const braille_dots_578: Key = Key(ffi::GDK_KEY_braille_dots_578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1578")]
    pub const braille_dots_1578: Key = Key(ffi::GDK_KEY_braille_dots_1578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2578")]
    pub const braille_dots_2578: Key = Key(ffi::GDK_KEY_braille_dots_2578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12578")]
    pub const braille_dots_12578: Key = Key(ffi::GDK_KEY_braille_dots_12578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3578")]
    pub const braille_dots_3578: Key = Key(ffi::GDK_KEY_braille_dots_3578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13578")]
    pub const braille_dots_13578: Key = Key(ffi::GDK_KEY_braille_dots_13578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23578")]
    pub const braille_dots_23578: Key = Key(ffi::GDK_KEY_braille_dots_23578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123578")]
    pub const braille_dots_123578: Key = Key(ffi::GDK_KEY_braille_dots_123578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4578")]
    pub const braille_dots_4578: Key = Key(ffi::GDK_KEY_braille_dots_4578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14578")]
    pub const braille_dots_14578: Key = Key(ffi::GDK_KEY_braille_dots_14578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24578")]
    pub const braille_dots_24578: Key = Key(ffi::GDK_KEY_braille_dots_24578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124578")]
    pub const braille_dots_124578: Key = Key(ffi::GDK_KEY_braille_dots_124578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34578")]
    pub const braille_dots_34578: Key = Key(ffi::GDK_KEY_braille_dots_34578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134578")]
    pub const braille_dots_134578: Key = Key(ffi::GDK_KEY_braille_dots_134578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234578")]
    pub const braille_dots_234578: Key = Key(ffi::GDK_KEY_braille_dots_234578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234578")]
    pub const braille_dots_1234578: Key = Key(ffi::GDK_KEY_braille_dots_1234578 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_678")]
    pub const braille_dots_678: Key = Key(ffi::GDK_KEY_braille_dots_678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1678")]
    pub const braille_dots_1678: Key = Key(ffi::GDK_KEY_braille_dots_1678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2678")]
    pub const braille_dots_2678: Key = Key(ffi::GDK_KEY_braille_dots_2678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12678")]
    pub const braille_dots_12678: Key = Key(ffi::GDK_KEY_braille_dots_12678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_3678")]
    pub const braille_dots_3678: Key = Key(ffi::GDK_KEY_braille_dots_3678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_13678")]
    pub const braille_dots_13678: Key = Key(ffi::GDK_KEY_braille_dots_13678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_23678")]
    pub const braille_dots_23678: Key = Key(ffi::GDK_KEY_braille_dots_23678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_123678")]
    pub const braille_dots_123678: Key = Key(ffi::GDK_KEY_braille_dots_123678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_4678")]
    pub const braille_dots_4678: Key = Key(ffi::GDK_KEY_braille_dots_4678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_14678")]
    pub const braille_dots_14678: Key = Key(ffi::GDK_KEY_braille_dots_14678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_24678")]
    pub const braille_dots_24678: Key = Key(ffi::GDK_KEY_braille_dots_24678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_124678")]
    pub const braille_dots_124678: Key = Key(ffi::GDK_KEY_braille_dots_124678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_34678")]
    pub const braille_dots_34678: Key = Key(ffi::GDK_KEY_braille_dots_34678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_134678")]
    pub const braille_dots_134678: Key = Key(ffi::GDK_KEY_braille_dots_134678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_234678")]
    pub const braille_dots_234678: Key = Key(ffi::GDK_KEY_braille_dots_234678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1234678")]
    pub const braille_dots_1234678: Key = Key(ffi::GDK_KEY_braille_dots_1234678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_5678")]
    pub const braille_dots_5678: Key = Key(ffi::GDK_KEY_braille_dots_5678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_15678")]
    pub const braille_dots_15678: Key = Key(ffi::GDK_KEY_braille_dots_15678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_25678")]
    pub const braille_dots_25678: Key = Key(ffi::GDK_KEY_braille_dots_25678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_125678")]
    pub const braille_dots_125678: Key = Key(ffi::GDK_KEY_braille_dots_125678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_35678")]
    pub const braille_dots_35678: Key = Key(ffi::GDK_KEY_braille_dots_35678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_135678")]
    pub const braille_dots_135678: Key = Key(ffi::GDK_KEY_braille_dots_135678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_235678")]
    pub const braille_dots_235678: Key = Key(ffi::GDK_KEY_braille_dots_235678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1235678")]
    pub const braille_dots_1235678: Key = Key(ffi::GDK_KEY_braille_dots_1235678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_45678")]
    pub const braille_dots_45678: Key = Key(ffi::GDK_KEY_braille_dots_45678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_145678")]
    pub const braille_dots_145678: Key = Key(ffi::GDK_KEY_braille_dots_145678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_245678")]
    pub const braille_dots_245678: Key = Key(ffi::GDK_KEY_braille_dots_245678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1245678")]
    pub const braille_dots_1245678: Key = Key(ffi::GDK_KEY_braille_dots_1245678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_345678")]
    pub const braille_dots_345678: Key = Key(ffi::GDK_KEY_braille_dots_345678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_1345678")]
    pub const braille_dots_1345678: Key = Key(ffi::GDK_KEY_braille_dots_1345678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_2345678")]
    pub const braille_dots_2345678: Key = Key(ffi::GDK_KEY_braille_dots_2345678 as u32);
    #[doc(alias = "GDK_KEY_braille_dots_12345678")]
    pub const braille_dots_12345678: Key = Key(ffi::GDK_KEY_braille_dots_12345678 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ng")]
    pub const Sinh_ng: Key = Key(ffi::GDK_KEY_Sinh_ng as u32);
    #[doc(alias = "GDK_KEY_Sinh_h2")]
    pub const Sinh_h2: Key = Key(ffi::GDK_KEY_Sinh_h2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_a")]
    pub const Sinh_a: Key = Key(ffi::GDK_KEY_Sinh_a as u32);
    #[doc(alias = "GDK_KEY_Sinh_aa")]
    pub const Sinh_aa: Key = Key(ffi::GDK_KEY_Sinh_aa as u32);
    #[doc(alias = "GDK_KEY_Sinh_ae")]
    pub const Sinh_ae: Key = Key(ffi::GDK_KEY_Sinh_ae as u32);
    #[doc(alias = "GDK_KEY_Sinh_aee")]
    pub const Sinh_aee: Key = Key(ffi::GDK_KEY_Sinh_aee as u32);
    #[doc(alias = "GDK_KEY_Sinh_i")]
    pub const Sinh_i: Key = Key(ffi::GDK_KEY_Sinh_i as u32);
    #[doc(alias = "GDK_KEY_Sinh_ii")]
    pub const Sinh_ii: Key = Key(ffi::GDK_KEY_Sinh_ii as u32);
    #[doc(alias = "GDK_KEY_Sinh_u")]
    pub const Sinh_u: Key = Key(ffi::GDK_KEY_Sinh_u as u32);
    #[doc(alias = "GDK_KEY_Sinh_uu")]
    pub const Sinh_uu: Key = Key(ffi::GDK_KEY_Sinh_uu as u32);
    #[doc(alias = "GDK_KEY_Sinh_ri")]
    pub const Sinh_ri: Key = Key(ffi::GDK_KEY_Sinh_ri as u32);
    #[doc(alias = "GDK_KEY_Sinh_rii")]
    pub const Sinh_rii: Key = Key(ffi::GDK_KEY_Sinh_rii as u32);
    #[doc(alias = "GDK_KEY_Sinh_lu")]
    pub const Sinh_lu: Key = Key(ffi::GDK_KEY_Sinh_lu as u32);
    #[doc(alias = "GDK_KEY_Sinh_luu")]
    pub const Sinh_luu: Key = Key(ffi::GDK_KEY_Sinh_luu as u32);
    #[doc(alias = "GDK_KEY_Sinh_e")]
    pub const Sinh_e: Key = Key(ffi::GDK_KEY_Sinh_e as u32);
    #[doc(alias = "GDK_KEY_Sinh_ee")]
    pub const Sinh_ee: Key = Key(ffi::GDK_KEY_Sinh_ee as u32);
    #[doc(alias = "GDK_KEY_Sinh_ai")]
    pub const Sinh_ai: Key = Key(ffi::GDK_KEY_Sinh_ai as u32);
    #[doc(alias = "GDK_KEY_Sinh_o")]
    pub const Sinh_o: Key = Key(ffi::GDK_KEY_Sinh_o as u32);
    #[doc(alias = "GDK_KEY_Sinh_oo")]
    pub const Sinh_oo: Key = Key(ffi::GDK_KEY_Sinh_oo as u32);
    #[doc(alias = "GDK_KEY_Sinh_au")]
    pub const Sinh_au: Key = Key(ffi::GDK_KEY_Sinh_au as u32);
    #[doc(alias = "GDK_KEY_Sinh_ka")]
    pub const Sinh_ka: Key = Key(ffi::GDK_KEY_Sinh_ka as u32);
    #[doc(alias = "GDK_KEY_Sinh_kha")]
    pub const Sinh_kha: Key = Key(ffi::GDK_KEY_Sinh_kha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ga")]
    pub const Sinh_ga: Key = Key(ffi::GDK_KEY_Sinh_ga as u32);
    #[doc(alias = "GDK_KEY_Sinh_gha")]
    pub const Sinh_gha: Key = Key(ffi::GDK_KEY_Sinh_gha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ng2")]
    pub const Sinh_ng2: Key = Key(ffi::GDK_KEY_Sinh_ng2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_nga")]
    pub const Sinh_nga: Key = Key(ffi::GDK_KEY_Sinh_nga as u32);
    #[doc(alias = "GDK_KEY_Sinh_ca")]
    pub const Sinh_ca: Key = Key(ffi::GDK_KEY_Sinh_ca as u32);
    #[doc(alias = "GDK_KEY_Sinh_cha")]
    pub const Sinh_cha: Key = Key(ffi::GDK_KEY_Sinh_cha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ja")]
    pub const Sinh_ja: Key = Key(ffi::GDK_KEY_Sinh_ja as u32);
    #[doc(alias = "GDK_KEY_Sinh_jha")]
    pub const Sinh_jha: Key = Key(ffi::GDK_KEY_Sinh_jha as u32);
    #[doc(alias = "GDK_KEY_Sinh_nya")]
    pub const Sinh_nya: Key = Key(ffi::GDK_KEY_Sinh_nya as u32);
    #[doc(alias = "GDK_KEY_Sinh_jnya")]
    pub const Sinh_jnya: Key = Key(ffi::GDK_KEY_Sinh_jnya as u32);
    #[doc(alias = "GDK_KEY_Sinh_nja")]
    pub const Sinh_nja: Key = Key(ffi::GDK_KEY_Sinh_nja as u32);
    #[doc(alias = "GDK_KEY_Sinh_tta")]
    pub const Sinh_tta: Key = Key(ffi::GDK_KEY_Sinh_tta as u32);
    #[doc(alias = "GDK_KEY_Sinh_ttha")]
    pub const Sinh_ttha: Key = Key(ffi::GDK_KEY_Sinh_ttha as u32);
    #[doc(alias = "GDK_KEY_Sinh_dda")]
    pub const Sinh_dda: Key = Key(ffi::GDK_KEY_Sinh_dda as u32);
    #[doc(alias = "GDK_KEY_Sinh_ddha")]
    pub const Sinh_ddha: Key = Key(ffi::GDK_KEY_Sinh_ddha as u32);
    #[doc(alias = "GDK_KEY_Sinh_nna")]
    pub const Sinh_nna: Key = Key(ffi::GDK_KEY_Sinh_nna as u32);
    #[doc(alias = "GDK_KEY_Sinh_ndda")]
    pub const Sinh_ndda: Key = Key(ffi::GDK_KEY_Sinh_ndda as u32);
    #[doc(alias = "GDK_KEY_Sinh_tha")]
    pub const Sinh_tha: Key = Key(ffi::GDK_KEY_Sinh_tha as u32);
    #[doc(alias = "GDK_KEY_Sinh_thha")]
    pub const Sinh_thha: Key = Key(ffi::GDK_KEY_Sinh_thha as u32);
    #[doc(alias = "GDK_KEY_Sinh_dha")]
    pub const Sinh_dha: Key = Key(ffi::GDK_KEY_Sinh_dha as u32);
    #[doc(alias = "GDK_KEY_Sinh_dhha")]
    pub const Sinh_dhha: Key = Key(ffi::GDK_KEY_Sinh_dhha as u32);
    #[doc(alias = "GDK_KEY_Sinh_na")]
    pub const Sinh_na: Key = Key(ffi::GDK_KEY_Sinh_na as u32);
    #[doc(alias = "GDK_KEY_Sinh_ndha")]
    pub const Sinh_ndha: Key = Key(ffi::GDK_KEY_Sinh_ndha as u32);
    #[doc(alias = "GDK_KEY_Sinh_pa")]
    pub const Sinh_pa: Key = Key(ffi::GDK_KEY_Sinh_pa as u32);
    #[doc(alias = "GDK_KEY_Sinh_pha")]
    pub const Sinh_pha: Key = Key(ffi::GDK_KEY_Sinh_pha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ba")]
    pub const Sinh_ba: Key = Key(ffi::GDK_KEY_Sinh_ba as u32);
    #[doc(alias = "GDK_KEY_Sinh_bha")]
    pub const Sinh_bha: Key = Key(ffi::GDK_KEY_Sinh_bha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ma")]
    pub const Sinh_ma: Key = Key(ffi::GDK_KEY_Sinh_ma as u32);
    #[doc(alias = "GDK_KEY_Sinh_mba")]
    pub const Sinh_mba: Key = Key(ffi::GDK_KEY_Sinh_mba as u32);
    #[doc(alias = "GDK_KEY_Sinh_ya")]
    pub const Sinh_ya: Key = Key(ffi::GDK_KEY_Sinh_ya as u32);
    #[doc(alias = "GDK_KEY_Sinh_ra")]
    pub const Sinh_ra: Key = Key(ffi::GDK_KEY_Sinh_ra as u32);
    #[doc(alias = "GDK_KEY_Sinh_la")]
    pub const Sinh_la: Key = Key(ffi::GDK_KEY_Sinh_la as u32);
    #[doc(alias = "GDK_KEY_Sinh_va")]
    pub const Sinh_va: Key = Key(ffi::GDK_KEY_Sinh_va as u32);
    #[doc(alias = "GDK_KEY_Sinh_sha")]
    pub const Sinh_sha: Key = Key(ffi::GDK_KEY_Sinh_sha as u32);
    #[doc(alias = "GDK_KEY_Sinh_ssha")]
    pub const Sinh_ssha: Key = Key(ffi::GDK_KEY_Sinh_ssha as u32);
    #[doc(alias = "GDK_KEY_Sinh_sa")]
    pub const Sinh_sa: Key = Key(ffi::GDK_KEY_Sinh_sa as u32);
    #[doc(alias = "GDK_KEY_Sinh_ha")]
    pub const Sinh_ha: Key = Key(ffi::GDK_KEY_Sinh_ha as u32);
    #[doc(alias = "GDK_KEY_Sinh_lla")]
    pub const Sinh_lla: Key = Key(ffi::GDK_KEY_Sinh_lla as u32);
    #[doc(alias = "GDK_KEY_Sinh_fa")]
    pub const Sinh_fa: Key = Key(ffi::GDK_KEY_Sinh_fa as u32);
    #[doc(alias = "GDK_KEY_Sinh_al")]
    pub const Sinh_al: Key = Key(ffi::GDK_KEY_Sinh_al as u32);
    #[doc(alias = "GDK_KEY_Sinh_aa2")]
    pub const Sinh_aa2: Key = Key(ffi::GDK_KEY_Sinh_aa2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ae2")]
    pub const Sinh_ae2: Key = Key(ffi::GDK_KEY_Sinh_ae2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_aee2")]
    pub const Sinh_aee2: Key = Key(ffi::GDK_KEY_Sinh_aee2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_i2")]
    pub const Sinh_i2: Key = Key(ffi::GDK_KEY_Sinh_i2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ii2")]
    pub const Sinh_ii2: Key = Key(ffi::GDK_KEY_Sinh_ii2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_u2")]
    pub const Sinh_u2: Key = Key(ffi::GDK_KEY_Sinh_u2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_uu2")]
    pub const Sinh_uu2: Key = Key(ffi::GDK_KEY_Sinh_uu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ru2")]
    pub const Sinh_ru2: Key = Key(ffi::GDK_KEY_Sinh_ru2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_e2")]
    pub const Sinh_e2: Key = Key(ffi::GDK_KEY_Sinh_e2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ee2")]
    pub const Sinh_ee2: Key = Key(ffi::GDK_KEY_Sinh_ee2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ai2")]
    pub const Sinh_ai2: Key = Key(ffi::GDK_KEY_Sinh_ai2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_o2")]
    pub const Sinh_o2: Key = Key(ffi::GDK_KEY_Sinh_o2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_oo2")]
    pub const Sinh_oo2: Key = Key(ffi::GDK_KEY_Sinh_oo2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_au2")]
    pub const Sinh_au2: Key = Key(ffi::GDK_KEY_Sinh_au2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_lu2")]
    pub const Sinh_lu2: Key = Key(ffi::GDK_KEY_Sinh_lu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_ruu2")]
    pub const Sinh_ruu2: Key = Key(ffi::GDK_KEY_Sinh_ruu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_luu2")]
    pub const Sinh_luu2: Key = Key(ffi::GDK_KEY_Sinh_luu2 as u32);
    #[doc(alias = "GDK_KEY_Sinh_kunddaliya")]
    pub const Sinh_kunddaliya: Key = Key(ffi::GDK_KEY_Sinh_kunddaliya as u32);
    #[doc(alias = "GDK_KEY_ModeLock")]
    pub const ModeLock: Key = Key(ffi::GDK_KEY_ModeLock as u32);
    #[doc(alias = "GDK_KEY_MonBrightnessUp")]
    pub const MonBrightnessUp: Key = Key(ffi::GDK_KEY_MonBrightnessUp as u32);
    #[doc(alias = "GDK_KEY_MonBrightnessDown")]
    pub const MonBrightnessDown: Key = Key(ffi::GDK_KEY_MonBrightnessDown as u32);
    #[doc(alias = "GDK_KEY_KbdLightOnOff")]
    pub const KbdLightOnOff: Key = Key(ffi::GDK_KEY_KbdLightOnOff as u32);
    #[doc(alias = "GDK_KEY_KbdBrightnessUp")]
    pub const KbdBrightnessUp: Key = Key(ffi::GDK_KEY_KbdBrightnessUp as u32);
    #[doc(alias = "GDK_KEY_KbdBrightnessDown")]
    pub const KbdBrightnessDown: Key = Key(ffi::GDK_KEY_KbdBrightnessDown as u32);
    #[doc(alias = "GDK_KEY_Standby")]
    pub const Standby: Key = Key(ffi::GDK_KEY_Standby as u32);
    #[doc(alias = "GDK_KEY_AudioLowerVolume")]
    pub const AudioLowerVolume: Key = Key(ffi::GDK_KEY_AudioLowerVolume as u32);
    #[doc(alias = "GDK_KEY_AudioMute")]
    pub const AudioMute: Key = Key(ffi::GDK_KEY_AudioMute as u32);
    #[doc(alias = "GDK_KEY_AudioRaiseVolume")]
    pub const AudioRaiseVolume: Key = Key(ffi::GDK_KEY_AudioRaiseVolume as u32);
    #[doc(alias = "GDK_KEY_AudioPlay")]
    pub const AudioPlay: Key = Key(ffi::GDK_KEY_AudioPlay as u32);
    #[doc(alias = "GDK_KEY_AudioStop")]
    pub const AudioStop: Key = Key(ffi::GDK_KEY_AudioStop as u32);
    #[doc(alias = "GDK_KEY_AudioPrev")]
    pub const AudioPrev: Key = Key(ffi::GDK_KEY_AudioPrev as u32);
    #[doc(alias = "GDK_KEY_AudioNext")]
    pub const AudioNext: Key = Key(ffi::GDK_KEY_AudioNext as u32);
    #[doc(alias = "GDK_KEY_HomePage")]
    pub const HomePage: Key = Key(ffi::GDK_KEY_HomePage as u32);
    #[doc(alias = "GDK_KEY_Mail")]
    pub const Mail: Key = Key(ffi::GDK_KEY_Mail as u32);
    #[doc(alias = "GDK_KEY_Start")]
    pub const Start: Key = Key(ffi::GDK_KEY_Start as u32);
    #[doc(alias = "GDK_KEY_Search")]
    pub const Search: Key = Key(ffi::GDK_KEY_Search as u32);
    #[doc(alias = "GDK_KEY_AudioRecord")]
    pub const AudioRecord: Key = Key(ffi::GDK_KEY_AudioRecord as u32);
    #[doc(alias = "GDK_KEY_Calculator")]
    pub const Calculator: Key = Key(ffi::GDK_KEY_Calculator as u32);
    #[doc(alias = "GDK_KEY_Memo")]
    pub const Memo: Key = Key(ffi::GDK_KEY_Memo as u32);
    #[doc(alias = "GDK_KEY_ToDoList")]
    pub const ToDoList: Key = Key(ffi::GDK_KEY_ToDoList as u32);
    #[doc(alias = "GDK_KEY_Calendar")]
    pub const Calendar: Key = Key(ffi::GDK_KEY_Calendar as u32);
    #[doc(alias = "GDK_KEY_PowerDown")]
    pub const PowerDown: Key = Key(ffi::GDK_KEY_PowerDown as u32);
    #[doc(alias = "GDK_KEY_ContrastAdjust")]
    pub const ContrastAdjust: Key = Key(ffi::GDK_KEY_ContrastAdjust as u32);
    #[doc(alias = "GDK_KEY_RockerUp")]
    pub const RockerUp: Key = Key(ffi::GDK_KEY_RockerUp as u32);
    #[doc(alias = "GDK_KEY_RockerDown")]
    pub const RockerDown: Key = Key(ffi::GDK_KEY_RockerDown as u32);
    #[doc(alias = "GDK_KEY_RockerEnter")]
    pub const RockerEnter: Key = Key(ffi::GDK_KEY_RockerEnter as u32);
    #[doc(alias = "GDK_KEY_Back")]
    pub const Back: Key = Key(ffi::GDK_KEY_Back as u32);
    #[doc(alias = "GDK_KEY_Forward")]
    pub const Forward: Key = Key(ffi::GDK_KEY_Forward as u32);
    #[doc(alias = "GDK_KEY_Stop")]
    pub const Stop: Key = Key(ffi::GDK_KEY_Stop as u32);
    #[doc(alias = "GDK_KEY_Refresh")]
    pub const Refresh: Key = Key(ffi::GDK_KEY_Refresh as u32);
    #[doc(alias = "GDK_KEY_PowerOff")]
    pub const PowerOff: Key = Key(ffi::GDK_KEY_PowerOff as u32);
    #[doc(alias = "GDK_KEY_WakeUp")]
    pub const WakeUp: Key = Key(ffi::GDK_KEY_WakeUp as u32);
    #[doc(alias = "GDK_KEY_Eject")]
    pub const Eject: Key = Key(ffi::GDK_KEY_Eject as u32);
    #[doc(alias = "GDK_KEY_ScreenSaver")]
    pub const ScreenSaver: Key = Key(ffi::GDK_KEY_ScreenSaver as u32);
    #[doc(alias = "GDK_KEY_WWW")]
    pub const WWW: Key = Key(ffi::GDK_KEY_WWW as u32);
    #[doc(alias = "GDK_KEY_Sleep")]
    pub const Sleep: Key = Key(ffi::GDK_KEY_Sleep as u32);
    #[doc(alias = "GDK_KEY_Favorites")]
    pub const Favorites: Key = Key(ffi::GDK_KEY_Favorites as u32);
    #[doc(alias = "GDK_KEY_AudioPause")]
    pub const AudioPause: Key = Key(ffi::GDK_KEY_AudioPause as u32);
    #[doc(alias = "GDK_KEY_AudioMedia")]
    pub const AudioMedia: Key = Key(ffi::GDK_KEY_AudioMedia as u32);
    #[doc(alias = "GDK_KEY_MyComputer")]
    pub const MyComputer: Key = Key(ffi::GDK_KEY_MyComputer as u32);
    #[doc(alias = "GDK_KEY_VendorHome")]
    pub const VendorHome: Key = Key(ffi::GDK_KEY_VendorHome as u32);
    #[doc(alias = "GDK_KEY_LightBulb")]
    pub const LightBulb: Key = Key(ffi::GDK_KEY_LightBulb as u32);
    #[doc(alias = "GDK_KEY_Shop")]
    pub const Shop: Key = Key(ffi::GDK_KEY_Shop as u32);
    #[doc(alias = "GDK_KEY_History")]
    pub const History: Key = Key(ffi::GDK_KEY_History as u32);
    #[doc(alias = "GDK_KEY_OpenURL")]
    pub const OpenURL: Key = Key(ffi::GDK_KEY_OpenURL as u32);
    #[doc(alias = "GDK_KEY_AddFavorite")]
    pub const AddFavorite: Key = Key(ffi::GDK_KEY_AddFavorite as u32);
    #[doc(alias = "GDK_KEY_HotLinks")]
    pub const HotLinks: Key = Key(ffi::GDK_KEY_HotLinks as u32);
    #[doc(alias = "GDK_KEY_BrightnessAdjust")]
    pub const BrightnessAdjust: Key = Key(ffi::GDK_KEY_BrightnessAdjust as u32);
    #[doc(alias = "GDK_KEY_Finance")]
    pub const Finance: Key = Key(ffi::GDK_KEY_Finance as u32);
    #[doc(alias = "GDK_KEY_Community")]
    pub const Community: Key = Key(ffi::GDK_KEY_Community as u32);
    #[doc(alias = "GDK_KEY_AudioRewind")]
    pub const AudioRewind: Key = Key(ffi::GDK_KEY_AudioRewind as u32);
    #[doc(alias = "GDK_KEY_BackForward")]
    pub const BackForward: Key = Key(ffi::GDK_KEY_BackForward as u32);
    #[doc(alias = "GDK_KEY_Launch0")]
    pub const Launch0: Key = Key(ffi::GDK_KEY_Launch0 as u32);
    #[doc(alias = "GDK_KEY_Launch1")]
    pub const Launch1: Key = Key(ffi::GDK_KEY_Launch1 as u32);
    #[doc(alias = "GDK_KEY_Launch2")]
    pub const Launch2: Key = Key(ffi::GDK_KEY_Launch2 as u32);
    #[doc(alias = "GDK_KEY_Launch3")]
    pub const Launch3: Key = Key(ffi::GDK_KEY_Launch3 as u32);
    #[doc(alias = "GDK_KEY_Launch4")]
    pub const Launch4: Key = Key(ffi::GDK_KEY_Launch4 as u32);
    #[doc(alias = "GDK_KEY_Launch5")]
    pub const Launch5: Key = Key(ffi::GDK_KEY_Launch5 as u32);
    #[doc(alias = "GDK_KEY_Launch6")]
    pub const Launch6: Key = Key(ffi::GDK_KEY_Launch6 as u32);
    #[doc(alias = "GDK_KEY_Launch7")]
    pub const Launch7: Key = Key(ffi::GDK_KEY_Launch7 as u32);
    #[doc(alias = "GDK_KEY_Launch8")]
    pub const Launch8: Key = Key(ffi::GDK_KEY_Launch8 as u32);
    #[doc(alias = "GDK_KEY_Launch9")]
    pub const Launch9: Key = Key(ffi::GDK_KEY_Launch9 as u32);
    #[doc(alias = "GDK_KEY_LaunchA")]
    pub const LaunchA: Key = Key(ffi::GDK_KEY_LaunchA as u32);
    #[doc(alias = "GDK_KEY_LaunchB")]
    pub const LaunchB: Key = Key(ffi::GDK_KEY_LaunchB as u32);
    #[doc(alias = "GDK_KEY_LaunchC")]
    pub const LaunchC: Key = Key(ffi::GDK_KEY_LaunchC as u32);
    #[doc(alias = "GDK_KEY_LaunchD")]
    pub const LaunchD: Key = Key(ffi::GDK_KEY_LaunchD as u32);
    #[doc(alias = "GDK_KEY_LaunchE")]
    pub const LaunchE: Key = Key(ffi::GDK_KEY_LaunchE as u32);
    #[doc(alias = "GDK_KEY_LaunchF")]
    pub const LaunchF: Key = Key(ffi::GDK_KEY_LaunchF as u32);
    #[doc(alias = "GDK_KEY_ApplicationLeft")]
    pub const ApplicationLeft: Key = Key(ffi::GDK_KEY_ApplicationLeft as u32);
    #[doc(alias = "GDK_KEY_ApplicationRight")]
    pub const ApplicationRight: Key = Key(ffi::GDK_KEY_ApplicationRight as u32);
    #[doc(alias = "GDK_KEY_Book")]
    pub const Book: Key = Key(ffi::GDK_KEY_Book as u32);
    #[doc(alias = "GDK_KEY_CD")]
    pub const CD: Key = Key(ffi::GDK_KEY_CD as u32);
    #[doc(alias = "GDK_KEY_WindowClear")]
    pub const WindowClear: Key = Key(ffi::GDK_KEY_WindowClear as u32);
    #[doc(alias = "GDK_KEY_Close")]
    pub const Close: Key = Key(ffi::GDK_KEY_Close as u32);
    #[doc(alias = "GDK_KEY_Copy")]
    pub const Copy: Key = Key(ffi::GDK_KEY_Copy as u32);
    #[doc(alias = "GDK_KEY_Cut")]
    pub const Cut: Key = Key(ffi::GDK_KEY_Cut as u32);
    #[doc(alias = "GDK_KEY_Display")]
    pub const Display: Key = Key(ffi::GDK_KEY_Display as u32);
    #[doc(alias = "GDK_KEY_DOS")]
    pub const DOS: Key = Key(ffi::GDK_KEY_DOS as u32);
    #[doc(alias = "GDK_KEY_Documents")]
    pub const Documents: Key = Key(ffi::GDK_KEY_Documents as u32);
    #[doc(alias = "GDK_KEY_Excel")]
    pub const Excel: Key = Key(ffi::GDK_KEY_Excel as u32);
    #[doc(alias = "GDK_KEY_Explorer")]
    pub const Explorer: Key = Key(ffi::GDK_KEY_Explorer as u32);
    #[doc(alias = "GDK_KEY_Game")]
    pub const Game: Key = Key(ffi::GDK_KEY_Game as u32);
    #[doc(alias = "GDK_KEY_Go")]
    pub const Go: Key = Key(ffi::GDK_KEY_Go as u32);
    #[doc(alias = "GDK_KEY_iTouch")]
    pub const iTouch: Key = Key(ffi::GDK_KEY_iTouch as u32);
    #[doc(alias = "GDK_KEY_LogOff")]
    pub const LogOff: Key = Key(ffi::GDK_KEY_LogOff as u32);
    #[doc(alias = "GDK_KEY_Market")]
    pub const Market: Key = Key(ffi::GDK_KEY_Market as u32);
    #[doc(alias = "GDK_KEY_Meeting")]
    pub const Meeting: Key = Key(ffi::GDK_KEY_Meeting as u32);
    #[doc(alias = "GDK_KEY_MenuKB")]
    pub const MenuKB: Key = Key(ffi::GDK_KEY_MenuKB as u32);
    #[doc(alias = "GDK_KEY_MenuPB")]
    pub const MenuPB: Key = Key(ffi::GDK_KEY_MenuPB as u32);
    #[doc(alias = "GDK_KEY_MySites")]
    pub const MySites: Key = Key(ffi::GDK_KEY_MySites as u32);
    #[doc(alias = "GDK_KEY_New")]
    pub const New: Key = Key(ffi::GDK_KEY_New as u32);
    #[doc(alias = "GDK_KEY_News")]
    pub const News: Key = Key(ffi::GDK_KEY_News as u32);
    #[doc(alias = "GDK_KEY_OfficeHome")]
    pub const OfficeHome: Key = Key(ffi::GDK_KEY_OfficeHome as u32);
    #[doc(alias = "GDK_KEY_Open")]
    pub const Open: Key = Key(ffi::GDK_KEY_Open as u32);
    #[doc(alias = "GDK_KEY_Option")]
    pub const Option: Key = Key(ffi::GDK_KEY_Option as u32);
    #[doc(alias = "GDK_KEY_Paste")]
    pub const Paste: Key = Key(ffi::GDK_KEY_Paste as u32);
    #[doc(alias = "GDK_KEY_Phone")]
    pub const Phone: Key = Key(ffi::GDK_KEY_Phone as u32);
    #[doc(alias = "GDK_KEY_Reply")]
    pub const Reply: Key = Key(ffi::GDK_KEY_Reply as u32);
    #[doc(alias = "GDK_KEY_Reload")]
    pub const Reload: Key = Key(ffi::GDK_KEY_Reload as u32);
    #[doc(alias = "GDK_KEY_RotateWindows")]
    pub const RotateWindows: Key = Key(ffi::GDK_KEY_RotateWindows as u32);
    #[doc(alias = "GDK_KEY_RotationPB")]
    pub const RotationPB: Key = Key(ffi::GDK_KEY_RotationPB as u32);
    #[doc(alias = "GDK_KEY_RotationKB")]
    pub const RotationKB: Key = Key(ffi::GDK_KEY_RotationKB as u32);
    #[doc(alias = "GDK_KEY_Save")]
    pub const Save: Key = Key(ffi::GDK_KEY_Save as u32);
    #[doc(alias = "GDK_KEY_ScrollUp")]
    pub const ScrollUp: Key = Key(ffi::GDK_KEY_ScrollUp as u32);
    #[doc(alias = "GDK_KEY_ScrollDown")]
    pub const ScrollDown: Key = Key(ffi::GDK_KEY_ScrollDown as u32);
    #[doc(alias = "GDK_KEY_ScrollClick")]
    pub const ScrollClick: Key = Key(ffi::GDK_KEY_ScrollClick as u32);
    #[doc(alias = "GDK_KEY_Send")]
    pub const Send: Key = Key(ffi::GDK_KEY_Send as u32);
    #[doc(alias = "GDK_KEY_Spell")]
    pub const Spell: Key = Key(ffi::GDK_KEY_Spell as u32);
    #[doc(alias = "GDK_KEY_SplitScreen")]
    pub const SplitScreen: Key = Key(ffi::GDK_KEY_SplitScreen as u32);
    #[doc(alias = "GDK_KEY_Support")]
    pub const Support: Key = Key(ffi::GDK_KEY_Support as u32);
    #[doc(alias = "GDK_KEY_TaskPane")]
    pub const TaskPane: Key = Key(ffi::GDK_KEY_TaskPane as u32);
    #[doc(alias = "GDK_KEY_Terminal")]
    pub const Terminal: Key = Key(ffi::GDK_KEY_Terminal as u32);
    #[doc(alias = "GDK_KEY_Tools")]
    pub const Tools: Key = Key(ffi::GDK_KEY_Tools as u32);
    #[doc(alias = "GDK_KEY_Travel")]
    pub const Travel: Key = Key(ffi::GDK_KEY_Travel as u32);
    #[doc(alias = "GDK_KEY_UserPB")]
    pub const UserPB: Key = Key(ffi::GDK_KEY_UserPB as u32);
    #[doc(alias = "GDK_KEY_User1KB")]
    pub const User1KB: Key = Key(ffi::GDK_KEY_User1KB as u32);
    #[doc(alias = "GDK_KEY_User2KB")]
    pub const User2KB: Key = Key(ffi::GDK_KEY_User2KB as u32);
    #[doc(alias = "GDK_KEY_Video")]
    pub const Video: Key = Key(ffi::GDK_KEY_Video as u32);
    #[doc(alias = "GDK_KEY_WheelButton")]
    pub const WheelButton: Key = Key(ffi::GDK_KEY_WheelButton as u32);
    #[doc(alias = "GDK_KEY_Word")]
    pub const Word: Key = Key(ffi::GDK_KEY_Word as u32);
    #[doc(alias = "GDK_KEY_Xfer")]
    pub const Xfer: Key = Key(ffi::GDK_KEY_Xfer as u32);
    #[doc(alias = "GDK_KEY_ZoomIn")]
    pub const ZoomIn: Key = Key(ffi::GDK_KEY_ZoomIn as u32);
    #[doc(alias = "GDK_KEY_ZoomOut")]
    pub const ZoomOut: Key = Key(ffi::GDK_KEY_ZoomOut as u32);
    #[doc(alias = "GDK_KEY_Away")]
    pub const Away: Key = Key(ffi::GDK_KEY_Away as u32);
    #[doc(alias = "GDK_KEY_Messenger")]
    pub const Messenger: Key = Key(ffi::GDK_KEY_Messenger as u32);
    #[doc(alias = "GDK_KEY_WebCam")]
    pub const WebCam: Key = Key(ffi::GDK_KEY_WebCam as u32);
    #[doc(alias = "GDK_KEY_MailForward")]
    pub const MailForward: Key = Key(ffi::GDK_KEY_MailForward as u32);
    #[doc(alias = "GDK_KEY_Pictures")]
    pub const Pictures: Key = Key(ffi::GDK_KEY_Pictures as u32);
    #[doc(alias = "GDK_KEY_Music")]
    pub const Music: Key = Key(ffi::GDK_KEY_Music as u32);
    #[doc(alias = "GDK_KEY_Battery")]
    pub const Battery: Key = Key(ffi::GDK_KEY_Battery as u32);
    #[doc(alias = "GDK_KEY_Bluetooth")]
    pub const Bluetooth: Key = Key(ffi::GDK_KEY_Bluetooth as u32);
    #[doc(alias = "GDK_KEY_WLAN")]
    pub const WLAN: Key = Key(ffi::GDK_KEY_WLAN as u32);
    #[doc(alias = "GDK_KEY_UWB")]
    pub const UWB: Key = Key(ffi::GDK_KEY_UWB as u32);
    #[doc(alias = "GDK_KEY_AudioForward")]
    pub const AudioForward: Key = Key(ffi::GDK_KEY_AudioForward as u32);
    #[doc(alias = "GDK_KEY_AudioRepeat")]
    pub const AudioRepeat: Key = Key(ffi::GDK_KEY_AudioRepeat as u32);
    #[doc(alias = "GDK_KEY_AudioRandomPlay")]
    pub const AudioRandomPlay: Key = Key(ffi::GDK_KEY_AudioRandomPlay as u32);
    #[doc(alias = "GDK_KEY_Subtitle")]
    pub const Subtitle: Key = Key(ffi::GDK_KEY_Subtitle as u32);
    #[doc(alias = "GDK_KEY_AudioCycleTrack")]
    pub const AudioCycleTrack: Key = Key(ffi::GDK_KEY_AudioCycleTrack as u32);
    #[doc(alias = "GDK_KEY_CycleAngle")]
    pub const CycleAngle: Key = Key(ffi::GDK_KEY_CycleAngle as u32);
    #[doc(alias = "GDK_KEY_FrameBack")]
    pub const FrameBack: Key = Key(ffi::GDK_KEY_FrameBack as u32);
    #[doc(alias = "GDK_KEY_FrameForward")]
    pub const FrameForward: Key = Key(ffi::GDK_KEY_FrameForward as u32);
    #[doc(alias = "GDK_KEY_Time")]
    pub const Time: Key = Key(ffi::GDK_KEY_Time as u32);
    #[doc(alias = "GDK_KEY_SelectButton")]
    pub const SelectButton: Key = Key(ffi::GDK_KEY_SelectButton as u32);
    #[doc(alias = "GDK_KEY_View")]
    pub const View: Key = Key(ffi::GDK_KEY_View as u32);
    #[doc(alias = "GDK_KEY_TopMenu")]
    pub const TopMenu: Key = Key(ffi::GDK_KEY_TopMenu as u32);
    #[doc(alias = "GDK_KEY_Red")]
    pub const Red: Key = Key(ffi::GDK_KEY_Red as u32);
    #[doc(alias = "GDK_KEY_Green")]
    pub const Green: Key = Key(ffi::GDK_KEY_Green as u32);
    #[doc(alias = "GDK_KEY_Yellow")]
    pub const Yellow: Key = Key(ffi::GDK_KEY_Yellow as u32);
    #[doc(alias = "GDK_KEY_Blue")]
    pub const Blue: Key = Key(ffi::GDK_KEY_Blue as u32);
    #[doc(alias = "GDK_KEY_Suspend")]
    pub const Suspend: Key = Key(ffi::GDK_KEY_Suspend as u32);
    #[doc(alias = "GDK_KEY_Hibernate")]
    pub const Hibernate: Key = Key(ffi::GDK_KEY_Hibernate as u32);
    #[doc(alias = "GDK_KEY_TouchpadToggle")]
    pub const TouchpadToggle: Key = Key(ffi::GDK_KEY_TouchpadToggle as u32);
    #[doc(alias = "GDK_KEY_TouchpadOn")]
    pub const TouchpadOn: Key = Key(ffi::GDK_KEY_TouchpadOn as u32);
    #[doc(alias = "GDK_KEY_TouchpadOff")]
    pub const TouchpadOff: Key = Key(ffi::GDK_KEY_TouchpadOff as u32);
    #[doc(alias = "GDK_KEY_AudioMicMute")]
    pub const AudioMicMute: Key = Key(ffi::GDK_KEY_AudioMicMute as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_1")]
    pub const Switch_VT_1: Key = Key(ffi::GDK_KEY_Switch_VT_1 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_2")]
    pub const Switch_VT_2: Key = Key(ffi::GDK_KEY_Switch_VT_2 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_3")]
    pub const Switch_VT_3: Key = Key(ffi::GDK_KEY_Switch_VT_3 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_4")]
    pub const Switch_VT_4: Key = Key(ffi::GDK_KEY_Switch_VT_4 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_5")]
    pub const Switch_VT_5: Key = Key(ffi::GDK_KEY_Switch_VT_5 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_6")]
    pub const Switch_VT_6: Key = Key(ffi::GDK_KEY_Switch_VT_6 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_7")]
    pub const Switch_VT_7: Key = Key(ffi::GDK_KEY_Switch_VT_7 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_8")]
    pub const Switch_VT_8: Key = Key(ffi::GDK_KEY_Switch_VT_8 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_9")]
    pub const Switch_VT_9: Key = Key(ffi::GDK_KEY_Switch_VT_9 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_10")]
    pub const Switch_VT_10: Key = Key(ffi::GDK_KEY_Switch_VT_10 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_11")]
    pub const Switch_VT_11: Key = Key(ffi::GDK_KEY_Switch_VT_11 as u32);
    #[doc(alias = "GDK_KEY_Switch_VT_12")]
    pub const Switch_VT_12: Key = Key(ffi::GDK_KEY_Switch_VT_12 as u32);
    #[doc(alias = "GDK_KEY_Ungrab")]
    pub const Ungrab: Key = Key(ffi::GDK_KEY_Ungrab as u32);
    #[doc(alias = "GDK_KEY_ClearGrab")]
    pub const ClearGrab: Key = Key(ffi::GDK_KEY_ClearGrab as u32);
    #[doc(alias = "GDK_KEY_Next_VMode")]
    pub const Next_VMode: Key = Key(ffi::GDK_KEY_Next_VMode as u32);
    #[doc(alias = "GDK_KEY_Prev_VMode")]
    pub const Prev_VMode: Key = Key(ffi::GDK_KEY_Prev_VMode as u32);
    #[doc(alias = "GDK_KEY_LogWindowTree")]
    pub const LogWindowTree: Key = Key(ffi::GDK_KEY_LogWindowTree as u32);
    #[doc(alias = "GDK_KEY_LogGrabInfo")]
    pub const LogGrabInfo: Key = Key(ffi::GDK_KEY_LogGrabInfo as u32);
}
