/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*  This file is part of gnome-spell bonobo component
    copied from echo.c written by Miguel de Icaza and updated for Spell.idl needs

    Copyright (C) 1999, 2000 Helix Code, Inc.
    Authors:                 Radek Doulik <rodo@helixcode.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <config.h>
#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <bonobo.h>

#include "Spell.h"
#include "dictionary.h"

static BonoboObjectClass                  *dictionary_parent_class;
static POA_GNOME_Spell_Dictionary__vepv    dictionary_vepv;

#define DICT_DEBUG(x) x

static void release_engines (GNOMESpellDictionary *dict);

static void
raise_error (CORBA_Environment * ev, const gchar *s)
{
	GNOME_Spell_Dictionary_Error *exception;
	exception = GNOME_Spell_Dictionary_Error__alloc ();
		
	exception->error = CORBA_string_dup (s);
	CORBA_exception_set (ev, CORBA_USER_EXCEPTION,
			     ex_GNOME_Spell_Dictionary_Error,
			     exception);
}

static void
dictionary_object_init (GtkObject *object)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (object);

	dict->changed = TRUE;
	dict->engines = NULL;
}

static void
dictionary_object_destroy (GtkObject *object)
{
	GNOMESpellDictionary *dictionary = GNOME_SPELL_DICTIONARY (object);

	release_engines (dictionary);

	GTK_OBJECT_CLASS (dictionary_parent_class)->destroy (object);
}

static SpellEngine *
new_engine (const gchar *language)
{
	SpellEngine *se;

	se = g_new0 (SpellEngine, 1);
	se->config = new_pspell_config ();
	pspell_config_replace (se->config, "language-tag", language);
	pspell_config_replace (se->config, "encoding", "utf-8");
	se->changed = TRUE;

	return se;
}

static void
release_engines (GNOMESpellDictionary *dict)
{
	for (; dict->engines; ) {
		SpellEngine *se = dict->engines->data;

		if (se->manager)
			delete_pspell_manager (se->manager);
		if (se->config)
			delete_pspell_config (se->config);
		g_free (se);
		dict->engines = g_slist_remove (dict->engines, se);
	}

	dict->engines = NULL;
	dict->changed = TRUE;
}

#define KNOWN_LANGUAGES 22
static gchar *known_languages [KNOWN_LANGUAGES*2 + 1] = {
	"br", N_("Breton"),
	"ca", N_("Catalan"),
	"cs", N_("Czech"),
	"da", N_("Danish"),
	"de_de", N_("German (Germany)"),
	"de_ch", N_("German (Swiss)"),
	"en_us", N_("English (American)"),
	"en_gb", N_("English (British)"),
	"en_ca", N_("English (Canadian)"),
	"eo", N_("Esperanto"),
	"es", N_("Spanish"),
	"fo", N_("Faroese"),
	"fr_fr", N_("French (France)"),
	"fr_ch", N_("French (Swiss)"),
	"it", N_("Italian"),
	"nl", N_("Dutch"),
	"no", N_("Norwegian"),
	"pl", N_("Polish"),
	"pt_pt", N_("Portuguese (Portugal)"),
	"pt_br", N_("Portuguese (Brazilian)"),
	"ru", N_("Russian"),
	"sv", N_("Swedish"),
	NULL
};

static GSList *
get_languages_real (gint *ln)
{
	GSList *langs;
	PspellCanHaveError *err;
	PspellConfig  *config;
	PspellManager *manager;
	gint i;

	/* printf ("get_languages_real\n"); */

	langs = NULL;
	*ln = 0;
	for (i=0; known_languages [i]; i++) {
		config = new_pspell_config ();
		pspell_config_replace (config, "language-tag", known_languages [i]);
		i++;
		err = new_pspell_manager (config);
		if (pspell_error_number (err) == 0) {
			manager = to_pspell_manager (err);
			DICT_DEBUG (printf ("Language: %s\n", known_languages [i]));
			delete_pspell_manager (manager);
			langs = g_slist_prepend (langs, GINT_TO_POINTER (i - 1));
			(*ln) ++;
		}
	}

	return langs;
}

static GSList *
get_languages_load (gint *ln)
{
	GString *str;
	GSList *langs = NULL;
	gint i, lang_num;

	/* printf ("get_languages_load\n"); */

	str = g_string_new (NULL);
	*ln = gnome_config_get_int ("languages");
	for (i = 0; i < *ln; i++) {
		g_string_sprintf (str, "language%d", i);
		lang_num = gnome_config_get_int (str->str);
		langs = g_slist_prepend (langs, GINT_TO_POINTER (lang_num));
	}

	return langs;
}

static GSList *
get_languages (gint *ln)
{
	GSList *langs, *l;
	time_t mtime;
	struct stat buf;
	gint i, kl;

	gnome_config_push_prefix ("gnome-spell/Languages/");
	mtime = gnome_config_get_int ("mtime=0");
	kl = gnome_config_get_int ("known_languages=0");

	stat ("/opt/gnome/share" "/pspell", &buf);
	if (buf.st_mtime != mtime || kl != KNOWN_LANGUAGES) {
		GString *str;
		langs = get_languages_real (ln);

		str = g_string_new (NULL);
		gnome_config_set_int ("languages", *ln);
		for (l = langs, i = 0; i < *ln; i ++) {
			g_string_sprintf (str, "language%d", *ln - i - 1);
			gnome_config_set_int (str->str, GPOINTER_TO_INT (l->data));
			l = l->next;
		}
		gnome_config_set_int ("mtime", buf.st_mtime);
		gnome_config_set_int ("known_languages", KNOWN_LANGUAGES);
		g_string_free (str, TRUE);
		gnome_config_sync ();
	} else
		langs = get_languages_load (ln);

	gnome_config_pop_prefix ();

	return langs;
}

static GNOME_Spell_LanguageSeq *
impl_gnome_spell_dictionary_get_languages (PortableServer_Servant servant, CORBA_Environment *ev)
{
	GNOME_Spell_LanguageSeq *seq;
	GSList *l, *langs;
	gint i, ln, pos;

	langs = get_languages (&ln);

	seq = GNOME_Spell_LanguageSeq__alloc ();
	seq->_length = ln;

	if (seq->_length == 0)
		return seq;

	seq->_buffer = CORBA_sequence_GNOME_Spell_Language_allocbuf (seq->_length);

	for (i = ln - 1, l = langs; l; l = l->next, i--) {

		pos = GPOINTER_TO_INT (l->data);
		seq->_buffer [i].name = CORBA_string_dup (_(known_languages [pos + 1]));
		seq->_buffer [i].abrev = CORBA_string_dup (known_languages [pos]);
	}
	CORBA_sequence_set_release (seq, CORBA_TRUE);
	g_slist_free (langs);

	return seq;
}

static void
impl_gnome_spell_dictionary_set_language (PortableServer_Servant  servant,
					  const CORBA_char       *language,
					  CORBA_Environment      *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	const gchar *s, *begin, *end;
	gchar *one_language;
	gint len;

	g_assert (dict);
	g_assert (language);

	DICT_DEBUG (printf ("setLanguage: %s\n", language));

	release_engines (dict);
	for (s = language; *s; s = end) {
		begin = s;
		while (*begin && *begin == ' ')
			begin++;
		end = begin;
		len = 0;
		while (*end && *end != ' ') {
			end ++;
			len ++;
		}

		if (len) {
			one_language = g_strndup (begin, len);
			dict->engines = g_slist_prepend (dict->engines, new_engine (one_language));
			g_free (one_language);

			dict->changed = TRUE;
		}
	}
}

static void
engines_config_replace (GSList *engines, const gchar *key, const gchar *value)
{
	for (; engines; engines = engines->next) {
		SpellEngine *se = (SpellEngine *) engines->data;

		pspell_config_replace (se->config, key, value);
	}
}

static void
update_engine (SpellEngine *se, CORBA_Environment * ev)
{
	PspellCanHaveError *err;

	DICT_DEBUG (printf ("Dictionary: creating new pspell manager\n"));

	if (se->changed) {
		if (se->manager)
			delete_pspell_manager (se->manager);
		err = new_pspell_manager (se->config);
		if (pspell_error_number (err) != 0) {
			g_warning ("pspell error: %s\n", pspell_error_message (err));
			se->manager = NULL;
			raise_error (ev, pspell_error_message (err));
		} else {
			se->manager = to_pspell_manager (err);
			se->changed = FALSE;
		}
	}
}

static void
update_engines (GNOMESpellDictionary *dict, CORBA_Environment * ev)
{
	g_assert (IS_GNOME_SPELL_DICTIONARY (dict));

	if (dict->changed) {
		GSList *l;

		for (l = dict->engines; l; l = l->next) {
			update_engine ((SpellEngine *) l->data, ev);
		}

		dict->changed = FALSE;
	}
}

static CORBA_boolean
engine_check_word (SpellEngine *se, const gchar *word, CORBA_Environment *ev)
{
	CORBA_boolean result = CORBA_TRUE;
	gint pspell_result;

	g_return_val_if_fail (se->manager, CORBA_TRUE);

	pspell_result = pspell_manager_check (se->manager, word, strlen (word));
	if (pspell_result == 0)
		result = CORBA_FALSE;
	if (pspell_result == -1) {
		g_warning ("pspell error: %s\n", pspell_manager_error_message (se->manager));
		raise_error (ev, pspell_manager_error_message (se->manager));
	}

	return result;
}

static CORBA_boolean
impl_gnome_spell_dictionary_check_word (PortableServer_Servant servant, const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	CORBA_boolean result = CORBA_FALSE;
	GSList *l;
	gboolean valid_manager = FALSE;

	g_return_val_if_fail (word, result);

	if (!strcmp (word, "Ximian"))
		return CORBA_TRUE;

	update_engines (dict, ev);
	for (l = dict->engines; l; l = l->next) {
		if (((SpellEngine *) l->data)->manager) {
			valid_manager = TRUE;
			if (engine_check_word ((SpellEngine *) l->data, word, ev))
				result = CORBA_TRUE;
		}
	}

	if (!valid_manager) {
		DICT_DEBUG (printf ("Dictionary check_word: %s --> 1 (not valid manager)\n", word));
		return CORBA_TRUE;
	}
	DICT_DEBUG (printf ("Dictionary check_word: %s --> %d\n", word, result));
	return result;
}

static void
impl_gnome_spell_dictionary_add_word_to_session (PortableServer_Servant servant, const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	GSList *l;

	g_return_if_fail (word);

	update_engines (dict, ev);
	DICT_DEBUG (printf ("Dictionary add_word_to_session: %s\n", word));
	for (l = dict->engines; l; l = l->next) {
		if (((SpellEngine *) l->data)->manager)
			pspell_manager_add_to_session (((SpellEngine *) l->data)->manager, word, strlen (word));
	}
}

static void
impl_gnome_spell_dictionary_add_word_to_personal (PortableServer_Servant servant,
						  const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	GSList *l;

	g_return_if_fail (word);

	update_engines (dict, ev);
	DICT_DEBUG (printf ("Dictionary add_word_to_personal: %s\n", word));
	for (l = dict->engines; l; l = l->next) {
		SpellEngine *se = (SpellEngine *) l->data;

		if (se->manager) {
			pspell_manager_add_to_personal (se->manager, word, strlen (word));
			pspell_manager_save_all_word_lists (se->manager);
		}
	}
}

static void
impl_gnome_spell_dictionary_set_correction (PortableServer_Servant servant,
					    const CORBA_char *word, const CORBA_char *replacement, CORBA_Environment *ev)
{
	GNOMESpellDictionary *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	GSList *l;

	g_return_if_fail (word);
	g_return_if_fail (replacement);

	update_engines (dict, ev);
	DICT_DEBUG (printf ("Dictionary correction: %s <-- %s\n", word, replacement));
	for (l = dict->engines; l; l = l->next) {
		SpellEngine *se = (SpellEngine *) l->data;

		if (se->manager) {
			pspell_manager_store_replacement (se->manager, word, strlen (word),
							  replacement, strlen (replacement));
			pspell_manager_save_all_word_lists (se->manager);
		}
	}
}

static GNOME_Spell_StringSeq *
impl_gnome_spell_dictionary_get_suggestions (PortableServer_Servant servant,
					     const CORBA_char *word, CORBA_Environment *ev)
{
	GNOMESpellDictionary  *dict = GNOME_SPELL_DICTIONARY (bonobo_object_from_servant (servant));
	const PspellWordList  *suggestions;
	PspellStringEmulation *elements;
	GNOME_Spell_StringSeq *seq = NULL;
	GSList *l, *suggestion_list = NULL;
	gint i, len, pos;

	g_return_if_fail (word);

	DICT_DEBUG (printf ("Dictionary correction: %s\n", word));
	update_engines (dict, ev);

	len = 0;
	for (l = dict->engines; l; l = l->next) {
		SpellEngine *se = (SpellEngine *) l->data;

		if (se->manager) {
			suggestions  = pspell_manager_suggest (se->manager, word, strlen (word));
			suggestion_list = g_slist_prepend (suggestion_list, (gpointer) suggestions);
			len += pspell_word_list_size (suggestions);
		}
	}

	seq          = GNOME_Spell_StringSeq__alloc ();
	seq->_length = len;

	if (seq->_length == 0)
		return seq;

	seq->_buffer = CORBA_sequence_CORBA_string_allocbuf (seq->_length);

	pos = 0;
	for (l = suggestion_list; l; l = l->next) {
		gint list_len;

		suggestions = (const PspellWordList  *) l->data;
		elements = pspell_word_list_elements (suggestions);
		list_len = pspell_word_list_size (suggestions);
		for (i = 0; i < list_len; i ++, pos ++)
			seq->_buffer [pos] = CORBA_string_dup (pspell_string_emulation_next (elements));
		delete_pspell_string_emulation (elements);
	}
	CORBA_sequence_set_release (seq, CORBA_TRUE);
	g_slist_free (suggestion_list);

	return seq;
}

/*
 * If you want users to derive classes from your implementation
 * you need to support this method.
 */
POA_GNOME_Spell_Dictionary__epv *
gnome_spell_dictionary_get_epv (void)
{
	POA_GNOME_Spell_Dictionary__epv *epv;

	epv = g_new0 (POA_GNOME_Spell_Dictionary__epv, 1);

	epv->getLanguages         = impl_gnome_spell_dictionary_get_languages;
	epv->setLanguage          = impl_gnome_spell_dictionary_set_language;
	epv->checkWord            = impl_gnome_spell_dictionary_check_word;
	epv->addWordToSession     = impl_gnome_spell_dictionary_add_word_to_session;
	epv->addWordToPersonal    = impl_gnome_spell_dictionary_add_word_to_personal;
	epv->getSuggestions       = impl_gnome_spell_dictionary_get_suggestions;
	epv->setCorrection        = impl_gnome_spell_dictionary_set_correction;

	return epv;
}

static void
init_dictionary_corba_class (void)
{
	dictionary_vepv.Bonobo_Unknown_epv         = bonobo_object_get_epv ();
	dictionary_vepv.GNOME_Spell_Dictionary_epv = gnome_spell_dictionary_get_epv ();
}

static void
dictionary_class_init (GNOMESpellDictionaryClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	dictionary_parent_class  = gtk_type_class (bonobo_object_get_type ());
	object_class->destroy    = dictionary_object_destroy;

	init_dictionary_corba_class ();
}

GtkType
gnome_spell_dictionary_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"Dictionary",
			sizeof (GNOMESpellDictionary),
			sizeof (GNOMESpellDictionaryClass),
			(GtkClassInitFunc) dictionary_class_init,
			(GtkObjectInitFunc) dictionary_object_init,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (bonobo_object_get_type (), &info);
	}

	return type;
}

BonoboObject *
dictionary_construct (BonoboObject *dictionary, GNOME_Spell_Dictionary corba_dictionary)
{
	CORBA_Environment ev;
	static int i;
	
	g_return_val_if_fail (dictionary != NULL, NULL);
	g_return_val_if_fail (IS_GNOME_SPELL_DICTIONARY (dictionary), NULL);
	g_return_val_if_fail (corba_dictionary != CORBA_OBJECT_NIL, NULL);

	if (!bonobo_object_construct (dictionary, (CORBA_Object) corba_dictionary))
		return NULL;

	CORBA_exception_init (&ev);
	CORBA_exception_free (&ev);

	return dictionary;
}

/*
 * This routine creates the ORBit CORBA server and initializes the
 * CORBA side of things
 */
static GNOME_Spell_Dictionary
create_dictionary (BonoboObject *dictionary)
{
	POA_GNOME_Spell_Dictionary *servant;
	CORBA_Environment ev;

	servant       = (POA_GNOME_Spell_Dictionary *) g_new0 (BonoboObjectServant, 1);
	servant->vepv = &dictionary_vepv;

	CORBA_exception_init (&ev);
	POA_GNOME_Spell_Dictionary__init ((PortableServer_Servant) servant, &ev);
	ORBIT_OBJECT_KEY (servant->_private)->object = NULL;

	if (ev._major != CORBA_NO_EXCEPTION){
		g_free (servant);
		CORBA_exception_free (&ev);
		return CORBA_OBJECT_NIL;
	}

	CORBA_exception_free (&ev);

	return (GNOME_Spell_Dictionary) bonobo_object_activate_servant (dictionary, servant);
}

BonoboObject *
gnome_spell_dictionary_new (void)
{
	BonoboObject *dictionary;
	GNOME_Spell_Dictionary corba_dictionary;

              dictionary = gtk_type_new (gnome_spell_dictionary_get_type ());
	corba_dictionary = create_dictionary (dictionary);

	if (corba_dictionary == CORBA_OBJECT_NIL) {
		bonobo_object_unref (BONOBO_OBJECT (dictionary));
		return NULL;
	}
	
	return dictionary_construct (dictionary, corba_dictionary);
}
