/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* This file is part of the GNOME Spell

   Copyright (C) 2002 Ximian, Inc.
   Authors:           Radek Doulik (rodo@ximian.com)

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHcANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <string.h>
#include <gnome.h>
#include <capplet-widget.h>
#include <gconf/gconf-client.h>
#include <liboaf/liboaf.h>
#include <bonobo.h>
#include <glade/glade.h>

#include "../gnome-spell/Spell.h"

#define GNOME_SPELL_GCONF_DIR "/GNOME/Spell"

static CORBA_sequence_GNOME_Spell_Language *seq;

static GtkWidget *capplet;
static GtkWidget *language_clist;
static GtkWidget *color_picker;

static gboolean active = FALSE;

static GError      *error  = NULL;
static GConfClient *client = NULL;

static gchar *language_str = NULL, *language_str_orig = NULL;
static GdkColor spell_error_color, spell_error_color_orig;

static void
select_lang (const gchar *abrev)
{
	gint i;

	for (i = 0; i < seq->_length; i ++) {
		if (!strcasecmp (abrev, seq->_buffer [i].abrev)) {
			gtk_clist_select_row (GTK_CLIST (language_clist), i, 0);
		}
	}
}

static void
set_ui_language ()
{
	gchar *l, *last, *lang;

	gtk_clist_freeze (GTK_CLIST (language_clist));
	gtk_clist_unselect_all (GTK_CLIST (language_clist));
	last = language_str;
	while ((l = strchr (last, ' '))) {
		if (l != last) {
			lang = g_strndup (last, l - last);
			select_lang (lang);
			g_free (lang);
		}

		last = l + 1;
	}
	if (last)
		select_lang (last);
	gtk_clist_thaw (GTK_CLIST (language_clist));
}

static void
set_ui ()
{
	active = FALSE;

	set_ui_language ();
	gnome_color_picker_set_i16 (GNOME_COLOR_PICKER (color_picker),
				    spell_error_color.red, spell_error_color.green, spell_error_color.blue, 0xffff);

	active = TRUE;
}

static gchar *
get_language_str ()
{
	GList *selection = GTK_CLIST (language_clist)->selection;
	GString *str = g_string_new (NULL);
	gchar *rv;

	for (; selection; selection = selection->next) {
		g_string_append (str, gtk_clist_get_row_data (GTK_CLIST (language_clist),
							      GPOINTER_TO_INT (selection->data)));
		if (selection->next)
			g_string_append_c (str, ' ');
	}

	rv = str->str;
	g_string_free (str, FALSE);

	return rv;
}

static void
get_ui ()
{
	gnome_color_picker_get_i16 (GNOME_COLOR_PICKER (color_picker),
				    &spell_error_color.red, &spell_error_color.green, &spell_error_color.blue, NULL);
	g_free (language_str);
	language_str = get_language_str ();
}

#define GET(t,x,prop,f,c) \
        val = gconf_client_get_without_default (client, GNOME_SPELL_GCONF_DIR x, NULL); \
        if (val) { f; prop = c (gconf_value_get_ ## t (val)); \
        gconf_value_free (val); }

static void
save_orig ()
{
	g_free (language_str_orig);
	language_str_orig = g_strdup (language_str);
	spell_error_color_orig = spell_error_color;
}

static void
load_orig ()
{
	g_free (language_str);
	language_str = g_strdup (language_str_orig);
	spell_error_color = spell_error_color_orig;
}

static void
load_values ()
{
	GConfValue *val;

	g_free (language_str);
	language_str = g_strdup (_("en"));
	spell_error_color.red   = 0xffff;
	spell_error_color.green = 0;
	spell_error_color.blue  = 0;

	GET (int, "/spell_error_color_red",   spell_error_color.red,,);
	GET (int, "/spell_error_color_green", spell_error_color.green,,);
	GET (int, "/spell_error_color_blue",  spell_error_color.blue,,);
	GET (string, "/language", language_str, g_free (language_str), g_strdup);

	save_orig ();
}

#define SET(t,x,prop) \
        gconf_client_set_ ## t (client, GNOME_SPELL_GCONF_DIR x, prop, NULL);

static void
save_values (gboolean force)
{
	GConfValue *val;

	if (force || !gdk_color_equal (&spell_error_color, &spell_error_color_orig)) {
		SET (int, "/spell_error_color_red",   spell_error_color.red);
		SET (int, "/spell_error_color_green", spell_error_color.green);
		SET (int, "/spell_error_color_blue",  spell_error_color.blue);
	}
	if (force || strcmp (language_str, language_str_orig)) {
		SET (string, "/language", language_str);
	}

	gconf_client_suggest_sync (client, NULL);
}

static void
apply (void)
{
	get_ui ();
	save_values (FALSE);
}

static void
revert (void)
{
	load_orig ();
	set_ui ();
	save_values (TRUE);
}

static void
changed (GtkWidget *widget, gpointer null)
{
	if (active)
		capplet_widget_state_changed (CAPPLET_WIDGET (capplet), TRUE);
}

static void
setup ()
{
	GtkWidget *hbox, *ebox;
	GladeXML *xml;
	gint i;

	glade_gnome_init ();
	xml = glade_xml_new (GLADE_DATADIR "/gnome-spell-capplet.glade", "prefs_widget");

	if (!xml)
		g_error (_("Could not load glade file."));

        capplet = capplet_widget_new();
	hbox    = glade_xml_get_widget (xml, "prefs_widget");

	language_clist = glade_xml_get_widget (xml, "clist_language");
	gtk_clist_freeze (GTK_CLIST (language_clist));
	for (i = 0; i < seq->_length; i ++) {
		gchar *texts [1];

		texts [0] = _(seq->_buffer [i].name);
		gtk_clist_append (GTK_CLIST (language_clist), texts);
		gtk_clist_set_row_data (GTK_CLIST (language_clist), i, seq->_buffer [i].abrev);
	}
	gtk_clist_thaw (GTK_CLIST (language_clist));

	color_picker = glade_xml_get_widget (xml, "color_picker");

        gtk_container_add (GTK_CONTAINER (capplet), hbox);
        gtk_widget_show_all (capplet);

	load_values ();
	set_ui ();

	glade_xml_signal_connect (xml, "changed", GTK_SIGNAL_FUNC (changed));
}

static void
init_bonobo (int argc, char *argv [])
{
	CORBA_ORB orb;

        gnome_init_with_popt_table (
		"gnome-spell-capplet", VERSION,
		argc, argv,
		oaf_popt_options, 0, NULL); 

	orb = oaf_init (argc, argv);

	if (!bonobo_init (orb, CORBA_OBJECT_NIL, CORBA_OBJECT_NIL))
		g_error (_("I could not initialize Bonobo"));

	bonobo_activate ();
}

int
main (int argc, char **argv)
{
	BonoboObjectClient *dictionary_client;
	GNOME_Spell_Dictionary dict;
	CORBA_Environment ev;
	gchar *dictionary_id;

        bindtextdomain ("gnome-spell-" VERSION, GNOMELOCALEDIR);
        textdomain ("gnome-spell-" VERSION);

	init_bonobo (argc, argv);
	dictionary_id = "OAFIID:GNOME_Spell_Dictionary:" API_VERSION;
	dictionary_client = bonobo_object_activate (dictionary_id, 0);

	if (!dictionary_client)
		g_error ("Cannot activate %s", dictionary_id);
	dict = bonobo_object_corba_objref (BONOBO_OBJECT (dictionary_client));

	CORBA_exception_init (&ev);
	seq = GNOME_Spell_Dictionary_getLanguages (dict, &ev);
	CORBA_exception_free (&ev);

        if (gnome_capplet_init ("gnome-spell-properties", VERSION, argc, argv, NULL, 0, NULL) < 0)
		return 1;

	if (!gconf_init(argc, argv, &error)) {
		g_assert(error != NULL);
		g_warning("GConf init failed:\n  %s", error->message);
		return 1;
	}

	client = gconf_client_get_default ();
	gconf_client_add_dir(client, GNOME_SPELL_GCONF_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);

        setup ();

	/* connect signals */
        gtk_signal_connect (GTK_OBJECT (capplet), "try",
                            GTK_SIGNAL_FUNC (apply), NULL);
        gtk_signal_connect (GTK_OBJECT (capplet), "revert",
                            GTK_SIGNAL_FUNC (revert), NULL);
        gtk_signal_connect (GTK_OBJECT (capplet), "ok",
                            GTK_SIGNAL_FUNC (apply), NULL);
        gtk_signal_connect (GTK_OBJECT (capplet), "cancel",
                            GTK_SIGNAL_FUNC (revert), NULL);

        capplet_gtk_main ();

        return 0;
}
