/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gnome-scan
 * Copyright (C) Étienne Bersac 2007 <bersace03@laposte.net>
 * 
 * gnome-scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

#include <glib/gi18n.h>
#include "gsfile-pspec.h"
#include "gsfile-filenames-widget.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GSFILE_TYPE_FILENAMES_WIDGET, GSFileFilenamesWidgetPrivate))
#define	PREVIEW_SIZE	96

typedef struct _GSFileFilenamesWidgetPrivate GSFileFilenamesWidgetPrivate;

struct _GSFileFilenamesWidgetPrivate
{
	GtkWidget			*filechooser;
	GtkListStore		*liststore;
	GtkTreeSelection	*selection;
};

enum
{
	FILENAMES_PREVIEW,
	FILENAMES_BASENAME,
	FILENAMES_PATHNAME,
	FILENAMES_N_COLUMNS
};

static void	gsffw_filenames_add (GtkButton *button,
								   GSFileFilenamesWidget *widget);
static void	gsffw_filenames_populate (GSFileFilenamesWidget *widget);
static gboolean gsffw_filenames_preview_foreach_func (GtkTreeModel *model,
													 GtkTreePath *path,
													 GtkTreeIter *iter,
													 GSFileFilenamesWidget *widget);
static void	gsffw_filenames_remove (GtkButton *button,
									  GSFileFilenamesWidget *widget);
static void	gsffw_filenames_clear (GtkButton *button,
									 GSFileFilenamesWidget *widget);
static gboolean	gsffw_filenames_foreach_func (GtkTreeModel *model,
												GtkTreePath *path,
												GtkTreeIter *iter,
												GSList **widget);
static void	gsffw_filenames_changed (GSFileFilenamesWidget *widget);

GS_DEFINE_PARAM_WIDGET (GSFileFilenamesWidget, gsfile_filenames_widget)

static void
gsfile_filenames_widget_init (GSFileFilenamesWidget *object)
{
	/* TODO: Add initialization code here */
}

static void
gsfile_filenames_widget_finalize (GObject *object)
{
	/* TODO: Add deinitalization code here */

	G_OBJECT_CLASS (gsfile_filenames_widget_parent_class)->finalize (object);
}



static void
gsfile_filenames_widget_build (GnomeScanParamWidget *gspw)
{
	GSFileFilenamesWidget *gsffw = GSFILE_FILENAMES_WIDGET (gspw);
	GSFileFilenamesWidgetPrivate *priv = GET_PRIVATE (gsffw);
	GtkWidget *scrolled, *tree_view, *box, *button;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	
	gtk_box_set_spacing (GTK_BOX (gspw), 6);
	gspw->shows_label = TRUE; /* yes, that's a lie :o) */
	gspw->expands = TRUE;
	
	priv->liststore = gtk_list_store_new (FILENAMES_N_COLUMNS,
										  GDK_TYPE_PIXBUF,
										  G_TYPE_STRING,
										  G_TYPE_STRING);
	
	tree_view = gtk_tree_view_new_with_model (GTK_TREE_MODEL (priv->liststore));
	gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tree_view), TRUE);
	gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tree_view), TRUE);
	priv->selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
	
	/* preview */
	renderer = gtk_cell_renderer_pixbuf_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Preview"), renderer,
													   "pixbuf", FILENAMES_PREVIEW,
													   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);
	
	/* basename */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Filename"), renderer,
													   "markup", FILENAMES_BASENAME,
													   NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);
	
	/* scrolled window */
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
									GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scrolled),
										   tree_view);
	gtk_box_pack_start (GTK_BOX (gspw), scrolled, TRUE, TRUE, 0);
	
	/* buttons */
	box = gtk_vbutton_box_new ();
	gtk_box_set_spacing (GTK_BOX (box), 4);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (box), GTK_BUTTONBOX_START);
	gtk_box_pack_start (GTK_BOX (gspw), box, FALSE, FALSE, 0);
	
	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	g_signal_connect (button, "clicked",
					  (GCallback) gsffw_filenames_add, gsffw);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);
	
	button = gtk_button_new_from_stock (GTK_STOCK_REMOVE);
	g_signal_connect (button, "clicked",
					  (GCallback) gsffw_filenames_remove, gsffw);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);
	
	button = gtk_button_new_from_stock (GTK_STOCK_CLEAR);
	g_signal_connect (button, "clicked",
					  (GCallback) gsffw_filenames_clear, gsffw);
	gtk_box_pack_start (GTK_BOX (box), button, FALSE, FALSE, 0);
}

static void
gsfile_filenames_widget_set (GnomeScanParamWidget *gspw, GValue *value)
{
}
/* INTERNALS */

static void
gsffw_filenames_add (GtkButton *button, GSFileFilenamesWidget *gsffw)
{
	GnomeScanParamWidget *gspw = GNOME_SCAN_PARAM_WIDGET (gsffw);
	GSFileFilenamesWidgetPrivate *priv = GET_PRIVATE (gsffw);
	GtkFileFilter *filter, *filter1;
	GSList *node;
	gchar **mimes, **exts;
	gchar *pattern;
	gint i;
	GError *error = NULL;
	GnomeScanFormat *format;
	
	
	/* filechooser */
	if (!priv->filechooser) {
		priv->filechooser =
			gtk_file_chooser_dialog_new (_("Select files"),
										 GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (gsffw))),
										 GTK_FILE_CHOOSER_ACTION_OPEN,
										 GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
										 GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
										 NULL);
		gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (priv->filechooser),
											 g_get_home_dir ());
		gtk_file_chooser_set_select_multiple (GTK_FILE_CHOOSER (priv->filechooser),
											  TRUE);
		
		/* add a global filter and a per format filter */
		filter = gtk_file_filter_new ();
		gtk_file_filter_set_name (filter, _("Supported formats"));
		gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (priv->filechooser), filter);
		node = gsfile_param_spec_filenames_get_formats (gspw->pspec);
		for (; node ; node = node->next) {
			format = node->data;
			filter1 = gtk_file_filter_new ();
			
			mimes = format->mime_types;
			for (i = 0; mimes[i]; i++) {
				gtk_file_filter_add_mime_type (filter, mimes[i]);
				gtk_file_filter_add_mime_type (filter1, mimes[i]);
			}
			
			exts = format->extensions;
			for (i = 0; exts[i]; i++) {
				pattern = g_strdup_printf ("*.%s", exts[i]);
				gtk_file_filter_add_pattern (filter, pattern);
				gtk_file_filter_add_pattern (filter1, pattern);
			}
			
			gtk_file_filter_set_name (filter1,
									  g_strdup_printf ("%s (*.%s)",
													   dgettext(format->domain, format->description),
													   g_strjoinv(", *.", format->extensions)));
			gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (priv->filechooser),
										 filter1);
		}
	}
	
	/* run filechooser */
	if (gtk_dialog_run (GTK_DIALOG (priv->filechooser)) == GTK_RESPONSE_ACCEPT) {
		gtk_widget_hide (priv->filechooser);
		/* delegate populate creation in a thread */
		g_thread_create ((GThreadFunc) gsffw_filenames_populate,
						 gsffw, FALSE, &error);
	}
	gtk_widget_hide (priv->filechooser);
}

static void
gsffw_filenames_populate (GSFileFilenamesWidget *gsffw)
{
	GSFileFilenamesWidgetPrivate *priv = GET_PRIVATE (gsffw);
	GSList *node;
	GtkTreeIter iter;
	
	/* first populate list store */
	node = gtk_file_chooser_get_filenames (GTK_FILE_CHOOSER (priv->filechooser));
	for (; node ; node = node->next) {
		gtk_list_store_insert_with_values (priv->liststore, &iter, G_MAXINT,
										   FILENAMES_PREVIEW, NULL,
										   FILENAMES_BASENAME, g_path_get_basename (node->data),
										   FILENAMES_PATHNAME, node->data,
										   -1);
	}
	
	/* trigger preview update (this may take some time considering list length and image size) */
	gtk_tree_model_foreach (GTK_TREE_MODEL (GET_PRIVATE (gsffw)->liststore),
							(GtkTreeModelForeachFunc) gsffw_filenames_preview_foreach_func,
							gsffw);
	gsffw_filenames_changed (gsffw);
}

/* create a preview if not present */
gboolean
gsffw_filenames_preview_foreach_func (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, GSFileFilenamesWidget *gsffw)
{
	GSFileFilenamesWidgetPrivate *priv = GET_PRIVATE (gsffw);
	GdkPixbuf *preview;
	GError *error = NULL;
	gchar *pathname;
	
	gtk_tree_model_get (model, iter,
						FILENAMES_PREVIEW, &preview,
						FILENAMES_PATHNAME, &pathname,
						-1);
	if (!preview) {
		preview = gdk_pixbuf_new_from_file_at_scale (pathname,
													 PREVIEW_SIZE, PREVIEW_SIZE,
													 TRUE, &error);
		gtk_list_store_set (priv->liststore, iter,
							FILENAMES_PREVIEW, preview,
							-1);
	}
	return FALSE;
}

static void
gsffw_filenames_remove (GtkButton *button, GSFileFilenamesWidget *gsffw)
{
	GSFileFilenamesWidgetPrivate *priv = GET_PRIVATE (gsffw);
	GtkTreeModel *model = GTK_TREE_MODEL (priv->liststore);
	GList *node, *first = gtk_tree_selection_get_selected_rows (priv->selection,
																&model);
	GtkTreeIter iter;

	for (node = first; node ; node = node->next) {
		gtk_tree_model_get_iter (model, &iter, node->data);
		gtk_list_store_remove (priv->liststore, &iter);
		gtk_tree_path_free (node->data);
	}
	g_list_free (first);
	gsffw_filenames_changed (gsffw);
}

static void
gsffw_filenames_clear (GtkButton *button, GSFileFilenamesWidget *gsffw)
{
	gtk_list_store_clear (GET_PRIVATE (gsffw)->liststore);
	gsffw_filenames_changed (gsffw);
}

static void
gsffw_filenames_changed (GSFileFilenamesWidget *gsffw)
{
	GnomeScanParamWidget *gspw = GNOME_SCAN_PARAM_WIDGET (gsffw);
	GSFileFilenamesWidgetPrivate *priv = GET_PRIVATE (gsffw);
	GSList *list = NULL;
	
	gtk_tree_model_foreach (GTK_TREE_MODEL (priv->liststore),
							(GtkTreeModelForeachFunc) gsffw_filenames_foreach_func,
							&list);
	g_value_set_pointer (gspw->value, list);

	gnome_scan_param_widget_changed (gspw);
}

static gboolean
gsffw_filenames_foreach_func (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, GSList **list)
{
	gchar*pathname;
	gtk_tree_model_get (model, iter,
						FILENAMES_PATHNAME, &pathname,
						-1);
	*list = g_slist_append (*list, g_strdup (pathname));
	
	return FALSE;
}


