/* Gnome Scan - Scan as easy as you print
 * Copyright © 2007  Étienne Bersac <bersace03@laposte.net>
 *
 * Gnome Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

/**
 * SECTION: gnome-scan-settings
 * @short_description: Load settings accross plugins
 *
 * When user switch from a scanner to another, he expects the settings
 * to keep almost the same. e.g. resolution should not be resetted on
 * device selection. The settings are remember in a #GnomeScanSettings
 * which maintains a table of #GValue using @GParamSpec:name as
 * key. This allow to apply the same #GnomeScanSettings on various
 * #GnomeScanPlugin through gnome_scan_plugin_configure().
 *
 * In the future, automatic saving to GConf or .ini would be such
 * feature easily added with this design.
 **/
#include "gnome-scan-settings.h"

#define	GET_PRIVATE(o)	(G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_TYPE_SCAN_SETTINGS, GnomeScanSettingsPrivate))

typedef struct _GnomeScanSettingsPrivate GnomeScanSettingsPrivate;

struct _GnomeScanSettingsPrivate
{
  /* using GParamSpec name as key, and GValue as value */
  GHashTable	*values;
};

enum
  {
    CHANGED,
    N_SIGNALS
  };

static GObjectClass* parent_class = NULL;
static guint signals[N_SIGNALS] = {0};

G_DEFINE_TYPE (GnomeScanSettings, gnome_scan_settings, G_TYPE_OBJECT);

static void
gnome_scan_settings_init (GnomeScanSettings *object)
{
  GnomeScanSettingsPrivate *priv = GET_PRIVATE (object);
	
  priv->values = g_hash_table_new (g_str_hash, g_str_equal);
}

static void
gnome_scan_settings_finalize (GObject *object)
{
  /* TODO: Add deinitalization code here */

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
gnome_scan_settings_class_init (GnomeScanSettingsClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));

  g_type_class_add_private (klass, sizeof (GnomeScanSettingsPrivate));
  object_class->finalize = gnome_scan_settings_finalize;
	
  /**
   * GnomeScanParamWidget::changed:
   * @widget: The emitting widget
   * @value: The new value
   *
   * This signal is emitted when the value is changed by the user.
   **/
  signals[CHANGED] =
    g_signal_new ("changed",
		  G_OBJECT_CLASS_TYPE (klass),
		  G_SIGNAL_RUN_LAST,
		  G_STRUCT_OFFSET (GnomeScanSettingsClass, changed),
		  NULL, NULL,
		  g_cclosure_marshal_VOID__STRING,
		  G_TYPE_NONE, 1, G_TYPE_STRING);
}




/**
 * gnome_scan_settings_new:
 * 
 * Create a new empty #GnomeScanSettings.
 * 
 * Returns: a new #GnomeScanSettings
 **/
GnomeScanSettings*
gnome_scan_settings_new (void)
{
  GObject *object = g_object_new (GNOME_TYPE_SCAN_SETTINGS, NULL);
  return GNOME_SCAN_SETTINGS (object);
}

void
gnome_scan_settings_init_value (GnomeScanSettings *settings,
				GParamSpec *pspec)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings));
  GValue *value = g_new0 (GValue, 1);
  g_value_init (value, G_PARAM_SPEC_VALUE_TYPE (pspec));
  g_param_value_set_default (pspec, value);
  gnome_scan_settings_set (settings,
			   g_param_spec_get_name (pspec),
			   value);
  g_value_reset (value);
  g_free (value);
}

/**
 * gnome_scan_settings_set:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: #GParamSpec user value
 * 
 * Store @value for @name key.
 * See: gnome_scan_settings_get()
 **/
void
gnome_scan_settings_set (GnomeScanSettings *settings,
                         const gchar *name,
                         const GValue *value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings) && G_IS_VALUE (value));
  g_hash_table_insert (GET_PRIVATE (settings)->values, g_strdup (name),
                       g_boxed_copy (G_TYPE_VALUE, value));
  g_signal_emit (settings, signals[CHANGED], 0, name);
}

/**
 * gnome_scan_settings_set_transform:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: #GParamSpec user value
 * 
 * Store @value for @name key, keeping existing value type by
 * transforming @value. @value type is used if the @name setting has
 * never been set.
 *
 * See: gnome_scan_settings_get_transformed()
 **/
void
gnome_scan_settings_set_transform (GnomeScanSettings *settings,
				   const gchar *name,
				   const GValue *value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings) && G_IS_VALUE (value));
  GValue *v = gnome_scan_settings_get (settings, name);
  if (v) {
    g_value_transform (value, v);
  }
  else {
    v = g_boxed_copy (G_TYPE_VALUE, value);
  }

  g_hash_table_insert (GET_PRIVATE (settings)->values, g_strdup (name),
		       v);
  g_signal_emit (settings, signals[CHANGED], 0, name);
}

/**
 * gnome_scan_settings_set_double:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: gdouble value
 * 
 *
 * See: gnome_scan_settings_set_transform()
 **/
void
gnome_scan_settings_set_double (GnomeScanSettings *settings,
                                const gchar *name,
                                gdouble value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings));
  GValue *v = g_new0 (GValue, 1);
  g_value_init (v, G_TYPE_DOUBLE);
  g_value_set_double (v, value);
  gnome_scan_settings_set_transform (settings, name, v);
}

/**
 * gnome_scan_settings_set_int:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: gint value
 * 
 *
 * See: gnome_scan_settings_set_transform()
 **/
void
gnome_scan_settings_set_int (GnomeScanSettings *settings,
                             const gchar *name,
                             gint value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings));
  GValue *v = g_new0 (GValue, 1);
  g_value_init (v, G_TYPE_INT);
  g_value_set_int (v, value);
  gnome_scan_settings_set_transform (settings, name, v);
}

/**
 * gnome_scan_settings_set_enum:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: genum value
 * 
 *
 * See: gnome_scan_settings_set_transform()
 **/
void
gnome_scan_settings_set_enum (GnomeScanSettings *settings,
                              const gchar *name,
                              GType type,
                              gint value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings));
  GValue *v = g_new0 (GValue, 1);
  g_value_init (v, type);
  g_value_set_enum (v, value);
  gnome_scan_settings_set_transform (settings, name, v);
}

/**
 * gnome_scan_settings_set_boxed:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: a #GBoxed
 * 
 *
 * See: gnome_scan_settings_set_transform()
 **/
void
gnome_scan_settings_set_boxed (GnomeScanSettings *settings,
                               const gchar *name,
                               GType type,
                               gpointer value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings));
  GValue *v = g_new0 (GValue, 1);
  g_value_init (v, type);
  g_value_set_boxed (v, g_boxed_copy (type, value));
  gnome_scan_settings_set_transform (settings, name, v);
}

/**
 * gnome_scan_settings_set_object:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: a #GObject
 * 
 *
 * See: gnome_scan_settings_set_transform()
 **/
void
gnome_scan_settings_set_object (GnomeScanSettings *settings,
                                const gchar *name,
                                GObject* value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings));
  GValue *v = g_new0 (GValue, 1);
  g_value_init (v, G_TYPE_OBJECT);
  g_value_set_object (v, value);
  gnome_scan_settings_set_transform (settings, name, v);
}

/**
 * gnome_scan_settings_set_pointer:
 * @settings: a #GnomeScanSettings
 * @name: #GParamSpec name
 * @value: a gpointer
 * 
 *
 * See: gnome_scan_settings_set_transform()
 **/
void
gnome_scan_settings_set_pointer (GnomeScanSettings *settings,
                                 const gchar *name,
                                 gpointer value)
{
  g_return_if_fail (GNOME_IS_SCAN_SETTINGS (settings));
  GValue *v = g_new0 (GValue, 1);
  g_value_init (v, G_TYPE_POINTER);
  g_value_set_pointer (v, value);
  gnome_scan_settings_set_transform (settings, name, v);
}

/**
 * gnome_scan_settings_get:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the value corresponding to @key or NULL.
 * See: gnome_scan_settings_set()
 **/
GValue*
gnome_scan_settings_get (GnomeScanSettings *settings,
                         const gchar *name)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), NULL);
  GValue *v = g_hash_table_lookup (GET_PRIVATE (settings)->values, name);
  if (v)
    v = g_boxed_copy (G_TYPE_VALUE, v);
	
  return v;
}

/**
 * gnome_scan_settings_get:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * @type:	Transformation destination type.
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the value corresponding to @key or NULL.
 * See: gnome_scan_settings_set()
 **/
GValue*
gnome_scan_settings_get_transformed	(GnomeScanSettings *settings,
                                        const gchar *name,
                                        GType type)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), NULL);
  GValue *v = gnome_scan_settings_get (settings, name);
  GValue *t = g_new0 (GValue, 1);
  g_value_init (t, type);
  if (v) {
    g_value_transform (v, t);
    g_value_unset (v);
    g_free (v);
  }
  else {
    g_warning ("Option %s not set", name);
  }
  return t;
}


/**
 * gnome_scan_settings_get_string:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the string corresponding to @key.
 * See: gnome_scan_settings_get()
 **/
gchar*
gnome_scan_settings_get_string (GnomeScanSettings *settings,
                                const gchar *name)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), NULL);
  GValue *v = gnome_scan_settings_get_transformed (settings,
						   name,
						   G_TYPE_STRING);
  gchar* val = g_value_dup_string (v);
    g_value_unset(v);
  g_free (v);
  return val;
}

/**
 * gnome_scan_settings_get_double:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the string corresponding to @key or NULL.
 * See: gnome_scan_settings_get()
 **/
gdouble
gnome_scan_settings_get_double (GnomeScanSettings *settings,
                                const gchar *name)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), 0.);
  GValue *v = gnome_scan_settings_get_transformed (settings,
						   name,
						   G_TYPE_DOUBLE);
  gdouble val = g_value_get_double (v);
  g_free (v);
  return val;
}

/**
 * gnome_scan_settings_get_int:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the string corresponding to @key or NULL.
 * See: gnome_scan_settings_get()
 **/
gint
gnome_scan_settings_get_int (GnomeScanSettings *settings,
                             const gchar *name)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), 0);
  GValue *v = gnome_scan_settings_get_transformed (settings,
						   name,
						   G_TYPE_INT);
  gint val = g_value_get_int (v);
  g_value_unset (v);
  g_free (v);
  return val;
}

/**
 * gnome_scan_settings_get_enum:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * @type: #GEnum subtype
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the enum value corresponding to @key or 0.
 * See: gnome_scan_settings_get()
 **/
gint
gnome_scan_settings_get_enum (GnomeScanSettings *settings,
			      const gchar *name,
			      GType type)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), 0);
  GValue *v = gnome_scan_settings_get_transformed (settings,
						   name,
						   type);
  gint val = g_value_get_enum (v);
  g_value_unset (v);
  g_free (v);
  return val;
}

/**
 * gnome_scan_settings_get_boxed:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the boxed-type value corresponding to @key or NULL.
 * See: gnome_scan_settings_get()
 **/
gpointer
gnome_scan_settings_get_boxed (GnomeScanSettings *settings,
			       const gchar *name,
			       GType type)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), NULL);
  GValue *v = gnome_scan_settings_get_transformed (settings,
						   name,
						   type);
  gpointer val = g_value_dup_boxed (v);
  g_value_unset (v);
  g_free (v);
  return val;
}

/**
 * gnome_scan_settings_get_object:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the object corresponding to @key or NULL.
 * See: gnome_scan_settings_get()
 **/
GObject*
gnome_scan_settings_get_object (GnomeScanSettings *settings,
				const gchar *name)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), NULL);
  GValue *v = gnome_scan_settings_get_transformed (settings,
						   name,
						   G_TYPE_OBJECT);
  GObject* val = g_value_dup_object (v);
  g_value_unset (v);
  g_free (v);
  return val;
}

/**
 * gnome_scan_settings_get_pointer:
 * @settings: a #GnomeScanSettings
 * @name: a key
 * 
 * Look for a value stored in the table for @name key.
 * 
 * Returns: the pointer corresponding to @key or NULL.
 * See: gnome_scan_settings_get()
 **/
gpointer
gnome_scan_settings_get_pointer (GnomeScanSettings *settings,
				 const gchar *name)
{
  g_return_val_if_fail (GNOME_IS_SCAN_SETTINGS (settings), NULL);
  GValue *v = gnome_scan_settings_get_transformed (settings,
						   name,
						   G_TYPE_POINTER);
  gpointer val = g_value_get_pointer (v);
  g_value_unset (v);
  g_free (v);
  return val;
}

/**
 * gnome_scan_settings_get_all:
 * @settings: a #GnomeScanSettings
 *
 * Retrun all stored values in a #GHashTable. Use g_hash_table_ref() if 
 * you want to keep the #GHashTable in memory after @settings destruction.
 *
 * Returns: a GHashTable containings #GValues
 */
GHashTable*
gnome_scan_settings_get_all (GnomeScanSettings *settings)
{
  return GET_PRIVATE (settings)->values;
}
