/*
 *
 * gnome-scancontext.h
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/**
 * SECTION:gnomescancontext
 * @short_description: The marshaller object that holds all other objects
 *
 * The #GnomeScanContext is the main object that applications will
 * use. #GnomeScanContext holds the list of scanners, scan options such
 * as resolution, area, rotation, etc. Only one #GnomeScanContext
 * should be used in one application. The object must be shared
 * between all object and widget of the application. Also, if the
 * application do not show widgets for handling such or such option,
 * the application must set those options to the desired value itself.
 * 
 */

#ifndef GNOME_SCAN_CONTEXT_H
#define GNOME_SCAN_CONTEXT_H

#include <glib.h>
#include <glib-object.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "gnomescanbackend.h"
#include "gnomescanner.h"
#include "gnomescanneroption.h"

#define	GNOME_TYPE_SCAN_CONTEXT			(gnome_scan_context_get_type ())
#define GNOME_SCAN_CONTEXT(obj)			(G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_TYPE_SCAN_CONTEXT, GnomeScanContext))
#define GNOME_SCAN_CONTEXT_CLASS(klass)		(G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_TYPE_SCAN_CONTEXT, GnomeScanContextClass))
#define GNOME_IS_SCAN_CONTEXT(obj)		(G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_TYPE_SCAN_CONTEXT))
#define GNOME_IS_SCAN_CONTEXT_CLASS(klass)	(G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_TYPE_SCAN_CONTEXT))
#define GNOME_SCAN_CONTEXT_GET_CLASS(obj)	(G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_TYPE_SCAN_CONTEXT, GnomeScanContextClass))

typedef struct _GnomeScanContext		GnomeScanContext;
typedef struct _GnomeScanContextClass		GnomeScanContextClass;

struct _GnomeScanContext {
  GObject			parent;
};

struct _GnomeScanContextClass {
  GObjectClass				parent;
  /* SIGNALS */
  void (* probe_done)			(GnomeScanContext *context,
					 gpointer data);

  void (* scanner_selected)		(GnomeScanContext *context,
					 GnomeScanner *scanner,
					 gpointer data);

  void (* acquisition_started)		(GnomeScanContext *context,
					 GnomeScanForecast *forecast,
					 gpointer data);

  void (* data_received)		(GnomeScanContext *context,
					 gsize data_size,
					 gpointer data);

  void (* acquisition_paused)		(GnomeScanContext *context,
					 gpointer data);

  void (* acquisition_resumed)		(GnomeScanContext *context,
					 gpointer data);

  void (* acquisition_terminated)	(GnomeScanContext *context,
					 gpointer data);

  void (* image_received)		(GnomeScanContext *context,
					 GnomeScanResult *result,
					 gpointer data);

  void (* preview_started)		(GnomeScanContext *context,
					 GnomeScanForecast *forecast,
					 gpointer data);

  void (* preview_terminated)		(GnomeScanContext *context,
					 gpointer data);

  void (* preview_received)		(GnomeScanContext *context,
					 GnomeScanResult *result,
					 gpointer data);

  void (* error)			(GnomeScanContext *context,
					 GError *error,
					 gpointer data);
};

typedef enum {
  GNOME_SCAN_CONTEXT_ERROR_FAILED,
  GNOME_SCAN_CONTEXT_ERROR_NO_DEVICES
} GnomeScanContextError;

/* GObject functions */
GType			gnome_scan_context_get_type			(void) G_GNUC_CONST;

/* Methods */

GnomeScanContext*	gnome_scan_context_new 				(void);
void			gnome_scan_context_destroy 			(GnomeScanContext *context);

void			gnome_scan_context_probe_scanners		(GnomeScanContext *context);
GSList*			gnome_scan_context_get_scanners			(GnomeScanContext *context);

GnomeScanner*		gnome_scan_context_get_scanner			(GnomeScanContext *context);
void			gnome_scan_context_select_scanner		(GnomeScanContext *context,
									 GnomeScanner *scanner);


gdouble			gnome_scan_context_get_resolution 		(GnomeScanContext *context);
void			gnome_scan_context_set_resolution		(GnomeScanContext *context,
									 gdouble resolution);

GnomeScanArea*		gnome_scan_context_get_area			(GnomeScanContext *context);
void			gnome_scan_context_set_area			(GnomeScanContext *context,
									 GnomeScanArea *area);

GdkPixbufRotation	gnome_scan_context_get_rotation			(GnomeScanContext *context);
void			gnome_scan_context_set_rotation			(GnomeScanContext *context,
									 GdkPixbufRotation rotation);

GnomeScannerSource	gnome_scan_context_get_source			(GnomeScanContext *context);
void			gnome_scan_context_set_source			(GnomeScanContext *context,
									 GnomeScannerSource source);

GnomeScanForecast*	gnome_scan_context_forecast			(GnomeScanContext *context);
void			gnome_scan_context_acquire_preview		(GnomeScanContext *context);
void			gnome_scan_context_start_acquisition		(GnomeScanContext *context);
void			gnome_scan_context_resume_acquisition		(GnomeScanContext *context);
void			gnome_scan_context_stop_acquisition		(GnomeScanContext *context);

#endif
