/*
 *
 * gnome-scanner.c
 *
 * Copyright © 2006 Étienne Bersac
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "gnomescanner.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define GNOME_SCANNER_ERROR			(g_type_qname (GNOME_TYPE_SCANNER))
#define GET_PRIVATE(obj)			(G_TYPE_INSTANCE_GET_PRIVATE ((obj), GNOME_TYPE_SCANNER, GnomeScannerPrivate))

typedef struct _GnomeScannerPrivate		GnomeScannerPrivate;

struct _GnomeScannerPrivate {
  gpointer			id;
};

/* GOBJECT STUFFS */

enum {
  PROP_0,
  PROP_ID,
  PROP_VENDOR,
  PROP_PRODUCT,
  PROP_TYPE,
  PROP_RESOLUTION_RANGE,
  PROP_OPTIONS,
  PROP_GEOMETRY,
  PROP_SOURCES
};

enum {
  N_SIGNALS
};

static guint signals[N_SIGNALS];

void 	gnome_scanner_set_property 		(GObject *obj,
						 guint property_id,
						 const GValue *value,
						 GParamSpec *pspec);

void 	gnome_scanner_get_property	 	(GObject *obj,
						 guint property_id,
						 GValue *value,
						 GParamSpec *pspec);

G_DEFINE_TYPE (GnomeScanner, gnome_scanner, G_TYPE_OBJECT);

void
gnome_scanner_class_init (GnomeScannerClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

  gobject_class->set_property = gnome_scanner_set_property;
  gobject_class->get_property = gnome_scanner_get_property;

  g_type_class_add_private (gobject_class, sizeof (GnomeScannerPrivate));

  /* properties */

  g_object_class_install_property (gobject_class,
				   PROP_ID,
				   g_param_spec_pointer ("id",
							 "Id",
							 "Backend unique device identifier",
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_VENDOR,
				   g_param_spec_string ("vendor",
							"Vendor",
							"Device vendor name.",
							"Unknown",
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_PRODUCT,
				   g_param_spec_string ("product",
							"Product",
							"Device product name.",
							"Unknown",
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_TYPE,
				   g_param_spec_enum ("type",
						      "Type",
						      "Device type.",
						      GNOME_TYPE_SCANNER_TYPE,
						      GNOME_SCANNER_UNKNOWN,
						      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_RESOLUTION_RANGE,
				   g_param_spec_pointer ("resolution-range",
							 "Resolution range",
							 "Range of available resolutions on the device.",
							 G_PARAM_READWRITE | G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class,
				   PROP_GEOMETRY,
				   g_param_spec_pointer ("geometry",
							 "Geometry",
							 "Geometry of the device's scan area in mm",
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_SOURCES,
				   g_param_spec_value_array ("sources",
							     "Sources",
							     "Available document sources on the device.",
							     g_param_spec_enum ("scanner-source",
										"Scanner source",
										"A source the scanner is available to scan from.",
										GNOME_TYPE_SCANNER_SOURCE,
										GNOME_SCANNER_SOURCE_NONE,
										G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY),
							     G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  g_object_class_install_property (gobject_class,
				   PROP_OPTIONS,
				   g_param_spec_pointer ("options",
							 "Options",
							 "All device's options.",
							 G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

  /* SIGNALS */

  /*
   * GnomeScanner::option-changed:
   * @scanner: The emitting #GnomeScanner
   * @option: The changed #GnomeScannerOption
   *
   */
  /*   signals[SIGNAL_OPTION_CHANGED] = g_signal_new ("option-changed", */
  /* 						 GNOME_TYPE_SCANNER, */
  /* 						 G_SIGNAL_RUN_FIRST, */
  /* 						 G_STRUCT_OFFSET (GnomeScannerClass, option_changed), */
  /* 						 NULL, */
  /* 						 NULL, */
  /* 						 g_cclosure_marshal_VOID__OBJECT, */
  /* 						 G_TYPE_NONE, */
  /* 						 1, */
  /* 						 GNOME_TYPE_SCANNER_OPTION); */
}

void
gnome_scanner_init (GnomeScanner *scanner)
{
  GnomeScannerPrivate *priv = GET_PRIVATE (scanner);

  priv->id 			= NULL;
  scanner->product 		= NULL;
  scanner->vendor 		= NULL;
  scanner->geometry 		= NULL;
  scanner->resolution_range	= NULL;
  scanner->sources		= NULL;
  /* options hash table is set at constructin, not filled. Therefore
     the backend has to create the new GHashTable. */
  scanner->options 		= NULL;
}

void
gnome_scanner_set_property (GObject *obj,
			    guint property_id,
			    const GValue *value,
			    GParamSpec *pspec)
{
  GnomeScanner *scanner = GNOME_SCANNER(obj);
  GnomeScannerPrivate *priv = GET_PRIVATE (scanner);
  GValue *source;

  switch (property_id) {
  case PROP_ID:
    priv->id = g_value_get_pointer (value);
    break;
  case PROP_VENDOR:
    scanner->vendor = g_value_dup_string (value);
    break;
  case PROP_PRODUCT:
    scanner->product = g_value_dup_string (value);
    break;
  case PROP_TYPE:
    scanner->type = g_value_get_enum (value);
    break;
  case PROP_GEOMETRY:
    scanner->geometry = g_value_get_pointer (value);
    break;
  case PROP_SOURCES:
    scanner->sources = g_boxed_copy (G_TYPE_VALUE_ARRAY,
				     g_value_get_boxed (value));
/*     g_debug ("%p %p", g_value_get_boxed (value), g_value_get_boxed (value)); */

/*     source = g_value_array_get_nth (scanner->sources, */
/* 				    0); */
/*     g_debug ("%s sources = %p : %i ; source = %p : %s", __FUNCTION__, scanner->sources, scanner->sources->n_values, source, */
/* 	     g_enum_get_nick (GNOME_TYPE_SCANNER_SOURCE, */
/* 			      g_value_get_enum (source))); */
    break;
  case PROP_RESOLUTION_RANGE:
    scanner->resolution_range = g_value_get_pointer (value);
    break;
  case PROP_OPTIONS:
    scanner->options = g_value_get_pointer (value);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (obj,
				       property_id,
				       pspec);
    break;
  }
}

void
gnome_scanner_get_property (GObject *obj,
			    guint property_id,
			    GValue *value,
			    GParamSpec *pspec)
{
  GnomeScanner *scanner = GNOME_SCANNER(obj);
  GnomeScannerPrivate *priv = GET_PRIVATE (scanner);

  switch (property_id) {
  case PROP_ID:
    g_value_set_pointer (value,
			 (gpointer) priv->id);
    break;
  case PROP_VENDOR:
    g_value_set_string (value,
			scanner->vendor);
    break;
  case PROP_PRODUCT:
    g_value_set_string (value,
			scanner->product);
    break;
  case PROP_TYPE:
    g_value_set_enum (value,
		      scanner->type);
    break;
  case PROP_GEOMETRY:
    g_value_set_pointer (value,
			 scanner->geometry);
    break;
  case PROP_SOURCES:
    g_value_set_boxed (value,
		       scanner->sources);
    break;
  case PROP_RESOLUTION_RANGE:
    g_value_set_pointer (value,
			 scanner->resolution_range);
    break;
  case PROP_OPTIONS:
    g_value_set_pointer (value,
			 (gpointer) scanner->options);
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (obj,
				       property_id,
				       pspec);
    break;
  }
}

/* PROPERTIES */

/**
 * gnome_scanner_get_id:
 * @scanner: a #GnomeScanner
 * 
 * Returns the unique id of the device. This function should be called
 * only by #GnomeScanBackend.
 * 
 * Return value: the device's id
 **/
gpointer
gnome_scanner_get_id (GnomeScanner *scanner)
{
  g_return_val_if_fail (GNOME_IS_SCANNER (scanner), NULL);

  return GET_PRIVATE (scanner)->id;
}

/**
 * gnome_scanner_get_vendor:
 * @scanner: a #GnomeScanner
 * 
 * Returns a string describing the device's vendor.
 * 
 * Return value: the device's vendor
 **/
const gchar*
gnome_scanner_get_vendor (GnomeScanner *scanner)
{
  g_return_val_if_fail (GNOME_IS_SCANNER (scanner), NULL);

  return scanner->vendor;
}

/**
 * gnome_scanner_get_product:
 * @scanner: a #GnomeScanner
 * 
 * Returns a string describing the device's product name.
 * 
 * Return value: the product name
 **/
const gchar*
gnome_scanner_get_product (GnomeScanner *scanner)
{
  g_return_val_if_fail (GNOME_IS_SCANNER (scanner), NULL);

  return scanner->product;
}

/**
 * gnome_scanner_get_type:
 * @scanner: a #GnomeScanner
 * 
 * Return the device type as a string.
 *
 * Return value: the device type
 **/
GnomeScannerType
gnome_scanner_get_scanner_type (GnomeScanner *scanner)
{
  g_return_val_if_fail (GNOME_IS_SCANNER (scanner), GNOME_SCANNER_UNKNOWN);

  return scanner->type;
}

/**
 * gnome_scanner_get_sources:
 * @scanner: a #GnomeScanner
 * 
 * 
 * 
 * Return value: the availables document sources
 **/
GValueArray*
gnome_scanner_get_sources (GnomeScanner *scanner)
{
  g_return_val_if_fail (GNOME_IS_SCANNER (scanner), NULL);

  return scanner->sources;
}

/**
 * gnome_scanner_get_geometry:
 * @scanner: a #GnomeScanner
 * 
 * 
 * 
 * Return value: the device size
 **/
GnomeScanGeometry*
gnome_scanner_get_geometry (GnomeScanner *scanner)
{
  g_return_val_if_fail (GNOME_IS_SCANNER (scanner), NULL);
  return scanner->geometry;
}

/**
 * gonme_scanner_get_resolution_range:
 * @scanner: a #GnomeScanner
 * 
 * 
 * 
 * Return value: the resolution range
 **/
GnomeScanRange*
gnome_scanner_get_resolution_range (GnomeScanner *scanner)
{
  g_return_val_if_fail (GNOME_IS_SCANNER (scanner), NULL);

  return scanner->resolution_range;
}

/**
 * gnome_scanner_foreach_options:
 * @scanner: a #GnomeScanner
 * @func: a #GHFunc callback
 * @data: an optional parameter to pass to @func
 * 
 * Retreive each option and apply the function @func with a
 * #GnomeScannerOption and @data as parameter.
 * 
 **/
void
gnome_scanner_foreach_options (GnomeScanner *scanner,
			       GHFunc func,
			       gpointer data)
{
  g_return_if_fail (GNOME_IS_SCANNER (scanner));

  g_hash_table_foreach (scanner->options,
			func,
			data);
}

/**
 * gnome_scanner_find_option:
 * @scanner: a #GnomeScanner
 * @name: a unique option's name
 * 
 * Search an option by its name in the lists of options. If the search
 * fail, a NULL pointer is returned.
 * 
 * Return value: the corresponding #GnomeScannerOption or NULL
 **/
GnomeScannerOption*
gnome_scanner_find_option (GnomeScanner *scanner,
			   const gchar *name)
{
  return g_hash_table_lookup (scanner->options,
			      name);
}
