/* -*- mode: C; c-basic-offset: 4 -*- */
#include <pygobject.h>
#include "pygnomevfs.h"

static PyObject *pygnome_vfs_result_exc = NULL;

gboolean
pygnome_vfs_result_check(GnomeVFSResult result)
{
    if (result != GNOME_VFS_OK) {
	const char *message;
	PyObject *exc_instance, *d;

	message = gnome_vfs_result_to_string(result);
	exc_instance = PyObject_CallFunction(pygnome_vfs_result_exc, "z",
					     message);
	PyObject_SetAttrString(exc_instance, "result",
			       d=PyInt_FromLong(result));
	Py_DECREF(d);
	PyErr_SetObject(pygnome_vfs_result_exc, exc_instance);
	Py_DECREF(exc_instance);
	return TRUE;
    }
    return FALSE;
}

static PyObject *
pygvfs_create(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "open_mode", "exclusive", "perm", NULL };
    PyObject *uri;
    GnomeVFSOpenMode open_mode = GNOME_VFS_OPEN_NONE;
    gboolean exclusive = FALSE;
    guint perm = 0666;
    GnomeVFSHandle *handle;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "O|iii:gnome.vfs.create",
				     kwlist, &uri, &open_mode, &exclusive,
				     &perm))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_create_uri(&handle, pygnome_vfs_uri_get(uri),
				      open_mode, exclusive, perm);
    else if (PyString_Check(uri))
	result = gnome_vfs_create(&handle, PyString_AsString(uri),
				  open_mode, exclusive, perm);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    return pygnome_vfs_handle_new(handle);
}

static PyObject *
pygvfs_get_file_info(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "options", NULL };
    PyObject *uri;
    GnomeVFSFileInfo *finfo;
    GnomeVFSFileInfoOptions options = GNOME_VFS_FILE_INFO_DEFAULT;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O|i:gnome.vfs.get_file_info",
				     kwlist, &uri, &options))
	return NULL;

    finfo = gnome_vfs_file_info_new();
    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_get_file_info_uri(pygnome_vfs_uri_get(uri), finfo,
					     options);
    else if (PyString_Check(uri))
	result = gnome_vfs_get_file_info(PyString_AsString(uri), finfo,
					 options);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	gnome_vfs_file_info_unref(finfo);
	return NULL;
    }

    if (pygnome_vfs_result_check(result)) {
	gnome_vfs_file_info_unref(finfo);
	return NULL;
    }
    return pygnome_vfs_file_info_new(finfo);
}

static PyObject *
pygvfs_truncate(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "length", NULL };
    PyObject *uri, *py_length;
    GnomeVFSFileSize length;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "OO:gnome.vfs.truncate",
				     kwlist, &uri, &py_length))
	return NULL;

    length = PyLong_Check(py_length) ? PyLong_AsUnsignedLongLong(py_length)
	: PyInt_AsLong(py_length);
    if (PyErr_Occurred()) return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_truncate_uri(pygnome_vfs_uri_get(uri), length);
    else if (PyString_Check(uri))
	result = gnome_vfs_truncate(PyString_AsString(uri), length);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_make_directory(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", "perm", NULL };
    PyObject *uri;
    gint perm;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "Oi:gnome.vfs.make_directory", kwlist,
				     &uri, &perm))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_make_directory_for_uri(pygnome_vfs_uri_get(uri),
						  perm);
    else if (PyString_Check(uri))
	result = gnome_vfs_make_directory(PyString_AsString(uri), perm);
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_remove_directory(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *uri;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O:gnome.vfs.remove_directory", kwlist,
				     &uri))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_remove_directory_from_uri(pygnome_vfs_uri_get(uri));
    else if (PyString_Check(uri))
	result = gnome_vfs_remove_directory(PyString_AsString(uri));
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_unlink(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *uri;
    GnomeVFSResult result;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs,
				     "O:gnome.vfs.unlink", kwlist, &uri))
	return NULL;

    if (PyObject_TypeCheck(uri, &PyGnomeVFSURI_Type))
	result = gnome_vfs_unlink_from_uri(pygnome_vfs_uri_get(uri));
    else if (PyString_Check(uri))
	result = gnome_vfs_unlink(PyString_AsString(uri));
    else {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }

    if (pygnome_vfs_result_check(result))
	return NULL;

    Py_INCREF(Py_None);
    return Py_None;
}

static PyObject *
pygvfs_exists(PyObject *self, PyObject *args, PyObject *kwargs)
{
    static char *kwlist[] = { "uri", NULL };
    PyObject *py_uri;
    GnomeVFSURI *uri = NULL;
    gboolean exists;

    if (!PyArg_ParseTupleAndKeywords(args, kwargs, "O:gnome.vfs.exists",
				     kwlist, &py_uri))
	return NULL;

    if (PyObject_TypeCheck(py_uri, &PyGnomeVFSURI_Type))
	uri = gnome_vfs_uri_ref(pygnome_vfs_uri_get(py_uri));
    else if (PyString_Check(py_uri))
	uri = gnome_vfs_uri_new(PyString_AsString(py_uri));

    if (!uri) {
	PyErr_SetString(PyExc_TypeError,
			"uri must be a gnome.vfs.URI or a string");
	return NULL;
    }
    exists = gnome_vfs_uri_exists(uri);
    gnome_vfs_uri_unref(uri);

    return PyInt_FromLong(exists);
}

static PyObject *
pygvfs_get_mime_type(PyObject *self, PyObject *args)
{
    char *text_uri;
    
    if(!PyArg_ParseTuple(args, "s:gnome.vfs.get_mime_type",
			 &text_uri))
	return NULL;

    return PyString_FromString(gnome_vfs_get_mime_type(text_uri));
}

static PyObject *
pygvfs_get_mime_type_for_data(PyObject *self, PyObject *args)
{
    char *data;
    int data_size;
    
    if(!PyArg_ParseTuple(args, "si:gnome.vfs.get_mime_type_for_data",
			 &data, &data_size))
	return NULL;

    return PyString_FromString(gnome_vfs_get_mime_type_for_data(data,
								data_size));
}

static PyMethodDef pygnomevfs_functions[] = {
    { "create", (PyCFunction)pygvfs_create, METH_VARARGS|METH_KEYWORDS },
    { "get_file_info", (PyCFunction)pygvfs_get_file_info,
      METH_VARARGS|METH_KEYWORDS },
    { "truncate", (PyCFunction)pygvfs_truncate, METH_VARARGS|METH_KEYWORDS },
    { "make_directory", (PyCFunction)pygvfs_make_directory,
      METH_VARARGS|METH_KEYWORDS },
    { "remove_directory", (PyCFunction)pygvfs_remove_directory,
      METH_VARARGS|METH_KEYWORDS },
    { "unlink", (PyCFunction)pygvfs_unlink, METH_VARARGS|METH_KEYWORDS },
    { "exists", (PyCFunction)pygvfs_exists, METH_VARARGS|METH_KEYWORDS },
    { "get_mime_type", (PyCFunction)pygvfs_get_mime_type, METH_VARARGS },
    { "get_mime_type_for_data", (PyCFunction)pygvfs_get_mime_type_for_data,
      METH_VARARGS },
    { NULL, NULL, 0 }
};

static void
register_constants(PyObject *module)
{
#define regconst(x) PyModule_AddIntConstant(module, #x, GNOME_VFS_ ## x)
    regconst(FILE_FLAGS_NONE);
    regconst(FILE_FLAGS_SYMLINK);
    regconst(FILE_FLAGS_LOCAL);
    regconst(FILE_TYPE_UNKNOWN);
    regconst(FILE_TYPE_REGULAR);
    regconst(FILE_TYPE_DIRECTORY);
    regconst(FILE_TYPE_FIFO);
    regconst(FILE_TYPE_SOCKET);
    regconst(FILE_TYPE_CHARACTER_DEVICE);
    regconst(FILE_TYPE_BLOCK_DEVICE);
    regconst(FILE_TYPE_SYMBOLIC_LINK);
    regconst(FILE_INFO_FIELDS_NONE);
    regconst(FILE_INFO_FIELDS_TYPE);
    regconst(FILE_INFO_FIELDS_PERMISSIONS);
    regconst(FILE_INFO_FIELDS_FLAGS);
    regconst(FILE_INFO_FIELDS_DEVICE);
    regconst(FILE_INFO_FIELDS_INODE);
    regconst(FILE_INFO_FIELDS_LINK_COUNT);
    regconst(FILE_INFO_FIELDS_SIZE);
    regconst(FILE_INFO_FIELDS_BLOCK_COUNT);
    regconst(FILE_INFO_FIELDS_IO_BLOCK_SIZE);
    regconst(FILE_INFO_FIELDS_ATIME);
    regconst(FILE_INFO_FIELDS_MTIME);
    regconst(FILE_INFO_FIELDS_CTIME);
    regconst(FILE_INFO_FIELDS_SYMLINK_NAME);
    regconst(FILE_INFO_FIELDS_MIME_TYPE);
    regconst(PERM_SUID);
    regconst(PERM_SGID);
    regconst(PERM_STICKY);
    regconst(PERM_USER_READ);
    regconst(PERM_USER_WRITE);
    regconst(PERM_USER_EXEC);
    regconst(PERM_USER_ALL);
    regconst(PERM_GROUP_READ);
    regconst(PERM_GROUP_WRITE);
    regconst(PERM_GROUP_EXEC);
    regconst(PERM_GROUP_ALL);
    regconst(PERM_OTHER_READ);
    regconst(PERM_OTHER_WRITE);
    regconst(PERM_OTHER_EXEC);
    regconst(PERM_OTHER_ALL);
    regconst(FILE_INFO_DEFAULT);
    regconst(FILE_INFO_GET_MIME_TYPE);
    regconst(FILE_INFO_FORCE_FAST_MIME_TYPE);
    regconst(FILE_INFO_FORCE_SLOW_MIME_TYPE);
    regconst(FILE_INFO_FOLLOW_LINKS);
    regconst(SET_FILE_INFO_NONE);
    regconst(SET_FILE_INFO_NAME);
    regconst(SET_FILE_INFO_PERMISSIONS);
    regconst(SET_FILE_INFO_OWNER);
    regconst(SET_FILE_INFO_TIME);
    regconst(DIRECTORY_VISIT_DEFAULT);
    regconst(DIRECTORY_VISIT_SAMEFS);
    regconst(DIRECTORY_VISIT_LOOPCHECK);
    regconst(OPEN_NONE);
    regconst(OPEN_READ);
    regconst(OPEN_WRITE);
    regconst(OPEN_RANDOM);
    regconst(SEEK_START);
    regconst(SEEK_CURRENT);
    regconst(SEEK_END);
#undef regconst
}

DL_EXPORT(void)
initvfs(void)
{
    PyObject *m, *d;

    PyGnomeVFSURI_Type.ob_type = &PyType_Type;
    PyGnomeVFSFileInfo_Type.ob_type = &PyType_Type;
    PyGnomeVFSDirectoryHandle_Type.ob_type = &PyType_Type;
    PyGnomeVFSHandle_Type.ob_type = &PyType_Type;

    init_pygobject();
    if (!gnome_vfs_init()) {
	PyErr_SetString(PyExc_RuntimeError, "could not initialise gnome.vfs");
	return;
    }

    if (PyType_Ready(&PyGnomeVFSURI_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSFileInfo_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSDirectoryHandle_Type) < 0)
	return;
    if (PyType_Ready(&PyGnomeVFSHandle_Type) < 0)
	return;

    m = Py_InitModule("gnome.vfs", pygnomevfs_functions);
    d = PyModule_GetDict(m);

    register_constants(m);

    pygnome_vfs_result_exc = PyErr_NewException("gnome.vfs.error",
						PyExc_RuntimeError, NULL);
    PyDict_SetItemString(d, "error", pygnome_vfs_result_exc);

    PyDict_SetItemString(d, "URI", (PyObject *)&PyGnomeVFSURI_Type);
    PyDict_SetItemString(d, "FileInfo", (PyObject *)&PyGnomeVFSFileInfo_Type);
    PyDict_SetItemString(d, "DirectoryHandle",
			 (PyObject *)&PyGnomeVFSDirectoryHandle_Type);
    PyDict_SetItemString(d, "Handle", (PyObject *)&PyGnomeVFSHandle_Type);

    PyDict_SetItemString(d, "open_directory",
			 (PyObject *)&PyGnomeVFSDirectoryHandle_Type);
    PyDict_SetItemString(d, "open", (PyObject *)&PyGnomeVFSHandle_Type);

    if (PyErr_Occurred()) {
	PyErr_Print();
	Py_FatalError("can't initialise module gnome.vfs");
    }
}
