/*
 * gnome-print-rbuf.c: A rasterizer for GnomePrint
 *
 * Author:
 *   Lauris Kaplinski <lauris@ariman.ee>
 *
 * Copyright (C) Lauris Kaplinski, 2000
 *
 * TODO
 *
 * - figure out, how is path & currentpoint handled during fill, clip, show...
 * - implement new libart rendering/clipping, when available
 * - glyph outline cache (should be in gnome-font)
 * - are dashes in device or current coordinates?
 *
 */

#include <config.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>

#include <libart_lgpl/art_misc.h>
#include <libart_lgpl/art_affine.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_bpath.h>
#include <libart_lgpl/art_vpath_bpath.h>
#include <libart_lgpl/art_svp.h>
#include <libart_lgpl/art_svp_wind.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <libart_lgpl/art_svp_vpath_stroke.h>
#include <libart_lgpl/art_svp_ops.h>
#include <libart_lgpl/art_rect_svp.h>
#include <libart_lgpl/art_vpath_dash.h>
#include <libart_lgpl/art_vpath_svp.h>
#include <libart_lgpl/art_rgb_svp.h>
#include <libart_lgpl/art_rgb_rgba_affine.h>

#include <libgnomeprint/gp-gc.h>
#include "art_rgba_svp.h"
#include "art_rgba_rgba_affine.h"

#include <libgnomeprint/gnome-print-private.h>
#include <libgnomeprint/gnome-print-rbuf.h>
#include <libgnomeprint/gnome-print-rbuf-private.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprint/gt1-parset1.h>

/*
 * Private structures
 */

struct _GnomePrintRBufPrivate {
	guchar * pixels;
	gint width;
	gint height;
	gint rowstride;

	gdouble page2buf[6];

	GPGC * gc;

	guint32 alpha : 1;
};

static void gpb_class_init (GnomePrintRBufClass *class);
static void gpb_init (GnomePrintRBuf *rbuf);
static void gpb_destroy (GtkObject *object);

static gint gpb_fill (GnomePrintContext * pc);
static gint gpb_eofill (GnomePrintContext * pc);
static gint gpb_stroke (GnomePrintContext * pc);
static gint gpb_show_ucs4 (GnomePrintContext * pc, guint32 * buf, gint length);

static gint gpb_grayimage (GnomePrintContext * pc, const gchar * data, gint width, gint height, gint rowstride);
static gint gpb_rgbimage (GnomePrintContext * pc, const gchar * data, gint width, gint height, gint rowstride);
static gint gpb_rgbaimage (GnomePrintContext * pc, const gchar * data, gint width, gint height, gint rowstride);

static gint gpb_gsave (GnomePrintContext * pc);
static gint gpb_grestore (GnomePrintContext * pc);

static gint gpb_newpath (GnomePrintContext * pc);
static gint gpb_moveto (GnomePrintContext * pc, gdouble x, gdouble y);
static gint gpb_lineto (GnomePrintContext * pc, gdouble x, gdouble y);
static gint gpb_curveto (GnomePrintContext * pc, gdouble x1, gdouble y1, gdouble x2, gdouble y2, gdouble x3, gdouble y3);
static gint gpb_closepath (GnomePrintContext * pc);
static gint gpb_strokepath (GnomePrintContext * pc);

static gint gpb_setrgbcolor (GnomePrintContext * pc, gdouble r, gdouble g, gdouble b);
static gint gpb_setlinewidth (GnomePrintContext * pc, gdouble width);
static gint gpb_setmiterlimit (GnomePrintContext * pc, gdouble limit);
static gint gpb_setlinejoin (GnomePrintContext * pc, gint jointype);
static gint gpb_setlinecap (GnomePrintContext * pc, gint captype);
static gint gpb_setdash (GnomePrintContext * pc, gint n_values, const gdouble * values, gdouble offset);
static gint gpb_setopacity (GnomePrintContext * pc, gdouble opacity);

static gint gpb_setfont (GnomePrintContext * pc, GnomeFont * font);

static gint gpb_concat (GnomePrintContext * pc, const gdouble matrix[6]);
#ifdef SETMATRIX_DEPRECATED
static gint gpb_setmatrix (GnomePrintContext * pc, const gdouble matrix[6]);
#endif 

static gint gpb_clip (GnomePrintContext * pc);
static gint gpb_eoclip (GnomePrintContext * pc);

static gint gpb_textline (GnomePrintContext * pc, GnomeTextLine * textline);

static gint gpb_beginpage (GnomePrintContext * pc, const gchar * name);
static gint gpb_showpage (GnomePrintContext * pc);
static gint gpb_close (GnomePrintContext * pc);

static void gp_svp_uncross_to_render (GnomePrintRBufPrivate * rbp, const ArtSVP * svp, ArtWindRule rule);
static void gp_vpath_to_render (GnomePrintRBufPrivate * rbp, const ArtBpath * bpath, ArtWindRule rule);

static void gp_render_silly_rgba (GnomePrintRBufPrivate * rbp, const guchar * pixels, gint width, gint height, gint rowstride);

static GnomePrintContextClass * print_rbuf_parent_class;

/**
 * gnome_print_rbuf_get_type:
 *
 * GTK type identification routine for #GnomePrintRBuf
 *
 * Returns: The Gtk type for the #GnomePrintRBuf object
 */

GtkType
gnome_print_rbuf_get_type (void)
{
	static GtkType type = 0;

	if (!type){
		GtkTypeInfo info = {
			"GnomePrintRBuf",
			sizeof (GnomePrintRBuf),
			sizeof (GnomePrintRBufClass),
			(GtkClassInitFunc) gpb_class_init,
			(GtkObjectInitFunc) gpb_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		type = gtk_type_unique (gnome_print_context_get_type (), &info);
	}
	return type;
}

static void
gpb_class_init (GnomePrintRBufClass *class)
{
	GtkObjectClass * object_class;
	GnomePrintContextClass * pc_class;

	object_class = (GtkObjectClass *) class;
	pc_class = (GnomePrintContextClass *) class;

	print_rbuf_parent_class = gtk_type_class (gnome_print_context_get_type ());

	object_class->destroy = gpb_destroy;
	
	pc_class->fill = gpb_fill;
	pc_class->eofill = gpb_eofill;
	pc_class->stroke = gpb_stroke;
	pc_class->show_ucs4 = gpb_show_ucs4;
	pc_class->grayimage = gpb_grayimage;
	pc_class->rgbimage = gpb_rgbimage;
	pc_class->rgbaimage = gpb_rgbaimage;

	pc_class->newpath = gpb_newpath;
	pc_class->moveto = gpb_moveto;
	pc_class->lineto = gpb_lineto;
	pc_class->curveto = gpb_curveto;
	pc_class->closepath = gpb_closepath;
	pc_class->strokepath = gpb_strokepath;

	pc_class->setrgbcolor = gpb_setrgbcolor;
	pc_class->setopacity = gpb_setopacity;
	pc_class->setlinewidth = gpb_setlinewidth;
	pc_class->setmiterlimit = gpb_setmiterlimit;
	pc_class->setlinejoin = gpb_setlinejoin;
	pc_class->setlinecap = gpb_setlinecap;
	pc_class->setdash = gpb_setdash;

	pc_class->setfont = gpb_setfont;

	pc_class->concat = gpb_concat;
#ifdef SETMATRIX_DEPRECATED
	pc_class->setmatrix = gpb_setmatrix;
#endif	
	pc_class->gsave = gpb_gsave;
	pc_class->grestore = gpb_grestore;

	pc_class->clip = gpb_clip;
	pc_class->eoclip = gpb_eoclip;

	pc_class->textline = gpb_textline;
	pc_class->showpage = gpb_showpage;

	pc_class->beginpage = gpb_beginpage;

	pc_class->close = gpb_close;
}

static void
gpb_init (GnomePrintRBuf * rbuf)
{
	rbuf->private = g_new (GnomePrintRBufPrivate, 1);
	g_assert (rbuf->private != NULL);

	rbuf->private->pixels = NULL;
	art_affine_identity (rbuf->private->page2buf);
	rbuf->private->gc = NULL;
}

static void
gpb_destroy (GtkObject *object)
{
	GnomePrintRBuf * rbuf;

	rbuf = GNOME_PRINT_RBUF (object);

	if (rbuf->private) {
		if (rbuf->private->gc) {
			gp_gc_unref (rbuf->private->gc);
		}
		g_free (rbuf->private);
	}

	if (GTK_OBJECT_CLASS (print_rbuf_parent_class)->destroy)
		(* GTK_OBJECT_CLASS (print_rbuf_parent_class)->destroy) (object);
}

/*
 * Drawing methods
 */

static gint
gpb_fill (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
	GPGC * gc;
	const GPPath * gppath;
	GPPath * closedpath;
	ArtBpath * bpath;

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), -1);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;
	gc = rbuf->private->gc;

	g_return_val_if_fail (gp_gc_has_currentpath (gc), -1);

	gppath = gp_gc_get_currentpath (gc);
	g_assert (gppath != NULL);

	closedpath = gp_path_close_all (gppath);
	g_assert (closedpath != NULL);

	bpath = gp_path_bpath (closedpath);
	g_assert (bpath != NULL);

	gp_vpath_to_render (rbp, bpath, ART_WIND_RULE_NONZERO);

	gp_path_unref (closedpath);

	gp_gc_newpath (gc);

	return 1;
}

static gint
gpb_eofill (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
	GPGC * gc;
	const GPPath * gppath;
	GPPath * closedpath;
	ArtBpath * bpath;

	g_return_val_if_fail (pc != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), -1);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;
	gc = rbuf->private->gc;

	g_return_val_if_fail (gp_gc_has_currentpath (gc), -1);

	gppath = gp_gc_get_currentpath (gc);
	g_assert (gppath != NULL);

	closedpath = gp_path_close_all (gppath);
	g_assert (closedpath != NULL);

	bpath = gp_path_bpath (closedpath);
	g_assert (bpath != NULL);

	gp_vpath_to_render (rbp, bpath, ART_WIND_RULE_ODDEVEN);

	gp_path_unref (closedpath);

	gp_gc_newpath (gc);

	return 1;
}

/* fixme: use real pen transforming */

static gint
gpb_stroke (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
	GPGC * gc;
	const GPPath * gppath;
	ArtBpath * bpath;
	ArtVpath * vpath;
	ArtSVP * svp;
	const ArtVpathDash * dash;
	gdouble linewidth;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;
	gc = rbuf->private->gc;

	g_return_val_if_fail (gp_gc_currentpath_points (gc) > 1, -1);

	gppath = gp_gc_get_currentpath (gc);
	g_assert (gppath != NULL);

	bpath = gp_path_bpath (gppath);
	g_assert (bpath != NULL);

	vpath = art_bez_path_to_vec (bpath, 0.25);
	g_assert (vpath != NULL);

	dash = gp_gc_get_dash (gc);

	if ((dash->n_dash > 0) && (dash->dash != NULL)) {
		ArtVpath * dvp;
		dvp = art_vpath_dash (vpath, dash);
		g_assert (dvp != NULL);
		art_free (vpath);
		vpath = dvp;
	}

	/* fixme */

	linewidth = gp_gc_get_linewidth (gc);

	svp = art_svp_vpath_stroke (vpath,
		gp_gc_get_linejoin (gc),
		gp_gc_get_linecap (gc),
		linewidth,
		gp_gc_get_miterlimit (gc),
		0.25);
	g_assert (svp != NULL);
	art_free (vpath);

	gp_svp_uncross_to_render (rbp, svp, ART_WIND_RULE_NONZERO);

	art_svp_free (svp);

	/* fixme: is this correct? */
	gp_gc_newpath (gc);

	return 1;
}

/* fixme: */

static gint
gpb_show_ucs4 (GnomePrintContext * pc, guint32 * buf, gint length)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
	GPGC * gc;
	const GnomeFont * font;
#if 0
	GnomeFontMap * fm;
#else
	const GnomeFontFace * face;
	gdouble size;
	const ArtBpath * bpath;
	ArtPoint advance;
	gint code;
#endif
	const ArtPoint * p;
#if 0
	Gt1LoadedFont * gt1f;
	Gt1GlyphOutline * gol;
#endif
	gdouble affine[6];
	const gdouble * ctm;
	gdouble x, y;
	gint i;
	ArtBpath * abp;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;
	gc = rbuf->private->gc;

	font = gp_gc_get_font (gc);
#if 0
	fm = font->fontmap_entry;
	g_return_val_if_fail (fm != NULL, -1);

	gt1f = gt1_load_font (fm->pfb_fn);
	g_assert (gt1f != NULL);
#else
	face = gnome_font_get_face ((GnomeFont *) font);
	size = gnome_font_get_size ((GnomeFont *) font);
#endif

	p = gp_gc_get_currentpoint (gc);
	g_return_val_if_fail (p != NULL, -1);

	ctm = gp_gc_get_ctm (gc);

	x = y = 0.0;

	for (i = 0; i < length; i++) {
		code = gnome_font_face_lookup_default ((GnomeFontFace *) face, GUINT32_FROM_BE (buf[i]));
		bpath = gnome_font_face_get_glyph_stdoutline ((GnomeFontFace *) face, code);

		art_affine_scale (affine,
			size / 1000.0,
			size / 1000.0);
		affine[4] = x;
		affine[5] = y;
		art_affine_multiply (affine, affine, ctm);
		affine[4] = affine[4] - ctm[4] + p->x;
		affine[5] = affine[5] - ctm[5] + p->y;

#if 0
		abp = art_bpath_affine_transform (gol->bpath, affine);
		g_assert (abp != NULL);
#else
		abp = art_bpath_affine_transform (bpath, affine);
#endif

		gp_vpath_to_render (rbp, abp, ART_WIND_RULE_NONZERO);

#if 0
		x += gol->wx * font->size / 1000.0;
#else
		gnome_font_face_get_glyph_stdadvance ((GnomeFontFace *) face, code, &advance);
		x += advance.x * size / 1000.0;
#endif
	}

#if 0
	gt1_unload_font (gt1f);
#endif

	/* fixme: is this correct? */
	gp_gc_newpath (gc);

	return 1;
}

/* fixme: clipping */

static gint
gpb_grayimage (GnomePrintContext * pc, const gchar * data, gint width, gint height, gint rowstride)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
	GPGC * gc;
	art_u8 * ib, * ipd;
	const art_u8 * ips;
	gint x, y;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);
	g_return_val_if_fail (data != NULL, 0);
	g_return_val_if_fail (width > 0, 0);
	g_return_val_if_fail (height > 0, 0);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;
	gc = rbuf->private->gc;

	ib = g_new (art_u8, width * height * 4);

	for (y = 0; y < height; y++) {
		ips = data + y * rowstride;
		ipd = ib + y * width * 4;
		for (x = 0; x < width; x++) {
			* ipd++ = * ips;
			* ipd++ = * ips;
			* ipd++ = * ips++;
			* ipd++ = 0xff;
		}
	}

	gp_render_silly_rgba (rbp, ib, width, height, rowstride);

	g_free (ib);

	return 1;
}

static gint
gpb_rgbimage (GnomePrintContext * pc, const gchar * data, gint width, gint height, gint rowstride)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
	GPGC * gc;
	art_u8 * ib, * ipd;
	const art_u8 * ips;
	gint x, y;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);
	g_return_val_if_fail (data != NULL, 0);
	g_return_val_if_fail (width > 0, 0);
	g_return_val_if_fail (height > 0, 0);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;
	gc = rbuf->private->gc;

	ib = g_new (art_u8, width * height * 4);

	for (y = 0; y < height; y++) {
		ips = data + y * rowstride;
		ipd = ib + y * width * 4;
		for (x = 0; x < width; x++) {
			* ipd++ = * ips++;
			* ipd++ = * ips++;
			* ipd++ = * ips++;
			* ipd++ = 0xff;
		}
	}

	gp_render_silly_rgba (rbp, ib, width, height, width * 4);

	g_free (ib);

	return 1;
}

/* fixme: clipping */

static gint
gpb_rgbaimage (GnomePrintContext * pc, const gchar * data, gint width, gint height, gint rowstride)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
	GPGC * gc;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);
	g_return_val_if_fail (data != NULL, 0);
	g_return_val_if_fail (width > 0, 0);
	g_return_val_if_fail (height > 0, 0);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;
	gc = rbuf->private->gc;

	gp_render_silly_rgba (rbp, data, width, height, rowstride);

	return 1;
}

/*
 * Gsave & Grestore
 */

static gint
gpb_gsave (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_gsave (rbuf->private->gc);

	return 1;
}

static gint
gpb_grestore (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_grestore (rbuf->private->gc);

	return 1;
}


/*
 * Path operations
 */

static gint
gpb_newpath (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_newpath (rbuf->private->gc);

	return 1;
}

static gint
gpb_moveto (GnomePrintContext * pc, gdouble x, gdouble y)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_moveto (rbuf->private->gc, x, y);

	return 1;
}

static gint
gpb_lineto (GnomePrintContext * pc, gdouble x, gdouble y)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_lineto (rbuf->private->gc, x, y);

	return 1;
}

static gint
gpb_curveto (GnomePrintContext * pc, gdouble x1, gdouble y1, gdouble x2, gdouble y2, gdouble x3, gdouble y3)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_curveto (rbuf->private->gc, x1, y1, x2, y2, x3, y3);

	return 1;
}

static gint
gpb_closepath (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_closepath (rbuf->private->gc);

	return 1;
}

static gint
gpb_strokepath (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_strokepath (rbuf->private->gc);

	return 1;
}

/*
 * Simple GC Properties
 */

static gint
gpb_setrgbcolor (GnomePrintContext * pc, gdouble r, gdouble g, gdouble b)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_set_rgbcolor (rbuf->private->gc, r, g, b);

	return 1;
}

static gint
gpb_setopacity (GnomePrintContext * pc, gdouble opacity)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_set_opacity (rbuf->private->gc, opacity);

	return 1;
}

static gint
gpb_setlinewidth (GnomePrintContext * pc, gdouble width)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_set_linewidth (rbuf->private->gc, width);

	return 1;
}

static gint
gpb_setmiterlimit (GnomePrintContext * pc, gdouble limit)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_set_miterlimit (rbuf->private->gc, limit);

	return 1;
}

static gint
gpb_setlinejoin (GnomePrintContext * pc, gint jointype)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	switch (jointype) {
	case 0:
		gp_gc_set_linejoin (rbuf->private->gc, ART_PATH_STROKE_JOIN_MITER);
		break;
	case 1:
		gp_gc_set_linejoin (rbuf->private->gc, ART_PATH_STROKE_JOIN_ROUND);
		break;
	case 2:
		gp_gc_set_linejoin (rbuf->private->gc, ART_PATH_STROKE_JOIN_BEVEL);
		break;
	default:
		return -1;
		break;
	}

	return 1;
}

static gint
gpb_setlinecap (GnomePrintContext * pc, gint captype)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	switch (captype) {
	case 0:
		gp_gc_set_linecap (rbuf->private->gc, ART_PATH_STROKE_CAP_BUTT);
		break;
	case 1:
		gp_gc_set_linecap (rbuf->private->gc, ART_PATH_STROKE_CAP_ROUND);
		break;
	case 2:
		gp_gc_set_linecap (rbuf->private->gc, ART_PATH_STROKE_CAP_SQUARE);
		break;
	default:
		return -1;
		break;
	}

	return 1;
}

static gint
gpb_setdash (GnomePrintContext * pc, gint n_values, const gdouble * values, gdouble offset)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_set_dash (rbuf->private->gc, n_values, values, offset);

	return 1;
}

/*
 * Font loading
 */

static gint
gpb_setfont (GnomePrintContext * pc, GnomeFont * font)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);
	g_return_val_if_fail (font != NULL, 0);
	g_return_val_if_fail (GNOME_IS_FONT (font), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_set_font (rbuf->private->gc, font);

	return 1;
}

/*
 * CTM transformation
 */

static gint
gpb_concat (GnomePrintContext * pc, const gdouble matrix[6])
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);
	g_return_val_if_fail (matrix != NULL, 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_concat (rbuf->private->gc, matrix);

	return 1;
}

#ifdef SETMATRIX_DEPRECATED
static gint
gpb_setmatrix (GnomePrintContext * pc, const gdouble matrix[6])
{
	GnomePrintRBuf * rbuf;
	gdouble m[6];

	g_warning ("setmatrix is deprecated!");

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);
	g_return_val_if_fail (matrix != NULL, 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	/* We have to preserve page2buf transform :) */

	art_affine_multiply (m, matrix, rbuf->private->page2buf);

	gp_gc_setmatrix (rbuf->private->gc, m);

	return 1;
}
#endif

/* Clipping */

static gint
gpb_clip (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_clip (rbuf->private->gc);

	return 1;
}

static gint
gpb_eoclip (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_eoclip (rbuf->private->gc);

	return 1;
}

/* Misc */

static gint
gpb_showpage (GnomePrintContext * pc)
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);

	gp_gc_reset (rbuf->private->gc);

	return 1;
}

static gint
gpb_close (GnomePrintContext * pc)
{
	return 1;
}

static gint
gpb_textline (GnomePrintContext * pc, GnomeTextLine * textline)
{
	g_warning ("GnomePrintRBuf::textline is not implemented");

	return 1;
}

/*
 * Beginpage
 *
 * fixme: Currently we simply clear rbuffer
 *
 * showpage, close - do nothing
 */

static gint
gpb_beginpage (GnomePrintContext * pc, const gchar * name)
{
	GnomePrintRBuf * rbuf;
	GnomePrintRBufPrivate * rbp;
#if 0
	guint32 * p;
	gint x, y;
#endif

	g_return_val_if_fail (pc != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (pc), 0);

	rbuf = GNOME_PRINT_RBUF (pc);
	rbp = rbuf->private;

	/* fixme: What exactly should be reset? */
	gp_gc_reset (rbuf->private->gc);
	gp_gc_concat (rbuf->private->gc, rbuf->private->page2buf);

#if 0
	if (rbp->alpha) {
		for (y = 0; y < rbp->height; y++) {
			p = (guint32 *) (rbp->pixels + y * rbp->rowstride);
			for (x = 0; x < rbp->width; x++)
				*p++ = 0xffffff00;
	}
	} else {
		for (y = 0; y < rbp->height; y++) {
			memset (rbp->pixels + y * rbp->rowstride,
				0xff,
				 rbp->width * 3);
		}
	}
#endif

	return 1;
}

/* Constructors */

GnomePrintContext *
gnome_print_rbuf_new (guchar * pixels,
	gint width,
	gint height,
	gint rowstride,
	gdouble page2buf[6],
	gboolean alpha)
	
{
	GnomePrintRBuf * rbuf;

	g_return_val_if_fail (pixels != NULL, NULL);
	g_return_val_if_fail (width > 0, NULL);
	g_return_val_if_fail (height > 0, NULL);
	g_return_val_if_fail (rowstride >= 3 * width, NULL);
	g_return_val_if_fail (page2buf != NULL, NULL);

	rbuf = gtk_type_new (GNOME_TYPE_PRINT_RBUF);

	if (!gnome_print_rbuf_construct (rbuf, pixels, width, height, rowstride, page2buf, alpha)) {
		gtk_object_unref (GTK_OBJECT (rbuf));
	}

	return GNOME_PRINT_CONTEXT (rbuf);
}

GnomePrintRBuf *
gnome_print_rbuf_construct (GnomePrintRBuf * rbuf,
	guchar * pixels,
	gint width,
	gint height,
	gint rowstride,
	gdouble page2buf[6],
	gboolean alpha)
{
	GPGC * gc;

	g_return_val_if_fail (rbuf != NULL, NULL);
	g_return_val_if_fail (GNOME_IS_PRINT_RBUF (rbuf), NULL);
	g_return_val_if_fail (pixels != NULL, NULL);
	g_return_val_if_fail (width > 0, NULL);
	g_return_val_if_fail (height > 0, NULL);
	g_return_val_if_fail (rowstride >= 3 * width, NULL);
	g_return_val_if_fail (page2buf != NULL, NULL);

	g_assert (rbuf->private != NULL);

	rbuf->private->pixels = pixels;
	rbuf->private->width = width;
	rbuf->private->height = height;
	rbuf->private->rowstride = rowstride;
	rbuf->private->alpha = alpha;

	memcpy (rbuf->private->page2buf, page2buf, sizeof (gdouble) * 6);

	gc = gp_gc_new ();
	g_assert (gc != NULL);
	gp_gc_concat (gc, page2buf);

	rbuf->private->gc = gc;

	return rbuf;
}

/* Private helpers */

static void
gp_svp_uncross_to_render (GnomePrintRBufPrivate * rbp, const ArtSVP * svp, ArtWindRule rule)
{
	ArtSVP * svp1, * svp2;

	g_assert (rbp != NULL);
	g_assert (svp != NULL);

	svp2 = art_svp_uncross ((ArtSVP *) svp);
	g_assert (svp2 != NULL);

	svp1 = art_svp_rewind_uncrossed (svp2, rule);
	g_assert (svp1 != NULL);
	art_svp_free (svp2);

	if (gp_gc_has_clipsvp (rbp->gc)) {
		svp2 = art_svp_intersect (svp1, gp_gc_get_clipsvp (rbp->gc));
		g_assert (svp2 != NULL);
		art_svp_free (svp1);
		svp1 = svp2;
	}

	if (rbp->alpha) {
		art_rgba_svp_alpha (svp1,
			0, 0, rbp->width, rbp->height,
			gp_gc_get_rgba (rbp->gc),
			rbp->pixels, rbp->rowstride,
			NULL);
	} else {
		art_rgb_svp_alpha (svp1,
			0, 0, rbp->width, rbp->height,
			gp_gc_get_rgba (rbp->gc),
			rbp->pixels, rbp->rowstride,
			NULL);
	}

	art_svp_free (svp1);
}

static void
gp_vpath_to_render (GnomePrintRBufPrivate * rbp, const ArtBpath * bpath, ArtWindRule rule)
{
	ArtVpath * vpath1, * vpath2;
	ArtSVP * svp;

	g_assert (rbp != NULL);
	g_assert (bpath != NULL);

	vpath1 = art_bez_path_to_vec (bpath, 0.25);
	g_assert (vpath1 != NULL);

	vpath2 = art_vpath_perturb (vpath1);
	g_assert (vpath2 != NULL);
	art_free (vpath1);

	svp = art_svp_from_vpath (vpath2);
	g_assert (svp != NULL);
	art_free (vpath2);

	gp_svp_uncross_to_render (rbp, svp, rule);

	art_svp_free (svp);
}

static void
gp_render_silly_rgba (GnomePrintRBufPrivate * rbp,
	const guchar * pixels, gint width, gint height, gint rowstride)
{
	gdouble affine[6];
	ArtVpath vp[6];
	ArtVpath * vpath;
	ArtSVP * svp1, * svp2;
	ArtDRect bbox, pbox;
	ArtIRect ibox;
	gdouble ba[6];
	guchar * cbuf, * ibuf;
	guchar * p, * ip, * cp;
	gint bw, bh, x, y;

	art_affine_scale (affine, 1.0 / width, -1.0 / height);
	affine[5] = 1.0;
	art_affine_multiply (affine, affine, gp_gc_get_ctm (rbp->gc));

	vp[0].code = ART_MOVETO;
	vp[0].x = 0.0;
	vp[0].y = 0.0;
	vp[1].code = ART_LINETO;
	vp[1].x = width;
	vp[1].y = 0.0;
	vp[2].code = ART_LINETO;
	vp[2].x = width;
	vp[2].y = height;
	vp[3].code = ART_LINETO;
	vp[3].x = 0.0;
	vp[3].y = height;
	vp[4].code = ART_LINETO;
	vp[4].x = 0.0;
	vp[4].y = 0.0;
	vp[5].code = ART_END;

	vpath = art_vpath_affine_transform (vp, affine);

	svp1 = art_svp_from_vpath (vpath);
	art_free (vpath);

	svp2 = art_svp_uncross (svp1);
	art_svp_free (svp1);

	svp1 = art_svp_rewind_uncrossed (svp2, ART_WIND_RULE_NONZERO);
	art_svp_free (svp2);

	if (gp_gc_has_clipsvp (rbp->gc)) {
		svp2 = art_svp_intersect (svp1, gp_gc_get_clipsvp (rbp->gc));
		art_svp_free (svp1);
		svp1 = svp2;
	}

	art_drect_svp (&bbox, svp1);

	pbox.x0 = pbox.y0 = 0.0;
	pbox.x1 = rbp->width;
	pbox.y1 = rbp->height;

	art_drect_intersect (&bbox, &bbox, &pbox);

	if (art_drect_empty (&bbox)) {
		art_svp_free (svp1);
		return;
	}

	art_drect_to_irect (&ibox, &bbox);

	bw = ibox.x1 - ibox.x0;
	bh = ibox.y1 - ibox.y0;

	/* Create coverage */

	cbuf = g_new (guchar, bw * bh * 4);
	for (y = 0; y < bh; y++) {
		p = cbuf + y * bw * 4;
		for (x = 0; x < bw; x++) {
			*p++ = 0;
			*p++ = 0;
			*p++ = 0;
			*p++ = 0;
		}
	}

	art_rgba_svp_alpha (svp1,
		ibox.x0, ibox.y0, ibox.x1, ibox.y1,
		0xffffffff,
		cbuf, bw * 4,
		NULL);

	art_svp_free (svp1);

	/* Create image */

	ibuf = g_new (guchar, bw * bh * 4);
	for (y = 0; y < bh; y++) {
		p = ibuf + y * bw * 4;
		for (x = 0; x < bw; x++) {
			*p++ = 0;
			*p++ = 0;
			*p++ = 0;
			*p++ = 0;
		}
	}

	memcpy (ba, affine, 6 * sizeof (gdouble));
	ba[4] -= ibox.x0;
	ba[5] -= ibox.y0;

	art_rgba_rgba_affine (ibuf,
		0, 0, bw, bh, bw * 4,
		pixels, width, height, rowstride,
		ba,
		ART_FILTER_NEAREST, NULL);

	/* Composite */

	for (y = 0; y < bh; y++) {
		ip = ibuf + y * bw * 4;
		cp = cbuf + y * bw * 4;
		for (x = 0; x < bw; x++) {
			ip += 3;
			cp += 3;
			*ip = (*ip) * (*cp) >> 8;
			ip++;
			cp++;
		}
	}

	art_affine_translate (ba, ibox.x0, ibox.y0);

	/* Render */

	if (rbp->alpha) {
		art_rgba_rgba_affine (rbp->pixels,
			0, 0, rbp->width, rbp->height, rbp->rowstride,
			ibuf, bw, bh, bw * 4,
			ba,
			ART_FILTER_NEAREST, NULL);
	} else {
		art_rgb_rgba_affine (rbp->pixels,
			0, 0, rbp->width, rbp->height, rbp->rowstride,
			ibuf, bw, bh, bw * 4,
			ba,
			ART_FILTER_NEAREST, NULL);
	}

	g_free (ibuf);
	g_free (cbuf);
}


