/* Hacktext item type for GnomeCanvas widget
 *
 * GnomeCanvas is basically a port of the Tk toolkit's most excellent canvas widget.  Tk is
 * copyrighted by the Regents of the University of California, Sun Microsystems, and other parties.
 *
 * Copyright (C) 1998,1999 The Free Software Foundation
 *
 * Authors: Federico Mena <federico@nuclecu.unam.mx>
 *          Raph Levien <raph@acm.org>
 */

/* These includes are set up for standalone compile. If/when this codebase
   is integrated into libgnomeui, the includes will need to change. */
#include <math.h>
#include <string.h>
#include <gnome.h>
#include <libart_lgpl/art_rect.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_bpath.h>
#include <libart_lgpl/art_vpath_bpath.h>
#include <libart_lgpl/art_vpath_dash.h>
#include <libart_lgpl/art_svp.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <libart_lgpl/art_svp_wind.h>
#include <libart_lgpl/art_svp_point.h>
#include "gt1-parset1.h"
#include "gnome-canvas-hacktext.h"

#define NUM_STATIC_POINTS 256	/* Number of static points to use to avoid allocating arrays */


#define GROW_BOUNDS(bx1, by1, bx2, by2, x, y) {	\
	if (x < bx1)				\
		bx1 = x;			\
						\
	if (x > bx2)				\
		bx2 = x;			\
						\
	if (y < by1)				\
		by1 = y;			\
						\
	if (y > by2)				\
		by2 = y;			\
}


enum {
	ARG_0,
	ARG_TEXT,
	ARG_FILL_COLOR,
	ARG_FILL_COLOR_GDK,
	ARG_FILL_COLOR_RGBA,
	ARG_OUTLINE_COLOR,
	ARG_OUTLINE_COLOR_GDK,
	ARG_OUTLINE_COLOR_RGBA,
	ARG_FILL_STIPPLE,
	ARG_OUTLINE_STIPPLE,
	ARG_WIDTH_PIXELS,
	ARG_WIDTH_UNITS,
	ARG_CAP_STYLE,
	ARG_JOIN_STYLE,
	ARG_FONT,
	ARG_SIZE,
	ARG_X,
	ARG_Y
};


static void gnome_canvas_hacktext_class_init (GnomeCanvasHacktextClass *class);
static void gnome_canvas_hacktext_init       (GnomeCanvasHacktext      *hacktext);
static void gnome_canvas_hacktext_destroy    (GtkObject               *object);
static void gnome_canvas_hacktext_set_arg    (GtkObject               *object,
					   GtkArg                  *arg,
					   guint                    arg_id);
static void gnome_canvas_hacktext_get_arg    (GtkObject               *object,
					   GtkArg                  *arg,
					   guint                    arg_id);

static void   gnome_canvas_hacktext_update      (GnomeCanvasItem *item, double *affine, ArtSVP *clip_path, int flags);
static void   gnome_canvas_hacktext_realize     (GnomeCanvasItem *item);
static void   gnome_canvas_hacktext_unrealize   (GnomeCanvasItem *item);
static void   gnome_canvas_hacktext_draw        (GnomeCanvasItem *item, GdkDrawable *drawable,
						int x, int y, int width, int height);
static double gnome_canvas_hacktext_point       (GnomeCanvasItem *item, double x, double y,
						int cx, int cy, GnomeCanvasItem **actual_item);
static void   gnome_canvas_hacktext_bounds      (GnomeCanvasItem *item, double *x1, double *y1, double *x2, double *y2);
static void   gnome_canvas_hacktext_render      (GnomeCanvasItem *item, GnomeCanvasBuf *buf);
static void   gnome_canvas_hacktext_req_repaint (GnomeCanvasHacktext *hacktext, ArtIRect *bbox);


static GnomeCanvasItemClass *parent_class;

typedef struct _GnomeCanvasHacktextCharInfo GnomeCanvasHacktextCharInfo;

/* All coordinates in CharInfo are _unscaled_ so an em = ??? */
struct _GnomeCanvasHacktextCharInfo {
	ArtBpath *bpath; /* bpath shape for the glyph, or NULL if not yet loaded */
	double wx;
	ArtDRect bbox;
};

static GnomeCanvasHacktextCharInfo *gnome_canvas_hacktext_get_info (GnomeCanvasHacktext *hacktext, gint32 ch);

struct _GnomeCanvasHacktextPriv {
	Gt1LoadedFont *font;
	double affine[6]; /* the item to world transform */
	GnomeCanvasHacktextCharInfo info[256];
};

GtkType
gnome_canvas_hacktext_get_type (void)
{
	static GtkType hacktext_type = 0;

	if (!hacktext_type) {
		GtkTypeInfo hacktext_info = {
			"GnomeCanvasHacktext",
			sizeof (GnomeCanvasHacktext),
			sizeof (GnomeCanvasHacktextClass),
			(GtkClassInitFunc) gnome_canvas_hacktext_class_init,
			(GtkObjectInitFunc) gnome_canvas_hacktext_init,
			NULL, /* reserved_1 */
			NULL, /* reserved_2 */
			(GtkClassInitFunc) NULL
		};

		hacktext_type = gtk_type_unique (gnome_canvas_item_get_type (), &hacktext_info);
	}

	return hacktext_type;
}

static void
gnome_canvas_hacktext_class_init (GnomeCanvasHacktextClass *class)
{
	GtkObjectClass *object_class;
	GnomeCanvasItemClass *item_class;

	object_class = (GtkObjectClass *) class;
	item_class = (GnomeCanvasItemClass *) class;

	parent_class = gtk_type_class (gnome_canvas_item_get_type ());

	/* when this gets checked into libgnomeui, change the
           GTK_TYPE_POINTER to GTK_TYPE_GNOME_CANVAS_HACKTEXT, and add an
           entry to gnome-boxed.defs */
	gtk_object_add_arg_type ("GnomeCanvasHacktext::text", GTK_TYPE_STRING, GTK_ARG_READWRITE, ARG_TEXT);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::fill_color", GTK_TYPE_STRING, GTK_ARG_WRITABLE, ARG_FILL_COLOR);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::fill_color_gdk", GTK_TYPE_GDK_COLOR, GTK_ARG_READWRITE, ARG_FILL_COLOR_GDK);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::fill_color_rgba", GTK_TYPE_UINT, GTK_ARG_READWRITE, ARG_FILL_COLOR_RGBA);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::outline_color", GTK_TYPE_STRING, GTK_ARG_WRITABLE, ARG_OUTLINE_COLOR);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::outline_color_gdk", GTK_TYPE_GDK_COLOR, GTK_ARG_READWRITE, ARG_OUTLINE_COLOR_GDK);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::outline_color_rgba", GTK_TYPE_UINT, GTK_ARG_READWRITE, ARG_OUTLINE_COLOR_RGBA);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::fill_stipple", GTK_TYPE_GDK_WINDOW, GTK_ARG_READWRITE, ARG_FILL_STIPPLE);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::outline_stipple", GTK_TYPE_GDK_WINDOW, GTK_ARG_READWRITE, ARG_OUTLINE_STIPPLE);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::width_pixels", GTK_TYPE_UINT, GTK_ARG_WRITABLE, ARG_WIDTH_PIXELS);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::width_units", GTK_TYPE_DOUBLE, GTK_ARG_WRITABLE, ARG_WIDTH_UNITS);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::cap_style", GTK_TYPE_GDK_CAP_STYLE, GTK_ARG_READWRITE, ARG_CAP_STYLE);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::join_style", GTK_TYPE_GDK_JOIN_STYLE, GTK_ARG_READWRITE, ARG_JOIN_STYLE);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::font", GTK_TYPE_POINTER, GTK_ARG_READWRITE, ARG_FONT);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::size", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_SIZE);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::x", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_X);
	gtk_object_add_arg_type ("GnomeCanvasHacktext::y", GTK_TYPE_DOUBLE, GTK_ARG_READWRITE, ARG_Y);

	object_class->destroy = gnome_canvas_hacktext_destroy;
	object_class->set_arg = gnome_canvas_hacktext_set_arg;
	object_class->get_arg = gnome_canvas_hacktext_get_arg;

	item_class->update = gnome_canvas_hacktext_update;
	item_class->realize = gnome_canvas_hacktext_realize;
	item_class->unrealize = gnome_canvas_hacktext_unrealize;
	item_class->draw = gnome_canvas_hacktext_draw;
	item_class->point = gnome_canvas_hacktext_point;
	item_class->bounds = gnome_canvas_hacktext_bounds;
	item_class->render = gnome_canvas_hacktext_render;
}

static void
gnome_canvas_hacktext_init (GnomeCanvasHacktext *hacktext)
{
	int i;

	hacktext->width = 0.0;
	hacktext->cap = GDK_CAP_BUTT;
	hacktext->join = GDK_JOIN_MITER;
	hacktext->text = NULL;
	hacktext->priv = g_new (GnomeCanvasHacktextPriv, 1);
	hacktext->priv->font = NULL;

	art_affine_identity (hacktext->priv->affine);

	for (i = 0; i < 256; i++)
		hacktext->priv->info[i].bpath = NULL;
}

static void
gnome_canvas_hacktext_destroy (GtkObject *object)
{
	GnomeCanvasHacktext *hacktext;
	int i;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_HACKTEXT (object));

	hacktext = GNOME_CANVAS_HACKTEXT (object);

	if (hacktext->text)
		art_free (hacktext->text);

	if (hacktext->fill_stipple)
		gdk_bitmap_unref (hacktext->fill_stipple);

	if (hacktext->outline_stipple)
		gdk_bitmap_unref (hacktext->outline_stipple);

	for (i = 0; i < 256; i++)
		if (hacktext->priv->info[i].bpath)
			art_free (hacktext->priv->info[i].bpath);
	g_free (hacktext->priv);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
art_drect_hacktext (ArtDRect *bbox, GnomeCanvasHacktext *hacktext)
{
	GnomeCanvasHacktextCharInfo *info;
	double x, y;
	const unsigned char *text;
	int i;
	ArtDRect w_bbox;
	double affine[6];

	x = hacktext->x;
	y = hacktext->y;
	text = (const unsigned char *)hacktext->text;
	if (text == NULL)
		return;

	art_affine_identity (affine);

	art_affine_scale (affine, hacktext->size * 0.001, hacktext->size * -0.001);	
	affine[4] = x;
	affine[5] = y;
	for (i = 0; text[i]; i++) {
		info = gnome_canvas_hacktext_get_info (hacktext, text[i]);
		art_drect_affine_transform (&w_bbox, &info->bbox, affine);

		if (bbox)
			art_drect_union (bbox, bbox, &w_bbox);

		affine[4] += info->wx * affine[0];
		affine[5] += info->wx * affine[1];
	}
}

/* Computes the bounding box of the hacktext.  Assumes that the number of points in the hacktext is
 * not zero.
 */
static void
get_bounds (GnomeCanvasHacktext *hacktext, double *bx1, double *by1, double *bx2, double *by2)
{
	double width;
	ArtDRect bbox;

	/* Compute bounds of hacktext */
	art_drect_hacktext (&bbox, hacktext);

	/* Add outline width */

	if (hacktext->width_pixels)
		width = hacktext->width / hacktext->item.canvas->pixels_per_unit;
	else
		width = hacktext->width;

	width /= 2.0;

	bbox.x0 -= width;
	bbox.y0 -= width;
	bbox.x1 += width;
	bbox.y1 += width;

	/* Done */

	*bx1 = bbox.x0;
	*by1 = bbox.y0;
	*bx2 = bbox.x1;
	*by2 = bbox.y1;
}

/* Computes the bounding box of the hacktext, in canvas coordinates.  Assumes that the number of points in the hacktext is
 * not zero.
 */
static void
get_bounds_canvas (GnomeCanvasHacktext *hacktext, double *bx1, double *by1, double *bx2, double *by2, double affine[6])
{
	GnomeCanvasItem *item;
	ArtDRect bbox_world;
	ArtDRect bbox_canvas;

	item = GNOME_CANVAS_ITEM (hacktext);

	get_bounds (hacktext, &bbox_world.x0, &bbox_world.y0, &bbox_world.x1, &bbox_world.y1);

	/* art_drect_affine_transform (&bbox_canvas, &bbox_world, affine);  */
	/* include 1 pixel of fudge */
	*bx1 = bbox_canvas.x0 - 1;
	*by1 = bbox_canvas.y0 - 1;
	*bx2 = bbox_canvas.x1 + 1;
	*by2 = bbox_canvas.y1 + 1;
}

#if 0
static void
recalc_bounds (GnomeCanvasHacktext *hacktext)
{
	GnomeCanvasItem *item;
	double x1, y1, x2, y2;
	int cx1, cy1, cx2, cy2;
	double dx, dy;

	item = GNOME_CANVAS_ITEM (hacktext);

	if (hacktext->text == NULL) {
		item->x1 = item->y1 = item->x2 = item->y2 = 0;
		return;
	}

	/* Get bounds in world coordinates */

	get_bounds (hacktext, &x1, &y1, &x2, &y2);

	/* Convert to canvas pixel coords */

	dx = dy = 0.0;
	gnome_canvas_item_i2w (item, &dx, &dy);

	gnome_canvas_w2c (item->canvas, x1 + dx, y1 + dy, &cx1, &cy1);
	gnome_canvas_w2c (item->canvas, x2 + dx, y2 + dy, &cx2, &cy2);
	item->x1 = cx1;
	item->y1 = cy1;
	item->x2 = cx2;
	item->y2 = cy2;

	/* Some safety fudging */

	item->x1--;
	item->y1--;
	item->x2++;
	item->y2++;

	gnome_canvas_group_child_bounds (GNOME_CANVAS_GROUP (item->parent), item);
}
#endif

/* Convenience function to set a GC's foreground color to the specified pixel value */
static void
set_gc_foreground (GdkGC *gc, gulong pixel)
{
	GdkColor c;

	if (!gc)
		return;

	c.pixel = pixel;
	gdk_gc_set_foreground (gc, &c);
}

/* Sets the stipple pattern for the specified gc */
static void
set_stipple (GdkGC *gc, GdkBitmap **internal_stipple, GdkBitmap *stipple, int reconfigure)
{
	if (*internal_stipple && !reconfigure)
		gdk_bitmap_unref (*internal_stipple);

	*internal_stipple = stipple;
	if (stipple && !reconfigure)
		gdk_bitmap_ref (stipple);

	if (gc) {
		if (stipple) {
			gdk_gc_set_stipple (gc, stipple);
			gdk_gc_set_fill (gc, GDK_STIPPLED);
		} else
			gdk_gc_set_fill (gc, GDK_SOLID);
	}
}

/* Recalculate the outline width of the hacktext and set it in its GC */
static void
set_outline_gc_width (GnomeCanvasHacktext *hacktext)
{
	int width;

	if (!hacktext->outline_gc)
		return;

	if (hacktext->width_pixels)
		width = (int) hacktext->width;
	else
		width = (int) (hacktext->width * hacktext->item.canvas->pixels_per_unit + 0.5);

	gdk_gc_set_line_attributes (hacktext->outline_gc, width,
				    GDK_LINE_SOLID, GDK_CAP_ROUND, GDK_JOIN_ROUND);
}

static void
gnome_canvas_hacktext_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeCanvasItem *item;
	GnomeCanvasHacktext *bp;
	char *text;
	GdkColor color;

	item = GNOME_CANVAS_ITEM (object);
	bp = GNOME_CANVAS_HACKTEXT (object);

	switch (arg_id) {
	case ARG_TEXT:
		text = GTK_VALUE_POINTER (*arg);

		if (bp->text) {
			art_free (bp->text);
			bp->text = NULL;
		}

		if (text)
			bp->text = g_strdup (text);

		gnome_canvas_item_request_update (item);
		break;

	case ARG_FILL_COLOR:
		if (gnome_canvas_get_color (item->canvas, GTK_VALUE_STRING (*arg), &color)) {
			bp->fill_set = TRUE;
			bp->fill_pixel = color.pixel;
			if (item->canvas->aa)
				bp->fill_rgba =
					((color.red & 0xff00) << 16) |
					((color.green & 0xff00) << 8) |
					(color.blue & 0xff00) |
					0xff;
			else
				set_gc_foreground (bp->fill_gc, bp->fill_pixel);
		} else {
			bp->fill_set = FALSE;
			bp->fill_rgba = 0;
		}

		gnome_canvas_item_request_update (item);
		break;

	case ARG_FILL_COLOR_GDK:
		bp->fill_set = TRUE;
		bp->fill_pixel = ((GdkColor *) GTK_VALUE_BOXED (*arg))->pixel;
		set_gc_foreground (bp->fill_gc, bp->fill_pixel);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_FILL_COLOR_RGBA:
		bp->fill_set = TRUE;
		bp->fill_rgba = GTK_VALUE_UINT (*arg);

		/* should probably request repaint on the fill_svp */
		gnome_canvas_item_request_update (item);

		break;

	case ARG_OUTLINE_COLOR:
		if (gnome_canvas_get_color (item->canvas, GTK_VALUE_STRING (*arg), &color)) {
			bp->outline_set = TRUE;
			bp->outline_pixel = color.pixel;
			if (item->canvas->aa)
				bp->outline_rgba =
					((color.red & 0xff00) << 16) |
					((color.green & 0xff00) << 8) |
					(color.blue & 0xff00) |
					0xff;
			else
				set_gc_foreground (bp->outline_gc, bp->outline_pixel);
		} else {
			bp->outline_set = FALSE;
			bp->outline_rgba = 0;
		}

		gnome_canvas_item_request_update (item);
		break;

	case ARG_OUTLINE_COLOR_GDK:
		bp->outline_set = TRUE;
		bp->outline_pixel = ((GdkColor *) GTK_VALUE_BOXED (*arg))->pixel;
		set_gc_foreground (bp->outline_gc, bp->outline_pixel);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_OUTLINE_COLOR_RGBA:
		bp->outline_set = TRUE;
		bp->outline_rgba = GTK_VALUE_UINT (*arg);

		/* should probably request repaint on the outline_svp */
		gnome_canvas_item_request_update (item);

		break;

	case ARG_FILL_STIPPLE:
		set_stipple (bp->fill_gc, &bp->fill_stipple, GTK_VALUE_BOXED (*arg), FALSE);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_OUTLINE_STIPPLE:
		set_stipple (bp->outline_gc, &bp->outline_stipple, GTK_VALUE_BOXED (*arg), FALSE);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_WIDTH_PIXELS:
		bp->width = GTK_VALUE_UINT (*arg);
		bp->width_pixels = TRUE;
		set_outline_gc_width (bp);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_WIDTH_UNITS:
		bp->width = fabs (GTK_VALUE_DOUBLE (*arg));
		bp->width_pixels = FALSE;
		set_outline_gc_width (bp);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_CAP_STYLE:
		bp->cap = GTK_VALUE_ENUM (*arg);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_JOIN_STYLE:
		bp->join = GTK_VALUE_ENUM (*arg);
		gnome_canvas_item_request_update (item);
		break;
	
	case ARG_FONT:
		bp->priv->font = GTK_VALUE_POINTER (*arg);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_SIZE:
		bp->size = GTK_VALUE_DOUBLE (*arg);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_X:
		bp->x = GTK_VALUE_DOUBLE (*arg);
		gnome_canvas_item_request_update (item);
		break;

	case ARG_Y:
		bp->y = GTK_VALUE_DOUBLE (*arg);
		gnome_canvas_item_request_update (item);
		break;

	default:
		break;
	}
}

/* Allocates a GdkColor structure filled with the specified pixel, and puts it into the specified
 * arg for returning it in the get_arg method.
 */
static void
get_color_arg (GnomeCanvasHacktext *hacktext, gulong pixel, GtkArg *arg)
{
	GdkColor *color;

	color = g_new (GdkColor, 1);
	color->pixel = pixel;
	gdk_color_context_query_color (hacktext->item.canvas->cc, color);
	GTK_VALUE_BOXED (*arg) = color;
}

static void
gnome_canvas_hacktext_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeCanvasHacktext *bp;

	bp = GNOME_CANVAS_HACKTEXT (object);

	switch (arg_id) {
	case ARG_TEXT:
		if (bp->text) {
			GTK_VALUE_POINTER (*arg) = g_strdup (bp->text);
		} else
			GTK_VALUE_POINTER (*arg) = NULL;
		break;

	case ARG_FILL_COLOR_GDK:
		get_color_arg (bp, bp->fill_pixel, arg);
		break;
		
	case ARG_OUTLINE_COLOR_GDK:
		get_color_arg (bp, bp->outline_pixel, arg);
		break;

	case ARG_FILL_COLOR_RGBA:
		GTK_VALUE_UINT (*arg) = bp->fill_color;
		break;

	case ARG_FILL_STIPPLE:
		GTK_VALUE_BOXED (*arg) = bp->fill_stipple;
		break;

	case ARG_OUTLINE_STIPPLE:
		GTK_VALUE_BOXED (*arg) = bp->outline_stipple;
		break;

	case ARG_CAP_STYLE:
		GTK_VALUE_ENUM (*arg) = bp->cap;
		break;

	case ARG_JOIN_STYLE:
		GTK_VALUE_ENUM (*arg) = bp->join;
		break;

	case ARG_FONT:
		GTK_VALUE_POINTER (*arg) = bp->priv->font;
		break;

	case ARG_SIZE:
		GTK_VALUE_DOUBLE (*arg) = bp->size;
		break;

	case ARG_X:
		GTK_VALUE_DOUBLE (*arg) = bp->x;
		break;

	case ARG_Y:
		GTK_VALUE_DOUBLE (*arg) = bp->y;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

#if 0
static void
gnome_canvas_hacktext_render(GnomeCanvasItem *item,
			     GnomeCanvasBuf *buf)
{
	GnomeCanvasHacktext *hacktext;
	guint32 fg_color, bg_color;

	hacktext = GNOME_CANVAS_HACKTEXT (item);

	if (hacktext->fill_svp != NULL)
		gnome_canvas_render_svp (buf, hacktext->fill_svp, hacktext->fill_rgba);

	if (hacktext->outline_svp != NULL)
		gnome_canvas_render_svp (buf, hacktext->outline_svp, hacktext->outline_rgba);
}
#endif

static void
gnome_canvas_hacktext_update (GnomeCanvasItem *item, double *affine, ArtSVP *clip_path, int flags)
{
	GnomeCanvasHacktext *hacktext;
	double x1, y1, x2, y2;
	ArtIRect ibbox = {0, 0, 0, 0};

	hacktext = GNOME_CANVAS_HACKTEXT (item);

	if (parent_class->update)
		(* parent_class->update) (item, affine, clip_path, flags);

	if (item->canvas->aa) {
		gnome_canvas_hacktext_req_repaint (hacktext, &ibbox);
		if (!art_affine_equal (hacktext->priv->affine, affine)) {
			memcpy (hacktext->priv->affine, affine, 6 * sizeof(double));
			gnome_canvas_hacktext_req_repaint (hacktext, &ibbox);
		}
		hacktext->item.x1 = ibbox.x0;
		hacktext->item.y1 = ibbox.y0;
		hacktext->item.x2 = ibbox.x1;
		hacktext->item.y2 = ibbox.y1;
	} else {
		set_outline_gc_width (hacktext);
		set_gc_foreground (hacktext->fill_gc, hacktext->fill_pixel);
		set_gc_foreground (hacktext->outline_gc, hacktext->outline_pixel);
		set_stipple (hacktext->fill_gc, &hacktext->fill_stipple, hacktext->fill_stipple, TRUE);
		set_stipple (hacktext->outline_gc, &hacktext->outline_stipple, hacktext->outline_stipple, TRUE);

		get_bounds_canvas (hacktext, &x1, &y1, &x2, &y2, affine);
		gnome_canvas_update_bbox (item, x1, y1, x2, y2);		
	}
}

static void
gnome_canvas_hacktext_realize (GnomeCanvasItem *item)
{
	GnomeCanvasHacktext *hacktext;

	hacktext = GNOME_CANVAS_HACKTEXT (item);

	if (parent_class->realize)
		(* parent_class->realize) (item);

	hacktext->fill_gc = gdk_gc_new (item->canvas->layout.bin_window);
	hacktext->outline_gc = gdk_gc_new (item->canvas->layout.bin_window);
}

static void
gnome_canvas_hacktext_unrealize (GnomeCanvasItem *item)
{
	GnomeCanvasHacktext *hacktext;

	hacktext = GNOME_CANVAS_HACKTEXT (item);

	gdk_gc_unref (hacktext->fill_gc);
	gdk_gc_unref (hacktext->outline_gc);

	if (parent_class->unrealize)
		(* parent_class->unrealize) (item);
}

/* Converts an array of world coordinates into an array of canvas pixel coordinates.  Takes in the
 * item->world deltas and the drawable deltas.
 */
static void
item_to_canvas (GnomeCanvas *canvas, double *item_coords, GdkPoint *canvas_coords, int num_points,
		double i2c[6])
{
	int i;
	ArtPoint pi, pc;

#ifdef VERBOSE
	{
		char str[128];
		art_affine_to_string (str, i2c);
		g_print ("hacktext item_to_canvas %s\n", str);
	}
#endif

	for (i = 0; i < num_points; i++) {
		pi.x = item_coords[i * 2];
		pi.y = item_coords[i * 2 + 1];
		art_affine_point (&pc, &pi, i2c);
		canvas_coords->x = floor (pc.x + 0.5);
		canvas_coords->y = floor (pc.y + 0.5);
		canvas_coords++; 
	}
}

static void
gnome_canvas_hacktext_draw (GnomeCanvasItem *item, GdkDrawable *drawable,
			   int x, int y, int width, int height)
{
#if 0
	/* TODO: implement this */
	GnomeCanvasHacktext *hacktext;
	GdkPoint static_points[NUM_STATIC_POINTS];
	GdkPoint *points;
	double dx, dy;
	int cx, cy;
	int i;
	double i2c[6];

	hacktext = GNOME_CANVAS_HACKTEXT (item);

	if (hacktext->num_points == 0)
		return;

	/* Build array of canvas pixel coordinates */

	if (hacktext->num_points <= NUM_STATIC_POINTS)
		points = static_points;
	else
		points = g_new (GdkPoint, hacktext->num_points);

	gnome_canvas_item_i2c_affine (item, i2c);

	i2c[4] -= x;
	i2c[5] -= y;

	item_to_canvas (item->canvas, hacktext->coords, points, hacktext->num_points, i2c);

	if (hacktext->fill_set) {
		if (hacktext->fill_stipple)
			gnome_canvas_set_stipple_origin (item->canvas, hacktext->fill_gc);

		gdk_draw_hacktext (drawable, hacktext->fill_gc, TRUE, points, hacktext->num_points);
	}

	if (hacktext->outline_set) {
		if (hacktext->outline_stipple)
			gnome_canvas_set_stipple_origin (item->canvas, hacktext->outline_gc);

		gdk_draw_hacktext (drawable, hacktext->outline_gc, FALSE, points, hacktext->num_points);
	}

	/* Done */

	if (points != static_points)
		g_free (points);
#endif
}

static double
gnome_canvas_hacktext_point (GnomeCanvasItem *item, double mx, double my,
			    int cx, int cy, GnomeCanvasItem **actual_item)
{
	double dist;
	int wind;

	GnomeCanvasHacktext *hacktext;
	GnomeCanvasHacktextCharInfo *info;
	double x, y, width;
	const unsigned char *text;
	int i;
	ArtBpath *affine_bpath;
	ArtVpath *vpath;
	ArtSVP *svp;
	double size_affine[6], affine[6];

	hacktext = GNOME_CANVAS_HACKTEXT (item);
	x = hacktext->x;
	y = hacktext->y;
	text = (const unsigned char *)hacktext->text;
	if (text == NULL)
		return 100.0;

	/* If we get the empty string, this default will be used.  FIXME: check
	   sanity.  */
	dist = 100.0;

	art_affine_scale (size_affine, hacktext->size * 0.001, hacktext->size * -0.001);
	art_affine_multiply (affine, size_affine, hacktext->priv->affine);
	affine[4] += x * hacktext->priv->affine[0] + y * hacktext->priv->affine[2];
	affine[5] += x * hacktext->priv->affine[1] + y * hacktext->priv->affine[3];
	for (i = 0; text[i]; i++) {
		info = gnome_canvas_hacktext_get_info (hacktext, text[i]);
		/* it intersects with the draw rect - draw the glyph */
		affine_bpath = art_bpath_affine_transform (info->bpath, affine);
		vpath = art_bez_path_to_vec (affine_bpath, 0.25);
		art_free (affine_bpath);
		if (hacktext->fill_set) {
		  svp = art_svp_from_vpath (vpath);
		  wind = art_svp_point_wind (svp, cx, cy);
		  if (wind) {
		    *actual_item = item;
		    return 0.0;
		  }
		  dist = art_svp_point_dist (svp, cx, cy);
		  art_svp_free (svp);
		}
		if (hacktext->outline_set) {
		  if (hacktext->width_pixels)
		    width = hacktext->width;
		  else
		    width = hacktext->width * item->canvas->pixels_per_unit;
		  
		  if (width < 0.5)
		    width = 0.5;
		  
		  svp = art_svp_vpath_stroke (vpath,
					      gnome_canvas_join_gdk_to_art (hacktext->join),
					      gnome_canvas_cap_gdk_to_art (hacktext->cap),
					      width,
					      4,
					      0.25);
		  wind = art_svp_point_wind (svp, cx, cy);
		  if (wind) {
		    *actual_item = item;
		    return 0.0;
		  }
		  
		  dist = art_svp_point_dist (svp, cx, cy);
		  art_svp_free (svp);
		}
		
		art_free (vpath);
		affine[4] += info->wx * affine[0];
		affine[5] += info->wx * affine[1];
	}

	if (!hacktext->outline_set && !hacktext->fill_set) 
		return 1e12;
			
#ifdef VERBOSE
	g_print ("dist = %g\n", dist);
#endif
	/* should dist be scaled by zoom somehow? */
	*actual_item = item;
	
	return dist;
}

static void
gnome_canvas_hacktext_bounds (GnomeCanvasItem *item, double *x1, double *y1, double *x2, double *y2)
{
	GnomeCanvasHacktext *hacktext;

	g_return_if_fail (item != NULL);
	g_return_if_fail (GNOME_IS_CANVAS_HACKTEXT (item));

	hacktext = GNOME_CANVAS_HACKTEXT (item);

	if (hacktext->text == NULL) {
		*x1 = *y1 = *x2 = *y2 = 0.0;
		return;
	}

	get_bounds (hacktext, x1, y1, x2, y2);
}

/* Find the bounding box of a bezier path. */
static void
art_drect_bpath (ArtDRect *bbox, const ArtBpath *bpath)
{
	/* This implementation is a hack. The real thing needs to go
	   into libart. */
	ArtVpath *vpath;

	vpath = art_bez_path_to_vec (bpath, 0.25);
	art_vpath_bbox_drect (vpath, bbox);
	art_free (vpath);
}

/* Get the info for a character, loading it if necessary */
static GnomeCanvasHacktextCharInfo *
gnome_canvas_hacktext_get_info (GnomeCanvasHacktext *hacktext, gint32 ch)
{
	Gt1LoadedFont *lf;
	double wx;
	ArtBpath *bpath;

	if (hacktext->priv->info[ch].bpath == NULL) {
		lf = hacktext->priv->font;
		bpath = gt1_get_glyph_outline (lf, ch, &wx);

		hacktext->priv->info[ch].bpath = bpath;
		hacktext->priv->info[ch].wx = wx;
		art_drect_bpath (&hacktext->priv->info[ch].bbox, bpath);
	}

	return &hacktext->priv->info[ch];
}

static void
gnome_canvas_hacktext_req_repaint (GnomeCanvasHacktext *hacktext,
				   ArtIRect *bbox)
{
	GnomeCanvasHacktextCharInfo *info;
	double x, y;
	const unsigned char *text;
	int i;
	ArtDRect w_bbox;
	ArtIRect wbi;
	double size_affine[6], affine[6];

	x = hacktext->x;
	y = hacktext->y;
	text = (const unsigned char *)hacktext->text;
	if (text == NULL)
		return;

	art_affine_scale (size_affine, hacktext->size * 0.001, hacktext->size * -0.001);
	art_affine_multiply (affine, size_affine, hacktext->priv->affine);
	affine[4] += x * hacktext->priv->affine[0] + y * hacktext->priv->affine[2];
	affine[5] += x * hacktext->priv->affine[1] + y * hacktext->priv->affine[3];
	for (i = 0; text[i]; i++) {
		info = gnome_canvas_hacktext_get_info (hacktext, text[i]);
		art_drect_affine_transform (&w_bbox, &info->bbox, affine);
		art_drect_to_irect (&wbi, &w_bbox);
		gnome_canvas_request_redraw (hacktext->item.canvas,
					     wbi.x0 - 2, wbi.y0 - 2,
					     wbi.x1 + 2 , wbi.y1 + 2);
		if (bbox)
			art_irect_union (bbox, bbox, &wbi);

		affine[4] += info->wx * affine[0];
		affine[5] += info->wx * affine[1];
	}
}

static void
gnome_canvas_hacktext_render (GnomeCanvasItem *item,
			      GnomeCanvasBuf *buf)
{
	GnomeCanvasHacktext *hacktext;
	GnomeCanvasHacktextCharInfo *info;
	double x, y, width;
	const unsigned char *text;
	int i;
	ArtDRect w_bbox;
	ArtIRect wi_bbox, int_rect;
	ArtBpath *affine_bpath;
	ArtVpath *vpath;
	ArtSVP *svp;
	double size_affine[6], affine[6];
	int j;
	ArtVpathDash *dash = NULL;

#ifdef USE_DASH_HACK
	double dots[8] = {12, 3, 6, 3, 24, 3, 6, 3};
	dash = g_new0 (ArtVpathDash, 1);
	dash->offset = 0;
	dash->n_dash = 8;
	dash->dash = dots;
#endif

	for (j = 0; dash && j < dash->n_dash; j++)
	  dash->dash[j] *= item->canvas->pixels_per_unit;

	hacktext = GNOME_CANVAS_HACKTEXT (item);
	x = hacktext->x;
	y = hacktext->y;
	text = (const unsigned char *)hacktext->text;
	if (text == NULL)
		return;

#ifdef VERBOSE
	g_print ("buf->rect = (%d, %d) - (%d, %d)\n", buf->rect.x0, buf->rect.y0, buf->rect, buf->rect.y1);
#endif
	art_affine_scale (size_affine, hacktext->size * 0.001, hacktext->size * -0.001);
	art_affine_multiply (affine, size_affine, hacktext->priv->affine);
	affine[4] += x * hacktext->priv->affine[0] + y * hacktext->priv->affine[2];
	affine[5] += x * hacktext->priv->affine[1] + y * hacktext->priv->affine[3];
	for (i = 0; text[i]; i++) {
#ifdef VERBOSE      
		g_print ("character %c, affine = [%f %f %f %f %f %f]\n", text[i],
			 affine[0], affine[1], affine[2], affine[3], affine[4], affine[5]);
#endif
		info = gnome_canvas_hacktext_get_info (hacktext, text[i]);
		art_drect_affine_transform (&w_bbox, &info->bbox, affine);
		art_drect_to_irect (&wi_bbox, &w_bbox);
#ifdef VERBOSE
		g_print ("wi_bbox = (%d, %d) - (%d, %d)\n", wi_bbox.x0, wi_bbox.y0, wi_bbox.x1, wi_bbox.y1);
#endif
		art_irect_intersect (&int_rect, &wi_bbox, &buf->rect);
		if (!art_irect_empty (&int_rect)) {
			/* it intersects with the draw rect - draw the glyph */
			affine_bpath = art_bpath_affine_transform (info->bpath, affine);
			vpath = art_bez_path_to_vec (affine_bpath, 0.25);
			art_free (affine_bpath);
			if (hacktext->fill_set) {
			    svp = art_svp_from_vpath (vpath);
			    gnome_canvas_render_svp (buf, svp, hacktext->fill_rgba);
			    art_svp_free (svp);
			}
			if (hacktext->outline_set) {
			    if (hacktext->width_pixels)
				width = hacktext->width;
			    else
				width = hacktext->width * item->canvas->pixels_per_unit;
			    
			    if (width < 0.5)
				width = 0.5;

			    if (dash)
			      {
				ArtVpath *old = vpath;
						    
				vpath = art_vpath_dash (vpath, dash);
				art_free (old);
			      }
			    svp = art_svp_vpath_stroke (vpath,
							gnome_canvas_join_gdk_to_art (hacktext->join),
							gnome_canvas_cap_gdk_to_art (hacktext->cap),
							width,
							4,
							0.25);
			    gnome_canvas_render_svp (buf, svp, hacktext->outline_rgba);
			    art_svp_free (svp);
			}
			art_free (vpath);

		}

		affine[4] += info->wx * affine[0];
		affine[5] += info->wx * affine[1];
	}
}













