/*
 * Metafile printer driver.
 *
 * Author:
 *    Miguel de Icaza (miguel@gnu.org
 *
 * TODO:
 *    If the GnomeFont is not found during the replay of a metafile
 *    we should load any font at the specified size.
 *
 *    We could keep a cache of all the fonts referenced during a
 *    playback (instead of using gnome_font_new always) and unref
 *    all the fonts when the render is done.
 *
 *    Handle GNOME_META_TEXTLINE
 */
#include <gtk/gtk.h>
#include <string.h>
#include <math.h>

#include "config.h"
#include "gnome-print-meta.h"

static GnomePrintContextClass *parent_class = NULL;

#define GNOME_METAFILE_SIGNATURE "GNOME_METAFILE-1.0"
#define BLOCKSIZE 4096

typedef struct {
	char signature [sizeof (GNOME_METAFILE_SIGNATURE)];
	int  size;
} GnomeMetaFileHeader;

typedef enum {
	GNOME_META_NEWPATH,
	GNOME_META_MOVETO,
	GNOME_META_LINETO,
	GNOME_META_CURVETO,
	GNOME_META_CLOSEPATH,
	GNOME_META_SETRGBCOLOR,
	GNOME_META_FILL,
	GNOME_META_EOFILL,
	GNOME_META_SETLINEWIDTH,
	GNOME_META_SETMITERLIMIT,
	GNOME_META_SETLINEJOIN,
	GNOME_META_SETLINECAP,
	GNOME_META_SETDASH,
	GNOME_META_STROKEPATH,
	GNOME_META_STROKE,
	GNOME_META_SETFONT,
	GNOME_META_SHOW,
	GNOME_META_CONCAT,
	GNOME_META_SETMATRIX,
	GNOME_META_GSAVE,
	GNOME_META_GRESTORE,
	GNOME_META_CLIP,
	GNOME_META_EOCLIP,
	GNOME_META_GRAYIMAGE,
	GNOME_META_RGBIMAGE,
	GNOME_META_SHOWPAGE,
	GNOME_META_TEXTLINE,
	GNOME_META_CLOSE
} GnomeMetaType;
	
static void
gnome_print_meta_finalize (GtkObject *object)
{
	GnomePrintMeta *meta;

	meta = GNOME_PRINT_META (object);

	if (meta->buffer)
		g_free (meta->buffer);
				      
	(*GTK_OBJECT_CLASS (parent_class)->finalize) (object);
}

static void
encode_block (GnomePrintMeta *meta, int size, void *data)
{
	int bytes_left;

	bytes_left = meta->buffer_size - meta->current;

	if (bytes_left < size){
		int min_size = MAX (BLOCKSIZE, size-bytes_left);
		void *newp;
		
		newp = g_realloc (meta->buffer, meta->buffer_size + min_size);
		if (!newp)
			return;

		meta->buffer = newp;
		meta->buffer_size = meta->buffer_size + min_size;
	}

	memcpy (meta->buffer + meta->current, data, size);
	meta->current += size;
}

static void
encode_int (GnomePrintContext *pc, gint32 value)
{
	gint32 new = g_htonl (value);
	
	encode_block (GNOME_PRINT_META (pc), sizeof (gint32), &new);
}

static void
encode_double (GnomePrintContext *pc, double d)
{
	static int message_shown;

	if (!message_shown){
		g_warning ("We lack double marshalling\n");
		message_shown = 1;
	}
	encode_block (GNOME_PRINT_META (pc), sizeof (double), &d); 
}

static void
encode_six_doubles (GnomePrintContext *pc, double x1, double y1, double x2, double y2, double x3, double y3)
{
	encode_double (pc, x1);
	encode_double (pc, y1);
	encode_double (pc, x2);
	encode_double (pc, y2);
	encode_double (pc, x3);
	encode_double (pc, y3);
}

static int
meta_newpath (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_NEWPATH);
	return 0;
}

static int
meta_moveto (GnomePrintContext *pc, double x, double y)
{
	encode_int (pc, GNOME_META_MOVETO);
	encode_double (pc, x);
	encode_double (pc, y);
	return 0;
}

static int
meta_lineto (GnomePrintContext *pc, double x, double y)
{
	encode_int (pc, GNOME_META_LINETO);
	encode_double (pc, x);
	encode_double (pc, y);
	return 0;
}

static int
meta_curveto (GnomePrintContext *pc, double x1, double y1, double x2, double y2, double x3, double y3)
{
	encode_int (pc, GNOME_META_CURVETO);
	encode_six_doubles (pc, x1, y1, x2, y2, x3, y3);
	return 0;
}

static int
meta_closepath (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_CLOSEPATH);
	return 0;
}

static int
meta_setrgbcolor (GnomePrintContext *pc, double r, double g, double b)
{
	encode_int (pc, GNOME_META_SETRGBCOLOR);
	encode_double (pc, r);
	encode_double (pc, g);
	encode_double (pc, b);
	return 0;
}

static int
meta_fill (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_FILL);
	return 0;
}

static int
meta_eofill (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_EOFILL);
	return 0;
}

static int
meta_setlinewidth (GnomePrintContext *pc, double width)
{
	encode_int (pc, GNOME_META_SETLINEWIDTH);
	encode_double (pc, width);
	return 0;
}

static int
meta_setmiterlimit (GnomePrintContext *pc, double limit)
{
	encode_int (pc, GNOME_META_SETMITERLIMIT);
	encode_double (pc, limit);
	return 0;
}

static int
meta_setlinejoin (GnomePrintContext *pc, int jointype)
{
	encode_int (pc, GNOME_META_SETLINEJOIN);
	encode_int (pc, jointype);
	return 0;
}

static int
meta_setlinecap (GnomePrintContext *pc, int captype)
{
	encode_int (pc, GNOME_META_SETLINECAP);
	encode_int (pc, captype);
	return 0;
}

static int
meta_setdash (GnomePrintContext *pc, int n_values, double *values, double offset)
{
	int i;
	
	encode_int (pc, GNOME_META_SETDASH);
	encode_int (pc, n_values);

	for (i = 0; i < n_values; i++)
		encode_double (pc, values [i]);
	encode_double (pc, offset);
	return 0;
}

static int
meta_strokepath (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_STROKEPATH);
	return 0;
}

static int
meta_stroke (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_STROKE);
	return 0;
}

static int
meta_setfont (GnomePrintContext *pc, GnomeFont *font)
{
	char *fontname = font->fontmap_entry->font_name;
	int  len = strlen (fontname);
			  
	encode_int (pc, GNOME_META_SETFONT);
	encode_double (pc, font->size);
	encode_int (pc, len);
	encode_block (GNOME_PRINT_META (pc), len, fontname);
	
	return 0;
}


static int
meta_show (GnomePrintContext *pc, char *text)
{
	int len = strlen (text);
	
	encode_int (pc, GNOME_META_SHOW);
	encode_int (pc, len);
	encode_block (GNOME_PRINT_META (pc), len, text);
	return 0;
}

static int
meta_concat (GnomePrintContext *pc, double matrix [6])
{
	encode_int (pc, GNOME_META_CONCAT);
	encode_six_doubles (pc, matrix [0], matrix [1], matrix [2], matrix [3], matrix [4], matrix [5]);
	return 0;
}

static int
meta_setmatrix (GnomePrintContext *pc, double matrix [6])
{
	encode_int (pc, GNOME_META_SETMATRIX);
	encode_six_doubles (pc, matrix [0], matrix [1], matrix [2], matrix [3], matrix [4], matrix [5]);
	return 0;
}

static int
meta_gsave (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_GSAVE);
	return 0;
}

static int
meta_grestore (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_GRESTORE);
	return 0;
}

static int
meta_clip (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_CLIP);
	return 0;
}

static int
meta_eoclip (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_EOCLIP);
	return 0;
}

static void
encode_image (GnomePrintContext *pc, char *data, int width, int height, int rowstride, int bytes_per_pixel)
{
	int ix, y;
	
	encode_int (pc, height);
	encode_int (pc, width);

	ix = 0;
	for (y = 0; y < height; y++){
		encode_block (GNOME_PRINT_META (pc), width * bytes_per_pixel, &data [ix]);
		ix += rowstride;
	}
}

static int
meta_grayimage (GnomePrintContext *pc, char *data, int width, int height, int rowstride)
{
	encode_int (pc, GNOME_META_GRAYIMAGE);
	encode_image (pc, data, width, height, rowstride, 1);
	return 0;
}

static int
meta_rgbimage (GnomePrintContext *pc, char *data, int width, int height, int rowstride)
{
	encode_int (pc, GNOME_META_RGBIMAGE);
	encode_image (pc, data, width, height, rowstride, 3);
	return 0;
}

static int
meta_textline (GnomePrintContext *pc, GnomeTextLine *line)
{
	encode_int (pc, GNOME_META_TEXTLINE);
	g_warning ("meta_textline: not implemented yet\n");
	return 0;
}

static int
meta_showpage (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_SHOWPAGE);
	return 0;
}

static int
meta_close (GnomePrintContext *pc)
{
	encode_int (pc, GNOME_META_CLOSE);
	return 0;
}

static void
gnome_print_meta_class_init (GnomePrintMetaClass *class)
{
	GtkObjectClass *object_class;
	GnomePrintContextClass *pc_class;
	
	object_class = (GtkObjectClass *)class;
	pc_class = (GnomePrintContextClass *)class;
	
	parent_class = gtk_type_class (gnome_print_context_get_type ());
	
	object_class->finalize = gnome_print_meta_finalize;
	
	/* initialization code, autogenned */
	pc_class->newpath = meta_newpath;
	pc_class->moveto = meta_moveto;
	pc_class->lineto = meta_lineto;
	pc_class->curveto = meta_curveto;
	pc_class->closepath = meta_closepath;
	pc_class->setrgbcolor = meta_setrgbcolor;
	pc_class->fill = meta_fill;
	pc_class->eofill = meta_eofill;
	pc_class->setlinewidth = meta_setlinewidth;
	pc_class->setmiterlimit = meta_setmiterlimit;
	pc_class->setlinejoin = meta_setlinejoin;
	pc_class->setlinecap = meta_setlinecap;
	pc_class->setdash = meta_setdash;
	pc_class->strokepath = meta_strokepath;
	pc_class->stroke = meta_stroke;
	pc_class->setfont = meta_setfont;
	pc_class->show = meta_show;
	pc_class->concat = meta_concat;
	pc_class->setmatrix = meta_setmatrix;
	pc_class->gsave = meta_gsave;
	pc_class->grestore = meta_grestore;
	pc_class->clip = meta_clip;
	pc_class->eoclip = meta_eoclip;
	pc_class->grayimage = meta_grayimage;
	pc_class->rgbimage = meta_rgbimage;
	pc_class->textline = meta_textline;
	pc_class->showpage = meta_showpage;
	
	pc_class->close = meta_close;
}

static void
gnome_print_meta_init (GnomePrintMeta *meta)
{
	meta->buffer_size = BLOCKSIZE;
	meta->buffer = g_malloc (meta->buffer_size);

	strcpy (meta->buffer, GNOME_METAFILE_SIGNATURE);
	meta->current = sizeof (GnomeMetaFileHeader);
}

/**
 * gnome_print_meta_get_type:
 *
 * Returns the GtkType for the GnomePrintMeta object
 */
GtkType
gnome_print_meta_get_type (void)
{
	static GtkType meta_type = 0;
	
	if (!meta_type){
		GtkTypeInfo meta_info =
		{
			"GnomePrintMeta",
			sizeof (GnomePrintMeta),
			sizeof (GnomePrintMetaClass),
			(GtkClassInitFunc) gnome_print_meta_class_init,
			(GtkObjectInitFunc) gnome_print_meta_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};
		
		meta_type = gtk_type_unique (gnome_print_context_get_type (), &meta_info);
	}
	
	return meta_type;
}

/**
 * gnome_print_meta_new:
 *
 * Creates a new Metafile context GnomePrint object.
 */
GnomePrintMeta *
gnome_print_meta_new (void)
{
	GnomePrintMeta *meta;

	meta = gtk_type_new (gnome_print_meta_get_type ());

	return meta;
}

/**
 * gnome_print_meta_new:
 * @data: Pointer to a gnome_print_meta metadata.
 *
 * Creates a new Metafile context GnomePrint object.
 *
 * Initializes the contents from a buffer that contains a
 * GNOME_METAFILE stream.
 */
GnomePrintMeta *
gnome_print_meta_new_from (void *data)
{
	GnomePrintMeta *meta;
	GnomeMetaFileHeader *mf = data;
	int len;

	g_return_val_if_fail (data != NULL, NULL);

	if (strcmp (data, GNOME_METAFILE_SIGNATURE) != 0)
		return NULL;

	len = mf->size;
	meta = gtk_type_new (gnome_print_meta_get_type ());

	if (meta->buffer_size < len){
		g_free (meta->buffer);
		meta->buffer = g_malloc (len);

		if (!meta->buffer){
			gtk_object_destroy (GTK_OBJECT (meta));
			return NULL;
		}
	}
	memcpy (meta->buffer, data, len);
	meta->current = len;
	return meta;
}

/**
 * gnome_print_meta_access_buffer:
 * @buffer: points to a void * that will be modified to
 *          point to the new buffer
 * @buflen: pointer to an integer that will be set to the lenght of buffer.
 *
 * Makes buffer point to the internal GnomePrintMeta information. 
 *
 * Returns 0 on a failure, any other value on success
 */
int
gnome_print_meta_access_buffer (GnomePrintMeta *meta, void **buffer, int *buflen)
{
	g_return_val_if_fail (meta != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_META (meta), 0);

	((GnomeMetaFileHeader *)meta->buffer)->size = meta->buffer_size;
	*buffer = meta->buffer;
	*buflen = meta->buffer_size;

	return 1;
}

/**
 * gnome_print_meta_get_copy:
 * @buffer: points to a void * that will be modified to
 *          point to the new buffer
 * @buflen: pointer to an integer that will be set to the lenght of buffer.
 *
 * Duplicates the internal buffer and stores a pointer to the block with
 * a copy of this in *@buffer.  The size of the buffer is stored in @buflen.
 *
 * Returns 0 on a failure, any other value on success
 */
int
gnome_print_meta_get_copy (GnomePrintMeta *meta, void **buffer, int *buflen)
{
	g_return_val_if_fail (meta != NULL, 0);
	g_return_val_if_fail (GNOME_IS_PRINT_META (meta), 0);

	*buffer = g_malloc (meta->buffer_size);
	if (*buffer == NULL)
		return 0;

	memcpy (*buffer, meta->buffer, meta->buffer_size);
	((GnomeMetaFileHeader *)*buffer)->size = meta->buffer_size;
	*buflen = meta->buffer_size;

	return 1;
}

static char *
decode_int (char *data, gint32 *dest)
{
	*dest = g_ntohl (*(int *) data);
	return data + sizeof (gint32);
}

static char *
decode_double (char *data, double *dest)
{
	*dest = *(double *) data;
	return data + sizeof (double);
}

static char *
decode_image (char *data, GnomePrintContext *dest, int bytes_per_pixel)
{
	void *buffer;
	gint32 height, width;
	int size;
	
	data = decode_int (data, &height);
	data = decode_int (data, &width);

	size = height * width * bytes_per_pixel;
	buffer = g_malloc (size);
	memcpy (buffer, data, size);

	if (bytes_per_pixel == 1)
		gnome_print_grayimage (dest, buffer, width, height, width);
	else
		gnome_print_rgbimage (dest, buffer, width, height, width * bytes_per_pixel);

	g_free (buffer);
	return data + size;
}

static gboolean
do_render (GnomePrintContext *dest, char *data, int size)
{
	char *end = data + size;
	double matrix [6];
	double x1, y1, x2, y2, x3, y3;
	double x, y;
	double r, g, b;
	
	while (data < end){
		gint32 opcode, i;
		
		data = decode_int (data, &opcode);
		switch ((GnomeMetaType) opcode){
		case GNOME_META_NEWPATH:
			gnome_print_newpath (dest);
			break;
			
		case GNOME_META_MOVETO:
			data = decode_double (data, &x);
			data = decode_double (data, &y);
			gnome_print_moveto (dest, x, y);
			break;

		case GNOME_META_LINETO:
			data = decode_double (data, &x);
			data = decode_double (data, &y);
			gnome_print_lineto (dest, x, y);
			break;
			
		case GNOME_META_CURVETO:
			data = decode_double (data, &x1);
			data = decode_double (data, &y1);
			data = decode_double (data, &x2);
			data = decode_double (data, &y2);
			data = decode_double (data, &x3);
			data = decode_double (data, &y3);
			gnome_print_curveto (dest, x1, y1, x2, y2, x3, y3);
			break;
			
		case GNOME_META_CLOSEPATH:
			gnome_print_closepath (dest);
			break;

		case GNOME_META_SETRGBCOLOR:
			data = decode_double (data, &r);
			data = decode_double (data, &g);
			data = decode_double (data, &b);
			gnome_print_setrgbcolor (dest, r, g, b);
			break;
			
		case GNOME_META_FILL:
			gnome_print_fill (dest);
			break;
			
		case GNOME_META_EOFILL:
			gnome_print_eofill (dest);
			break;
			
		case GNOME_META_SETLINEWIDTH:
			data = decode_double (data, &x);
			gnome_print_setlinewidth (dest, x);
			break;
			
		case GNOME_META_SETMITERLIMIT:
			data = decode_double (data, &x);
			gnome_print_setmiterlimit (dest, x);
			break;
			
		case GNOME_META_SETLINEJOIN:
			data = decode_int (data, &i);
			gnome_print_setlinejoin (dest, i);
			break;
			
		case GNOME_META_SETLINECAP:
			data = decode_int (data, &i);
			gnome_print_setlinecap (dest, i);
			break;

		case GNOME_META_SETDASH: {
			int n;
			double *values, offset;

			data = decode_int (data, &n);
			values = g_new (double, n);
			for (i = 0; i < n; i++)
				data = decode_double (data, &values [i]);
			data = decode_double (data, &offset);
			gnome_print_setdash (dest, n, values, offset);
			g_free (values);
			break;
		}
		
		case GNOME_META_STROKEPATH:
			gnome_print_strokepath (dest);
			break;
			
		case GNOME_META_STROKE:
			gnome_print_stroke (dest);
			break;
			
		case GNOME_META_SETFONT: {
			GnomeFont *font;
			char *name;
			
			data = decode_double (data, &x);
			data = decode_int (data, &i);
			name = g_malloc (i + 1);
			memcpy (name, data, i);
			name [i] = 0;
			data += i;

			font = gnome_font_new (name, x);
			g_free (name);

			if (font){
				gnome_print_setfont (dest, font);
				gtk_object_unref (GTK_OBJECT (font));
			}
				
			g_warning ("GNOME_META_SETFONT: No fallback code defined\n");
			break;
		}
			
		case GNOME_META_SHOW: {
			char *buffer;
			
			data = decode_int (data, &i);
			buffer = g_malloc (i + 1);
			memcpy (buffer, data, i);
			buffer [i] = 0;
			data += i;
			gnome_print_show (dest, buffer);
			g_free (buffer);
			break;
		}
		
		case GNOME_META_CONCAT:
		case GNOME_META_SETMATRIX:
			data = decode_double (data, &matrix [0]);
			data = decode_double (data, &matrix [1]);
			data = decode_double (data, &matrix [2]);
			data = decode_double (data, &matrix [3]);
			data = decode_double (data, &matrix [4]);
			data = decode_double (data, &matrix [5]);
			if (opcode == GNOME_META_CONCAT)
				gnome_print_concat (dest, matrix);
			else
				gnome_print_setmatrix (dest, matrix);
			break;
			
		case GNOME_META_GSAVE:
			gnome_print_gsave (dest);
			break;
			
		case GNOME_META_GRESTORE:
			gnome_print_grestore (dest);
			break;
			
		case GNOME_META_CLIP:
			gnome_print_clip (dest);
			break;
			
		case GNOME_META_EOCLIP:
			gnome_print_eoclip (dest);
			break;
			
		case GNOME_META_GRAYIMAGE:
			data = decode_image (data, dest, 1);
			break;
			
		case GNOME_META_RGBIMAGE:
			data = decode_image (data, dest, 3);
			break;
			
		case GNOME_META_SHOWPAGE:
			gnome_print_showpage (dest);
			break;

		case GNOME_META_TEXTLINE:
			g_warning ("FIXME: Text line missing");
			break;
			
		case GNOME_META_CLOSE:
			return TRUE;
		}
	}
	return TRUE;
}

/**
 * gnome_print_meta_render:
 * @destination: Destination printer context.
 * @meta_stream: a metadata stream to replay
 *
 * Plays the @meta_steam metadata stream into the @destination printer.
 *
 * Returns TRUE on success.
 */
gboolean
gnome_print_meta_render (GnomePrintContext *destination, void *meta_stream)
{
	char *data;
	int size;
	
	g_return_val_if_fail (destination != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (destination), FALSE);
	g_return_val_if_fail (meta_stream != NULL, FALSE);

	if (strcmp (meta_stream, GNOME_METAFILE_SIGNATURE) != 0)
		return FALSE;

	data = meta_stream;
	size = ((GnomeMetaFileHeader *)meta_stream)->size;
	return do_render (destination, data + sizeof (GnomeMetaFileHeader), size);
}

/**
 * gnome_print_meta_render_from_object
 * @destination: Destination printer context.
 * @source: an existing GnomePrintMeta printer context.
 *
 * Plays the contents of the GnomePrintMeta @source metadata stream
 * into the @destination printer.
 *
 * Returns TRUE on success.
 */
gboolean
gnome_print_meta_render_from_object (GnomePrintContext *destination, GnomePrintMeta *source)
{
	g_return_val_if_fail (destination != NULL, FALSE);
	g_return_val_if_fail (source != NULL, FALSE);
	g_return_val_if_fail (GNOME_IS_PRINT_CONTEXT (destination), FALSE);
	g_return_val_if_fail (GNOME_IS_PRINT_META (source), FALSE);

	return gnome_print_meta_render (destination, source->buffer);
}

