/* font-dialog.c - functions that do things to fonts.
 *
 * Copyright (C) 1998 Chris Lahey.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "config.h"
#include "gnome-font-dialog.h"
#include <gnome.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtklistitem.h>
#include <gtk/gtkframe.h>
#include <libgnomeui/gnome-stock.h>

/* These are what we use as the standard font sizes, for the size clist.
   Note that when using points we still show these integer point values but
   we work internally in decipoints (and decipoint values can be typed in). */
const static char *font_sizes[] = {
  "8", "9", "10", "11", "12", "13", "14", "16", "18", "20", "22", "24", "26", "28",
  "32", "36", "40", "48", "56", "64", "72"
};

typedef struct _GnomeFontWeightTab GnomeFontWeightTab;

struct _GnomeFontWeightTab {
  const char *name;
  int code;
};

static const GnomeFontWeightTab weight_tab[] = {
  { "Extra Light", GNOME_FONT_EXTRA_LIGHT },

  { "Thin", GNOME_FONT_THIN },

  { "Light", GNOME_FONT_LIGHT },

  { "Regular", GNOME_FONT_BOOK },

  { "Medium", GNOME_FONT_MEDIUM },

  { "Semibold", GNOME_FONT_SEMI },

  { "Bold", GNOME_FONT_BOLD },

  { "Heavy", GNOME_FONT_HEAVY },
 
  { "Extra Bold", GNOME_FONT_EXTRABOLD },

  { "Black", GNOME_FONT_BLACK },

  { "Extra Black", GNOME_FONT_EXTRABLACK }
};

static void    gnome_font_selection_class_init	     (GnomeFontSelectionClass *klass);
static void    gnome_font_selection_init		     (GnomeFontSelection *fontsel);
static void    gnome_font_selection_destroy	     (GtkObject      *object);

/* FontSelectionDialog */
static void    gnome_font_selection_dialog_class_init  (GnomeFontSelectionDialogClass *klass);
static void    gnome_font_selection_dialog_init	     (GnomeFontSelectionDialog *fontseldiag);

static GnomeDialogClass *font_selection_parent_class = NULL;
static GtkTableClass    *font_selection_dialog_parent_class = NULL;

GtkType
gnome_font_selection_get_type()
{
  static GtkType font_selection_type = 0;
  
  if(!font_selection_type)
    {
      static const GtkTypeInfo fontsel_type_info =
      {
	"GnomeFontSelection",
	sizeof (GnomeFontSelection),
	sizeof (GnomeFontSelectionClass),
	(GtkClassInitFunc) gnome_font_selection_class_init,
	(GtkObjectInitFunc) gnome_font_selection_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };
      
      font_selection_type = gtk_type_unique (GTK_TYPE_TABLE,
					     &fontsel_type_info);
    }
  
  return font_selection_type;
}

static void
gnome_font_selection_class_init(GnomeFontSelectionClass *klass)
{
  GtkObjectClass *object_class;
  
  object_class = (GtkObjectClass *) klass;
  
  font_selection_parent_class = gtk_type_class (GTK_TYPE_TABLE);
  
  object_class->destroy = gnome_font_selection_destroy;
}

static void
gnome_font_selection_init(GnomeFontSelection *fontsel)
{
  GnomeFontClass *klass;
  GtkWidget *combo;
  GtkWidget *listitem;
  GList *list; /* Of type gchar */
  gint i;
  GtkTable *table = GTK_TABLE( fontsel );

  table->homogeneous = FALSE;
  gtk_table_resize (table, 2, 2);

  combo = gtk_combo_new();
  klass = gtk_type_class (gnome_font_get_type ());
  list = gnome_font_family_list (klass);
  gtk_combo_set_popdown_strings( GTK_COMBO( combo ), list );
  gnome_font_family_list_free (list);
  gtk_table_attach( GTK_TABLE( table ), combo, 0, 2, 0, 1,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->family_name = combo;
  gtk_widget_show( combo );

  combo = gtk_combo_new();
  for ( i = 0; i < sizeof(font_sizes) / sizeof(font_sizes[0]); i++ )
    {
      listitem = gtk_list_item_new_with_label ( font_sizes[ i ] );
      gtk_widget_show( listitem );
      gtk_container_add( GTK_CONTAINER( GTK_COMBO( combo )->list ), listitem );
    }
  gtk_table_attach( GTK_TABLE( table ), combo, 0, 1, 1, 2,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->font_size = combo;
  gtk_widget_show( combo );
  
  combo = gtk_combo_new();
  for ( i = 0; i < sizeof(weight_tab) / sizeof(weight_tab[0]); i++ )
    {
      listitem = gtk_list_item_new_with_label ( weight_tab[ i ].name);
      gtk_widget_show( listitem );
      gtk_container_add( GTK_CONTAINER( GTK_COMBO( combo )->list ), listitem );
    }

  gtk_table_attach( GTK_TABLE( table ), combo, 1, 2, 1, 2,
		    GTK_EXPAND | GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    5, 5);
  fontsel->font_weight = combo;
  gtk_widget_show( combo );
}

GtkWidget *
gnome_font_selection_new()
{
  GnomeFontSelection *fontsel;
  
  fontsel = gtk_type_new (gnome_font_selection_get_type());
  
  return GTK_WIDGET (fontsel);
}

static void
gnome_font_selection_destroy (GtkObject *object)
{
  GnomeFontSelection *fontsel;
  
  g_return_if_fail (object != NULL);
  g_return_if_fail (GNOME_IS_FONT_SELECTION (object));
  
  fontsel = GNOME_FONT_SELECTION (object);
  
  if (GTK_OBJECT_CLASS (font_selection_parent_class)->destroy)
    (* GTK_OBJECT_CLASS (font_selection_parent_class)->destroy) (object);
}


/*****************************************************************************
 * These functions are the main public interface for getting/setting the font.
 *****************************************************************************/

GnomeDisplayFont*
gnome_font_selection_get_font (GnomeFontSelection *fontsel)
{
  gchar *family_chars;
  gchar *temp;
  gint font_size;
  GnomeFontWeight font_weight;
  GnomeDisplayFont *display_font;
  gint i;
  
  g_return_val_if_fail (fontsel != NULL, NULL);

  family_chars = gtk_editable_get_chars( GTK_EDITABLE( GTK_COMBO( fontsel->family_name )->entry ), 0, -1 );
  
  temp = gtk_editable_get_chars( GTK_EDITABLE( GTK_COMBO( fontsel->font_size )->entry ), 0, -1 );
  font_size = atoi( temp );
  g_free ( temp );
  
  temp = gtk_editable_get_chars( GTK_EDITABLE( GTK_COMBO( fontsel->font_weight )->entry ), 0, -1 );
  font_weight = GNOME_FONT_BOOK;
  for (i = 0; i < sizeof(weight_tab) / sizeof(weight_tab[0]); i++)
    if (!g_strcasecmp (temp, weight_tab[i].name))
      {
	font_weight = weight_tab[i].code;
	break;
      }
  g_free ( temp );
  
  display_font = gnome_get_display_font( family_chars,
					 font_weight,
					 FALSE,
					 font_size,
					 1);
  g_free( family_chars );

  return display_font;
}


/*****************************************************************************
 * GtkFontSelectionDialog
 *****************************************************************************/

guint
gnome_font_selection_dialog_get_type (void)
{
  static guint font_selection_dialog_type = 0;
  
  if (!font_selection_dialog_type)
    {
      GtkTypeInfo fontsel_diag_info =
      {
	"GnomeFontSelectionDialog",
	sizeof (GnomeFontSelectionDialog),
	sizeof (GnomeFontSelectionDialogClass),
	(GtkClassInitFunc) gnome_font_selection_dialog_class_init,
	(GtkObjectInitFunc) gnome_font_selection_dialog_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };
      
      font_selection_dialog_type = gtk_type_unique (gnome_dialog_get_type(), &fontsel_diag_info);
    }
  
  return font_selection_dialog_type;
}

static void
gnome_font_selection_dialog_class_init (GnomeFontSelectionDialogClass *klass)
{
  GtkObjectClass *object_class;
  
  object_class = (GtkObjectClass*) klass;
  
  font_selection_dialog_parent_class = gtk_type_class (gnome_dialog_get_type());
}

static void
gnome_font_selection_dialog_init (GnomeFontSelectionDialog *fontseldiag)
{
  GtkWidget *frame;
  gnome_dialog_append_button ( GNOME_DIALOG(fontseldiag),
			       GNOME_STOCK_BUTTON_OK);

  gnome_dialog_append_button ( GNOME_DIALOG(fontseldiag),
			       GNOME_STOCK_BUTTON_CANCEL);

  gnome_dialog_set_default( GNOME_DIALOG( fontseldiag ), 0 );

  frame = gtk_frame_new (_("Select font"));
  
  fontseldiag->fontsel = gnome_font_selection_new();
  gtk_container_add (GTK_CONTAINER (frame), fontseldiag->fontsel);
  
  gtk_container_set_border_width (GTK_CONTAINER (fontseldiag), 4);
  gtk_window_set_policy(GTK_WINDOW(fontseldiag), FALSE, TRUE, TRUE);
  
  gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (fontseldiag)->vbox), frame, TRUE, TRUE, 0);
  gtk_widget_show_all (frame);  
}

GtkWidget*
gnome_font_selection_dialog_new	(const gchar	  *title)
{
  GnomeFontSelectionDialog *fontseldiag;
  
  fontseldiag = gtk_type_new (GNOME_TYPE_FONT_SELECTION_DIALOG);
  gtk_window_set_title (GTK_WINDOW (fontseldiag),
			title ? title : _("Font Selection"));
  
  return GTK_WIDGET (fontseldiag);
}

GnomeDisplayFont*
gnome_font_selection_dialog_get_font	(GnomeFontSelectionDialog *fsd)
{
  return gnome_font_selection_get_font(GNOME_FONT_SELECTION(fsd->fontsel));
}
