/* 
 * Copyright (C) 1998-2000 Free Software Foundation
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Authors : Vadim Strizhevsky
 *           Eskil Heyn Olsen
 */


#include <config.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <gnome.h>
#include <gdk_imlib.h>
#include <glade/glade.h>
#include <libgnomeui/gnome-window-icon.h>
#include "applet-widget.h"

/* headers for select & child stuff */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <signal.h>

#include "gpilotd/gnome-pilot-client.h"

enum {
  TARGET_URI_LIST,
};

typedef struct _pilot_properties pilot_properties;

struct _pilot_properties {
	GList* pilot_ids;
	GList* cradles;
	gchar * exec_when_clicked;
	gboolean popups;
};

typedef struct {
	gchar *cradle;
	gchar *backupdir;
	gchar *username;
	gint id;
} restore_properties;

static pilot_properties properties = { NULL };

typedef enum { INITIALISING, CONNECTING_TO_DAEMON, SYNCING, WAITING, NUM_STATES } state;

char *pixmaps[] = 
{
    "sync_broken.png",
    "sync_icon.png",
    "syncing_icon.png",
    "sync_icon.png",
};

static state curstate;
static GtkWidget *applet;
static GtkWidget *pixmap; 
static GtkWidget *dialogWindow; 
static GtkWidget *operationDialogWindow;
static GtkWidget *pb = NULL;

guint timeout_handler_id;
gboolean double_clicked;

static GtkWidget *progressDialog=NULL;
static GtkWidget *sync_label,*overall_progress_bar,*conduit_progress_bar, *message_area;
static GtkWidget *cancel_button;
static GtkWidget *chooseDialog=NULL; 
static GtkWidget *restoreDialog=NULL;
static GdkColor  errorColor;
static gchar* glade_file=NULL;

static GnomePilotClient *gpc = NULL;

static void show_error_dialog (gchar*,...);
static void show_warning_dialog (gchar*,...);
static void show_message_dialog (gchar*,...);
static void resume_cb (AppletWidget *widget, gpointer data);
static void cancel_cb (GtkButton* button,gpointer whatever);

static void pilot_draw (GtkWidget*);

static void pilot_execute_program (const gchar *);
#define GPILOTD_DRUID "gpilotd-control-applet --druid"
#define GPILOTD_CAPPLET "gpilotd-control-applet"
#define CONDUIT_CAPPLET "gpilotd-control-applet --cap-id=1"

/******************************************************************/

static void 
gpilotd_connect_cb (GnomePilotClient* client, 
		    const gchar *id,
		    const GNOME_Pilot_PilotUser *user,
		    gpointer unused)
{
	GdkColormap *colormap;

	gchar *buf;
	applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Synchronizing..."));
  
	if (!GTK_WIDGET_REALIZED (pixmap)) {
		g_warning ("! realized");
		return;
	}
	curstate = SYNCING;
	pilot_draw (pixmap);

	if (properties.popups == FALSE) return;

	if (progressDialog == NULL) {
		GladeXML *xml =glade_xml_new (glade_file,"ProgressDialog");
		progressDialog=glade_xml_get_widget (xml,"ProgressDialog");
		sync_label=glade_xml_get_widget (xml,"sync_label");
		message_area=glade_xml_get_widget (xml,"message_area");
		overall_progress_bar=glade_xml_get_widget (xml,"overall_progress_bar");
		conduit_progress_bar=glade_xml_get_widget (xml,"conduit_progress_bar");
		cancel_button=glade_xml_get_widget (xml,"cancel_button");
		gtk_signal_connect (GTK_OBJECT (cancel_button),"clicked",
				   GTK_SIGNAL_FUNC (cancel_cb),NULL);
	} else {
		gtk_text_set_point (GTK_TEXT (message_area),0);
		gtk_text_forward_delete (GTK_TEXT (message_area),
				gtk_text_get_length (GTK_TEXT (message_area)));
	}

	gtk_widget_set_sensitive (cancel_button,FALSE);
	buf=g_strdup_printf (_("%s Synchronizing"),id);
	gtk_label_set_text (GTK_LABEL (sync_label),buf);
	g_free (buf);
	gtk_widget_show_all (progressDialog);

	gtk_progress_set_format_string (GTK_PROGRESS (overall_progress_bar), _("Database %v of %u"));
	gtk_progress_set_format_string (GTK_PROGRESS (conduit_progress_bar), "");
	gtk_progress_configure (GTK_PROGRESS (overall_progress_bar),0 ,0, 1);
	gtk_progress_configure (GTK_PROGRESS (conduit_progress_bar),0 ,0, 100.0);

	colormap = gdk_window_get_colormap (message_area->window);
	gdk_color_parse ("red",&errorColor);
	gdk_colormap_alloc_color (colormap,&errorColor,FALSE,TRUE);
}

static void 
gpilotd_disconnect_cb (GnomePilotClient* client, 
		       const gchar *id,
		       gpointer unused)
{
	applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Ready to synchronize"));

	curstate = WAITING;
	pilot_draw (pixmap);
	if (properties.popups && progressDialog!=NULL) {
		gtk_widget_hide (progressDialog);
	}
}

static void 
gpilotd_request_completed (GnomePilotClient* client,
			   const gchar *id,
			   unsigned long handle,
			   gpointer unused)
{
	gnome_dialog_close (GNOME_DIALOG (operationDialogWindow));
	if (properties.popups && progressDialog !=NULL) {
		gchar *txt=g_strdup_printf (_("Request %ld has been completed\n"),handle);
		gtk_text_insert (GTK_TEXT (message_area),NULL,NULL,NULL,txt,-1);
		g_free (txt);

	}

/*	show_message_dialog ("Pilot %s has completed request %ld",
			    strlen (id)==0?"(noname)":id,
			    handle);
*/
}

static void 
gpilotd_conduit_start (GnomePilotClient* client, 
		       const gchar *id, 
		       const gchar *conduit, 
		       const gchar *db_name,
		       gpointer unused) 
{ 
	if (properties.popups && progressDialog!=NULL) {
		gchar *txt=g_strdup_printf (_("%s Synchronizing : %s"),id, conduit);
		gtk_label_set_text (GTK_LABEL (sync_label),txt);
		gtk_progress_configure (GTK_PROGRESS (conduit_progress_bar),0,0,100);
		g_free (txt);
		txt=g_strdup_printf (_("%s: Started\n"),conduit);
		gtk_text_insert (GTK_TEXT (message_area),NULL,NULL,NULL,txt,-1);
		g_free (txt);
	}
}

static void 
gpilotd_conduit_end (GnomePilotClient* client, 
		     const gchar *id, 
		     const gchar *conduit,
		     gpointer unused) 
{
	if (properties.popups && progressDialog!=NULL) {
		gchar *txt=g_strdup_printf (_("%s Finished : %s"),id, conduit);
		gtk_progress_configure (GTK_PROGRESS (conduit_progress_bar),100,0,100);
		gtk_label_set_text (GTK_LABEL (sync_label),txt);
		g_free (txt);
		txt=g_strdup_printf (_("%s: Ended\n"),conduit);
		gtk_text_insert (GTK_TEXT (message_area),NULL,NULL,NULL,txt,-1);
		g_free (txt);
		gtk_progress_set_format_string (GTK_PROGRESS (conduit_progress_bar), "");
	}
}

static void 
gpilotd_conduit_progress (GnomePilotClient* client, 
			  const gchar *id, 
			  const gchar *conduit, 
			  guint current, 
			  guint total,
			  gpointer unused)
{
	gfloat tot_f,cur_f;
	tot_f = total;
	cur_f = current;
/*
        g_message ("%s : %s : %d/%d = %d%%",id, conduit, current, 
        total,abs (cur_f/(tot_f/100)));
*/
	if (properties.popups && progressDialog!=NULL) {
		gtk_progress_configure (GTK_PROGRESS (conduit_progress_bar),cur_f,1,tot_f); 
		gtk_progress_set_format_string (GTK_PROGRESS (conduit_progress_bar), _("%v of %u records"));
	}
}

static void 
gpilotd_overall_progress (GnomePilotClient* client, 
			  const gchar *id, 
			  guint current, 
			  guint total,
			  gpointer unused)
{
	gfloat tot_f,cur_f;
	tot_f = total;
	cur_f = current;


	if (properties.popups && progressDialog!=NULL) {
		gtk_progress_configure (GTK_PROGRESS (overall_progress_bar),cur_f,0,tot_f);
	}
}

static void 
gpilotd_conduit_message (GnomePilotClient* client, 
			 const gchar *id, 
			 const gchar *conduit, 
			 const gchar *message,
			 gpointer unused) 
{
	if (properties.popups && progressDialog != NULL) {
		gchar *txt=g_strdup_printf ("%s: %s\n",conduit,message);
		gtk_text_insert (GTK_TEXT (message_area),NULL,NULL,NULL,txt,-1);
		g_free (txt);
	}
}

static void 
gpilotd_daemon_message (GnomePilotClient* client, 
			const gchar *id, 
			const gchar *conduit,
			const gchar *message,
			gpointer unused) 
{
	if (properties.popups && progressDialog != NULL) {
		gchar *txt=g_strdup_printf ("%s\n", message);
		gtk_text_insert (GTK_TEXT (message_area),NULL,NULL,NULL,txt,-1);
		g_free (txt);
	}
}

static void 
gpilotd_daemon_error (GnomePilotClient* client, 
			 const gchar *id, 
			 const gchar *message,
			 gpointer unused) 
{
	if (properties.popups && progressDialog != NULL) {
		gchar *txt=g_strdup_printf ("Error: %s\n", message);
		gtk_text_insert (GTK_TEXT (message_area),NULL,NULL,NULL,txt,-1);
		g_free (txt);
	}
}

static void 
gpilotd_conduit_error (GnomePilotClient* client, 
		       const gchar *id, 
		       const gchar *conduit, 
		       const gchar *message,
		       gpointer unused) 
{
	if (properties.popups && progressDialog != NULL) {
		gchar *txt=g_strdup_printf ("%s: %s\n",conduit,message);
		gtk_text_insert (GTK_TEXT (message_area),NULL,&errorColor,NULL,txt,-1);
		g_free (txt);
	}
}

/******************************************************************/

static GList*
get_pilot_ids_from_gpilotd () 
{
	GList *pilots=NULL;
	gnome_pilot_client_get_pilots (gpc,&pilots);
	return pilots;
}

#if 0
/* not currently used */
static char *
file_type (char *name) 
{
	static char command[128];
	char *tmp;
	FILE *f;

	if (!access (name,R_OK)) return NULL;

	tmp = tmpnam (NULL);
	snprintf (command,127,"file \"%s\" > %s",name,tmp);
	if (system (command)==-1) {
		return NULL;
	}
  
	if ((f = fopen (tmp,"rt"))==NULL) {
		return NULL;
	}

	fgets (command,128,f);
	fclose (f);
	unlink (tmp);

	tmp = (char*)strstr (command,": "); tmp+=2;
	return tmp;
}

static long 
file_size (char *name) 
{
	FILE *FS;
	long ret;

	if ((FS = fopen (name,"rt"))==NULL) {
		return 0;
	}
	fseek (FS,0L,SEEK_END);
	ret = ftell (FS);
	fclose (FS);
	return ret;
}
#endif
/******************************************************************/

static void 
pilot_draw (GtkWidget *pixmap)
{
	if (!GTK_WIDGET_REALIZED (pixmap)) {
		g_warning ("pilot_draw ! realized");
		return;
	}

	gnome_pixmap_load_file (GNOME_PIXMAP (pixmap), pixmaps[curstate]);
  
}

/******************************************************************/

static int 
timeout (gpointer data) 
{
	switch (curstate) {
		
	case INITIALISING: {
		g_message ("state = INITIALISING");
		pilot_draw (GTK_WIDGET (data));
		
		if (gnome_pilot_client_connect_to_daemon (gpc) == GPILOTD_OK) {
			curstate = CONNECTING_TO_DAEMON;
			pilot_draw (GTK_WIDGET (data));
		} else {
			/* timeout_handler_id = gtk_timeout_add (3000,timeout,data); */
		}
		return TRUE;
	} break;
	
	case CONNECTING_TO_DAEMON: {
		g_message ("state = CONNECTING_TO_DAEMON");

                gtk_timeout_remove (timeout_handler_id); 
		timeout_handler_id = gtk_timeout_add (5000,timeout,pixmap);
		
		if (properties.pilot_ids) {
			GList *tmp =properties.pilot_ids;
			while (tmp) {
				g_free ((gchar*)tmp->data);
				tmp = g_list_next (tmp);
			} 
			g_list_free (properties.pilot_ids);
		}
		properties.pilot_ids = get_pilot_ids_from_gpilotd ();
		if (properties.pilot_ids==NULL){
			applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Not connected. Restart daemon to reconnect"));
			pilot_execute_program (GPILOTD_DRUID);
			curstate=INITIALISING;
			pilot_draw (GTK_WIDGET (data));
			/* FIXME: add gpilot_monitor_state_change () */
			return FALSE;
		}
		gnome_pilot_client_monitor_on_all_pilots (gpc);
		
		gnome_pilot_client_notify_on (gpc,GNOME_Pilot_NOTIFY_CONNECT);
		gnome_pilot_client_notify_on (gpc,GNOME_Pilot_NOTIFY_CONDUIT);
		gnome_pilot_client_notify_on (gpc,GNOME_Pilot_NOTIFY_DISCONNECT);
		
		applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Ready to synchronize"));
		
		curstate = WAITING;
		pilot_draw (GTK_WIDGET (data));    
	} break;
	case WAITING:
		if (gnome_pilot_client_noop (gpc) == GPILOTD_ERR_NOT_CONNECTED) {
			curstate = INITIALISING;
			gtk_timeout_remove (timeout_handler_id); 
			timeout_handler_id = gtk_timeout_add (1000,timeout,pixmap);
		}
		break;
	case SYNCING: 
	break;
	default: g_error ("curstate == default ?!"); break;
	}
	
	return TRUE;
}

static void 
show_error_dialog (gchar *mesg,...) 
{
	char *tmp;
	va_list ap;

	va_start (ap,mesg);
	tmp = g_strdup_vprintf (mesg,ap);
	dialogWindow = gnome_message_box_new (tmp,GNOME_MESSAGE_BOX_ERROR,
					     GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close (GNOME_DIALOG (dialogWindow));
	g_free (tmp);
	va_end (ap);
}

static void 
show_warning_dialog (gchar *mesg,...) 
{
	char *tmp;
	va_list ap;

	va_start (ap,mesg);
	tmp = g_strdup_vprintf (mesg,ap);
	dialogWindow = gnome_message_box_new (tmp,GNOME_MESSAGE_BOX_WARNING,
					     GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close (GNOME_DIALOG (dialogWindow));
	g_free (tmp);
	va_end (ap);
}


static void 
show_message_dialog (char *mesg,...)
{
	char *tmp;
	va_list ap;

	va_start (ap,mesg);
	tmp = g_strdup_vprintf (mesg,ap);
	dialogWindow = gnome_message_box_new (tmp,GNOME_MESSAGE_BOX_GENERIC,
					     GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close (GNOME_DIALOG (dialogWindow));
	g_free (tmp);
	va_end (ap);
}

static void
handle_client_error (void)
{
	if (curstate == SYNCING) {
		show_warning_dialog (_("Pilot is currently synchronizing.\nPlease wait for it to finish."));
	} else {
		curstate=INITIALISING;
		applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Not connected. Please restart daemon."));	
		pilot_draw (pixmap);
		show_error_dialog (_("Not connected to gpilotd.\nPlease restart daemon."));
	}
}

static void
load_properties (char *cfgpath)
{
	if (properties.exec_when_clicked) g_free (properties.exec_when_clicked);
  
	gnome_config_push_prefix (cfgpath);
	properties.exec_when_clicked = gnome_config_get_string ("exec_when_clicked=" GPILOTD_CAPPLET);
/*	properties.text_limit = gnome_config_get_int ("text_limit=10240");*/
	properties.popups = gnome_config_get_bool ("pop-ups=true");

	gnome_config_pop_prefix ();
}

static void
save_properties (const char *cfgpath,
		const char *globcfgpath) {
	gnome_config_push_prefix (cfgpath);
	gnome_config_set_string ("exec_when_clicked",properties.exec_when_clicked);
	/* gnome_config_set_int ("text_limit",properties.text_limit); */
	gnome_config_set_bool ("pop-ups",properties.popups);

	gnome_config_pop_prefix ();
	gnome_config_sync ();
	gnome_config_drop_all ();
  
}

static void
apply_exec_when_clicked_cb (GnomePropertyBox * pb, gint page, gpointer data)
{
	gchar * new_list;

	if (page != -1) return; /* Only honor global apply */

	new_list = gtk_entry_get_text (GTK_ENTRY (data));

	if (new_list) {
		g_free (properties.exec_when_clicked);
		properties.exec_when_clicked = g_strdup (new_list);
	}

}

static void
toggle_notices_cb (GtkWidget *toggle,gpointer data) 
{
	properties.popups = GTK_TOGGLE_BUTTON (toggle)->active;
	gnome_property_box_changed (GNOME_PROPERTY_BOX (pb));
}

static void
apply_properties_cb (GtkWidget* pb,gpointer data)
{
	save_properties (APPLET_WIDGET (applet)->privcfgpath,
			APPLET_WIDGET (applet)->globcfgpath);
}

static void cancel_restore (GnomeDialog *w,gpointer data) {
	g_message (_("cancelling %d"),GPOINTER_TO_INT (data));
	gnome_pilot_client_remove_request (gpc,GPOINTER_TO_INT (data));  
}


static void
activate_pilot_menu (GtkMenuItem *widget, gpointer data)
{
	gtk_object_set_data (GTK_OBJECT (chooseDialog),"pilot",data);	
}

static gchar*
pick_pilot (void)
{
	gchar * pilot=NULL;
	if (properties.pilot_ids) {
		if (g_list_length (properties.pilot_ids)==1) {
                        /* there's only one */
			pilot = (gchar*)properties.pilot_ids->data;
		} else {
			GList *tmp;
			GtkWidget *optionMenu,*menuItem,*menu;
			if (chooseDialog == NULL) {
				GladeXML * xml;

				xml = glade_xml_new (glade_file,"ChoosePilot");
				chooseDialog = glade_xml_get_widget (xml,"ChoosePilot");
				optionMenu=glade_xml_get_widget (xml,"pilot_menu");
				gtk_object_set_data (GTK_OBJECT (chooseDialog),"pilot_menu",optionMenu);
			} else {
				optionMenu=gtk_object_get_data (GTK_OBJECT (chooseDialog),"pilot_menu");
			}
			menu = gtk_menu_new ();
	
			tmp=properties.pilot_ids;
			while (tmp!=NULL){
				menuItem = gtk_menu_item_new_with_label ((gchar*)tmp->data);
				gtk_widget_show (menuItem);
				gtk_signal_connect (GTK_OBJECT (menuItem),"activate",
						   GTK_SIGNAL_FUNC (activate_pilot_menu),
						   tmp->data);
				gtk_menu_append (GTK_MENU (menu),menuItem);
				tmp=tmp->next;
			}
			gtk_option_menu_set_menu (GTK_OPTION_MENU (optionMenu),menu);
			gtk_option_menu_set_history (GTK_OPTION_MENU (optionMenu),0);
			gtk_object_set_data (GTK_OBJECT (chooseDialog),"pilot",properties.pilot_ids->data);	
			if (gnome_dialog_run_and_close (GNOME_DIALOG (chooseDialog))==0) {
				pilot=(gchar *)gtk_object_get_data (GTK_OBJECT (chooseDialog),"pilot");
			} else {
				pilot=NULL;
			}
		}
	}

	return pilot;
}

static void
restore_dialog (AppletWidget * w, gpointer data)
{
	int handle,i;
	gchar *pilot_name;
	gint pilot_id;
	GladeXML *xml;
	GtkWidget *dir_entry, *id_entry,*frame;
	GtkWidget *device_menu, *menu_item,*menu;
	GList *list;
	gchar *buf;
	restore_properties restore_props; 

	
	if (curstate!=WAITING) {
		handle_client_error ();
		return;
	}
	
	if ((pilot_name=pick_pilot ()) == NULL) {
		return;
	}

	if (gnome_pilot_client_get_pilot_base_dir_by_name (gpc,pilot_name,&buf) == GPILOTD_OK) {
		restore_props.backupdir = g_strdup_printf ("%s/backup/",buf);
		g_free (buf);
	} else {
		handle_client_error ();
		return;
	}
	if (gnome_pilot_client_get_pilot_id_by_name (gpc,pilot_name,&pilot_id) != GPILOTD_OK) {
		handle_client_error ();
		return;
	}

	xml = glade_xml_new (glade_file,"RestoreDialog");
	restoreDialog = glade_xml_get_widget (xml,"RestoreDialog");

	dir_entry = glade_xml_get_widget (xml,"dir_entry");

	gtk_entry_set_text (GTK_ENTRY (dir_entry),restore_props.backupdir);
	g_free (restore_props.backupdir);
	restore_props.backupdir=NULL;
	/* FIXME: do we need to preserve the backupdir somewhere? */
	
	frame = glade_xml_get_widget (xml,"main_frame");
	buf = g_strdup_printf (_("Restoring %s"), pilot_name);
	gtk_frame_set_label (GTK_FRAME (frame), buf);
	g_free (buf);
	

	id_entry =  glade_xml_get_widget (xml,"pilotid_entry");
	
	buf = g_strdup_printf ("%d",pilot_id);
	gtk_entry_set_text (GTK_ENTRY (id_entry),buf);
	g_free (buf);

	properties.cradles=NULL;
	gnome_pilot_client_get_cradles (gpc,&properties.cradles);
	list =properties.cradles;
	
	device_menu =  glade_xml_get_widget (xml,"device_menu");
	menu =gtk_menu_new ();
	
	i=0;
	while (list){
		menu_item = gtk_menu_item_new_with_label ((gchar*)list->data);
		gtk_widget_show (menu_item);
/*
		gtk_signal_connect (GTK_OBJECT (menu_item),"activate",
				   GTK_SIGNAL_FUNC (activate_device),
				   (gchar*)list->data);
*/
		gtk_menu_append (GTK_MENU (menu),menu_item);
		list=list->next;
		i++;
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (device_menu),menu);	
	if (i<=1) gtk_widget_set_sensitive (GTK_WIDGET (device_menu),FALSE);
	else  gtk_widget_set_sensitive (GTK_WIDGET (device_menu),TRUE);
	
	if (gnome_dialog_run_and_close (GNOME_DIALOG (restoreDialog))==0) {

		restore_props.backupdir=gtk_entry_get_text (GTK_ENTRY (dir_entry));
		if ( restore_props.backupdir == NULL || 
		     strlen (restore_props.backupdir)==0) {
			show_warning_dialog (_("No directory to restore from."));
			return;
		}
		
		/* FIXME: how do we specify which device to restore on? */
		if (gnome_pilot_client_restore (gpc,pilot_name,restore_props.backupdir,GNOME_Pilot_IMMEDIATE,0, &handle) == GPILOTD_OK) {
			operationDialogWindow = gnome_message_box_new (_("Press synchronize on the cradle to restore\n" 
									" or cancel the operation."),
								      GNOME_MESSAGE_BOX_GENERIC,
								      GNOME_STOCK_BUTTON_CANCEL,
								      NULL);
			gnome_dialog_button_connect (GNOME_DIALOG (operationDialogWindow),0,
						    cancel_restore,GINT_TO_POINTER (handle));
			gnome_dialog_run_and_close (GNOME_DIALOG (operationDialogWindow));
		} else {
			show_warning_dialog (_("Restore request failed"));
		}
	}
}
static void 
properties_dialog (GtkWidget *w, gpointer data)
{
	static GnomeHelpMenuEntry help_entry = {"gpilot-applet","properties"};

	GtkWidget *button, *entry;
	GladeXML *xml;

	xml =glade_xml_new (glade_file,"PropertiesDialog");
	pb=glade_xml_get_widget (xml,"PropertiesDialog");
	
	gtk_window_set_title (GTK_WINDOW (pb), _("GNOME PilotSync Properties"));


	/* execute when clicked */

	entry = glade_xml_get_widget (xml,"exec_entry");
	gtk_entry_set_text (GTK_ENTRY (entry), properties.exec_when_clicked);
	gtk_signal_connect_object (GTK_OBJECT (entry), "changed",
				  GTK_SIGNAL_FUNC (gnome_property_box_changed),
				  GTK_OBJECT (pb));
	gtk_signal_connect (GTK_OBJECT (pb), "apply", 
			   GTK_SIGNAL_FUNC (apply_exec_when_clicked_cb),entry);

  
	button = glade_xml_get_widget (xml,"notices_button");
	gtk_toggle_button_set_state (GTK_TOGGLE_BUTTON (button),properties.popups);
	gtk_signal_connect (GTK_OBJECT (button), "toggled",
			   GTK_SIGNAL_FUNC (toggle_notices_cb),
			   NULL);


	gtk_signal_connect (GTK_OBJECT (pb), 
			   "apply", 
			   GTK_SIGNAL_FUNC (apply_properties_cb),
			   NULL);
	gtk_signal_connect (GTK_OBJECT (pb),
			   "destroy",
			   GTK_SIGNAL_FUNC (gtk_widget_destroy),
			   pb);
	gtk_signal_connect (GTK_OBJECT (pb),
			   "help",
			   GTK_SIGNAL_FUNC (gnome_help_pbox_display),
			   &help_entry);

	gtk_widget_show_all (pb);
}

static void
pilot_execute_program (const gchar *str)
{
	int err = -1;
	int argc=0;
	gchar **targv;
	
	targv=g_strsplit (str," ",0);
	while (targv[argc]) argc++;
	g_message ("pilot_execute_program (%s)",str);
	err = gnome_execute_async (NULL,argc,targv);
	if (err==-1) {
		show_warning_dialog (_("Execution of %s failed"),str);
	} else {
		sleep (1);
	}

	g_strfreev (targv);
}

static gint 
pilot_clicked_cb (GtkWidget * widget, 
		 GdkEventButton * e, 
		 gpointer data)
{
	pilot_execute_program (properties.exec_when_clicked);
	return TRUE; 
}

static void
pilot_link_dialog (GtkWidget *widget, gpointer data)
{
	if (data==NULL) {
		pilot_execute_program (GPILOTD_CAPPLET);
	} else {
		pilot_execute_program (CONDUIT_CAPPLET);
	}
}

static void
lastlog_dialog (GtkWidget *widget, gpointer data)
{
	if (progressDialog != NULL) {
		gtk_label_set_text (GTK_LABEL (sync_label),"Last Sync Log");
		gtk_widget_set_sensitive (cancel_button,TRUE);
		gtk_widget_show_all (progressDialog);
	} else {
		show_message_dialog (_("There's no last sync on record."));
	}
}

static void 
cancel_cb (GtkButton* button,gpointer whatever)
{
	if (progressDialog != NULL) {
		gtk_widget_hide (progressDialog);
	}
}


/*
static gint
pilot_timeout (gpointer data)
{
	GtkWidget *pixmap = data;

	gtk_widget_queue_draw (pixmap);
	return TRUE;
}
*/

static void
dnd_drop_internal (GtkWidget        *widget,
		  GdkDragContext   *context,
		  gint              x,
		  gint              y,
		  GtkSelectionData *selection_data,
		  guint             info,
		  guint             time,
		  gpointer          data)
{
	gchar *pilot_id;
	gint  file_handle;
#if WITH_VFS
	gchar **names;
	int idx;
#else /* WITH_VFS */
	GList *names, *iterator;
#endif /* WITH_VFS */

	pilot_id=pick_pilot ();

	switch (info)
	{
	case TARGET_URI_LIST:
#if WITH_VFS
		names = g_strsplit ((char *)selection_data->data, "\n", -1);
		for (idx=0; names[idx]; idx++) {
			char *ptr=names[idx];
			while (iscntrl (ptr[strlen(ptr)-1])) {
				ptr[strlen(ptr)-1] = 0;
			}
#else /* WITH_VFS */
		names = gnome_uri_list_extract_filenames ((char *)selection_data->data);
		for (iterator = names; iterator;iterator = iterator->next) {
			char *ptr=iterator->data;
#endif /* WITH_VFS */
			if (ptr) {
				/*
				  if (strstr (file_type (ptr),"text")) {
				  if (file_size (ptr)>properties.text_limit)
				  g_message ("installing textfile as Doc (not implemented)");
				  else
				  g_message ("installing textfile as Memo (not implemented)");
				  }
				*/
				g_message (_("installing \"%s\""), ptr);
				if (pilot_id) {
					gnome_pilot_client_install_file (gpc,pilot_id,
									ptr,
									GNOME_Pilot_PERSISTENT,0,&file_handle);
				}
			}
		}
#if WITH_VFS
		g_strfreev (names);
#else /* WITH_VFS */
		gnome_uri_list_free_strings (names);
#endif /* WITH_VFS */
		break;
	default:
		g_warning (_("unknown dnd type"));
		break;
	}
}


static void
about_cb (AppletWidget *widget, gpointer data)
{
	GtkWidget *about;
	const gchar *authors[] = {"Vadim Strizhevsky, vadim@optonline.net"," Eskil Heyn Olsen, eskil@eazel.com",NULL};
	gchar version[] = VERSION;

	about = gnome_about_new (_("Gnome Pilot Applet"), 
				version,
				_("(C) 2000 Free Software Foundation "),
				(const gchar**)authors,
				_("A dnd and gui monitor for gpilotd.\n"
				  "Visit Website at http://www.gnome.org/gnome-pilot"),
				NULL);
	gtk_widget_show (about);

	return;
}

static gint
applet_save_session (GtkWidget *w,
		    const char *cfgpath,
		    const char *globcfgpath,
		    gpointer data)
{
	g_message (_("saving session"));
	save_properties (cfgpath,globcfgpath);
	return FALSE;
}

static void
applet_destroy (GtkWidget *applet,gpointer horse)
{
	g_message (_("destroy gpilot-applet"));
}

static void
restart_cb (AppletWidget *widget, gpointer data)
{
	if ( curstate == SYNCING) {
		if ( gnome_dialog_run_and_close (GNOME_DIALOG (gnome_question_dialog (_("Pilot sync is currently in progress.\nAre you sure you want to restart daemon?"),NULL,NULL))) !=0 ) {
			return;
		}
		if (progressDialog!=NULL) {
			gtk_widget_hide (progressDialog);
		}
	}
	curstate = INITIALISING;
	gnome_pilot_client_restart_daemon (gpc);
	applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Trying to connect to "
							  "the GnomePilot Daemon"));
	timeout_handler_id = gtk_timeout_add (1000,timeout,pixmap);
}

static void
pause_cb (AppletWidget *widget, gpointer data)
{
	if (curstate==WAITING &&	gnome_pilot_client_pause_daemon (gpc)==GPILOTD_OK) {
		applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Daemon paused"));
		applet_widget_register_stock_callback (APPLET_WIDGET (applet),
						      "stop",
						      GNOME_STOCK_MENU_REFRESH,
						      _("Resume daemon"),
						      resume_cb,
						      NULL);
	} else {
		handle_client_error ();
	}
}

static void
resume_cb (AppletWidget *widget, gpointer data)
{
	if (curstate==WAITING &&	gnome_pilot_client_unpause_daemon (gpc)==GPILOTD_OK) {

		applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Ready to synchronize"));
		applet_widget_register_stock_callback (APPLET_WIDGET (applet),
						      "stop",
						      GNOME_STOCK_PIXMAP_STOP,
						      _("Pause daemon"),
						      pause_cb,
						      NULL);
	} else {
		handle_client_error ();
	}
        return;
}

static GtkWidget *
create_pilot_widget (GtkWidget *window)
{
	GtkWidget *event_box;
	GtkStyle *style;
	int i;
  
	static GtkTargetEntry drop_types [] = { 
		{ "text/uri-list", 0, TARGET_URI_LIST },
	};
	static gint n_drop_types = sizeof (drop_types) / sizeof (drop_types[0]);

	style = gtk_widget_get_style (window);

	applet_widget_set_tooltip (APPLET_WIDGET (applet),_("Trying to connect to "
							  "the GnomePilot Daemon"));


	curstate = INITIALISING;

	for (i = 0; i < sizeof (pixmaps)/sizeof (pixmaps[0]); i++)
		pixmaps[i] = gnome_unconditional_pixmap_file (pixmaps[i]);
  
	event_box = gtk_event_box_new ();
  
	pixmap = gnome_pixmap_new_from_file (pixmaps[curstate]);

  
	gtk_widget_set_events (event_box, gtk_widget_get_events (applet) |
			      GDK_BUTTON_PRESS_MASK);
	gtk_signal_connect (GTK_OBJECT (event_box), "button-press-event",
			   GTK_SIGNAL_FUNC (pilot_clicked_cb), NULL);


	gtk_widget_show (pixmap);  
	gtk_container_add (GTK_CONTAINER (event_box), pixmap);

	timeout_handler_id = gtk_timeout_add (1000,timeout,pixmap);
  
	gtk_signal_connect (GTK_OBJECT (applet),"save_session",
			   GTK_SIGNAL_FUNC (applet_save_session),
			   NULL);
	gtk_signal_connect (GTK_OBJECT (applet),"destroy",
			   GTK_SIGNAL_FUNC (applet_destroy),NULL);

	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "restore",
					      GNOME_STOCK_PIXMAP_OPEN,
					      _("Restore..."),
					      restore_dialog,
					      NULL);	
	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "stop",
					      GNOME_STOCK_PIXMAP_STOP,
					      _("Pause daemon"),
					      pause_cb,
					      NULL);
	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "restart",
					      GNOME_STOCK_PIXMAP_REFRESH,
					      _("Restart daemon"),
					      restart_cb,
					      NULL);
	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "lastlog",
					      GNOME_STOCK_MENU_PREF,
					      _("Last sync log..."),
					      GTK_SIGNAL_FUNC (lastlog_dialog),
					      NULL);	
#if 0 
        /* FIXME: remove undef when the new capplet is checked in */
	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "conduitprops",
					      GNOME_STOCK_MENU_PROP,
					      _("Conduit Settings..."),
					      GTK_SIGNAL_FUNC (pilot_link_dialog),
					      GINT_TO_POINTER (1));	
#endif
	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "pilotlinkprops",
					      GNOME_STOCK_MENU_PROP,
					      _("Pilot Link Settings..."),
					      GTK_SIGNAL_FUNC (pilot_link_dialog),
					      NULL);	
	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "properties",
					      GNOME_STOCK_MENU_PROP,
					      _("Properties..."),
					      GTK_SIGNAL_FUNC (properties_dialog),
					      NULL);	
	applet_widget_register_stock_callback (APPLET_WIDGET (applet),
					      "about",
					      GNOME_STOCK_MENU_ABOUT,
					      _("About..."),
					      about_cb,
					      NULL);
	/* Now set the dnd features */
	gtk_drag_dest_set (GTK_WIDGET (pixmap),
			   GTK_DEST_DEFAULT_ALL,
			   drop_types, n_drop_types,
			   GDK_ACTION_COPY);


	gtk_signal_connect (GTK_OBJECT (pixmap),
			   "drag_data_received",
			   GTK_SIGNAL_FUNC (dnd_drop_internal),
			   NULL);


	return event_box;
}

static GtkWidget * applet_start_new_applet (const gchar *goad_id, const char **params, int nparams) {
	GtkWidget *pilot;
	GtkWidget *applet;
  
	if (strcmp (goad_id, "gpilot-applet")) return NULL;
     
	applet = applet_widget_new (goad_id);
	if (!applet)
		g_error (_("Can't create applet!\n"));
  
	gtk_widget_realize (applet);
	pilot = create_pilot_widget (applet);
	gtk_widget_show_all (pilot);


	applet_widget_add (APPLET_WIDGET (applet), pilot);
	gtk_widget_show (applet);
  
	return applet;
}



static CORBA_Object
pilot_applet_activator (PortableServer_POA poa,
		       const char *goad_id,
		       const char **params,
		       gpointer *impl_ptr,
		       CORBA_Environment *ev)
{
	GtkWidget *pilot;

	applet = applet_widget_new (goad_id);
	if (!applet)
		g_error (_("Can't create applet!\n"));

	load_properties (APPLET_WIDGET (applet)->privcfgpath);
  
	gtk_widget_realize (applet);
	pilot = create_pilot_widget (applet);
	gtk_widget_show_all (pilot);
  
  
	applet_widget_add (APPLET_WIDGET (applet), pilot);
	gtk_widget_show (applet);
 
	return applet_widget_corba_activate (applet, poa, goad_id, params,
					    impl_ptr, ev);
}

static void
pilot_applet_deactivator (PortableServer_POA poa,
			 const char *goad_id,
			 gpointer impl_ptr,
			 CORBA_Environment *ev)
{
	applet_widget_corba_deactivate (poa, goad_id, impl_ptr, ev);
}

#if 0 /* defined (SHLIB_APPLETS) */
static const char *repo_id[]={"IDL:GNOME/Applet:1.0", NULL};
static GnomePluginObject applets_list[] = {
	{repo_id, "gpilot-applet", NULL, "gnome-pilot applet",
	 &pilot_applet_activator, &pilot_applet_deactivator},
	{NULL}
};

GnomePlugin GNOME_Plugin_info = {
	applets_list, NULL
};
#else
int
main (int argc, char *argv[])
{
	gpointer pilot_applet_impl;
  
	/* initialize the i18n stuff */
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	/* initialize applet stuff */
	gnomelib_init ("gpilot-applet", VERSION);
	applet_widget_init ("gpilot-applet", VERSION, 
			   argc, argv, NULL, 0,NULL);

	/* initialize glade */
	glade_gnome_init ();
	glade_file="./pilot-applet.glade";
	if (!g_file_exists (glade_file)) {
		glade_file = g_concat_dir_and_file (GLADEDATADIR, "pilot-applet.glade");
	}
	if (!g_file_exists (glade_file)) {
		show_error_dialog (_("Cannot find pilot-applet.glade"));
		return -1;
	}
	gnome_window_icon_set_default_from_file (GNOME_ICONDIR"/gnome-palm.png");

	gpc = GNOME_PILOT_CLIENT (gnome_pilot_client_new ());
	
	gtk_signal_connect_object (GTK_OBJECT (gpc), 
				   GNOME_PILOT_CLIENT_SIGNAL_PILOT_CONNECT (gpilotd_connect_cb),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc), 
				  GNOME_PILOT_CLIENT_SIGNAL_PILOT_DISCONNECT (gpilotd_disconnect_cb),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc),
				  GNOME_PILOT_CLIENT_SIGNAL_COMPLETED_REQUEST (gpilotd_request_completed),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc), 
				  GNOME_PILOT_CLIENT_SIGNAL_START_CONDUIT (gpilotd_conduit_start),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc), 
				  GNOME_PILOT_CLIENT_SIGNAL_END_CONDUIT (gpilotd_conduit_end),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc), 
				  GNOME_PILOT_CLIENT_SIGNAL_PROGRESS_CONDUIT (gpilotd_conduit_progress),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc), 
				  GNOME_PILOT_CLIENT_SIGNAL_PROGRESS_OVERALL (gpilotd_overall_progress),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc), 
				  GNOME_PILOT_CLIENT_SIGNAL_ERROR_CONDUIT (gpilotd_conduit_error),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc),
				  GNOME_PILOT_CLIENT_SIGNAL_MESSAGE_CONDUIT (gpilotd_conduit_message),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc),
				  GNOME_PILOT_CLIENT_SIGNAL_MESSAGE_DAEMON (gpilotd_daemon_message),
				   NULL);
	gtk_signal_connect_object (GTK_OBJECT (gpc),
				  GNOME_PILOT_CLIENT_SIGNAL_ERROR_DAEMON (gpilotd_daemon_error),
				   NULL);

	applet_factory_new ("gpilot-applet_factory", NULL, applet_start_new_applet);

	APPLET_ACTIVATE (pilot_applet_activator, "gpilot-applet", &pilot_applet_impl);

	applet_widget_gtk_main ();

	APPLET_DEACTIVATE (pilot_applet_deactivator, "gpilot-applet", pilot_applet_impl);

	return 0;
}
#endif
