#include <glib.h>
#include <gnome.h>

#include <pi-source.h>
#include <pi-socket.h>
#include <pi-file.h>
#include <pi-dlp.h>
#include <pi-version.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <utime.h>
#include <unistd.h>
#include <pwd.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>
#include <gpilotd/gnome-pilot-conduit.h>
#include <gpilotd/gnome-pilot-conduit-standard-abs.h>

#include "memo_file_conduit.h"

#define MC_DEBUG

#ifdef MC_DEBUG
#define LOG(args...) g_log (G_LOG_DOMAIN, \
                            G_LOG_LEVEL_MESSAGE, \
                             "memo_file: "##args)
#else
#define LOG(args...)
#endif

GnomePilotConduit *conduit_get_gpilot_conduit (guint32 pilotId);
void conduit_destroy_gpilot_conduit (GnomePilotConduit *conduit);

#define CONDUIT_VERSION "0.7"

static void 
load_configuration(ConduitCfg **c,guint32 pilotId) 
{
	gchar prefix[256];
	gchar *buf;
	g_return_if_fail(c!=NULL);
	
	g_snprintf(prefix,255,CONFIG_PREFIX,pilotId);
 
	*c = g_new0(ConduitCfg,1);
	gnome_config_push_prefix(prefix);
	(*c)->sync_type = GnomePilotConduitSyncTypeCustom; /* this will be reset by capplet */
	(*c)->open_secret = gnome_config_get_bool("open secret=FALSE"); 
	
	buf = gnome_config_get_string("file mode=0600");
	(*c)->file_mode =(mode_t)strtol(buf,NULL,0);
	g_free(buf);

	buf = gnome_config_get_string("dir mode=0700");
	(*c)->dir_mode =(mode_t)strtol(buf,NULL,0);
	g_free(buf);

	buf = gnome_config_get_string("secret mode=0600");
	(*c)->secret_mode =(mode_t)strtol(buf,NULL,0);
	g_free(buf);
  
	(*c)->dir = gnome_config_get_string("dir");
	(*c)->ignore_end=gnome_config_get_string("ignore end");
	(*c)->ignore_start=gnome_config_get_string("ignore start");
	gnome_config_pop_prefix();
	(*c)->pilotId = pilotId;
}

static void 
destroy_configuration(ConduitCfg **c) 
{
	g_return_if_fail(c!=NULL);
	if((*c)->dir) g_free((*c)->dir);
	if((*c)->ignore_start) g_free((*c)->ignore_start);
	if((*c)->ignore_end) g_free((*c)->ignore_end);
	g_free(*c);
	*c = NULL;
}

static IterateData *
new_iterate_data(int _flag,int _archived) 
{
	static IterateData d;
	d.flag = _flag;
	d.archived = _archived;
	d.prev = NULL;
	d.first = NULL;
	return &d;
}

/* from pilot-xfer */
static void 
protect_name(char *d, char *s) 
{
	while(*s) {
		switch(*s) {
		case '/': *(d++) = '='; *(d++) = '2'; *(d++) = 'F'; break;
		case '=': *(d++) = '='; *(d++) = '3'; *(d++) = 'D'; break;
		case '\x0A': *(d++) = '='; *(d++) = '0'; *(d++) = 'A'; break;
		case '\x0D': *(d++) = '='; *(d++) = '0'; *(d++) = 'D'; break;
			/*case ' ': *(d++) = '='; *(d++) = '2'; *(d++) = '0'; break;*/
		default: 
			if(*s < ' ') {
				gchar tmp[6];
				g_snprintf(tmp,5,"=%2X",(unsigned char)*s);
				*(d++) = tmp[0]; *(d++) = tmp[1]; *(d++) = tmp[2];
			} else
				*(d++) = *s;
			break;
		}
		++s;
	}
	*d = '\0';
}

/** GCompareFunc for finding a record */

static gint 
match_record_id(MemoLocalRecord *a,int *b) 
{
	if(!a) return -1;
	return !(a->local.ID == *b);
}



/* from sync-memodir */
static gchar * 
newfilename(MemoLocalRecord * r) 
{
	static gchar name[256];
	static gchar buf[256];
	char *rec, *end;
	int i;
	
	rec = r->record;
	end = &r->record[r->length];

	/* use first line as file name
	 * but change whitespace chars into '.'
	 */
	while( rec < end && isspace(*rec) )		/* skip whitespace */
		++rec;
	
	/* Go only upto 63 characters for file names. That should be enough. 
	   protect_name() can maximally generate 4*n character name
	*/
	for( i = 0; rec < end && i < 63; ++i, ++rec) {
		if( *rec == '\n' )
			break;
		else
			name[i] = *rec;
	}
	name[i] = '\0';
	
	if( *name == '\0' )				/* an empty memo */
		strcpy( name, "empty" );
	
	strcpy(buf,name);
	protect_name(name,buf);
	/*strcpy(name,g_strecape(name));*/
	return name;
}

/** 
    generates a pathname for a category 
 */
static gchar *
category_path(int category,GnomePilotConduitStandardAbs *abs) 
{
	static gchar filename[FILENAME_MAX];
	gchar buf[64];
	
	if(category==16) 
		strcpy(buf,"Archived");
	else
		protect_name(buf,GET_CONDUIT_DATA(abs)->ai.category.name[category]);
  
	g_snprintf(filename,FILENAME_MAX-1,"%s/%s",
		   GET_CONDUIT_CFG(abs)->dir,
		   buf);
	     
	return filename;
}

static void 
generate_name(MemoLocalRecord *local,GnomePilotConduitStandardAbs *abs) 
{
	struct stat stbuf;
	gchar *fname;
	int i = 1;
	
	local->filename = (gchar*)g_malloc(FILENAME_MAX);
	fname = newfilename(local);

	g_snprintf(local->filename,FILENAME_MAX-1,"%s/%s",
		   category_path(local->local.archived?16:local->category,abs),
		   fname);

	/* file name already exists, tack on a unique number */
	if(stat(local->filename,&stbuf) != 0) return;

	for (i = 2; ; ++i) {
		g_snprintf(local->filename,FILENAME_MAX-1, "%s/%s.%d", 
			   category_path(local->local.archived?16:local->category,abs),
			   fname,i);
		if (stat(local->filename, &stbuf) != 0)
			return;
	}
}

/** generates a name for a .ids filename
    returns NULL if no name specified for given category (which is a FIXME:)
*/
static gchar *
idfile_name(int cnt,GnomePilotConduitStandardAbs *abs) 
{
	static gchar filename[FILENAME_MAX];

	if(category_path(cnt,abs))
		g_snprintf(filename,FILENAME_MAX-1,"%s/.ids",category_path(cnt,abs));
	else
		return NULL;

	return filename;
}

/**
   spools a records, gets a unique filename based on the first line and saves it
*/

static void 
spool_foreach(MemoLocalRecord *local,GnomePilotConduitStandardAbs *abs)
{
	int f;
	gchar entry[128];
	mode_t mode;
	
	if(!local) return;
	if (local->local.attr==GnomePilotRecordDeleted) return;
	
	LOG("spool_foreach");
	generate_name(local,abs);
	
	if(local->local.secret) mode=GET_CONDUIT_CFG(abs)->secret_mode;
	else mode=GET_CONDUIT_CFG(abs)->file_mode;
	f = open(local->filename,O_WRONLY|O_CREAT|O_TRUNC,mode);
	g_return_if_fail(f!=-1);
	write(f,local->record,local->length-1);
	close(f);
	
	f = open(idfile_name(local->category,abs),O_WRONLY|O_APPEND|O_CREAT,0600);
	g_return_if_fail(f!=-1);
	g_snprintf(entry,127,"%lu:%d:%lu;%s\n",local->local.ID,local->local.secret,time(NULL),local->filename);
	write(f,entry,strlen(entry));
	close(f);
}

/** 
    moves the Memo directory to dir.backup, and recreates dir, used by spool_records
*/
static void 
backup_directory(GnomePilotConduitStandardAbs *abs) 
{
	gchar filename[FILENAME_MAX];
	gchar tmp[FILENAME_MAX];
	strcpy(tmp,GET_CONDUIT_CFG(abs)->dir);
	while(tmp[strlen(tmp)-1]=='/') tmp[strlen(tmp)-1]='\0';

	g_snprintf(filename,FILENAME_MAX,"%s.old",tmp); 
	LOG("renaming %s to %s",GET_CONDUIT_CFG(abs)->dir,filename);
	if(rename(GET_CONDUIT_CFG(abs)->dir,filename)!=0) {
		LOG("%s",g_strerror(errno));
	}
	mkdir(GET_CONDUIT_CFG(abs)->dir,GET_CONDUIT_CFG(abs)->dir_mode);
}

/** obliterates the backup dir, called by spool_records after memos are saved */
static void 
nuke_backup(GnomePilotConduitStandardAbs *abs) 
{
	DIR *dir,*subdir;
	struct dirent *de;
	gchar filename[FILENAME_MAX],filename2[FILENAME_MAX];
	gchar tmp[FILENAME_MAX];
	
	strcpy(tmp,GET_CONDUIT_CFG(abs)->dir);
	while(tmp[strlen(tmp)-1]=='/') tmp[strlen(tmp)-1]='\0';
	
	g_snprintf(filename,FILENAME_MAX-1,"%s.old",tmp);
	if((dir=opendir(filename))==NULL) {
		LOG("nuke_backup cannot open %s",GET_CONDUIT_CFG(abs)->dir);
		return;
	}
	while((de=readdir(dir))) {
		if(strcmp(de->d_name,".")==0) continue;
		if(strcmp(de->d_name,"..")==0) continue;
		if(strcmp(de->d_name,".categories")==0) continue;
		g_snprintf(filename,FILENAME_MAX-1,"%s.old/%s",tmp,de->d_name); /* backup ensures that GET_CONDUIT_CFG(abs)->dir doesn't end with / */
		if((subdir=opendir(filename))==NULL) {
			LOG("nuke_backup cannot open subdir %s",filename);
			continue;
		}
		while((de=readdir(subdir))) {
			g_snprintf(filename2,FILENAME_MAX-1,"%s/%s",filename,de->d_name);
			unlink(filename2);
		}
		closedir(subdir);
		if(rmdir(filename)<0) {
			LOG("cannot rmdir %s",filename);
		}
	}
	closedir(dir);
	g_snprintf(filename,FILENAME_MAX-1,"%s.old/.categories",tmp);
	unlink(filename);
	g_snprintf(filename,FILENAME_MAX-1,"%s.old",tmp);
	if(rmdir(filename)<0) {
		LOG("cannot rmdir %s",filename);
	}
}

/**
   saves all the records, called by the destructor.
   First it deletes all the files
*/

static void 
spool_records(GnomePilotConduitStandardAbs *abs) 
{
	int cnt;
	gchar filename[FILENAME_MAX];
	gchar entry[128];
	int f;

	g_return_if_fail(GET_CONDUIT_CFG(abs)->dir != NULL);
	/* backup, in case we die before all is spooled */
	backup_directory(abs);
	g_snprintf(filename,FILENAME_MAX-1,"%s/.categories",GET_CONDUIT_CFG(abs)->dir);
	f = open(filename,O_WRONLY|O_APPEND|O_CREAT,0600);

	mkdir(GET_CONDUIT_CFG(abs)->dir,GET_CONDUIT_CFG(abs)->dir_mode);
	for(cnt=0;cnt<17;cnt++) {
		mkdir(category_path(cnt,abs),GET_CONDUIT_CFG(abs)->dir_mode);
		g_snprintf(entry,127,"%d;%s\n",cnt,category_path(cnt,abs));
		write(f,entry,strlen(entry));
	}
	close(f);

	g_list_foreach(GET_CONDUIT_DATA(abs)->records,(GFunc)spool_foreach,abs);

	nuke_backup(abs);
};


/**
   free a record, called by destroy_abs for all records
*/

static void 
free_records_foreach(MemoLocalRecord *local, gpointer whatever) 
{
	if(!local) return;
	if(local->record)  g_free(local->record);
	if(local->filename) g_free(local->filename);
	local->record=NULL;
	local->filename=NULL;
	g_free(local);
}

/**
   frees and destroys a record
*/

static void 
destroy_records_foreach(MemoLocalRecord *local,gpointer whatever) 
{
	if(!local) return;
	if(local->filename) unlink(local->filename);
}

/**
   marks a record as deleted
*/

static void 
delete_records_foreach(MemoLocalRecord *local,gpointer whatever) 
{
	if(!local) return;
	local->local.attr = GnomePilotRecordDeleted;
}

/**
   deletes a record if attr says so
*/

static void 
purge_records_foreach(MemoLocalRecord *local,gpointer whatever) 
{
	if(!local) return;
	if(local->local.attr == GnomePilotRecordDeleted) {
		destroy_records_foreach(local,whatever);
	}
}

/** 
  foreach function that sets the .next pointer 
*/
static void 
iterate_foreach(MemoLocalRecord *local,IterateData *d) 
{
	gboolean accept;
	if(!local) return; 
	accept = TRUE;
	
	local->next = NULL;

	/* only check if archived = 0 | = 1 */
	if(d->archived>=0) 
		if(d->archived != local->local.archived) accept = FALSE;
	if(d->flag>=0)
		if(d->flag != local->local.attr) accept = FALSE;

	if(local->ignore == TRUE) accept = FALSE;

	if(accept) { 
		if(d->prev) 
			d->prev->next = local;
		else
			d->first = local;
		d->prev = local;
	}
}

static void 
create_deleted_record_foreach(gchar *key, LoadInfo *li,GList **records) 
{
	MemoLocalRecord *local;
	local = (MemoLocalRecord*)g_malloc(sizeof(MemoLocalRecord));
	local->local.ID = li->id;
	local->local.secret = li->secret;
	local->next = NULL;
	local->category = 0;
	local->local.attr = GnomePilotRecordDeleted;
	local->local.archived = 0;
	local->length = 0;
	local->record = NULL;
	local->filename = g_strdup(key);
	local->ignore = FALSE;

	*records = g_list_append(*records,local);
}

/** 
    fills a record from a file, assuming at least filename is set properly
    archived and secret aren't touched
  */ 
static void 
load_record(GnomePilotConduitStandardAbs *abs,MemoLocalRecord *local) 
{
	FILE *rec;
	struct stat st;
	
	local->record = NULL;
	local->length = 0;
	local->local.attr = GnomePilotRecordNothing;
	/* stat the file and get modtime */
	if(stat(local->filename,&st)<0) {
		LOG("load_record cannot stat record file \"%s\"",local->filename);
		local->local.attr = GnomePilotRecordDeleted;
		return;
	}
	if(st.st_mtime > local->mtime) { 
		if(local->local.ID == 0)
			local->local.attr = GnomePilotRecordNew;
		else
			local->local.attr = GnomePilotRecordModified;
	}
	
	/* open the file and read the contents */
	if((rec=fopen(local->filename,"rb"))==NULL) {
		local->local.attr = GnomePilotRecordDeleted; /* FIXME: is this safe ? what if the access is wrong ? */
		return;
	}
	
	fseek(rec,0L,SEEK_END);
	local->length = ftell(rec)+1;
	rewind(rec);
	local->record = (unsigned char*)g_malloc(local->length);
	fread(local->record,local->length-1,1,rec);
	local->record[local->length-1]='\0';
	fclose(rec);
}

/** loads the .categories file */
static GHashTable *
load_categories(GnomePilotConduitStandardAbs *abs) 
{
	GHashTable *categories;
	FILE *f;
	gchar filename[FILENAME_MAX];
	
	LOG("load_categories");
	categories = g_hash_table_new(g_str_hash,g_str_equal);
	g_snprintf(filename,FILENAME_MAX-1,"%s/.categories",GET_CONDUIT_CFG(abs)->dir);

	if((f = fopen(filename,"r"))==NULL) return NULL;
	
	while(fgets(filename,FILENAME_MAX-1,f)!=NULL) {
		gint cat;
		gchar *ptr;
		
		cat = atol(filename);
		ptr = strchr(filename,';');
		if(ptr) {
			ptr++;
			ptr[strlen(ptr)-1] = '\0';
			g_hash_table_insert(categories,g_strdup(ptr),GINT_TO_POINTER(cat));
		}
	}
	fclose(f);
	return categories;
}

static gboolean 
ignore_file_name(GnomePilotConduitStandardAbs* abs, gchar* name)
{
	ConduitCfg *cfg;
	if(name[0]=='.') return TRUE;

	cfg=GET_CONDUIT_CFG(abs);
	if(cfg->ignore_start && strlen(cfg->ignore_start)>0 &&
	   strncmp(name,cfg->ignore_start,strlen(cfg->ignore_start))==0)
		return TRUE;
	
	if(cfg->ignore_end && strlen(cfg->ignore_end)>0 &&
	   strcmp(name+strlen(name)-strlen(cfg->ignore_end),cfg->ignore_end)==0)
		return TRUE;
	
	return FALSE;
}

static void 
free_str_foreach(gchar *s, gpointer whatever) 
{
	g_free(s);
}

static void
free_loadinfo_foreach(LoadInfo *info, gpointer whatever) 
{
	g_free(info);
}

/** loads the records into the abs structure */
static void 
load_records(GnomePilotConduitStandardAbs *abs) 
{
	FILE *idfile;
	DIR *dir;
	struct dirent *de;
	gchar filename[FILENAME_MAX];
	gchar entry[1024];
	gchar *ptr;
	MemoLocalRecord *local;
	GHashTable *categories;
	int category;
	int total=0,updated=0,deleted=0,newrecs=0;
	
	LOG("load_records");

	if((dir=opendir(GET_CONDUIT_CFG(abs)->dir))==NULL) {
		LOG("load_records cannot open %s",GET_CONDUIT_CFG(abs)->dir);
		return;
	}
	if((categories = load_categories(abs))==NULL) {
		LOG("no categories, no records");
		closedir(dir);
		return;
	}
	while((de=readdir(dir))) {
		GHashTable *loadinfo;
		LoadInfo *info;
		
		if(strcmp(de->d_name,".")==0) continue;
		if(strcmp(de->d_name,"..")==0) continue;
		if(strcmp(de->d_name,".categories")==0) continue;
		
		/* first load id:sec:names from the .ids file */
		loadinfo = g_hash_table_new(g_str_hash,g_str_equal);

		g_snprintf(filename,FILENAME_MAX-1,"%s/%s",GET_CONDUIT_CFG(abs)->dir,de->d_name);
		category = GPOINTER_TO_INT(g_hash_table_lookup(categories,filename));
		if(category<0 || category > 16) category = 0;
		
		g_snprintf(filename,FILENAME_MAX-1,"%s/%s/.ids",GET_CONDUIT_CFG(abs)->dir,de->d_name);
		if((idfile=fopen(filename,"rt"))!=NULL) {
			while(fgets(entry,1024,idfile)!=NULL) {
				gchar *key;
				info = (LoadInfo*)g_malloc(sizeof(LoadInfo));
				sscanf(entry,"%lu:%d:%lu;",&info->id,&info->secret,&info->mtime);
				ptr = strchr(entry,';'); ptr++;
				
				key = g_strdup(ptr);
				key[strlen(key)-1] = '\0';
				
				g_hash_table_insert(loadinfo,key,info);
			}
			fclose(idfile);
		}
		
		/* now check all files in directory */
		{
			DIR *l_dir;
			struct dirent *l_de;
			g_snprintf(filename,FILENAME_MAX-1,"%s/%s",GET_CONDUIT_CFG(abs)->dir,de->d_name);
			if((l_dir=opendir(filename))==NULL) {
				LOG("load_records cannot open %s",filename);
			} else {
				LOG("Reading directory %s",filename);
				while((l_de=readdir(l_dir))) {
					if(ignore_file_name(abs,l_de->d_name)){
						LOG("Ignoring %s",l_de->d_name);
						continue;
					}
					
					local = g_new0(MemoLocalRecord,1);
					local->filename = g_strdup_printf("%s/%s",filename,l_de->d_name);
					
					if((info=g_hash_table_lookup(loadinfo,local->filename))!=NULL) {
						local->local.ID = info->id;
						local->local.secret = info->secret;
						local->mtime=info->mtime;
						g_hash_table_remove(loadinfo,local->filename);
						g_free(info);
					} else {
						local->local.ID = 0;
						local->local.secret = 0;
						local->mtime=0;
					}
					local->local.archived = 0;
					local->category = category;
					local->ignore = FALSE;
					load_record(abs,local);
					
					GET_CONDUIT_DATA(abs)->records = g_list_append(GET_CONDUIT_DATA(abs)->records,local);
					total++;
					switch(local->local.attr) {
					case GnomePilotRecordDeleted: deleted++; break;
					case GnomePilotRecordModified: updated++;break;
					case GnomePilotRecordNew: newrecs++; break;
					default: break;
					}
					LOG("Found local file %s, state %d",l_de->d_name,local->local.attr);
				}
				closedir(l_dir);      
			}
		}
		if(g_hash_table_size(loadinfo)>0) {
			deleted += g_hash_table_size(loadinfo);
			g_hash_table_foreach(loadinfo,(GHFunc)create_deleted_record_foreach,&GET_CONDUIT_DATA(abs)->records);
			g_hash_table_foreach(loadinfo,(GHFunc)free_loadinfo_foreach,NULL);
		}
		g_hash_table_destroy(loadinfo);
	}
	closedir(dir);
	g_hash_table_foreach(categories,(GHFunc)free_str_foreach,NULL);
	g_hash_table_destroy(categories);
		
	gnome_pilot_conduit_standard_abs_set_num_local_records(abs,total);
	gnome_pilot_conduit_standard_abs_set_num_updated_local_records(abs,updated);
	gnome_pilot_conduit_standard_abs_set_num_new_local_records(abs,newrecs);
	gnome_pilot_conduit_standard_abs_set_num_deleted_local_records(abs,deleted);
	
	LOG("records: total = %d updated = %d new = %d deleted = %d",total,updated,newrecs,deleted);
}


static gint 
pre_sync(GnomePilotConduit *c, GnomePilotDBInfo *dbi) 
{
	int l;
	unsigned char *buf;
  
	g_message ("MemoFile Conduit v %s", CONDUIT_VERSION);

	LOG("PreSync");

	GET_CONDUIT_DATA(c)->dbi=dbi;
  
	buf = (unsigned char*)g_malloc(0xffff);
	if((l=dlp_ReadAppBlock(dbi->pilot_socket,dbi->db_handle,0,(unsigned char *)buf,0xffff))<0) {
		LOG("dlp_ReadAppBlock(...) failed");
		g_free(buf);
		return -1;
	}
	unpack_MemoAppInfo(&(GET_CONDUIT_DATA(c)->ai),buf,l);
	g_free(buf);

	if (GET_CONDUIT_CFG(c)->dir==NULL) {
		return -1;
	}

	load_records((GnomePilotConduitStandardAbs*)c); 

	/* If local store is empty force the slow sync. */
	if(g_list_length(GET_CONDUIT_DATA(c)->records)==0){
		gnome_pilot_conduit_standard_set_slow((GnomePilotConduitStandard*)c);
	}
	return 0;
}

static gint
match_record	(GnomePilotConduitStandardAbs *abs,
		 MemoLocalRecord **local,
		 PilotRecord *remote,
		 gpointer data)
{
	GList *tmp;
	LOG("MatchRecord");

	g_return_val_if_fail(local!=NULL,-1);
	g_return_val_if_fail(remote!=NULL,-1);

	tmp = g_list_find_custom(GET_CONDUIT_DATA(abs)->records,(gpointer)&remote->ID,(GCompareFunc)match_record_id);
	if(tmp==NULL) 
		*local = NULL;
	else {
		*local = tmp->data;
	}
	return 0;
}

static gint
free_match	(GnomePilotConduitStandardAbs *abs,
		 MemoLocalRecord **local,
		 gpointer data)
{
	LOG("FreeMatch");

	g_return_val_if_fail(local!=NULL,-1);
	g_return_val_if_fail(*local!=NULL,-1);
	
	*local = NULL;
	return 0;
}

static gint
archive_local (GnomePilotConduitStandardAbs *abs,
	       MemoLocalRecord *local,
	       gpointer data)
{
	LOG("ArchiveLocal");
	g_return_val_if_fail(local!=NULL,-1);
	local->local.archived = 1; 
	return 0;
}

static gint
archive_remote (GnomePilotConduitStandardAbs *abs,
		MemoLocalRecord *local,
		PilotRecord *remote,
		gpointer data)
{
	LOG("ArchiveRemote");
	g_return_val_if_fail(remote!=NULL,-1);

	if(local!=NULL)  local->local.archived = 1; 
	else
	{
		/* 
		   FIXME: what to do here? The following doesn't quite work. but is something
		   similar needed? Or is this belong here or in general abs-conduit?
		   store_remote(abs,remote,data);
		*/
	}
	return 0;
}

static gint
store_remote (GnomePilotConduitStandardAbs *abs,
	      PilotRecord *remote,
	      gpointer data)
{
	MemoLocalRecord *local;
	GList *tmp;
	ConduitData *cd;

	LOG("StoreRemote");
	
	g_return_val_if_fail(remote!=NULL,-1);
	
	cd = GET_CONDUIT_DATA(abs);
	tmp = g_list_find_custom(cd->records,(gpointer)&remote->ID,(GCompareFunc)match_record_id);
  
	if(tmp==NULL) {
		/* new record */
		local = (MemoLocalRecord*)g_malloc(sizeof(MemoLocalRecord));
		cd->records = g_list_append(cd->records,local);
		
	} else {
		local = tmp->data;
		if(local->record) {
			g_free(local->record);
			local->record=NULL;
		}
	}

	local->local.ID = remote->ID; 
	local->local.attr = remote->attr;
	local->local.archived = remote->archived;
	local->local.secret = remote->secret;
	local->length = remote->length;
	local->category = remote->category; 
	local->ignore = FALSE;
	if(local->length) {
		/* paranoia check */
		if (remote->record==NULL) {
			LOG("record with NULL contents encountered");
			local->record = NULL;
			local->length = 0;
		} else {
			local->record = (unsigned char*)g_malloc(local->length);
			memcpy(local->record,remote->record,local->length);
		}
	}
	
	return 0;
}

static gint
clear_status_archive_local (GnomePilotConduitStandardAbs *abs,
			    MemoLocalRecord *local,
			    gpointer data)
{
	LOG("ClearStatusArchiveLocal");
	g_return_val_if_fail(local!=NULL,-1);
	local->local.archived=0;
	return 0;
}

static gint
iterate (GnomePilotConduitStandardAbs *abs,
	 MemoLocalRecord **local,
	 gpointer data)
{
	LOG("Iterate");
	g_return_val_if_fail(local!=NULL,-1);
	if(!*local) {
		/* setup the links */
		IterateData *d;
		d = new_iterate_data(-1,-1);
		g_list_foreach(GET_CONDUIT_DATA(abs)->records,(GFunc)iterate_foreach,d);
		*local = d->first;
	} else {
		*local = (*local)->next;
	}
	if(*local==NULL) return 0;
	else return 1;
}

static gint
iterate_specific (GnomePilotConduitStandardAbs *abs,
		  MemoLocalRecord **local,
		  gint flag,
		  gint archived,
		  gpointer data)
{
	LOG("IterateSpecific, *local %s NULL,    flag = %d, archived = %d",
	    *local==NULL?"==":"!=",flag,archived);
	g_return_val_if_fail(local!=NULL,-1);
	if(! *local) {
		/* setup the links */
		IterateData *d;
		d = new_iterate_data(flag,archived);
		if(g_list_length(GET_CONDUIT_DATA(abs)->records)>0) {
			g_list_foreach(GET_CONDUIT_DATA(abs)->records,(GFunc)iterate_foreach,d);
			*local = d->first;
		} else {
			*local=NULL;
		}
	} else {
		*local = (*local)->next;
	}
	if(*local == NULL) return 0;
	else return 1;
}

static gint
purge (GnomePilotConduitStandardAbs *abs,
       gpointer data)
{
	LOG("Purge");

	g_list_foreach(GET_CONDUIT_DATA(abs)->records,
		       (GFunc)purge_records_foreach,
		       GET_CONDUIT_DATA(abs)->records);
	spool_records(abs);
	
	return 0;
}

static gint
set_status (GnomePilotConduitStandardAbs *abs,
	    MemoLocalRecord *local,
	    gint status,
	    gpointer data)
{
	LOG("SetStatus %d",status);
	g_return_val_if_fail(local!=NULL,-1);
	local->local.attr = status; 
	if(status==GnomePilotRecordDeleted) local->ignore=TRUE;
	return 0;
}

static gint
set_archived (GnomePilotConduitStandardAbs *abs,
	      MemoLocalRecord *local,
	      gint archived,
	      gpointer data)
{
	LOG("SetArchived");
	g_return_val_if_fail(local!=NULL,-1);
	local->local.archived = archived; 
	return 0;
}

static gint
set_pilot_id (GnomePilotConduitStandardAbs *abs,
	      MemoLocalRecord *local,
	      guint32 ID,
	      gpointer data)
{
	LOG("SetPilotId, ID = %u",ID);
	g_return_val_if_fail(local!=NULL,-1);
	local->local.ID = ID; 
	return 0;

}
static gint
compare (GnomePilotConduitStandardAbs *abs,
	 MemoLocalRecord *local,
	 PilotRecord *remote,
	 gpointer data)
{
	LOG("Compare");
	
	g_return_val_if_fail(local!=NULL,-1);
	g_return_val_if_fail(remote!=NULL,-1);
	if(local->record==NULL || remote->record == NULL) return -1;
	return strncmp(local->record,remote->record,local->length);
}

static gint
compare_backup (GnomePilotConduitStandardAbs *abs,
		MemoLocalRecord *local,
		PilotRecord *remote,
		gpointer data)
{
	LOG("CompareBackup");
	g_return_val_if_fail(local!=NULL,-1);
	g_return_val_if_fail(remote!=NULL,-1);
	if(local->record==NULL || remote->record == NULL) return -1;

	return -1;
}
static gint
free_transmit (GnomePilotConduitStandardAbs *abs,
	       MemoLocalRecord *local,
	       PilotRecord **remote,
	       gpointer data)
{
	LOG("FreeTransmit");
	g_return_val_if_fail(local!=NULL,-1);
	g_return_val_if_fail(remote!=NULL,-1);
	g_return_val_if_fail(*remote!=NULL,-1);
  
	if((*remote)->record) g_free((*remote)->record); 
	*remote = NULL;
	return 0;
}

static gint
delete_all (GnomePilotConduitStandardAbs *abs,
	    gpointer data)
{
	LOG("DeleteAll");
	g_list_foreach(GET_CONDUIT_DATA(abs)->records,
		       (GFunc)delete_records_foreach,
		       NULL);
	return 0;
}

static gint
transmit (GnomePilotConduitStandardAbs *abs,
	  MemoLocalRecord *local,
	  PilotRecord **remote,
	  gpointer data)
{
	static PilotRecord p;
	LOG("Transmit, local %s NULL",local==NULL?"==":"!=");
	
	g_return_val_if_fail(local!=NULL,-1);
	g_return_val_if_fail(remote!=NULL,-1);
	
	p.record = NULL;

	p.ID = local->local.ID;
	p.attr = local->local.attr;
	p.archived = local->local.archived;
	p.secret = local->local.secret;
	p.length = local->length;
	p.category = local->category;
	if(p.length) {
		p.record = (unsigned char*)g_malloc(p.length);
		memcpy(p.record,local->record,p.length);
	}
	*remote = &p;
	return 0;
}

GnomePilotConduit *
conduit_get_gpilot_conduit (guint32 pilotId)
{
	GtkObject *retval;

	ConduitData *cd = g_new0(ConduitData,1);
	ConduitCfg *cc;
	
	cd->records=NULL;
  
	retval = gnome_pilot_conduit_standard_abs_new ("MemoDB", 0x6d656d6f);
	g_assert (retval != NULL);
	gnome_pilot_conduit_construct(GNOME_PILOT_CONDUIT(retval),"memo_file");

	
	LOG("creating memo_file conduit");

	g_assert (retval != NULL);
	gtk_signal_connect (retval, "match_record", (GtkSignalFunc) match_record, NULL);
	gtk_signal_connect (retval, "free_match", (GtkSignalFunc) free_match, NULL);
	gtk_signal_connect (retval, "archive_local", (GtkSignalFunc) archive_local, NULL);
	gtk_signal_connect (retval, "archive_remote", (GtkSignalFunc) archive_remote, NULL);
	gtk_signal_connect (retval, "store_remote", (GtkSignalFunc) store_remote, NULL);
	gtk_signal_connect (retval, "clear_status_archive_local", (GtkSignalFunc) clear_status_archive_local, NULL);
	gtk_signal_connect (retval, "iterate", (GtkSignalFunc) iterate, NULL);
	gtk_signal_connect (retval, "iterate_specific", (GtkSignalFunc) iterate_specific, NULL);
	gtk_signal_connect (retval, "purge", (GtkSignalFunc) purge, NULL);
	gtk_signal_connect (retval, "set_status", (GtkSignalFunc) set_status, NULL);
	gtk_signal_connect (retval, "set_archived", (GtkSignalFunc) set_archived, NULL);
	gtk_signal_connect (retval, "set_pilot_id", (GtkSignalFunc) set_pilot_id, NULL);
	gtk_signal_connect (retval, "compare", (GtkSignalFunc) compare, NULL);
	gtk_signal_connect (retval, "compare_backup", (GtkSignalFunc) compare_backup, NULL);
	gtk_signal_connect (retval, "free_transmit", (GtkSignalFunc) free_transmit, NULL);
	gtk_signal_connect (retval, "delete_all", (GtkSignalFunc) delete_all, NULL);
	gtk_signal_connect (retval, "transmit", (GtkSignalFunc) transmit, NULL);
	gtk_signal_connect (retval, "pre_sync", (GtkSignalFunc) pre_sync, NULL);

	load_configuration(&cc,pilotId);
	gtk_object_set_data(GTK_OBJECT(retval),OBJ_DATA_CONFIG,cc);
	gtk_object_set_data(GTK_OBJECT(retval),OBJ_DATA_CONDUIT,cd);

	if(cc->dir==NULL) {
		g_warning(_("No dir specified. Please run memo_file conduit capplet first."));
		gnome_pilot_conduit_error(GNOME_PILOT_CONDUIT(retval),
					  _("No dir specified. Please run memo_file conduit capplet first."));
		/* FIXME: the following is probably the right way to go, 
		   but with current state it doesn't work. (capplet wouldn't start up)
                destroy_configuration(&cc);
		return NULL;
		*/
	} 
	
	if(cc->open_secret) gnome_pilot_conduit_standard_abs_set_db_open_mode(GNOME_PILOT_CONDUIT_STANDARD_ABS(retval),
									      dlpOpenReadWrite|dlpOpenSecret);
	return GNOME_PILOT_CONDUIT (retval);
}

void
conduit_destroy_gpilot_conduit (GnomePilotConduit *conduit)
{
	ConduitData *cd=GET_CONDUIT_DATA(conduit);
	ConduitCfg  *cfg=GET_CONDUIT_CFG(conduit);
	LOG("destroying memo_file conduit");
	
	g_list_foreach(cd->records,(GFunc)free_records_foreach,cd->records);
	g_list_free(cd->records);
	g_free(cd);
	
	destroy_configuration(&cfg);
	gtk_object_destroy (GTK_OBJECT (conduit));
}


