#include <glib.h>
#include <gmodule.h>
#include <pi-socket.h>
#include <pi-sockaddr.h>
#include <pi-source.h>
#include <pi-sync.h>
#include <pi-dlp.h>

#include <libgnome/libgnome.h>

#include "orbit_daemon_glue.h"
#include "manager.h"
#include "queue_io.h"
#include "gnome-pilot-conduit-backup.h"
#include "gnome-pilot-conduit-file.h"
#include "gnome-pilot-conduit-standard.h"
#include "gpilot-gui.h"

#define PROGRESS_STEPPING 10
#define USE_GMODULE 

#ifndef USE_GMODULE
/* we're using dlopen instead of g_module_open, include nessecary .h's */
#include <dlfcn.h>
#endif

/* File and Backup conduits are handled completely separately from the
   other conduits */
/* Typedefs. */
typedef struct GnomePilotConduitManagerInfo {
	gchar *name;
	GModule *dlhandle;
	conduit_destroy_gpilot_conduit destroy_gpilot_conduit;

	gchar *config_name;
	GnomePilotConduitSyncType sync_type;
	GnomePilotConduitSyncType first_sync_type;
	gboolean slow;
} GnomePilotConduitManagerInfo;

typedef struct {
	GnomePilotConduit *conduit;
	GPilotContext *context;
	struct PilotUser *pu;
	int pilot_socket;
        GList *conduit_list;
        GnomePilotSyncStamp *syncstamp;
} GnomePilotInstallCarrier;

typedef gint (*iterate_func)(GnomePilotConduitStandard *,GnomePilotDBInfo *);

/* Prototypes */

static GnomePilotConduit *gpilot_load_conduit (guint32 pilotId,
					       gchar *conduit_fn,
					       GnomePilotConduitManagerInfo *cmaninfo);
static void unload_foreach (gpointer *conduit, gpointer unused);
static gint find_matching_conduit_compare_func (GnomePilotConduit *a,
						GnomePilotDBInfo *b);
static GnomePilotConduit *find_matching_conduit (GList *conlist, GnomePilotDBInfo *dbinfo);
static void backup_foreach (GnomePilotConduitBackup *conduit, GnomePilotDBInfo *info);
static gint iterate_dbs (gint pfd,
			 GnomePilotSyncStamp *stamp,
			 struct PilotUser *pu,
			 GList *conlist,
			 GList *bconlist,
			 iterate_func iter,
			 GPilotContext *context);
static gint conduit_synchronize (GnomePilotConduitStandard *conduit,
				 GnomePilotDBInfo *dbinfo);
static gint conduit_copy_to_pilot (GnomePilotConduitStandard *conduit,
				   GnomePilotDBInfo *dbinfo);
static gint conduit_copy_from_pilot (GnomePilotConduitStandard *conduit,
				     GnomePilotDBInfo *dbinfo);
static gint conduit_sync_default (GnomePilotConduitStandard *conduit,
			     GnomePilotDBInfo *dbinfo);
/* This is defined in gpilotd.  But we need to use it... */
gchar *pilot_name_from_id(guint32 id,GPilotContext *context);

/**************************************************************************/
/* These are various match methods, used to do g_list_find's              */

static gint
find_matching_conduit_compare_func (GnomePilotConduit *conduit,
				    GnomePilotDBInfo *dbinfo)
{
	if (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit)) {
		if ((gnome_pilot_conduit_standard_get_creator_id (GNOME_PILOT_CONDUIT_STANDARD (conduit)) ==
		     PI_DBINFO (dbinfo)->creator) &&
		    (strcmp(gnome_pilot_conduit_standard_get_db_name (GNOME_PILOT_CONDUIT_STANDARD (conduit)),
			    PI_DBINFO(dbinfo)->name) == 0)) {
			return 0;
		}
	}
	return -1;
}

static GnomePilotConduit *
find_matching_conduit (GList *conlist,
		       GnomePilotDBInfo *dbinfo)
{
	GList *elem;
	elem = g_list_find_custom(conlist,
				  dbinfo,
				  (GCompareFunc) find_matching_conduit_compare_func);
	if (elem)
		return GNOME_PILOT_CONDUIT (elem->data);
	else
		return NULL;
}

static gint 
find_matching_conduit_compare_func_name(GnomePilotConduit *conduit,
					gchar *name) 
{
  if(GNOME_IS_PILOT_CONDUIT_STANDARD(conduit)) {
    if(strcmp(gnome_pilot_conduit_standard_get_db_name(GNOME_PILOT_CONDUIT_STANDARD(conduit)),
	      name)==0) {
      return 0;
    }
  }
  return -1;
}

static GnomePilotConduit*
find_matching_conduit_name(GList *conlist,gchar *name)
{
  GList *elem;
  elem=g_list_find_custom(conlist,name,
			  (GCompareFunc)find_matching_conduit_compare_func_name);
  if(elem)
    return GNOME_PILOT_CONDUIT(elem->data);
  else
    return NULL;
}

/**************************************************************************/
/* Here comes conduit signal handler, used to call methods from 
   orbit_daemon_glue                                                      */

static void 
conduit_progress_callback(GnomePilotConduit *conduit, 
			  gint total,
			  gint current,
			  GnomePilotDBInfo *db) {
	static gint top = 0;
	gfloat tot_f,cur_f;
	gint pct;

	tot_f = total;
	cur_f = current;
	pct = abs(cur_f/(tot_f/100));
	
	if (pct < PROGRESS_STEPPING) top = 0; /* ensures resetting top */
	if (pct >= top+PROGRESS_STEPPING) {
		/* for every PROGRESS_STEPPING%, call the corba thingy */
		top = pct;
		orbed_notify_conduit_progress(db->pilotInfo->name,
					      conduit,
					      current,
					      total);
	}
}

static void 
conduit_message_callback(GnomePilotConduit *conduit, 
			 gchar *message,
			 GnomePilotDBInfo *db) {
	orbed_notify_conduit_message(db->pilotInfo->name,
				     conduit,
				     message);
}

static void 
conduit_error_callback(GnomePilotConduit *conduit, 
		       gchar *message,
		       GnomePilotDBInfo *db) {
	orbed_notify_conduit_error(db->pilotInfo->name,
				   conduit,
				   message);
}

/**************************************************************************/

static void
backup_foreach (GnomePilotConduitBackup *conduit, GnomePilotDBInfo *info)
{

	guint progress_h,error_h,message_h;

	if (GNOME_IS_PILOT_CONDUIT_BACKUP (conduit) == FALSE) {
		g_error(_("non-backup conduit in backup conduit list"));
		return;
	}
	g_assert (info != NULL);

	progress_h = gtk_signal_connect(GTK_OBJECT(conduit),"progress",
					(GtkSignalFunc)conduit_progress_callback,
					info);
	error_h = gtk_signal_connect(GTK_OBJECT(conduit),"error",
				     (GtkSignalFunc)conduit_error_callback,
				     info);
	message_h = gtk_signal_connect(GTK_OBJECT(conduit),"message",
				       (GtkSignalFunc)conduit_message_callback,
				       info);

	gnome_pilot_conduit_backup_backup (conduit, info);

	gtk_signal_disconnect(GTK_OBJECT(conduit),progress_h);
	gtk_signal_disconnect(GTK_OBJECT(conduit),error_h);
	gtk_signal_disconnect(GTK_OBJECT(conduit),message_h);
}

static gint
iterate_dbs (gint pfd,
	     GnomePilotSyncStamp *stamp,
	     struct PilotUser *pu,
	     GList *conlist,
	     GList *bconlist,
	     iterate_func iter,
	     GPilotContext *context)
{
	GnomePilotDBInfo dbinfo;
	GnomePilotConduit *conduit;
	int index = 0;
	int error = 0;

	index = 0;
	error = 0;
	dbinfo.pilot_socket = pfd;
	dbinfo.pilotInfo = gpilot_find_pilot_by_id(pu->userID,context->pilots);

	while(1) {
		if(dlp_OpenConduit(pfd) < 0) {
			g_error("Unable to open conduit!");
			return -1;
		}
		/* load next dbinfo block */
		if(dlp_ReadDBList(pfd, 0, 0x80, index, PI_DBINFO (&dbinfo)) < 0)
			break;
		index = PI_DBINFO (&dbinfo)->index + 1;
		/*
		g_message("DB: %20s: Exclude=%3s, Backup=%3s Flags=0x%8.8x Creator=0x%lx",
			   PI_DBINFO(&dbinfo)->name,
			   (PI_DBINFO(&dbinfo)->miscFlags & dlpDBMiscFlagExcludeFromSync)?"yes":"no",
			   (PI_DBINFO(&dbinfo)->flags & dlpDBFlagBackup)?"yes":"no",
			   PI_DBINFO(&dbinfo)->flags,
			   PI_DBINFO(&dbinfo)->creator); 
		*/
		/* check if the base is marked as being excluded from sync */
		if (!(PI_DBINFO(&dbinfo)->miscFlags & dlpDBMiscFlagExcludeFromSync)) {
			guint progress_h,error_h,message_h;

			dbinfo.pu = pu;
			dbinfo.pilot_socket = pfd;
			dbinfo.manager_data = (void *) stamp;
			conduit = find_matching_conduit (conlist,&dbinfo);
			if(!conduit) {
				/* FIXME: check if base is marked as to-be-backupped using
				   fields in dbinfo.
				   Or not, it doesn't look like dbinfo->flags are set when the
				   base is changed, nor is modifyDate > backupDate...
				 */
				g_list_foreach (bconlist,
						(GFunc) backup_foreach,
						&dbinfo);
				continue;
			}
			/* Now call the iter method on the conduit, after setting up
			   the signals. The signals aren't set before, since we 
			   want to pass dbinfo as the userdata, and we don't do it
			   in "iter", since it's a parameterized funtion */
			progress_h = gtk_signal_connect(GTK_OBJECT(conduit),"progress",
							(GtkSignalFunc)conduit_progress_callback,
							&dbinfo);
			error_h = gtk_signal_connect(GTK_OBJECT(conduit),"error",
						     (GtkSignalFunc)conduit_error_callback,
						     &dbinfo);
			message_h = gtk_signal_connect(GTK_OBJECT(conduit),"message",
						       (GtkSignalFunc)conduit_message_callback,
						       &dbinfo);
			error = iter (GNOME_PILOT_CONDUIT_STANDARD (conduit), &dbinfo);

			gtk_signal_disconnect(GTK_OBJECT(conduit),progress_h);
			gtk_signal_disconnect(GTK_OBJECT(conduit),error_h);
			gtk_signal_disconnect(GTK_OBJECT(conduit),message_h);
		} else {
			LOG("Base %s is to be ignored by sync",PI_DBINFO(&dbinfo)->name);
			continue;
		}
	}
	return error;
}

static gint
conduit_synchronize(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Synchronizing %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeSynchronize);
	err = gnome_pilot_conduit_standard_synchronize (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_copy_to_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Copy to pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeCopyToPilot);
	err = gnome_pilot_conduit_standard_copy_to_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_copy_from_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Copy from pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeCopyFromPilot);
	err = gnome_pilot_conduit_standard_copy_from_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_merge_to_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Merge to pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeMergeToPilot);
	err = gnome_pilot_conduit_standard_merge_to_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_merge_from_pilot(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	int err;
	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	if(gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit))!=NULL)
		gpilot_add_log_entry(dbinfo->pilot_socket,"Merge from pilot %s\n",
				     gnome_pilot_conduit_get_name(GNOME_PILOT_CONDUIT(conduit)));
	orbed_notify_conduit_start(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit),
				   GnomePilotConduitSyncTypeMergeFromPilot);
	err = gnome_pilot_conduit_standard_merge_from_pilot (conduit,dbinfo);
	orbed_notify_conduit_end(dbinfo->pilotInfo->name,
				   GNOME_PILOT_CONDUIT(conduit));
	return err;
}

static gint
conduit_sync_default(GnomePilotConduitStandard *conduit, GnomePilotDBInfo *dbinfo)
{
	GnomePilotConduitManagerInfo *manager;
	GnomePilotConduitSyncType action;

	g_return_val_if_fail (conduit != NULL, -1);
	g_return_val_if_fail (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit), -1);
	g_return_val_if_fail (dbinfo!=NULL, -1);
	g_return_val_if_fail (dbinfo->pu!=NULL, -1);

	manager = gtk_object_get_data (GTK_OBJECT (conduit), "manager_info");
	g_return_val_if_fail(manager!=NULL,-1);
	if (manager->first_sync_type != GnomePilotConduitSyncTypeNotSet) {
		g_message(_("Conduit %s's first synchronization..."),manager->name);
		gpilot_remove_first_sync_settings(dbinfo->pu->userID,
						  manager->config_name);
		action = manager->first_sync_type;
		if(manager->slow==TRUE) {
			gnome_pilot_conduit_standard_set_slow(conduit);
		}
	} else {
		action = manager->sync_type;
	}
		
	switch (action) {
	case GnomePilotConduitSyncTypeSynchronize:
		return conduit_synchronize (conduit, dbinfo);
	case GnomePilotConduitSyncTypeCopyToPilot:
		return conduit_copy_to_pilot (conduit, dbinfo);
	case GnomePilotConduitSyncTypeCopyFromPilot:
		return conduit_copy_from_pilot (conduit, dbinfo);
	case GnomePilotConduitSyncTypeMergeToPilot:
		return conduit_merge_to_pilot (conduit, dbinfo);
	case GnomePilotConduitSyncTypeMergeFromPilot:
		return conduit_merge_from_pilot (conduit, dbinfo);
	default:
		g_error (_("unknown syncing action (%d = %s).\n"),action,
			 gnome_pilot_conduit_sync_type_int_to_str(action));
		return -1;
	}
}

static void
install_db_foreach (GPilotRequest *req, GnomePilotInstallCarrier *carrier) {
	switch(req->type) {
	case GREQ_INSTALL:
		gnome_pilot_conduit_file_install_db (GNOME_PILOT_CONDUIT_FILE (carrier->conduit),
						     carrier->pilot_socket,
						     req->parameters.install.filename);
		break;
	case GREQ_RESTORE:
		gnome_pilot_conduit_file_restore_directory (GNOME_PILOT_CONDUIT_FILE (carrier->conduit),
							    carrier->pilot_socket,
							    req->parameters.restore.directory);
		break;
	default:
		g_warning (_("%s this request.type == %d is not a install/restore"), __PRETTY_FUNCTION__, req->type);
		break;
	}
	orbed_notify_completion (&req);
};

static void
install_foreach (GnomePilotConduit *conduit,
		 GnomePilotInstallCarrier *info)
{
	GList *request_list;

	g_return_if_fail (conduit != NULL);
	g_return_if_fail (GNOME_IS_PILOT_CONDUIT_FILE (conduit));

	info->conduit = conduit;
	request_list = gpc_queue_load_requests(info->pu->userID, GREQ_INSTALL,FALSE);
	g_message ("Pilot has %d entries in file install queue", g_list_length (request_list));
	g_list_foreach (request_list, (GFunc)install_db_foreach, info);
	g_list_free (request_list);
}

static void
send_sysinfo_foreach(GPilotRequest *req, GnomePilotInstallCarrier *carrier)
{
	struct CardInfo card;
	char *pilot_name;

	pilot_name = pilot_name_from_id (carrier->pu->userID, carrier->context);
	dlp_ReadStorageInfo (carrier->pilot_socket,0,&card);
	orbed_notify_sysinfo (pilot_name,
			      card, 
			      &req);
	orbed_notify_completion (&req);
	g_free(pilot_name);
}

static void
do_restore_foreach(GPilotRequest *req, GnomePilotInstallCarrier *carrier)
{
	GList *iterator;
	g_assert(carrier!=NULL);

	/* FIXME: if carrier.conduit_list is NULL, the user hasn't
	  enabled any file conduits. The corba restore call should
	  throw an exception or something.  */
	iterator = carrier->conduit_list;
	while( iterator != NULL ) {
		/* FIXME: Do some amazing stuff here ! */
		GnomePilotConduitFile *conduit;
		conduit = GNOME_PILOT_CONDUIT_FILE(iterator->data);
		gnome_pilot_conduit_file_restore_directory(conduit,carrier->pilot_socket,
							   req->parameters.restore.directory);
		iterator = iterator->next;
	}


	orbed_notify_completion (&req);
}

static void
unload_foreach(gpointer *conduit,
	       gpointer unused)
{
	GnomePilotConduitManagerInfo *manager;
	g_return_if_fail (conduit != NULL);
	g_return_if_fail (GNOME_IS_PILOT_CONDUIT (conduit));
	
	manager = gtk_object_get_data (GTK_OBJECT (conduit), "manager_info");
	g_assert(manager!=NULL);
	manager->destroy_gpilot_conduit (GNOME_PILOT_CONDUIT (conduit));
#ifdef USE_GMODULE
	g_module_close (manager->dlhandle);
#else
	dlclose (manager->dlhandle);
#endif
	g_free(manager->name);
	g_free(manager->config_name);
	g_free(manager);
}

static GnomePilotConduit *
gpilot_load_conduit(guint32 pilotId,
		    gchar *conduit_fn,
		    GnomePilotConduitManagerInfo *cmaninfo)
{
	GModule *dlhandle;
	GnomePilotConduit *conduit;
	conduit_get_gpilot_conduit get_gpilot_conduit;
	conduit_destroy_gpilot_conduit destroy_gpilot_conduit;

	g_assert(cmaninfo!=NULL);
	g_assert(conduit_fn!=NULL);

#ifdef USE_GMODULE
	if((dlhandle = g_module_open(conduit_fn, G_MODULE_BIND_LAZY))) {
		if(!g_module_symbol(dlhandle,"conduit_get_gpilot_conduit",
				    (gpointer) &get_gpilot_conduit)) {
			g_error(_("Malformed conduit %s, cannot find "
				  "conduit_get_gpilot_conduit: %s"),conduit_fn,g_module_error());
			g_module_close(dlhandle);
			return NULL;
		}

		if(!g_module_symbol(dlhandle,"conduit_destroy_gpilot_conduit",
				    (gpointer) &destroy_gpilot_conduit)) {
			g_error(_("Malformed conduit %s, cannot find "
				  "conduit_destroy_gpilot_conduit: %s"),conduit_fn,g_module_error());
			g_module_close(dlhandle);
			return NULL;
		}
#else
	if((dlhandle = dlopen(conduit_fn, RTLD_LAZY))) {
		if((get_gpilot_conduit = dlsym(dlhandle,"conduit_get_gpilot_conduit"))==NULL) {
			g_error(_("Malformed conduit %s, cannot find "
				  "conduit_get_gpilot_conduit: %s"),conduit_fn,g_module_error());
			dlclose(dlhandle);
			return NULL;
		}

		if((destroy_gpilot_conduit = dlsym(dlhandle,"conduit_destroy_gpilot_conduit")) == NULL) {

			g_error(_("Malformed conduit %s, cannot find "
				  "conduit_destroy_gpilot_conduit: %s"),conduit_fn,g_module_error());
			dlclose(dlhandle);
			return NULL;
		}
#endif

		conduit = get_gpilot_conduit(pilotId);
		cmaninfo->dlhandle = dlhandle;
		cmaninfo->name = g_strdup (gnome_pilot_conduit_get_name (conduit));
		cmaninfo->destroy_gpilot_conduit = destroy_gpilot_conduit;
		gtk_object_set_data (GTK_OBJECT (conduit), "manager_info", cmaninfo);
	} else {
#ifdef USE_GMODULE
		g_warning(_("Unable to open conduit %s: %s"),conduit_fn,g_module_error());
#else
		g_warning(_("Unable to open conduit %s: %s"),conduit_fn,dlerror());
#endif
		return NULL;
	}
	return conduit;
}


void
gpilot_load_conduits (GPilotPilot *pilot,
		      GList **clist,
		      GList **blist,
		      GList **flist)
{
	int i, cnt;
	gchar *conduit_fn, *conduit_sync_type_str;
	gchar *prefix;
	GnomePilotConduit *conduit;
	guint32 pilotId;
	gchar **conduit_name;
	GnomePilotConduitManagerInfo *cmaninfo;

	pilotId = pilot->pilot_id;

	*clist = NULL;
	*blist = NULL;
	*flist = NULL;

	/* Read the conduit configuration */
	prefix = g_strdup_printf ("gnome-pilot.d/conduits%d/General/", pilot->pilot_id);
	gnome_config_push_prefix(prefix);
	gnome_config_get_vector("conduits",&cnt,&conduit_name);
	gnome_config_pop_prefix();
	g_free(prefix);
	g_message(_("Loading %d conduits..."),cnt);
	for(i = 0;i<cnt;i++) {
		prefix = g_strdup_printf("/gnome-pilot.d/conduits%d/%s/",
			   pilot->pilot_id,conduit_name[i]);
		if(gnome_config_has_section(prefix)) {
			cmaninfo = g_new0(GnomePilotConduitManagerInfo,1);
			cmaninfo->config_name = g_strdup(conduit_name[i]);
			gnome_config_push_prefix(prefix);
			conduit_fn = gnome_config_get_string("libfile");

			conduit_sync_type_str = gnome_config_get_string("sync_type");
			cmaninfo->sync_type = gnome_pilot_conduit_sync_type_str_to_int (conduit_sync_type_str);
			g_free(conduit_sync_type_str);

			conduit_sync_type_str = gnome_config_get_string("first_sync_type");
			if (conduit_sync_type_str != NULL) {
				cmaninfo->first_sync_type = gnome_pilot_conduit_sync_type_str_to_int (conduit_sync_type_str);
				g_free(conduit_sync_type_str);
			} else
				cmaninfo->first_sync_type = GnomePilotConduitSyncTypeNotSet;
			cmaninfo->slow = gnome_config_get_bool("slow_sync");


			conduit = gpilot_load_conduit(pilotId,conduit_fn,cmaninfo);
			if (conduit == NULL) {
				g_message(_("Loading conduit \"%s\" failed!"),conduit_name[i]);
				gpilot_gui_warning_dialog(_("Loading conduit \"%s\" failed!\n"
							    "It is likely a problem with the file\n"
							    "%s"),conduit_name[i],conduit_fn);
				g_free(conduit_fn);
				continue;
			}

			if (GNOME_IS_PILOT_CONDUIT_BACKUP (conduit)) {
				*blist = g_list_append (*blist, conduit);
			} else if (GNOME_IS_PILOT_CONDUIT_FILE (conduit)) {
				*flist = g_list_append (*flist, conduit);
			} else if (GNOME_IS_PILOT_CONDUIT_STANDARD (conduit)) {
				*clist = g_list_append (*clist, conduit);
			} else {
				g_warning("Error in conduits definition file for pilot %d, %s is not a valid conduit",
					  pilot->number,
					  conduit_name[i]);
			}
			g_free(conduit_fn);
			gnome_config_pop_prefix();
		} else {
			g_warning("Error in conduits definition file for pilot %d, %s is not a valid section",
				  pilot->number,
				  conduit_name[i]);
		}
		g_free(prefix);
		g_free(conduit_name[i]);
	}
	g_free(conduit_name);
}

void
gpilot_unload_conduits(GList *list)
{
	g_list_foreach (list, (GFunc) unload_foreach, NULL);
	g_list_free(list);
}

static void 
run_conduit_sync_foreach(GPilotRequest *req,
			 GnomePilotInstallCarrier *carrier)
{
	GnomePilotConduit *conduit;
	GList *clist,*blist,*flist;
	int err;

	g_return_if_fail(conduit!=NULL);
	g_return_if_fail(carrier!=NULL);

	err = 0;
	clist = NULL;
	blist = NULL;
	flist = NULL;
	conduit = NULL;

	conduit = find_matching_conduit_name(carrier->conduit_list,
					     req->parameters.conduit.name);
	if(conduit==NULL) {
		/* FIXME: remove the bloody request, or the user will get this
		   message every time */
		g_error(_("Conduit %s requested, but not found"),
			req->parameters.conduit.name);
	} else {
		if(GNOME_IS_PILOT_CONDUIT_BACKUP(conduit)) {
			blist = g_list_append(blist,conduit);
		} else if(GNOME_IS_PILOT_CONDUIT_STANDARD(conduit)) {
			clist = g_list_append(clist,conduit);
		} else if(GNOME_IS_PILOT_CONDUIT_FILE(conduit)) {
			flist = g_list_append(flist,conduit);
		} else {
			g_error(_("Conduit %s cannot be requested"),
				req->parameters.conduit.name);
		}

		if(flist) {
			g_list_foreach(flist,(GFunc)install_foreach,&carrier);
		} else {
	    
			err= iterate_dbs(carrier->pilot_socket,
					 carrier->syncstamp,
					 carrier->pu,
					 clist,
					 blist,
					 conduit_sync_default,
					 carrier->context); 
		}
	}
	orbed_notify_completion(&req);
}

gboolean gpilot_initial_synchronize_operations(int pilot_socket,
					       GnomePilotSyncStamp *stamp,
					       struct PilotUser *pu,
					       GList *conduit_list,
					       GList *backup_conduit_list,
					       GList *file_conduit_list,
					       GPilotContext *context)
{
	GList *request_list;
	gboolean do_normal_sync_stuff;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	/* If this variable is true, run all the normal conduit stuff in
	   conduit_list, backup_conduit_list and file_conduit_list in the end.
	   If false (getsysinfo, restore and/or single conduit run), dont
	   use the normal conduits */
	do_normal_sync_stuff = TRUE;

	/* elements in request_list freed by gpc_request_purge calls
           in send_sysinfo_foreach */
	request_list = gpc_queue_load_requests(pu->userID, GREQ_GET_SYSINFO,FALSE);
	g_message("Pilot has %d entries in get_sysinfo queue",g_list_length(request_list));
	if (g_list_length (request_list)) {
		g_list_foreach(request_list,(GFunc)send_sysinfo_foreach,&carrier);
	}

	/* elements in request_list freed by gpc_request_purge calls
           in do_restore_foreach */
	request_list = gpc_queue_load_requests(pu->userID, GREQ_RESTORE,FALSE);
	g_message("Pilot has %d entries in restore queue",g_list_length(request_list));
	if (g_list_length (request_list)) {
		do_normal_sync_stuff = FALSE;
		carrier.conduit_list = g_list_copy(file_conduit_list);
		g_list_foreach(request_list,(GFunc)do_restore_foreach,&carrier);
	}

	/* elements in request_list freed by gpc_request_purge calls
           in run_conduit_sync_foreach */
	request_list = gpc_queue_load_requests(pu->userID, GREQ_CONDUIT,FALSE);
	g_message("Pilot has %d entries in conduit queue",g_list_length(request_list));
	if (g_list_length (request_list)) {
		carrier.conduit_list = g_list_copy(conduit_list);
		carrier.conduit_list = g_list_concat(carrier.conduit_list,backup_conduit_list);
		carrier.conduit_list = g_list_concat(carrier.conduit_list,file_conduit_list);
		do_normal_sync_stuff = FALSE;
		g_list_foreach(request_list,(GFunc)run_conduit_sync_foreach,&carrier);
		g_list_free(carrier.conduit_list);
	}

	return do_normal_sync_stuff;
}

gint
gpilot_synchronize(int pilot_socket,
		   GnomePilotSyncStamp *stamp,
		   struct PilotUser *pu,
		   GList *conduit_list,
		   GList *backup_conduit_list,
		   GList *file_conduit_list,
		   GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	/* now run through all the file install requests */
	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);
	/* and then iterate over all conduits */
	error = iterate_dbs(pilot_socket,
			    stamp,
			    pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_synchronize,
			    context);
	/* FIXME: empty queue for pilot here */

	return error;
}

gint 
gpilot_copy_to_pilot(int pilot_socket, 
		     GnomePilotSyncStamp *stamp,
		     struct PilotUser *pu,
		     GList *conduit_list,
		     GList *backup_conduit_list,
		     GList *file_conduit_list,
		     GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);

	error = iterate_dbs(pilot_socket,stamp,pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_copy_to_pilot,
			    context);
	return error;
}

gint 
gpilot_copy_from_pilot(int pilot_socket, 
		       GnomePilotSyncStamp *stamp,
		       struct PilotUser *pu,
		       GList *conduit_list,
		       GList *backup_conduit_list,
		       GPilotContext *context)
{
	gint error;
	error = iterate_dbs(pilot_socket,stamp,pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_copy_from_pilot,
			    context);
	return error;
}

gint 
gpilot_merge_to_pilot(int pilot_socket, 
		      GnomePilotSyncStamp *stamp,
		      struct PilotUser *pu,
		      GList *conduit_list,
		      GList *backup_conduit_list, 
		      GList *file_conduit_list,
		      GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;

	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);

	error = iterate_dbs(pilot_socket,stamp,pu,
			    conduit_list,
			    backup_conduit_list,
			    conduit_merge_to_pilot,
			    context);

	return error;
}

gint 
gpilot_merge_from_pilot(int pilot_socket, 
			GnomePilotSyncStamp *stamp,
			struct PilotUser *pu,
			GList *conduit_list,
			GList *backup_conduit_list,
			GPilotContext *context)
{
	gint error;
	error = iterate_dbs(pilot_socket,stamp,pu,conduit_list,backup_conduit_list,
			  conduit_merge_from_pilot,context);
	return error;
}

gint 
gpilot_sync_default(int pilot_socket, GnomePilotSyncStamp *stamp,
		    struct PilotUser *pu,
		    GList *conduit_list,
		    GList *backup_conduit_list,
		    GList *file_conduit_list,
		    GPilotContext *context)
{
	gint error;
	GnomePilotInstallCarrier carrier;

	carrier.pu = pu;
	carrier.pilot_socket = pilot_socket;
	carrier.context = context;
	g_list_foreach(file_conduit_list,(GFunc)install_foreach,&carrier);

	error=iterate_dbs(pilot_socket,stamp,pu,conduit_list,backup_conduit_list,
			  conduit_sync_default,context);
	return error;
}

void 
gpilot_add_log_entry(int pilot_socket,
		     gchar *entry, ...) 
{
	gchar *e;

	/* Pilot link before 0.9.3 has a bug in dlp_AddSyncLogEntry,
	 that eats last char (the \n) from input. Thus we add a space
	 if pilot link is an older version */

#if PILOT_LINK_VERSION == 0
  #if PILOT_LINK_MAJOR == 9
    #if PILOT_LINK_MINOR < 3
      /* pilot-link version less then 0.9.3 */
      #define PILOT_NAME_NEED_SPACE
    #endif /* PILOT_LINK_MINOR < 3 */
  #else /* PILOT_LINK_MAJOR == 9 */
    #if PILOT_LINK_MAJOR < 9
      /* "pilot-link version less then 0.9.x */
      #define PILOT_NAME_NEED_SPACE
    #endif /* PILOT_LINK_MAJOR < 9 */
  #endif /* PILOT_LINK_MAJOR == 9 */
#endif /* PILOT_LINK_VERSION == 0 */

#ifdef PILOT_NAME_NEED_SPACE
	gchar *f;
#endif
	va_list ap;

	va_start(ap,entry);
	e = g_strdup_vprintf(entry,ap);
#ifdef PILOT_NAME_NEED_SPACE
	f = g_strdup_printf("%s ",e);
	dlp_AddSyncLogEntry(pilot_socket,f);
	g_free (f);
#else
	dlp_AddSyncLogEntry(pilot_socket,e);
#endif
	g_free (e);
}

void
gpilot_remove_first_sync_settings(guint32 pilot_id,
				  gchar *config_name) 
{
	gchar *prefix;
	prefix = g_strdup_printf("/gnome-pilot.d/conduits%d/%s/",pilot_id,config_name);
	if (gnome_config_has_section(prefix)) {
		gnome_config_push_prefix(prefix);
		gnome_config_clean_key("first_sync_type");
		gnome_config_clean_key("slow_sync");
		gnome_config_pop_prefix();
		gnome_config_sync();
	}
	g_free(prefix);

}
