#include "config.h"
#include "gpilotd-app.h"
#include <libgnorba/gnorba.h>

/* #define LIBGPILOTD_DEBUG 1 */
#define USE_GNORBA

GNOME_Pilot_Daemon gpilotddaemon;
GNOME_Pilot_Client callback_obj;
CORBA_Environment ev;
CORBA_ORB orb;
CORBA_char* client_id;

/* the LOG macro cannot by used without a format */
#ifdef LIBGPILOTD_DEBUG
#define LOG(format,args...) g_log (G_LOG_DOMAIN, \
                                   G_LOG_LEVEL_MESSAGE, \
                                   "lib-gpilotd: "##format, ##args)
#else
#define LOG(format,args...)
#endif

static CORBA_char* 
cb_get_client_id(PortableServer_Servant servant, 
		 CORBA_Environment *ev) 
{
	LOG("client_id returned : %.20s...",client_id==NULL?"(null)":client_id);
	if (client_id==NULL) return CORBA_string_dup("");
	return CORBA_string_dup(client_id);
}

static void 
cb_set_client_id(PortableServer_Servant servant, 
		 const CORBA_char * value, 
		 CORBA_Environment *ev) 
{

	/* FIXME: erh, the exception id ? 
	   if(client_id!=NULL)
	   CORBA_exception_set (ev, CORBA_USER_EXCEPTION, "halla walla", NULL);
	*/

	client_id = g_strdup(value);
	LOG("client_id = %.20s",client_id);
}

static void
cb_pilot_connect(PortableServer_Servant servant,
		 const CORBA_char *pilot_id,
		 const GNOME_Pilot_PilotUser *pilotuser,
		 CORBA_Environment *ev) 
{
	LOG("cb_pilot_connect(%s)",pilot_id);
	(*gpilotd_connect_cb)(pilot_id,pilotuser);
	return;
}

static void
cb_pilot_disconnect(PortableServer_Servant servant,
		    const CORBA_char *pilot_id,
		    CORBA_Environment *ev) 
{
	LOG("cb_pilot_disconnect(%s)",pilot_id);
	(*gpilotd_disconnect_cb)(pilot_id);
	return;
}

static void
cb_request_completed(PortableServer_Servant servant,
		     const CORBA_char *pilot_id,
		     const CORBA_unsigned_long id,
		     CORBA_Environment *ev) 
{
	LOG("cb_request_completed(id=%s,req=%d)",pilot_id,id);
	(*gpilotd_request_completed)(pilot_id,id);
	return;
}

static void
cb_userinfo_requested(PortableServer_Servant servant,
		      const CORBA_char *device,
		      const GNOME_Pilot_PilotUser *user,
		      CORBA_Environment *ev) {
	LOG("cb_userinfo_requested(device=%s,user=%d:%s)",device,user->userID,user->username);
	(*gpilotd_userinfo_requested)(device,user);
	return;
}

static void
cb_sysinfo_requested(PortableServer_Servant servant,
		     const CORBA_char *pilot_id,
		     const GNOME_Pilot_SysInfo *sys,
		     CORBA_Environment *ev) 
{
	LOG("cb_sysinfo_requested(pilot=%s,...)",pilot_id);
	(*gpilotd_sysinfo_requested)(pilot_id,sys);
	return;
}

static void
cb_conduit_start(PortableServer_Servant servant,
		 const CORBA_char *pilot_id,
		 const CORBA_char *conduit_name,
		 const CORBA_char *database,
		 CORBA_Environment *ev) 
{
	LOG("cb_conduit_start(pilot=%s,conduit=%s,db=%s)",pilot_id,conduit_name,database);
	(*gpilotd_conduit_start)(pilot_id, conduit_name, database);
}

static void
cb_conduit_end(PortableServer_Servant servant,
	       const CORBA_char *pilot_id,
	       const CORBA_char *conduit_name,
	       CORBA_Environment *ev) 
{
	LOG("cb_conduit_end(pilot=%s,conduit=%s)",pilot_id,conduit_name);
	(*gpilotd_conduit_end)(pilot_id,conduit_name);
}

static void
cb_conduit_progress(PortableServer_Servant servant,
		    const CORBA_char *pilot_id,
		    const CORBA_char *conduit,
		    CORBA_long current,
		    CORBA_long total,
		    CORBA_Environment *ev) 
{
	/*LOG("cb_conduit_progress(%s,%s,%d/%d)",pilot_id,conduit,current,total);*/
	(*gpilotd_conduit_progress)(pilot_id,conduit,current,total);
}

static void
cb_conduit_message(PortableServer_Servant servant,
		   const CORBA_char *pilot_id,
		   const CORBA_char *conduit,
		   const CORBA_char *message,
		   CORBA_Environment *ev) 
{
	LOG("cb_conduit_message(pilot=%s,message=%s)",pilot_id,message);
	(*gpilotd_conduit_message)(pilot_id,conduit,message);
}

static void
cb_conduit_error(PortableServer_Servant servant,
		 const CORBA_char *pilot_id,
		 const CORBA_char *conduit,
		 const CORBA_char *message,
		 CORBA_Environment *ev) 
{
	LOG("cb_conduit_error(pilot=%s,message=%s)",pilot_id,message);
	(*gpilotd_conduit_error)(pilot_id,conduit,message);
}

PortableServer_ServantBase__epv base_epv = {
	NULL,
	NULL,
	NULL
};

POA_GNOME_Pilot_Client__epv callback_epv = { 
	NULL, 
	(gpointer)&cb_get_client_id,
	(gpointer)&cb_set_client_id,
	(gpointer)&cb_pilot_connect,
	(gpointer)&cb_pilot_disconnect,
	(gpointer)&cb_request_completed,
	(gpointer)&cb_userinfo_requested,
	(gpointer)&cb_sysinfo_requested,

	(gpointer)&cb_conduit_start,
	(gpointer)&cb_conduit_progress,
	(gpointer)&cb_conduit_end,
	(gpointer)&cb_conduit_message,
	(gpointer)&cb_conduit_error,
};

POA_GNOME_Pilot_Client__vepv callback_vepv = { &base_epv, &callback_epv };
POA_GNOME_Pilot_Client callback_servant = { NULL, &callback_vepv };

gint 
gpilotd_connect() 
{
	gpilotddaemon = goad_server_activate_with_repo_id(NULL,
							  "IDL:GNOME/Pilot/Daemon:1.0",
							  0,NULL);
	if (gpilotddaemon == NULL) return -1;
	return 0;
}

gint 
gpilotd_init(int *argc,char *argv[]) 
{
	PortableServer_POA the_poa;

	client_id = NULL;

	CORBA_exception_init(&ev);

	orb = gnome_CORBA_ORB();
  
	POA_GNOME_Pilot_Client__init(&callback_servant,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
    
	the_poa = (PortableServer_POA)CORBA_ORB_resolve_initial_references(orb, "RootPOA", &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -2;
	}
    
	PortableServer_POAManager_activate(PortableServer_POA__get_the_POAManager(the_poa, &ev), &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -3;
	}
    
	PortableServer_POA_activate_object(the_poa, &callback_servant, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -4;
	}
    
	callback_obj = PortableServer_POA_servant_to_reference(the_poa, &callback_servant, &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -5;
	}

	return 0;
}

void gpilotd_shutdown() {
/*
  What the hell, I'm not supposed to unregister the server ?!
	if (1) {
		goad_server_unregister(CORBA_OBJECT_NIL,"","object",&ev);
		if (ev._major != CORBA_NO_EXCEPTION) {
			g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
			CORBA_exception_free(&ev);
			return;
		}
	}
*/
	CORBA_Object_release(gpilotddaemon, &ev);
	CORBA_Object_release(callback_obj, &ev);
}

gint 
gpilotd_monitor_on(gchar *pilot_id) 
{
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(pilot_id!=NULL,-1);
	GNOME_Pilot_Daemon_monitor_on(gpilotddaemon,callback_obj,pilot_id,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	return 0;
}

gint 
gpilotd_monitor_off(gchar *pilot_id) 
{
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(pilot_id!=NULL,-1);
	GNOME_Pilot_Daemon_monitor_off(gpilotddaemon,callback_obj,pilot_id,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	return 0;
}

gint 
gpilotd_notify_on(GNOME_Pilot_EventType type) 
{
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	GNOME_Pilot_Daemon_notify_on(gpilotddaemon,type,callback_obj,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	return 0;
}

gint 
gpilotd_notify_off(GNOME_Pilot_EventType type) 
{
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	GNOME_Pilot_Daemon_notify_off(gpilotddaemon,type,callback_obj,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	return 0;
}

gint 
gpilotd_restore(gchar *pilot_id,
		gchar *directory,
		GNOME_Pilot_Survival survival,
		int timeout) 
{
	gint val;

	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(pilot_id!=NULL,-1);
	g_return_val_if_fail(directory!=NULL,-1);
	g_return_val_if_fail(timeout>=0,-1);
	
	val = GNOME_Pilot_Daemon_request_restore(gpilotddaemon,
						 callback_obj,
						 pilot_id,
						 directory,
						 survival,
						 timeout,
						 &ev);
	switch(ev._major) {
	case CORBA_NO_EXCEPTION: 
		break;
	case CORBA_SYSTEM_EXCEPTION:
	case CORBA_USER_EXCEPTION:
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		LOG("install failed with exception %s",CORBA_exception_id(&ev));
		if (strcmp(ex_GNOME_Pilot_MissingFile,CORBA_exception_id(&ev))==0) {
			LOG("Missing file");
		} 
		CORBA_exception_free(&ev);
		return -1;
		break;
	default:
		break;
	}
	
	LOG("request handle = %d",val);
	return val;
}

gint 
gpilotd_install_file(gchar *pilot_id,
		     gchar *name,
		     GNOME_Pilot_Survival survival,
		     int timeout) 
{
	gint val;
	FILE *in,*out;
	gchar *filename;
	gchar queue[128];
	
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(pilot_id!=NULL,-1);
	g_return_val_if_fail(name!=NULL,-1);
	g_return_val_if_fail(timeout>=0,-1);
	
	g_snprintf(queue,127,"%s/.gnome/gnome-pilot.d/",g_get_home_dir());
	
	filename = tempnam(queue,"PDB");
	LOG("tmpfile = %s",filename);
	
	in = fopen(name,"rb");
	g_return_val_if_fail(in!=NULL,-12);
	out = fopen(filename,"wb");
	g_return_val_if_fail(out!=NULL,-13);
	
	while(!feof(in)) {
		size_t act;
		char block[1024];
		act=fread(block,1,1024,in);
		fwrite(block,1,act,out);
	}
	
	fflush(out);
	fclose(in); 
	fclose(out);
	
	val = GNOME_Pilot_Daemon_request_install(gpilotddaemon,
						 callback_obj,
						 pilot_id,
						 filename,
						 name,
						 survival,
						 timeout,
						 &ev);
	switch(ev._major) {
	case CORBA_NO_EXCEPTION: 
		break;
	case CORBA_SYSTEM_EXCEPTION:
	case CORBA_USER_EXCEPTION:
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		LOG("install failed with exception %s",CORBA_exception_id(&ev));
		if (strcmp(ex_GNOME_Pilot_MissingFile,CORBA_exception_id(&ev))==0) {
			LOG("Missing file");
		} else {
			unlink(filename);
		}
		CORBA_exception_free(&ev);
		return -1;
		break;
	default:
		break;
	}
	
	LOG("request handle = %d",val);
	return val;
}

gint 
gpilotd_set_user_info(gchar *cradle,
		      GNOME_Pilot_PilotUser user,
		      GNOME_Pilot_Survival survival,
		      int timeout) 
{
	gint val;

	/* FIXME: needs code to check user contents */
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(cradle!=NULL,-1);
	g_return_val_if_fail(timeout>=0,-1);

	val = GNOME_Pilot_Daemon_set_user_info(gpilotddaemon,
					       callback_obj,
					       &user,
					       cradle,
					       survival,timeout,
					       &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	LOG("val = %d",val);
	return val;
}

gint 
gpilotd_get_user_info(gchar *cradle,
		      GNOME_Pilot_Survival survival,
		      int timeout) 
{
	gint val;

	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(cradle!=NULL,-1);
	g_return_val_if_fail(timeout>=0,-1);

	val = GNOME_Pilot_Daemon_get_user_info(gpilotddaemon,
					       callback_obj,
					       cradle,
					       survival,timeout,
					       &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	LOG("val = %d",val);
	return val;
}

gint 
gpilotd_get_system_info(gchar *pilot_id,
			GNOME_Pilot_Survival survival,
			int timeout) 
{
	gint val;

	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(pilot_id!=NULL,-1);
	g_return_val_if_fail(timeout>=0,-1);

	val = GNOME_Pilot_Daemon_get_system_info(gpilotddaemon,
						 callback_obj,
						 pilot_id,
						 survival,timeout,
						 &ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	LOG("val = %d",val);
	return val;
}

int 
gpilotd_get_users(gchar ***users) 
{
	GNOME_Pilot_StringSequence *userseq;

	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(users!=NULL,-1);

	userseq = GNOME_Pilot_Daemon_get_users(gpilotddaemon,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	if (userseq->_length>0) {
		int i;
		(*users) = (gchar**)g_malloc((1+userseq->_length) * sizeof(gchar*));
		for(i=0;i<userseq->_length;i++) {
			(*users)[i] = g_strdup(userseq->_buffer[i]);
		}
		(*users)[userseq->_length] = NULL;
	} else
		*users = NULL;

	CORBA_free(userseq);
	return 0;
}

int 
gpilotd_get_cradles(gchar ***cradles) 
{
	GNOME_Pilot_StringSequence *cradleseq;

	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(cradles!=NULL,0);

	cradleseq = GNOME_Pilot_Daemon_get_cradles(gpilotddaemon,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	if (cradleseq->_length>0) {
		int i;
		(*cradles) = (gchar**)g_malloc((1+cradleseq->_length) * sizeof(gchar*));
		for(i=0;i<cradleseq->_length;i++) {
			(*cradles)[i] = g_strdup(cradleseq->_buffer[i]);
		}
		(*cradles)[cradleseq->_length] = NULL;
	} else
		*cradles = NULL;

	CORBA_free(cradleseq);
	return 0;
}

enum gpilotd_app_pilots { GPILOTD_APP_ALL_PILOTS,
			  GPILOTD_APP_BY_NAME,
			  GPILOTD_APP_BY_UID };

static int 
gpilotd_get_pilots_action(gchar *name,
			  enum gpilotd_app_pilots action,
			  gchar ***pilots) 
{
	GNOME_Pilot_StringSequence *pilotseq;

	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(pilots!=NULL,-1);

	switch(action) {
	case GPILOTD_APP_ALL_PILOTS:
		pilotseq = GNOME_Pilot_Daemon_get_pilot_names(gpilotddaemon,&ev);
		break;
	case GPILOTD_APP_BY_NAME:
		pilotseq = GNOME_Pilot_Daemon_get_pilots_by_name(gpilotddaemon,name,&ev);
		break;
	case GPILOTD_APP_BY_UID:
		pilotseq = GNOME_Pilot_Daemon_get_pilots_by_login(gpilotddaemon,name,&ev);
		break;
	}
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	if (pilotseq->_length>0) {
		int i;
		(*pilots) = (gchar**)g_malloc((1+pilotseq->_length) * sizeof(gchar*));
		for(i=0;i<pilotseq->_length;i++) {
			(*pilots)[i] = g_strdup(pilotseq->_buffer[i]);
		}
		(*pilots)[pilotseq->_length] = NULL;
	} else
		*pilots = NULL;

	CORBA_free(pilotseq);
	return 0;
}

int 
gpilotd_get_pilot_names(gchar ***pilots) 
{
	return gpilotd_get_pilots_action(NULL,GPILOTD_APP_ALL_PILOTS,pilots);
}

int 
gpilotd_get_pilot_ids(gint **pilots) 
{
	GNOME_Pilot_LongSequence *pilotseq;

	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(pilots!=NULL,-1);

	pilotseq = GNOME_Pilot_Daemon_get_pilot_ids(gpilotddaemon,&ev);

	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	if (pilotseq->_length>0) {
		int i;
		(*pilots) = (gint*)g_new(gint,1+pilotseq->_length);
		for(i=0;i<pilotseq->_length;i++) {
			(*pilots)[i] = pilotseq->_buffer[i];
		}
		(*pilots)[pilotseq->_length] = -1;
	} else
		*pilots = NULL;

	CORBA_free(pilotseq);
	return 0;
}

int 
gpilotd_get_pilots_by_name(gchar *name,gchar ***pilots) 
{
	g_return_val_if_fail(name!=NULL,0);
	return gpilotd_get_pilots_action(name,GPILOTD_APP_BY_NAME,pilots);
}

int 
gpilotd_get_pilots_by_login(gchar *login,gchar ***pilots) 
{
	g_return_val_if_fail(login!=NULL,0);
	return gpilotd_get_pilots_action(login,GPILOTD_APP_BY_UID,pilots);
}

int 
gpilotd_get_conduits(gchar ***conduits) 
{
	GNOME_Pilot_StringSequence *conduitseq;
	
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(conduits!=NULL,-1);
	
	conduitseq = GNOME_Pilot_Daemon_get_conduits(gpilotddaemon,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	if (conduitseq->_length>0) {
		int i;
		(*conduits) = (gchar**)g_malloc((1+conduitseq->_length) * sizeof(gchar*));
		for(i=0;i<conduitseq->_length;i++) {
			(*conduits)[i] = g_strdup(conduitseq->_buffer[i]);
		}
		(*conduits)[conduitseq->_length] = NULL;
	} else
		*conduits = NULL;
	
	CORBA_free(conduitseq);
	return 0;
}

void 
gpilotd_monitor_on_all_pilots() 
{
	gchar **pilots;
	int i;

	if (gpilotd_get_pilot_names(&pilots)!=0) return;
	i=0;
	if (pilots) {
		while(pilots[i]) { 
			LOG("gpilotd_monitor_on(pilot %d = \"%s\");",i,pilots[i]); 
			gpilotd_monitor_on(pilots[i]);
			g_free(pilots[i]);
			i++; 
		}
		g_free(pilots);
	}
}

void 
gpilotd_monitor_off_all_pilots() 
{
	gchar **pilots;
	int i;

	if (gpilotd_get_pilot_names(&pilots)!=0) return;
	i=0;
	if (pilots) {
		while(pilots[i]) { 
			LOG("gpilotd_monitor_off(pilot %d = \"%s\");",i,pilots[i]); 
			gpilotd_monitor_off(pilots[i]);
			g_free(pilots[i]);
			i++; 
		}
		g_free(pilots);
	}
}

gint 
gpilotd_remove_request(gint handle) 
{
	gint retval;
	g_return_val_if_fail(handle>0,-1);
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	retval = GNOME_Pilot_Daemon_remove_request(gpilotddaemon,handle,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}
	return retval;
}

gint 
gpilotd_conduit(gchar *pilot_id,
		gchar *conduit,
		GNOME_Pilot_ConduitOperation operation,
		GNOME_Pilot_Survival survival, 
		int timeout) 
{
	gint retval;
	g_return_val_if_fail(gpilotddaemon!=NULL,-10);
	g_return_val_if_fail(callback_obj!=NULL,-11);
	g_return_val_if_fail(pilot_id!=NULL,0);
	g_return_val_if_fail(conduit!=NULL,0);
	g_return_val_if_fail(timeout>=0,0);
	
	retval = GNOME_Pilot_Daemon_request_conduit(gpilotddaemon,
						    callback_obj,
						    pilot_id,
						    conduit,
						    operation,
						    survival,
						    timeout,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return 0;
	}
	return retval;
}

/* return val must be freed */
gchar* 
gpilotd_get_pilot_basedir(gchar *pilot) 
{
	CORBA_char *basedir;
	gchar *retval;
	
	g_return_val_if_fail(pilot!=NULL,NULL);

	basedir = GNOME_Pilot_Daemon_get_pilot_base_dir(gpilotddaemon,pilot,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return NULL;
	}

	retval = g_strdup(basedir);
	CORBA_free(basedir);

	if (strlen(retval)==0) {
		g_free(retval);
		return g_strdup_printf("%s/%s",g_get_home_dir(),pilot);
	}

	return retval;	
}

/* return val must be freed */
gchar* 
gpilotd_get_pilot_basedir_by_id(guint32 pilot_id) 
{
	gchar *retval;
	gchar *pilot_name;

	pilot_name = gpilotd_get_name_from_id(pilot_id);
	retval = gpilotd_get_pilot_basedir(pilot_name);

	g_free(pilot_name);

	return retval;
}

/* return val must be freed */
gchar* gpilotd_get_name_from_id(guint32 id) {

	CORBA_char *name;
	gchar *retval;

	name = GNOME_Pilot_Daemon_get_pilot_name_from_id(gpilotddaemon,id,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return NULL;
	}

	retval = g_strdup(name);
	CORBA_free(name);

	if (strlen(retval)==0) {
		g_free(retval);
		return NULL;
	}

	return retval;	
}

guint32 gpilotd_get_id_from_name(gchar *name) {
	CORBA_long id;
	guint32 retval;

	id = GNOME_Pilot_Daemon_get_pilot_id_from_name(gpilotddaemon,name,&ev);
	if (ev._major != CORBA_NO_EXCEPTION) {
		g_warning ("lib-gpilotd:%d: Caught exception: %s",__LINE__, CORBA_exception_id (&ev));
		CORBA_exception_free(&ev);
		return -1;
	}

	retval = id;

	return retval;	
}
