/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* Control applet ("capplet") for the gnome-pilot email conduit,           */
/* based on                                                                 */
/* gpilotd control applet ('capplet') for use with the GNOME control center */

#include <pwd.h>
#include <sys/types.h>
#include <signal.h>
#include <gnome.h>

#include <config.h>
#include <capplet-widget.h>

#include <gpilotd/gpilotd-conduit-mgmt.h>
#include <gpilotd/gpilotd-app.h>
#include <gpilotd/gpilotd-app-dummy-callbacks.h>

#include "email_conduit.h"

/* tell changes callbacks to ignore changes or not */
static gboolean ignore_changes=FALSE;

/* capplet widget */
static GtkWidget *capplet=NULL;

/* host/device/pilot configuration windows */
GtkWidget *cfgOptionsWindow=NULL;
GtkWidget *cfgStateWindow=NULL;
GtkWidget *dialogWindow=NULL;

gboolean activated,org_activation_state;
GnomePilotConduitMgmt *conduit;
ConduitCfg *origState = NULL;
ConduitCfg *curState = NULL;

static void doTrySettings(GtkWidget *widget, gpointer dummy);
static void doRevertSettings(GtkWidget *widget, gpointer dummy);
static void doSaveSettings(GtkWidget *widget, gpointer dummy);
static void doCancelSettings(GtkWidget *widget, gpointer dummy);

static void readOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void setOptionsCfg(GtkWidget *w, ConduitCfg *state);
static void readStateCfg(GtkWidget *w, ConduitCfg *state);
static void setStateCfg(GtkWidget *w, ConduitCfg *state);

gint pilotId;
CORBA_Environment ev;

#define ACTION_OPTIONS_COUNT 2
static gchar* action_options[] = { N_("file"), 
                                   N_("delete") };
static void
setSettings(ConduitCfg* conduitCfg)
{
        if(activated)
                gpilotd_conduit_mgmt_enable(conduit,pilotId,GnomePilotConduitSyncTypeSynchronize);
        else
                gpilotd_conduit_mgmt_disable(conduit,pilotId);
        save_configuration(conduitCfg);
}

static void
doTrySettings(GtkWidget *widget, gpointer dummy)
{
        readStateCfg(cfgStateWindow, curState);
        readOptionsCfg(cfgOptionsWindow, curState);
        setSettings(curState);
}

static void
doSaveSettings(GtkWidget *widget, gpointer dummy)
{
        doTrySettings(widget, curState);
}


static void
doRevertSettings(GtkWidget *widget, gpointer dummy)
{
        activated = org_activation_state;
        destroy_configuration(&curState);
        curState = dupe_configuration(origState);
        setOptionsCfg(cfgOptionsWindow, curState);
        setStateCfg(cfgStateWindow, curState);
        setSettings(curState);
}

static void
doCancelSettings(GtkWidget *widget, gpointer dummy)
{
        setSettings(origState);
}

/* Don't allow any spaces */
static void
insert_callback (GtkEditable    *editable, const gchar    *text,
                 gint len, gint *position, void *data)
{
    gint i;
    gchar *curname;

    curname = gtk_entry_get_text(GTK_ENTRY(editable));
    if (*curname == '\0' && len > 0) {
	if (isspace(text[0])) {
	    gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), "insert_text");
	    return;
	}
    } else {
	for (i=0; i<len; i++) {
	    if (isspace(text[i])) {
		gtk_signal_emit_stop_by_name(GTK_OBJECT(editable), 
					     "insert_text");
		return;
	    }
	}
    }
}

/* make the ok/try/cancel buttons active */
static void
statechange_cb(GtkEditable    *editable, const gchar    *text,
                 gint            length, gint           *position,
                 void *data)
{
        if (!ignore_changes)
                capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

/* option menu callback for 'sendAction' */
static void
sendaction_cb(GtkMenuItem *widget, gpointer data)
{
        curState->sendAction = g_strdup((gchar*)data);
        if(!ignore_changes)
                capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}

/* option menu callback for 'receiverAction' */
static void
recvaction_cb(GtkMenuItem *widget, gpointer data)
{
        curState->receiveAction = g_strdup((gchar*)data);
        if(!ignore_changes)
                capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
}
	
void about_cb (GtkWidget *widget, gpointer data) {
  GtkWidget *about;
  const gchar *authors[] = {_("Eskil Heyn Olsen <deity@eskil.dk>"),NULL};
  
  about = gnome_about_new(_("Gpilotd SendMail conduit"), VERSION,
			  _("(C) 1998 the Free Software Foundation"),
			  authors,
			  _("Configuration utility for the SendMail conduit.\n"
			    "The SendMail conduit is responsible for sending "
			    "out mail composed on the pilot"),
			  _("gnome-unknown.xpm"));
  gtk_widget_show (about);
  
  return;
}

static void toggled_cb(GtkWidget *widget, gpointer data) {
  if(!ignore_changes) {
          gtk_widget_set_sensitive(cfgOptionsWindow,GTK_TOGGLE_BUTTON(widget)->active);
    capplet_widget_state_changed(CAPPLET_WIDGET(capplet), TRUE);
  }
}

static GtkWidget
*createStateCfgWindow(void)
{
    GtkWidget *vbox, *table;
    GtkWidget *entry, *label;
    GtkWidget *button;

    vbox = gtk_vbox_new(FALSE, GNOME_PAD);

    table = gtk_table_new(2, 2, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 4);
    gtk_table_set_col_spacings(GTK_TABLE(table), 10);
    gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

    label = gtk_label_new(_("Enabled"));
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1,2);

    button = gtk_check_button_new();
    gtk_object_set_data(GTK_OBJECT(vbox), "conduit_on_off", button);
    gtk_signal_connect(GTK_OBJECT(button), "toggled",
		       GTK_SIGNAL_FUNC(toggled_cb),
		       NULL);
    gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, 1,2);

    return vbox;
}

static void
setStateCfg(GtkWidget *widget,ConduitCfg *cfg)
{
    GtkWidget *button;
    gchar num[40];

    button = gtk_object_get_data(GTK_OBJECT(widget), "conduit_on_off");

    g_assert(button!=NULL);

    ignore_changes = TRUE;
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button),activated);
    gtk_widget_set_sensitive(cfgOptionsWindow,GTK_TOGGLE_BUTTON(button)->active);
    ignore_changes = FALSE;
}


static void
readStateCfg(GtkWidget *widget,ConduitCfg *cfg)
{
        GtkWidget *button;

        button  = gtk_object_get_data(GTK_OBJECT(widget), "conduit_on_off");
        
        g_assert(button!=NULL);

        activated = GTK_TOGGLE_BUTTON(button)->active;
}

static GtkWidget
*createCfgWindow(void)
{
	GtkWidget *vbox, *table;
	GtkWidget *entry, *label;
        GtkWidget *menuItem, *optionMenu;
        GtkMenu   *menu;
        int i;

	vbox = gtk_vbox_new(FALSE, GNOME_PAD);

	table = gtk_table_new(2, 4, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(table), 10);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, GNOME_PAD);

        /* Get the From Address */
	label = gtk_label_new(_("From E-mail Address"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
    
	entry = gtk_entry_new_with_max_length(128);
	gtk_object_set_data(GTK_OBJECT(vbox), "FromAddress", entry);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 0, 1);
        /* no spaces in the fields */
  	gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
  			   GTK_SIGNAL_FUNC(insert_callback),
  			   NULL);

        /* change the state of the capplet on delete chars as well */
  	gtk_signal_connect(GTK_OBJECT(entry), "delete_text",
  			   GTK_SIGNAL_FUNC(statechange_cb),
  			   NULL);

	gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
                                 GTK_SIGNAL_FUNC(statechange_cb),
                                 NULL);

        /* Get the MH Directory */
	label = gtk_label_new(_("MH Mail directory"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

	entry = gtk_entry_new_with_max_length(256);
	gtk_object_set_data(GTK_OBJECT(vbox), "MailDir", entry);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 1, 2);
        /* no spaces in the fields */
  	gtk_signal_connect(GTK_OBJECT(entry), "insert_text",
  			   GTK_SIGNAL_FUNC(insert_callback),
  			   NULL);

        /* change the state of the capplet on delete chars as well */
  	gtk_signal_connect(GTK_OBJECT(entry), "delete_text",
  			   GTK_SIGNAL_FUNC(statechange_cb),
  			   NULL);

        gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
                                 GTK_SIGNAL_FUNC(statechange_cb),
                                 NULL);

        /* Get sendmail location */
	label = gtk_label_new(_("Mailer location"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

	entry = gtk_entry_new_with_max_length(256);
	gtk_object_set_data(GTK_OBJECT(vbox), "SendmailLoc", entry);
	gtk_table_attach_defaults(GTK_TABLE(table), entry, 1, 2, 2, 3);

        /* change the state of the capplet on delete chars as well */
  	gtk_signal_connect(GTK_OBJECT(entry), "delete_text",
  			   GTK_SIGNAL_FUNC(statechange_cb),
  			   NULL);

        gtk_signal_connect_after(GTK_OBJECT(entry), "insert_text",
                                 GTK_SIGNAL_FUNC(statechange_cb),
                                 NULL);
        
        /* what to do with mail sent from the pilot */
        label = gtk_label_new(_("Mail sent from pilot"));
        gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);

        menu = GTK_MENU(gtk_menu_new());
        for (i = 0; i < ACTION_OPTIONS_COUNT; i++) {
                action_options[i] = _(action_options[i]);
                menuItem = gtk_menu_item_new_with_label(action_options[i]);
                gtk_widget_show(menuItem);
                  gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
                                     GTK_SIGNAL_FUNC(sendaction_cb), 
                                     action_options[i]); 
                gtk_menu_append(menu, menuItem);
        }

        optionMenu = gtk_option_menu_new(); 
        gtk_option_menu_set_menu(GTK_OPTION_MENU(optionMenu),GTK_WIDGET(menu));
        gtk_signal_connect(GTK_OBJECT(menu), "selection-done",
                           GTK_SIGNAL_FUNC(statechange_cb),
                           NULL);

        gtk_table_attach_defaults(GTK_TABLE(table), optionMenu, 1, 2, 3, 4);
	gtk_object_set_data(GTK_OBJECT(vbox), "SendAction", optionMenu);

        /* is the mail on the host computer archived? */
        label = gtk_label_new(_("Mail on Host"));
        gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
        menu = GTK_MENU(gtk_menu_new());
        for (i = 0; i < ACTION_OPTIONS_COUNT; i++) {
                action_options[i] = _(action_options[i]);
                menuItem = gtk_menu_item_new_with_label(action_options[i]);
                gtk_widget_show(menuItem);
                  gtk_signal_connect(GTK_OBJECT(menuItem),"activate",
                                     GTK_SIGNAL_FUNC(recvaction_cb), 
                                     action_options[i]); 
                gtk_menu_append(menu, menuItem);
        }
        optionMenu = gtk_option_menu_new(); 
        gtk_option_menu_set_menu(GTK_OPTION_MENU(optionMenu),GTK_WIDGET(menu));
        gtk_signal_connect(GTK_OBJECT(menu), "selection-done",
                           GTK_SIGNAL_FUNC(statechange_cb),
                           NULL);

        gtk_table_attach_defaults(GTK_TABLE(table), optionMenu, 1, 2, 4, 5);
	gtk_object_set_data(GTK_OBJECT(vbox), "RecvAction", optionMenu);

	return vbox;
}

static void
setOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *fromAddress, *mailDir, *sendmailLoc, *sendAct, *recvAct;
        int i;

	fromAddress = gtk_object_get_data(GTK_OBJECT(pilotcfg), "FromAddress");
	mailDir = gtk_object_get_data(GTK_OBJECT(pilotcfg), "MailDir");
        sendmailLoc = gtk_object_get_data(GTK_OBJECT(pilotcfg), "SendmailLoc");
        sendAct = gtk_object_get_data(GTK_OBJECT(pilotcfg), "SendAction");
        recvAct = gtk_object_get_data(GTK_OBJECT(pilotcfg), "RecvAction");

	g_assert(fromAddress != NULL);
	g_assert(mailDir != NULL);
	g_assert(sendmailLoc != NULL);
  	g_assert(sendAct != NULL); 
  	g_assert(recvAct != NULL);

	ignore_changes = TRUE;

	gtk_entry_set_text(GTK_ENTRY(fromAddress), state->fromAddr);
	gtk_entry_set_text(GTK_ENTRY(mailDir), state->mhDirectory);
	gtk_entry_set_text(GTK_ENTRY(sendmailLoc), state->sendmail);

        /* find the entry in the option menu. if not found, 
           default to the last */
        for(i = 0; i < ACTION_OPTIONS_COUNT && g_strncasecmp(state->sendAction, action_options[i], 20) != 0; i++);
        gtk_option_menu_set_history(GTK_OPTION_MENU(sendAct), i);

        for(i = 0; i < ACTION_OPTIONS_COUNT && g_strncasecmp(state->receiveAction, action_options[i], 20) != 0; i++);
        gtk_option_menu_set_history(GTK_OPTION_MENU(recvAct), i);

	ignore_changes = FALSE;
}

static void
readOptionsCfg(GtkWidget *pilotcfg, ConduitCfg *state)
{
	GtkWidget *fromAddress, *mailDir, *sendmailLoc, *sendAct, *recvAct;

	fromAddress = gtk_object_get_data(GTK_OBJECT(pilotcfg), "FromAddress");
	mailDir = gtk_object_get_data(GTK_OBJECT(pilotcfg), "MailDir");
	sendmailLoc = gtk_object_get_data(GTK_OBJECT(pilotcfg), "SendmailLoc");

	state->fromAddr = g_strdup(gtk_entry_get_text(GTK_ENTRY(fromAddress)));
	state->mhDirectory = g_strdup(gtk_entry_get_text(GTK_ENTRY(mailDir)));
	state->sendmail = g_strdup(gtk_entry_get_text(GTK_ENTRY(sendmailLoc)));
        /* the sendAction and receiveAction were already updated in the cb */
}

static void
pilot_capplet_setup(void)
{
    GtkWidget *frame, *table;

    capplet = capplet_widget_new();

    table = gtk_table_new(1, 2, FALSE);
    gtk_container_border_width(GTK_CONTAINER(table), GNOME_PAD);
    gtk_container_add(GTK_CONTAINER(capplet), table); 

    frame = gtk_frame_new(_("Conduit state"));
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
    gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 0, 1);
    cfgStateWindow = createStateCfgWindow();
    gtk_container_add(GTK_CONTAINER(frame), cfgStateWindow);

    frame = gtk_frame_new(_("SendMail options"));
    gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD_SMALL);
    gtk_table_attach_defaults(GTK_TABLE(table), frame, 0, 1, 1, 2);
    cfgOptionsWindow = createCfgWindow();
    gtk_container_add(GTK_CONTAINER(frame), cfgOptionsWindow);

    gtk_signal_connect(GTK_OBJECT(capplet), "try",
			GTK_SIGNAL_FUNC(doTrySettings), NULL);
    gtk_signal_connect(GTK_OBJECT(capplet), "revert",
			GTK_SIGNAL_FUNC(doRevertSettings), NULL);
    gtk_signal_connect(GTK_OBJECT(capplet), "ok",
			GTK_SIGNAL_FUNC(doSaveSettings), NULL);
    gtk_signal_connect(GTK_OBJECT(capplet), "cancel",
			GTK_SIGNAL_FUNC(doCancelSettings), NULL);
    gtk_signal_connect(GTK_OBJECT(capplet), "help",
			GTK_SIGNAL_FUNC(about_cb), NULL);


    setStateCfg(cfgStateWindow, curState);
    setOptionsCfg(cfgOptionsWindow, curState);

    gtk_widget_show_all(capplet);
}

void run_error_dialog(gchar *mesg,...) {
  char tmp[80];
  va_list ap;

  va_start(ap,mesg);
  vsnprintf(tmp,79,mesg,ap);
  dialogWindow = gnome_message_box_new(mesg,GNOME_MESSAGE_BOX_ERROR,GNOME_STOCK_BUTTON_OK,NULL);
  gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
  va_end(ap);
}

gint get_pilot_id_from_gpilotd() {
	gint *pilots;
	int i;
  
	i=0;
	gpilotd_get_pilot_ids(&pilots);
	if(pilots) {
		while(pilots[i]!=-1) { g_message("pilot %d = \"%d\"",i,pilots[i]); i++; }
		if(i==0) {
			run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
			return -1;
		} else {
			if(i==1) 
				return pilots[0];
			else {
				g_message("too many pilots...");
				return pilots[0];
			}
		}
	} else {
		run_error_dialog(_("No pilot configured, please choose the\n'Pilot Link Properties' capplet first."));
		return -1;
	}  
}

int
main( int argc, char *argv[] )
{
    /* we're a capplet */
    gnome_capplet_init ("email conduit control applet", NULL, argc, argv, 
			NULL,
			0, NULL);

    /* get pilot name from gpilotd */
    /* 1. initialize the gpilotd connection */
    if (gpilotd_init(&argc,argv)!=0) {
	    run_error_dialog(_("Cannot initialze the GnomePilot Daemon"));
	    g_error(_("Cannot initialze the GnomePilot Daemon"));
	    return -1;
    }
    
    /* 2 connect to gpilotd */
    if (gpilotd_connect()!=0) {
	    run_error_dialog(_("Cannot connect to the GnomePilot Daemon"));
	    g_error(_("Cannot connect to the GnomePilot Daemon"));
	    return -1;
    }
    
    pilotId = get_pilot_id_from_gpilotd();
    if(pilotId==-1) return -1;

    /* put all code to set things up in here */
    load_configuration(&origState,pilotId);
    curState = dupe_configuration(origState);
    conduit = gpilotd_conduit_mgmt_new("email_conduit",pilotId);

    org_activation_state = activated = gpilotd_conduit_mgmt_is_enabled(conduit,pilotId);
    
    pilot_capplet_setup();


    /* done setting up, now run main loop */
    capplet_gtk_main();
    return 0;
}    
