#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include "gpilot-structures.h"
#include <libgnome/libgnome.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>

static GList *get_devices (void);
static GList *get_pilots (void);


/* context stuff first */
GPilotContext *gpilot_context_new (void)
{
  gchar str[56];
  gint id;

  GPilotContext *retval;

  retval = (GPilotContext *)g_malloc(sizeof(GPilotContext));
  retval->devices = NULL;
  retval->pilots = NULL;
  retval->user = NULL;

  /* get the id.  Does anyone know the valid range for this? */
  srandom (time (NULL));
  id = 1 + ((guint) 1000000.0*random());
  g_snprintf (str, 56,
	    "/gnome-pilot.d/gpilotd/General/sync_PC_Id=%d",
	    id);
  retval->sync_PC_Id=gnome_config_get_int(str);

  /* if retval is the default value, store it now */
  if (retval == id) 
	  gnome_config_set_int("/gnome-pilot.d/gpilotd/General/sync_PC_Id",retval->sync_PC_Id);


  return retval;
}

void
gpilot_context_init_user (GPilotContext *context)
     /* this will initialize the user context from their config
      * files.  If it has already been initialized, it will reread
      * the files, and free the old data. */
{
  gchar *str;

  if (!context->user)
    context->user = gpilot_user_new ();

  str=getenv("USER");
  if(str) {
    g_free (context->user->username);
    context->user->username = g_strdup (str);
  }
 
  g_list_foreach(context->devices,(GFunc) gpilot_device_free, NULL);
  context->devices = get_devices ();
  g_list_foreach(context->pilots,(GFunc)gpilot_pilot_free,NULL);
  context->pilots = get_pilots ();

  context->sync_PC_Id=gnome_config_get_int("/gnome-pilot.d/gpilotd/General/sync_PC_Id");
}
void
gpilot_context_free (GPilotContext *context)
{
  g_free(context->user->username);
  g_free(context->user);
  g_list_foreach(context->pilots,(GFunc)gpilot_pilot_free,NULL);
  g_list_free(context->pilots);
  g_list_foreach(context->devices,(GFunc)gpilot_device_free,NULL);
  g_list_free(context->devices);
}


/* device stuff next */
GPilotDevice *
gpilot_device_new (void)
{
  GPilotDevice *device;
  device = (GPilotDevice *)g_malloc(sizeof(GPilotDevice));
  device->device_name = NULL;
  device->speed = 0;
  device->fd = 0;
  return device;
}

gint
gpilot_device_init (GPilotDevice *device, gint i)
{
	gchar prefix[40];
	g_return_val_if_fail (device != NULL,-1);
	g_return_val_if_fail (prefix != NULL,-1);
	
	g_snprintf(prefix,39,"/gnome-pilot.d/gpilotd/Device%d/",i);
	gnome_config_push_prefix (prefix);
	g_free (device->device_name);
	device->device_name = gnome_config_get_string("device=/dev/pilot");
	device->fd=open(device->device_name,O_RDWR);
	if(device->fd < 0) {
		g_warning(_("Could not open device %s: reason: \"%s\"."),
			  device->device_name,
			  g_strerror(errno));
		g_free(device->device_name);
		close(device->fd);
		device->fd = 0;
		return -1;
	}
	device->io = g_io_channel_unix_new(device->fd);
	g_io_channel_ref(device->io);
	device->speed = (guint)gnome_config_get_int("speed=57600");
	gnome_config_pop_prefix ();
	return 0;
}

void
gpilot_device_free (GPilotDevice *device)
{
  g_io_channel_close(device->io);
  g_free(device->device_name);
  g_free(device);
}

static GList *
get_devices (void)
{
  GList * retval = NULL;
  int n,i,final;
  
  final = n = gnome_config_get_int("/gnome-pilot.d/gpilotd/General/num_devices=0");

  if(n==0) {
    g_warning("number of devices is configured to 0");
    g_warning("Please run gpilotd-control-applet (use gnomecc) to configure.");    
  }

  for(i=0;i<n;i++) {
    GPilotDevice *device;

    device = gpilot_device_new ();

    if(gpilot_device_init (device, i)==0)
	    retval = g_list_append (retval, device);
    else
	    final --;
  }

  if (final == 0) {
    g_warning(_("No accessible devices available"));
  }

  return retval;
}
static GList *
get_pilots (void)
{
  GList * retval = NULL;
  int n, i;
  
  n = gnome_config_get_int("/gnome-pilot.d/gpilotd/General/num_pilots=0");

  if(n==0) {
    g_warning("number of pilots is configured to 0");
    g_warning("Winging it anyways...");
    g_warning("Please run gpilotd-control-applet (eg. use gnomecc) to configure.");
  }


  for(i=0;i<n;i++) {
    GPilotPilot *pilot;
    pilot = gpilot_pilot_new ();
    gpilot_pilot_init (pilot, i);
    retval = g_list_append (retval, pilot);
  }
  return retval;
}

