#ifndef __MEMO_FILE_CONDUIT_H__
#define __MEMO_FILE_CONDUIT_H__

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <pi-appinfo.h>
#include <pi-memo.h>
#include <glib.h>
#include <gnome.h>
#include <errno.h>

#include <gpilotd/gnome-pilot-conduit.h>
#include <gpilotd/gnome-pilot-conduit-standard-abs.h>

#define MC_DEBUG

#ifdef MC_DEBUG
#define LOG(format,args...) g_log (G_LOG_DOMAIN, \
                                   G_LOG_LEVEL_MESSAGE, \
                                   "%s:%d "##format,__FILE__,__LINE__, ##args)
#else
#define LOG(format,args...)
#endif

#define OBJ_DATA_CONDUIT "conduit_data"
#define OBJ_DATA_CONFIG  "conduit_config"
#define CONFIG_PREFIX    "/gnome-pilot.d/memo_file-conduit/Pilot_%u/"


typedef struct _MemoLocalRecord MemoLocalRecord;
struct _MemoLocalRecord {
  LocalRecord local;

  MemoLocalRecord *next;

  time_t mtime;
  gint category;

  gint length;
  gchar *record;
  gchar *filename;
};

typedef struct _ConduitData ConduitData;

struct _ConduitData {
  struct MemoAppInfo ai;
  GList *records;
  GnomePilotDBInfo *dbi;
};

#define GET_CONDUIT_CFG(s) ((ConduitCfg*)gtk_object_get_data(GTK_OBJECT(s),OBJ_DATA_CONFIG))
#define GET_CONDUIT_DATA(s) ((ConduitData*)gtk_object_get_data(GTK_OBJECT(s),OBJ_DATA_CONDUIT))

typedef struct IterateData {
  int flag;
  int archived;
  MemoLocalRecord *prev;
  MemoLocalRecord *first;
} IterateData;

typedef struct LoadInfo {
  recordid_t id;
  gint secret;
  time_t mtime;
} LoadInfo;


typedef struct _ConduitCfg ConduitCfg;

struct _ConduitCfg 
{
	GnomePilotConduitSyncType  sync_type;   /* only used by capplet */
	mode_t   file_mode;
	mode_t   dir_mode;
	guint32  pilotId;
	gchar   *dir;
	gchar   *ignore_start;
	gchar   *ignore_end;

	gboolean open_secret;
	mode_t   secret_mode;
};


static void load_configuration(ConduitCfg **c,guint32 pilotId) 
{
  gchar prefix[256];
  gchar *buf;
  g_return_if_fail(c!=NULL);

  g_snprintf(prefix,255,CONFIG_PREFIX,pilotId);
 
  *c = g_new0(ConduitCfg,1);
  gnome_config_push_prefix(prefix);
  (*c)->sync_type = GnomePilotConduitSyncTypeCustom; /* this will be reset by capplet */
  (*c)->open_secret = gnome_config_get_bool("open secret=FALSE"); 

  buf = gnome_config_get_string("file mode=0600");
  (*c)->file_mode =(mode_t)strtol(buf,NULL,0);
  g_free(buf);

  buf = gnome_config_get_string("dir mode=0700");
  (*c)->dir_mode =(mode_t)strtol(buf,NULL,0);
  g_free(buf);

  buf = gnome_config_get_string("secret mode=0600");
  (*c)->secret_mode =(mode_t)strtol(buf,NULL,0);
  g_free(buf);
  
  (*c)->dir = gnome_config_get_string("dir");
  (*c)->ignore_end=gnome_config_get_string("ignore end");
  (*c)->ignore_start=gnome_config_get_string("ignore start");
  gnome_config_pop_prefix();
  if((*c)->dir==NULL) {
    (*c)->dir =g_strdup_printf("%s/pilotMemos",g_get_home_dir());
  } 

  (*c)->pilotId = pilotId;
}

static void save_configuration(ConduitCfg *c) 
{
  gchar prefix[256];

  g_return_if_fail(c!=NULL);
  g_snprintf(prefix,255,CONFIG_PREFIX,c->pilotId);

  gnome_config_push_prefix(prefix);
  gnome_config_set_bool("open secret",c->open_secret); 
  g_snprintf(prefix,255,"0%o",c->secret_mode);
  gnome_config_set_string("secret mode",prefix);
  g_snprintf(prefix,255,"0%o",c->file_mode);
  gnome_config_set_string("file mode",prefix);
  g_snprintf(prefix,255,"0%o",c->dir_mode);
  gnome_config_set_string("dir mode",prefix);
  gnome_config_set_string("dir",c->dir);
  gnome_config_set_string("ignore end",c->ignore_end);
  gnome_config_set_string("ignore start",c->ignore_start);
  gnome_config_pop_prefix();

  gnome_config_sync();
  gnome_config_drop_all();

}

static void 
copy_configuration(ConduitCfg *d, ConduitCfg *c)
{
  g_return_if_fail(c!=NULL);
  g_return_if_fail(d!=NULL);
  d->sync_type=c->sync_type;
  if(d->dir) g_free(d->dir);
  d->dir = g_strdup(c->dir);
  if(d->ignore_start) g_free(d->ignore_start);
  d->ignore_start = g_strdup(c->ignore_start);
  if(d->ignore_end) g_free(d->ignore_end);
  d->ignore_end = g_strdup(c->ignore_end);
  d->file_mode = c->file_mode;
  d->dir_mode = c->dir_mode;
  d->secret_mode = c->secret_mode;
  d->open_secret = c->open_secret;
  d->pilotId = c->pilotId;
}

static ConduitCfg*
dupe_configuration(ConduitCfg *c) 
{
  ConduitCfg *d;
  g_return_val_if_fail(c!=NULL,NULL);
  d = g_new0(ConduitCfg,1);
  d->dir=NULL;
  copy_configuration(d,c);
  return d;
}


/** this method frees all data from the conduit config */
static void destroy_configuration(ConduitCfg **c) 
{
  g_return_if_fail(c!=NULL);
  if((*c)->dir) g_free((*c)->dir);
  if((*c)->ignore_start) g_free((*c)->ignore_start);
  if((*c)->ignore_end) g_free((*c)->ignore_end);
  g_free(*c);
  *c = NULL;
}

#endif
