/* handles pilot issues */

#include <config.h>
#include <gnome.h>

#include "pilot.h"

const char *deviceList[] = {"/dev/pilot", 
			    "/dev/ttyS0",
			    "/dev/ttyS1",
			    "/dev/ttyS2",
			    "/dev/ttyS3", NULL};
const guint  speedList[] = {9600, 19200, 38400, 57600, 115200, 0};


/* create new pilotstate structure, initialize to sane state */
PilotState
*newPilotState(void)
{
    PilotState *p;

    p = g_new0(PilotState, 1);

    return p;
}

void
copyPilotState(PilotState *dest, PilotState *src)
{
    memcpy(dest, src, sizeof(PilotState));
}

PilotState
*dupPilotState(PilotState *src)
{
    PilotState *p;
    
    p = g_new0(PilotState, 1);
    copyPilotState(p, src);
    return p;
}

gint
loadHostID(PilotState *p)
{
    guint32  id;
    gboolean notfound;

    gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
    id = gnome_config_get_int_with_default("sync_PC_Id=0", &notfound);
    gnome_config_pop_prefix();
  
    if (notfound) {
	p->syncPCid = random();
	return -1;
    } else {
	p->syncPCid = id;
	return 0;
    }
}    

gint
loadDeviceCfg(PilotState *p)
{
  GPilotDevice *device;
  gchar prefix[256];
  int num;

  gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
  num = gnome_config_get_int("num_devices=0");
  gnome_config_pop_prefix();

  if (num == 0) {
      g_message("No pilot craddle information located");
      p->device = NULL;
      return -1;
  } else if (num > 1) {
      g_message("More than one set of cradle information located.");
      g_message("This tool only handles a single cradle, aborting...");
      run_error_dialog(_("More than one set of cradle information located.\n"
			 "This tool only handles a single cradle, aborting..."));
      exit(1);
  } else {
	  device = g_new0(GPilotDevice, 1);
	  gnome_config_push_prefix("/gnome-pilot.d/gpilotd/Device0/");
	  device->device_name = gnome_config_get_string("device");
	  g_message("cradle device name -> %s",device->device_name);
	  device->speed = gnome_config_get_int("speed=9600");
	  g_message("Pilot Speed  -> %d",device->speed);  
	  gnome_config_pop_prefix();
  }
  
  p->device = device;
  
  return 0;
}


gint
loadPilotPilot(PilotState *p)
{
  GPilotPilot *pilot;
  gboolean notfound;
  int num;

  gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
  num = gnome_config_get_int("num_pilots=0");
  gnome_config_pop_prefix();

  if (num == 0) {
      g_message("No pilot userid/username information located");
      p->pilot = NULL;
      return -1;
  } else if (num > 1) {
      g_message("More than one set of pilot information located.");
      g_message("This tool only handles a single pilot, aborting...");
      run_error_dialog(_("More than one set of pilot information located.\n"
			 "This tool only handles a single pilot, aborting..."));
      exit(1);
  } else {
	  gchar *tmp;
	  pilot = g_new0(GPilotPilot, 1);
	  gnome_config_push_prefix("/gnome-pilot.d/gpilotd/Pilot0/");
	  pilot->name = gnome_config_get_string("name=My Pilot");
	  g_message("Pilot name -> %s",pilot->name);
	  pilot->pilot_id = gnome_config_get_int_with_default("pilotid",&notfound);
	  if (notfound)
		  pilot->pilot_id = getuid();
	  g_message("Pilot id   -> %d",pilot->pilot_id);
	  pilot->pilot_username=gnome_config_get_string("pilotusername");
	  g_message("Pilot username -> %s",pilot->pilot_username);
	  
	  tmp = gnome_config_get_string("default_sync_action=custom");
	  g_message("Pilot sync_action -> %s",tmp);
	  p->sync_type = gnome_pilot_conduit_sync_type_str_to_int(tmp);
	  g_free(tmp);
	  
	  /* set these to something sane since we arent using them */
	  pilot->trusted_users = NULL;
	  pilot->number = 0;
	  
	  gnome_config_pop_prefix();
  }

  p->pilot = pilot;
  
  return 0;
}

/* allocates a pilotstate, load pilot state, return 0 if ok, -1 otherwise*/
gint
loadPilotState(PilotState **pilotState)
{
    PilotState *p;

    p = newPilotState();

    /* load host information */
    if (loadHostID(p) < 0) {
	g_message("Unable to load host id information, assuming unset");
    }

    if (loadPilotPilot(p) < 0) {
	g_message("Unable to load pilot id/username, assuming unset");	
    }

    if (loadDeviceCfg(p) < 0) {
	g_message("Unable to load pilot craddle info, assuming unset");
    }

    *pilotState = p;
    return 0;
}


gint
savePilotState(PilotState *state)
{
  gnome_config_clean_section("/gnome-pilot.d/gpilotd/General/");

  gnome_config_set_int("/gnome-pilot.d/gpilotd/General/sync_PC_Id",state->syncPCid);

  gnome_config_clean_section("/gnome-pilot.d/gpilotd/Device0/");
  if (state->device) {
      gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
      gnome_config_set_int("num_devices", 1);
      gnome_config_pop_prefix();

      gnome_config_push_prefix("/gnome-pilot.d/gpilotd/Device0/");
      gnome_config_set_string("device", state->device->device_name);
      gnome_config_set_int("speed", state->device->speed);
      gnome_config_pop_prefix();
  }

  gnome_config_clean_section("/gnome-pilot.d/gpilotd/Pilot0/");
  if (state->pilot) {
      gnome_config_push_prefix("/gnome-pilot.d/gpilotd/General/");
      gnome_config_set_int("num_pilots", 1);
      gnome_config_pop_prefix();

      gnome_config_push_prefix("/gnome-pilot.d/gpilotd/Pilot0/");
      gnome_config_set_string("name", state->pilot->name);
      gnome_config_set_int("pilotid", state->pilot->pilot_id);
      gnome_config_set_string("pilotusername", state->pilot->pilot_username);
      gnome_config_set_string("default_sync_action", 
			      gnome_pilot_conduit_sync_type_int_to_str(state->sync_type));
      gnome_config_pop_prefix();
  }

  gnome_config_sync();
  return 0;
}



