#include <config.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <errno.h>
#include <gnome.h>
#include <gdk_imlib.h>
#include "applet-widget.h"

/* headers for select & child stuff */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <signal.h>

#include "gpilotd/gpilotd-app.h"

enum {
  TARGET_URI_LIST,
};

typedef struct _pilot_properties pilot_properties;

struct _pilot_properties {
	gchar * pilot_id;
	gint  text_limit;
	gchar * exec_when_clicked;
	gchar * backupdir;
	GList *install_files;
	gboolean popups;
	pid_t child_alive;
};

static pilot_properties properties = { NULL };

typedef enum { initializing,syncing,waiting, num_states } state;

char *pixmaps[] = 
{
    "sync_broken.png",
    "syncing_icon.png",
    "sync_icon.png",
};

static state curstate;
static GtkWidget *applet;
static GtkWidget *pixmap; 
static GtkWidget *dialogWindow; 
static GtkWidget *pb;
/* static GtkWidget *dialog; */
guint timeout_handler_id;
gboolean double_clicked;

static char *pilot_file_drop_types[] = {"file:ALL"};
void show_error_dialog(gchar*,...);
void show_warning_dialog(gchar*,...);
void show_message_dialog(gchar*,...);

/******************************************************************/

void gpilotd_connect_cb(const gchar *id,const GNOME_Pilot_PilotUser *user) {
  applet_widget_set_tooltip(APPLET_WIDGET(applet),_("Synchronizing..."));
  
  if(!GTK_WIDGET_REALIZED(pixmap)) {
    g_warning("! realized");
    return;
  }
  curstate = syncing;
  pilot_draw(pixmap);
}

void gpilotd_disconnect_cb(const gchar *id) {
  applet_widget_set_tooltip(APPLET_WIDGET(applet),_("Ready to synchronize"));

  curstate = waiting;
  pilot_draw(pixmap);
}

void gpilotd_request_completed(const gchar *id,unsigned long handle) {
	show_message_dialog("Pilot %s has completed request %d",
			    strlen(id)==0?"(noname)":id,
			    handle);
}

void gpilotd_userinfo_requested(const gchar *device,const GNOME_Pilot_PilotUser *user) {
}

void gpilotd_sysinfo_requested(const gchar *id,const GNOME_Pilot_SysInfo *info) {
  g_message("pilot %s",id);
}

void gpilotd_conduit_start(const gchar *id, 
			   const gchar *conduit_name, 
			   const gchar *db_name) { 
	g_message("conduit_start(%s,%s,%s)",id,conduit_name,db_name);
}

void gpilotd_conduit_end(const gchar *id) { 
	g_message("conduit_end(%s)",id);
}

/******************************************************************/

gchar *get_pilot_id_from_gpilotd() {
  gchar **pilots;
  int i;
  
  i=0;
  gpilotd_get_pilot_names(&pilots);
  if(pilots) {
    while(pilots[i]) { g_message("pilot %d = \"%s\"",i,pilots[i]); i++; }
    if(i==1) 
      return pilots[0];
    else {
      g_message("too many pilots...");
      return pilots[0];
    }
  } else
    g_error("gpilotd says 'no pilots' !");
}

char *file_type(char *name) {
  static char command[128];
  char *tmp;
  FILE *f;

  if(!access(name,R_OK)) return NULL;

  tmp = tmpnam(NULL);
  snprintf(command,127,"file \"%s\" > %s",name,tmp);
  if(system(command)==-1) {
    return NULL;
  }
  
  if((f = fopen(tmp,"rt"))==NULL) {
    return NULL;
  }

  fgets(command,128,f);
  fclose(f);
  unlink(tmp);

  tmp = (char*)strstr(command,": "); tmp+=2;
  return tmp;
}

long file_size(char *name) {
  FILE *FS;
  long ret;

  if((FS = fopen(name,"rt"))==NULL) {
    return 0;
  }
  fseek(FS,0L,SEEK_END);
  ret = ftell(FS);
  fclose(FS);
  return ret;
}
/******************************************************************/

pilot_draw(GtkWidget *pixmap)
{
  if(!GTK_WIDGET_REALIZED(pixmap)) {
    g_warning("pilot_draw ! realized");
    return;
  }

  gnome_pixmap_load_file(GNOME_PIXMAP(pixmap), pixmaps[curstate]);
  
}

/******************************************************************/

static int 
timeout(gpointer data) 
{
	switch(curstate) {
		
	case initializing: {
		g_message("state = initializing");
		pilot_draw(GTK_WIDGET(data));
		
		if (gpilotd_connect()==0) {
			curstate = waiting;
			pilot_draw(GTK_WIDGET(data));
		} else {
			//timeout_handler_id = gtk_timeout_add(3000,timeout,data);
		}
		return TRUE;
	} break;
	
	case waiting: {
		g_message("state = waiting");
		gtk_timeout_remove(timeout_handler_id);
		
		if (properties.pilot_id) g_free(properties.pilot_id);
		properties.pilot_id = get_pilot_id_from_gpilotd();
		g_message("pilot_applet: monitoring %s",properties.pilot_id);
		gpilotd_monitor_on(properties.pilot_id);
		/*gpilotd_monitor_on_all_pilots();*/
		
		gpilotd_notify_on(GNOME_Pilot_NOTIFY_CONNECT);
		gpilotd_notify_on(GNOME_Pilot_NOTIFY_CONDUIT);
		gpilotd_notify_on(GNOME_Pilot_NOTIFY_DISCONNECT);
		
		applet_widget_set_tooltip(APPLET_WIDGET(applet),_("Ready to Synchronize"));
		
		pilot_draw(GTK_WIDGET(data));    
	} break;
	case syncing: break;
	default: g_error("curstate == default ?!"); break;
	}
	
	return TRUE;
}

void ChildDeathSignal(int signal)
{
  g_message("in ChildDeathSignal\n");
  waitpid(properties.child_alive,NULL,WNOHANG);
  properties.child_alive = -1;
}

void 
show_error_dialog(gchar *mesg,...) 
{
	char *tmp;
	va_list ap;

	va_start(ap,mesg);
	tmp = g_strdup_vprintf(mesg,ap);
	dialogWindow = gnome_message_box_new(tmp,GNOME_MESSAGE_BOX_ERROR,
					     GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	g_free(tmp);
	va_end(ap);
}

void 
show_warning_dialog(gchar *mesg,...) 
{
	char *tmp;
	va_list ap;

	va_start(ap,mesg);
	tmp = g_strdup_vprintf(mesg,ap);
	dialogWindow = gnome_message_box_new(tmp,GNOME_MESSAGE_BOX_WARNING,
					     GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	g_free(tmp);
	va_end(ap);
}


void 
show_message_dialog(char *mesg,...)
{
	char *tmp;
	va_list ap;

	va_start(ap,mesg);
	tmp = g_strdup_vprintf(mesg,ap);
	dialogWindow = gnome_message_box_new(tmp,GNOME_MESSAGE_BOX_GENERIC,
					     GNOME_STOCK_BUTTON_OK,NULL);
	gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	g_free(tmp);
	va_end(ap);
}

static void
load_properties(char *cfgpath)
{
  char *query;

  if (properties.exec_when_clicked) g_free(properties.exec_when_clicked);
  
  gnome_config_push_prefix(cfgpath);
  properties.exec_when_clicked = gnome_config_get_string("exec_when_clicked=gpilotd-control-applet");
  properties.backupdir = gnome_config_get_string("backupdir=");
  properties.text_limit = gnome_config_get_int("text_limit=10240");
  properties.popups = gnome_config_get_bool("pop-ups=true");

  gnome_config_pop_prefix();
}

void
save_properties(const char *cfgpath,
		const char *globcfgpath) {
  gnome_config_push_prefix(cfgpath);
  gnome_config_set_string("exec_when_clicked",properties.exec_when_clicked);
  gnome_config_set_string("backupdir",properties.backupdir);
  gnome_config_set_int("text_limit",properties.text_limit);
  gnome_config_set_bool("pop-ups",properties.popups);

  gnome_config_pop_prefix();
  gnome_config_sync();
  gnome_config_drop_all();
  
}

static void
apply_restore_directory_cb(GnomePropertyBox * pb, gint page, gpointer data)
{
	gchar * new_list;
	
	if (page != -1) return; /* Only honor global apply */
	
	new_list = gtk_entry_get_text(GTK_ENTRY(data));
	
	if (new_list) {
		g_free(properties.backupdir);
		properties.backupdir = g_strdup(new_list);
	}
}

static void
apply_exec_when_clicked_cb(GnomePropertyBox * pb, gint page, gpointer data)
{
  gchar * new_list;

  if (page != -1) return; /* Only honor global apply */

  new_list = gtk_entry_get_text(GTK_ENTRY(data));

  if (new_list) {
    g_free(properties.exec_when_clicked);
    properties.exec_when_clicked = g_strdup(new_list);
  }

}

static void
toggle_notices_cb(GtkWidget *toggle,gpointer data) 
{
	properties.popups = GTK_TOGGLE_BUTTON(toggle)->active;
}

static void
apply_properties_cb(GtkWidget* pb,gpointer data)
{
	save_properties(APPLET_WIDGET(applet)->privcfgpath,
			APPLET_WIDGET(applet)->globcfgpath);
}

static void
set_toggle_button_value (GtkWidget *widget, gpointer data)
{
        if(GTK_TOGGLE_BUTTON(widget)->active)
                *(gboolean *)data=TRUE;
        else
                *(gboolean *)data=FALSE;
        if(pb)
	  gnome_property_box_changed (GNOME_PROPERTY_BOX (pb));
}

static void cancel_restore(GnomeDialog *w,gpointer data) {
	g_message("cancelling %d",GPOINTER_TO_INT(data));
	gpilotd_remove_request(GPOINTER_TO_INT(data));  
}

static void
restore_dialog(AppletWidget * w, gpointer data)
{
	GtkWidget *dialogWindow;
	int handle;
	gchar *backupdir;
	gchar *backupdir_key;

	if ( properties.backupdir == NULL ) {
		show_warning_dialog(_("No directory to restore from.\n" "Please set the directory in the properties dialog."));
		return;
	}
		
	handle = gpilotd_restore(properties.pilot_id,backupdir,GNOME_Pilot_PERSISTENT,0);

	if (handle>0) {
		dialogWindow = gnome_message_box_new(_("Press synchronize on the cradle to restore\n" 
						       " or cancel the operation."),
						     GNOME_MESSAGE_BOX_GENERIC,
						     GNOME_STOCK_BUTTON_CANCEL,
						     NULL);
		gnome_dialog_button_connect(GNOME_DIALOG(dialogWindow),0,
					    cancel_restore,GINT_TO_POINTER(handle));
		gnome_dialog_run_and_close(GNOME_DIALOG(dialogWindow));
	} else {
		show_warning_dialog(_("Restore request failed"));
	}
}
static void 
properties_dialog(GtkWidget *w, gpointer data)
{
  GtkWidget 
	  *vbox,
	  *hbox,
	  *button,
	  *entry;
  static GnomeHelpMenuEntry help_entry = {"pilot_applet","properties.html"};

  pb = gnome_property_box_new();

  gtk_window_set_title(GTK_WINDOW(pb), _("GNOME PilotSync Properties"));

  /* comm options page */

  vbox = gtk_vbox_new(GNOME_PAD_SMALL, FALSE);
  gtk_container_border_width(GTK_CONTAINER(vbox), GNOME_PAD_SMALL);
  hbox = gtk_hbox_new(GNOME_PAD_SMALL, FALSE);
  gtk_container_border_width(GTK_CONTAINER(hbox), GNOME_PAD_SMALL);


  /* execute when clicked */
  hbox = gtk_hbox_new(GNOME_PAD_SMALL, FALSE);
  gtk_container_border_width(GTK_CONTAINER(hbox), GNOME_PAD_SMALL);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, GNOME_PAD_SMALL);
  entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(entry), properties.exec_when_clicked);
  gtk_signal_connect_object(GTK_OBJECT(entry), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(pb));
  gtk_signal_connect(GTK_OBJECT(pb), "apply", 
		     GTK_SIGNAL_FUNC(apply_exec_when_clicked_cb),entry);
  gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Execute when clicked ")), FALSE, FALSE, GNOME_PAD_SMALL);
  gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, TRUE, GNOME_PAD_SMALL);

  /* backupdir */
  hbox = gtk_hbox_new(GNOME_PAD_SMALL, FALSE);
  gtk_container_border_width(GTK_CONTAINER(hbox), GNOME_PAD_SMALL);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, GNOME_PAD_SMALL);
  entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(entry), properties.backupdir);
  gtk_signal_connect_object(GTK_OBJECT(entry), "changed",
			    GTK_SIGNAL_FUNC(gnome_property_box_changed),
			    GTK_OBJECT(pb));
  gtk_signal_connect(GTK_OBJECT(pb), "apply", 
		     GTK_SIGNAL_FUNC(apply_restore_directory_cb),entry);
  gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Directory to restore from ")), FALSE, FALSE, GNOME_PAD_SMALL);
  gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, TRUE, GNOME_PAD_SMALL);

  /* popups */
  hbox = gtk_hbox_new(GNOME_PAD_SMALL, FALSE);
  gtk_container_border_width(GTK_CONTAINER(hbox), GNOME_PAD_SMALL);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, GNOME_PAD_SMALL);
  
  button = gtk_check_button_new();
  gtk_signal_connect(GTK_OBJECT(button), "toggled",
		     GTK_SIGNAL_FUNC(toggle_notices_cb),
		     NULL);
  gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Display notices ")), FALSE, FALSE, GNOME_PAD_SMALL);
  gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, TRUE, GNOME_PAD_SMALL);

  /* * */

  gnome_property_box_append_page(GNOME_PROPERTY_BOX(pb), vbox,
				 gtk_label_new(_("Applet options")));

  gtk_signal_connect(GTK_OBJECT(pb), 
		     "apply", 
		     GTK_SIGNAL_FUNC(apply_properties_cb),
		     NULL);
  gtk_signal_connect(GTK_OBJECT(pb),
		     "destroy",
		     GTK_SIGNAL_FUNC(gtk_widget_destroy),
		     pb);
  gtk_signal_connect(GTK_OBJECT(pb),
		     "help",
		     GTK_SIGNAL_FUNC(gnome_help_pbox_display),
		     &help_entry);

  gtk_widget_show_all(pb);
}

static gint 
pilot_clicked_cb(GtkWidget * widget, 
		 GdkEventButton * e, 
		 gpointer data)
{
	int err = -1;
	const char *targv[2];
	
	targv[0]=properties.exec_when_clicked;
	targv[1]=NULL;
	g_message("pilot_clicked_cb(%s)",properties.exec_when_clicked);
	err = gnome_execute_async(NULL,1,targv);
	if(err==-1) {
		show_warning_dialog(_("Execution of %s failed"),properties.exec_when_clicked);
	} else {
		sleep(1);
	}

	return TRUE; 
}

/*
static gint
pilot_timeout(gpointer data)
{
	GtkWidget *pixmap = data;

	gtk_widget_queue_draw(pixmap);
	return TRUE;
}
*/

static void
dnd_drop_internal(GtkWidget        *widget,
		    GdkDragContext   *context,
		    gint              x,
		    gint              y,
		    GtkSelectionData *selection_data,
		    guint             info,
		    guint             time,
		    gpointer          data)
{
  int cnt;
  char *ptr;

  GList *names, *templist;
  g_message("dnd");
  switch (info)
    {
    case TARGET_URI_LIST:
      names = gnome_uri_list_extract_filenames ((char *)selection_data->data);
      for (templist = names; templist;templist = templist->next) {
	char *ptr=templist->data;
	  if(ptr) {
	    /*
	    if(strstr(file_type(ptr),"text")) {
	      if(file_size(ptr)>properties.text_limit)
		g_message("installing textfile as Doc (not implemented)");
	      else
		g_message("installing textfile as Memo (not implemented)");
	    }
	    */
	    properties.install_files = g_list_append(properties.install_files,g_strdup ((gchar*)templist->data));
	    g_message("installing %s",(gchar*)templist->data);
	    gpilotd_install_file(properties.pilot_id,
				 (gchar*)templist->data,
				 GNOME_Pilot_PERSISTENT,0);
	  }
      }
	
      gnome_uri_list_free_strings (names);
      break;
    default:
      g_warning("unknown dnd type");
      break;
    }
}


void
about_cb (AppletWidget *widget, gpointer data)
{
  GtkWidget *about;
  const gchar *authors[] = {"Eskil Heyn Olsen, deity@dbc.dk","and the people of pilot-link",NULL};
  gchar version[] = VERSION;

  about = gnome_about_new(_("Gnome Pilot Applet"), 
			  version,
			  _("(C) 1998 Eskil Heyn Olsen"),
			  (const gchar**)authors,
			  _("A dnd and gui monitor for gpilotd."),
			  NULL);
  gtk_widget_show (about);

  return;
}

static gint
applet_save_session(GtkWidget *w,
		    const char *cfgpath,
		    const char *globcfgpath,
		    gpointer data)
{
  g_message("saving session");
  save_properties(cfgpath,globcfgpath);
  return FALSE;
}

static void
applet_destroy(GtkWidget *applet,gpointer horse)
{
  g_message("destroy pilot_applet");
}

static GtkWidget *
create_pilot_widget(GtkWidget *window)
{
  GtkWidget *event_box;
  GtkStyle *style;
  int i;
  
  static GtkTargetEntry drop_types [] = { 
    { "text/uri-list", 0, TARGET_URI_LIST },
  };
  static gint n_drop_types = sizeof (drop_types) / sizeof(drop_types[0]);

  style = gtk_widget_get_style(window);

  applet_widget_set_tooltip(APPLET_WIDGET(applet),_("Trying to connect to "
						    "the GnomePilot Daemon"));


  curstate = initializing;

  for (i = 0; i < sizeof(pixmaps)/sizeof(pixmaps[0]); i++)
      pixmaps[i] = gnome_unconditional_pixmap_file(pixmaps[i]);
  
  event_box = gtk_event_box_new();
  
  pixmap = gnome_pixmap_new_from_file(pixmaps[curstate]);

  
  gtk_widget_set_events(event_box, gtk_widget_get_events(applet) |
			GDK_BUTTON_PRESS_MASK);
  gtk_signal_connect(GTK_OBJECT(event_box), "button-press-event",
		     GTK_SIGNAL_FUNC(pilot_clicked_cb), NULL);


  gtk_widget_show(pixmap);  
  gtk_container_add(GTK_CONTAINER(event_box), pixmap);

  timeout_handler_id = gtk_timeout_add(1000,timeout,pixmap);
  
  gtk_signal_connect(GTK_OBJECT(applet),"session-save",
		     GTK_SIGNAL_FUNC(applet_save_session),
		     NULL);
  gtk_signal_connect(GTK_OBJECT(applet),"destroy",
		    GTK_SIGNAL_FUNC(applet_destroy),NULL);

  applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					"restore",
					GNOME_STOCK_MENU_REFRESH,
					_("Restore..."),
					restore_dialog,
					NULL);	

  applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					"properties",
					GNOME_STOCK_MENU_PROP,
					_("Properties..."),
					GTK_SIGNAL_FUNC(properties_dialog),
					NULL);	
  applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					"about",
					GNOME_STOCK_MENU_ABOUT,
					_("About..."),
					about_cb,
					NULL);
  /* Now set the dnd features */
  gtk_drag_dest_set (GTK_WIDGET (pixmap),
		     GTK_DEST_DEFAULT_ALL,
		     drop_types, n_drop_types,
		     GDK_ACTION_COPY);


  gtk_signal_connect(GTK_OBJECT(pixmap),
		     "drag_data_received",
		     GTK_SIGNAL_FUNC(dnd_drop_internal),
		     NULL);


  return event_box;
}

static GtkWidget * applet_start_new_applet(const gchar *goad_id, const char **params, int nparams) {
  GtkWidget *pilot;
  GtkWidget *applet;
  
  if(strcmp(goad_id, "pilot_applet")) return NULL;
     
  applet = applet_widget_new(goad_id);
  if (!applet)
    g_error("Can't create applet!\n");
  
  gtk_widget_realize(applet);
  pilot = create_pilot_widget(applet);
  gtk_widget_show_all(pilot);


  applet_widget_add(APPLET_WIDGET(applet), pilot);
  gtk_widget_show(applet);
  
  return applet;
}



static CORBA_Object
pilot_applet_activator(PortableServer_POA poa,
		       const char *goad_id,
		       const char **params,
		       gpointer *impl_ptr,
		       CORBA_Environment *ev)
{
  GtkWidget *pilot;

  applet = applet_widget_new(goad_id);
  if (!applet)
    g_error("Can't create applet!\n");

  load_properties(APPLET_WIDGET(applet)->privcfgpath);
  
  gtk_widget_realize(applet);
  pilot = create_pilot_widget(applet);
  gtk_widget_show_all(pilot);
  
  
  applet_widget_add(APPLET_WIDGET(applet), pilot);
  gtk_widget_show(applet);
 
  return applet_widget_corba_activate(applet, poa, goad_id, params,
                                      impl_ptr, ev);
}

static void
pilot_applet_deactivator(PortableServer_POA poa,
			 const char *goad_id,
			 gpointer impl_ptr,
			 CORBA_Environment *ev)
{
  applet_widget_corba_deactivate(poa, goad_id, impl_ptr, ev);
}

#if 0 /* defined(SHLIB_APPLETS) */
static const char *repo_id[]={"IDL:GNOME/Applet:1.0", NULL};
static GnomePluginObject applets_list[] = {
  {repo_id, "pilot_applet", NULL, "Gnome-Pilot applet",
   &pilot_applet_activator, &pilot_applet_deactivator},
  {NULL}
};

GnomePlugin GNOME_Plugin_info = {
  applets_list, NULL
};
#else
int
main(int argc, char *argv[])
{
	GtkWidget *pilot;
	gpointer pilot_applet_impl;
	gchar *goad_id;
  
	/* initialize the i18n stuff */
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	/* initialize applet stuff */
	applet_widget_init("pilot_applet", VERSION, 
			   argc, argv, NULL, 0,NULL);

	/* initialize gpilotd stuff */
	if(gpilotd_init(&argc,argv)!=0) {
		show_error_dialog(_("Cannot initialize the GnomePilot Daemon"));
		g_error(_("Cannot initialize the GnomePilot Daemon"));
		return -1;
	}

	applet_factory_new("pilot_applet_factory", NULL, applet_start_new_applet);

	APPLET_ACTIVATE(pilot_applet_activator, "pilot_applet", &pilot_applet_impl);

	applet_widget_gtk_main();

	APPLET_DEACTIVATE(pilot_applet_deactivator, "pilot_applet", pilot_applet_impl);

	return 0;
}
#endif
