/* The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Mobile Application Link.
 *
 * The Initial Developer of the Original Code is AvantGo, Inc.
 * Portions created by AvantGo, Inc. are Copyright (C) 1997-1999
 * AvantGo, Inc. All Rights Reserved.
 *
 * Contributor(s):
 */

#include <AGDBConfig.h>
#include <AGUtil.h>
#ifndef REMOVE_SYNCHRONIZE_FEATURE
#include <AGSynchronize.h>
#endif

static AGArray *dupNewIdArray(AGArray *src);

ExportFunc AGDBConfig *AGDBConfigNew(char *dbname, AGDBConfigType type, 
                                     AGBool sendRecordPlatformData, 
                                     int32 platformDataLength, 
                                     void *platformData,
                                     AGArray *newids)
{
    AGDBConfig *dbconfig;

    dbconfig = (AGDBConfig*)malloc(sizeof(AGDBConfig));
    if (NULL == dbconfig)
        return NULL;
    return AGDBConfigInit(dbconfig, dbname, type, 
        sendRecordPlatformData, platformDataLength, platformData, newids);
}

ExportFunc AGDBConfig *AGDBConfigInit(AGDBConfig *dbconfig, 
                                      char *dbname, AGDBConfigType type,
                                      AGBool sendRecordPlatformData,
                                      int32 platformDataLength, 
                                      void *platformData,
                                      AGArray *newids)
{
    bzero(dbconfig, sizeof(*dbconfig));
    dbconfig->type = type;
    dbconfig->sendRecordPlatformData = sendRecordPlatformData;
    AGDBConfigSetDBName(dbconfig, dbname);
    AGDBConfigSetPlatformData(dbconfig, platformDataLength, platformData);
    AGDBConfigSetNewIds(dbconfig, newids);

    return dbconfig;
}

ExportFunc void AGDBConfigFinalize(AGDBConfig *dbconfig)
{
    if (dbconfig->dbname)
        free(dbconfig->dbname);
    if (dbconfig->platformData)
        free(dbconfig->platformData);
    if (dbconfig->newids) 
        AGArrayFree(dbconfig->newids);
    bzero(dbconfig, sizeof(*dbconfig));
}

ExportFunc void AGDBConfigFree(AGDBConfig *dbconfig)
{
    AGDBConfigFinalize(dbconfig);
    free(dbconfig);
}

ExportFunc AGDBConfig *AGDBConfigCopy(AGDBConfig *dstdbconfig,
                                      AGDBConfig *srcdbconfig)
{
    void * tmp;

    if (NULL != srcdbconfig->platformData) {
        tmp = malloc(srcdbconfig->platformDataLength);
        if (NULL != tmp)
            memcpy(tmp, srcdbconfig->platformData,
                srcdbconfig->platformDataLength);
    }
    else
        tmp = NULL;

    return AGDBConfigInit(dstdbconfig,
                          strdup(srcdbconfig->dbname),
                          srcdbconfig->type,
                          srcdbconfig->sendRecordPlatformData,
                          srcdbconfig->platformDataLength,
                          tmp,
                          dupNewIdArray(srcdbconfig->newids));

}

ExportFunc AGDBConfig *AGDBConfigDup(AGDBConfig *dbconfig)
{
    AGDBConfig * result;

    result = (AGDBConfig *)malloc(sizeof(AGDBConfig));
    if (NULL != result) {
        bzero(result, sizeof(AGDBConfig));
        AGDBConfigCopy(result, dbconfig);
    }
    return result;
}

ExportFunc void AGDBConfigSetDBName(AGDBConfig *dbconfig, char *dbname)
{
    if (dbconfig->dbname == dbname) {
        return;
    }

    if (dbconfig->dbname != NULL) {
        free(dbconfig->dbname);
    }

    dbconfig->dbname = dbname;
}

ExportFunc void AGDBConfigSetPlatformData(AGDBConfig *dbconfig, 
                                          int32 platformDataLength, 
                                          void *platformData)
{
    dbconfig->platformDataLength = platformDataLength;
    if (dbconfig->platformData == platformData) {
        return;
    }

    if (dbconfig->platformData != NULL) {
        free(dbconfig->platformData);
    }

    dbconfig->platformData = platformData;
}

ExportFunc void AGDBConfigSetNewIds(AGDBConfig *dbconfig, AGArray *newids)
{
    //PENDING(klobad) This scares me, but I'm in Rome.
    if (dbconfig->newids == newids) {
        return;
    }

    if(dbconfig->newids) {
        AGArrayFree(dbconfig->newids);
    }

    dbconfig->newids = newids;
}

ExportFunc void AGDBConfigReadData(AGDBConfig *dbconfig, AGReader *r)
{
    char *dbname;
    int32 i, count;

    dbname = AGReadCString(r);
    dbconfig->type = (AGDBConfigType)AGReadCompactInt(r);
    AGDBConfigSetDBName(dbconfig, dbname);
    dbconfig->sendRecordPlatformData = AGReadBoolean(r);
    dbconfig->platformDataLength = AGReadCompactInt(r);
    dbconfig->platformData = malloc(dbconfig->platformDataLength);
    AGReadBytes(r, dbconfig->platformData, dbconfig->platformDataLength);
    count = AGReadCompactInt(r);
    if(count > 0) {
        dbconfig->newids = AGArrayNew(AGIntegerElements, count);
        for(i = 0; i < count; i++) 
            AGArrayAppend(dbconfig->newids, (void *)AGReadInt32(r));
    }
}

ExportFunc AGDBConfig * AGDBConfigNewAndReadData(AGReader *r)
{
    AGDBConfig * result;

    result = (AGDBConfig *)malloc(sizeof(AGDBConfig));
    if (NULL != result) {
        bzero(result, sizeof(AGDBConfig));
        AGDBConfigReadData(result, r);
    }
    return result;
}

ExportFunc void AGDBConfigWriteData(AGDBConfig *dbconfig, AGWriter *w)
{
    int32 i, count;

    AGWriteCString(w, dbconfig->dbname);
    AGWriteCompactInt(w, dbconfig->type);
    AGWriteBoolean(w, dbconfig->sendRecordPlatformData);
    AGWriteCompactInt(w, dbconfig->platformDataLength);
    AGWriteBytes(w, dbconfig->platformData, dbconfig->platformDataLength);
    if(!dbconfig->newids || AGArrayCount(dbconfig->newids) < 1) {
        AGWriteCompactInt(w, 0);
    } else {
        count = AGArrayCount(dbconfig->newids);
        AGWriteCompactInt(w, count);
        for(i = 0; i < count; i++) 
            AGWriteInt32(w, (uint32)AGArrayElementAt(dbconfig->newids, i));
    }
}

#ifndef REMOVE_SYNCHRONIZE_FEATURE
AGDBConfig * AGDBConfigSynchronize(AGDBConfig *agreed,
                                   AGDBConfig *device,
                                   AGDBConfig *desktop)
{
    AGDBConfig * result;
    
    result = (AGDBConfig *)malloc(sizeof(AGDBConfig));

    if (NULL != result) {

        bzero(result, sizeof(AGDBConfig));

        result->dbname = AGSynchronizeString(agreed->dbname,
            device->dbname,
            desktop->dbname);

        result->type = (AGDBConfigType)AGSynchronizeInt32(
            (int32)agreed->type,
            (int32)device->type,
            (int32)desktop->type);

        result->sendRecordPlatformData =
            AGSynchronizeBoolean(agreed->sendRecordPlatformData,
                device->sendRecordPlatformData,
                desktop->sendRecordPlatformData);

        AGSynchronizeData(&result->platformData,
            &result->platformDataLength,
            agreed->platformData,
            agreed->platformDataLength,
            device->platformData,
            device->platformDataLength,
            desktop->platformData,
            desktop->platformDataLength);

        AGDBConfigSetNewIds(result, dupNewIdArray(device->newids));

    }
    
    return result;
}
#endif /*#ifndef REMOVE_SYNCHRONIZE_FEATURE*/

ExportFunc void AGDBConfigAppendNewId(AGDBConfig *config, 
                                      uint32 tmpUID, uint32 newUID)
{
    if(!config->newids) {
        config->newids = AGArrayNew(AGIntegerElements, 2);
    }
    AGArrayAppend(config->newids, (void *)tmpUID);
    AGArrayAppend(config->newids, (void *)newUID);
}

static AGArray *dupNewIdArray(AGArray *src)
{
    AGArray *newArray;

    if(src) {
        newArray = AGArrayNew(AGIntegerElements, AGArrayCount(src));
        AGArrayAppendArray(newArray, src);
    } else {
        newArray = NULL;
    }
    return newArray;
}

