/* gnome-remote-shell - Wrapper for opening remote shells
 * Copyright (C) 2002 by Rodrigo Moya
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <gconf/gconf-client.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkimage.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtkvbox.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-ui-init.h>

static GConfClient *conf_client;
static GtkWidget *dialog;
static GtkWidget *ssh_method;
static GtkWidget *telnet_method;
static GtkWidget *host_entry;
static GtkWidget *user_entry;
static GtkWidget *port_entry;
static GtkWidget *window_width;
static GtkWidget *window_height;

static void
dialog_destroyed_cb (GtkWidget *widget, gpointer user_data)
{
	gtk_main_quit ();
}

static void
dialog_response_cb (GtkDialog *dialog, gint response_id, gpointer user_data)
{
	GError *err = NULL;

	if (response_id == GTK_RESPONSE_OK) {
		gchar *cmd;
		const gchar *host, *user;
		gdouble port;
		gchar *geometry;

		geometry = g_strdup_printf (
			"--geometry=%dx%d",
			(gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (window_width)),
			(gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (window_height)));

		port = gtk_spin_button_get_value (GTK_SPIN_BUTTON (port_entry));
		host = gtk_entry_get_text (GTK_ENTRY (gnome_entry_gtk_entry (GNOME_ENTRY (host_entry))));
		if (!host || !strlen (host)) {
			gnome_error_dialog_parented (_("No host specified"), GTK_WINDOW (dialog));
			return;
		}
					   
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ssh_method))) {
			user = gtk_entry_get_text (GTK_ENTRY (user_entry));
			if (!user || !strlen (user)) {
				gnome_error_dialog_parented (_("A user name is required for SSH"),
							     GTK_WINDOW (dialog));
				return;
			}

			if (port > 0.0) {
				cmd = g_strdup_printf (
					"gnome-terminal %s --hide-menubar -t \"%s@%s - SSH\" -x ssh -p %d -X %s@%s",
					geometry, user, host, (gint) port, user, host);
			} else {
				cmd = g_strdup_printf (
					"gnome-terminal %s --hide-menubar -t \"%s@%s - SSH\" -x ssh -X %s@%s",
					geometry, user, host, user, host);
			}
		} else {
			if (port > 0.0) {
				cmd = g_strdup_printf (
					"gnome-terminal %s --hide-menubar -t \"%s@%s - Telnet\" -x telnet %s %d",
					geometry, user, host, host, (gint) port);
			} else {
				cmd = g_strdup_printf (
					"gnome-terminal %s --hide-menubar -t \"%s@%s - Telnet\" -x telnet %s",
					geometry, user, host, host);
			}
		}

		if (!g_spawn_command_line_async (cmd, &err)) {
			gnome_error_dialog_parented (err->message, GTK_WINDOW (dialog));
			g_error_free (err);
		}

		g_free (cmd);
		g_free (geometry);

		gconf_client_set_int (conf_client, "/apps/gnome-remote-shell/TerminalWidth",
				      (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (window_width)),
				      NULL);
		gconf_client_set_int (conf_client, "/apps/gnome-remote-shell/TerminalHeight",
				      (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (window_height)),
				      NULL);
	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
radio_button_toggled_cb (GtkToggleButton *button, gpointer user_data)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ssh_method))) {
		gtk_widget_set_sensitive (user_entry, TRUE);
	} else {
		gtk_widget_set_sensitive (user_entry, FALSE);
	}
}

static void
set_spin_from_config (GtkWidget *spin, const gchar *key, gint default_value)
{
	GConfValue *conf_value;

	conf_value = gconf_client_get (conf_client, key, NULL);
	if (conf_value) {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin),
					   (gdouble) gconf_value_get_int (conf_value));
		gconf_value_free (conf_value);
	} else {
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), (gdouble) default_value);
	}
}

static void
create_dialog_contents (void)
{
	GtkWidget *table, *box, *frame, *label, *menu, *pixmap;
	gchar *pixmap_file;

	table = gtk_table_new (5, 2, FALSE);

	box = gtk_hbox_new (FALSE, 0);
	gtk_table_attach (GTK_TABLE (table), box, 0, 2, 0, 1, GTK_FILL, GTK_FILL, 2, 2);

	pixmap_file = gnome_pixmap_file ("gnome-session.png");
	pixmap = gtk_image_new_from_file (pixmap_file);
	g_free (pixmap_file);
	gtk_box_pack_start (GTK_BOX (box), pixmap, FALSE, FALSE, 5);

	frame = gtk_frame_new (_("Method"));
	gtk_box_pack_start (GTK_BOX (box), frame, TRUE, TRUE, 0);
	box = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);
	ssh_method = gtk_radio_button_new_with_label (NULL, "SSH");
	g_signal_connect (G_OBJECT (ssh_method), "toggled",
			  G_CALLBACK (radio_button_toggled_cb), NULL);
	gtk_box_pack_start (GTK_BOX (box), ssh_method, TRUE, TRUE, 0);
	telnet_method = gtk_radio_button_new_with_label_from_widget (GTK_RADIO_BUTTON (ssh_method),
								     "Telnet");
	g_signal_connect (G_OBJECT (telnet_method), "toggled",
			  G_CALLBACK (radio_button_toggled_cb), NULL);
	gtk_box_pack_start (GTK_BOX (box), telnet_method, TRUE, TRUE, 0);

	label = gtk_label_new (_("Host"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	host_entry = gnome_entry_new ("GNOME_Network_RemoteShell_Host");
	gtk_table_attach (GTK_TABLE (table), host_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
	
	label = gtk_label_new (_("User"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 2, 2);
	user_entry = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), user_entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 2, 2);

	label = gtk_label_new (_("Port"));
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 3, 4, GTK_FILL, GTK_FILL, 2, 2);
	port_entry = gtk_spin_button_new_with_range (0.0, 1000000.0, 1.0);
	gtk_table_attach (GTK_TABLE (table), port_entry, 1, 2, 3, 4, GTK_FILL, GTK_FILL, 2, 2);

	frame = gtk_frame_new (_("Terminal Window"));
	gtk_table_attach (GTK_TABLE (table), frame, 0, 2, 4, 5, GTK_FILL, GTK_FILL, 2, 2);
	box = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (frame), box);

	label = gtk_label_new (_("Width"));
	gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 0);
	window_width = gtk_spin_button_new_with_range (0.0, 1024.0, 1.0);
	set_spin_from_config (window_width, "/apps/gnome-remote-shell/TerminalWidth", 80);
	gtk_box_pack_start (GTK_BOX (box), window_width, TRUE, TRUE, 0);

	label = gtk_label_new (_("Height"));
	gtk_box_pack_start (GTK_BOX (box), label, TRUE, TRUE, 0);
	window_height = gtk_spin_button_new_with_range (0.0, 768.0, 1.0);
	set_spin_from_config (window_height, "/apps/gnome-remote-shell/TerminalHeight", 25);
	gtk_box_pack_start (GTK_BOX (box), window_height, TRUE, TRUE, 0);
	
	gtk_widget_show_all (table);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), table, TRUE, TRUE, 0);
}

int
main (int argc, char **argv)
{
#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gnome_program_init ("gnome-remote-shell", VERSION,
			    LIBGNOMEUI_MODULE, argc, argv, NULL);

	conf_client = gconf_client_get_default ();

	/* create the main dialog */
	dialog = gtk_dialog_new_with_buttons (_("GNOME Remote Shell"), NULL, 0,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OK, GTK_RESPONSE_OK,
					      NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	g_signal_connect (G_OBJECT (dialog), "destroy",
			  G_CALLBACK (dialog_destroyed_cb), NULL);
	g_signal_connect (G_OBJECT (dialog), "response",
			  G_CALLBACK (dialog_response_cb), NULL);

	create_dialog_contents ();
	gtk_widget_show (dialog);

	gtk_main ();

	g_object_unref (G_OBJECT (conf_client));

	return 0;
}
