/* gnome-netinfo - A GUI Interface for network utilities
 * Copyright (C) 2002, 2003 by German Poo-Caaman~o
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome.h>
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef HAVE_SYS_SOCKIO_H 
#  include <sys/sockio.h>
#endif


#include <sys/socket.h>	/* basic socket definitions */
#include <arpa/inet.h>	/* inet(3) functions */
#include <sys/un.h>	/* for Unix domain sockets */
#include <sys/ioctl.h>
#include <stdlib.h>
#include <net/if.h>

#include "info.h"
#include "utils.h"

#ifdef IFCONFIG_PROGRAM

#define IFCONFIG IFCONFIG_PROGRAM " -s"

GList * info_get_interfaces ();

void
info_do (const gchar * nic, netinfo_info * info)
{

}

void
info_load_iface (netinfo_info * info, GtkWidget * combo)
{
	GList *items = NULL;
	FILE *pfp;
	GIOChannel *io = NULL;
	gchar *line, *iface, rest[1000];
	gboolean title = TRUE;

	items = info_get_interfaces ();
	if (items != NULL) {
		gtk_combo_set_popdown_strings (GTK_COMBO (combo), items);
	}
	
	g_list_free (items);
}

void
info_nic_changed (GtkEditable *editable, gpointer data)
{
	const gchar *text;
	netinfo_info *info = data;
	gchar mtu[10], met[10], rx[10], rx_error[10], rx_drop[10], rx_ovr[10];
	gchar tx[10], tx_error[10], tx_drop[10], tx_ovr[10];
	gchar iface[30], flags[30];
	FILE *pfp;
	GIOChannel *io = NULL;
	gchar *line;
	gboolean title = TRUE;
	
	g_return_if_fail (info != NULL);
	
	text = gtk_entry_get_text (GTK_ENTRY (editable));

#if defined(__linux__)
	pfp = popen (IFCONFIG, "r");
	io = g_io_channel_unix_new (fileno (pfp));
	
	while (g_io_channel_read_line (io, &line, NULL, NULL, NULL) == G_IO_STATUS_NORMAL) {
		if (title) {
			title = FALSE;
			g_free (line);
			continue;
		}
		sscanf (line, "%s %s %s %s %s %s %s %s %s %s %s %s", iface,
			mtu, met, rx, rx_error, rx_drop, rx_ovr, tx,
			tx_error, tx_drop, tx_ovr, flags);
#endif
		/* Fill the NIC configuration data */
		info_get_nic_information (text, info);

#if defined(__linux__)
		if (g_ascii_strcasecmp (iface, text) == 0) {
			gtk_label_set_text (GTK_LABEL (info->tx), tx);
			gtk_label_set_text (GTK_LABEL (info->tx_errors), tx_error);
			gtk_label_set_text (GTK_LABEL (info->rx), rx);
			gtk_label_set_text (GTK_LABEL (info->rx_errors), rx_error);
		}
		g_free (line);
		/*info_free_nic_info (ninfo);*/
	}
	g_io_channel_unref (io);
	pclose (pfp);
#endif
}

void
info_get_nic_information (const gchar *nic, netinfo_info *info) {
	gint sockfd, len;
	gchar *ptr, buf[2048], dst[INFO_ADDRSTRLEN];
	struct ifconf ifc;
	struct ifreq *ifr, ifrcopy;
	struct sockaddr_in *sinptr;
	gint flags;
	gboolean loopback;

	sockfd = socket (AF_INET, SOCK_DGRAM, 0);

	ifc.ifc_len = sizeof (buf);
	ifc.ifc_req = (struct ifreq *) buf;
	ioctl (sockfd, SIOCGIFCONF, &ifc);

	for (ptr = buf; ptr < buf + ifc.ifc_len;) {
		ifr = (struct ifreq *) ptr;
		len = sizeof (struct sockaddr);
#ifdef	HAVE_SOCKADDR_SA_LEN
		if (ifr->ifr_addr.sa_len > len)
			len = ifr->ifr_addr.sa_len;	/* length > 16 */
#endif
		ptr += sizeof (ifr->ifr_name) + len;	/* for next one in buffer */

		if (strcmp (ifr->ifr_name, nic) != 0) {
			continue;
		}
		
		switch (ifr->ifr_addr.sa_family) {
		case AF_INET:
			
			/* Get the IPv4 address */
			sinptr = (struct sockaddr_in *) &ifr->ifr_addr;
			inet_ntop (AF_INET, &sinptr->sin_addr, dst, INFO_ADDRSTRLEN);
			
			gtk_label_set_text (GTK_LABEL (info->ip_address), dst);
			bzero (dst, INFO_ADDRSTRLEN);
			
			ifrcopy = *ifr;
			flags = ifrcopy.ifr_flags;

			/* Get the Hardware Address */
#ifdef SIOCGIFHWADDR
			ioctl (sockfd, SIOCGIFHWADDR, &ifrcopy);
			sinptr =
				(struct sockaddr_in *) &ifrcopy.ifr_dstaddr;
			g_sprintf (dst, "%02x:%02x:%02x:%02x:%02x:%02x",
				 (int) ((guchar *) &ifrcopy.ifr_hwaddr.sa_data)[0],
				 (int) ((guchar*)  &ifrcopy.ifr_hwaddr.sa_data)[1],
				 (int) ((guchar *) &ifrcopy.ifr_hwaddr.sa_data)[2],
				 (int) ((guchar *) &ifrcopy.ifr_hwaddr.sa_data)[3],
				 (int) ((guchar *) &ifrcopy.ifr_hwaddr.sa_data)[4],
				 (int) ((guchar *) &ifrcopy.ifr_hwaddr.sa_data)[5]);
#else
			g_sprintf (dst, NOT_AVAILABLE);
#endif /* SIOCGIFHWADDR */
				
			gtk_label_set_text (GTK_LABEL (info->hw_address), dst);

			/* Get the netMask address */
#ifdef SIOCGIFNETMASK
			ioctl (sockfd, SIOCGIFNETMASK, &ifrcopy);
				
			sinptr = (struct sockaddr_in *) &ifrcopy.ifr_addr;	

/*			sinptr =  (struct sockaddr_in *) &ifrcopy.ifr_netmask;*/
			inet_ntop (AF_INET, &sinptr->sin_addr, dst, INFO_ADDRSTRLEN);
#else
			g_sprintf (dst, NOT_AVAILABLE);
#endif /* SIOCGIFNETMASK */
			gtk_label_set_text (GTK_LABEL (info->netmask), dst);
			bzero (dst, INFO_ADDRSTRLEN);
			
			/* Get the broadcast address */
			ioctl (sockfd, SIOCGIFBRDADDR, &ifrcopy);
			sinptr = (struct sockaddr_in *) &ifrcopy.ifr_broadaddr;
			inet_ntop (AF_INET, &sinptr->sin_addr, dst, INFO_ADDRSTRLEN);
			gtk_label_set_text (GTK_LABEL (info->broadcast), dst);
			bzero (dst, INFO_ADDRSTRLEN);
			
			/* Get the MTU */
			ioctl (sockfd, SIOCGIFMTU, &ifrcopy);
			g_sprintf (dst, "%d", ifrcopy.ifr_mtu);
			gtk_label_set_text (GTK_LABEL (info->mtu), dst);
			bzero (dst, INFO_ADDRSTRLEN);

			/* Get Flags to determine other properties */
			ioctl (sockfd, SIOCGIFFLAGS, &ifrcopy);
			flags = ifrcopy.ifr_flags;

			/* Interface is up */
			if (flags & IFF_UP) {
				gtk_label_set_text (GTK_LABEL (info->state), _("Active"));
			} else {
				gtk_label_set_text (GTK_LABEL (info->state), _("Inactive"));
			}

			/* Is a loopback device */
			if ((flags & IFF_LOOPBACK)) {
				gtk_label_set_text (GTK_LABEL (info->hw_address), _("Loopback"));
				gtk_label_set_text (GTK_LABEL (info->broadcast), " ");
				gtk_label_set_text (GTK_LABEL (info->link_speed), " ");
			} else {
				gtk_label_set_text (GTK_LABEL (info->link_speed), NOT_AVAILABLE);
			}
			
			/* Supports multicast */
			if (flags & IFF_MULTICAST) {
				gtk_label_set_text (GTK_LABEL (info->multicast), _("Enabled"));
			} else {
				gtk_label_set_text (GTK_LABEL (info->multicast), _("Disabled"));
			}

			/* Interface is a point to point link */
			if (flags & IFF_POINTOPOINT) {
				ioctl (sockfd, SIOCGIFDSTADDR, &ifrcopy);
				sinptr = (struct sockaddr_in *) &ifrcopy.ifr_dstaddr;
				
				printf ("\tP-t-P: %s\n",
					inet_ntop (AF_INET,
						   &sinptr->sin_addr, dst,
						   INFO_ADDRSTRLEN));
			}
			bzero (dst, INFO_ADDRSTRLEN);

			break;

		default:
			gtk_label_set_text (GTK_LABEL (info->hw_address), NOT_AVAILABLE);
			gtk_label_set_text (GTK_LABEL (info->ip_address), NOT_AVAILABLE);
			gtk_label_set_text (GTK_LABEL (info->broadcast), NOT_AVAILABLE);
			gtk_label_set_text (GTK_LABEL (info->netmask), NOT_AVAILABLE);
			/*gtk_label_set_text (GTK_LABEL (info->dst_address), NOT_AVAILABLE);*/
			gtk_label_set_text (GTK_LABEL (info->mtu), NOT_AVAILABLE);
			gtk_label_set_text (GTK_LABEL (info->state), NOT_AVAILABLE);
			gtk_label_set_text (GTK_LABEL (info->multicast), NOT_AVAILABLE);
			gtk_label_set_text (GTK_LABEL (info->link_speed), NOT_AVAILABLE);
			break;
		}
	}
	g_free (ifr);
}

GList *
info_get_interfaces ()
{
	GList *items = NULL;
	gchar *iface;
	gchar *ptr, buf[2048];
	struct sockaddr_in *sinptr;
	struct ifconf ifc;
	struct ifreq *ifr;
	int sockfd, len;

	sockfd = socket (AF_INET, SOCK_DGRAM, 0);

	ifc.ifc_len = sizeof (buf);
	ifc.ifc_req = (struct ifreq *) buf;

	ioctl (sockfd, SIOCGIFCONF, &ifc);

	for (ptr = buf; ptr < buf + ifc.ifc_len;) {
		ifr = (struct ifreq *) ptr;
		len = sizeof (struct sockaddr);

		iface = g_strdup (ifr->ifr_name);
		items = g_list_append (items, iface);

#ifdef	HAVE_SOCKADDR_SA_LEN
		if (ifr->ifr_addr.sa_len > len)
			len = ifr->ifr_addr.sa_len;	/* length > 16 */
#endif
		ptr += sizeof (ifr->ifr_name) + len;	/* for next one in buffer */
	}

	return items;
}

#endif
