/* main.c - GNOME remote desktop client/server

   Copyright (C) 1998 Tom Tromey

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */

#include <config.h>

#include <gtk/gtkdialog.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkimage.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-ui-init.h>
#include <glade/glade-xml.h>
#ifdef INCLUDE_RFB_SERVER
#include "rfbserver.h"
#endif

static GladeXML *xmlui;
static GtkWidget *x_method;
static GtkWidget *vnc_method;
static GtkWidget *host_entry;
static GtkWidget *display_entry;
static GtkWidget *desktop_width;
static GtkWidget *desktop_height;

static void
dialog_destroyed_cb (GtkWidget *widget, gpointer user_data)
{
	gtk_main_quit ();
}

static void
dialog_response_cb (GtkDialog *dialog, gint response_id, gpointer user_data)
{
	if (response_id == GTK_RESPONSE_OK) {
		gchar *cmd;
		const gchar *host;
		const gchar *display;
		GError *err = NULL;

		host = gtk_entry_get_text (GTK_ENTRY (gnome_entry_gtk_entry (GNOME_ENTRY (host_entry))));
		display = gtk_entry_get_text (GTK_ENTRY (display_entry));
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (x_method))) {
#ifdef XNEST_PROGRAM
			/* FIXME: must guess the nest display name */
			cmd = g_strdup_printf ("%s -query %s %s", XNEST_PROGRAM, host,
					       display && *display ? display : ":3000");
			if (!g_spawn_command_line_async (cmd, &err)) {
				gnome_error_dialog_parented (err->message, GTK_WINDOW (dialog));
				g_error_free (err);
			}

			g_free (cmd);
#else
			gnome_error_dialog_parented (_("You don't have the Xnest program installed in the system"),
						     GTK_WINDOW (dialog));
			return;
#endif
		} else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (vnc_method))) {
#ifdef VNCVIEWER_PROGRAM
			if (display && *display) {
				if (host && *host) {
					cmd = g_strdup_printf ("%s %s%s", VNCVIEWER_PROGRAM, host, display);
				} else {
					cmd = g_strdup_printf ("%s %s", VNCVIEWER_PROGRAM, display);
				}
			} else if (host && *host) {
				cmd = g_strdup_printf ("%s %s", VNCVIEWER_PROGRAM, host);
			} else {
				gnome_error_dialog_parented (_("You have to specify at least a host or display name"),
							     dialog);
				return;
			}

			if (!g_spawn_command_line_async (cmd, &err)) {
				gnome_error_dialog_parented (err->message, GTK_WINDOW (dialog));
				g_error_free (err);
			}

			g_free (cmd);
#else
			gnome_error_dialog_parented (_("You don't have a VNC viewer installed in the system"),
						     GTK_WINDOW (dialog));
			return;
#endif
		}

	}

	gtk_widget_destroy (GTK_WIDGET (dialog));
}

static GtkWidget *
load_dialog (void)
{
	GtkWidget *dialog, *image;
	gchar *pixmap_file;

	xmlui = glade_xml_new (GLADEDIR "/gnome-remote-desktop.glade", "remote-desktop-dialog", NULL);
	if (!xmlui)
		return gnome_error_dialog (_("Could not load UI for remote desktop client"));

	dialog = glade_xml_get_widget (xmlui, "remote-desktop-dialog");
	x_method = glade_xml_get_widget (xmlui, "x-method-radio");
	vnc_method = glade_xml_get_widget (xmlui, "vnc-method-radio");
	host_entry = glade_xml_get_widget (xmlui, "host-entry");
	display_entry = glade_xml_get_widget (xmlui, "display-entry");
	desktop_width = glade_xml_get_widget (xmlui, "desktop-width");
	desktop_height = glade_xml_get_widget (xmlui, "desktop-height");

	pixmap_file = gnome_pixmap_file ("gnome-session.png");
	image = glade_xml_get_widget (xmlui, "logo-method");
	gtk_image_set_from_file (GTK_IMAGE (image), pixmap_file);
	image = glade_xml_get_widget (xmlui, "about-image");
	gtk_image_set_from_file (GTK_IMAGE (image), pixmap_file);
	g_free (pixmap_file);

	pixmap_file = gnome_pixmap_file ("gnome-terminal.png");
	image = glade_xml_get_widget (xmlui, "logo-connection");
	gtk_image_set_from_file (GTK_IMAGE (image), pixmap_file);
	g_free (pixmap_file);

	pixmap_file = gnome_pixmap_file ("gnome-ccdesktop.png");
	image = glade_xml_get_widget (xmlui, "logo-desktop");
	gtk_image_set_from_file (GTK_IMAGE (image), pixmap_file);
	g_free (pixmap_file);

	return dialog;
}

int
main (int argc, char *argv[])
{
	GtkWidget *dialog;

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, GNOMELOCALEDIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	gnome_program_init ("gnome-remote-desktop", VERSION,
			    LIBGNOMEUI_MODULE, argc, argv, NULL);

	/* create the main dialog */
	dialog = load_dialog ();
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
	g_signal_connect (G_OBJECT (dialog), "destroy",
			  G_CALLBACK (dialog_destroyed_cb), NULL);
	g_signal_connect (G_OBJECT (dialog), "response",
			  G_CALLBACK (dialog_response_cb), NULL);
	gtk_widget_show (dialog);

#ifdef INCLUDE_RFB_SERVER
	rfbserver_init ();
	rfbserver_run (argc, argv);
#endif

	gtk_main ();

#ifdef INCLUDE_RFB_SERVER
	rfbserver_shutdown ();
#endif

	g_object_unref (G_OBJECT (xmlui));
	
	return 0;
}
