/* gnome-ppp - The GNOME PPP Dialer
 * Copyright (C) 1997 Jay Painter
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 *(at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <gnome.h>
#include "gnome-ppp.h"
#include "configure-window.h"
#include "dial-window.h"
#include "global.h"
#include "misc.h"


enum
{
  _DISCONNECTED,
  _IN_PROGRESS,
  _CONNECTED
};


/* prototype external function for quit callback,
 * this function is in main.c */
extern void exit_gppp();


/* static, single-instance dial window struct */
typedef struct _DialWindow 
{
  GtkWidget *window;
  GtkWidget *connect_menu;
  GtkWidget *status;
  GtkWidget *connect_button_label;
  GtkWidget *connect;
  GtkWidget *configure;
  GtkWidget *quit;

  Account *account;
  gint state;
  gint timer_id;
} DialWindow;
static DialWindow *dw = NULL;


/* close callback */
static void close_dial_window();
static void account_cb(GtkWidget * widget, gpointer data);
static void connect_button_cb(GtkWidget *widget);
static void configure_button_cb(GtkWidget *widget);
static gint timeout_cb();
static void connect_message_cb(Account *account, 
			       PPPMessageType message, 
			       gchar *text);


void
open_dial_window()
{
  GtkWidget *vbox;
  GtkWidget *hbox;
  GtkWidget *label;
  GtkWidget *separator;

  if (dw)
    {
      return;
    }

  dw = g_malloc(sizeof(DialWindow));
  dw->account = NULL;
  dw->state = _DISCONNECTED;
  dw->timer_id = -1;

  /* set up the main window */
  dw->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(dw->window), GNOME_PPP_NAME);
  gtk_window_set_policy(GTK_WINDOW(dw->window), FALSE, TRUE, TRUE);
  gtk_container_border_width(GTK_CONTAINER(dw->window), 10);
  gtk_signal_connect(
      GTK_OBJECT(dw->window),
      "destroy",
     (GtkSignalFunc) close_dial_window,
      NULL);
  gtk_signal_connect(
      GTK_OBJECT(dw->window),
      "delete_event",
     (GtkSignalFunc) gtk_false,
      NULL);

  /* boxes for widget layout */
  vbox = gtk_vbox_new(FALSE, 10);
  gtk_container_add(GTK_CONTAINER(dw->window), vbox);
  gtk_widget_show(vbox);

  /* first row */
  hbox = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show(hbox);

  /* label for the connection-selection box */
  label = gtk_label_new("Connect To:");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, TRUE, 0);
  gtk_widget_show(label);

  /* option menu to select dial-out */
  dw->connect_menu = gtk_option_menu_new();
  gtk_widget_set_usize(dw->connect_menu, 150, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), dw->connect_menu, TRUE, TRUE, 0);
  gtk_widget_show(dw->connect_menu);

  /* second row */
  hbox = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show(hbox);

  /* status label */
  dw->status = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(hbox), dw->status, TRUE, TRUE,0);
  gtk_widget_show(dw->status);

  /* third row */
  hbox = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show(hbox);

  /* horizontal seperator line */
  separator = gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(hbox), separator, TRUE, TRUE, 0);
  gtk_widget_show(separator);

  /* fourth row */
  hbox = gtk_hbox_new(FALSE, 10);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, FALSE, 0);
  gtk_widget_show(hbox);

  /* connect button */
  dw->connect_button_label = gtk_label_new("");
  gtk_widget_show(dw->connect_button_label);

  dw->connect = gtk_button_new();
  gtk_container_add(GTK_CONTAINER(dw->connect), dw->connect_button_label);
  gtk_widget_set_usize(dw->connect, 100, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), dw->connect, TRUE, FALSE, 10);
  gtk_signal_connect(
      GTK_OBJECT(dw->connect),
      "clicked",
     (GtkSignalFunc) connect_button_cb,
      NULL);
  gtk_widget_show(dw->connect);

  /* configure button */
  dw->configure = gtk_button_new_with_label("Configure");
  gtk_widget_set_usize(dw->configure, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), dw->configure, TRUE, FALSE, 5);
  gtk_signal_connect(
      GTK_OBJECT(dw->configure),
      "clicked",
     (GtkSignalFunc) configure_button_cb,
      NULL);
  gtk_widget_show(dw->configure);

  /* quit button */
  dw->quit = gtk_button_new_with_label("Quit");
  gtk_widget_set_usize(dw->quit, GPPP_BUTTON_WIDTH, GPPP_BUTTON_HEIGHT);
  gtk_box_pack_start(GTK_BOX(hbox), dw->quit, TRUE, FALSE, 5);
  gtk_signal_connect_object(
      GTK_OBJECT(dw->quit),
      "clicked",
      GTK_SIGNAL_FUNC(gtk_widget_destroy),
      GTK_OBJECT(dw->window));
  gtk_widget_show(dw->quit);

  /* set data in the dial window */
  refresh_dial_window();

  /* Show it */
  gtk_widget_show(dw->window);
}


void
refresh_dial_window()
{
  GList *list;
  GtkWidget *menu;
  Account *account;

  menu = gtk_menu_new();

  list = account_list();
  while (list)
    {
      account = (Account *) list->data;
      list = list->next;

      append_menuitem_connect(
          GTK_MENU(menu), 
	  account->name->str, 
	 (GtkSignalFunc) account_cb, 
	 (gpointer) account,
	  NULL);
    }
  gtk_option_menu_remove_menu(GTK_OPTION_MENU(dw->connect_menu));
  gtk_option_menu_set_menu(GTK_OPTION_MENU(dw->connect_menu), menu);

  /* re-select the default account, or select the
   * first account as default if there was no default
   */
  if (account_list())
    {
      if (!dw->account || !g_list_find(account_list(), dw->account))
	{
	  dw->account = (Account *) account_list()->data;
	}

      gtk_option_menu_set_history(
          GTK_OPTION_MENU(dw->connect_menu),
	  g_list_index(account_list(), dw->account));
    }
  else
    {
      dw->account = NULL;
    }

  /* set the active state of the connection button and the
   * account-selection option menu based on the connection
   * state
   */
  if (dw->account)
    {
      gtk_widget_set_sensitive(GTK_WIDGET(dw->connect), TRUE);

      switch (dw->state)
	{
	case _DISCONNECTED:
	  gtk_label_set_text(GTK_LABEL(dw->connect_button_label), "Connect");
	  gtk_widget_set_sensitive(GTK_WIDGET(dw->connect_menu), TRUE);
	  break;
	  
	case _IN_PROGRESS:
	  gtk_label_set_text(GTK_LABEL(dw->connect_button_label), "Cancel");
	  gtk_widget_set_sensitive(GTK_WIDGET(dw->connect_menu), FALSE);
	  break;
	  
	case _CONNECTED:
	  gtk_label_set_text(GTK_LABEL(dw->connect_button_label), "Disconnect");
	  gtk_widget_set_sensitive(GTK_WIDGET(dw->connect_menu), FALSE);
	  break;
	}
    }
  else
    {
      gtk_widget_set_sensitive(GTK_WIDGET(dw->connect), FALSE);
      gtk_widget_set_sensitive(GTK_WIDGET(dw->connect_menu), FALSE);
    }
}


static void
close_dial_window()
{
  if(dw)
    {
      g_free(dw);
      dw = NULL;
    }

  exit_gppp();
}


/* callback for option menu */
static void
account_cb(GtkWidget * widget, gpointer data)
{
  dw->account = (Account *) data;
}


static void
connect_button_cb(GtkWidget *widget)
{
  /* we need an account selected to connect */
  if (!g_list_find(account_list(), dw->account))
    {
      g_error("internal error !g_list_find(account_list(), dw->account\n");
    }

  switch (dw->state)
    {
    case _DISCONNECTED:
      if (connect_start(dw->account, connect_message_cb))
	{
	  dw->timer_id = 
	    gtk_timeout_add(500, (GtkFunction) timeout_cb, NULL);
	}
    break;

    case _IN_PROGRESS:
    case _CONNECTED:
      connect_stop(dw->account);
    break;
    }
}

static gint
timeout_cb()
{
  if (connect_engine_iteration(dw->account))
    {
      return 1;
    }
  else
    {
      /* blank out the timeout id and return 0 shutting
       * down the gtk_timeout
       */
      dw->timer_id = -1;
      return 0;
    }
}


static void
configure_button_cb(GtkWidget *widget)
{
  open_configure_window();
}


/*
 * message callback from connect
 */
static void
connect_message_cb(Account *account, PPPMessageType message, gchar *text)
{
  gint saved_state;

  g_assert(account != NULL);
  g_assert(dw->account == account);

  saved_state = dw->state;

  /* map messages from the connect engine to internal
   * states of the dial window
   */
  switch (message)
    {
    case PPP_DISCONNECTED:
      dw->state = _DISCONNECTED;
      break;

    case PPP_IN_PROGRESS:
      dw->state = _IN_PROGRESS;
      break;

    case PPP_CONNECTED:
      dw->state = _CONNECTED;
      break;
    }

  /* refresh the dial window if the state changed */
  if (dw->state != saved_state)
    {
      refresh_dial_window();
    }

  /* display a message if there is one */
  if (text)
    {
      gtk_label_set_text(GTK_LABEL(dw->status), text);
    }
  else
    { 
      /* blank out the message area if we disconnected normally */
      if (saved_state == _CONNECTED && dw->state == _DISCONNECTED)
	{
	  gtk_label_set_text(GTK_LABEL(dw->status), "");
	}
    }
}

