# Infrared Remote Control Properties for GNOME
# Copyright (C) 2008 Fluendo Embedded S.L. (www.fluendo.com)
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
'''
Hardware Abstraction Layer (hal) related classes.
'''

import dbus
import gobject
import gtk.gdk
import logging
import os, os.path

from ConfigParser import SafeConfigParser
from gnome_lirc_properties import lirc

HAL_SERVICE = 'org.freedesktop.Hal'
HAL_MANAGER_PATH = '/org/freedesktop/Hal/Manager'
HAL_MANAGER_IFACE = 'org.freedesktop.Hal.Manager'

class HalDevice(object):
    '''A device as announced by HAL.'''

    def __init__(self, bus, hal, udi):
        proxy_object = bus.get_object ('org.freedesktop.Hal', udi)
        self.__obj = dbus.Interface (proxy_object, 'org.freedesktop.Hal.Device')
        self.__hal = hal
        self.__bus = bus
        self.__udi = udi

    def __getitem__(self, key):
        try:
            return self.__obj.GetProperty(key)

        except dbus.exceptions.DBusException, ex:
            if ('org.freedesktop.Hal.NoSuchProperty' == ex.get_dbus_name()):
                raise KeyError, key

            raise

    def get(self, key, default = None):
        '''
        Returns the value of the property described by key,
        or default if the property doesn't exist.
        '''
        try:
            return self.__obj.GetProperty(key)

        except dbus.exceptions.DBusException, ex:
            if ('org.freedesktop.Hal.NoSuchProperty' == ex.get_dbus_name()):
                return default

            raise

    def lookup_parent(self):
        '''Find the parent device of this device.'''
        udi = self['info.parent']
        return HalDevice(self.__bus, udi)

    def find_children(self):
        '''Lists all child devices of this device.'''
        children = self.__hal.FindDeviceStringMatch('info.parent', self.__udi)
        return [HalDevice(self.__bus, self.__hal, udi) for udi in children]

    def check_kernel_module(self, kernel_module):
        '''Checks if the driver uses the specified kernel module.'''
        for device in self.find_children():
            if device.get('info.linux.driver') == kernel_module:
                return True

        return False

    def find_device_by_class(self, device_class):
        '''Find the LIRC device node associated with this device.'''
        class_root = os.path.join(os.sep, 'sys', 'class', device_class)
        sysfs_path = self['linux.sysfs_path']

        for device in os.listdir(class_root):
            try:
                # resolve the device link found in the sysfs folder:
                root = os.path.join(class_root, device)
                link = os.path.join(root, 'device')
                link = os.path.join(root, os.readlink(link))

                # compare the resolved link and its parent with the device path
                # stored in the 'linux.sysfs_path' property:
                while link:
                    if not os.path.samefile(sysfs_path, link):
                        link = os.path.dirname(link)
                        continue

                    for filename in (
                        os.path.join(os.sep, 'dev', device_class, device),
                        os.path.join(os.sep, 'dev', device),
                    ):
                        if os.path.exists(filename):
                            return filename

                    break

            except (IOError, OSError), ex:
                logging.warning(ex)

        return None

    def find_input_device(self):
        '''Find the Linux Input System device node associated with this device.'''
        for device in self.find_children():
            if device.get('info.category') == 'input':
                return device['input.device']

        return None

    def find_device_node(self, kernel_module, lirc_driver):
        '''Finds the device node associated with this device.'''
        if 'usbhid' == kernel_module:
            return self.find_device_by_class('usb')
        if 'default' == lirc_driver:
            return self.find_device_by_class('lirc')
        if lirc_driver in ('devinput', 'dev/input'):
            return self.find_input_device()

        return None

    def __str__(self):
        return self.__udi
    def __repr__(self):
        return '<HalDevice: %s>' % self.__udi

    # pylint: disable-msg=W0212
    udi = property(lambda self: self.__udi)

class HardwareDatabase(SafeConfigParser):
    '''Information about supported hardware.'''
    def __init__(self, filename):
        SafeConfigParser.__init__(self)
        self.read(filename)

class HardwareManager(gobject.GObject):
    '''This object provides hardware detection.'''

    __gsignals__ = {
        'search-progress': (
            gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
            (gobject.TYPE_FLOAT, gobject.TYPE_STRING)),
        'search-finished': (
            gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
            tuple()),
        'receiver-found': (
            gobject.SIGNAL_RUN_LAST, gobject.TYPE_NONE,
            (gobject.TYPE_PYOBJECT, gobject.TYPE_STRING, gobject.TYPE_STRING)),
    }

    def __init__(self, hardware_db):
        # pylint: disable-msg=E1002

        assert(isinstance(hardware_db, HardwareDatabase))

        super(HardwareManager, self).__init__()
        self.__search_canceled = False

        # reading hardware database, and reording as
        # dictionary with (product-id, vendor-id) keys
        self.__receivers = dict()

        for section in hardware_db.sections():
            vendor, product = [s.strip() for s in section.split(':', 1)]
            properties = dict(hardware_db.items(section))

            receiver = lirc.Receiver(vendor, product, **properties)
            key = (receiver.vendor_id, receiver.product_id)
            self.__receivers[key] = receiver

    def resolve_device_nodes(self, nodes):
        '''Resolve the requested device nodes.'''
        device_nodes, hal, bus = list(), None, None

        for name in nodes:
            # Use HAL manager to find device nodes,
            # when name starts with "hal-capability:"
            if name.startswith('hal-capability:'):
                capability = name.split(':')[1]

                if not hal:
                    hal, bus = self._create_hal_manager()

                device_nodes += [
                    str(HalDevice(bus, hal, udi)['serial.device'])
                    for udi in hal.FindDeviceByCapability(capability)]

            elif name.startswith('numeric:'):
                device_nodes.append(name)

            # Otherwise check if name is an existing device node.
            elif os.path.isabs(name) and os.path.exists(name):
                device_nodes.append(name)

        # Ensure that each entry exists only once, and sort them.
        device_nodes = list(set(device_nodes))
        device_nodes.sort()

        return device_nodes

    @staticmethod
    def parse_numeric_device_node(device_node):
        '''
        Parses a numeric device node as used for describing for instance UDP
        receivers. Returns the tuple label, default-value, minimum-value,
        maximum-value.
        '''

        parts  = device_node.split(':', 4)
        limits = map(int, parts[1:4])
        label  = parts[4]

        return [label] + limits

    @staticmethod
    def _create_hal_manager():
        '''Retreive HAL manager instance from DBus'''

        bus = dbus.SystemBus()
        hal = bus.get_object(HAL_SERVICE, HAL_MANAGER_PATH)
        hal = dbus.Interface(hal, HAL_MANAGER_IFACE)

        return hal, bus

    def search_receivers(self):
        '''
        Search for supported IR receivers. This search actually happens
        in a separate thread and can be aborted with the cancel() method.
        Connect to the signals of this object, to monitor search progress.

        TODO: Currently the search happens in the main thread, as DBus' Python
        bindings still cause random dead-locks when used with threads. Despite
        claiming being thread-safe.
        '''

        self.__search_canceled = False

        # retreive list of USB devices from HAL
        hal, bus = self._create_hal_manager()
        usb_devices = hal.FindDeviceStringMatch('info.subsystem', 'usb_device')

        for i in range(len(usb_devices)):
            # provide cancelation point:
            if self.__search_canceled:
                break

            # process GTK+ events:
            while gtk.events_pending():
                if gtk.main_iteration():
                    return

            # lookup the current device:
            device = HalDevice(bus, hal, usb_devices[i])

            # report search progress:
            self._search_progress(float(i)/float(len(usb_devices)),
                                  '%s %s', device['info.vendor'],
                                           device['info.product'])

            # lookup the USB devices in our hardware database:
            vendor_id = device['usb_device.vendor_id']
            product_id = device['usb_device.product_id']

            # Skip devices without vendor id. Linux for instance
            # doesn't assign a vendor id to USB host controllers.
            if not vendor_id:
                continue

            # The Streamzap really has a product ID of 0000,
            # so don't skip like this on empty product ids:
            #
            #   if not vendor_id or not product_id:
            #       continue
            #

            # lookup receiver description:
            receiver_key = (vendor_id, product_id)
            receiver = self.__receivers.get(receiver_key)

            # skip unknown hardware:
            if not receiver:
                continue

            # skip devices, where the associated kernel module
            # doesn't match the expected kernel module:
            if (receiver.kernel_module and
                not device.check_kernel_module(receiver.kernel_module)):
                continue

            # find the LIRC device node, and signal search result:
            device_node = device.find_device_node(receiver.kernel_module,
                                                  receiver.lirc_driver)

            self._receiver_found(receiver, device.udi, device_node)

        self._search_finished()

    def _search_progress(self, progress, message, *args):
        '''Report search progress.'''

        # pylint: disable-msg=E1101
        self.emit('search-progress', progress, message % args)

    def _receiver_found(self, receiver, udi, device_node):
        '''Signal that a receiver was found.'''

        # pylint: disable-msg=E1101
        self.emit('receiver-found', receiver, udi, device_node)

    def _search_finished(self):
        '''Signal that search has finished.'''

        # pylint: disable-msg=E1101
        self.emit('search-finished')

    def cancel(self):
        '''Abort current search process.'''

        self.__search_canceled = True
