/* 
   Copyright (C) 2005 James Bowes <bowes@cs.dal.ca>
   Copyright (C) 2005 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdlib.h>

#include <gtk/gtkmarshal.h>
#include <gtk/gtkmessagedialog.h>
#include <glade/glade.h>
#include <libgnome/gnome-help.h>
#include <libgnomeui/gnome-stock-icons.h>

#include "gnome-keyring-manager.h"
#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-util.h"
#include "gnome-keyring-manager-attribute-display.h"
#include "gnome-keyring-manager-create-keyring-dialog.h"
#include "gnome-keyring-manager-keyring-editor.h"
#include "gnome-keyring-manager-keyring-manager.h"

#include "gnome-keyring-manager-main-ui.h"

#define GLADE_ROOT "Gnome Keyring Manager"

struct _GKMMainUIPrivate
{
  GladeXML *xml;

  GtkWidget *window;
  GtkWidget *keyrings_treeview;
  GtkWidget *keys_treeview;
  GtkWidget *attribute_display;

  gboolean keyrings_visible;

  GObject *manager;
  GObject *editor;
};

static void gkm_main_ui_class_init (GKMMainUIClass *class);
static void gkm_main_ui_init       (GObject *object);
static void gkm_main_ui_finalize   (GObject *object);

static void gkm_main_ui_initialize_keyrings_tree_view (GKMMainUI *self);
static void gkm_main_ui_initialize_keys_tree_view     (GKMMainUI *self);

static void gkm_main_ui_populate_key_attributes       (GKMMainUI   *self,
						       GtkTreeIter  iter);
static void gkm_main_ui_populate_key_access_list      (GKMMainUI   *self,
						       GtkTreeIter  iter);

static void gkm_main_ui_update_title                  (GKMMainUI *self);

static void gkm_main_ui_connect_glade_signals         (GKMMainUI *self);

static GObjectClass *parent_class = NULL;
static gint destroy_signal;

GType
gkm_main_ui_get_type (void)
{
  static GType type = 0;

  if (!type)
    {
      static const GTypeInfo info =
      {
        sizeof (GKMMainUIClass),
	NULL,		/* base_init */
	NULL,		/* base_finalize */
        (GClassInitFunc) gkm_main_ui_class_init,
	NULL,		/* class_finalize */
	NULL,		/* class_data */ 
	sizeof (GKMMainUI),
	0,		/* n_preallocs */
	(GInstanceInitFunc) gkm_main_ui_init,
	0
      };

      type = g_type_register_static (G_TYPE_OBJECT, "GKMMainUI", &info, 0);
    }

  return type;
}

static void
gkm_main_ui_class_init (GKMMainUIClass *class)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (class);

  gobject_class->finalize = gkm_main_ui_finalize;

  parent_class = g_type_class_peek_parent (class);

  destroy_signal = 
    g_signal_new ("destroy",
		  G_TYPE_FROM_CLASS (gobject_class),
		  G_SIGNAL_RUN_CLEANUP | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  0, NULL, NULL,
		  gtk_marshal_VOID__VOID,
		  G_TYPE_NONE, 0);

}

static void
gkm_main_ui_init (GObject *object)
{
  GKMMainUI *self;
  GtkTreeModel *tree_model;
  gchar *default_keyring;

  g_return_if_fail (GKM_IS_MAIN_UI (object));

  self = GKM_MAIN_UI (object);

  self->priv = g_new0 (GKMMainUIPrivate, 1);

  self->priv->keyrings_visible = FALSE;

  self->priv->xml = glade_xml_new (GLADEDIR GLADEFILE,
				   GLADE_ROOT, NULL);

  if(!self->priv->xml) 
    {
      g_warning ("Unable to load the glade file. Your install is broken.");
      exit (1);
    }

  self->priv->window = glade_xml_get_widget (self->priv->xml, 
					     GLADE_ROOT);

  self->priv->keyrings_treeview = glade_xml_get_widget (self->priv->xml,
							"keyrings_treeview");
  self->priv->keys_treeview 	= glade_xml_get_widget (self->priv->xml,
							"keys_treeview");
  self->priv->attribute_display = glade_xml_get_widget (self->priv->xml,
							"attributes_display");

  gkm_main_ui_initialize_keyrings_tree_view (self);
  gkm_main_ui_initialize_keys_tree_view (self);

  self->priv->manager = gkm_keyring_manager_new ();
  tree_model = gkm_keyring_manager_get_model (GKM_KEYRING_MANAGER (self->priv->manager));
  gtk_tree_view_set_model (GTK_TREE_VIEW (self->priv->keyrings_treeview), 
			   tree_model);
  /* FIXME: choose the keyring to open by default elsewhere */
  /* or by default just open the first in the list, and allow for another to be 
     chosen through a function call */
  default_keyring = 
    gkm_keyring_manager_get_default_keyring (GKM_KEYRING_MANAGER (self->priv->manager));

  self->priv->editor = gkm_keyring_editor_new ("default");
  tree_model = gkm_keyring_editor_get_model (GKM_KEYRING_EDITOR (self->priv->editor));
  gtk_tree_view_set_model (GTK_TREE_VIEW (self->priv->keys_treeview), 
			   tree_model);

  g_free (default_keyring);

  gkm_main_ui_connect_glade_signals (self);
}

static void
gkm_main_ui_finalize (GObject *object)
{
  GKMMainUI *self;

  g_return_if_fail (GKM_IS_MAIN_UI (object));

  self = GKM_MAIN_UI (object);

  g_object_unref (self->priv->xml);

  g_free (self->priv);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}

GObject *
gkm_main_ui_new ()
{
  GKMMainUI *main_ui;

  main_ui = g_object_new (GKM_TYPE_MAIN_UI, NULL);

  gkm_main_ui_update_title (main_ui);

  return G_OBJECT (main_ui);
}

static void
format_timeout_cell_data_func (GtkTreeViewColumn *column G_GNUC_UNUSED,
                               GtkCellRenderer   *cell,
                               GtkTreeModel      *model,
                               GtkTreeIter       *iter,
                               gpointer           data)
{
  gint     column_id;
  guint32  timeout;
  gchar   *readable_timeout;

  column_id = GPOINTER_TO_INT (data);

  gtk_tree_model_get(model, iter, column_id, &timeout, -1); 
  readable_timeout = g_strdup_printf (_("%d seconds"), timeout);
  
  g_object_set(cell, "text", readable_timeout, NULL);

  g_free (readable_timeout);
}

static void
format_keyring_name_cell_data_func (GtkTreeViewColumn *column G_GNUC_UNUSED,
				    GtkCellRenderer   *cell,
				    GtkTreeModel      *model,
				    GtkTreeIter       *iter,
				    gpointer           data)
{
  gint      column_id;
  gchar    *keyring_name;
  gchar    *display_name;
  gboolean  is_default;

  column_id = GPOINTER_TO_INT (data);

  gtk_tree_model_get(model, iter, 
		     column_id, &keyring_name, 
		     MANAGER_COLUMN_DEFAULT, &is_default,
		     -1); 

  if (is_default)
    {
      display_name = g_strdup_printf (_("<span weight='bold'>%s</span>"), 
				      keyring_name);
    }
  else
    {
      display_name = g_strdup (keyring_name);
    }

  g_object_set(cell, "markup", display_name, NULL);

  g_free (keyring_name);
  g_free (display_name);
}

static void
gkm_main_ui_initialize_keyrings_tree_view (GKMMainUI *self)
{
  GtkTreeViewColumn *column;
  GtkCellRenderer *renderer;
  
  g_return_if_fail (GKM_IS_MAIN_UI (self));

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Locked"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keyrings_treeview), 
			       column);
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "stock-id", 
				      MANAGER_COLUMN_LOCK);
  gtk_tree_view_column_set_sort_column_id (column, MANAGER_COLUMN_LOCK);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Keyring"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keyrings_treeview), 
			       column);
  gtk_tree_view_column_set_expand (column, TRUE);
  renderer = gtk_cell_renderer_text_new ();
  g_object_set (G_OBJECT (renderer), "editable", TRUE, NULL);
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, MANAGER_COLUMN_KEYRING);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_keyring_name_cell_data_func,
				           GINT_TO_POINTER (MANAGER_COLUMN_KEYRING), 
					   NULL);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Lock on Idle"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keyrings_treeview), 
			       column);
  renderer = gtk_cell_renderer_toggle_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "active", 
				      MANAGER_COLUMN_LOCK_ON_IDLE);
  gtk_tree_view_column_set_sort_column_id (column, MANAGER_COLUMN_LOCK_ON_IDLE);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Lock Timeout"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keyrings_treeview), 
			       column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_timeout_cell_data_func,
				           GINT_TO_POINTER (MANAGER_COLUMN_LOCK_TIMEOUT), 
					   NULL);
  gtk_tree_view_column_set_sort_column_id (column, MANAGER_COLUMN_LOCK_TIMEOUT);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Modification Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keyrings_treeview), 
			       column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (MANAGER_COLUMN_MTIME), 
					   NULL);
  gtk_tree_view_column_set_sort_column_id (column, MANAGER_COLUMN_MTIME);
 
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Creation Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keyrings_treeview), 
			       column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (MANAGER_COLUMN_CTIME), 
					   NULL);
  gtk_tree_view_column_set_sort_column_id (column, MANAGER_COLUMN_CTIME);
}

static void 
key_type_get_pixbuf (GtkTreeViewColumn *column G_GNUC_UNUSED, 
		     GtkCellRenderer *renderer, 
		     GtkTreeModel *model,
		     GtkTreeIter *iter, 
		     gpointer *data G_GNUC_UNUSED)
{
  int type;

  gtk_tree_model_get (model, iter, EDITOR_COLUMN_TYPE, &type, -1);
  switch (type)
    {
      case GNOME_KEYRING_ITEM_GENERIC_SECRET:
        g_object_set (G_OBJECT (renderer), "stock-id", GNOME_STOCK_AUTHENTICATION, NULL);
        break;
      case GNOME_KEYRING_ITEM_NETWORK_PASSWORD:
        g_object_set (G_OBJECT (renderer), "stock-id", GTK_STOCK_NETWORK, NULL);
        break;
      case GNOME_KEYRING_ITEM_NOTE:
        g_object_set (G_OBJECT (renderer), "stock-id", GNOME_STOCK_BOOK_OPEN, NULL);
        break;
      default:
        g_object_set (G_OBJECT (renderer), "stock-id", GNOME_STOCK_BLANK, NULL);
    }
}

static void
gkm_main_ui_initialize_keys_tree_view (GKMMainUI *self)
{
  GtkTreeViewColumn   *column;
  GtkCellRenderer     *renderer;
  
  g_return_if_fail (GKM_IS_MAIN_UI (self));
  
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Type"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keys_treeview), column);
  renderer = gtk_cell_renderer_pixbuf_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_cell_data_func (column, renderer, 
					   (GtkTreeCellDataFunc) key_type_get_pixbuf, 
					   NULL, NULL);
  gtk_tree_view_column_set_sort_column_id (column, EDITOR_COLUMN_TYPE);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Name"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keys_treeview), column);
  gtk_tree_view_column_set_expand (column, TRUE);
  renderer = gtk_cell_renderer_text_new ();
  g_object_set (G_OBJECT (renderer), "editable", TRUE, NULL);
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_add_attribute (column, renderer, "text", EDITOR_COLUMN_NAME);
  gtk_tree_view_column_set_sort_column_id (column, EDITOR_COLUMN_NAME);
  
  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Modification Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keys_treeview), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (EDITOR_COLUMN_MTIME), NULL);
  gtk_tree_view_column_set_sort_column_id (column, EDITOR_COLUMN_MTIME);

  column = gtk_tree_view_column_new ();
  gtk_tree_view_column_set_title (column, _("Creation Time"));
  gtk_tree_view_append_column (GTK_TREE_VIEW (self->priv->keys_treeview), column);
  renderer = gtk_cell_renderer_text_new ();
  gtk_tree_view_column_pack_start (column, renderer, TRUE);
  gtk_tree_view_column_set_visible (column, FALSE);
  gtk_tree_view_column_set_cell_data_func (column,
  				           renderer,
				           (GtkTreeCellDataFunc) format_date_cell_data_func,
				           GINT_TO_POINTER (EDITOR_COLUMN_CTIME), NULL);
  gtk_tree_view_column_set_sort_column_id (column, EDITOR_COLUMN_CTIME);
}

static void
gkm_main_ui_populate_key_attributes_callback (GKMMainUI                 *self,
					      GnomeKeyringItemType       item_type,
					      GnomeKeyringAttributeList *attributes)
{
  gkm_attribute_display_set_item_type  (self->priv->attribute_display, item_type);
  gkm_attribute_display_set_attributes (self->priv->attribute_display, attributes);
}

static void
gkm_main_ui_populate_key_attributes (GKMMainUI   *self,
				     GtkTreeIter  iter)
{
  gchar *secret;
  GtkTreeModel *model;

  g_return_if_fail (GKM_IS_MAIN_UI (self));

  model = gkm_keyring_editor_get_model (self->priv->editor);

  gtk_tree_model_get (model, &iter, 
		      EDITOR_COLUMN_SECRET, &secret, 
		      -1);

  gkm_attribute_display_set_secret (self->priv->attribute_display, secret);

  g_free (secret);

  gkm_keyring_editor_get_attributes_list (self->priv->editor,
					  iter,
					  (GKMKeyringEditorGetAttributesCallback *)
					  gkm_main_ui_populate_key_attributes_callback,
					  self);
}

static void
gkm_main_ui_populate_key_access_list (GKMMainUI   *self,
				      GtkTreeIter  iter)
{
  g_return_if_fail (GKM_IS_MAIN_UI (self));
}

static void
gkm_main_ui_update_title (GKMMainUI *self)
{
  gchar *keyring_name;

  g_return_if_fail (GKM_IS_MAIN_UI (self));

  keyring_name = gkm_keyring_editor_get_keyring_name (self->priv->editor);

  if (keyring_name == NULL)
    {
      gtk_window_set_title (GTK_WINDOW (self->priv->window), _("Keyring Manager"));
    }
  else
    {
      gchar *title;

      title = g_strdup_printf (_("%s Keyring"), keyring_name);
      gtk_window_set_title (GTK_WINDOW (self->priv->window), title);

      g_free (title);
    }

  g_free (keyring_name);
}

/************************************************************
 * Manually connected signals
 */

static void
on_gkm_window_destroy (GtkWidget *window G_GNUC_UNUSED,
		       GKMMainUI *self)
{
  g_return_if_fail (GKM_IS_MAIN_UI (self));

  g_signal_emit (self, destroy_signal, 0);

  g_object_run_dispose (G_OBJECT (self));
}

/* Manager menu functions */

static void
create_keyring_dialog_cb (GtkDialog *dialog,
			  gint       response_id,
			  GKMMainUI *self)
{
  gchar *name;
  gchar *password;

  switch (response_id)
    {
    case GTK_RESPONSE_DELETE_EVENT:
    case GTK_RESPONSE_CANCEL:
      break;
    case GTK_RESPONSE_ACCEPT:
      name = gkm_create_keyring_dialog_get_name (dialog);
      password = gkm_create_keyring_dialog_get_password (dialog);

      gkm_keyring_manager_create_keyring (self->priv->manager,
					  name, password);

      g_free (name);
      g_free (password);
      break;
    default:
      g_assert_not_reached ();
    }

  gtk_widget_destroy (dialog);
}

static void
on_new_keyring_activate (GtkWidget *widget G_GNUC_UNUSED,
			 GKMMainUI *self)
{
  GtkWidget *dialog;

  dialog = gkm_create_keyring_dialog_new ();

  g_signal_connect (G_OBJECT (dialog),
		    "response",
		    G_CALLBACK (create_keyring_dialog_cb),
		    self);

  gtk_window_present (GTK_WINDOW (dialog));
}

/* Deleting a key or keyring */

typedef struct _DeleteActionCBData
{
  gpointer    object;
  GtkTreeIter iter;
  void        (*deletion_func)();
} DeleteActionCBData;

static void
delete_action_dialog_cb (GtkDialog          *dialog,
			 gint                response_id,
			 DeleteActionCBData *data)
{
  switch (response_id)
    {
    case GTK_RESPONSE_DELETE_EVENT:
    case GTK_RESPONSE_CANCEL:
      break;
    case GTK_RESPONSE_ACCEPT:
      data->deletion_func (data->object, data->iter);
      break;
    default:
      g_assert_not_reached ();
    }

  g_free (data);
  gtk_widget_destroy (GTK_WIDGET (dialog));
}

static void
delete_action_common (GKMMainUI          *self,
		      const gchar        *primary_text,
		      const gchar        *secondary_text,
		      DeleteActionCBData *data,
		      GtkTreeSelection   *selection,
		      gint                column)
{
  GtkWidget    *dialog;
  GtkTreeModel *model;
  gchar        *name;
  
  if (gtk_tree_selection_get_selected (selection, &model, &data->iter))
    {
      gtk_tree_model_get (model, &data->iter, 
			  column, &name, -1);
      
      dialog = gtk_message_dialog_new (GTK_WINDOW (self->priv->window),
				       GTK_DIALOG_DESTROY_WITH_PARENT,
				       GTK_MESSAGE_QUESTION,
				       GTK_BUTTONS_CANCEL,
				       primary_text,
				       name);
      
      gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
						secondary_text);
      
      gtk_dialog_add_buttons (GTK_DIALOG (dialog),
			      GTK_STOCK_DELETE,
			      GTK_RESPONSE_ACCEPT,
			      NULL);
      
      g_signal_connect (G_OBJECT (dialog),
			"response",
			G_CALLBACK (delete_action_dialog_cb),
			data);
      
      gtk_window_present (GTK_WINDOW (dialog));
      
      g_free (name);
    }
  else
    {
      g_free (data);
    }
}

static void
on_delete_key_activate (GtkWidget *widget G_GNUC_UNUSED,
			GKMMainUI *self)
{
  GtkTreeSelection   *selection;
  DeleteActionCBData *data;

  data = g_new0 (DeleteActionCBData, 1);
  data->object = self->priv->editor;
  data->deletion_func = gkm_keyring_editor_delete_key;
  
  selection = gtk_tree_view_get_selection (self->priv->keys_treeview);

  delete_action_common (self,
			_("Delete the '%s' key?"),
			_("Deleting a key cannot be undone."),
			data, selection, EDITOR_COLUMN_NAME);
}

static void
on_delete_keyring_activate (GtkWidget *widget G_GNUC_UNUSED,
			    GKMMainUI *self)
{
  GtkTreeSelection    *selection;
  DeleteActionCBData  *data;

  data = g_new0 (DeleteActionCBData, 1);
  data->object = self->priv->manager;
  data->deletion_func = gkm_keyring_manager_delete_keyring;
  
  selection = gtk_tree_view_get_selection (self->priv->keyrings_treeview);

  delete_action_common (self,
			_("Delete the '%s' keyring?"),
			_("Deleting a keyring cannot be undone."),
			data, selection, MANAGER_COLUMN_KEYRING);
}

/* End manager menu functions */

/* View menu functions */

static void
on_show_keyrings_activate (GtkWidget *widget G_GNUC_UNUSED,
			   GKMMainUI *self)
{
  GtkWidget *scroll_window;

  g_return_if_fail (GKM_IS_MAIN_UI (self));

  scroll_window = gtk_widget_get_parent (self->priv->keyrings_treeview);

  if (!self->priv->keyrings_visible) 
    {
      gtk_widget_show (scroll_window);
    }
  else 
    {
      gtk_widget_hide (scroll_window);
    }

  self->priv->keyrings_visible = !self->priv->keyrings_visible;
}

/* End of view menu functions */

/* Help menu functions */

static void
on_help_activate (GtkWidget *widget G_GNUC_UNUSED,
		  GKMMainUI *self)
{
  GError *error;

  g_return_if_fail (GKM_IS_MAIN_UI (self));

  error = NULL;
  
  /* TODO: handle the error */
  gnome_help_display ("gnome-keyring-manager", NULL, &error);

  if (error != NULL) {
    g_error ("Unable to load help: %s", error->message);

    g_error_free (error);
  }
}

static void
on_about_activate (GtkWidget *widget G_GNUC_UNUSED,
		   GKMMainUI *self)
{
  g_return_if_fail (GKM_IS_MAIN_UI (self));

  gkm_application_open_about_dialog (GTK_WINDOW (self->priv->window));
}

/* End of help menu functions */

/* Keyrings treeview functions */

static gboolean
on_keyrings_treeview_button_press_event (GtkWidget         *keyrings_treeview,
					 GdkEventButton    *event,
					 GKMMainUI         *self)
{
  g_return_if_fail (GKM_IS_MAIN_UI (self));
  
  if (event->type == GDK_BUTTON_PRESS && event->button == 3)
    {
      GtkTreeSelection *selection;
      GtkTreePath *path;
      
      selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (keyrings_treeview));
      
      /* Only bring up the menu if there is a row under the mouse. */
      if (gtk_tree_view_get_path_at_pos (GTK_TREE_VIEW (keyrings_treeview),
					 event->x, event->y,
					 &path, NULL, NULL, NULL))
	{
	  GtkTreeIter iter;
	  
	  gtk_tree_selection_unselect_all (selection);
	  gtk_tree_selection_select_path (selection, path);
	  //	  gtk_tree_model_get_iter (GTK_TREE_MODEL (self->priv->keyrings), 
	  //		   &iter, path);
	  gtk_tree_path_free (path);
	  
	  //	   keyring_tree_popup_menu (keyrings_treeview, &iter, event, self);
	}
      
      return TRUE;
    }
  else
    {
      return FALSE;
    }
} 

static gboolean
on_keyrings_treeview_popup_menu (GtkWidget *widget G_GNUC_UNUSED,
				 GKMMainUI *self)
{
  g_return_val_if_fail (GKM_IS_MAIN_UI (self), FALSE);

  return FALSE;
}

static void
on_keyrings_treeview_selection_changed (GtkTreeSelection *selection,
					GKMMainUI        *self)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  g_return_if_fail (GKM_IS_MAIN_UI (self));

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      GtkTreePath *path;
      gchar *keyring_name;
      
      gtk_tree_model_get (model, &iter, MANAGER_COLUMN_KEYRING, &keyring_name, -1);
      
      gkm_keyring_editor_set_keyring_name (self->priv->editor, keyring_name);

      gkm_main_ui_update_title (self);
    }
}

/* End of keyrings treeview functions */

/* Key treeview functions */

static void
on_keys_treeview_selection_changed (GtkTreeSelection *selection,
				    GKMMainUI        *self)
{
  GtkTreeModel *model;
  GtkTreeIter   iter;

  g_return_val_if_fail (GKM_IS_MAIN_UI (self), FALSE);

  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    {
      gkm_main_ui_populate_key_attributes (self, iter);
    }
}

/* End of key treeview functions */

static void 
gkm_main_ui_connect_glade_signals (GKMMainUI *self)
{
  GtkWidget *widget;
  GtkTreeSelection *selection;

  g_signal_connect (G_OBJECT (self->priv->window),
		    "destroy",
		    G_CALLBACK (on_gkm_window_destroy),
		    self);

  g_signal_connect (self->priv->keyrings_treeview, 
		    "button-press-event", 
		    G_CALLBACK (on_keyrings_treeview_button_press_event), 
		    self);
  g_signal_connect (self->priv->keyrings_treeview, 
		    "popup-menu", 
		    G_CALLBACK (on_keyrings_treeview_popup_menu), 
		    self);

  widget = glade_xml_get_widget (self->priv->xml, "new_keyring");
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK (on_new_keyring_activate),
		    self);

  widget = glade_xml_get_widget (self->priv->xml, "delete_key");
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK (on_delete_key_activate),
		    self);

  widget = glade_xml_get_widget (self->priv->xml, "delete_keyring");
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK (on_delete_keyring_activate),
		    self);

  widget = glade_xml_get_widget (self->priv->xml, "close");
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK (on_gkm_window_destroy),
		    self);

  widget = glade_xml_get_widget (self->priv->xml, "show_keyrings");
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK (on_show_keyrings_activate),
		    self);

  widget = glade_xml_get_widget (self->priv->xml, "contents");
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK (on_help_activate),
		    self);

  widget = glade_xml_get_widget (self->priv->xml, "about");
  g_signal_connect (G_OBJECT (widget),
		    "activate",
		    G_CALLBACK (on_about_activate),
		    self);

  selection = 
    gtk_tree_view_get_selection (GTK_TREE_VIEW (self->priv->keyrings_treeview));
  g_signal_connect (G_OBJECT (selection),
		    "changed",
		    G_CALLBACK (on_keyrings_treeview_selection_changed), 
		    self);

  selection = 
    gtk_tree_view_get_selection (GTK_TREE_VIEW (self->priv->keys_treeview));
  g_signal_connect (G_OBJECT (selection),
		    "changed",
		    G_CALLBACK (on_keys_treeview_selection_changed), 
		    self);
}
