/* 
   Copyright (C) 2004 Fernando Herrera <fherrera@onirica.com>
   Copyright (C) 2004 Mariano Suárez-Alvarez <mariano@gnome.org>
   Copyright (C) 2004 GNOME Love Project <gnome-love@gnome.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <config.h>

#include <glib.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-about.h>

#include <time.h>
#include <string.h>

#include "gnome-keyring-manager.h"
#include "gnome-keyring-manager-i18n.h"
#include "gnome-keyring-manager-util.h"

/********************************************************************
 * Utility functions
 */

void
error_dialog (GtkWindow *transient_parent, const char *fmt, ...)
{
  GtkWidget *dialog;
  char *msg;
  va_list args;

  va_start (args, fmt);
  
  msg = g_strdup_vprintf (fmt, args);

  va_end (args);
  
  dialog = gtk_message_dialog_new (transient_parent,
                                   GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_ERROR,
                                   GTK_BUTTONS_OK,
                                   "%s",
                                   msg);
  g_free (msg);
  
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}

void
complain_about_gnome_keyring_bad_result (GtkWindow          *transient_parent,
                                         GnomeKeyringResult  result)
{
  /*
   * FIXME: There are messages missing here...
   */

  switch (result) 
    {
      case GNOME_KEYRING_RESULT_NO_KEYRING_DAEMON:
        error_dialog (GTK_WINDOW (transient_parent), _("GNOME Keyring daemon is not running."));
        break;
        
      case GNOME_KEYRING_RESULT_IO_ERROR:
        error_dialog (GTK_WINDOW (transient_parent), _("Cannot communicate with GNOME Keyring daemon."));
        break;
        
      case GNOME_KEYRING_RESULT_DENIED:
        error_dialog (GTK_WINDOW (transient_parent), _("Access denied to GNOME Keyring."));
        break;

      case GNOME_KEYRING_RESULT_ALREADY_EXISTS:
        error_dialog (GTK_WINDOW (transient_parent), _("Keyring already exists."));
        break;
      
      case GNOME_KEYRING_RESULT_NO_SUCH_KEYRING:
        error_dialog (GTK_WINDOW (transient_parent), _("No such keyring."));
        break;
        
      case GNOME_KEYRING_RESULT_ALREADY_UNLOCKED:
        error_dialog (GTK_WINDOW (transient_parent), _("Keyring already unlocked."));
        break;
      
      case GNOME_KEYRING_RESULT_OK:
        g_warning ("complain_about_gnome_keyring_bad_result called about GNOME_KEYRING_RESULT_OK");
        break;

      default:
        error_dialog (GTK_WINDOW (transient_parent), _("Unknown GNOME Keyring error."));
    }
}

/**
 * format_date_for_display:
 * @date: a #time_t
 *
 * Returns: @date formatted in a way suitable for display.
 * 
 * This code is taken from evolution/mail/message-list.c
 */

char *
format_date_for_display (time_t date)
{
  time_t nowdate = time(NULL);
  time_t yesdate;
  struct tm then, now, yesterday;
  char buf[26];
  gboolean done = FALSE;

  if (date == 0)
    {
      return g_strdup (_("?"));
    }

  localtime_r (&date, &then);
  localtime_r (&nowdate, &now);
  if (then.tm_mday == now.tm_mday &&
      then.tm_mon == now.tm_mon &&
      then.tm_year == now.tm_year) 
    {
      strftime (buf, 26, _("Today %l:%M %p"), &then);
      done = TRUE;
    }

  if (!done) 
    {
      yesdate = nowdate - 60 * 60 * 24;
      localtime_r (&yesdate, &yesterday);
      if (then.tm_mday == yesterday.tm_mday &&
          then.tm_mon == yesterday.tm_mon &&
          then.tm_year == yesterday.tm_year) 
        {
          strftime (buf, 26, _("Yesterday %l:%M %p"), &then);
          done = TRUE;
        }
    }

  if (!done) 
    {
      int i;
      for (i = 2; i < 7; i++) 
        {
          yesdate = nowdate - 60 * 60 * 24 * i;
          localtime_r (&yesdate, &yesterday);
          if (then.tm_mday == yesterday.tm_mday &&
              then.tm_mon == yesterday.tm_mon &&
              then.tm_year == yesterday.tm_year) 
            {
              strftime (buf, 26, _("%a %l:%M %p"), &then);
              done = TRUE;
              break;
            }
        }
    }

  if (!done) 
    {
      if (then.tm_year == now.tm_year) 
        {
          strftime (buf, 26, _("%b %d %l:%M %p"), &then);
        } 
      else 
        {
            strftime (buf, 26, _("%b %d %Y"), &then);
        }
    }

  return g_strdup (buf);
}

GtkWidget *
gkm_about_dialog_new (GtkWindow *transient_parent)
{
  GtkWidget *about;
  GtkIconTheme *icon_theme;
  GdkPixbuf *pixbuf;

  g_return_val_if_fail (GTK_IS_WINDOW (transient_parent), NULL);

  const char *authors[] = {
    "Fernando Herrera <fherrera@onirica.com>",
    "Mariano Su\303\241rez-Alvarez <mariano@gnome.org>",
    NULL
  };
  const char *documenters[] = {
    NULL
  };
  const char *translator_credits = _("translator_credits");
  
  icon_theme = gtk_icon_theme_get_default ();
  pixbuf = gtk_icon_theme_load_icon (icon_theme, "stock_keyring", 48, 0, NULL);
	   
  about = gnome_about_new (_("GNOME Keyring Manager"), VERSION, 
                           _("Copyright \302\251 2004 The GNOME Love Project"),
                           _("A simple keyring manager for GNOME 2"),
                           (const char **) authors, 
                           (const char **) documenters,
                           strcmp (translator_credits, "translator_credits") != 0 ? translator_credits : NULL,
                           pixbuf);

  if (pixbuf != NULL)
    {
      gdk_pixbuf_unref (pixbuf);
    }
              
  gtk_window_set_transient_for (GTK_WINDOW (about), transient_parent);
  
  return about;
}
/********************************************************************
 * Toggle display of columns
 */

void
toggle_view_column_action (GtkToggleAction *toggle_action, 
                           GConfClient	   *gconf_client)
{
  gboolean old_state, state;
  char *key;
  GError *error;

  state = gtk_toggle_action_get_active (toggle_action);
  key = g_object_get_data (G_OBJECT (toggle_action), "key");

  error = NULL;
  old_state = gconf_client_get_bool (gconf_client, key, &error);
  if (error != NULL)
    {
      g_warning (_("Error while trying to read the gconf key %s: %s"), key, error->message);
      g_error_free (error);
    }
 
  if (state == old_state)
    {
      /* Avoid an infinite loop */
      return;
    }

  /* FIXME: This key might be read-only... In that case we need to call the
   * gconf callback ourselves. Sigh.
   */
  error = NULL;
  gconf_client_set_bool (gconf_client, key, state, &error);
  if (error != NULL)
    {
      g_warning (_("Error while setting the gconf key %s: %s"), key, error->message);
      g_error_free (error);
    }
}

void
update_column_visibility (GConfClient       *client G_GNUC_UNUSED,
                          guint              cnxn_id G_GNUC_UNUSED,
                          GConfEntry        *entry,
                          GtkTreeViewColumn *column)
{
  GConfValue *value;
  gboolean state;
  GtkToggleAction *action;

  g_return_if_fail (GTK_IS_TREE_VIEW_COLUMN (column));

  value = gconf_entry_get_value (entry);
  state = gconf_value_get_bool (value);
  action = g_object_get_data (G_OBJECT (column), "action");

  gtk_tree_view_column_set_visible (column, state);
  gtk_toggle_action_set_active (action, state);
}

GSList *
set_column_visibility_preferences (const SetColumnPrefsData *toggle,
				   guint		     num_elements,
				   const gchar              *dir_name,
				   GtkTreeView              *tree_view,
				   GtkActionGroup           *action_group,
				   GConfClient		    *gconf_client)
{
  unsigned int i;
  GtkAction *action;
  GtkTreeViewColumn *column;
  char *dir;
  GError *error;
  GSList *gconf_cnxn_ids;

  dir = gconf_concat_dir_and_key (dir_name, "columns");

  error = NULL;
  gconf_client_add_dir (gconf_client, dir, GCONF_CLIENT_PRELOAD_ONELEVEL, &error); 
  if (error != NULL)
    {
      g_warning (_("Error trying to add %s to the list of gconf directories we listen to: %s"), dir, error->message);
      g_error_free (error);
    }

  gconf_cnxn_ids = NULL;

  for (i = 0; i < num_elements; i++)
    {
      char *key;
      gboolean state;
      int id;

      column = gtk_tree_view_get_column ((tree_view), toggle[i].column_id);
      action = gtk_action_group_get_action (action_group, toggle[i].action_name);

      key = gconf_concat_dir_and_key (dir, toggle[i].key_name);
      g_object_set_data_full (G_OBJECT (action), "key", key, g_free);

      g_object_set_data (G_OBJECT (column), "action", action);

      error = NULL;
      state = gconf_client_get_bool (gconf_client, key, &error); 
      if (error != NULL)
        {
          g_warning (_("Error while trying to read the gconf key %s: %s"), key, error->message);
          g_error_free (error);
        }

      gtk_tree_view_column_set_visible (column, state);
      gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action), state);

      error = NULL;
      id = gconf_client_notify_add (gconf_client, key, (GConfClientNotifyFunc) update_column_visibility, column, NULL, &error);
      if (error != NULL)
        {
          g_warning (_("Error while trying to add a listener for changed on the gconf key %s: %s"), key, error->message);
          g_error_free (error);
        }

      gconf_cnxn_ids = g_slist_prepend (gconf_cnxn_ids, GINT_TO_POINTER (id));
    }

  g_free (dir);

  return gconf_cnxn_ids;
}

static void
disconnect_gconf_listener (gpointer data,
                           GConfClient *gconf_client)
{
  gconf_client_notify_remove (gconf_client, GPOINTER_TO_INT (data));
}

void
close_gconf_connections (GSList      *connections, 
	                 const gchar *dir_name,
			 GConfClient *gconf_client)
{
  char   *dir;
  GError *error;

  g_slist_foreach (connections, (GFunc) disconnect_gconf_listener, gconf_client);
  g_slist_free (connections);

  dir = gconf_concat_dir_and_key (dir_name, "columns");
  error = NULL;
  gconf_client_remove_dir (gconf_client, dir, &error);
  if (error != NULL)
    {
      g_printerr (_("Error while trying to remove the %s dir from the list of gconf directories we listen to: %s"), 
                  dir, error->message);
      g_error_free (error);
    }
  g_free (dir);
}

