/*  -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Symbol viewing for libdryad.
 * 
 * Copyright (C) 1999 Dave Camp <campd@oit.edu>, 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include "libdryad.h"
#include <string.h>
#include <ctype.h>

#include "gdf-libdryad-debugger.h"

static void read_type (GdbInstance *inst, GDF_Symbol *sym);
static void read_value (GdbInstance *inst, GDF_Symbol *sym);
static void expand_pointer (GdbInstance *inst, GdfLibdryadSymbolSet *ss,
			    GDF_Symbol *sym);
static void expand_structure (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
			      GDF_Symbol *sym);
static void expand_array (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
			  GDF_Symbol *sym);
static void destroy_children (GDF_Symbol *sym);
static GDF_SymbolSequence *init_seq (void);
static void grow_seq (GDF_SymbolSequence *seq, int amount);
void
gdb_get_locals (GdbInstance *inst, GdfLibdryadSymbolSet *ss)
{
    char *line;
    GList *names = NULL;
    GList *i;
    char *work;

    if (ss->symbols) {
	CORBA_free (ss->symbols);
    }
    ss->set_size = 0;
    
    gdb_execute_command (inst, "info locals");

    ss->symbols = gdb_allocate_symbols ();

    /* Read the names of all the locals */
    line = gdb_read_line (inst);
    while (strcmp (line, "\32\32pre-prompt")) {
	if ((work = strstr (line, " = "))) {
	    *work = '\0';
	    work += 3;
	    
	    names = g_list_prepend (names, g_strdup (line));
	    
	    if (!strncmp (work, "{", 1)) {
		int levels = 1;
		while (levels > 0) {
		    line = gdb_read_line (inst);
		    if (!strncmp (line, "{", 1) || !strncmp (line, ", {", 3))
			levels++;
		    if (!strncmp (line, "}", 1))
			levels--;
		}
	    }
	}
	line = gdb_read_line (inst);
    }

    /* Add all the locals to the symbol set */
    names = g_list_reverse (names);

    for (i = names; i != NULL; i = i->next) {
	gdb_add_expression (inst, ss, i->data);
	g_free (i->data);
    }
    g_list_free (names);

    /* Notify the symbol set that it has changed */
    gdf_libdryad_symbol_set_changed (ss);
}

void
gdb_get_expression (GdbInstance *inst, GDF_Symbol *sym, const char *expr)
{
    sym->name = CORBA_string_dup (expr);
    sym->expression = CORBA_string_dup (expr);

    read_type (inst, sym);
    read_value (inst, sym);
}

void
gdb_expand_symbol (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		   GDF_Symbol *sym)
{
    g_return_if_fail (sym->expandable);

    if (!sym->expanded) {
	switch (sym->type) {
	case GDF_TYPE_PTR :
	    expand_pointer (inst, ss, sym);
	    break;
	case GDF_TYPE_STRUCT :
	    expand_structure (inst, ss, sym);
	    break;
	case GDF_TYPE_ARRAY :
	    expand_array (inst, ss, sym);
	default:
	    return;
	}
    }

    sym->expanded = TRUE;
}

static void
update_seq (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
	    GDF_SymbolSequence *seq) 
{
    char *old_value;
    char *old_typename;
    GDF_Symbol *sym;
    CORBA_boolean was_expandable;

    int i;

    for (i = 0; i < seq->_length; i++) {
	sym = &seq->_buffer[i];

	old_value = sym->value;
	sym->value = NULL;
	old_typename = sym->typename;
	sym->typename = NULL;
	was_expandable = sym->expandable;

	read_type (inst, sym);
	read_value (inst, sym);
	
	if (strcmp (old_typename, sym->typename)) {
	    CORBA_any *event_any;
	    
	    destroy_children (sym);

	    event_any = gdf_marshal_event_long ("symbol_type_changed", 
						sym->handle);
	    
	    gdf_event_channel_client_push (ss->event_channel,
					   event_any);
	    CORBA_free (event_any);
	}
	
	if (strcmp (old_value, sym->value) 
	    || was_expandable != sym->expandable) {
	    CORBA_any *event_any;

	    event_any = gdf_marshal_event_long ("symbol_changed", 
						sym->handle);
	    
	    gdf_event_channel_client_push (ss->event_channel,
					   event_any);
	    CORBA_free (event_any);
	}
	
	CORBA_free (old_value);
	CORBA_free (old_typename);
	
	update_seq (inst, ss, 
		    (GDF_SymbolSequence*)sym->children._value);
    }
}

void
gdb_update_symbols (GdbInstance *inst, GdfLibdryadSymbolSet *ss)
{
    update_seq (inst, ss, ss->symbols);
}

GDF_SymbolSequence *
gdb_allocate_symbols (void)
{
    return init_seq ();
}

static void
add_expression_to_seq  (GdbInstance *inst,
			GDF_SymbolSequence *seq, 
			GdfLibdryadSymbolSet *ss, 
			const char *expr)
{
    GDF_Symbol *sym;
    
    grow_seq (seq, 1);
    sym = &seq->_buffer[seq->_length - 1];
    gdb_get_expression (inst, sym, expr);
    
    sym->handle = ss->set_size++;
}

void
gdb_add_expression (GdbInstance *inst,
		    GdfLibdryadSymbolSet *ss, 
		    const char *expr)
{
    add_expression_to_seq (inst, ss->symbols, ss, expr);
}

void
read_type (GdbInstance *inst, GDF_Symbol *sym)
{
    char *line;
    char *cmd_str;

    g_assert (sym->expression);
    
    cmd_str = g_strdup_printf ("ptype %s", sym->expression);
    gdb_execute_command (inst, cmd_str);
    g_free (cmd_str);

    if (sym->typename) 
	CORBA_free (sym->typename);
    sym->typename = NULL;

    line = gdb_read_line (inst);
    while (strcmp (line, "\32\32pre-prompt")) {
	if (line[0] != '\0' && strcmp (line, "\32\32post-prompt")) {
	    char *type;
	    if (!strcmp (line, "\32\32error-begin"))
		line = gdb_read_line (inst);
	    if (!strncmp (line, "type = ", strlen ("type = ")))
		line += 7;

	    type = g_strdup (line);

	    if (line[strlen (line) - 1] == '{') {
		int depth = 1;
		char *newtype;
		while (depth) {
		    line = gdb_read_line (inst);
		    while (*line) {
			if (*line == '{')
			    depth++;
			if (*line++ == '}')
			    depth--;
		    }
		}
		
		newtype = g_strdup_printf ("%s%s", type, line);
		g_free (type);
		type = newtype;
	    }

	    sym->typename = CORBA_string_dup (type);
	    g_free (type);
	    
	    break;
	}
	line = gdb_read_line (inst);
    }

    g_assert (sym->typename);
    
    sym->expandable = FALSE;
    sym->type = GDF_TYPE_NORMAL;
    if (!strncmp (sym->typename, "struct", strlen ("struct"))) {
	sym->expandable = TRUE;
	sym->type = GDF_TYPE_STRUCT;
    }
    if (sym->typename[strlen (sym->typename) - 1] == ']') {
	sym->expandable = TRUE;
	sym->type = GDF_TYPE_ARRAY;
    }
    if (sym->typename[strlen (sym->typename) - 1] == '*') {
	sym->expandable = TRUE;
	sym->type = GDF_TYPE_PTR;
    }
}

void
read_value (GdbInstance *inst, GDF_Symbol *sym) 
{
    char *cmd_str;
    char *line;

    g_assert (!sym->value);
    sym->value = NULL;

    if (sym->type == GDF_TYPE_STRUCT) {
	sym->value = CORBA_string_dup ("{...}");
	return;
    } else if (sym->type == GDF_TYPE_ARRAY) {
	sym->value = CORBA_string_dup ("[...]");
	return;
    }

    cmd_str = g_strdup_printf ("output %s", sym->expression);
    gdb_execute_command (inst, cmd_str);
    g_free (cmd_str);

    line = gdb_read_line (inst);
    while (strcmp (line, "\32\32pre-prompt")) {
	if (!strncmp(line, "\32\32value-begin", strlen("\32\32value-begin"))) {
	    line = gdb_read_line (inst);
	    
	    while (line[0] == '\0' && strcmp (line, "\32\32\error-begin")) {
		line = gdb_read_line (inst);
	    }
	    
	    if (strcmp (line, "\32\32error-begin")) {
		sym->value = CORBA_string_dup (line);
	    } else {
		line = gdb_read_line (inst);
		while (strcmp (line, "\32\32error")) {
		    char *new;
		    new = g_strdup_printf ("%s%s",
					   sym->value ? sym->value : "", 
					   line);
		    if (sym->value) {
			CORBA_free (sym->value);
		    }
		    sym->value = CORBA_string_dup (new);
		    g_free (new);
		    line = gdb_read_line (inst);
		}
		break;
	    }
	    break;
	}
	if (!strcmp (line, "\32\32error-begin")) {
	    line = gdb_read_line (inst);
	    while (strcmp (line, "\32\32error")) {
		char *new;
		new = g_strdup_printf ("%s%s",
				       sym->value ? sym->value : "", 
				       line);
		if (sym->value) {
		    CORBA_free (sym->value);
		}
		sym->value = CORBA_string_dup (new);
		g_free (new);	
		line = gdb_read_line (inst);
	    }
	}
	line = gdb_read_line (inst);
    }
    g_assert (sym->value != NULL);
}

static void
add_child_expression (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		      GDF_Symbol *sym, const char *name, const char *expr)
{
    GDF_Symbol *new_sym;
    GDF_SymbolSequence *children;

    children = sym->children._value;
    add_expression_to_seq (inst, children, ss, expr);
    new_sym = &(children->_buffer[children->_length - 1]);
    CORBA_free (new_sym->name);
    new_sym->name = CORBA_string_dup (name);
}

void
expand_pointer (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		GDF_Symbol *sym)
{
    char *new_name;
    char *new_expr;
    new_name = g_strdup_printf ("*%s", sym->name);
    new_expr = g_strdup_printf ("*(%s)", sym->expression);
    add_child_expression (inst, ss, sym, new_name, new_expr);
    g_free (new_expr);
    g_free (new_name);
}

static char *
read_child_from_ptype (GdbInstance *inst)
{
    char *name;
    char *work;
    char *line;

    line = gdb_read_line (inst);

    if (line[0] == '}') 
	return NULL;

    /* Clean out the trailing ;, and any array indices if necessary */
    work = line + (strlen (line) - 1);
    if (*work == '{') {
	do {
	    line = gdb_read_line (inst);
	    while (isspace (*line++))
		;    
	    line--;
	} while (*line != '}');
    } 

    work = strrchr (line, ';');
    if (work) {
	*work-- = '\0';
    } else {
	work = line + (strlen(line) - 1);
    }    

    while (*work == ']') {
	while (*work-- != '[') 
	    ;
	*(work + 1) = '\0';
    }

    while (*work != '*' && *work-- != ' ')
	;
    name = work + 1;
    return g_strdup (name);
}

void
expand_structure (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
		  GDF_Symbol *sym)
{
    char *expr;
    char *cmd_str;
    char *line;
    char *name;
    GList *names = NULL;
    GList *i;

    cmd_str = g_strdup_printf ("ptype %s", sym->expression);
    gdb_execute_command (inst, cmd_str);
    g_free (cmd_str);

    do {
	line = gdb_read_line (inst);
    } while (strncmp (line, "type = ", strlen ("type = ")));

    name = read_child_from_ptype (inst);
    while (name) {
	names = g_list_prepend (names, name);
	name = read_child_from_ptype (inst);
    }

    names = g_list_reverse (names);
    for (i = names; i != NULL; i = i->next) {
	g_assert (i != NULL);
	expr = g_strdup_printf ("(%s).%s", sym->expression, (char*)i->data);
	g_assert (i != NULL);
	add_child_expression (inst, ss, sym, i->data, expr);
	g_free (expr);
	g_free (i->data);
    }
    g_list_free (names); 
}

void
expand_array (GdbInstance *inst, GdfLibdryadSymbolSet *ss, 
	      GDF_Symbol *sym)
{
    char *typecpy;
    char *p;
    int size;
    int i;
    
    typecpy = g_strdup (sym->typename);

    /* isolate the first index size */
    p = strchr (typecpy, ']');
    g_assert (p);
    *p = '\0';
    
    p = strchr (typecpy, '[');
    g_assert (p);

    size = atoi (p + 1);
    
    g_free (typecpy);

    for (i = 0; i < size; i++) {
	char *expr;
	char *name;
 	expr = g_strdup_printf ("(%s)[%d]", sym->expression, i);
	name = g_strdup_printf ("[%d]", i);
	add_child_expression (inst, ss, sym, name, expr);
	g_free (expr);
	g_free (name);
    }
}

static void
destroy_children (GDF_Symbol *sym)
{
    CORBA_free (sym->children._value);
    sym->children._value = init_seq ();
    sym->expanded = CORBA_FALSE;
}

static GDF_SymbolSequence *
init_seq (void)
{
    GDF_SymbolSequence *seq = GDF_SymbolSequence__alloc ();
    seq->_length = 0;
    seq->_maximum = 0;
    seq->_buffer = NULL;
    /* We tell it not to release now, but will set release to TRUE when the
     * buffer is grown for the first time */
    CORBA_sequence_set_release (seq, CORBA_FALSE);
    return seq;
}

void
grow_seq (GDF_SymbolSequence *seq, int amount)
{
    int i;

    int old_max = seq->_maximum;
    GDF_Symbol *old_buf = seq->_buffer;

    seq->_maximum += amount;
    seq->_length += amount;
    seq->_buffer = CORBA_sequence_GDF_Symbol_allocbuf (seq->_maximum);

    if (old_buf) {
	ORBit_mem_info *block;

	for (i = 0; i < old_max; i++) {
	    GDF_Symbol *sym = &(seq->_buffer[i]);
	    GDF_Symbol *old_sym = &old_buf[i];
	    sym->symbol_set = old_sym->symbol_set;
	    sym->handle = old_sym->handle;
	    sym->name = old_sym->name;
	    sym->expression = old_sym->expression;
	    sym->type = old_sym->type;
	    sym->value = old_sym->value;
	    sym->typename = old_sym->typename;
	    sym->expandable = old_sym->expandable;
	    sym->expanded = old_sym->expanded;
	    sym->children = old_sym->children;
	}

	/* This is somewhat not nice.  Basically, we want to free the previous
	 * buffer, but we don't want the deep free that is done by 
	 * CORBA_free.  To get around this we cheat and free the buffer
	 * ourselves */

	block = PTR_TO_MEMINFO (old_buf);
	g_free (block);
    } else {
	/* _release was previously set to FALSE (because there was no buffer).
	 * set it to true now that a buffer has been created. */
	CORBA_sequence_set_release (seq, CORBA_TRUE);
    }

    for (i = seq->_length - amount; i < seq->_length; i++) {
	GDF_Symbol *sym = &(seq->_buffer[i]);
	sym->symbol_set = -1;
	sym->handle = -1;
	sym->type = GDF_TYPE_NORMAL;
	sym->expandable = CORBA_FALSE;
	sym->expanded = CORBA_FALSE;
	sym->children._type = TC_CORBA_sequence_GDF_Symbol;
	sym->children._value = init_seq ();
	CORBA_any_set_release (&sym->children, CORBA_TRUE);
    }
}












