/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include "gdf-variable-viewer.h"
#include <gdf.h>

struct _GdfVariableViewerPriv {
	GtkWidget *ctree;
    GtkWidget *entry;
    GtkWidget *add_btn;
	GtkWidget *scrolled;
    GtkWidget *entry_hbox;

    GTree *nodes;

	/* signal identifiers */
    guint symbol_changed_sig;
    guint symbol_set_changed_sig;
	guint program_unloaded_sig;

	GdfDebuggerClient *dbg;
    GdfSymbolSetClient *ss;
    gboolean view_locals;
};

enum {
    DEBUGGER_SET,
    LAST_SIGNAL
};

enum {
    ARG_0,
    ARG_DEBUGGER,
    ARG_SHOW_ENTRY
};

static gint variable_viewer_signals[LAST_SIGNAL];

static void variable_viewer_class_init (GdfVariableViewerClass *klass);
static void variable_viewer_init (GdfVariableViewer *vv);
static void variable_viewer_destroy (GtkObject *object);
static void get_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void set_arg (GtkObject *object, 
                     GtkArg *arg, 
                     guint arg_id);
static void create_children (GdfVariableViewer *vv);
static void add_node (GdfVariableViewer *vv, GtkCTreeNode *parent, 
                      GDF_Symbol *sym);
static void add_dummy_child (GdfVariableViewer *vv, GtkCTreeNode *parent);
static void clear_variable_viewer (GdfVariableViewer *vv);
static void connect_debugger_signals (GdfVariableViewer *vv);
static void disconnect_debugger_signals (GdfVariableViewer *vv);
static void connect_symbol_set_signals (GdfVariableViewer *vv);
static void disconnect_symbol_set_signals (GdfVariableViewer *vv);
static void add_btn_clicked_cb (GtkWidget *btn, gpointer data);
static void entry_activate_cb (GtkWidget *btn, gpointer data);
static void tree_expand_cb (GtkWidget *ctree, GtkCTreeNode *node, 
                            gpointer data);
static gint int_cmp (gconstpointer a, gconstpointer b);

static GtkFrameClass *parent_class;

/* 
 * Public interface 
 */

GtkType
gdf_variable_viewer_get_type ()
{
	static GtkType variable_viewer_type = 0;
	
	if (!variable_viewer_type) {
		static const GtkTypeInfo variable_viewer_info = {
			"GdfVariableViewer",
			sizeof (GdfVariableViewer),
			sizeof (GdfVariableViewerClass),
			(GtkClassInitFunc) variable_viewer_class_init,
			(GtkObjectInitFunc) variable_viewer_init,
			NULL,
			NULL,
			(GtkClassInitFunc)NULL
		};
	
		variable_viewer_type = gtk_type_unique (gtk_frame_get_type (),
                                                &variable_viewer_info);
	}
	
	return variable_viewer_type;
}

GdfVariableViewer *
gdf_variable_viewer_new ()
{
	GdfVariableViewer *vv;
	
	vv = gtk_type_new (gdf_variable_viewer_get_type ());
	
	return vv;
}

void 
gdf_variable_viewer_view_symbol_set (GdfVariableViewer *vv,
                                     GdfSymbolSetClient *ss)
{
    g_return_if_fail (vv != NULL);
	g_return_if_fail (GDF_IS_VARIABLE_VIEWER (vv));

    vv->priv->view_locals = FALSE;

    if (vv->priv->dbg) {
        if (!ss) {
            vv->priv->ss =
                gdf_debugger_client_allocate_symbol_set (vv->priv->dbg);
        } else {
            vv->priv->ss = ss;
        }
    }
    
    /* FIXME: Get the current symbol set */
}

void 
gdf_variable_viewer_view_locals (GdfVariableViewer *vv)
{
    g_return_if_fail (vv != NULL);
	g_return_if_fail (GDF_IS_VARIABLE_VIEWER (vv));

    vv->priv->view_locals = TRUE;

    if (vv->priv->dbg) {
        vv->priv->ss =
            gdf_debugger_client_get_locals (vv->priv->dbg);
    }
    
    /* FIXME: Get the current symbol set */
}

void 
gdf_variable_viewer_view_symbol (GdfVariableViewer *vv,
                                 GDF_Symbol *sym)
{
	g_return_if_fail (vv != NULL);
	g_return_if_fail (sym != NULL);
	g_return_if_fail (GDF_IS_VARIABLE_VIEWER (vv));

    add_node (vv, NULL, sym);
}

void
gdf_variable_viewer_clear (GdfVariableViewer *vv)
{
	g_return_if_fail (vv != NULL);
	g_return_if_fail (GDF_IS_VARIABLE_VIEWER (vv));

    clear_variable_viewer (vv);
}

void
gdf_variable_viewer_set_debugger (GdfVariableViewer *vv,
                                  GdfDebuggerClient *dbg)
{
	g_return_if_fail (vv != NULL);
	g_return_if_fail (GDF_IS_VARIABLE_VIEWER (vv));

    gdf_variable_viewer_clear (vv);

	if (vv->priv->dbg) {
		disconnect_debugger_signals (vv);
		gtk_object_unref (GTK_OBJECT (vv->priv->dbg));
	} 

    if (vv->priv->ss) {
        disconnect_symbol_set_signals (vv);
        gtk_object_unref (GTK_OBJECT (vv->priv->ss));
        vv->priv->ss = NULL;
    }
	
	vv->priv->dbg = dbg;

    if (dbg) {
        gtk_object_ref (GTK_OBJECT (dbg));
        if (GTK_OBJECT_FLOATING (GTK_OBJECT (dbg))) 
            gtk_object_sink (GTK_OBJECT (dbg));
        if (vv->priv->view_locals) {
            vv->priv->ss = gdf_debugger_client_get_locals (vv->priv->dbg);
        } else {
            vv->priv->ss = 
                gdf_debugger_client_allocate_symbol_set (vv->priv->dbg);
        }

        connect_debugger_signals (vv);
        connect_symbol_set_signals (vv);
    }

    /* FIXME: View the symbol set */

    gtk_signal_emit (GTK_OBJECT (vv), variable_viewer_signals[DEBUGGER_SET],
                     dbg);
}

/*
 * Class/object functions.
 */

void
variable_viewer_class_init (GdfVariableViewerClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *)klass;

    gtk_object_add_arg_type ("GdfVariableViewer::debugger",
                             GTK_TYPE_OBJECT,
                             GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT, 
                             ARG_DEBUGGER);
    gtk_object_add_arg_type ("GdfVariableViewer::show_entry",
                             GTK_TYPE_BOOL,
                             GTK_ARG_READWRITE, 
                             ARG_SHOW_ENTRY);
    
	parent_class = gtk_type_class (gtk_frame_get_type ());
    
    variable_viewer_signals [DEBUGGER_SET] =
        gtk_signal_new ("debugger_set",
                        GTK_RUN_FIRST,
                        object_class->type,
                        GTK_SIGNAL_OFFSET (GdfVariableViewerClass,
                                           debugger_set),
                        gtk_marshal_NONE__POINTER,
                        GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
    
    gtk_object_class_add_signals (object_class, 
                                  variable_viewer_signals,
                                  LAST_SIGNAL);

    klass->debugger_set = NULL;
	object_class->destroy = variable_viewer_destroy;
    object_class->get_arg = get_arg;
    object_class->set_arg = set_arg;
}

void
variable_viewer_init (GdfVariableViewer *vv)
{

    vv->priv = g_new0 (GdfVariableViewerPriv, 1);

	vv->priv->dbg = NULL;
    vv->priv->ss = NULL;
    vv->priv->nodes = NULL;
    vv->priv->view_locals = FALSE;

	create_children (vv);
}

void 
variable_viewer_destroy (GtkObject *obj)
{
	GdfVariableViewer *vv = GDF_VARIABLE_VIEWER (obj);
	
	g_return_if_fail (obj != NULL);
	
	if (vv->priv->dbg)
		gtk_object_unref (GTK_OBJECT (vv->priv->dbg));
    if (vv->priv->ss)
        gtk_object_unref (GTK_OBJECT (vv->priv->ss));

    if (vv->priv->nodes)
        g_tree_destroy (vv->priv->nodes);

	g_free (vv->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (obj);
}

void 
get_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfVariableViewer *vv = GDF_VARIABLE_VIEWER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        GTK_VALUE_OBJECT (*arg) = GTK_OBJECT (vv->priv->dbg);
        break;
    case ARG_SHOW_ENTRY :
        GTK_VALUE_BOOL (*arg) = GTK_WIDGET_VISIBLE (vv->priv->entry_hbox);
        break;
    default :
        arg->type = GTK_TYPE_INVALID;
    }   
}

void 
set_arg (GtkObject *object, 
         GtkArg *arg, 
         guint arg_id)
{
    GdfVariableViewer *vv = GDF_VARIABLE_VIEWER (object);
    
    switch (arg_id) {
    case ARG_DEBUGGER :
        gdf_variable_viewer_set_debugger (vv, GTK_VALUE_OBJECT (*arg));
        break;
    case ARG_SHOW_ENTRY :
        if (GTK_VALUE_BOOL (*arg)) {
            gtk_widget_show (vv->priv->entry_hbox);
        } else {
            gtk_widget_hide (vv->priv->entry_hbox);
        }
        break;
    default :
        break;
    }   
}

/* 
 * Helper functions
 */
void
create_children (GdfVariableViewer *vv)
{
	static gchar *titles[] = {
		N_("Name"),
		N_("Value")
	};
    GtkWidget *vbox;

	vv->priv->ctree = gtk_ctree_new_with_titles (2, 0, titles);
	
    gtk_signal_connect (GTK_OBJECT (vv->priv->ctree), "tree_expand",
                        GTK_SIGNAL_FUNC (tree_expand_cb), vv);

    gtk_clist_set_column_width (GTK_CLIST (vv->priv->ctree), 0, 150);

	vv->priv->scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (vv->priv->scrolled),
									GTK_POLICY_AUTOMATIC,
									GTK_POLICY_AUTOMATIC);	

    vv->priv->entry = gnome_entry_new ("variable-viewer");
    gtk_signal_connect (GTK_OBJECT (gnome_entry_gtk_entry (GNOME_ENTRY (vv->priv->entry))), 
                        "activate", GTK_SIGNAL_FUNC (entry_activate_cb), vv);
    vv->priv->add_btn = gtk_button_new_with_label (_("Add"));
    gtk_signal_connect (GTK_OBJECT (vv->priv->add_btn), "clicked", 
                        GTK_SIGNAL_FUNC (add_btn_clicked_cb), vv);

    vbox = gtk_vbox_new (FALSE, 0);
    vv->priv->entry_hbox = gtk_hbox_new (FALSE, 0);

    gtk_box_pack_start (GTK_BOX (vv->priv->entry_hbox), 
                        vv->priv->entry, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (vv->priv->entry_hbox), 
                        vv->priv->add_btn, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (vv->priv->scrolled), vv->priv->ctree);
    gtk_box_pack_start (GTK_BOX (vbox), vv->priv->entry_hbox, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), vv->priv->scrolled, TRUE, TRUE, 0); 
	gtk_container_add (GTK_CONTAINER (vv), vbox);
    
    gtk_widget_show (vv->priv->entry);
    gtk_widget_show (vv->priv->add_btn);
    gtk_widget_show (vv->priv->entry_hbox);
    gtk_widget_show (vbox);
	gtk_widget_show (vv->priv->ctree);
	gtk_widget_show (vv->priv->scrolled);
}

void
add_dummy_child (GdfVariableViewer *vv, GtkCTreeNode *parent)
{       
    GtkCTreeNode *node;
    char *text[2];
    text[0] = "";
    text[1] = "";
    
    node = gtk_ctree_insert_node (GTK_CTREE (vv->priv->ctree), parent, 
                                  NULL, text, 0, NULL, NULL, NULL, NULL, 
                                  FALSE, FALSE);
    gtk_ctree_node_set_row_data (GTK_CTREE (vv->priv->ctree), node, 
                                 GINT_TO_POINTER (-1));
}

void
add_node (GdfVariableViewer *vv, GtkCTreeNode *parent, GDF_Symbol *sym)
{
    GtkCTreeNode *node;
    char *text[2];

    text[0] = sym->name;
    text[1] = sym->value;
    
    node = gtk_ctree_insert_node (GTK_CTREE (vv->priv->ctree), parent, NULL, 
                                  text, 0, NULL, NULL, NULL, NULL, 
                                  FALSE, 
                                  FALSE);
    g_tree_insert (vv->priv->nodes, GINT_TO_POINTER (sym->handle), node);
    gtk_ctree_node_set_row_data (GTK_CTREE (vv->priv->ctree), node, 
                                 GINT_TO_POINTER (sym->handle));
    if (sym->expandable) {
        add_dummy_child (vv, node);
    }
}

void 
clear_variable_viewer (GdfVariableViewer *vv)
{
	gtk_clist_clear (GTK_CLIST (vv->priv->ctree));
    if (vv->priv->nodes) {
        g_tree_destroy (vv->priv->nodes);
        vv->priv->nodes = NULL;
    }
}

static void
symbol_set_changed_cb (GdfSymbolSetClient *ss, GdfVariableViewer *vv)
{
    GDF_SymbolSequence *syms;
    int i;

    gdf_variable_viewer_clear (vv);

    syms = gdf_symbol_set_client_get_symbols (ss);
    
    if (syms->_length > 0)
        vv->priv->nodes = g_tree_new (int_cmp);
    else 
        vv->priv->nodes = NULL;

    for (i = 0; i < syms->_length; i++)
        gdf_variable_viewer_view_symbol (vv, &syms->_buffer[i]);

    CORBA_free (syms);
}

static void
symbol_changed_cb (GdfSymbolSetClient *ss, int handle, GdfVariableViewer *vv)
{
    GDF_Symbol *sym;
    GtkCTreeNode *node;
    GtkCTreeNode *child;
    GtkCTreeRow *row;

    sym = gdf_symbol_set_client_get_symbol (ss, handle);

    node = g_tree_lookup (vv->priv->nodes, GINT_TO_POINTER (handle));
    row = GTK_CTREE_ROW (node);
    gtk_ctree_node_set_text (GTK_CTREE (vv->priv->ctree),
                             node, 1, sym->value);

    child = GTK_CTREE_ROW (node)->children;
    if (sym->expandable && !child) {
        add_dummy_child (vv, node);
    }

    CORBA_free (sym);
}

void
connect_debugger_signals (GdfVariableViewer *vv)
{
	vv->priv->program_unloaded_sig = 
		gtk_signal_connect_object (GTK_OBJECT (vv->priv->dbg), 
								   "program_unloaded",
								   GTK_SIGNAL_FUNC (clear_variable_viewer),
								   (gpointer)vv);

}

void
disconnect_debugger_signals (GdfVariableViewer *vv)
{
	gtk_signal_disconnect (GTK_OBJECT (vv->priv->dbg), 
						   vv->priv->program_unloaded_sig);
}

void 
connect_symbol_set_signals (GdfVariableViewer *vv)
{
    vv->priv->symbol_changed_sig =
        gtk_signal_connect (GTK_OBJECT (vv->priv->ss),
                            "symbol_changed",
                            GTK_SIGNAL_FUNC (symbol_changed_cb),
                            (gpointer)vv);
    vv->priv->symbol_set_changed_sig =
        gtk_signal_connect (GTK_OBJECT (vv->priv->ss),
                            "symbol_set_changed",
                            GTK_SIGNAL_FUNC (symbol_set_changed_cb),
                            (gpointer)vv);
}

void
disconnect_symbol_set_signals (GdfVariableViewer *vv)
{
	gtk_signal_disconnect (GTK_OBJECT (vv->priv->ss), 
						   vv->priv->symbol_changed_sig);
    gtk_signal_disconnect (GTK_OBJECT (vv->priv->ss),
                           vv->priv->symbol_set_changed_sig);
}

static void
add_current_expression (GdfVariableViewer *vv)
{ 
    char *text;
    
    text = 
        gtk_entry_get_text (GTK_ENTRY (gnome_entry_gtk_entry (GNOME_ENTRY (vv->priv->entry))));
    
    if (strcmp (text, "")) {
        gdf_symbol_set_client_add_expression (vv->priv->ss, 
                                              text);
    }

    gtk_entry_set_text (GTK_ENTRY (gnome_entry_gtk_entry (GNOME_ENTRY (vv->priv->entry))), "");
}


void
add_btn_clicked_cb (GtkWidget *btn, gpointer data)
{
    add_current_expression (GDF_VARIABLE_VIEWER (data));
}

void
entry_activate_cb (GtkWidget *editable, gpointer data)
{
    add_current_expression (GDF_VARIABLE_VIEWER (data));
}

void
tree_expand_cb (GtkWidget *ctree, GtkCTreeNode *node, gpointer data)
{
    GdfVariableViewer *vv = GDF_VARIABLE_VIEWER (data);
    GtkCTreeNode *child;
    int sym_id;

    child = GTK_CTREE_ROW (node)->children;
    if (!child)
        return;
    
    sym_id = GPOINTER_TO_INT (gtk_ctree_node_get_row_data (GTK_CTREE (ctree),
                                                           child));
    
    /* if this is a dummy node, sym_id will be -1 */
    if (sym_id == -1) {
        int i;        
        GDF_SymbolSequence *seq;
        sym_id = 
            GPOINTER_TO_INT (gtk_ctree_node_get_row_data (GTK_CTREE (ctree), 
                                                          node));
        seq = gdf_symbol_set_client_get_symbol_children (vv->priv->ss, sym_id);

        /* Remove the old/dummy children */
        child = GTK_CTREE_ROW (node)->children;
        while (child) {
            GtkCTreeNode *next;
            next = GTK_CTREE_ROW (child)->sibling;
            gtk_ctree_remove_node (GTK_CTREE (vv->priv->ctree), child);
            child = next;
        }

        for (i = 0; i < seq->_length; i++) {
            add_node (vv, node, &(seq->_buffer[i]));
        }

        CORBA_free (seq);
    }
}

gint
int_cmp (gconstpointer a, gconstpointer b) 
{
    return (GPOINTER_TO_INT (a) < GPOINTER_TO_INT (b) 
            ? -1 
            : (GPOINTER_TO_INT (a) == GPOINTER_TO_INT (b) ? 0 : 1));
}
