/* GNOME-DB Components
 * Copyright (C) 2000-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <libgda/gda-config.h>
#include <bonobo/bonobo-i18n.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtknotebook.h>
#include <libgnomedb/gnome-db-dsn-config.h>
#include <libgnomedb/gnome-db-grid.h>
#include "components-manager-view.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _ComponentsManagerViewPrivate {
	GtkWidget *notebook;

	/* 'Data Sources' tab */
	GtkWidget *dsn_tab;
	GtkWidget *dsn_list;

	/* 'Providers' tab */
	GtkWidget *providers_tab;
	GtkWidget *providers_list;

	/* configuration listener */
	guint config_listener;
};

static void components_manager_view_class_init (ComponentsManagerViewClass *klass);
static void components_manager_view_init       (ComponentsManagerView *manager,
						ComponentsManagerViewClass *klass);
static void components_manager_view_finalize   (GObject *object);

static GObjectClass *parent_class = NULL;

/*
 * Private functions
 */

static void
show_data_sources (ComponentsManagerView *view)
{
	GdaDataModel *model;

	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	model = gda_config_get_data_source_model ();
	gnome_db_grid_set_model (GNOME_DB_GRID (view->priv->dsn_list), model);

	g_object_unref (G_OBJECT (model));
}

static void
show_providers (ComponentsManagerView *view)
{
	GList *list;
	GList *l;
	GdaDataModel *model;

	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	list = gda_config_get_provider_list ();

	/* FIXME: implement */

	/* free memory */
	gda_config_free_provider_list (list);
}

/*
 * Callbacks
 */

static void
config_listener_cb (const gchar *path, gpointer user_data)
{
	ComponentsManagerView *view = (ComponentsManagerView *) user_data;

	g_return_if_fail (path != NULL);
	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	if (!strncmp (path, GDA_CONFIG_SECTION_DATASOURCES, strlen (GDA_CONFIG_SECTION_DATASOURCES))) {
		show_data_sources (view);
	}
}

/*
 * ComponentsManagerView class implementation
 */

static void
components_manager_view_class_init (ComponentsManagerViewClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = components_manager_view_finalize;
}

static void
components_manager_view_init (ComponentsManagerView *view, ComponentsManagerViewClass *klass)
{
	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	/* allocate private structure */
	view->priv = g_new0 (ComponentsManagerViewPrivate, 1);
	view->priv->config_listener = gda_config_add_listener (
		GDA_CONFIG_SECTION_DATASOURCES,
		(GdaConfigListenerFunc) config_listener_cb,
		view);

	/* setup widgets */
	view->priv->notebook = gnome_db_new_notebook_widget ();
	gtk_box_pack_start (GTK_BOX (view), view->priv->notebook, TRUE, TRUE, 0);

	/* dsn tab */
	view->priv->dsn_tab = gnome_db_new_hbox_widget ();
	gtk_notebook_append_page (GTK_NOTEBOOK (view->priv->notebook),
				  view->priv->dsn_tab, gtk_label_new (_("Data Sources")));

	view->priv->dsn_list = gnome_db_grid_new ();
	gtk_widget_show (view->priv->dsn_list);
	show_data_sources (view);
	gtk_box_pack_start (GTK_BOX (view->priv->dsn_tab), view->priv->dsn_list, TRUE, TRUE, 0);

	/* providers tab */
	view->priv->providers_tab = gnome_db_new_hbox_widget ();
	gtk_notebook_append_page (GTK_NOTEBOOK (view->priv->notebook),
				  view->priv->providers_tab,
				  gtk_label_new (_("Providers")));

	view->priv->providers_list = gnome_db_grid_new ();
	gtk_widget_show (view->priv->providers_list);
	show_providers (view);
	gtk_box_pack_start (GTK_BOX (view->priv->providers_tab),
			    view->priv->providers_list, TRUE, TRUE, 0);
}

static void
components_manager_view_finalize (GObject *object)
{
	ComponentsManagerView *view = (ComponentsManagerView *) object;

	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	/* free memory */
	g_free (view->priv);
	view->priv = NULL;

	parent_class->finalize (object);
}

GType
components_manager_view_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (ComponentsManagerViewClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) components_manager_view_class_init,
			NULL,
			NULL,
			sizeof (ComponentsManagerView),
			0,
			(GInstanceInitFunc) components_manager_view_init
		};
		type = g_type_register_static (GTK_TYPE_VBOX, "ComponentsManagerView", &info, 0);
	}

	return type;
}

GtkWidget *
components_manager_view_new (void)
{
	ComponentsManagerView *view;

	view = g_object_new (COMPONENTS_TYPE_MANAGER_VIEW, NULL);
	return GTK_WIDGET (view);
}

void
components_manager_view_show_properties (ComponentsManagerView *view)
{
	GtkWidget *dialog;
	GtkWidget *props;
	GdaDataSourceInfo *dsn_info;
	GList *selection;
	gchar *str;
	GdaDataModel *model;
	gint current;

	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (view->priv->notebook));

	/* retrieve selection */
	selection = gnome_db_grid_get_selection (GNOME_DB_GRID (view->priv->dsn_list));
	if (!selection)
		return;

	model = gnome_db_grid_get_model (GNOME_DB_GRID (view->priv->dsn_list));
	str = gda_value_stringify (
		gda_data_model_get_value_at (model, 0, GPOINTER_TO_INT (selection->data)));

	dsn_info = gda_config_find_data_source (str);
	g_free (str);
	if (!dsn_info)
		return;

	/* show the properties */
	dialog = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (dialog), _("Properties"));
	gtk_dialog_add_button (GTK_DIALOG (dialog), _("Cancel"), GTK_RESPONSE_CANCEL);
	gtk_dialog_add_button (GTK_DIALOG (dialog), _("Save"), GTK_RESPONSE_OK);

	props = gnome_db_dsn_config_new ();
	gnome_db_dsn_config_set_info (GNOME_DB_DSN_CONFIG (props), dsn_info);
	gtk_widget_show (props);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), props, TRUE, TRUE, 0);

	gda_config_free_data_source_info (dsn_info);

	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK) {
		dsn_info = gnome_db_dsn_config_get_info (GNOME_DB_DSN_CONFIG (props));
		if (dsn_info) {
			gda_config_save_data_source (dsn_info->name,
						     dsn_info->provider,
						     dsn_info->cnc_string,
						     dsn_info->description,
						     dsn_info->username);
			gda_config_free_data_source_info (dsn_info);
		}
	}

	gtk_widget_destroy (dialog);
}

void
components_manager_view_select_all (ComponentsManagerView *view)
{
	gint current;

	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (view->priv->notebook));
	switch (current) {
	case 0 : /* data sources */
		gnome_db_grid_select_all (GNOME_DB_GRID (view->priv->dsn_list));
		break;
	}
}

void
components_manager_view_unselect_all (ComponentsManagerView *view)
{
	gint current;

	g_return_if_fail (COMPONENTS_IS_MANAGER_VIEW (view));

	current = gtk_notebook_get_current_page (GTK_NOTEBOOK (view->priv->notebook));
	switch (current) {
	case 0 : /* data sources */
		gnome_db_grid_unselect_all (GNOME_DB_GRID (view->priv->dsn_list));
		break;
	}
}
