/* GNOME-DB Manager component
 * Copyright (C) 1998,1999 Michael Lausch
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gda-config.h>
#include "gnomedb-mgr.h"
#if defined(HAVE_GAL)
#  include <gal/e-paned/e-hpaned.h>
#endif

typedef struct {
	GtkWidget* notebook;
	GtkWidget* dsn_tab;
	GtkWidget* dsn_list;
	GtkWidget* dsn_config;
} MANAGER_DsnInfo;

static void add_dsn_cb      (GtkWidget *w, MANAGER_DsnInfo *dsn_info);
static void refresh_list_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info);
static void remove_dsn_cb   (GtkWidget *w, MANAGER_DsnInfo *dsn_info);
static void test_dsn_cb     (GtkWidget *w, MANAGER_DsnInfo *dsn_info);

static GnomeUIInfo dsn_toolbar[] = {
	{ GNOME_APP_UI_ITEM, N_("Add"), N_("Add new data source"),
	  add_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_NEW, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Remove"), N_("Remove current data source"),
	  remove_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Refresh"), N_("Refresh Data Sources list"),
	  refresh_list_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
	{ GNOME_APP_UI_ITEM, N_("Test"), N_("Test settings for this datasource"),
	  test_dsn_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
	  GNOME_STOCK_MENU_CONVERT, 0, 0, NULL },
	GNOMEUIINFO_END
};

/*
 * Private functions
 */
void
fill_dsn_list (GtkCList *clist) {
	GList* dsns;
	GList* node;
	
	g_return_if_fail(GTK_IS_CLIST(clist));
	
	gnome_db_clear_clist(clist);
	dsns = gda_dsn_list();
	gtk_clist_freeze(clist);
	for (node = g_list_first(dsns); node != NULL; node = g_list_next(node)) {
		Gda_Dsn* dsn = (Gda_Dsn *) node->data;
		
		if (dsn) {
			gchar* row[2] = { "", "" };
			row[1] = GDA_DSN_GDA_NAME(dsn);
			gtk_clist_append(clist, row);
		}
	}
	gtk_clist_thaw(clist);
	gda_dsn_free_list(dsns);
}

/*
 * Callbacks
 */
static void
add_dsn_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info) {
	g_return_if_fail(dsn_info != NULL);
	g_return_if_fail(GNOME_DB_IS_DSN_CONFIG(dsn_info->dsn_config));
	
	gnome_db_dsn_config_save(GNOME_DB_DSN_CONFIG(dsn_info->dsn_config));
	fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
}

static void
clist_selection_cb (GtkCList *clist, gint row, gint col, GdkEvent *event, MANAGER_DsnInfo *dsn_info) {
	gchar* txt = NULL;
	
	g_return_if_fail(dsn_info != NULL);
	g_return_if_fail(GTK_IS_CLIST(dsn_info->dsn_list));
	g_return_if_fail(GNOME_DB_IS_DSN_CONFIG(dsn_info->dsn_config));
	
	gnome_db_dsn_config_clear(GNOME_DB_DSN_CONFIG(dsn_info->dsn_config));
	if (gtk_clist_get_text(GTK_CLIST(dsn_info->dsn_list), row, 1, &txt)) {
		gnome_db_dsn_config_set_name(GNOME_DB_DSN_CONFIG(dsn_info->dsn_config), txt);
	}
}

static void
refresh_list_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info) {
	g_return_if_fail(dsn_info != NULL);
	
	gnome_db_dsn_config_clear(GNOME_DB_DSN_CONFIG(dsn_info->dsn_config));
	fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
}

static void
remove_dsn_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info) {
	g_return_if_fail(dsn_info != NULL);
	g_return_if_fail(GNOME_DB_IS_DSN_CONFIG(dsn_info->dsn_config));
	
	gnome_db_dsn_config_remove(GNOME_DB_DSN_CONFIG(dsn_info->dsn_config));
	fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
}

static void
test_dsn_cb (GtkWidget *w, MANAGER_DsnInfo *dsn_info) {
	gchar* gda_name;
	gchar* username;
	
	g_return_if_fail(dsn_info != NULL);
	g_return_if_fail(GNOME_DB_IS_DSN_CONFIG(dsn_info->dsn_config));
	
	gda_name = gnome_db_dsn_config_get_name(GNOME_DB_DSN_CONFIG(dsn_info->dsn_config));
	if (gda_name) {
		Gda_Dsn* dsn = gda_dsn_find_by_name(gda_name);
		if (dsn) {
			GtkWidget* dialog;
			GtkWidget* frame;
			GtkWidget* table;
			GtkWidget* label;
			GtkWidget* gda_entry;
			GtkWidget* user_entry;
			GtkWidget* password_entry;
			
			/* create login dialog */
			dialog = gnome_dialog_new(gda_name,
			                          GNOME_STOCK_BUTTON_OK,
			                          GNOME_STOCK_BUTTON_CANCEL,
			                          NULL);
			frame = gnome_db_new_frame_widget(NULL);
			gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
			table = gnome_db_new_table_widget(2, 3, FALSE);
			gtk_container_add(GTK_CONTAINER(frame), table);
			
			label = gnome_db_new_label_widget(_("GDA Name"));
			gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
			gda_entry = gnome_db_new_entry_widget(0, FALSE);
			gtk_entry_set_text(GTK_ENTRY(gda_entry), gda_name);
			gtk_table_attach(GTK_TABLE(table), gda_entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
			
			label = gnome_db_new_label_widget(_("Username"));
			gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
			user_entry = gnome_db_new_entry_widget(0, TRUE);
			gtk_entry_set_text(GTK_ENTRY(user_entry), GDA_DSN_USERNAME(dsn));
			gtk_table_attach(GTK_TABLE(table), user_entry, 1, 2, 1, 2, GTK_FILL, GTK_FILL, 3, 3);
			
			label = gnome_db_new_label_widget(_("Password"));
			gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
			password_entry = gnome_db_new_entry_widget(0, TRUE);
			gtk_entry_set_visibility(GTK_ENTRY(password_entry), FALSE);
			gtk_table_attach(GTK_TABLE(table), password_entry, 1, 2, 2, 3, GTK_FILL, GTK_FILL, 3, 3);
			
			/* run the dialog */
			if (!gnome_dialog_run(GNOME_DIALOG(dialog))) {
				Gda_Connection* cnc = gda_connection_new(gda_corba_get_orb());
				
				gda_connection_set_provider(cnc, GDA_DSN_PROVIDER(dsn));
				if (gda_connection_open(cnc,
				                        GDA_DSN_DSN(dsn),
				                        gtk_entry_get_text(GTK_ENTRY(user_entry)),
				                        gtk_entry_get_text(GTK_ENTRY(password_entry))) < 0) {
					if (cnc->errors_head) {
						gnome_db_show_error(gda_error_description((Gda_Error *) cnc->errors_head->data));
					}
					else gnome_db_show_error(_("Test failed"));
				}
				else gnome_db_show_message(_("Connection successful"));
				
				gda_connection_close(cnc);
				gda_connection_free(cnc);
			}
			
			gnome_dialog_close(GNOME_DIALOG(dialog));
			gda_dsn_free(dsn);
		}
		else gnome_db_show_error(_("Datasource %s not found in configuration"), gda_name);		
	}
}

/*
 * Public functions
 */
void
manager_dsn_init (GtkNotebook *notebook) {
	MANAGER_DsnInfo* dsn_info;
	GtkWidget*       scroll;
	GtkWidget*       button;
	GtkWidget*       toolbar;
	
	dsn_info = g_new0(MANAGER_DsnInfo, 1);
	dsn_info->notebook = notebook;
	dsn_info->dsn_tab = gnome_db_new_table_widget(4, 2, FALSE);
	
	scroll = gnome_db_new_scrolled_window_widget();
	gtk_table_attach(GTK_TABLE(dsn_info->dsn_tab), scroll, 0, 1, 1, 2,
	                 GTK_FILL | GTK_EXPAND, GTK_FILL, 3, 3);
	dsn_info->dsn_list = gnome_db_new_clist_widget(NULL, 2);
	gtk_signal_connect(GTK_OBJECT(dsn_info->dsn_list),
	                   "select_row",
	                   GTK_SIGNAL_FUNC(clist_selection_cb),
	                   (gpointer) dsn_info);
	gtk_signal_connect(GTK_OBJECT(dsn_info->dsn_list),
	                   "unselect_row",
	                   GTK_SIGNAL_FUNC(clist_selection_cb),
	                   (gpointer) dsn_info);
	fill_dsn_list(GTK_CLIST(dsn_info->dsn_list));
	gtk_container_add(GTK_CONTAINER(scroll), dsn_info->dsn_list);
	
	dsn_info->dsn_config = gnome_db_dsn_config_new(NULL);
	gtk_widget_show(dsn_info->dsn_config);
	gtk_table_attach(GTK_TABLE(dsn_info->dsn_tab), dsn_info->dsn_config, 1, 4, 1, 2,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 3, 3);
	                 
	toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
	                                      GTK_TOOLBAR_ICONS,
	                                      dsn_toolbar,
	                                      (gpointer) dsn_info);
	gtk_table_attach(GTK_TABLE(dsn_info->dsn_tab), toolbar, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
  
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dsn_info->dsn_tab, gtk_label_new(_("Data Sources")));
}

