/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome-db-browser.h>
#include <gnome-db-grid.h>
#include <gnome-db-list.h>

struct _GnomeDbBrowserPrivate {
	Gda_Connection* cnc;
	GtkWidget*      notebook;
};

typedef struct _object_data_t {
	gchar                label[16];
	GDA_Connection_QType qtype;
	GtkWidget*           (*create_detail)(void);
	void                 (*update_detail)(GnomeDbBrowser *browser, const gchar *name, GtkWidget *container);
} object_data_t;

static GtkWidget* tables_create_detail (void);
static void       tables_update_detail (GnomeDbBrowser *browser, const gchar *name, GtkWidget *container);

enum
{
  CONNECTION_CHANGED,
  LAST_SIGNAL
};

static gint db_browser_signals[LAST_SIGNAL] = { 0, };
static object_data_t objects[] = {
	{ N_("Tables"), GDA_Connection_GDCN_SCHEMA_TABLES, tables_create_detail, tables_update_detail },
	{ N_("Views"), GDA_Connection_GDCN_SCHEMA_VIEWS, tables_create_detail, tables_update_detail },
	{ N_("Types"), GDA_Connection_GDCN_SCHEMA_PROV_TYPES, NULL, NULL },
};

/*
 * Private functions
 */
static void
show_object_list (GnomeDbBrowser* browser, GtkWidget *container) {
	object_data_t* obj_data;
	
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(GTK_IS_WIDGET(container));
	
	obj_data = (object_data_t *) gtk_object_get_data(GTK_OBJECT(container),
		                                             "GNOME_DB_Browser_ObjectData");
	if (obj_data) {
		GtkWidget* list = gtk_object_get_data(GTK_OBJECT(container), "GNOME_DB_Browser_ListWidget");
		if (GNOME_DB_IS_LIST(list)) {
			Gda_Recordset* recset;
			
			recset = gda_connection_open_schema(gnome_db_browser_get_connection(browser),
			                                    obj_data->qtype,
			                                    GDA_Connection_no_CONSTRAINT,
			                                    0);
			gnome_db_list_set_recordset(GNOME_DB_LIST(list), recset, 0);
		}
	}
}

static GtkWidget *
tables_create_detail (void) {
	GtkWidget* table;
	GtkWidget* grid;
	
	table = gnome_db_new_table_widget(3, 2, FALSE);
	
	grid = gnome_db_grid_new(NULL);
	gtk_object_set_data(GTK_OBJECT(table), "GNOME_DB_Browser_Tables_GridWidget", (gpointer) grid);
	gtk_widget_show(grid);
	gtk_table_attach(GTK_TABLE(table), grid, 0, 3, 1, 2,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 3, 3);
	
	return table;
}

static void
tables_update_detail (GnomeDbBrowser *browser, const gchar *name, GtkWidget *container) {
	GtkWidget* detail_pane;
	GtkWidget* grid;
	
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(name != NULL);
	g_return_if_fail(GTK_IS_WIDGET(container));
	g_return_if_fail(gda_connection_is_open(gnome_db_browser_get_connection(browser)));
	
	detail_pane = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(container), "GNOME_DB_Browser_DetailPane"));
	grid = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(detail_pane), "GNOME_DB_Browser_Tables_GridWidget"));
	if (GNOME_DB_IS_GRID(grid)) {
		Gda_Recordset* recset;
		
		recset = gda_connection_open_schema(gnome_db_browser_get_connection(browser),
		                                    GDA_Connection_GDCN_SCHEMA_COLS,
		                                    GDA_Connection_OBJECT_NAME,
		                                    name,
		                                    GDA_Connection_no_CONSTRAINT);
		gnome_db_grid_set_recordset(GNOME_DB_GRID(grid), recset);
	}
}

/**
 * Callbacks
 */
static void
object_selected_cb (GnomeDbList *dblist, GtkWidget *container) {
	GnomeDbBrowser* browser;
	
	g_return_if_fail(GNOME_DB_IS_LIST(dblist));
	g_return_if_fail(GTK_IS_WIDGET(container));
	
	browser = GNOME_DB_BROWSER(gtk_object_get_data(GTK_OBJECT(container), "GNOME_DB_Browser_BrowserWidget"));
	if (GNOME_DB_IS_BROWSER(browser)) {
		object_data_t* obj_data = (object_data_t *) gtk_object_get_data(GTK_OBJECT(container),
		                                                                "GNOME_DB_Browser_ObjectData"); 
		if (obj_data && obj_data->update_detail) {
			obj_data->update_detail(browser,
			                        gnome_db_list_get_string(GNOME_DB_LIST(dblist)),
			                        container);
		}
	}
}

/*
 * GnomeDbBrowser interface
 */
static void
_gnome_db_browser_class_init (GnomeDbBrowserClass *klass) {
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	
	db_browser_signals[CONNECTION_CHANGED] = gtk_signal_new("connection_changed",
	                                                        GTK_RUN_LAST,
	                                                        object_class->type,
	                                                        GTK_SIGNAL_OFFSET(GnomeDbBrowserClass, connection_changed),
	                                                        gtk_signal_default_marshaller,
	                                                        GTK_TYPE_NONE, 0);
	gtk_object_class_add_signals(object_class, db_browser_signals, LAST_SIGNAL);
	klass->connection_changed = 0;
}

static void
_gnome_db_browser_destroy (GnomeDbBrowser *browser, gpointer data) {
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	
	if (browser->priv) {
		g_free((gpointer) browser->priv);
	}
}

static void
_gnome_db_browser_init (GnomeDbBrowser *browser) {
	gint cnt;
	
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	
	/* create private data */
	browser->priv = g_new0(GnomeDbBrowserPrivate, 1);
	gtk_signal_connect(GTK_OBJECT(browser),
	                   "destroy",
	                   GTK_SIGNAL_FUNC(_gnome_db_browser_destroy),
	                   NULL);

	/* create widgets */	
	browser->priv->notebook = gnome_db_new_notebook_widget();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(browser->priv->notebook), GTK_POS_LEFT);
	gtk_box_pack_start(GTK_BOX(browser), browser->priv->notebook, 1, 1, 0);
	
	/* add tabs */
	for (cnt = 0; cnt < sizeof(objects) / sizeof(objects[0]); cnt++) {
		GtkWidget* list;
		GtkWidget* container;
		GtkWidget* detail;

		container = gnome_db_new_table_widget(4, 1, TRUE);
		gtk_widget_show(container);
		gtk_object_set_data(GTK_OBJECT(container), "GNOME_DB_Browser_BrowserWidget", (gpointer) browser);
		gtk_object_set_data(GTK_OBJECT(container), "GNOME_DB_Browser_ObjectData", (gpointer) &objects[cnt]);
		list = gnome_db_list_new(NULL, 0);
		gtk_signal_connect(GTK_OBJECT(list),
		                   "select_row",
		                   GTK_SIGNAL_FUNC(object_selected_cb),
		                   (gpointer) container);
		gtk_object_set_data(GTK_OBJECT(container), "GNOME_DB_Browser_ListWidget", (gpointer) list);
		gtk_widget_show(list);
		gtk_table_attach(GTK_TABLE(container), list, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

		if (objects[cnt].create_detail) {
			detail = objects[cnt].create_detail();
			gtk_object_set_data(GTK_OBJECT(container), "GNOME_DB_Browser_DetailPane", (gpointer) detail);
			gtk_widget_show(detail);
			gtk_table_attach(GTK_OBJECT(container), detail, 1, 4, 0, 1,
			                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			                 3, 3);
		}

		gtk_notebook_append_page(GTK_NOTEBOOK(browser->priv->notebook),
		                         container,
		                         gtk_label_new(objects[cnt].label));
		show_object_list(browser, container);
	}
}

GtkType
gnome_db_browser_get_type (void) {
	static GtkType db_browser_type = 0;
	
	if (!db_browser_type) {
		GtkTypeInfo db_browser_info = {
			"GnomeDbBrowser",
			sizeof (GnomeDbBrowser),
			sizeof (GnomeDbBrowserClass),
			(GtkClassInitFunc) _gnome_db_browser_class_init,
			(GtkObjectInitFunc) _gnome_db_browser_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_browser_type = gtk_type_unique(gtk_vbox_get_type(), &db_browser_info);
	}
	return (db_browser_type);
}

/**
 * gnome_db_browser_new
 */
GtkWidget *
gnome_db_browser_new (Gda_Connection *cnc) {
	GnomeDbBrowser* browser;
	
	browser = GNOME_DB_BROWSER(gtk_type_new(gnome_db_browser_get_type()));
	if (IS_GDA_CONNECTION(cnc)) {
		gnome_db_browser_set_connection(browser, cnc);
	}
	return GTK_WIDGET(browser);
}

/**
 * gnome_db_browser_get_connection
 */
Gda_Connection *
gnome_db_browser_get_connection (GnomeDbBrowser *browser) {
	g_return_val_if_fail(GNOME_DB_IS_BROWSER(browser), NULL);
	g_return_val_if_fail(browser->priv != NULL, NULL);
	return browser->priv->cnc;
}

/**
 * gnome_db_browser_set_connection
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *browser, Gda_Connection *cnc) {
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(browser->priv != NULL);
	
	browser->priv->cnc = cnc;
	gnome_db_browser_refresh(browser);
	gtk_signal_emit(GTK_OBJECT(browser), db_browser_signals[CONNECTION_CHANGED]);
}

/**
 * gnome_db_browser_refresh
 */
void
gnome_db_browser_refresh (GnomeDbBrowser *browser) {
	GList* node;
	gint   cnt;
	
	g_return_if_fail(GNOME_DB_IS_BROWSER(browser));
	g_return_if_fail(browser->priv != NULL);
	g_return_if_fail(GTK_IS_NOTEBOOK(browser->priv->notebook));
	
	for (cnt = 0; cnt < g_list_length(GTK_NOTEBOOK(browser->priv->notebook)->children); cnt++) {
		show_object_list(browser, gtk_notebook_get_nth_page(GTK_NOTEBOOK(browser->priv->notebook),
		                                                    cnt));
	}
}

