/* GNOME DB libary
 * Copyright (C) 1999,2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(HAVE_CONFIG_H)
#  include <config.h>
#endif
#include "gnome-db.h"

/*
 * Functions
 */
GtkWidget *
gnome_db_new_button_widget (gchar *label)
{
  GtkWidget *button = label ? gtk_button_new_with_label(label) : gtk_button_new();
  gtk_widget_show(button);
  return (button);
}

GtkWidget *
gnome_db_new_button_widget_with_pixmap (const gchar *label, const gchar *icon)
{
  GtkWidget *button;
  
  g_return_val_if_fail(label != 0, 0);
  g_return_val_if_fail(icon != 0, 0);
  
  button = gnome_pixmap_button(gnome_stock_new_with_icon(icon), label);
  gtk_widget_show(button);
  return (button);
}

GtkWidget *
gnome_db_new_check_button_widget (gchar *label, gint state)
{
  GtkWidget *check = label ? gtk_check_button_new_with_label(label) : gtk_check_button_new();
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(check), state);
  gtk_widget_show(check);
  return check;
}

static void
clist_column_clicked_cb (GtkCList *clist, gint col, gpointer data)
{
  static gint        last_col = -1;
  static GtkSortType last_sort = GTK_SORT_ASCENDING;

  g_return_if_fail(GTK_IS_CLIST(clist));

  gtk_clist_set_sort_column(clist, col);
  if (last_col == col)
    {
      if (last_sort == GTK_SORT_ASCENDING) last_sort = GTK_SORT_DESCENDING;
      else last_sort = GTK_SORT_ASCENDING;
    }
  else last_sort = GTK_SORT_ASCENDING;
  gtk_clist_set_sort_type(clist, last_sort);
  gtk_clist_sort(clist);
  gtk_clist_set_reorderable(clist, TRUE);

  last_col = col;
}

GtkWidget *
gnome_db_new_clist_widget (gchar *titles[], gint count)
{
  gint cnt;
  GtkWidget *clist = titles ? gtk_clist_new_with_titles(count, titles) :
                              gtk_clist_new(count);
  gtk_clist_set_selection_mode(GTK_CLIST(clist), GTK_SELECTION_SINGLE);
  for (cnt = 0; cnt < count; cnt++)
    {
      gtk_clist_set_column_resizeable(GTK_CLIST(clist), cnt, TRUE);
      gtk_clist_set_column_auto_resize(GTK_CLIST(clist), cnt, TRUE);
    }

  /* set ordering ability */
  gtk_clist_set_reorderable(GTK_CLIST(clist), TRUE);
  gtk_signal_connect(GTK_OBJECT(clist), "click_column",
                     GTK_SIGNAL_FUNC(clist_column_clicked_cb), 0);

  gtk_widget_show(clist);
  return clist;
}

GtkWidget *
gnome_db_new_color_picker_widget (void)
{
  GtkWidget* color_picker = gnome_color_picker_new();
  gnome_color_picker_set_title(GNOME_COLOR_PICKER(color_picker), _("Select Color"));
  gtk_widget_show(color_picker);
  return color_picker;
}

GtkWidget *
gnome_db_new_ctree_widget (gchar *titles[], gint count)
{
  GtkWidget *ctree = titles ? gtk_ctree_new_with_titles(count, 1, titles) :
                              gtk_ctree_new(count, 1);
  
  gtk_clist_set_selection_mode(GTK_CLIST(ctree), GTK_SELECTION_BROWSE);
  gtk_clist_set_auto_sort(GTK_CLIST(ctree), TRUE);
  gtk_clist_set_sort_type(GTK_CLIST(ctree), GTK_SORT_ASCENDING);
  gtk_ctree_set_expander_style(GTK_CTREE(ctree), GTK_CTREE_EXPANDER_TRIANGLE);
  gtk_ctree_set_line_style(GTK_CTREE(ctree), GTK_CTREE_LINES_NONE);
  gtk_widget_show(ctree);
  return ctree;
}

GtkWidget *
gnome_db_new_combo_widget (void)
{
  GtkWidget *combo;
  combo = gtk_combo_new();
  gtk_combo_set_use_arrows(GTK_COMBO(combo), 1);
  gtk_combo_set_case_sensitive(GTK_COMBO(combo), 0);
  gtk_widget_show(combo);
  return combo;
}

GtkWidget *
gnome_db_new_entry_widget (gint max_len, gboolean editable)
{
  GtkWidget *entry;
  if (max_len > 0)
    entry = gtk_entry_new_with_max_length((guint16) max_len);
  else entry = gtk_entry_new();
  gtk_entry_set_visibility(GTK_ENTRY(entry), TRUE);
  gtk_entry_set_editable(GTK_ENTRY(entry), editable);
  gtk_widget_show(entry);
  return entry;  
}

GtkWidget *
gnome_db_new_file_entry_widget (const gchar *history_id)
{
  GtkWidget* file_entry = gnome_file_entry_new(history_id, _("Select File"));
  gtk_widget_show(file_entry);
  return file_entry;
}

GtkWidget *
gnome_db_new_file_viewer_widget (const gchar *file)
{
  GtkWidget* viewer = gnome_less_new();
  gtk_widget_show(viewer);
  if (file) gnome_less_show_file(GNOME_LESS(viewer), file);
  return viewer;
}

GtkWidget *
gnome_db_new_font_picker_widget (void)
{
  GtkWidget* font_picker = gnome_font_picker_new();
  gnome_font_picker_set_title(GNOME_FONT_PICKER(font_picker), _("Select Font"));
  gnome_font_picker_set_mode(GNOME_FONT_PICKER(font_picker),
			     GNOME_FONT_PICKER_MODE_FONT_INFO);
  gtk_widget_show(font_picker);
  return font_picker;
}

GtkWidget *
gnome_db_new_frame_widget (const gchar *label)
{
  GtkWidget* frame = gtk_frame_new(label);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_OUT);
  gtk_widget_show(frame);
  return frame;
}

GtkWidget *
gnome_db_new_grid_widget (Gda_Recordset *recset)
{
  GtkWidget* grid = gnome_db_grid_new(recset);

  /* FIXME: set saved properties */
  gnome_db_grid_show_column_titles(GNOME_DB_GRID(grid));

  gtk_widget_show(grid);
  return grid;
}

GtkWidget *
gnome_db_new_icon_widget (const gchar *name)
{
  GtkWidget *icon;
  
  g_return_val_if_fail(name != 0, 0);
  
  icon = gnome_stock_new_with_icon(name);
  gtk_widget_show(icon);
  return (icon);
}

GtkWidget *
gnome_db_new_icon_list_widget (void)
{
  GtkWidget* icon_list;

  icon_list = gnome_icon_list_new(80, NULL, TRUE);
  gtk_container_set_border_width(GTK_CONTAINER(icon_list), 3);
  gnome_icon_list_set_separators(GNOME_ICON_LIST(icon_list), "/-_.");
  gnome_icon_list_set_row_spacing(GNOME_ICON_LIST(icon_list), 1);
  gnome_icon_list_set_col_spacing(GNOME_ICON_LIST(icon_list), 1);
  gnome_icon_list_set_icon_border(GNOME_ICON_LIST(icon_list), 2);
  gnome_icon_list_set_text_spacing(GNOME_ICON_LIST(icon_list), 2);
  gnome_icon_list_set_selection_mode(GNOME_ICON_LIST(icon_list), GTK_SELECTION_SINGLE);
  GTK_WIDGET_SET_FLAGS(icon_list, GTK_CAN_FOCUS);
  gtk_widget_show(icon_list);
  return icon_list;
}

GtkWidget *
gnome_db_new_label_widget (const gchar *txt)
{
  GtkWidget *label = gtk_label_new(txt);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
  gtk_widget_show(label);
  return (label);
}

GtkWidget *
gnome_db_new_menu_item_widget (GtkMenu *menu, const gchar *label)
{
  GtkWidget* menu_item = label ? gtk_menu_item_new_with_label(label) :
                                 gtk_menu_item_new();
  gtk_widget_show(menu_item);
  if (GTK_IS_MENU(menu)) gtk_menu_append(menu, menu_item);
  return menu_item;
}

GtkWidget *
gnome_db_new_notebook_widget (void)
{
  GtkWidget *notebook;
  notebook = gtk_notebook_new();
  gtk_notebook_set_scrollable(GTK_NOTEBOOK(notebook), TRUE);
  gtk_notebook_popup_enable(GTK_NOTEBOOK(notebook));
  gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), TRUE);
  gtk_widget_show(notebook);
  return (notebook);
}

GtkWidget *
gnome_db_new_option_menu_widget (void)
{
  GtkWidget* option_menu = gtk_option_menu_new();
  gtk_widget_show(option_menu);
  return option_menu;
}

GtkWidget *
gnome_db_new_progress_bar_widget (void)
{
  GtkWidget* progress_bar;
  
  progress_bar = gtk_progress_bar_new();
  gtk_progress_bar_set_orientation(GTK_PROGRESS_BAR(progress_bar), GTK_PROGRESS_LEFT_TO_RIGHT);
  gtk_progress_bar_set_bar_style(GTK_PROGRESS_BAR(progress_bar), GTK_PROGRESS_CONTINUOUS);
  gtk_widget_show(progress_bar);
  return progress_bar;
}

GtkWidget *
gnome_db_new_radio_button_widget (const gchar *label, GtkRadioButton *group)
{
  GtkWidget *radio_button;
  GSList *lst_group;
  g_return_val_if_fail(label != NULL, NULL);
  lst_group = group ? gtk_radio_button_group(group) : NULL;
  radio_button = gtk_radio_button_new_with_label(lst_group, label);
  gtk_widget_show(radio_button);
  return (radio_button);
}

GtkWidget *
gnome_db_new_scrolled_window_widget (void)
{
  GtkWidget *scrolled_window;
  scrolled_window = gtk_scrolled_window_new(0, 0);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
                                 GTK_POLICY_AUTOMATIC,
                                 GTK_POLICY_AUTOMATIC);
  gtk_widget_show(scrolled_window);
  return (scrolled_window);
}

GtkWidget *
gnome_db_new_status_bar_widget (void)
{
  GtkWidget* status;
  status = gtk_statusbar_new();
  gtk_widget_show(status);
  return status;
}

GtkWidget *
gnome_db_new_table_widget (guint cols, guint rows, gboolean homogenous)
{
  GtkWidget* table = gtk_table_new(rows, cols, homogenous);
  gtk_widget_show(table);
  return table;
}

GtkWidget *
gnome_db_new_text_widget (void)
{
  GtkWidget* text = gtk_text_new(NULL, NULL);
  gtk_text_set_editable(GTK_TEXT(text), TRUE);
  gtk_widget_show(text);
  return text;
}

GtkWidget *
gnome_db_new_tree_widget (void)
{
  GtkWidget *tree = gtk_tree_new();
  gtk_widget_show(tree);
  return (tree);
}

void
gnome_db_clear_clist (GtkCList *clist)
{
  g_return_if_fail(GTK_IS_CLIST(clist));
  gtk_clist_clear(clist);
}

GnomePixmap *
gnome_db_get_pixmap (const gchar *stock_pixmap)
{
  struct loaded_pixmap_info
  {
    gchar*       name;
    GnomePixmap* pixmap;
  };

  GList*                     node;
  struct loaded_pixmap_info* info;
  static GList*              loaded_pixmaps = NULL;

  g_return_val_if_fail(stock_pixmap != NULL, NULL);

  /* first, try to find if the pixmap is already loaded */
  node = g_list_first(loaded_pixmaps);
  while (node)
    {
      info = (struct loaded_pixmap_info *) node->data;
      if (info && !strcmp(info->name, stock_pixmap))
	return info->pixmap;
      node = g_list_next(node);
    }

  /* it's not loaded, so get it */
  info = g_new0(struct loaded_pixmap_info, 1);
  info->name = g_strdup(stock_pixmap);
  info->pixmap = gnome_stock_pixmap_widget(NULL, info->name);
  loaded_pixmaps = g_list_append(loaded_pixmaps, (gpointer) info);

  return info->pixmap;
}

GtkWidget *
gnome_db_new_popup_menu (GtkWidget *parent, GnomeUIInfo *ui_info, gpointer data)
{
  GtkWidget *popup_menu;
  g_return_val_if_fail(parent != NULL, NULL);
  g_return_val_if_fail(ui_info != NULL, NULL);
  popup_menu = gnome_popup_menu_new(ui_info);
  gnome_popup_menu_attach(popup_menu, parent, (gpointer) data);
  return (popup_menu);
}

GtkWidget *
gnome_db_new_toolbar_widget (GtkOrientation orientation,
                             GtkToolbarStyle style,
                             GnomeUIInfo uiinfo[],
                             gpointer user_data)
{
  GtkWidget* toolbar;
  gint cnt;

  toolbar = gtk_toolbar_new(orientation, style);
  gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);
  if (uiinfo)
    {
      for (cnt = 0; uiinfo[cnt].type != GNOME_APP_UI_ENDOFINFO; cnt++)
        {
          uiinfo[cnt].user_data = user_data;
        }
      gnome_app_fill_toolbar(GTK_TOOLBAR(toolbar), uiinfo, NULL);
    }
  gtk_widget_show(toolbar);
  return toolbar;
}

void
gnome_db_show_about_dialog (const gchar *title)
{
  const gchar *authors[] = { "Michael Lausch <michael@lausch.at>",
                             "Rodrigo Moya <rodrigo@gnome-db.org>",
                             "Stephan Heinze <stephan.heinze@xcom.de>",
                             "Vivien Malerba <malerba@linuxave.net>",
			              "Nick Gorham <nick@lurcher.org>",
			              "Chris Wiegand <cwiegand@urgentmail.com>",
			              "Alvaro del Castillo <acs@futurnet.es>",
                             "Akira TAGOH <tagoh@gnome.gr.jp>",
                             "Carlos Perell Marn <carlos@hispalinux.es>",
                             "Holger Thon <holger@gidayu.max.uni-duisburg.de>",
                             "Joaqun Cuenca <e98cuenc@yahoo.com>",
                             NULL };
  GtkWidget *about = gnome_about_new(title, VERSION,
	     _("Copyright The Free Software Foundation (C) 1998-2000"),
	     authors,
             _("This program is part of the GNOME project for LINUX. "
	       "GNOME Data Access comes with ABSOLUTELY NO WARRANTY. "
	       "This is free software, and you are welcome to redistribute "
	       "it under the conditions of the GNU General Public Licence."),
	     "gnome-db.png");
  gtk_widget_show(about);
}

void
gnome_db_show_error (const gchar *format, ...)
{
  va_list args;
  gchar sz[256];
    
  /* build the message string */
  va_start(args, format);
  vsprintf(sz, format, args);
  va_end(args);
    
  gnome_error_dialog(sz);
}

void
gnome_db_show_file (const gchar *filename)
{
  GtkWidget* dialog;
  GtkWidget* viewer;

  g_return_if_fail(filename != 0);

  dialog = gnome_dialog_new(filename, GNOME_STOCK_BUTTON_CLOSE, 0);
  viewer = gnome_less_new();
  gtk_widget_show(viewer);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), viewer, 1, 1, 0);
  if (gnome_less_show_file(GNOME_LESS(viewer), filename))
    {
      gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
    }
  else gnome_db_show_error(_("Could not open file\n%s"), filename);
}

void
gnome_db_set_bg_color (GtkWidget *w, gint red, gint green, gint blue)
{
  GdkColormap* colormap;
  GdkColor     color;
  
  g_return_if_fail(GTK_IS_WIDGET(w));

  colormap = gdk_colormap_get_system();
  color.red = red;
  color.green = green;
  color.blue = blue;
  if (gdk_color_alloc(colormap, &color))
    {
      GtkStyle *style, *style_default;

      style_default = gtk_widget_get_default_style();
      style = gtk_style_copy(style_default);
      style->base[GTK_STATE_NORMAL] = color;
      gtk_widget_set_style(w, style);
    }
  else g_warning(_("Could not allocate color"));
}

static gint
get_config_int (const gchar *format, ...)
{
  gchar   buffer[2048];
  va_list args;

  g_return_val_if_fail(format, 0);

  va_start(args, format);
  vsprintf(buffer, format, args);
  va_end(args);

  return gda_config_get_int(buffer);
}

static void
set_config_int (gint value, const gchar *format, ...)
{
  gchar   buffer[2048];
  va_list args;

  g_return_if_fail(format);

  va_start(args, format);
  vsprintf(buffer, format, args);
  va_end(args);

  gda_config_set_int(buffer, value);
}

void
gnome_db_load_window_config (const gchar *name, GtkWidget *window)
{
  g_return_if_fail(name != NULL);
  g_return_if_fail(GTK_IS_WIDGET(window));

  gdk_window_move(window->window,
                  get_config_int("/%s/Placement/%s_X",
                                 g_get_prgname(),
                                 name) + 20,
                  get_config_int("/%s/Placement/%s_Y",
                                 g_get_prgname(),
                                 name) + 20);
  gtk_widget_set_usize(window,
                       get_config_int("/%s/Placement/%s_Width",
                                      g_get_prgname(),
                                      name),
                       get_config_int("/%s/Placement/%s_Height",
                                      g_get_prgname(),
                                      name));
}

void
gnome_db_save_window_config (const gchar *name, GtkWidget *window)
{
  gint x, y, width, height;

  g_return_if_fail(name != NULL);
  g_return_if_fail(GTK_IS_WIDGET(window));

  gdk_window_get_geometry(window->window, &x, &y, &width, &height, NULL);
  set_config_int(x, "/%s/Placement/%s_X", g_get_prgname(), name);
  set_config_int(y, "/%s/Placement/%s_Y", g_get_prgname(), name);
  set_config_int(width, "/%s/Placement/%s_Width", g_get_prgname(), name);
  set_config_int(height, "/%s/Placement/%s_Height", g_get_prgname(), name);
}

/*
 * File selection dialog
 */
static void
filesel_button_clicked (GtkWidget *w, glong *selected)
{
  *selected = (glong) gtk_object_get_data(GTK_OBJECT(w), "button");
}

gchar *
gnome_db_select_file (const gchar *title)
{
  GtkWidget* filesel;
  glong*     selected;
  gchar*     res = NULL;

  selected = (glong *) g_malloc(sizeof(glong));
  *selected = 0;

  /* create dialog */
  filesel = gtk_file_selection_new(title);
  gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
		      "button",
		      (gpointer) 1);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(filesel_button_clicked),
		     (gpointer) selected);
  gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
		      "button",
		      (gpointer) 2);
  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(filesel_button_clicked),
		     (gpointer) selected);
  gtk_widget_show(filesel);
  gtk_file_selection_show_fileop_buttons(GTK_FILE_SELECTION(filesel));

  /* wait for selection from user */
  while (!*selected)
    gtk_main_iteration_do(TRUE);
  if (*selected == 1) /* OK button */
    {
      res = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
    }
  else res = NULL;

  /* clean up */
  gtk_widget_destroy(filesel);
  g_free((gpointer) selected);
  return res;
}

void
gnome_db_submit_bug_cb (GtkWidget *w, gpointer data)
{
  gchar* args[1];

  args[0] = "bug-buddy";
  if (gnome_execute_async(NULL, 1, args) == -1)
    gnome_db_show_error(_("Unable to execute Bug Report Tool"));
}





