/* GNOME DB library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome-db.h>

/**
 * gnome_db_config_component_new
 */
GnomeDbConfigComponent *
gnome_db_config_component_new (void)
{
  return g_new0(GnomeDbConfigComponent, 1);
}

/**
 * gnome_db_config_component_free
 */
void
gnome_db_config_component_free (GnomeDbConfigComponent *component)
{
  g_return_if_fail(component != NULL);
  
  if (component->name) g_free((gpointer) component->name);
  if (component->location) g_free((gpointer) component->location);
  if (component->description) g_free((gpointer) component->description);
  if (component->repo_ids) g_free((gpointer) component->repo_ids);
  if (component->type) g_free((gpointer) component->type);
  
  g_free((gpointer) component);
}

/**
 * gnome_db_config_component_set_name
 */
void
gnome_db_config_component_set_name (GnomeDbConfigComponent *component, const gchar *name)
{
  g_return_if_fail(component != NULL);
  g_return_if_fail(name != NULL);
  
  if (component->name) g_free((gpointer) component->name);
  component->name = g_strdup(name);
}

/**
 * gnome_db_config_component_set_location
 */
void
gnome_db_config_component_set_location (GnomeDbConfigComponent *component, const gchar *location)
{
  g_return_if_fail(component != NULL);
  g_return_if_fail(location != NULL);
  
  if (component->location) g_free((gpointer) component->location);
  component->location = g_strdup(location);
}

/**
 * gnome_db_config_component_set_description
 */
void
gnome_db_config_component_set_description (GnomeDbConfigComponent *component, const gchar *desc)
{
  g_return_if_fail(component != NULL);
  g_return_if_fail(desc != NULL);
  
  if (component->description) g_free((gpointer) component->description);
  component->description = g_strdup(desc);
}

/**
 * gnome_db_config_component_set_repo_ids
 */
void
gnome_db_config_component_set_repo_ids (GnomeDbConfigComponent *component, const gchar *repo_ids)
{
  g_return_if_fail(component != NULL);
  g_return_if_fail(repo_ids != NULL);
  
  if (component->repo_ids) g_free((gpointer) component->repo_ids);
  component->repo_ids = g_strdup(repo_ids);
}

/**
 * gnome_db_config_component_set_type
 */
void
gnome_db_config_component_set_type (GnomeDbConfigComponent *component, const gchar *type)
{
  g_return_if_fail(component != NULL);
  g_return_if_fail(type != NULL);
  
  if (component->type) g_free((gpointer) component->type);
  component->type = g_strdup(type);
}

/**
 * gnome_db_config_component_set_icon
 */
void
gnome_db_config_component_set_icon (GnomeDbConfigComponent *component, const gchar *icon)
{
  g_return_if_fail(component != NULL);
  g_return_if_fail(icon != NULL);
  
  if (component->icon) g_free((gpointer) component->icon);
  component->icon = g_strdup(icon);
}

/**
 * gnome_db_config_get_shell_components
 *
 * Return a list of all registered components that can be used in the
 * gnome-db frontend
 */
GList *
gnome_db_config_get_shell_components (void)
{
  OAF_ServerInfoList* servlist;
  CORBA_Environment   ev;
  GList*              ret = NULL;
  gint                i;
  
  CORBA_exception_init(&ev);
  servlist = oaf_query("repo_ids.has('IDL:BonoboControl/gnome-db-shell-component:1.0')", NULL, &ev);
  if (servlist)
    {
      for (i = 0; i < servlist->_length; i++)
        {
          GnomeDbConfigComponent* component = gnome_db_config_component_new();
          gnome_db_config_component_set_name(component, servlist->_buffer[i].iid);
          gnome_db_config_component_set_location(component, servlist->_buffer[i].location_info);
          gnome_db_config_component_set_description(component,
                                                    gda_corba_get_oaf_attribute(servlist->_buffer[i].props, "description"));
          gnome_db_config_component_set_repo_ids(component,
                                                 gda_corba_get_oaf_attribute(servlist->_buffer[i].props, "repo_ids"));
          gnome_db_config_component_set_type(component, servlist->_buffer[i].server_type);
          gnome_db_config_component_set_icon(component,
                                             gda_corba_get_oaf_attribute(servlist->_buffer[i].props, "icon"));
          
          ret = g_list_append(ret, (gpointer) component);
        }
    }
  CORBA_exception_free(&ev);
  
  return ret;
}

/**
 * gnome_db_config_component_free_list
 */
void
gnome_db_config_component_free_list (GList *list)
{
  GList* node;
  
  while ((node = g_list_first(list)))
    {
      GnomeDbConfigComponent* component = (GnomeDbConfigComponent *) node->data;
      list = g_list_remove(list, node->data);
      gnome_db_config_component_free(component);
      
    }
}
