/* GNOME DB libary
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gnome-db.h"
#include <config.h>
#if defined(HAVE_GAL)
#  include <gal/e-paned/e-hpaned.h>
#endif

static void gnome_db_browser_class_init (GnomeDbBrowserClass *);
static void gnome_db_browser_init       (GnomeDbBrowser *);
static void gnome_db_browser_destroy    (GnomeDbBrowser *);

static void browse_data_cb              (GtkWidget *w, gpointer data);
static void browse_data_loaded_cb       (GnomeDbGrid *grid, gpointer data);
static void extra_info_cb               (GtkWidget *w, gpointer data);
static void show_aggrs_cb               (GtkWidget *w, gpointer data);
static void show_tables_cb              (GtkWidget *w, gpointer data);
static void show_procs_cb               (GtkWidget *w, gpointer data);
static void show_views_cb               (GtkWidget *w, gpointer data);

/*
 * GnomeDbBrowser widget signals
 */
enum
{
  GNOME_DB_BROWSER_CONNECTION_CHANGED,
  GNOME_DB_BROWSER_LAST_SIGNAL
};

static gint gnome_db_browser_signals[GNOME_DB_BROWSER_LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static void
add_list_row_cb (GnomeDbList *list, gint row, GnomeDbBrowser *brw)
{
  GnomePixmap* pixmap;

  g_return_if_fail(GNOME_DB_IS_LIST(list));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));

  pixmap = gnome_db_get_pixmap(GNOME_STOCK_MENU_BOOK_RED);
  gnome_db_list_set_pixmap(list, row, pixmap->pixmap, pixmap->mask);
}

static void
browse_data_cb (GtkWidget *w, gpointer data)
{
  gchar*          name;
  GnomeDbBrowser* brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  g_return_if_fail(GNOME_DB_IS_LIST(brw->object_list));
  
  name = gnome_db_list_get_string(GNOME_DB_LIST(brw->object_list));
  if (name)
    {
      gchar *str = 0;
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)) ||
	  gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
	    {
	      str = g_strdup_printf("select * from %s", name);
	    }
          
      if (str != 0)
        {
          GtkWidget *dialog, *frame, *table, *grid, *label, *entry;
          gulong reccount;
          Gda_Recordset *recset;
          
          dialog = gnome_dialog_new(name, GNOME_STOCK_BUTTON_CLOSE, NULL);
          gtk_widget_set_usize(dialog, 600, 450);
          frame = gnome_db_new_frame_widget(NULL);
          gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
          
          table = gtk_table_new(4, 2, FALSE);
          gtk_widget_show(table);
          gtk_container_add(GTK_CONTAINER(frame), table);
          
          label = gnome_db_new_label_widget(_("Object Name"));
          gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1,
                           GTK_FILL,
                           GTK_FILL,
                           3, 3);
                           
          entry = gnome_db_new_entry_widget(0, FALSE);
          gtk_entry_set_text(GTK_ENTRY(entry), name);
          gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);
          
          grid = gnome_db_new_grid_widget(NULL);
          gtk_table_attach(GTK_TABLE(table), grid, 0, 2, 1, 3,
                           GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                           GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                           3, 3);

          label = gnome_db_new_label_widget(_("0 rows"));
          gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
          gtk_table_attach(GTK_TABLE(table), label, 0, 2, 3, 4, GTK_FILL, GTK_FILL, 3, 3);
          gtk_signal_connect(GTK_OBJECT(grid),
                             "loaded",
                             GTK_SIGNAL_FUNC(browse_data_loaded_cb),
                             (gpointer) label);
          
          /* show recordset */
          recset = gda_connection_execute(brw->cnc, str, &reccount, 0);
          gnome_db_grid_set_recordset(GNOME_DB_GRID(grid), recset);
          
          gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
          g_free((gpointer) str);
        }
    }
}

static void
browse_data_loaded_cb (GnomeDbGrid *grid, gpointer data)
{
  gchar *str;

  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  g_return_if_fail(GTK_IS_LABEL(data));

  str = g_strdup_printf(_("%d rows"), gnome_db_grid_get_row_count(grid));
  gtk_label_set_text(GTK_LABEL(data), str);
  g_free((gpointer) str);
}

static void
drop_object_cb (GtkWidget *w, gpointer data)
{
}

static void
extra_info_cb (GtkWidget *w, gpointer data)
{
  gchar*          name;
  GnomeDbBrowser* brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  g_return_if_fail(GNOME_DB_IS_LIST(brw->object_list));
  
  name = gnome_db_list_get_string(GNOME_DB_LIST(brw->object_list));
  if (name)
    {
      Gda_Recordset* recset;

      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)))
        {
          recset = gda_connection_open_schema(brw->cnc,
                                              GDA_Connection_GDCN_SCHEMA_TABLES,
                                              GDA_Connection_EXTRA_INFO,
                                              "",
                                              GDA_Connection_OBJECT_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
        {
          recset = gda_connection_open_schema(brw->cnc,
                                              GDA_Connection_GDCN_SCHEMA_VIEWS,
                                              "",
                                              GDA_Connection_OBJECT_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->procs_button)))
        {
          recset = gda_connection_open_schema(brw->cnc,
                                              GDA_Connection_GDCN_SCHEMA_PROCS,
                                              "",
                                              GDA_Connection_OBJECT_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->types_button)))
        {
          recset = gda_connection_open_schema(brw->cnc,
                                              GDA_Connection_GDCN_SCHEMA_PROV_TYPES,
                                              "",
                                              GDA_Connection_OBJECT_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->aggrs_button)))
        {
          recset = gda_connection_open_schema(brw->cnc,
                                              GDA_Connection_GDCN_SCHEMA_AGGREGATES,
                                              "",
                                              GDA_Connection_OBJECT_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      
      /* show object properties */
      if (IS_GDA_RECORDSET(recset))
        {
          GtkWidget* dialog;
          GtkWidget* dataset;
          
          dialog = gnome_dialog_new(_("Object properties"), GNOME_STOCK_BUTTON_CLOSE, NULL);
          dataset = gnome_db_dataset_new(recset);
          gtk_widget_show(dataset);
          gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), dataset, 1, 1, 0);
          gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
          gda_recordset_free(recset);
        }
    }
}

static void
select_object_cb (GnomeDbList *icon_list, GnomeDbBrowser *brw)
{
  gchar *name = 0;
  
  g_return_if_fail(GNOME_DB_IS_LIST(icon_list));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  name = gnome_db_list_get_string(icon_list);
  if (name)
    {
      Gda_Recordset *recset = 0;
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)) ||
          gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
        {
          recset = gda_connection_open_schema(brw->cnc, 
                                              GDA_Connection_GDCN_SCHEMA_COLS,
                                              GDA_Connection_OBJECT_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->procs_button)))
        {
          recset = gda_connection_open_schema(brw->cnc,
                                              GDA_Connection_GDCN_SCHEMA_PROC_PARAMS,
                                              GDA_Connection_OBJECT_NAME,
                                              name,
                                              GDA_Connection_no_CONSTRAINT);
        }
      gnome_db_grid_set_recordset(GNOME_DB_GRID(brw->object_description),
                                  recset);
    }
}

static void
show_aggrs_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_RADIO_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->aggrs_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc,
                                      GDA_Connection_GDCN_SCHEMA_AGGREGATES,
                                      GDA_Connection_no_CONSTRAINT, 0);
  if (recset != 0)
    {
      gnome_db_list_set_recordset(GNOME_DB_LIST(brw->object_list), recset, 0);
    }
}

static void
show_tables_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_RADIO_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc,
                                      GDA_Connection_GDCN_SCHEMA_TABLES,
                                      GDA_Connection_no_CONSTRAINT, 0);
  if (recset != 0)
    {
      gnome_db_list_set_recordset(GNOME_DB_LIST(brw->object_list), recset, 0);
    }
}

static void
show_procs_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_RADIO_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->procs_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc,
                                      GDA_Connection_GDCN_SCHEMA_PROCS,
                                      GDA_Connection_no_CONSTRAINT, 0);
  if (recset != 0)
    {
      gnome_db_list_set_recordset(GNOME_DB_LIST(brw->object_list), recset, 0);
    }
}

static void
show_types_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_RADIO_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->types_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc,
                                      GDA_Connection_GDCN_SCHEMA_PROV_TYPES,
                                      GDA_Connection_no_CONSTRAINT, 0);
  if (recset != 0)
    {
      gnome_db_list_set_recordset(GNOME_DB_LIST(brw->object_list), recset, 0);
    }
}

static void
show_views_cb (GtkWidget *w, gpointer data)
{
  Gda_Recordset *recset;
  GnomeDbBrowser *brw = GNOME_DB_BROWSER(data);
  
  g_return_if_fail(GTK_IS_RADIO_BUTTON(w));
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
    {
      return;
    }
    
  /* clean up previous lists */
  gnome_db_browser_clear(brw);
  
  recset = gda_connection_open_schema(brw->cnc,
                                      GDA_Connection_GDCN_SCHEMA_VIEWS,
                                      GDA_Connection_no_CONSTRAINT, 0);
                                      
  if (recset != 0)
    {
      gnome_db_list_set_recordset(GNOME_DB_LIST(brw->object_list), recset, 0);
    }
}

/*
 * GnomeDbBrowser interface
 */
static void
gnome_db_browser_class_init (GnomeDbBrowserClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  gnome_db_browser_signals[GNOME_DB_BROWSER_CONNECTION_CHANGED] =
           gtk_signal_new("connection_changed", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbBrowserClass, connection_changed),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, gnome_db_browser_signals,
                               GNOME_DB_BROWSER_LAST_SIGNAL);
  klass->connection_changed = 0;
  object_class->destroy = gnome_db_browser_destroy;
}

static void
gnome_db_browser_init (GnomeDbBrowser *brw)
{
  //GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *box;
  GtkWidget *button;
  GtkWidget *paned;

#if defined(HAVE_GAL)
  paned = e_hpaned_new();
#else
  paned = gtk_hpaned_new();
#endif
  gtk_box_pack_start(GTK_BOX(brw), paned, 1, 1, 0);
  gtk_widget_show(paned);

  table = gtk_table_new(3, 4, FALSE);
#if defined(HAVE_GAL)
  e_paned_add2(E_PANED(paned), table);
#else
  gtk_paned_add2(GTK_PANED(paned), table);
#endif
  gtk_widget_show(table);
  
  brw->object_list = gnome_db_list_new(0, 0);
  gtk_signal_connect(GTK_OBJECT(brw->object_list),
		     "add_row",
		     GTK_SIGNAL_FUNC(add_list_row_cb),
		     (gpointer) brw);
  gtk_signal_connect(GTK_OBJECT(brw->object_list), "select_row",
                     GTK_SIGNAL_FUNC(select_object_cb), (gpointer) brw);
  gtk_widget_show(brw->object_list);
#if defined(HAVE_GAL)
  e_paned_add1(E_PANED(paned), brw->object_list);
#else
  gtk_paned_add1(GTK_PANED(paned), brw->object_list);
#endif

  brw->object_description = gnome_db_new_grid_widget(NULL);
  gtk_table_attach(GTK_TABLE(table), brw->object_description, 0, 3, 1, 4,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  
  /* create tabs */                 
  box = gtk_hbox_new(FALSE, 0);
  brw->tables_button = gtk_radio_button_new_with_label(0, _("Tables"));
  gtk_signal_connect(GTK_OBJECT(brw->tables_button), "toggled",
                     GTK_SIGNAL_FUNC(show_tables_cb), (gpointer) brw);
  gtk_widget_show(brw->tables_button);
  gtk_box_pack_start(GTK_BOX(box), brw->tables_button, 0, 0, 0);

  brw->views_button = gtk_radio_button_new_with_label(
  				gtk_radio_button_group(GTK_RADIO_BUTTON(brw->tables_button)),
  				_("Views"));
  gtk_signal_connect(GTK_OBJECT(brw->views_button), "toggled",
                     GTK_SIGNAL_FUNC(show_views_cb), (gpointer) brw); 
  gtk_widget_show(brw->views_button);
  gtk_box_pack_start(GTK_BOX(box), brw->views_button, 0, 0, 0);
      
  brw->procs_button = gtk_radio_button_new_with_label(
  				gtk_radio_button_group(GTK_RADIO_BUTTON(brw->tables_button)),
  				_("Procedures"));
  gtk_signal_connect(GTK_OBJECT(brw->procs_button), "toggled",
                     GTK_SIGNAL_FUNC(show_procs_cb), (gpointer) brw);
  gtk_widget_show(brw->procs_button);
  gtk_box_pack_start(GTK_BOX(box), brw->procs_button, 0, 0, 0);

  brw->types_button = gtk_radio_button_new_with_label(
				gtk_radio_button_group(GTK_RADIO_BUTTON(brw->tables_button)),
				_("Types"));
  gtk_signal_connect(GTK_OBJECT(brw->types_button), "toggled",
                     GTK_SIGNAL_FUNC(show_types_cb), (gpointer) brw);
  gtk_widget_show(brw->types_button);
  gtk_box_pack_start(GTK_BOX(box), brw->types_button, 0, 0, 0);

  brw->aggrs_button = gtk_radio_button_new_with_label(
                                gtk_radio_button_group(GTK_RADIO_BUTTON(brw->tables_button)),
                                _("Aggregates"));
  gtk_signal_connect(GTK_OBJECT(brw->aggrs_button), "toggled",
                     GTK_SIGNAL_FUNC(show_aggrs_cb), (gpointer) brw);
  gtk_widget_show(brw->aggrs_button);
  gtk_box_pack_start(GTK_BOX(box), brw->aggrs_button, 0, 0, 0);
  
  gtk_table_attach(GTK_TABLE(table), box, 0, 3, 0, 1, GTK_FILL, GTK_FILL, 1, 1);
  gtk_widget_show(box);
  
  /* create button bar */
  box = gtk_hbox_new(TRUE, 0);
  gtk_widget_show(box);
  gtk_table_attach(GTK_TABLE(table), box, 0, 3, 4, 5, GTK_FILL, GTK_FILL, 1, 1);
                   
  button = gnome_pixmap_button(gnome_stock_new_with_icon(GNOME_STOCK_MENU_PROP),
                               _("Data"));
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(browse_data_cb), (gpointer) brw);
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  
  button = gnome_pixmap_button(gnome_stock_new_with_icon(GNOME_STOCK_MENU_TRASH),
                               _("Remove"));
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(drop_object_cb), (gpointer) brw);
  gtk_widget_show(button);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
  
  button = gnome_db_new_button_widget_with_pixmap(_("Extra Info"), GNOME_STOCK_MENU_PROP);
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(extra_info_cb), (gpointer) brw);
  gtk_box_pack_start(GTK_BOX(box), button, 0, 0, 0);
}

guint
gnome_db_browser_get_type (void)
{
  static guint db_browser_type = 0;

  if (!db_browser_type)
    {
      GtkTypeInfo db_browser_info =
      {
        "GnomeDbBrowser",
        sizeof (GnomeDbBrowser),
        sizeof (GnomeDbBrowserClass),
        (GtkClassInitFunc) gnome_db_browser_class_init,
        (GtkObjectInitFunc) gnome_db_browser_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_browser_type = gtk_type_unique(gtk_vbox_get_type(), &db_browser_info);
    }
  return (db_browser_type);
}

/**
 * gnome_db_browser_new
 * @cnc: the connection to be used for browsing
 *
 * Create a new GnomeDbBrowser widget, binding it to an existing GDA connection
 * if given
 *
 * Returns: a pointer to the new widget if success, NULL if error
 */
GtkWidget *
gnome_db_browser_new (Gda_Connection *cnc)
{
  GtkWidget *brw;

  brw = gtk_type_new(gnome_db_browser_get_type());
  if (IS_GDA_CONNECTION(cnc))
    gnome_db_browser_set_connection(GNOME_DB_BROWSER(brw), cnc);
  return (brw);
}

/**
 * gnome_db_browser_destroy
 * @brw: the browser widget to destroy
 *
 * Destroy the widget pointed to by the @brw parameter
 */
static void
gnome_db_browser_destroy (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));

  gtk_widget_destroy(brw->object_list);  
  gtk_widget_destroy(brw->object_description);
}

/**
 * gnome_db_browser_get_connection
 * @brw: the browser widget
 *
 * Return the Gda_Connection object actually being used by this browser
 *
 * Returns: a pointer to a Gda_Connection object, or NULL on error
 */
Gda_Connection *
gnome_db_browser_get_connection (GnomeDbBrowser *brw)
{
  g_return_val_if_fail(GNOME_DB_IS_BROWSER(brw), 0);
  return (brw->cnc);
}

/**
 * gnome_db_browser_set_connection
 * @brw: the browser widget
 * @cnc: the Gda_Connection object to be used
 *
 * Dynamically change the Gda_Connection object being used by the GnomeDbBrowser
 * widget
 */
void
gnome_db_browser_set_connection (GnomeDbBrowser *brw, Gda_Connection *cnc)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  g_return_if_fail(IS_GDA_CONNECTION(cnc));
  g_return_if_fail(gda_connection_is_open(cnc));
  
  brw->cnc = cnc;
  
  /* update object list */
  gnome_db_browser_clear(brw);
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)))
    show_tables_cb(brw->tables_button, (gpointer) brw);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
    show_views_cb(brw->views_button, (gpointer) brw);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->procs_button)))
    show_procs_cb(brw->procs_button, (gpointer) brw);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->types_button)))
    show_types_cb(brw->types_button, (gpointer) brw);

  gtk_signal_emit(GTK_OBJECT(brw), 
                  gnome_db_browser_signals[GNOME_DB_BROWSER_CONNECTION_CHANGED]);
}

/**
 * gnome_db_browser_clear
 * @brw: the browser widget
 *
 * Clear all widgets in the browser
 */
void
gnome_db_browser_clear (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  gnome_db_list_clear(GNOME_DB_LIST(brw->object_list));
  gnome_db_grid_set_recordset(GNOME_DB_GRID(brw->object_description), 0);
}

/**
 * gnome_db_browser_refresh
 * @brw: the browser widget
 *
 * Refresh the browser widgets' contents to display possible changes in
 * the underlying recordset
 */
void
gnome_db_browser_refresh (GnomeDbBrowser *brw)
{
  g_return_if_fail(brw != 0);
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  
  gnome_db_browser_clear(brw);
  
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->tables_button)))
    show_tables_cb(GTK_RADIO_BUTTON(brw->tables_button), brw);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->views_button)))
    show_views_cb(GTK_RADIO_BUTTON(brw->views_button), brw);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->procs_button)))
    show_procs_cb(GTK_RADIO_BUTTON(brw->procs_button), brw);
  else if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(brw->types_button)))
    show_types_cb(GTK_RADIO_BUTTON(brw->types_button), brw);
}

/**
 * gnome_db_browser_show_procs
 * @brw: the GnomeDbBrowser widget
 */
void
gnome_db_browser_show_procs (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(brw->procs_button), TRUE);
}

/**
 * gnome_db_browser_show_tables
 * @brw: the GnomeDbBrowser widget
 */
void
gnome_db_browser_show_tables (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(brw->tables_button), TRUE);
}

/**
 * gnome_db_browser_show_types
 * @brw: the GnomeDbBrowser widget
 */
void
gnome_db_browser_show_types (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(brw->types_button), TRUE);
}

/**
 * gnome_db_browser_show_views
 * @brw: the GnomeDbBrowser widget
 */
void
gnome_db_browser_show_views (GnomeDbBrowser *brw)
{
  g_return_if_fail(GNOME_DB_IS_BROWSER(brw));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(brw->views_button), TRUE);
}
