/* GNOME DB library
 * Copyright (C) 2000 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gnome-db.h>

#define LOGVIEWER_DateEntry   "LOGVIEWER_DateEntry"
#define LOGVIEWER_TimeEntry   "LOGVIEWER_TimeEntry"
#define LOGVIEWER_MessageText "LOGVIEWER_MessageText"
#define LOGVIEWER_RecordLabel "LOGVIEWER_RecordLabel"

static void gnome_db_log_viewer_class_init (GnomeDbLogViewerClass *klass);
static void gnome_db_log_viewer_init       (GnomeDbLogViewer *viewer);
static void gnome_db_log_viewer_destroy    (GnomeDbLogViewer *viewer);

/*
 * GnomeDbLogViewer signals
 */
enum
{
  GNOME_DB_LOG_VIEWER_LAST_SIGNAL
};

static gint gnome_db_log_viewer_signals[GNOME_DB_LOG_VIEWER_LAST_SIGNAL] = { 0 };

/*
 * Private functions
 */
static void
update_view_dialog (GnomeDialog *dialog, GtkCList *list, gint row)
{
  gchar*     str;
  GtkWidget* w;

  g_return_if_fail(GNOME_IS_DIALOG(dialog));
  g_return_if_fail(GTK_IS_CLIST(list));

  /* update date entry */
  w = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(dialog), LOGVIEWER_DateEntry));
  if (GTK_IS_ENTRY(w))
    {
      if (gtk_clist_get_text(list, row, 0, &str))
	{
	  gtk_entry_set_text(GTK_ENTRY(w), str);
	}
    }

  /* update time entry */
  w = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(dialog), LOGVIEWER_TimeEntry));
  if (GTK_IS_ENTRY(w))
    {
      if (gtk_clist_get_text(list, row, 1, &str))
	{
	  gtk_entry_set_text(GTK_ENTRY(w), str);
	}
    }

  /* update message */
  w = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(dialog), LOGVIEWER_MessageText));
  if (GTK_IS_TEXT(w))
    {
      if (gtk_clist_get_text(list, row, 2, &str))
	{
	  gtk_editable_delete_text(GTK_EDITABLE(w), 0,
				   gtk_text_get_length(GTK_TEXT(w)));
	  gtk_text_insert(GTK_TEXT(w), NULL, NULL, NULL, str, strlen(str));
	}
    }

  /* update record label */
  w = GTK_WIDGET(gtk_object_get_data(GTK_OBJECT(dialog), LOGVIEWER_RecordLabel));
  if (GTK_IS_LABEL(w))
    {
      str = g_strdup_printf(_("Entry %d of %d"),
			    row + 1,
			    GTK_CLIST(list)->rows);
      gtk_label_set_text(GTK_LABEL(w), str);
      g_free((gpointer) str);
    }
}

/*
 * Callbacks
 */
static void
view_entry_clicked_cb (GnomeDialog *dialog, gint button_number, gpointer data)
{
  GList* selection;
  gint   row;

  g_return_if_fail(GNOME_IS_DIALOG(dialog));
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(data));
  g_return_if_fail(GTK_IS_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list));

  /* retrieve current selection */
  selection = GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list)->selection;
  g_return_if_fail(selection != NULL);
  row = GPOINTER_TO_INT(selection->data);

  switch (button_number)
    {
    case 0 : /* prev button */
      if (row > 0)
	{
	  update_view_dialog(dialog, GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list), row - 1);
	  gtk_clist_unselect_row(GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list), row, 0);
	  gtk_clist_select_row(GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list), row - 1, 0);
	}
      break;
    case 1 : /* next button */
      if (row < GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list)->rows - 1)
	{
	  update_view_dialog(dialog, GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list), row + 1);
	  gtk_clist_unselect_row(GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list), row, 0);
	  gtk_clist_select_row(GTK_CLIST(GNOME_DB_LOG_VIEWER(data)->msg_list), row + 1, 0);
	}
      break;
    case 2 : /* close button */
      gnome_dialog_close(GNOME_DIALOG(dialog));
      break;
    }
}

/*
 * GnomeDbLogViewer widget interface
 */
static void
gnome_db_log_viewer_class_init (GnomeDbLogViewerClass *klass)
{
  GtkObjectClass* object_class = GTK_OBJECT_CLASS(klass);

  object_class->destroy = gnome_db_log_viewer_destroy;
}

static void
gnome_db_log_viewer_init (GnomeDbLogViewer *viewer)
{
  GtkWidget* frame;
  GtkWidget* scroll;
  gchar*     col_titles[] = { N_("Date"), N_("Time"), N_("Message") };

  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));

  frame = gnome_db_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(viewer), frame, 1, 1, 0);

  scroll = gnome_db_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);
  viewer->msg_list = gnome_db_new_clist_widget(col_titles,
					     sizeof(col_titles) / sizeof(col_titles[0]));
  gtk_container_add(GTK_CONTAINER(scroll), viewer->msg_list);
}

GtkType
gnome_db_log_viewer_get_type (void)
{
  static GtkType db_log_viewer_type = 0;
  
  if (!db_log_viewer_type)
    {
      GtkTypeInfo db_log_viewer_info =
      {
        "GnomeDbLogViewer",
        sizeof (GnomeDbLogViewer),
        sizeof (GnomeDbLogViewerClass),
        (GtkClassInitFunc) gnome_db_log_viewer_class_init,
        (GtkObjectInitFunc) gnome_db_log_viewer_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_log_viewer_type = gtk_type_unique(gtk_vbox_get_type(), &db_log_viewer_info);
    }
  return (db_log_viewer_type);
}

/**
 * gnome_db_log_viewer_new
 *
 * Create a new #GnomeDbLogViewer widget, which lets you navigate through
 * the different log files generated by the GNOME-DB processes.
 *
 * Returns: a pointer to the newly created widget, or NULL on error
 */
GtkWidget *
gnome_db_log_viewer_new (void)
{
  GtkWidget* viewer = GTK_WIDGET(gtk_type_new(gnome_db_log_viewer_get_type()));
  return viewer;
}

static void
gnome_db_log_viewer_destroy (GnomeDbLogViewer *viewer)
{
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));
}

/**
 * gnome_db_log_viewer_load_file
 * @viewer: a log viewer widget
 * @prgname: program name
 */
gboolean
gnome_db_log_viewer_load_file (GnomeDbLogViewer *viewer, const gchar *prgname)
{
  GList* sections = NULL;
  GList* node_section;
  GList* keys = NULL;
  gchar* tmp;

  g_return_val_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer), FALSE);
  g_return_val_if_fail(prgname != NULL, FALSE);

  gnome_db_clear_clist(GTK_CLIST(viewer->msg_list));
  gtk_clist_freeze(GTK_CLIST(viewer->msg_list));

  tmp = g_strdup_printf("%s/%s", GDA_CONFIG_SECTION_LOG, prgname);
  sections = gda_config_list_sections(tmp);
  for (node_section = sections; node_section != NULL; node_section = g_list_next(node_section))
    {
      gchar* tmp2;
      GList* node_key;
      
      /* now get all keys in this section */
      tmp2 = g_strdup_printf("%s/%s", tmp, (gchar *) node_section->data);
      keys = gda_config_list_keys(tmp2);
      g_free((gpointer) tmp2);
      for (node_key = keys; node_key != NULL; node_key = g_list_next(node_key))
        {
          gchar* clist_row[3] = { "", "", "" };
          
          tmp2 = g_strdup_printf("%s/%s/%s", tmp, (gchar *) node_section->data, (gchar *) node_key->data);
          clist_row[0] = (gchar *) node_section->data;
          clist_row[1] = (gchar *) node_key->data;
          clist_row[2] = gda_config_get_string(tmp2);
          g_free((gpointer) tmp2);
          gtk_clist_append(GTK_CLIST(viewer->msg_list), clist_row);
        }
      gda_config_free_list(keys);
    }
  gda_config_free_list(sections);
  g_free((gpointer) tmp);
  gtk_clist_thaw(GTK_CLIST(viewer->msg_list));

  g_free((gpointer) viewer->filename);
  viewer->filename = g_strdup(prgname);
  return TRUE;
}

/**
 * gnome_db_log_viewer_refresh
 */
void
gnome_db_log_viewer_refresh (GnomeDbLogViewer *viewer)
{
  gchar* file;

  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));

  file = g_strdup(viewer->filename);
  gnome_db_log_viewer_load_file(viewer, file);
  g_free((gpointer) file);
}

/**
 * gnome_db_log_viewer_clear
 */
void
gnome_db_log_viewer_clear (GnomeDbLogViewer *viewer)
{
  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));

  gnome_db_clear_clist(GTK_CLIST(viewer->msg_list));
  g_free((gpointer) viewer->filename);
}

/**
 * gnome_db_log_viewer_show_current
 */
void
gnome_db_log_viewer_show_current (GnomeDbLogViewer *viewer)
{
  GList* selection;

  g_return_if_fail(GNOME_DB_IS_LOG_VIEWER(viewer));

  selection = GTK_CLIST(GNOME_DB_LOG_VIEWER(viewer)->msg_list)->selection;
  if (selection)
    {
      GtkWidget* dialog;
      GtkWidget* frame;
      GtkWidget* table;
      GtkWidget* label;
      GtkWidget* entry;
      GtkWidget* scroll;

      /* create dialog */
      dialog = gnome_dialog_new(_("Log Entry"),
				GNOME_STOCK_BUTTON_PREV,
				GNOME_STOCK_BUTTON_NEXT,
				GNOME_STOCK_BUTTON_CLOSE,
				NULL);
      gtk_signal_connect(GTK_OBJECT(dialog),
			 "clicked",
			 GTK_SIGNAL_FUNC(view_entry_clicked_cb),
			 (gpointer) viewer);
      frame = gnome_db_new_frame_widget(NULL);
      gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
      table = gnome_db_new_table_widget(10, 4, FALSE);
      gtk_container_add(GTK_CONTAINER(frame), table);

      /* create entries */
      label = gnome_db_new_label_widget(_("Date"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 3, 3);
      entry = gnome_db_new_entry_widget(0, FALSE);
      gtk_object_set_data(GTK_OBJECT(dialog), LOGVIEWER_DateEntry, (gpointer) entry);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 0, 1, GTK_SHRINK, GTK_SHRINK, 3, 3);
      
      label = gnome_db_new_label_widget(_("Time"));
      gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, GTK_SHRINK, GTK_SHRINK, 3, 3);
      entry = gnome_db_new_entry_widget(0, FALSE);
      gtk_object_set_data(GTK_OBJECT(dialog), LOGVIEWER_TimeEntry, (gpointer) entry);
      gtk_table_attach(GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_SHRINK, GTK_SHRINK, 3, 3);

      scroll = gnome_db_new_scrolled_window_widget();
      gtk_table_attach(GTK_TABLE(table), scroll, 0, 10, 2, 3,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       GTK_FILL | GTK_SHRINK | GTK_EXPAND,
		       3, 3);
      entry = gnome_db_new_text_widget();
      gtk_object_set_data(GTK_OBJECT(dialog), LOGVIEWER_MessageText, (gpointer) entry);
      gtk_text_set_editable(GTK_TEXT(entry), FALSE);
      gtk_container_add(GTK_CONTAINER(scroll), entry);

      label = gnome_db_new_label_widget(NULL);
      gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
      gtk_object_set_data(GTK_OBJECT(dialog), LOGVIEWER_RecordLabel, (gpointer) label);
      gtk_table_attach(GTK_TABLE(table), label, 0, 10, 3, 4, GTK_FILL, GTK_FILL, 3, 3);

      gtk_widget_set_usize(dialog, 450, 350);
      update_view_dialog(GNOME_DIALOG(dialog), GTK_CLIST(GNOME_DB_LOG_VIEWER(viewer)->msg_list), GPOINTER_TO_INT(selection->data));
      gtk_widget_show(dialog);
    }
}
