/*  GNOME-DB Front End
 *  Copyrigth (C) 2000 Rodrigo Moya
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gnomedb-fe.h>

#define FE_CONNECTION_DATA "FE_Connection_Data"

static void run_sql_cb    (GtkWidget *w, FE_ConnectionInfo *cnc_info);
static void select_sql_cb (GtkWidget *w, FE_ConnectionInfo *cnc_info);

static GList* open_connections = NULL;
static GnomeUIInfo sql_toolbar[] =
{
  { GNOME_APP_UI_ITEM, N_("Execute"), N_("Execute current command"),
    run_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Select"), N_("Select from previous commands"),
    select_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_UP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Open"), N_("Load file into buffer"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_OPEN, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Save"), N_("Save current file"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_SAVE, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Cut"), N_("Cut selected text"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_CUT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selected text"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Paste"), N_("Paste clipboard contents"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PASTE, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear"), N_("Clear window contents"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TRASH, 0, 0, NULL },
  GNOMEUIINFO_END
};
GnomeUIInfo glb_connection_toolbox[] =
{
  { GNOME_APP_UI_ITEM, N_("SQL"), N_("SQL window"),
    fe_activate_connection_sql_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_EXEC, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Batch"), N_("Batch commands window"),
    fe_activate_connection_batch_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_TIMER, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Queries"), N_("Query designer"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_INDEX, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Browser"), N_("Browse database"),
    fe_activate_connection_browser_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_PROPERTIES, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Export"), N_("Export database"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_REDO, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Import"), N_("Import data into current database"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_UNDO, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Disconnect"), N_("Close current connection"),
    fe_close_connection_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_PIXMAP_STOP, 0, 0, NULL },
  GNOMEUIINFO_END
};

/*
 * Private functions
 */
static void
init_sql_window (FE_ConnectionInfo *cnc_info)
{
  GtkWidget* scroll;
  GtkWidget* toolbar;
  
  g_return_if_fail(cnc_info != NULL);
  
  cnc_info->sql_tab = gnome_db_new_table_widget(3, 6, FALSE);
  toolbar = gnome_db_new_toolbar_widget(GTK_ORIENTATION_HORIZONTAL,
                                        GTK_TOOLBAR_ICONS,
                                        sql_toolbar,
                                        (gpointer) cnc_info);
  gtk_table_attach(GTK_TABLE(cnc_info->sql_tab), toolbar, 0, 3, 0, 1,
                   GTK_FILL, GTK_FILL, 3, 3);

  scroll = gnome_db_new_scrolled_window_widget();
  gtk_table_attach(GTK_TABLE(cnc_info->sql_tab), scroll, 0, 3, 1, 2,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);
  cnc_info->sql_area = gnome_db_new_text_widget();
  gtk_container_add(GTK_CONTAINER(scroll), cnc_info->sql_area);

  cnc_info->sql_grid = gnome_db_new_grid_widget(NULL);
  /*
  gtk_signal_connect(GTK_OBJECT(cnc_data->grid),
                     "loaded",
                     GTK_SIGNAL_FUNC(grid_loaded_cb),
                     NULL);
  gtk_signal_connect(GTK_OBJECT(cnc_data->grid),
                     "select_row",
                     GTK_SIGNAL_FUNC(grid_row_selected_cb),
                     NULL);
  gtk_signal_connect(GTK_OBJECT(cnc_data->grid),
                     "unselect_row",
                     GTK_SIGNAL_FUNC(grid_row_selected_cb),
                     NULL);*/
  gtk_table_attach(GTK_TABLE(cnc_info->sql_tab), cnc_info->sql_grid, 0, 3, 2, 6,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   GTK_FILL | GTK_EXPAND | GTK_SHRINK,
                   3, 3);

  /* add it to the notebook */
  gtk_notebook_append_page(GTK_NOTEBOOK(cnc_info->container),
                           cnc_info->sql_tab,
                           gtk_label_new("SQL"));
}

/*
 * Callbacks
 */
static void
run_sql_cb (GtkWidget *w, FE_ConnectionInfo *cnc_info)
{
  gchar* sql;
  
  g_return_if_fail(cnc_info != NULL);
  
  sql = gtk_editable_get_chars(GTK_EDITABLE(cnc_info->sql_area), 0, -1);
  if (sql)
    {
      gulong         reccount;
      Gda_Recordset* recset;
      
      /* execute command */
      recset = gda_connection_execute(cnc_info->cnc, sql, &reccount, 0);
      if (recset)
        {
          gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_mainwin),
                                        _("Command succeeded"));
        }
      else
        {
          gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_mainwin),
                                        _("Command failed"));
        }
      gnome_db_grid_set_recordset(GNOME_DB_GRID(cnc_info->sql_grid), recset);
      fe_config_add_sql_command(sql);
    }
}

static void
select_sql_cb (GtkWidget *w, FE_ConnectionInfo *cnc_info)
{
  GtkWidget* dialog;
  GtkWidget* frame;
  GtkWidget* scroll;
  GtkWidget* list;
  GList*     commands;
  
  g_return_if_fail(cnc_info != NULL);
  
  /* create the dialog */
  dialog = gnome_dialog_new(_("Select SQL Command"),
                            GNOME_STOCK_BUTTON_OK,
                            GNOME_STOCK_BUTTON_CANCEL,
                            NULL);
  gtk_widget_set_usize(dialog, 350, 250);
  frame = gnome_db_new_frame_widget(NULL);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), frame, 1, 1, 0);
  scroll = gnome_db_new_scrolled_window_widget();
  gtk_container_add(GTK_CONTAINER(frame), scroll);
  
  list = gnome_db_new_clist_widget(NULL, 2);
  commands = fe_config_get_last_commands();
  if (commands)
    {
      GList* node;
      gint   cnt;
      
      gtk_clist_freeze(GTK_CLIST(list));
      for (cnt = 1, node = g_list_first(commands); node; node = g_list_next(node), cnt++)
        {
          gchar* row[2];
          
          row[0] = g_strdup_printf("%d", cnt);
          row[1] = (gchar *) node->data;
          gtk_clist_append(GTK_CLIST(list), row);
        }
      gtk_clist_thaw(GTK_CLIST(list));
    }
  gtk_container_add(GTK_CONTAINER(scroll), list);

  /* run the dialog */
  if (!gnome_dialog_run(GNOME_DIALOG(dialog)))
    {
      gchar* sql = NULL;
      GList* selection;
      
      /* retrieve CList selection */
      selection = GTK_CLIST(list)->selection;
      if (selection)
        {
          gint row = GPOINTER_TO_INT(selection->data);
          if (gtk_clist_get_text(GTK_CLIST(list), row, 1, &sql))
            {
              gtk_editable_delete_text(GTK_EDITABLE(cnc_info->sql_area), 0, -1);
              gtk_text_insert(GTK_TEXT(cnc_info->sql_area), NULL, NULL, NULL, sql, strlen(sql));
            }
        }
    }
  gnome_dialog_close(GNOME_DIALOG(dialog));
}

/*
 * Public functions
 */
void
fe_connection_activate_browser (FE_ConnectionInfo *cnc_info)
{
  g_return_if_fail(cnc_info != NULL);
  
  if (!cnc_info->browser_tab)
    {
      cnc_info->browser_tab = gnome_db_browser_new(cnc_info->cnc);
      gtk_widget_show(cnc_info->browser_tab);
      gtk_notebook_append_page(GTK_NOTEBOOK(cnc_info->container),
                               cnc_info->browser_tab,
                               gtk_label_new(_("Browser")));
    }
  gtk_notebook_set_page(GTK_NOTEBOOK(cnc_info->container),
                        gtk_notebook_page_num(GTK_NOTEBOOK(cnc_info->container), cnc_info->browser_tab));
}

void
fe_connection_activate_sql (FE_ConnectionInfo *cnc_info)
{
  g_return_if_fail(cnc_info != NULL);
  gtk_notebook_set_page(GTK_NOTEBOOK(cnc_info->container),
                        gtk_notebook_page_num(GTK_NOTEBOOK(cnc_info->container), cnc_info->sql_tab));
}

void
fe_connection_begin_transaction (FE_ConnectionInfo *cnc_info)
{
  g_return_if_fail(cnc_info != NULL);
  
  if (gda_connection_begin_transaction(cnc_info->cnc) == 0)
    {
      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_mainwin),
                                    _("Transaction started successfully"));
    }
}

void
fe_connection_close (FE_ConnectionInfo *cnc_info)
{
  g_return_if_fail(cnc_info != NULL);
  
  gtk_widget_destroy(cnc_info->container);
  
  gda_connection_close(cnc_info->cnc);
  gda_connection_free(cnc_info->cnc);
  
  open_connections = g_list_remove(open_connections, (gpointer) cnc_info);
  g_free((gpointer) cnc_info->dsn_name);
  g_free((gpointer) cnc_info);
}

void
fe_connection_commit_transaction (FE_ConnectionInfo *cnc_info)
{
  g_return_if_fail(cnc_info != NULL);
  
  if (gda_connection_commit_transaction(cnc_info->cnc) == 0)
    {
      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_mainwin),
                                    _("Commit executed successfully"));
    }
}

FE_ConnectionInfo *
fe_connection_get_current (void)
{
  FE_ConnectionInfo* cnc_info;
  GtkWidget*         w;

  w = fe_workarea_get_current_tab();
  if (GTK_IS_WIDGET(w))
    {
      cnc_info = (FE_ConnectionInfo *) gtk_object_get_data(GTK_OBJECT(w), FE_CONNECTION_DATA);
      if (cnc_info && cnc_info->container == w)
        return cnc_info;
    }
  return NULL;
}

GList *
fe_connection_get_list (void)
{
  return open_connections;
}

FE_ConnectionInfo *
fe_connection_open (const gchar *dsn_name, Gda_Connection *cnc)
{
  FE_ConnectionInfo* cnc_info;
  gchar*             title;
  
  g_return_if_fail(gda_connection_is_open(cnc));
  
  cnc_info = g_new0(FE_ConnectionInfo, 1);
  cnc_info->cnc = cnc;
  cnc_info->dsn_name = g_strdup(dsn_name);
  
  /* create container */
  cnc_info->container = gnome_db_new_notebook_widget();
  gtk_notebook_set_show_border(GTK_NOTEBOOK(cnc_info->container), TRUE);
  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(cnc_info->container), FALSE);
  gtk_object_set_data(GTK_OBJECT(cnc_info->container), FE_CONNECTION_DATA, cnc_info);
  
  /* add SQL window */
  init_sql_window(cnc_info);
  
  open_connections = g_list_append(open_connections, (gpointer) cnc_info);
  return cnc_info;
}

void
fe_connection_rollback_transaction (FE_ConnectionInfo *cnc_info)
{
  g_return_if_fail(cnc_info != NULL);
  
  if (gda_connection_rollback_transaction(cnc_info->cnc) == 0)
    {
      gnome_db_container_set_status(GNOME_DB_CONTAINER(glb_mainwin),
                                    _("Rollback executed successfully"));
    }
}

void
fe_connection_show_errors (Gda_Connection *cnc)
{
  static GtkWidget* error_dialog = NULL;

  g_return_if_fail(IS_GDA_CONNECTION(cnc));

  if (!error_dialog)
    {
      error_dialog = gnome_db_errordlg_new(cnc, _("Error Viewer"));
    }
  gnome_db_errordlg_show_errors(GNOME_DB_ERRORDLG(error_dialog));
}