/*
 * structfill.C: implementation of the structure filler class
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later 
 * version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *
 * See the AUTHORS file for a list of people who have hacked on 
 * this code. 
 * See the ChangeLog file for a list of changes.
 *
 */

#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/* would prefer <iostream> but some linux systems seem to be 
   missing it */
#include <iostream.h>
#include <algorithm>

#include "snmpkit"
#include "oidseq.h"

void SNMP_structFiller::append(const std::string &oidstr,Tags tag,
			       void (*fp)())
  throw(std::bad_alloc,BerOidBadSubOidException,BerNoOidsException){
  tabdef.push_back(TableEntry(oidstr,tag,fp));
  if(!oidseq)
    oidseq=new OidSeq();
  oidseq->append(oidstr);
}

/* ITS4: ignore */
void SNMP_structFiller::remove(const std::string &oidstr)
  throw(FillerRemoveEmptyException,FillerRemoveNotFoundException,
	FillerCorruptException,OidSeqRemoveNotFoundException,
	OidSeqBadLayoutException){
  //delete the TableEntry
  if(tabdef.empty()) throw FillerRemoveEmptyException();
  std::list<TableEntry>::iterator pos=find_if(tabdef.begin(),tabdef.end(),
					 TableEntry_eq(oidstr));
  if(pos==tabdef.end()) throw FillerRemoveNotFoundException();
  tabdef.erase(pos);
  //delete it off of the oidseq
  if(oidseq==NULL) throw FillerCorruptException();
  oidseq->remove(oidstr); /* ITS4: ignore */
}

SNMP_structFiller::~SNMP_structFiller(){
  if(oidseq) delete oidseq;
  if(retseq) delete retseq;
}

/* This is a hack to get around a compiler bug. The compiler seems
   to get lost in the returning of the function pointer */
typedef void (*FP_RETURNED)(void *,const unsigned char*);
inline FP_RETURNED to_ipaddr_funct(void (*fp)(void)){
 return reinterpret_cast<void (*)(void*,const unsigned char *)>(fp);
}

inline const ustring &to_ipaddr_ustr(BerBase *cur){
  return dynamic_cast<BerIPAddr*>(cur)->ipaddr();
}

int SNMP_structFiller::fillStruct(OidSeq *data,void *curstruct)
  throw(FillerTypeMismatchException,BerUnexpectedTagException,
	OidSeqBadLayoutException){
  int retval=1;
  for(std::list<TableEntry>::iterator cur=tabdef.begin();cur!=tabdef.end();cur++){
    BerBase *curber;
    if(!(curber=data->child(cur->oidstr))){
      retval=0;
      break;
    }

    if(curber->type()!=cur->type){
      /* XXX this kind of fixup might cause problem with large 
	 counters or large ints */
      if(curber->type()==INT_TAG && cur->type==COUNTER_TAG)
	// these errors are minor enough and simple enough to fix
	cerr << "Warning: Counter returned when Integer expected for " 
	     << cur->oidstr << " Buggy firmware?\n";
      else if(curber->type()==COUNTER_TAG && cur->type==INT_TAG)
	// these errors are minor enough and simple enough to fix
	cerr << "Warning: Integer returned when Counter expected for "
	     << cur->oidstr << " Buggy firmware?\n";
      else{
	ios::fmtflags opts=ios::hex;
	opts=cerr.flags(opts);
	cerr << "Warning: Printer returned a value of type 0x"
	     << static_cast<long unsigned int>(curber->type()) 
	     << " when a value of 0x" 
	     << static_cast<long unsigned int>(cur->type)
	     << " was expected for " << cur->oidstr 
	     << " Buggy firmware? Skipping.\n";
	cerr.flags(opts);
	retval=0;
	break;
      }
    }
    
    switch(cur->type){
    case INT_TAG:
      if(dynamic_cast<BerInt*>(curber)==NULL) 
	throw FillerTypeMismatchException();
      // essentially (cur->fp)(curstruct,curber->value())
      reinterpret_cast<void (*)(void*,long)>(cur->fp)(curstruct,dynamic_cast<BerInt*>(curber)->value());
      break;
    case COUNTER_TAG:
      if(dynamic_cast<BerCounter*>(curber)==NULL) 
	throw FillerTypeMismatchException();
      // essentially (cur->fp)(curstruct,curber->value())
      reinterpret_cast<void (*)(void*,unsigned long)>(cur->fp)(curstruct,dynamic_cast<BerCounter*>(curber)->value());
      break;
    case TIME_TICK_TAG:
      if(dynamic_cast<BerTimeTick*>(curber)==NULL) 
	throw FillerTypeMismatchException();
      // essentially (cur->fp)(curstruct,curber->value())
      reinterpret_cast<void (*)(void*,unsigned long)>(cur->fp)(curstruct,dynamic_cast<BerTimeTick*>(curber)->value());
      break;
    case IPADDR_TAG:
      if(dynamic_cast<BerIPAddr*>(curber)==NULL) 
	throw FillerTypeMismatchException();
      else {
	// essentially (cur->fp)(curstruct,curber->IPaddr());
	/* XXX -- this is a nasty hack to work around some 
	   brokenness in libstdc++ 2.95.2 on linux. You can't call
	   c_str() on a basic std::string created from an unsigned 
	   type. */
	ustring tmp=to_ipaddr_ustr(curber);
	if(tmp.length()==0){
	  static const unsigned char foof[]={0};
	  tmp=foof;
	}
	to_ipaddr_funct(cur->fp)(curstruct,to_ipaddr_ustr(curber).data());
      }
      break;
    case STRING_TAG:
      if(dynamic_cast<BerString*>(curber)==NULL) 
	throw FillerTypeMismatchException();
      // essentially (cur->fp)(curstruct,curber->Str())
      reinterpret_cast<void (*)(void*,const char *)>(cur->fp)(curstruct,dynamic_cast<BerString*>(curber)->value().c_str());
      break;
    default:
      throw BerUnexpectedTagException();
    }
  }
  return retval;
}

void *SNMP_structFiller::get(void *tobefilled)
  throw(SNMPNoResponseException,SNMPPacketNotSequenceException,
	SNMPRespNotSequenceException,SNMPNotResponseTagException,
	SNMPSeqnoNotIntException,SNMPStateNotIntException,
	SNMPFaultOidNotIntException,OidSeqBadLayoutException,
	SNMPBadOidException,std::bad_alloc,SocketSendShortExecption,
	BerSequenceTagException,BerLengthException,BerIntTagException,
	BerIntLengthExecption,BerCounterTagException,
	BerCounterLengthExecption,BerStringTagException,BerNullTagException,
	BerNullLengthExecption,BerOidTagException,BerTimeTickTagException,
	BerTimeTickLengthExecption,BerIPAddrLengthExecption){
  retseq=session.__get(oidseq);
  if(retseq==NULL)
    throw SNMPNoResponseException();

  if(!fillStruct(retseq,(unsigned char*)tobefilled)){
    cerr << "Warning: printer did not respond with a value for one of the "
	 << "OIDs. Buggy firmware?\n";
    return NULL;
  }

  return tobefilled;
}

void *SNMP_structFiller::get_next(void *tobefilled)
  throw(SNMPNoResponseException,SNMPPacketNotSequenceException,
	SNMPRespNotSequenceException,SNMPNotResponseTagException,
	SNMPSeqnoNotIntException,SNMPStateNotIntException,
	SNMPFaultOidNotIntException,OidSeqBadLayoutException,
	SNMPBadOidException,std::bad_alloc,SocketSendShortExecption,
	BerSequenceTagException,BerLengthException,BerIntTagException,
	BerIntLengthExecption,BerCounterTagException,
	BerCounterLengthExecption,BerStringTagException,BerNullTagException,
	BerNullLengthExecption,BerOidTagException,BerTimeTickTagException,
	BerTimeTickLengthExecption,BerIPAddrLengthExecption){
  if(retseq){
    delete oidseq;
    oidseq=retseq;
    retseq=NULL;
  }
  retseq=session.__get_next(oidseq);
  if(retseq==NULL)
    throw SNMPNoResponseException();

  if(!fillStruct(retseq,(unsigned char*)tobefilled))
    return NULL;

  return tobefilled;
}

void SNMP_table::get(std::list<void*> &dest) 
  throw(SNMPNoResponseException,SNMPPacketNotSequenceException,
	SNMPRespNotSequenceException,SNMPNotResponseTagException,
	SNMPSeqnoNotIntException,SNMPStateNotIntException,
	SNMPFaultOidNotIntException,OidSeqBadLayoutException,
	SNMPBadOidException,std::bad_alloc,SocketSendShortExecption,
	BerSequenceTagException,BerLengthException,BerIntTagException,
	BerIntLengthExecption,BerCounterTagException,
	BerCounterLengthExecption,BerStringTagException,BerNullTagException,
	BerNullLengthExecption,BerOidTagException,BerTimeTickTagException,
	BerTimeTickLengthExecption,BerIPAddrLengthExecption){
  void *newone;
  for(newone=constructor();get_next(newone);
      newone=constructor()){ 
    dest.push_back(newone);
  }
}



