#include <errno.h>       
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <glib.h>

#include "parport.h"
#include "snmpinter.h"

#define IOCNR_GET_DEVICE_ID	1
#define LPIOC_GET_DEVICE_ID(len) _IOC(_IOC_READ, 'P', IOCNR_GET_DEVICE_ID, len)	/* get device_id string */

/*----------------------------------------------------------------
  libprinterconf - a function library for detecting and setting up
  printers in linux.

  03/08/2000
----------------------------------------------------------------*/

#include "printerconf.h"

#define PCONF_MAX_XREF_STRLEN 1024
#define PCONF_XREF_FILE "/extra/projects/printing/libprinterconf/printerconf.xref"

/*----------------------------------------------------------------
  Autodetect printer on a parallel port using IEEE1284 protocol.

  Attempt parallel port autodetection (IEEE1284). Return a
  string containing a unique printer ID
----------------------------------------------------------------*/
static char *pconf_autodetect_pport(long int port);

pconf_detmethod_t pconf_methods[4] = {
  {PCONF_DETECT_PPORT, "Parallel Port"},
  {PCONF_DETECT_NETWORK, "Network"},
  {PCONF_DETECT_USB,"USB"},
  // {PCONF_DETECT_IRDA,"irda"},
  // {PCONF_DETECT_HPMULTI, "HP Multicast"}, 
  {0, ""}
};

static pconf_xref_t *pconf_xref_top = NULL;

/*==============================================================*/
pconf_detmethod_t *pconf_get_detection_methods(int *count)
{
/*----------------------------------------------------------------
  Return a list (array) of the possible printer detection
  methods.

  Note: need to do better than this; what if we can't 
  (for example) parport probe? Do we want to even present an
  option if we can't do it?
----------------------------------------------------------------*/

  *count = 2;
  return &(pconf_methods[0]);

}

/*==============================================================*/
char *pconf_autodetect_pport(long int port)
{
/*----------------------------------------------------------------
  Note: Are we going to use parport for this? Does parport
  put the printer information into /proc/parport/0/devices,
  etc.?
----------------------------------------------------------------*/
  char *printer_name = NULL;

  /* These 2 calls use Red Hat printer-config parport.c calls. */
  probe_parport(port);
  printer_name = (char *)parport_probe_model(port);

  return printer_name;
}

static char **pconf_autodetect_usb(char *devstr){
  char **dev_strings=g_strsplit(devstr,",",0);
  char **cur;
  char **retval;
  char **curret;
  for(cur=dev_strings;*cur!=NULL;cur++);
  curret=retval=malloc(sizeof(char*)*(cur-dev_strings+1));
  memset(retval,0,sizeof(char*)*(cur-dev_strings+1));

  for(cur=dev_strings;*cur!=NULL;cur++){
     char buf[8192];
     int fd;
     int val;
     char *vendor;
     char *model;
     char *c2;
     char *orig_string;
     GString *str=g_string_new("/dev/usb/lp");
     g_string_append_printf(str,"%s",*cur);
     fd=open(str->str,O_RDONLY);
     memset(buf,0,8192);
     
     if(fd==-1 || (val=ioctl(fd,LPIOC_GET_DEVICE_ID(8192),buf))==-1){
       g_string_free(str,TRUE);
       continue;
     }
     if((vendor=strstr(buf+2,"MFG:"))==0){
       g_string_free(str,TRUE);
       continue;
     }
     vendor+=4;
     if((model=strstr(buf+2,"MDL:"))==0){
       g_string_free(str,TRUE);
       continue;
     }
     model+=4;
     if((c2=strchr(vendor,';'))==0){
       g_string_free(str,TRUE);
       continue;
     }
     *c2=0;
     if((c2=strchr(model,';'))==0){
       g_string_free(str,TRUE);
       continue;
     }
     *c2=0;
     orig_string = g_strdup (str->str);
     g_string_printf(str,"device=%s;vendor=%s;model=%s",orig_string,vendor,
		     model);
     g_free (orig_string);
     *curret=str->str;
     curret++;
     g_string_free(str,FALSE);
   }
  g_strfreev(dev_strings);
  return retval;
}

/*==============================================================*/
char **pconf_detect_printer(int dettype, char *detinfo, int *retval)
{
/*----------------------------------------------------------------
  Attempt to detect a printer of a specific type (attached to
  parallel port, networked, etc.). Return a list of unique 
  printer IDs that can be used to setup the detected printers.
----------------------------------------------------------------*/
  char *temp = NULL;
  char *portstr = NULL;
  char **top = NULL;
  char *endptr = NULL;
  char *p = NULL;
  char *info = NULL;
  char *sep = ",";
  long int port = -1;
  int count = 0;
  char *aline = NULL;

  *retval = 0;

  switch (dettype) {
  case PCONF_DETECT_PPORT:
    /* Parse the detinfo (a comma separated set of numbers. */
    temp = (char *)strdup(detinfo);

    portstr = (char *)strtok(temp, sep);
    /* Process a single port string. */
    while (portstr) {
      /* Convert to number */
      port = strtol(portstr, &endptr, 10);
      if (endptr != portstr) {
	/* Expand the list of char*'s as we go. */
	top = realloc(top, (++count) * sizeof(char *));

	/* Get the printer ID for this port */
	p = (char *)pconf_autodetect_pport(port);
	if (!p)
	  p = (char *)strdup("Port not configured");

	info = malloc(strlen(portstr) + strlen(p) + 25);
	sprintf(info, "port=%ld;model=%s", port, p);
	if (p)
	  free(p);

	top[count - 1] = info;
      }
      portstr = (char *)strtok(NULL, sep);
    }
    top = realloc(top, (++count) * sizeof(char *));
    top[count - 1] = NULL;


    free(temp);
    break;
  case PCONF_DETECT_NETWORK:
    p = (char *) get_snmp_printers (detinfo, retval);

    /* An error occurred. */
    if (*retval != 0)
      return NULL;

    /* Split p up at newlines and make array of string pointers.*/
    if (p != NULL) {
      temp = (char *)strdup(p);
      aline = (char *)strtok(temp, "\n");

      while (aline) {
	/* Expand the list of char*'s as we go. */
	top = realloc(top, (++count) * sizeof(char *));

	info = strdup(aline);
	top[count - 1] = info;

	aline = (char *)strtok(NULL, "\n");
      }
      /* Null terminate the list of strings (char pointers). */
      top = realloc(top, (++count) * sizeof(char *));
      top[count - 1] = NULL;

      free(temp);
    }
    break;
  case PCONF_DETECT_USB:
    return pconf_autodetect_usb(detinfo);
    break;
  default:
    break;
  }

  return (char **)top;
}

/*==============================================================*/
pconf_xref_t *pconf_read_xref_file(char *filename)
{
/*----------------------------------------------------------------
  Read a printer cross-reference file, which has the format:

  # comment
  printer-id\tppd-file-name

----------------------------------------------------------------*/

  FILE *fp;
  char *buf = NULL;
  char *id = NULL;
  char *ppd = NULL;
  char *shortdesc = NULL;
  char *longdesc = NULL;

  pconf_xref_t *top = NULL;
  pconf_xref_t *current = NULL;
  pconf_xref_t *new = NULL;

  /* Sanity checks */

  if (filename == NULL)
    return NULL;

  fp = fopen(filename, "r");	/* ITS4: ignore */
  if (!fp)
    return NULL;

  /* Read file into array */

  buf = (char *)malloc(PCONF_MAX_XREF_STRLEN);
  while (fgets(buf, PCONF_MAX_XREF_STRLEN, fp)) {

    /* Get rid of any trailing newline. */
    if (buf[strlen(buf) - 1] == '\n')
      buf[strlen(buf) - 1] = 0;

    /* Get rid of initial white space. */
    while ((buf[0] != 0) && ((buf[0] == ' ') || (buf[0] == '\t')))
      buf++;

    /* Continue parsing any non-null lines. */
    if ((buf[0] != 0) && (buf[0] != '#')) {

      /* Break on tab */
      id = (char *)strtok(buf, "\t");
      ppd = (char *)strtok(NULL, "\t");
      shortdesc = (char *)strtok(NULL, "\t");
      longdesc = (char *)strtok(NULL, "\t");

      printf("ID: %s FILE: %s\n", id, ppd);	/* ITS4: ignore */

      /* Add this data to xref list. */

      new = (pconf_xref_t *) malloc(sizeof(pconf_xref_t));
      new->id = id ? (char *)strdup(id) : NULL;
      new->ppd = ppd ? (char *)strdup(ppd) : NULL;
      new->shortdesc = shortdesc ? (char *)strdup(shortdesc) : NULL;
      new->longdesc = longdesc ? (char *)strdup(longdesc) : NULL;

      /* list maintenance */
      if (new->id && new->ppd) {
	new->next = NULL;
	if (!top)
	  top = new;
	else
	  current->next = new;

	current = new;
      } else
	free(new);

    }
  }


  fclose(fp);

  /* Sort array by printer-id */

  return top;
}

/*==============================================================*/
pconf_xref_t *pconf_find_xref_by_id(char *target_id)
{
/*----------------------------------------------------------------
  Given a printer id, return the matching xref struct, or NULL
  if not found.
----------------------------------------------------------------*/
  pconf_xref_t *curr;

  if (pconf_xref_top == NULL)
    pconf_xref_top = (pconf_xref_t *) pconf_read_xref_file(PCONF_XREF_FILE);

  for (curr = pconf_xref_top; curr; curr = curr->next)
    if (!strcmp(curr->id, target_id))
      return curr;

  return NULL;
}
