/* 
 * Copyright (C) 1999 JP Rosevear
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <gnome.h>

#include "board-window.h"
#include "board_window_menu.h"
#include "prefs.h"

typedef struct {
	guint8 dark_r;
	guint8 dark_g;
	guint8 dark_b;

	guint8 light_r;
	guint8 light_g;
	guint8 light_b;
	
	char *piecedir;
	
	gint notation;
	gboolean beeponmove;
	gboolean promotetoqueen;
	gboolean autoflag;

	GList *programs;
	GList *servers;

	gchar *telnet_program;
} Prefs;

static Prefs *prefs;
extern BoardWindow *main_board_window;

void 
prefs_init (void)
{
	int i, cnt;
	
	prefs = g_new0 (Prefs, 1);

	prefs->dark_r = gnome_config_get_int ("/gnome-chess/Board/Dark_R=0");
	prefs->dark_g = gnome_config_get_int ("/gnome-chess/Board/Dark_G=100");
	prefs->dark_b = gnome_config_get_int ("/gnome-chess/Board/Dark_B=0");

	prefs->light_r = gnome_config_get_int ("/gnome-chess/Board/Light_R=144");
	prefs->light_g = gnome_config_get_int ("/gnome-chess/Board/Light_G=238");
	prefs->light_b = gnome_config_get_int ("/gnome-chess/Board/Light_B=144");

	prefs->piecedir = gnome_config_get_string ("/gnome-chess/Board/PieceDir=" 
						   GNOMEDATADIR "/pixmaps/gnome-chess/");
	
	prefs->notation = gnome_config_get_int ("/gnome-chess/Game/Notation=0");
	prefs->beeponmove = gnome_config_get_bool ("/gnome-chess/Game/BeepOnMove=TRUE");
	prefs->promotetoqueen = gnome_config_get_bool ("/gnome-chess/Game/PromoteToQueen=TRUE");
	prefs->autoflag = gnome_config_get_bool ("/gnome-chess/Game/AutoFlag=TRUE");

	cnt = gnome_config_get_int ("/gnome-chess/Programs/Count=0");
	for (i = 0; i < cnt; i++) {
		PrefsProgram *pp = g_new0 (PrefsProgram, 1);
		char *str;
		
		str = g_strdup_printf ("/gnome-chess/Programs/ProgramName%d", i);
		pp->name = gnome_config_get_string (str);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Programs/ProgramLocation%d", i);
		pp->location = gnome_config_get_string (str);
		g_free (str);

		prefs->programs = g_list_append (prefs->programs, pp);
	}

	prefs->telnet_program = gnome_config_get_string ("/gnome-chess/Servers/TelnetProgram=telnet");

	cnt = gnome_config_get_int ("/gnome-chess/Servers/Count=0");
	for (i = 0; i < cnt; i++) {
		PrefsServer *ps = g_new0 (PrefsServer, 1);
		char *str;
		
		str = g_strdup_printf ("/gnome-chess/Servers/ServerName%d", i);
		ps->server = gnome_config_get_string (str);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerPort%d", i);
		ps->port = gnome_config_get_string (str);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerUser%d", i);
		ps->user = gnome_config_get_string (str);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerPassword%d", i);
		ps->password = gnome_config_get_string (str);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerConnect%d=%s", i,
				       prefs_get_telnetprogram ());
		ps->connect = gnome_config_get_string (str);
		g_free (str);

		prefs->servers = g_list_append (prefs->servers, ps);
	}
}

static void 
prefs_dialog_write (void)
{
	GList *l;
	int cnt = 0;
	
	gnome_config_set_int ("/gnome-chess/Board/Dark_R", prefs->dark_r);
	gnome_config_set_int ("/gnome-chess/Board/Dark_G", prefs->dark_g);
	gnome_config_set_int ("/gnome-chess/Board/Dark_B", prefs->dark_b);

	gnome_config_set_int ("/gnome-chess/Board/Light_R", prefs->light_r);
	gnome_config_set_int ("/gnome-chess/Board/Light_G", prefs->light_g);
	gnome_config_set_int ("/gnome-chess/Board/Light_B", prefs->light_b);

	gnome_config_set_string ("/gnome-chess/Game/PieceDir", prefs->piecedir);

	gnome_config_set_int ("/gnome-chess/Game/Notation", prefs->notation);
	gnome_config_set_bool ("/gnome-chess/Game/BeepOnMove", prefs->beeponmove);
	gnome_config_set_bool ("/gnome-chess/Game/PromoteToQueen", prefs->promotetoqueen);
	gnome_config_set_bool ("/gnome-chess/Game/AutoFlag", prefs->autoflag);

	for (l = prefs->programs, cnt = 0; l != NULL; l = l->next, cnt++) {
		PrefsProgram *pp = l->data;
		char *str;
		
		str = g_strdup_printf ("/gnome-chess/Programs/ProgramName%d", cnt);
		gnome_config_set_string (str, pp->name);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Programs/ProgramLocation%d", cnt);
		gnome_config_set_string (str, pp->location);
		g_free (str);
	}
	gnome_config_set_int ("/gnome-chess/Programs/Count", cnt);

	for (l = prefs->servers, cnt = 0; l != NULL; l = l->next, cnt++) {
		PrefsServer *ps = l->data;
		char *str;
		
		str = g_strdup_printf ("/gnome-chess/Servers/ServerName%d", cnt);
		gnome_config_set_string (str, ps->server);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerPort%d", cnt);
		gnome_config_set_string (str, ps->port);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerUser%d", cnt);
		gnome_config_set_string (str, ps->user);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerPassword%d", cnt);
		gnome_config_set_string (str, ps->password);
		g_free (str);

		str = g_strdup_printf ("/gnome-chess/Servers/ServerConnect%d", cnt);
		gnome_config_set_string (str, ps->connect);
		g_free (str);
	}
	gnome_config_set_int ("/gnome-chess/Servers/Count", cnt);

	gnome_config_set_string ("/gnome-chess/Servers/TelnetProgram", prefs->telnet_program);
	
	gnome_config_sync ();
}

void
prefs_sync (void) 
{
	prefs_dialog_write ();

	board_window_set_program_menu (main_board_window);
	board_window_set_server_menu (main_board_window);
}

void 
prefs_get_light_color (guint8 *r, guint8 *g, guint8 *b)
{
	*r = prefs->light_r;
	*g = prefs->light_g;
	*b = prefs->light_b;
}

void 
prefs_set_light_color (guint8 r, guint8 g, guint8 b)
{
	prefs->light_r = r;
	prefs->light_g = g;
	prefs->light_b = b;
}

void 
prefs_get_dark_color (guint8 *r, guint8 *g, guint8 *b)
{
	*r = prefs->dark_r;
	*g = prefs->dark_g;
	*b = prefs->dark_b;
}

void 
prefs_set_dark_color (guint8 r, guint8 g, guint8 b)
{
	prefs->dark_r = r;
	prefs->dark_g = g;
	prefs->dark_b = b;
}


const char *
prefs_get_piecedir (void)
{
	return prefs->piecedir;
}

void 
prefs_set_piecedir (const char *piecedir)
{
	if (prefs->piecedir)
		g_free (prefs->piecedir);
	
	prefs->piecedir = g_strdup (piecedir);
}

gint 
prefs_get_notation (void)
{
	return prefs->notation;
}

void
prefs_set_notation (gint notation)
{
	prefs->notation = notation;
}

gboolean 
prefs_get_beeponmove (void)
{
	return prefs->beeponmove;
}

void
prefs_set_beeponmove (gboolean beep)
{
	prefs->beeponmove = beep;
}

gboolean 
prefs_get_promotetoqueen (void)
{
	return prefs->promotetoqueen;
}

void
prefs_set_promotetoqueen (gboolean queen)
{
	prefs->promotetoqueen = queen;
}

gboolean
prefs_get_autoflag (void)
{
	return prefs->autoflag;
}

void
prefs_set_autoflag (gboolean autoflag)
{
	prefs->autoflag = autoflag;
}

GList *
prefs_get_programs (void)
{
	return prefs->programs;
}

void
prefs_set_programs (GList *programs)
{
	prefs_free_program_list (prefs->programs);
	prefs->programs = prefs_copy_program_list (programs);
}

GList *
prefs_copy_program_list (GList *list) 
{
	GList *l, *new = NULL;

	for (l = list; l != NULL; l = l->next) {
		PrefsProgram *pp = l->data;		
		PrefsProgram *pp2 = g_new0 (PrefsProgram, 1);

		pp2->name = g_strdup (pp->name);
		pp2->location = g_strdup (pp->location);

		new = g_list_append (new, pp2);
	}

	return new;
}

void 
prefs_free_program_list (GList *list)
{
	GList *l;
	
	for (l = list; l != NULL; l = l->next) {
		PrefsProgram *pp = l->data;
		
		g_free (pp->name);
		g_free (pp->location);
		g_free (pp);
	}
	
	g_list_free (list);
}

GList *
prefs_get_servers (void)
{
	return prefs->servers;;
}

void
prefs_set_servers (GList *servers)
{
	prefs_free_server_list (prefs->servers);
	prefs->servers = prefs_copy_server_list (servers);
}

GList *
prefs_copy_server_list (GList *list) 
{
	GList *l, *new = NULL;

	for (l = list; l != NULL; l = l->next) {
		PrefsServer *ps = l->data;		
		PrefsServer *ps2 = g_new0 (PrefsServer, 1);
		
		ps2->user = g_strdup (ps->user);
		ps2->server = g_strdup (ps->server);
		ps2->port = g_strdup (ps->port);
		ps2->password = g_strdup (ps->password);
		ps2->connect = g_strdup (ps->connect);

		new = g_list_append (new, ps2);
	}

	return new;
}

void 
prefs_free_server_list (GList *list)
{
	GList *l;
	
	for (l = list; l != NULL; l = l->next) {
		PrefsServer *ps = l->data;
		
		g_free (ps->server);
		g_free (ps->port);
		g_free (ps->user);
		g_free (ps->password);
		g_free (ps->connect);
		g_free (ps);
	}
	
	g_list_free (list);
}

const gchar *
prefs_get_telnetprogram (void)
{
	return prefs->telnet_program;
}

void 
prefs_set_telnetprogram (const gchar *telnet_program)
{
	prefs->telnet_program = g_strdup (telnet_program);
}
